;;; -*- Mode: Lisp; Package: gnode -*-

;;; $Header: /afs/athena.mit.edu/course/other/allp/nlp/tools/RCS/gnode-support.lisp,v 3.2 92/07/08 16:29:00 sfelshin Exp $

;;; Copyright (c) 1986-92 Massachusetts Institute of Technology.
;;; All rights reserved.
;;; Q: How many Athenians does it take to screw the Athena Language
;;;    Learning Project?
;;; A: Two.  One to update the system software and one to answer
;;;    the hardware hotline.

(in-package :gnode)

;;; This module provides general support useful for tree display and
;;; defines global variables which control default behavior of tree display
;;; and which are used by more than one tree display module.  Contrast
;;; module gnode-tools, which provides low-level support: process control
;;; and an interface to graphic display.
(provide-module :gnode-support)

#||
(global '(*gnode-snords* *gnode-new-bindings* *gnode-old-bindings*
	  *gnode-attend* *gnode-underlying-strings* *gnode-node-features*))
||#

(require-module :probes :macros t)	;for ENSURE-THAT
#||(use-package :probes)||#

(require-module :gnode-tools :macros t)
#+:ccl (require-module :mac-gnode)

(require-module :char-map)
(require-module :transliterate :macros t)
(require-module :std-chars)
#+:ccl (require-module :smarta-chars :when (eq (current-language) :greek))
#||(use-package :transliterate)||#

(require-module :snags)
#||(use-package :snags)||#

(require-module :feat-support :macros t)
(require-module :features :compile nil)
#||(use-package :features)||#

(require-module :nords :accessors t)
(require-module :snord)
#||(use-package :grammar)||#

(require-module :transfer-functions)
#||(use-package :transfer-functions)||#

(require-module :accents)
#||(use-package :accents)||#


;;; ================================================================
;;; Global variables for control of default display behavior.

(defvar *gnode-snords* t)
(defvar *gnode-old-bindings* t)
(defvar *gnode-new-bindings* t)
(defvar *gnode-attend* nil)
(defvar *gnode-underlying-strings* nil)
(defvar *gnode-node-features* nil)


;;; ================================================================
;;; Subscripts for coindexing.

;;; The table is used by various modules but they have their own algorithms
;;; for assigning subscripts, based on their local definitions of
;;; "coindexed".
(defvar *subscript-table* (make-hash-table :test #'eq))
(defvar *current-subscript* -1)

(defun node-subscript (node)
  (gethash node *subscript-table*))

(defun reset-subscripts ()
  (clrhash *subscript-table*)
  (setq *current-subscript* -1))

(defun subscript-char (node)
  (code-char (+ (char-code #\i) (node-subscript node))))


;;; ================================================================
;;; Titles for iconified windows.

(defun window-title (leaves &optional prefix)
  (gnode-command
   (if (eq (current-language) :greek) "K" "T")
   (transliterate-string
    (format nil "~@[~A~]~{~A~^ ~}" prefix (mapcar #'word-surface-string leaves))
    lexicon-internal
    (if (eq (current-language) :greek)
	*standard-symbol-character-set*
      *standard-character-set*))))
  
(defun collect-leaves (nord snord-box)
  (declare (type nord nord) (type (or null snord-box) snord-box))
  (when (and *gnode-snords* snord-box)
    (setq nord (snord nord snord-box)))
  (etypecase nord
    (word
     (unless (zerop (length (word-surface-string nord)))
       (list nord)))
    (node (mapcan
	   #'(lambda (nd)
	       (ensure-that () (nord-p nd))
	       (collect-leaves nd snord-box))
	   (node-children nord)))))


;;; ================================================================
;;; Snags.

(defun local-penalty (command snags)
  (declare (list snags) (special *reported-snags*))
  (let ((penalty 0) (args ()))
    (declare (fixnum penalty))
    (dolist (snag snags)
      (when (not (member (cdr snag) *reported-snags* :test #'eq))
	(push (cdr snag) *reported-snags*)
	(when args
	  (push "+" args))
	(push
	 (nstring-downcase
	  (princ-to-string
	   (or (snag-short-name (cdr snag)) "???")))
	 args)
	(incf penalty (car snag))))
    (when args
      (push "=" args)
      (push penalty args)
      (apply #'gnode-command command (nreverse args)))))


;;; ================================================================
;;; Proper string for word.

(defun word-string-and-face (word snord-box)
  (let ((face "a2"))
    (values
     (cond ((if *gnode-underlying-strings*
		(> (length (word-underlying-string word)) 0)
	      (> (length (word-surface-string word)) 0))
            (copy-seq (transliterate-string
                       (if *gnode-underlying-strings*
                         (word-underlying-string word)
                         (word-surface-string word))
                       lexicon-internal
                       (if (eq (current-language) :greek)
                         #-:ccl *standard-symbol-character-set*
                         #+:ccl smarta-character-set
                         *standard-character-set*))))
	   ((and (boundp '!pwo) (labeled-p word !pwo))
	    (setq face "a0")
	    "pro")
	   ((and *gnode-new-bindings*
		 snord-box
		 (sb-immediate-binder word snord-box))
	    (cond
	      ((labeled-p word !trace)
	       ;; &&& Should display tau unless Greek, plain t in Greek.
	       ;; Tau would be too confusing in Greek (likewise epsilon for
	       ;; empty).  Must check proper char-map to find out what
	       ;; character maps to tau.
	       (setq face "a1")
	       "t")
	      (t
	       ;; This should never happen!
	       "PRO")))
	   ;; Prefer imeanings on generation because dmeaning slot must be
	   ;; simple so imeaning is more interesting/accurate.  Could look
	   ;; at derivation, but this is D- or S-structure, not
	   ;; M-structure.
	   ((and (gword-p word)
		 (gword-imeaning word))
            (setq face "a0")
	    (if (consp (gword-imeaning word))
		(format nil "~(~{~A~^ ~}~)" (gword-imeaning word))
	      (string-downcase (gword-imeaning word))))
	   ((word-dmeaning word)
            (setq face "a0")
	    (string-downcase (word-dmeaning word)))
	   ;; Any better ideas?
	   (t "nil"))
     ;; Proper value of this variable is highly dependent on left-to-right
     ;; evaluation.
     face)))
