;;;; -*- Mode:Common-Lisp; Package:DSS; Base:10 -*-
;;;; *-* File: Hillary: RUBINSTEIN.DSS-ARM.UTILITIES DELAY-EXPERIMENT.LISP *-*
;;;; *-* Last-Edit: Thursday, June 3, 1993  16:47:48; Edited-By: RUBINSTEIN *-*
;;;; *-* Machine: Hillary (Explorer II, Microcode 489) *-*
;;;; *-* Software: TI Common Lisp System 6.49 *-*
;;;; *-* Lisp: TI Common Lisp System 6.49  *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *                                                                        *
;;;; *                            DELAY EXPERIMENT                            *
;;;; *                                                                        *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Zachary B. Rubinstein
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-29-93 File Created.  (RUBINSTEIN)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'DSS)

(proclaim '(optimize (speed 3) (safety 1)))

;;; ---------------------------------------------------------------------------
;;; Write Report Header
;;; ---------------------------------------------------------------------------

(defun write-report-header (stream additional-description)
  (format stream "Airport:    ~a~@
                  Timetable:  ~a~@
                  Rating Scheme:  ~a~%~a~%"
          (arm:get-loaded-airport)
          (arm:get-loaded-timetable)
          (get-rating-scheme)
          additional-description))

;;; ---------------------------------------------------------------------------
;;; Write Pre-Execution Reports to a File
;;; ---------------------------------------------------------------------------

(defun write-pre-execution-reports (filename &optional (additional-description ""))
  (with-open-file (stream filename :direction :output)
    (write-report-header stream additional-description)
    (prepare-dss-execution-stats-file stream)
    (terpri stream)))

;;; ---------------------------------------------------------------------------
;;; Resources Status Reporting Function
;;; ---------------------------------------------------------------------------

(defun report-resources (&optional (stream *standard-output*))
  (loop initially (format stream "~%RESOURCES:~%") 
        for resource-type in (mapcar #'class-name (class-direct-subclasses (find-class 'movable-resource))) do
        (format stream "  ~25<~A:~;~D~>~%" resource-type (number-of-existing-resources resource-type))))

;;; ---------------------------------------------------------------------------
;;; Write Post-Execution Reports to a File
;;; ---------------------------------------------------------------------------

(defun write-post-execution-reports (filename &optional (additional-description "") (header nil))
  (with-open-file (stream filename :direction :output :if-exists :append)
    (when header
      (write-report-header stream additional-description))
    (format stream "~a" additional-description)
    (format stream "~%~%")
    (check-schedule-integrity :stream stream)
    (terpri stream)
    (print-order-stats stream)
    (terpri stream)
    (print-resource-stats stream)
    (terpri stream)
    (print-finished-execution-stats stream)
    (terpri stream)
    (report-resources stream)))

;;; ---------------------------------------------------------------------------
;;; Utilities for Checking, Reporting, and Setting Variables
;;; ---------------------------------------------------------------------------

(defvar *delay-default-pathname* "hillary:rubinstein.dss-arm.reports;")

;;; ---------------------------------------------------------------------------

(defun delay-generate-filename (rating-scheme airport-delimiter suffix &optional (type "text"))
  (make-pathname :defaults *delay-default-pathname*
                 :name (format nil "~a-~a-BTS-~a-FTS-~a"
                               (rating-scheme-nickname rating-scheme)
                               (number-of-existing-resources 'arm::baggage-truck)
                               airport-delimiter
                               suffix)
                 :type type))

;;; ---------------------------------------------------------------------------

(defun delay-initialize-trial (rating-scheme airport-delimiter &optional airport (additional-description ""))
  (let (report-filename)
    (set-rating-scheme rating-scheme)
    (when (and airport
               (not (eq airport (arm:get-loaded-airport))))
      (arm::execute-load-airport :LOAD-SPECIFIC-FILE airport))
    (setf report-filename (delay-generate-filename rating-scheme airport-delimiter "STATS"))
    (write-pre-execution-reports report-filename additional-description)
    report-filename))

;;; ---------------------------------------------------------------------------

(defun delay-report (report-filename airport-delimiter rating-scheme)
  (clip:write-current-experiment-data)
  (write-post-execution-reports report-filename)
  (write-optimal-scheduled-plotter-data (delay-generate-filename rating-scheme airport-delimiter 
                                                                 "PLOTTER" "LISP")))

;;; ---------------------------------------------------------------------------
;;; Delay Information
;;; ---------------------------------------------------------------------------

(defun delays-information ()
  (LOOP WITH local-delay-time = 0
        AND network = nil
        AND release-time = nil
        AND due-date = nil
        AND start-time = nil
        AND finish-time = nil
        AND avg-delay = 0
        AND std-dev-delay = 0
        AND max-delay = 0
        FOR order IN (find-units 'order (make-paths '(scheduler * order)) :ALL)
        DO
        (setf network (order$network order))
        (setf release-time (order$release-time order))
        (setf due-date (order$due-date-time order))
        (setf start-time (task-network$start-time network))
        (setf finish-time (task-network$finish-time network))
        (setf local-delay-time (- finish-time due-date)) 
        SUM (- finish-time start-time) INTO total-actual-duration
        SUM (- due-date release-time) INTO total-desired-duration
        WHEN (not (= local-delay-time 0))
        COUNT local-delay-time INTO number-of-delays
        AND
        SUM local-delay-time INTO total-delay-time
        AND 
        COLLECT local-delay-time INTO delays
        FINALLY 
        (unless (zerop number-of-delays)
          (setf avg-delay (float (/ total-delay-time number-of-delays)))
          (setf std-dev-delay (LOOP FOR delay IN delays 
                                    SUM (expt (- delay avg-delay) 2) INTO sum-of-squares
                                    FINALLY (return (sqrt (float (/ sum-of-squares (1- number-of-delays)))))))
          (setf max-delay (apply #'max delays)))
        (return (values total-delay-time 
                        number-of-delays 
                        avg-delay
                        std-dev-delay
                        max-delay
                        (float (/ total-actual-duration total-desired-duration))))))

;;; ---------------------------------------------------------------------------
;;; CLIP Definitions
;;; ---------------------------------------------------------------------------

(clip::defclip airport ()
  "Return the Currently Loaded Airport"
  ()
  (arm::get-loaded-airport))

;;; ---------------------------------------------------------------------------

(clip::defclip timetable ()
  "Return the Currently Loaded timetable"
  ()
  (arm::get-loaded-timetable))

;;; ---------------------------------------------------------------------------

(clip:defclip delay ()
  "Super delay CLIP to calculate the various delay clips."
  (:components (total-delay number-of-delays average-delay std-dev-delay maximum-delay duration-ratio))
  (delays-information))

(clip:defclip total-delay (value))

(clip:defclip number-of-delays (value))

(clip:defclip average-delay (value))

(clip:defclip std-dev-delay (value))

(clip:defclip maximum-delay (value))

(clip:defclip duration-ratio (value))

;;; ---------------------------------------------------------------------------

(clip:defclip fragmentation ()
  "Super fragmentation CLIP to calculate the various fragmentation clips."
  (:components (total-servicing-time total-sig-frag-time total-setup-time total-travel-time
                                     ratio-of-frag-to-servicing ratio-of-setup-to-servicing 
                                     ratio-of-travel-to-servicing))
  (resource-fragmentation-information 'arm:baggage-truck-planner-unit))

(clip:defclip total-servicing-time (value))

(clip:defclip total-sig-frag-time (value))

(clip:defclip total-setup-time (value))

(clip:defclip total-travel-time (value))

(clip:defclip ratio-of-frag-to-servicing (value))

(clip:defclip ratio-of-setup-to-servicing (value))

(clip:defclip ratio-of-travel-to-servicing (value))

;;; ---------------------------------------------------------------------------
;;; Experiments
;;; ---------------------------------------------------------------------------

(clip:define-experiment limit-resources-numerically-over-rating-schemes 
                        (&key (rating-schemes *defined-rating-schemes*) 
                              (airport-delimiter 10)
                              (airports user::*10-FTS-AIRPORTS*)
                              (timetable-name :20-flights))
  "Using the various rating schemes, constrain the resources and report the delays."
  :locals ((report-filename nil))
  :start-system arm:run-arm
  :ivs ((airport-name in airports)
        (rating-scheme in rating-schemes))
  :instrumentation (clip:trial-number rating-scheme airport timetable total-delay number-of-delays 
                                      average-delay std-dev-delay maximum-delay duration-ratio 
                                      total-servicing-time total-sig-frag-time total-setup-time
                                      total-travel-time ratio-of-frag-to-servicing 
                                      ratio-of-setup-to-servicing ratio-of-travel-to-servicing)
  :before-experiment (progn (arm:zack-set-default-demo-parameters)
                            (arm:execute-load-timetable :LOAD-SPECIFIC-FILE timetable-name))
  :before-trial (setf report-filename
                      (delay-initialize-trial rating-scheme airport-delimiter airport-name))
  :after-trial (delay-report report-filename airport-delimiter rating-scheme))

;;; ---------------------------------------------------------------------------
;;; Example Invocation
;;; ---------------------------------------------------------------------------

#+COMMENT
(clip:run-experiment 'dss:limit-resources-numerically-over-rating-schemes
                      :args `(:rating-schemes (:zack-1 :default :obo-minest-heuristic)
                              :airport-delimiter :20-ref
                              :airports (:detroit-17-bts :detroit-16-bts :detroit-15-bts :detroit-14-bts
                                         :detroit-13-bts :detroit-12-bts)
                              :timetable-name :20-flights)
	              :output-file "hillary:rubinstein.dss-arm.reports;20-REF-FTS-DELAY-STATS.TEXT")

;;; ---------------------------------------------------------------------------
;;; Experiments
;;; ---------------------------------------------------------------------------

(clip:define-experiment run-over-rating-schemes 
                        (&key (rating-schemes *defined-rating-schemes*) 
                              (airport-delimiter 10))
  "Using the various rating schemes, constrain the resources, introduce orders, and report the delays."
  :locals ((report-filename nil))
  :start-system arm:run-arm
  :ivs ((rating-scheme in rating-schemes))
  :instrumentation (clip:trial-number rating-scheme airport timetable total-delay number-of-delays 
                                      average-delay std-dev-delay maximum-delay duration-ratio 
                                      total-servicing-time total-sig-frag-time total-setup-time
                                      total-travel-time ratio-of-frag-to-servicing 
                                      ratio-of-setup-to-servicing ratio-of-travel-to-servicing)
  :before-trial (setf report-filename
                      (delay-initialize-trial rating-scheme airport-delimiter))
  :after-trial (delay-report report-filename airport-delimiter rating-scheme))

;;; ---------------------------------------------------------------------------
;;; Example Invocation
;;; ---------------------------------------------------------------------------

#+COMMENT
(clip:run-experiment 'dss:run-over-rating-schemes
                     :ARGS `(:RATING-SCHEMES
                              ,(cons :DEFAULT (cons :ZACK-1
                                                    (remove :OPPORTUNISTIC
                                                            *defined-rating-schemes*
                                                            :TEST #'eq)))
                              :airport-delimiter 10)
                     :OUTPUT-FILE "hillary:rubinstein.dss-arm.reports;HILDUM-EXP-1.TEXT")

;;; ---------------------------------------------------------------------------
;;; Table Partitioning Functions
;;; ---------------------------------------------------------------------------

(defun create-data-file-from-data-table (data-table table-type table-filename-root data-pos)
  (with-open-file (stream (make-pathname :name (format nil "~a-~a" (pathname-name table-filename-root)
                                                       table-type)
                                         :type "text"
                                         :defaults table-filename-root)
                          :direction :output)
    (let ((sub-partition-data-table (loop initially (format stream "\"Resource Max\"~c" #\TAB)
                                          for partition in (list-table:lt$table-list data-table) collect
                                          (format stream "\"~a\"~c" (first partition) #\TAB)
                                          (second partition))))
      (terpri stream)
      (apply #'mapc #'(lambda (&rest data-entries)
                        (format stream "~s~c" (third (first data-entries)) #\TAB)
                        (loop for entry in data-entries do
                              (format stream "~s~c" (nth data-pos entry) #\TAB))
                        (terpri stream))
             sub-partition-data-table))))

;;; ---------------------------------------------------------------------------

(defun partition-by-rating-schemes (data-filename table-filename-root)
  (with-open-file (data-stream data-filename :direction :input)
    (loop with table = (list-table:make-list-table) 
          and data = ""
          initially (loop until (not (stringp (setf data (read data-stream nil :eof)))) do nil)
          while (not (eq data :eof)) do
          (list-table:add-list-table-value table (second data) data)
          (setf data (read data-stream nil :eof))
          finally 
          (create-data-file-from-data-table table :total table-filename-root 3)
          (create-data-file-from-data-table table :number table-filename-root 4)
          (create-data-file-from-data-table table :duration table-filename-root 5))))

;;; ***************************************************************************
;;; *                              End of File                                *
;;; ***************************************************************************
