;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File: styled-comments.lisp
;;; Author: Bob Kass, EDS Center for Advanced Research (kass@cmi.com)
;;; Date: 7/30/92
;;; 
;;; For MCL 2.0.
;;;
;;; This file is an extension similar to style-definitions.lisp by Derek White.
;;; It will format all the semicolon/Carriage Return delimited comments
;;; in a buffer using *comment-style*, with the exception of the modeline.
;;;
;;; This is nice to set all your comments in italics to help set them off 
;;; from the rest of your code. 
;;;
;;; Loading the file will add a "Styled Comments" entry to the Edit menu,
;;; and bind it to the command-I keystroke.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package :ccl)
(export '*comment-style*)

(defvar *comment-style* '(:italic) "Specify the character style to use for comments")

(defconstant *set-comment-menu-name* "Styled Comments"
  "The name of the 'Styled Comments' menu.")

(defmethod in-string-p ((b buffer-mark) position)
  "This function is something of a kludge to see whether <position> is in the middle of a 
   string.  First we find the beginning of the top-level sexp and then count the number of 
   quotemarks between the beginning and <position>.  An odd number of quotes implies we're
   in a string.  This will get confused by quotemarks that appear within a comment."
  (let ((sexp-start (ccl::ed-top-level-sexp-start-pos b position t))
        (quote-count 0)
        )
    (when sexp-start
      (loop for i from sexp-start to position
            if (and (eql (buffer-char b i) #\")
                    (> i 0)
                    (not (eql (buffer-char b (- i 1)) #\\)))    ; not a quote char
            do (incf quote-count)
            ))
    (oddp quote-count)
    ))

(defmethod set-comment-style ((w fred-window) font-spec)
  "Change all comments (starting with a leading ';') in the buffer for this window 
   to be displayed using *comment-style*"
  (let ((b (fred-buffer w))
        comment-begin
        comment-end
        )
    ;;; skip over the modeline if there is one -- like to keep it in a normal style
    (multiple-value-setq (comment-end comment-begin) (ccl::buffer-modeline-range b))
    (when (not comment-begin)
      (setf comment-begin 0)
      )
    (loop always (setf comment-begin (ccl::buffer-forward-search b #\; comment-begin))
          do 
          ;;; buffer-forward-search returns the position 1 passed the matching character,
          ;;; so we need to decrement by 1 to refer to the actual position of the match
          (unless (or (and (> comment-begin 1) 
                           (eql (buffer-char b (- comment-begin 2)) #\\))    ; #\; isn't really a comment
                      (in-string-p b (- comment-begin 1)))
            (setf comment-end (ccl::buffer-forward-search b #\return comment-begin))
            (buffer-set-font-spec b font-spec (- comment-begin 1) comment-end)
            (setf comment-begin comment-end))
          )
    (fred-update w)
    ))

(defun handle-set-comment-style (w)
  "Handle the menu invocation by calling set-comment-style and setting up the Undo/Redo menu."
  (set-comment-style w *comment-style*)
  (setup-undo w
              #'(lambda ()
                  (set-comment-style w :plain)
                  (setup-undo w
                              #'(lambda ()
                                  (handle-set-comment-style w))
                              "Redo Styled Comments"))
              "Undo Styled Comments"))

;;;
;;; Put an entry on the Edit menu
;;;
(add-menu-items *edit-menu*
                (make-instance 'menu-item
                  :menu-item-title *set-comment-menu-name*
                  :menu-item-action #'(lambda ()
                                        (handle-set-comment-style (front-window) ))
                  :command-key #\I
                  )
                )