;;;; Simple QuickTime movie player using the Movie Controller component
;;;; This code closely follows the samples in the QuickTime developer's guide.
;;;; It is not written to be directly reusable, rather as an example of what
;;;; can be done. The QuickTime Lisp interface files are necessary to compile
;;;; this.
;;;; *** REQUIRES 7.0.X ***
;;;; The 7.0.x dependency is in the choose-movie-file function. Different code
;;;; is needed for 6.0.x.
;;;; Use (play-movie) to choose and play a movie. You can play several movies at
;;;; the same time if your machine is fast enough.
;;;; Author: Daniel Ranson
;;;; This code is public domain. Use it any way you like, and at your own risk.

;;;; Is QuickTime available?
(defun quicktime-available-p ()
  (rlet ((response :pointer))
    (eql (#_Gestalt #$gestaltQuickTime response) #$noErr)))

;;;; Choose a movie file (7.0 only). Reply is a pointer to a StandardFileReply record
;;;; Return T if a file is chosen, else NIL
(defun choose-movie-dialog (reply)
  (rlet ((fileTypes :SFTypeList))
    (rset fileTypes (:SFTypeList.array 0) #$MovieFileType)
    (#_StandardGetFilePreview (%null-ptr) 1 fileTypes reply))
  (rref reply :StandardFileReply.sfGood))

;;;; Get a movie
(defun get-movie (w)
  (let ((err -1)(err1 0))
    (rlet ((reply :StandardFileReply)(resRefNum :signed-integer)(theMovie :pointer)
           (resID :signed-integer)(resName (:string 255))(changed :boolean))
      (when (choose-movie-dialog reply)
        (without-interrupts
         ;; This SetPort is very important. Movies somehow grab the current port
         ;; when they are created, and if you don't force the right port, you won't see anything.
         ;; The without-interrupts is for preventing MCL from changing the port again (e.g. to
         ;; update a window)
         (#_SetPort (wptr w))
         (setq err (#_OpenMovieFile (rref reply :StandardFileReply.sfFile) resRefNum #$fsRdPerm))
         (when (zerop err)
           (%put-word resID 0)
           (setq err (#_NewMovieFromFile theMovie (%get-word resRefNum) resID resName
                      #$newMovieActive changed))
           (setq err1 (#_CloseMovieFile (%get-word resRefNum)))
           (when (zerop err)
             (setq err err1)))))
      (when (zerop err)
        (%get-ptr theMovie)))))

;;;; A movie window class
(defclass movie-window
  (window)
  ((movie :initform nil :accessor movie-window-movie)
   (controller :initform nil :accessor movie-window-controller)
   (windows :initform nil :allocation :class :accessor movie-windows))
  (:default-initargs
    :view-position #@(100 100)
    :window-title "Movie"
    :window-show nil
    :color-p t
    :window-type :document
    :close-box-p t))

;;;; Initialize the window
(defmethod initialize-instance :after ((w movie-window) &key)
  ;; Initialize QT globals
  (#_EnterMovies)
  ;; Get a movie
  (let ((movie (get-movie w))(mc nil)(err 0))
    (when movie
      (setf (movie-window-movie w) movie)
      (rlet ((r :rect)(mcr :rect))
        ;; Get movie size (center it)
        (#_GetMovieBox movie r)
        (#_OffsetRect r (- (rref r :rect.left)) (- (rref r :rect.top)))
        (#_SetMovieBox movie r)
        ;; Get a controller
        (setq mc (#_NewMovieController movie (rref (wptr w) :grafport.portrect) #$mcTopLeftMovie))
        (cond ((%null-ptr-p mc) (setq err -1))
              (t
               (setf (movie-window-controller w) mc)
               ;; Add controller size to movie size
               (setq err (#_MCGetControllerBoundsRect mc mcr))
               (when (zerop err)
                 (#_UnionRect r mcr r)
                 ;; Set window size
                 (set-view-size w (- (rref r :rect.right) (rref r :rect.left))
                                (- (rref r :rect.bottom) (rref r :rect.top))))))))
    (unless (zerop err)
      (window-close w)
      (error "Sorry, cannot play movie."))
    (setq *eventhook* #'movie-event-hook)
    (push w (movie-windows w))
;    (#_MCSetVisible mc t)
    (window-show w)
    (window-select w)))

;;;; Clean up when closing window
(defmethod window-close :after ((w movie-window))
  (setf (movie-windows w) (remove w (movie-windows w)))
  (when (null (movie-windows w))
    (setq *eventhook* nil))
  (when (movie-window-controller w)
    (#_CloseComponent (movie-window-controller w)))
  (when (movie-window-movie w)
    (#_DisposeMovie (movie-window-movie w)))
  (#_ExitMovies))

;;;; A event hook function
(defun movie-event-hook ()
  (let ((ww (find-window "" 'movie-window))(event *current-event*))
    (when ww
      (dolist (w (movie-windows ww))
        (unless (zerop (#_MCIsPlayerEvent (movie-window-controller w) event))
          (return-from movie-event-hook t)))))
  nil)

(defun play-movie ()
  (unless (quicktime-available-p)
    (error "No QuickTime!"))
  (make-instance 'movie-window))
