;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

(defcommand-table Top-Level "IMAP Client top-level commands."
  "This is a command table that holds top-level commands for the IMAP client
mailer.  The sort of commands that you'll find in this table are those that
select messages, send, read BBoard and such-like."
)

;-------------------------------------------------------------------------------

(Defimmediate-command (mail-control-window :Beep-command)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Beep"
    :Description "Beeps according to yw:*beep-command-beep-type*."
   )
  (continuation (beep *beep-command-beep-type*))
)

(Defimmediate-command (mail-control-window :Notify)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Notify"
    :Description "Makes a notification according to its argument."
   )
  (let ((message (read-line *standard-input* t nil t)))
       (continuation (tv:notify tv:selected-window (string message)))
  )
)

(Defimmediate-command (mail-control-window :Edit-Rule)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Edit-Rule"
    :Description "Select a rule to edit."
    :Documentation '("Selects a rule to be edited")
   )
  (multiple-value-bind (rule-or-string found-rule-p)
      (read-rule *standard-input* nil)
    (continuation
      (if found-rule-p
	  (Edit-rule-with-menu
	    :Label "Edit Rule" :Rule-To-Edit rule-or-string
	  )
	  (if (y-or-n-p "There is no rule called ~A.  Create one?"
			rule-or-string
	      )
	      (Edit-rule-with-menu
		:Label "Create new rule" :Name rule-or-string
	      )
	      nil
	  )
      )
    )
  )
)

(Defimmediate-command (mail-control-window :Edit-Rule-Set)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Edit-Rule-Set"
    :Description "Select a rule set to edit."
    :Documentation '("Selects a rule set to be edited")
   )
  (multiple-value-bind (rule-set-or-string found-rule-set-p)
      (read-rule-set *standard-input* nil)
    (continuation
      (if found-rule-set-p
	  (edit-rule-set
	    :Label "Edit Rule Set" :Rule-Set-To-Edit rule-set-or-string
	  )
	  (if (y-or-n-p "There is no rule set called ~A.  Create one?"
			rule-set-or-string
	      )
	      (edit-rule-set
		:Label "Create new rule set"
		:Inits (list :Name rule-set-or-string)
	      )
	      nil
	  )
      )
    )
  )
)

(Defimmediate-command (mail-control-window :Invoke-Rule-Set)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Invoke-Rule-Set"
    :Description
      "Select a rule set and invoke it on the current mailbox/sequence."
    :Documentation '("Selects a rule set to be edited")
   )
  (multiple-value-bind (rule-set found-rule-set-p)
      (read-rule-set *standard-input* t)
    (continuation
     (if (and found-rule-set-p current-mailbox)
	 (let ((sequence (or current-sequence (simple-sequence :Sequence-All))))
	      (Send sequence :Map-Over-Messages
		    #'(lambda (seq message rule-set)
			(declare (optimize (speed 3) (safety 0)))
			(send rule-set :Apply-Self
			      (make-event :Mailbox (send seq :Mailbox)
					  :Message message
					  :Type :invoke
			      )
			      t
			)
		      )
		    rule-set
	      )
	 )
	 (yw-warn "There is no mailbox to invoke this rule set on.")
     )
    )
  )
)

(Defimmediate-command (mail-control-window :UnDigestify)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "UnDigestify"
    :Description "UnDigestifies a sequence of messages into separate windows."
    :Documentation
    '("UnDigestifies a sequence of messages into separate windows."
      *message-sequence-documentation*
     )
   )
  (send self :Apply-Method-To-Messages :UnDigestify)
)

(Defimmediate-command (mail-control-window :Digestify)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Digestify"
    :Description "Digestifies a sequence of messages into separate windows."
    :Documentation
    '("Digestifies a sequence of messages into separate windows."
      *message-sequence-documentation*
     )
   )
  (let ((sequence (send self :Read-Defaulted-Sequence)))
       (if (not sequence) (parse-error "~&No message sequence found.") nil)
       (continuation (send sequence :Digestify))
  )
)

(Defimmediate-command (mail-control-window :ReDo-Command)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "ReDo-Command"
    :Description "This command is not supported."
   )
  (parse-error "~&The ReDo-Command command is not supported.")
  (continuation nil)
)

(Defimmediate-command (mail-control-window :Parse-User-YW-Init-File)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Parse YW Init File"
    :Description "Open the yw init file on the mail server and parse it."
   )
  (continuation (parse-yw-init-file))
)

(Defimmediate-command (mail-control-window :Profile)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Profile"
    :Description "Enter the Profile tool to set useful variables."
   )
  (continuation
    (profile)
    (send tv:selected-window :Force-Kbd-Input
	 `(:Menu ,(assoc "YW"
			 (send (send tv:selected-window :Selection-Menu-Pane)
			       :Item-List
			 )
			 :Test #'equal
		  )
	   1 ,(send tv:selected-window :Selection-Menu-Pane)
	  )
    )
  )
)

(Defimmediate-command (mail-control-window :Read-Sequence)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Read"
    :Keys (#\M-R)
    :Description "Reads a sequence of messages."
    :Documentation '("Reads a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (let ((sequence (send self :Read-Defaulted-Sequence)))
       (if (not sequence) (parse-error "~&No message sequence found.") nil)
       (continuation (send self :read-type-command sequence :Read-Sequence))
  )
)

(defcommand-short-form (mail-control-window :Read-Sequence) "R")

(Defimmediate-command (mail-control-window :Reply)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Reply"
    :Description "Reply to a sequence of messages."
    :Documentation '("Replies a sequence of messages.  "
		     *message-sequence-documentation*
	            )
   )
  (declare (special *prompt-window*))
  (let ((*reply-to-all* *reply-to-all-by-default*)
	(*reply-to-be-inclusive* *reply-inclusive-by-default*)
       )
       (declare (special *reply-to-be-inclusive* *reply-to-all*))
       (with-these-command-tables ('(*reply-command-table*)) *prompt-window*
	 (let ((sequence (catch :Top-Level-Sequence
			   (Parse-a-message-sequence *standard-input*
		             (or current-sequence *default-message-sequence*)
			     nil
			   )
			 )
	       )
	       (all *reply-to-all*)
	       (incl *reply-to-be-inclusive*)
	      )
	      (if (not sequence)
		  (parse-error "~&No message sequence found.")
		  nil
	      )
	      (continuation
	        (send self :read-type-command sequence :Reply-To-Sequence
		      all incl
		)
	      )
	 )
       )
  )
)

(Defimmediate-command (mail-control-window :Forward)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Forward"
    :Description "Forward a sequence of messages."
    :Documentation '("Forwards a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (let ((sequence (Parse-a-message-sequence *standard-input* current-sequence)))
       (if (not sequence) (parse-error "~&No message sequence found.") nil)
       (continuation
	 (send self :read-type-command sequence :Forward-Sequence)
       )
  )
)

(Defimmediate-command (mail-control-window :Remail)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Remail"
    :Description "Remail a sequence of messages."
    :Documentation '("Remails a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (let ((sequence (Parse-a-message-sequence *standard-input* current-sequence)))
       (if (not sequence) (parse-error "~&No message sequence found.") nil)
       (continuation
	 (send self :read-type-command sequence :Remail-Sequence)
       )
  )
)

(Defimmediate-command (mail-control-window :Copy)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Copy"
    :Description "Copies a sequence of messages to a specified mailbox."
    :Documentation '("Copies a sequence of messages to another mailbox.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :move-sequence nil)
)

(Defimmediate-command (mail-control-window :Move)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Move"
    :Description "Moves a sequence of messages to a specified mailbox."
    :Documentation '("Moves a sequence of messages to another mailbox.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :move-sequence t)
)

(defcommand-short-form (mail-control-window :Move) "M")

(Defimmediate-command (mail-control-window :Delete)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Delete"
    :Keys (#\M-rubout)
    :Description "Deletes a sequence of messages."
    :Documentation '("Deletes a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :funcall-one-function-for-all-messages
      #'(lambda (sequence messages)
	  (flag/unflag-message
	    (send sequence :Mailstream) messages :Set :\\Deleted
	  )
	)
	nil
  )
)

(Defimmediate-command (mail-control-window :UnDelete)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "UnDelete"
    :Description "UnDeletes a sequence of messages."
    :Documentation '("UnDeletes a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :funcall-one-function-for-all-messages
      #'(lambda (sequence messages)
	  (flag/unflag-message
	    (send sequence :Mailstream) messages :Clear :\\Deleted
	  )
	)
	nil
  )
)

(Defimmediate-command (mail-control-window :Expunge)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Expunge"
    :Description "Expunges the current mailbox."
    :Documentation "Expunges the current mailbox."
   )
  (multiple-value-bind (mailbox string)
      (read-in-a-mailbox-name
	(and current-mailbox
	     (send (ucl:first-if-list current-mailbox) :mailbox)
	)
      )
    (if mailbox
	(continuation
	  (loop for box in (list-if-not mailbox)
		do (Expunge-MailBox box nil)
		   (mapcar
		     #'(lambda (sum)
			 (if (member box (send sum :Mailstreams) :Test #'eq)
			     (progn
			       (send sum :Mailbox-Expunged box)
			       (send Mailbox-Selector :Update-Label-For sum)
			     )
			     nil
			 )
		       )
		     all-summary-windows
		   )
	  )
	  (send self :Set-Current-Sequence nil)
	)
	(if (equal string "")
	    (parse-error "~&No current mailbox selected.")
	    (parse-error "~&~S is not the name of a mailbox." string)
	)
    )
  )
)

(defcommand-short-form (mail-control-window :Expunge) "Exp")

(Defimmediate-command (mail-control-window :Set-Keyword)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Keyword"
    :Description "Set a keyword for a sequence."
    :Documentation '("Sets a keyword for a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (let ((keyword (read-keyword *standard-input* current-mailbox t)))
       (send self :Apply-Method-To-Messages
	   #'(lambda (sequence number)
	       (flag/unflag-message
		 (send sequence :Mailstream) number :Set keyword
	       )
	     )
	     nil
       )
  )
)

(Defimmediate-command (mail-control-window :Set-UnKeyword)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "UnKeyword"
    :Description "UnSet a keyword for a sequence."
    :Documentation '("UnSets a keyword for a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (let ((keyword (read-keyword *standard-input* current-mailbox)))
       (send self :Apply-Method-To-Messages
	   #'(lambda (sequence number)
	       (flag/unflag-message
		 (send sequence :Mailstream) number :Clear keyword
	       )
	     )
	     nil
       )
  )
)

(Defimmediate-command (mail-control-window :Flag)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Flag"
    :Description "Flags a sequence of messages."
    :Documentation '("Flags a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :Apply-Method-To-Messages
      #'(lambda (sequence number)
	  (flag/unflag-message
	    (send sequence :Mailstream) number :Set :\\Flagged
	  )
	)
	nil
  )
)

(Defimmediate-command (mail-control-window :Unflag)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Unflag"
    :Description "Unflags a sequence of messages."
    :Documentation '("Unflags a sequence of messages.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :Apply-Method-To-Messages
      #'(lambda (sequence number)
	  (flag/unflag-message
	    (send sequence :Mailstream) number :Clear :\\Flagged
	  )
	)
	nil
  )
)

(Defimmediate-command (mail-control-window :Mark)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Mark"
    :Description "Marks a sequence of messages as seen."
    :Documentation '("Marks a sequence of messages as seen.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :funcall-one-function-for-all-messages
      #'(lambda (sequence messages)
	  (flag/unflag-message
	    (send sequence :Mailstream) messages :Set :\\Seen
	  )
	)
	nil
  )
)

(Defimmediate-command (mail-control-window :UnMark)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "UnMark"
    :Description "Marks a sequence of messages as unseen."
    :Documentation '("Marks a sequence of messages as unseen.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :funcall-one-function-for-all-messages
      #'(lambda (sequence messages)
	  (flag/unflag-message
	    (send sequence :Mailstream) messages :Clear :\\Seen
	  )
	)
	nil
  )
)

(Defimmediate-command (mail-control-window :UnAnswer)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "UnAnswer"
    :Description "Marks a sequence of messages as unanswered."
    :Documentation '("Marks a sequence of messages as unanswered.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :funcall-one-function-for-all-messages
      #'(lambda (sequence messages)
	  (flag/unflag-message
	    (send sequence :Mailstream) messages :Clear :\\Answered
	  )
	)
	nil
  )
)

(Defimmediate-command (mail-control-window :MultiHeaders)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "MultiHeaders"
    :Description "Selects a sequence of headers from multiple mailboxes."
    :Documentation
    '("Selects a sequence of message headers into a new summary fro mmultiple
       mailboxes.  "
      *message-sequence-documentation*
     )
   )
  (send self :select-multiheaders)
)

(Defimmediate-command (mail-control-window :Headers)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Headers"
    :Keys (#\M-H)
    :Description "Selects a sequence of headers."
    :Documentation
    '("Selects a sequence of message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (send self :select-headers)
)

(defcommand-short-form (mail-control-window :Headers) "H")

(Defimmediate-command (mail-control-window :Filter)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Filter"
    :Description "Selects a sequence of headers."
    :Documentation
    '("Filters the current sequence using a new sequence into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (if (typep current-sequence 'message-sequence)
      (send self :Select-Headers current-sequence)
      (parse-error "~&There is no current sequence so you cannot filter.")
  )
)

;;;Edited by Tom Gruber            8 Jan 92  13:06
;;;Edited by Tom Gruber            8 Jan 92  13:10
(defimmediate-command (mail-control-window :Close-Mailbox)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Close"
    :Description "Closes a mailbox."
   )
  (multiple-value-bind (mailboxes string)
      (read-in-a-mailbox-name
	(and current-mailbox (send (ucl:first-if-list current-mailbox) :Mailbox))
      )
    (continuation
      (if mailboxes
	  (with-daemons-reset-and-arrested ()
            (loop for box in (list-if-not mailboxes)
                  do (send box :Close)
            )
          )
	  (if current-mailbox
	      (Format-scroll-window self "~&~S is not the name of a mailbox"
				    string
	      )
	      (Format-scroll-window self "~&There is no current mailbox")
	  )
      )
      (send self :Get-Current-Summary-Window)
    )
  )
)

(defimmediate-command (mail-control-window :Get)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Get"
    :Description
    "Gets a possibly new mailbox optionally filtering with a sequence."
   )
  (let ((string
	  (string-trim *whitespace-chars*
		       (Read-Mailbox-Name *standard-input* t)
	  )
	)
	(sequence (let ((*make-sequence-with-no-mailbox-ok-p* t))
		       (parse-a-message-sequence *standard-input* nil)
		  )
	)
       )
       (continuation (send self :get-1 string nil sequence))
  )
)

(defimmediate-command (mail-control-window :Examine)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Examine"
    :Description "Examines a possibly new mailbox."
   )
  (let ((string
	  (string-trim *whitespace-chars*
		       (Read-Mailbox-Name *standard-input* t)
	  )
	)
	(sequence (let ((*make-sequence-with-no-mailbox-ok-p* t))
		       (parse-a-message-sequence *standard-input* nil)
		  )
	)
       )
       (continuation (send self :get-1 string t sequence))
  )
)

(defimmediate-command (mail-control-window :NNTP)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "News"
    :Description
    "Gets a possibly new NNTP newsgroup."
   )
  (let ((string
	  (string-trim
	    *whitespace-chars*
	    (Read-Mailbox-Name *standard-input* t
			       (first *netnews-search-list*) nil nil
			       'complete-newsgroup
	    )
	  )
	)
	(sequence (let ((*make-sequence-with-no-mailbox-ok-p* t))
		       (parse-a-message-sequence *standard-input* nil)
		  )
	)
       )
       (continuation
	 (send self :Get-1
	       (string-append *Site-Specific-Nntp-Server-Host* ":" String)
	       nil sequence :Nntp *site-specific-nntp-server-host*
	 )
	 (setq current-bboard string)
       )
  )
)

(defnonimmediate-command (mail-control-window :Check)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Check"
    :Description "Checks the current mailbox for new mail."
   )
  (if current-mailbox
      (continuation
	(loop for box in (list-if-not current-mailbox)
	      do (MAP-Check-MailBox box)
	)
      )
      (barf "~&There is no current mailbox.")
  )
)

(Defimmediate-command (mail-control-window :Hardcopy)
		      () (*All-Top-Level-Command-Names*)
  `(:Names ("Hardcopy" "Print" "List")
    :Description "Prints a sequence of messages on some printer."
    :Documentation '("Prints a sequence of messages on some printer.  "
		     *message-sequence-documentation*
		    )
   )
  (send self :hardcopy-sequence)
)

(defimmediate-command (mail-control-window :BBoard)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "BBoard"
    :Description "Gets a possibly new BBoard mailbox."
   )
  (let ((path (make-pathname :Host *bboard-source-host*
			     :Directory *bboard-source-directory*
			     :name *default-bboard*
			     :type *default-bboard-file-type*
			     :Version :Newest
	      )
	)
	(*user-host* *bboard-source-host*)
       )
       (let ((string (string-trim
		       *whitespace-chars*
		       (Read-Mailbox-Name *standard-input* t path path
					  *default-bboard-file-type*
		       )
		     )
	     )
	    )
	    (if (equal "" string)
		(parse-error "No bboard name found.")
	        (continuation
		  (let ((*make-imap-stream-read-only* t))
		       (declare (special *make-imap-stream-read-only*))
		       (send self :Get-1
			     (send self :Turn-Into-Bboard-Path string)
			     t (send self :Get-Bboard-Filter string)
		       )
		       (setq current-bboard string)
		  )
		)
	    )
       )
  )
)

(Defnonimmediate-command (mail-control-window :Find)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Find"
    :Description "Finds the next bboard with new mail."
   )
  (continuation (send self :Find-1))
)

(Defnonimmediate-command (mail-control-window :Ignore)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Ignore"
    :Description "Ignores all new messages in the current bboard and finds the next bboard with new mail."
   )
  (continuation
    (if current-bboard
	(let ((*make-imap-stream-read-only* t))
	     (declare (special *make-imap-stream-read-only*))
	     (Send self :Find-Or-Open-Mailbox
		   (send self :Turn-Into-Bboard-Path current-bboard)
	     )
	     (send self :mark-messages-as-seen current-mailbox)
	)
	nil
    )
    (send self :Find)
  )
)

(Defnonimmediate-command (mail-control-window :Send)
		      () (*All-Top-Level-Command-Names*)
   `(:Names ("Send" "Mail")
    :Description "Compose a message in zmacs and send it."
   )
  (continuation
    (multiple-value-bind (frame zwei:*window*) (find-zmacs-frame)
      (declare (special zwei:*window*))
      (Inside-Zmacs (frame)
	(let ((zwei:*unsent-message-query-p* *unsent-message-query-p*)
	      (old #'zwei:maybe-continue-unsent-message)
	     )
	     (letf ((#'zwei:maybe-continue-unsent-message
		     #'(lambda (&optional arg)
			 (let ((*query-io* (send *mailer* :Prompt-Window)))
			      (send *query-io* :Select)
			      (letf ((#'y-or-n-p #'yw-y-or-n-p))
				    (funcall old arg)
			      )
			 )
		       )
		    )
		   )
		   (send prompt-window :Clear-Input)
		   (zwei:com-mail)
	     )
	)
      )
      (send frame :Select)
      (send frame :Refresh)
      (inside-zmacs (frame)
	(if (not (get zwei:*interval* :source-mailer))
	    ;;; We might be selecting a preexisting buffer, which might either
	    ;;; already have a source mailer or might have been closed in which
	    ;;; case we don't want a source mailer.
	    (setf (get zwei:*interval* :source-mailer)
		  *mailer*
	    )
	    nil
	)
      )
    )
  )
)

(defcommand-short-form (mail-control-window :Send) "S")

(Defnonimmediate-command (mail-control-window :Expose-Summary-Window)
	    () (*All-Top-Level-Command-Names*)
  `(:Keys #\C-Sh-E
    :Description "Exposes the current summary window."
   )
  (continuation
    (if current-mailbox
	(if current-summary-window
	    (send current-summary-window :expose)
	    (Format-scroll-window
	      self "No header windows for mailbox ~S.~&~
		    Use the HEADERS command to make one."
	      (print-short-mailbox-name current-mailbox)
	    )
	)
	(mail-beep)
    )
  )
)

(Defnonimmediate-command (mail-control-window :Expose-All-Summary-Windows)
	    () (*All-Top-Level-Command-Names*)
  `(:Keys #\M-Sh-E
    :Description "Exposes all of the summary windows."
   )
  (continuation
    (let ((frame self))
	 (Expose-In-Order 'Summary-Windows
			  #'(lambda (win)
			      (member win (send frame :All-Summary-Windows)
				      :Test #'eq
			      )
			    )
	 )
    )
  )
)

(Defnonimmediate-command (mail-control-window :Page-Down)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Page Down"
    :Keys (#\C-V #\C-)
    :Description "Scrolls the current window down a screen full."
   )
  (continuation
    (if current-mailbox
	(send (send self :Get-current-summary-window) :page-down)
	(mail-beep)
    )
  )
)

(Defnonimmediate-command (mail-control-window :Page-Up)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Page Up"
    :Keys (#\M-V #\C-)
    :Description "Scrolls the current window back up a screen full."
   )
  (continuation
    (if current-mailbox
	(send (send self :Get-current-summary-window) :page-up)
	(mail-beep)
    )
  )
)

(Defnonimmediate-command (mail-control-window :Line-Down)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Line Down"
    :Keys (#\M-N #\C-Z)
    :Description "Scrolls the current window down a line."
   )
  (continuation
    (if current-mailbox
	(send (send self :Get-current-summary-window) :line-down)
	(mail-beep)
    )
  )
)

(Defnonimmediate-command (mail-control-window :Line-Up)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Line Up"
    :Keys (#\M-P #\M-Z)
    :Description "Scrolls the current window back up a screen full."
   )
  (continuation
    (if current-mailbox
	(send (send self :Get-current-summary-window) :line-up)
	(mail-beep)
    )
  )
)

(Defnonimmediate-command (mail-control-window :Goto-Beginning)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Goto Beginning"
    :Keys (#\M- #\S-)
    :Description "Scrolls the current window to the top."
   )
  (continuation
    (if current-mailbox
	(send (send self :Get-current-summary-window) :Goto-Beginning)
	(mail-beep)
    )
  )
)

(Defnonimmediate-command (mail-control-window :Goto-End)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Goto End"
    :Keys (#\M- #\S-)
    :Description "Scrolls the current window to the bottom."
   )
  (continuation
    (if current-mailbox
	(send (send self :Get-current-summary-window) :Goto-End)
	(mail-beep)
    )
  )
)

(Defimmediate-command (mail-control-window :Copy-Headers-In-Window)
		      () (*All-Top-Level-Command-Names*)
  `(:Keys #\M-Sh-W
    :Description
    "Copy the headers in the current headers window into the kill ring."
   )
  (if current-summary-window
      (let ((top (send current-summary-window :top-message-on-screen))
	    (bottom (send current-summary-window :bottom-message-on-screen))
	   )
	   (continuation
	     (maybe-preempt-envelopes current-mailbox top (- bottom top))
	     (zwei:kill-string
	       (apply #'string-append
		      (apply #'append
			     (loop for i from top to bottom
				   collect
				   (list (without-tabs-1
					   (cache-header-display-string
					     (cache-entry-of
					       i current-mailbox
					     )
					   )
					   0
					 )
					 #\newline
				   )
			     )
		      )
	       )
	     )
	   )
      )
      (barf "~&No current summary window.")
  )
)

(Defimmediate-command (mail-control-window :I-Search)
		      () (*All-Top-Level-Command-Names*)
  `(:Keys #\C-S
    :Description "I-Search through headers."
   )
  (if current-summary-window
      (i-search-1
	(if (send current-summary-window :Filter)
	    1
	    (send current-summary-window :Top-Message-On-Screen)
	)
	nil 'extract-and-match-header
      )
      (barf "~&No current summary window.")
  )
  (continuation nil)
)

(Defimmediate-command (mail-control-window :Reverse-I-Search)
		      () (*All-Top-Level-Command-Names*)
  `(:Keys #\C-R
    :Description "Reverse I-Search through headers."
   )
  (if current-summary-window
      (i-search-1
	(if (send current-summary-window :Filter)
	    (send current-mailbox :Messagecnt)
	    (send current-summary-window :bottom-message-on-screen)
	)
	t 'extract-and-match-header
      )
      (barf "~&No current summary window.")
  )
  (continuation nil)
)

(Defimmediate-command (mail-control-window :I-Search-Body)
		      () (*All-Top-Level-Command-Names*)
  `(:Keys #\C-M-S
    :Description "I-Search through all of messages."
   )
  (if current-summary-window
      (i-search-1 (send current-summary-window :top-message-on-screen)
		  nil 'extract-and-match-text
      )
      (barf "~&No current summary window.")
  )
  (continuation nil)
)

(Defimmediate-command (mail-control-window :Reverse-I-Search-Body)
		      () (*All-Top-Level-Command-Names*)
  `(:Keys #\C-M-R
    :Description "Reverse I-Search through all of messages."
   )
  (if current-summary-window
      (i-search-1
	(send current-summary-window :bottom-message-on-screen)
	t 'extract-and-match-text
      )
      (barf "~&No current summary window.")
  )
  (continuation nil)
)

(Defnonimmediate-command (mail-control-window :C-X-in-Zmacs)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "C-X in Zmacs"
    :Keys #\C-X
    :Description "Jumps you into Zmacs and types C-X again, since you think you ought to be there."
   )
  (tv:kbd-sys-1 #\E)
  (inside-zmacs ((find-frame tv:selected-window 'zwei:zmacs-frame))
    (send *standard-input* :Force-Kbd-Input #\C-X)
  )
  (continuation nil)
)

(Defnonimmediate-command (mail-control-window :M-X-in-Zmacs)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "M-X in Zmacs"
    :Keys #\M-X
    :Description "Jumps you into Zmacs and types M-X again, since you think you ought to be there."
   )
  (tv:kbd-sys-1 #\E)
  (inside-zmacs ((find-frame tv:selected-window 'zwei:zmacs-frame))
    (send *standard-input* :Force-Kbd-Input #\M-X)
  )
  (continuation nil)
)

(Defnonimmediate-command (mail-control-window :Reset)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Reset"
    :Description "Reset the mailer and all daemons."
   )
  (continuation (Reset-Daemons))
)

(Defnonimmediate-command (mail-control-window :RReset)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "RReset"
    :Description "Reset the mailer, all daemons and TCP."
   )
  (continuation
    (warm-reset)
    (ip:reset t)
    (mail-beep)
  )
)

(Defnonimmediate-command (mail-control-window :Bury-mailer)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Bury"
    :Keys #\
    :Description "Buries the mail window."
   )
  (if (fboundp 'w:shrink-window) (funcall 'w:shrink-window self) nil)
  (send self :Bury)
  (send self :Quit)
  (continuation nil)
)

(Defnonimmediate-command (mail-control-window :Exit)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Exit"
    :Description
    "Expunges and breaks all mail connections and cleans up windows."
   )
   (continuation
     (mapcar #'(lambda (mbx)
		 (if (send mbx :Open-P)
		     (expunge-mailbox mbx t)
		     nil
		 )
	       )
	       all-mailboxes
     )
     (send self :Quit-Mailer-Internal)
   )
)

;;;Edited by Tom Gruber            18 Feb 92  10:35
(Defnonimmediate-command (mail-control-window :Refresh-summaries)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Refresh"
    :Description "Refreshes the summary windows."
   )
  (continuation
    (mapcar #'(lambda (win)
		(send win :Clear-Window)
		(send win :fully-flush-display-cache)
                (send win :refresh)
	      )
	      all-summary-windows
    )
  )
)

(Defnonimmediate-command (mail-control-window :Quit-mailer)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Quit"
    :Description "Breaks all mail connections and cleans up windows."
   )
  (continuation (send self :Quit-Mailer-Internal))
)

(Defimmediate-command (mail-control-window :What)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "What"
    :Description "Expands an alias."
   )
  (let ((name
	  (string-upcase
	   (string-trim *whitespace-chars* (read-line *standard-input* t nil t))
	  )
	)
       )
       (if (stringp name)
	   (continuation
	     (send self :ensure-yw-init-file-parsed)
	     (if (find-symbol name 'YW-Variables)
		 (Format-scroll-window self
		   "~&~A -> ~&~A"
		   name (symbol-value (find-symbol name 'YW-Variables))
		 )
		 (Barf "~S was not defined in your .mminit file" name)
	     )
	   )
	   (parse-error "~S is not a valid name" name)
       )
  )
)

(Defimmediate-command (mail-control-window :Who)
	    () (*All-Top-Level-Command-Names*)
  `(:Names "Who"
    :Description "Expands an alias."
   )
  (let ((name
	  (string-upcase
	   (string-trim *whitespace-chars* (read-line *standard-input* t nil t))
	  )
	)
       )
       (cond ((not (stringp name)) (parse-error "~S is not a valid name" name))
	     ((not current-mailbox) (parse-error "No current mailbox"))
	     (t (multiple-value-bind (ignore host)
		    (mailbox-and-host-from-mailbox-name
			   (send current-mailbox :Mailbox)
		    )
		  (continuation
		    (with-more-p-enabled (prompt-window)
		      (Format-scroll-window self "~%~A~%" (mail:who host name))
		    )
		  )
	        )
	     )
       )
  )
)

(defimmediate-command (mail-control-window :Finger)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Finger"
    :Description "Finger a user on a specified host."
   )
  (let ((string
	  (string-trim
	    *whitespace-chars*
	    (read-string-maybe-with-blip
	      *standard-input* nil nil
	      #'(lambda (stream numbers)
		  (let ((entry (cache-entry-of (first numbers) stream)))
		       (if entry
			   (progn (maybe-preempt-envelopes stream numbers)
				  (let ((from
					  (envelope-from (cache-envelope entry))
					)
				       )
				       (format nil "~A@~A"
					       (address-mailbox (first from))
					       (address-host (first from))
				       )
				  )
			   )
			   ""
		       )
		  )
		)
	    )
	  )
	)
       )
       (let ((defaulted
	       (if (search "@" (the string string) :Test #'char-equal)
		   string
		   (string-append
		     string "@"
		     (if current-mailbox
			 (multiple-value-bind (ignore host)
			     (mailbox-and-host-from-mailbox-name
			       (send (ucl:first-if-list current-mailbox)
				     :Mailbox
			       )
			     )
			   host
			 )
			 (parse-error "No host to default to for ~S" string)
		     )
		   )
	       )
	     )
	    )
	    (continuation
	      (let ((result (with-more-p-enabled (prompt-window)
			      (ip:finger defaulted prompt-window t)
			    )
		    )
		   )
		   (if (typep result 'error)
		       (send result :Print-Error-Message
			     sys:%current-stack-group
			     t prompt-window
		       )
		       nil
		   )
	      )
	    )
       )
  )
)

(Defimmediate-command (mail-control-window :Sort)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Sort"
    :Description "Sorts the speciified or current mailbox."
    :Documentation "Sorts the specified or current mailbox."
   )
  (multiple-value-bind (mailbox string)
      (read-in-a-mailbox-name
	(and current-mailbox (send current-mailbox :Mailbox-name))
      )
    (if mailbox
	(if (feature-enabled-p :Renumber mailbox)
	    (continuation
	      (Renumber-MailBox mailbox :Date :Date)
	      (send self :Set-Current-Sequence nil)
	    )
	    (parse-error
	      "~&Sorry, the current server version does not support sorting."
	    )
	)
	(if (equal string "")
	    (parse-error "~&No current mailbox selected.")
	    (parse-error "~&~S is not the name of a mailbox." string)
	)
    )
  )
)


(Defimmediate-command (mail-control-window :Renumber)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Renumber"
    :Description "Renumbers the specified mailbox according to the appropriate
 field and the renumber-type.  Syntax is:
Renumber <mailbox> <field-name> [Alphabetic | Number | Date]." 
    :Documentation "Renumbers the current mailbox."
   )
  (multiple-value-bind (mailbox string)
      (read-in-a-mailbox-name
	(and current-mailbox (send current-mailbox :Mailbox-name))
      )
    (if mailbox
	(if (feature-enabled-p :Renumber mailbox)
	    (let ((field-name
		    (intern (string-upcase (read-string *standard-input*))
			    'keyword
		    )
		  )
		  (renumber-type
		    (read-symbol-from-alist
		      *standard-input*
		      '((:Alpha  "Alphabetic")
			(:Number "Number")
			(:Date   "Date")
		       )
		    )
		  )
		 )
		 (continuation
		   (Renumber-MailBox mailbox renumber-type field-name)
		   (send self :Set-Current-Sequence nil)
		 )
	    )
	    (parse-error
	     "~&Sorry, the current server version does not support renumbering."
	    )
	)
	(if (equal string "")
	    (parse-error "~&No current mailbox selected.")
	    (parse-error "~&~S is not the name of a mailbox." string)
	)
    )
  )
)


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------
;;; Boolean keywords.

(defcommand-table operator "IMAP Client operator commands."
  "This is a command table that holds operator commands for the IMAP
 client mailer.  The sort of commands that you'll find in this table are things
 like From, To, Subject..."
)

;-------------------------------------------------------------------------------

(defmethod (mail-control-window :diadic-operator-internal) (method-name)
  (declare (special *first-argument*))
  (let ((second-argument
	  (parse-a-message-sequence *standard-input*
				    *default-message-sequence*
	  )
	)
       )
       (if (not second-argument)
	   (parse-error "~&No message sequence found.")
	   nil
       )
       (throw :Sequence
	      (send self :Make-Operator-Sequence
		    method-name *first-argument* second-argument
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Or)
		      () (*All-Operator-Command-Names*)
  `(:Names "Or"
    :Description "Selects a disjunction of a pair of sequences of headers ...."
    :Documentation
    '("Selects a disjunction of a pair of sequences of message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (send self :diadic-operator-internal :Sequence-Or)
)

(Defimmediate-command (mail-control-window :XOr)
		      () (*All-Operator-Command-Names*)
  `(:Names "XOr"
    :Description "Selects the XOr of a pair of sequences of headers ...."
    :Documentation
    '("Selects a disjunction of a pair of sequences of message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (send self :diadic-operator-internal :Sequence-Xor)
)

(Defimmediate-command (mail-control-window :And)
		      () (*All-Operator-Command-Names*)
  `(:Names "And"
    :Description "Selects a conjunction of a pair of sequences of headers ...."
    :Documentation
    '("Selects a conjunction of a pair of sequences of message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (send self :diadic-operator-internal :Sequence-And)
)

(Defimmediate-command (mail-control-window :Then)
		      () (*All-Operator-Command-Names*)
  `(:Names "Then"
    :Description "Selects a disjunction of a pair of sequences of headers, such that one subsequence is processed before the other ...."
    :Documentation
    '("Selects a disjunction of a pair of sequences of headers, such that one subsequence is processed before the other into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (send self :diadic-operator-internal :Sequence-Then)
)

(Defimmediate-command (mail-control-window :Sort-by)
		      () (*All-Operator-Command-Names*)
  `(:Names "Sort by"
    :Description
    "Sorts a sequence of headers using a user defined predicate...."
    :Documentation
    '("Sorts a sequence of message headers using a user defined predicate.  "
      *message-sequence-documentation*
     )
   )
  (declare (special *first-argument* *prompt-window*))
  (let ((seq (let ((key (catch :Sort
			  (read-and-execute-command-from-different-tables
			    *standard-input* '(*sort-command-table*)
			  )
			)
		   )
		  )
		  (send self :Make-Operator-Sequence
			:Sequence-Sorted-by *first-argument* key
		  )
	     )
	)
       )
       (throw :Sequence seq)
  )
)

;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

(defcommand-table reply "IMAP Client reply subcommands."
  "This is a command table that holds subcommands to the reply command."
)

(Defimmediate-command (mail-control-window :Reply-to-All)
		      () (*All-Reply-Command-Names*)
  `(:Names "Everyone"
    :Description "Makes the reply go to all recipients of the original message."
   )
   (declare (special *reply-to-all*))
   (setq *reply-to-all* t)
   (throw :Top-Level-Sequence
	  (Parse-a-message-sequence *standard-input* current-sequence nil)
   )
)

(Defimmediate-command (mail-control-window :Reply-to-Sender)
		      () (*All-Reply-Command-Names*)
  `(:Names "Sender"
    :Description
      "Makes the reply go only to the sender of the original message."
   )
   (declare (special *reply-to-all*))
   (setq *reply-to-all* nil)
   (throw :Top-Level-Sequence
	  (Parse-a-message-sequence *standard-input* current-sequence nil)
   )
)

(Defimmediate-command (mail-control-window :Inclusive)
		      () (*All-Reply-Command-Names*)
  `(:Names "Inclusive"
    :Description "Makes the reply include the original message."
   )
   (declare (special *reply-to-be-inclusive*))
   (setq *reply-to-be-inclusive* t)
   (throw :Top-Level-Sequence
	  (Parse-a-message-sequence *standard-input* current-sequence nil)
   )
)

(Defimmediate-command (mail-control-window :Exclusive)
		      () (*All-Reply-Command-Names*)
  `(:Names "Exclusive"
    :Description "Makes the reply exclude the original message."
   )
   (declare (special *reply-to-be-inclusive*))
   (setq *reply-to-be-inclusive* nil)
   (throw :Top-Level-Sequence
	  (Parse-a-message-sequence *standard-input* current-sequence nil)
   )
)

(Defimmediate-command (mail-control-window :Test)
		      () (*All-Top-Level-Command-Names*)
  `(:Names "Test"
    :Description "Debug Command for developer use only."
   )
  (let ((sequence (Parse-a-message-sequence *standard-input*
					    *default-message-sequence*
		  )
	)
       )
       (continuation (print sequence))
  )
)

;-------------------------------------------------------------------------------
