;;; -*- Mode:Common-Lisp; Package:IMAP; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

(defun wait-for-file-lock-free (file)
"Process waits until the file lock for the FILE is free and then returns."
  (let ((path (fs:default-pathname file)))
       (if (equal (pathname-host (translate-pathname path)) si:local-host)
	   (let ((file (fs:lookup-file (pathname-directory path)
				       (pathname-name path)
				       (pathname-type path)
				       (pathname-version path)
		       )
		 )
		)
	        (process-wait "Wait for file lock"
			      #'(lambda (file) (not (fs:file-lock file))) file
                )
	   )
	   t
       )
  )
)

(defun compute-mail-file-format (file-name)
"A bit of a no-op right now, since we only support YW format mail files."
  (ignore file-name)
  :MM
)

(defun default-mailbox-path ()
  "Returns the pathname of the default mailbox."
  (declare (special yw:*default-mailbox-name*))
  (second (assoc yw:*default-mailbox-name*
		 *all-mailbox-name-translations*
		 :Test #'equal
	  )
  )
)

(defmethod parse-mail-file (file-name (type (eql :MM)))
"Parses a mail file named file-name, which is known to be in MM format.
Returns a MAIL-FILE object that represents the parsed mail file.
"
  (declare (values parsed-mail-file))
  (let ((defaulted-file-name
	  (fs:merge-pathnames file-name (Default-Mailbox-Path))
	)
       )
       (let ((file (make-mail-file :Name defaulted-file-name :Format type))
	     (length (file-length defaulted-file-name))
	    )
	    (parse-mail-file-1 file length type)
	    file
       )
  )
)

(defmethod parse-mail-file-1 (file length (type (eql :MM)))
"An internal method for parsing MM mail files.  It is passed a file object and
the length that we have yet to parse (this gets called incrementally for new
mail).  It parses any remaining messages in the mail file and updates its
message set.
"
  (let ((body (make-array (- length (mail-file-file-length file))
			  :Element-Type 'string-char
	      )
	)
	(delta (- length (mail-file-file-length file)))
       )
       (waiting-for-file-lock
	 (with-open-file (instream (mail-file-name file) :Direction :Input)
	   (file-position instream (mail-file-file-length file))
	   (loop for index from 0
		 for char = (read-char instream nil :Eof)
		 until (equal :Eof char)
		 do (setf (aref body index) char)
	   )
	 )
       )
       (let ((keywords (getf (fs:file-properties (mail-file-name file) nil)
			     :mailbox-keywords
		       )
	     )
	    )
	    (waiting-for-file-lock
	      (if (equal 0 (mail-file-file-length file))
		  (setf (fill-pointer (mail-file-messages file)) 0)
		  nil
	      )
	      (setf (mail-file-keywords file)
		    (remove-duplicates (append yw:*System-Flags* keywords))
	      )
	      (setf (mail-file-file-length file) length)
	      (setf (mail-file-last-written-date file) (file-write-date file))
	      (parse-messages file delta body keywords type)
	    )
       )
  )
)

(defun generate-MM-header-line (message-as-string date/time ends-ok-p)
"Given a message and the date at which the message was sent, generates an MM
format header line for it.  Ends-ok-p tells it whether the message ends in a NL
or not.
"
  (format nil "~A-~A,~D;000000000000~%"
	  (time:print-universal-time date/time nil nil :dd-mmm-yy)
	  (time:timezone-string)
	  (+ (if ends-ok-p 0 1) (length message-as-string))
  )
)


;;; Re: flags.
;There is 7 bits for flags and they are from right to left(starting with bit 0):
;Seen, Deleted, flagged, answered, recent, spare, spare.

;Starting from bit 35 and decreasing to bit 7 we store the keywords. The first
;keyword in the list is bit 35, the 2nd 34, etc ... Bizarre indeed. The 36 bits
;comes from the tops20 word size.

(defun as-bits (number number-of-bits &optional (result nil))
"Given a number which is at most number-of-bits long, returns a list of T or
NIL values for the bits starting with the least significant bit.
"
  (declare (optimize (speed 3) (safety 0)))
  (if (> number-of-bits 0)
      (let ((bit-value (expt 2 (- number-of-bits 1))))
	   (if (= 0 (floor number bit-value))
	       (as-bits number (- number-of-bits 1) (cons nil result))
	       (as-bits (- number bit-value) (- number-of-bits 1)
			(cons t result)
	       )
	   )
      )
      result
  )
)

(defun newline-index (string &optional (start 0))
  "Returns the index of the first newline in string."
  (sys:%string-search-char #\newline string start (length string))
)

(defmethod coerce-header-to-string ((header string))
  "Noop, since it is already a string."
  header
)

(defmethod coerce-header-to-string ((header cons))
"Turns a list of mail:headers into a string."
  (if (rest header)
      (string-append (send (first header) :String-For-Message) #\newline
		     (coerce-header-to-string (rest header))
      )
      (send (first header) :String-For-Message)
  )
)

(defmethod Message-Rfc822-Internal ((part mail:basic-header))
  (send part :String-Form-Message)
)

(defmethod Message-Rfc822-Internal ((part list))
  (if (equal part nil)
      ""
      (string-append (Message-Rfc822-Internal (first part))
		     (if (rest part) #\newline "")
		     (Message-Rfc822-Internal (rest  part))
      )
  )
)

(defmethod message-rfc822-internal ((part string))
  "This is just a string so return it."
  part
)

(defun Message-including-format-stuff (message)
  "Gets the whole of the message."
  (let ((old (message-whole-message message)))
       (setf (message-whole-message message) (message-rfc822-internal old))
       (let ((new (message-whole-message message)))
	    (if (eq old new)
		(setf (message-rfc822-string message) nil)
		nil
	    )
	    new
       )
  )
)

(defun Message-Rfc822 (message)
  "Gets the rfc822 equivalent text for the message."
  (if (message-rfc822-string message)
      (message-rfc822-string message)
      (let ((whole (Message-Including-Format-Stuff message)))
	   (setf (message-rfc822-string message)
		 (nsubstring whole (+ 1 (Newline-Index whole)))
	   )
	   (message-rfc822-string message)
      )
  )
)

(defun Message-Rfc822.header (message)
  "Gets the header of the message."
  (setf (message-header message)
	(message-rfc822-internal (message-header message))
  )
  (message-header message)
)

(defun Message-Rfc822.Text (message)
  "Gets the header of the message."
  (setf (message-body message)
	(message-rfc822-internal (message-body message))
  )
  (message-body message)
)

(defmethod Reset-Message-Header
	   (message message-number mailbox new-header (format (eql :MM)))
"Sets the recorded header of the message to have a new header.  Updates all of
the message parsing information.
"
  (let ((whole-message (Message-including-format-stuff message))
	(new-header (coerce-header-to-string new-header))
       )
       (multiple-value-bind
	 (date-start date-end ignore ignore flags-start flags-end)
	   (Find-MM-Date-Length-And-Flags
	     whole-message 0 (newline-index whole-message)
	   )
	 (let ((date  (subseq whole-message date-start date-end))
	       (flag-string (subseq whole-message flags-start flags-end))
	       (length
		 (+ 1 (length new-header) (length (message-body message)))
	       )
	      )
	      (setf (message-header message) new-header)
	      (let ((new-MM-header
		      (format nil "~A,~D;~A" date length flag-string)
		    )
		   )
		   (let ((whole
			  (string-append
			    new-MM-header #\newline new-header #\newline
			    (message-body message)
			  )
			 )
			 (flags (Decode-Message-Flags
				  flag-string (mail-file-keywords mailbox)
				  format
		                )
			 )
			)
		        (let ((new-message
				(Parse-Out-Message
				  0 whole 0 (length whole) date length flags
			        )
			      )
			     )
			     (setf (the-message message-number mailbox)
				   new-message
			     )
			)
		   )
	      )
	 )
       )
  )
)

(defmethod Reset-Message-Body
	   (message mailbox new-body (format (eql :MM)))
"Updates the representation of Message so that it contains the new message
body.
"
  (ignore mailbox)
  (let ((whole-message (Message-including-format-stuff message)))
       (multiple-value-bind
	 (date-start date-end ignore ignore flags-start flags-end)
	   (Find-MM-Date-Length-And-Flags
	     whole-message 0
	     (Newline-Index whole-message)
	   )
	 (let ((date  (subseq whole-message date-start date-end))
	       (flags (subseq whole-message flags-start flags-end))
	       (length
		 (+ 1 (length new-body) (length (message-header message)))
	       )
	      )
	      (setf (message-body message) new-body)
	      (let ((new-MM-header (format nil "~A,~D;~A" date length flags)))
		   (let ((whole
			  (string-append
			    new-MM-header #\newline (message-header message)
			    #\newline new-body
			  )
			 )
			)
		        (setf (message-whole-message message) whole)
			(setf (message-length message) length)
			message
		   )
	      )
	 )
       )
  )
)

(defmethod Set-Flags-In-Mail-File
	   (message flag-alist keywords (format (eql :MM)))
"Side-effects the message so that its internal representation of flags has been
updated to conform to flag-slist.
"
  (let ((flag-string (Encode-Message-Flags flag-alist keywords format))
	(header (Message-including-format-stuff message))
       )
       (multiple-value-bind (ignore ignore ignore ignore flags-start flags-end)
	   (Find-MM-Date-Length-And-Flags header 0 (Newline-Index header))
	 (copy-array-portion flag-string 0 (length flag-string)
			     header flags-start flags-end
         )
       )
       flag-alist
  )
)

(defmethod decode-message-flags (flag-string keywords (format (eql :MM)))
"Given a string of the flag list in an YW header returns an alist mapping the
flags names to the set status, i.e.
  ((:\\seen t) (:\\deleted nil) ..(keywordn nil)
"
  (let ((bits
	  (As-Bits (let ((*read-base* 8)) (read-from-string flag-string)) 36)
	)
       )
       (append (loop for bit in bits
		     for flag in yw:*System-Flags*
		     for index from 0 to 7
		     collect (list flag bit)
	       )
	       (loop for index from 35 downto 8
		     for key in keywords
		     collect (list key (nth index bits))
	       )
       )
  )
)
  
(defmethod encode-message-flags (flag-alist keywords (format (eql :MM)))
"Given a flag alist in the form ((:\\seen t) (:\\deleted nil) ..(keywordn nil)
returns a string for the MM header of the form 000000000011.
"
  (let ((number (+ (loop for index from 0
			 for flag in yw:*System-Flags*
			 for entry = (assoc flag flag-alist :Test #'eq)
			 sum (if (and entry (second entry))
				 (expt 2 index)
				 0
			     )
		   )
		   (loop for index from 35 downto 8
			 for key in keywords
			 for entry = (assoc key flag-alist :Test #'eq)
			 sum (if (and entry (second entry))
				 (expt 2 index)
				 0
			     )
		   )
		   0
		)
	)
       )
       (let ((*print-base* 8.)) (format nil "~12,,,'0@A" number))
  )
)


(defun append-message-to-MM-file (message-as-string date/time mail-file)
"Given a message and the date of the message being sent, appends the
message to the specified mail file."
  (waiting-for-file-lock
    (with-open-file (ostream mail-file
			     :Direction :Output
			     :If-Does-Not-Exist :Create
			     :If-Exists :Append
		    )
      (let ((ends-ok-p
	      (char= #\newline
		     (aref message-as-string (- (length message-as-string) 1))
	      )
	    )
	   )
	   (princ
	     (generate-MM-header-line message-as-string date/time ends-ok-p)
	     ostream
	   )
	   (princ message-as-string ostream)
	   (if ends-ok-p nil (terpri ostream))
      )
    )
  )
)

(defun translate-pathname (pathname)
"Makes sure that a pathname is translated into a physical pathname."
  (send pathname :Translated-Pathname)
)

(defun write-out-mailbox (mailbox)
"Given a mail file object writes it out to its appropriate file."
  (if (mail-file-modified-p mailbox)
      (progn
        (Wait-For-File-Lock-Free (mail-file-name mailbox))
	(with-write-lock (mailbox)
	  (with-open-file (ostream (mail-file-name mailbox)
				   :Direction :Output
				   :If-Does-Not-Exist :Create
;				   :If-Exists :Supersede ;;; {!!!!}
				   :If-Exists :New-Version
			  )
	    (loop for message-number from 1 to (Number-Of-Messages-In mailbox)
		  for message = (the-message message-number mailbox)
		  do (format ostream "~A"
			     (Message-including-format-stuff message)
		     )
	    )
	  )
	)
	(setf (mail-file-modified-p mailbox) nil)
      )
      nil
  )
)

(defun Find-MM-Date-Length-And-Flags (string start end)
"Given a string denoted by STRING, start and end, returns the positions
of the start and end points of the date, length and flag components in
an MM header line.
"
  (declare (values date-start date-end length-start length-end
		   flags-start flags-end
	   )
  )
  (let ((comma (position #\, string :Start start :End end)))
       (if comma
	   (let ((semicolon (position #\; string :Start comma :End end)))
	        (if semicolon
		    (values start comma
			    (+ 1 comma) semicolon
			    (+ 1 semicolon) end
		    )
		    nil
		)
	   )
	   nil
       )
  )
)

(defun parse-mm-header-line (string start end keywords format)
"Given an MM header line denoted by string, start and end, returns the
date string, the length and the flag-alist.  keywords and format are used to
decode the flags.
"
  (declare (values date-string length flags))
  (multiple-value-bind (date-start date-end length-start length-end
			flags-start flags-end
		       )
      (Find-MM-Date-Length-And-Flags string start end)
    (if date-start
        (values (nsubstring string date-start date-end)
		(let ((*read-base* 10.))
		     (read-from-string string t nil :Start length-start
				       :End length-end
		     )
		)
		(decode-message-flags
		  (nsubstring string flags-start flags-end)
		  keywords format
		)
	)
	nil
    )
  )
)

(defun union-of-addresses (&rest addresses)
  "Returns the set-union of the addresses."
  (let ((all-addresses (loop for addr in (apply #'append addresses)
			     collect (etypecase addr
				       (yw:address addr)
				       (mail:address
					(yw:make-address-from-explorer-address
					  addr
					)
				       )
				     )
		       )
	)
       )
       (remove-duplicates all-addresses :Test #'equalp)
  )
)

(defun canonicalize-envelope (envelope message)
  "Fills in the canonical keys of the envelope."
  (setf (yw:envelope-$cc envelope)
	(union-of-addresses
	  (yw:envelope-bcc envelope) (yw:envelope-cc envelope)
	)
  )
  (setf (yw:envelope-$from envelope)
	(union-of-addresses
	  (yw:envelope-from envelope)
	  (get-value-of-field :Apparently-From message)
	  (get-value-of-field :Resent-From message)
	)
  )
  (setf (yw:envelope-$to envelope)
	(union-of-addresses
	  (yw:envelope-bcc envelope)
	  (yw:envelope-cc envelope)
	  (yw:envelope-to envelope)
	  (get-value-of-field :Apparently-To message)
	  (get-value-of-field :Resent-To message)
	)
  )
  (setf (yw:envelope-$subject envelope) (yw:envelope-subject envelope))
  (setf (yw:envelope-canonicalized-p envelope) t)
)

(defun parse-out-message (message-start body start end date/time length flags)
"Given a pointer to the begging of a message in the body of a mail file or
message, the start and the end of the message and the date length and flags that
have been parsed from the message header line, builds a message object to
represent the message doing any necessary parsing and such.
"
  (ignore length)
  (let ((nl (yw:string-search yw:*blank-line* body start)))
       (let ((header (nsubstring body start (if (and nl (< nl end)) nl end)))
	     (message-body
	       (if (and nl (< nl end)) (nsubstring body (+ 1 nl) end) "")
	     )
	     (whole-message (nsubstring body message-start end))
	    )
	    (multiple-value-bind (internal-date flags envelope rfc822length
				  from-text subject to-text sender-text
				  list-of-headers
				 )
		(yw:parse-out-header-values header message-body flags nil nil)
	      (ignore from-text to-text sender-text internal-date subject)
	      (let ((message
		      (make-message :Header header
				    :Body message-body
				    :Envelope envelope
				    :Length rfc822length
				    :Flags flags
				    :internal-date date/time
				    :Whole-Message whole-message
				    :Parsed-Header list-of-headers
		      )
		    )
		   )
		   (canonicalize-envelope envelope message)
		   message
	      )
	    )
       )
  )
)

(defun parse-messages (file file-length body keywords format)
"Parses any remaining messages in the mail file FILE.  File length tells us
how much we have yet to parse.  Body is a string that represents the remainder
of the mail file.  Keywords is the keyword list for the mailbox that we
computed previously.  Format is the format of the mailbox.
"
  (let ((index 0)
	(messages (mail-file-messages file))
       )
       (loop for message-index from 0
	     until (>= index file-length)
	     for eol = (newline-index body index)
	     for last-index = index
	     when eol
	     do (multiple-value-bind (date/time length flags)
		    (parse-mm-header-line body index eol keywords format)
;		  (print (list date/time length flags index eol))
		  (setq index (+ eol 1 length))
		  (vector-push-extend
		    (Parse-Out-Message
		      last-index body (+ 1 eol) (+ length eol 1)
		      date/time length flags
		    )
		    messages
		  )
		)
       )
  )
)