;;; -*- Mode:Common-Lisp; Package:PROFILE; Base:10; Patch-File:t -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

;;; Patches

;;; TI code.
(defun DEFINE-PROFILE-VARIABLE-1 (VARIABLE-SYMBOL CLASSES TYPE PRETTY-NAME DOCUMENTATION)
  "This does the real work of DEFINE-PROFILE-VARIABLE."
  (declare (special sys:xr-bq-list))
  (ASSERT (OR *ALLOW-UNBOUND-SYMBOLS-P* (BOUNDP VARIABLE-SYMBOL))
;	    (VARIABLE-SYMBOL)
	  `(,VARIABLE-SYMBOL)
	  "The symbol ~S is unbound." VARIABLE-SYMBOL)
  (IF (NOT (BOUNDP VARIABLE-SYMBOL))
      (format t "~%The variable ~s is unbound." VARIABLE-SYMBOL))
  (progn
    ;;; Change put in here by JPR to deal with unbound profile variables.
    (SETF (VARIABLE-INITIAL-VALUE VARIABLE-SYMBOL)
	  (if (BOUNDP VARIABLE-SYMBOL)
	      (SYMBOL-VALUE VARIABLE-SYMBOL)
	      '.not-bound.))
    
    ;; If only one class is specified as a symbol, make it a list of one class
    (WHEN (SYMBOLP CLASSES)
      (SETQ CLASSES `(,CLASSES)))
    
    ;;Add the variable to each of the classes and add the classes to the variable
    (LOOP FOR CLASS IN CLASSES
	  DO
	  (ADD-VARIABLE VARIABLE-SYMBOL CLASS))
    (SETF (VARIABLE-CLASSES VARIABLE-SYMBOL) CLASSES)
    
    ;;Check and record the CVV type
    (ASSERT (OR *ALLOW-ILLEGAL-CVV-TYPES-P*
		(VALID-CVV-TYPE-P (IF (CONSP TYPE) (CAR TYPE) TYPE)))
	    (TYPE)
	    "~%TYPE, ~s, is not a valid CVV keyword or function." TYPE)
    (WHEN (AND (EQ :WARN *ALLOW-ILLEGAL-CVV-TYPES-P*)
	       (NOT (VALID-CVV-TYPE-P (IF (CONSP TYPE) (CAR TYPE)
					  TYPE))))
      (FORMAT T "~%TYPE, ~s, is not a valid CVV keyword or function." TYPE))
    (SETF (VARIABLE-TYPE VARIABLE-SYMBOL) TYPE)
    
    ;;PRETTY (Display) Name of Variable
    (WHEN (NULL PRETTY-NAME)
      (SETQ PRETTY-NAME (VARIABLE-NAME VARIABLE-SYMBOL)))
    (SETF (VARIABLE-NAME VARIABLE-SYMBOL) PRETTY-NAME)
    
    (COND
      ((NULL DOCUMENTATION)
       (SETQ DOCUMENTATION
	     (OR (DOCUMENTATION VARIABLE-SYMBOL 'VARIABLE)
		 (GET VARIABLE-SYMBOL 'ZWEI::DOCUMENTATION-PROPERTY)))
       (WHEN (NULL DOCUMENTATION)
	 (CASE *ALLOW-NO-DOCUMENTATION-P*
	   (NIL (FERROR () "~%No documentation for the profile variable ~s" VARIABLE-SYMBOL))
	   (:WARN (FORMAT T "~%No documentation for the profile variable ~s" VARIABLE-SYMBOL)))))
      (T (SETF (VARIABLE-DOCUMENTATION VARIABLE-SYMBOL) DOCUMENTATION)))))

;;; TI code.
(DEFUN REVERT-PROFILE-VARIABLES (&OPTIONAL (QUERY-P T) (DO-VARIABLES-CHANGED-BY-INIT-FILE-P T))
  "Reverts all profile variables to their initial values (bindings at boot time).
QUERY-P indicates whether the user should be asked to verify each change.
DO-VARIABLES-CHANGED-BY-INIT-FILE-P should be T unless variables changed by the user in his
init file are not to be reverted.  This should be done if these variables are to be restored after this
function (preventing needless setting of variables and execution of side effects)."
  (LOOP FOR CLASS IN *VARIABLE-CLASSES*
	FOR VARIABLES = (SYMBOL-VALUE (CLASS-VARIABLE CLASS))
	DO
	(LOOP FOR VARIABLE IN VARIABLES
	      FOR INITIAL-VALUE = (VARIABLE-INITIAL-VALUE VARIABLE)
	      FOR CURRENT-VALUE = (VARIABLE-CURRENT-VALUE VARIABLE)
	      UNLESS (EQUALP INITIAL-VALUE CURRENT-VALUE)
	      DO
	      (WHEN (AND (OR (NOT QUERY-P)
	                 (Y-OR-N-P "Revert the variable ~s from ~s back to ~s, its original value? "
				   VARIABLE CURRENT-VALUE INITIAL-VALUE))
	                 (OR DO-VARIABLES-CHANGED-BY-INIT-FILE-P
			     
	      ;;This is true if the variable has been set by a profile-setq form
			     (EQ (GET VARIABLE 'VALUE-FROM-INIT-FILE :NO-PROPERTY) :NO-PROPERTY)))
		(if (not (equal initial-value '.not-bound.))
		    ;;; This may have been bound in some way by the user.  Don't
		    ;;; Undo it, since it may have been bound by loading a tool.
		    (SETF (VARIABLE-CURRENT-VALUE VARIABLE) INITIAL-VALUE))))))


(advise (:method tv:basic-choose-variable-values :who-line-documentation-string)
	:Around :Look-For-Tools nil
   (if (typep self 'profile:profile-cvv-pane)
       (multiple-value-bind (window-x-offset window-y-offset)
	   (tv:sheet-calculate-offsets self tv:mouse-sheet)
	 (let ((x (- tv:mouse-x window-x-offset))
	       (y (- tv:mouse-y window-y-offset))
	      )
	      (multiple-value-bind (value type)
		  (send self :mouse-sensitive-item x y)
		(if type
		    (let ((result :do-it))
			 (if (and value (symbolp (second value))
				  (get (second value) :source-tool)
			     )
			     (string-append
			       "[Tool: "
			       (symbol-name
				 (get (second value) :source-tool)
			       )
			       " must be loaded] "
			       result
			     )
			     result
			 )
		    )
		    (if (and value (symbolp (second value)))
			(if (get (second value) :source-tool)
			    (string-append
			      "[Tool: "
			      (symbol-name
				(get (second value) :source-tool)
			      )
			      " must be loaded] "
			      (documentation (second value))
			    ) 
			    (documentation (second value))
			)
			(send self :default-who-line-documentation)
		    )
		)
	      )
	 )
       )
       :do-it
   )
)

;-------------------------------------------------------------------------------


;;; CVV Extensions:


;;; CVV type to read a universal time specifier.
(setf (get :time 'tv:choose-variable-values-keyword-function)
      'decode-time-item
)

(defun read-time (stream)
"Reads a time from the user."
  (let ((string (tv:read-string stream)))
       (multiple-value-bind (ignore error-p)
	   (catch-error
	     (multiple-value-list (time:parse-universal-time string 0 nil t))
	     nil
	   )
	 (if error-p
	     (ferror nil "A time is required.")
	     string
	 )
       )
  )
)

(defun decode-time-item (keyword-and-args)
"The CVV function to deal with the time CVV type."
  (ignore keyword-and-args)
  (values 'princ 'read-time nil nil nil
	  "Click left to enter a new time from the keyboard."
  )
)


;-------------------------------------------------------------------------------

;;; The following was commented out in the sources.  I can't think why.

(setf (get :font 'tv:choose-variable-values-keyword)
      '(print-font cvv-read-font)
)

(defun cvv-read-font (stream)
  "Read in and validate a font or font name."
  (let* ((*package* (find-package 'fonts))
	 (font (read stream t))
	)
        (unless (or (and (symbolp font)
			 (boundp font)
			 (typep (symbol-value font) 'font)
			 (eq (symbol-package font) *package*)
		    )
		    (typep font 'font))
	  (ferror () "A Font is required.")
	)
	(if (and (symbolp font) (boundp font))
	    (symbol-value font)
	    font
	)
  )
) 

;-------------------------------------------------------------------------------

(setf (get :beep-type 'tv:choose-variable-values-keyword-function)
      'cvv-beep-type
)

(defun cvv-beep-type (kwd-and-args)
"A CVV type that can read beep types."
  (ignore kwd-and-args)
  (let ((tv:menu (list 'quote
		       (loop for type in (copy-list tv:*beep-types*) collect
			     (list (let ((*print-case* :Capitalize))
				        (format nil "~A" type)
				   )
				   :Value type
			     )
		       )
		 )
	)
       )
       (declare (special tv:menu))
       (values (closure '(tv:menu) 'tv:choose-variable-values-menu-print)
	       nil nil nil
	       (closure '(tv:menu) 'tv:choose-variable-values-menu)
	       "Click left to select a new beep type from a menu"
       )
  )
)

;-------------------------------------------------------------------------------

(setf (get :font-list 'choose-variable-values-keyword)
      '(print-font cvv-read-font-list)
)

(defun print-font (font &optional (stream *standard-output*))
"Prints the name of a font."
  (let ((*package* (find-package 'fonts)))
       (typecase font
	 (font (princ (tv:font-name font) stream))
	 (cons (princ "(" stream)
	       (print-font (first font) stream)
	       (mapcar #'(lambda (f)
			   (princ " " stream)
			   (print-font f stream)
			 )
			 (rest font)
	       )
	       (princ ")" stream)
	 )
	 (otherwise (prin1 font stream))
       )
  )
)

(defun validate-font (font)
  "Given an object this function signals an error if it is not a font name."
  (unless (or (and (symbolp font)
		   (boundp font)
		   (typep (symbol-value font) 'font)
		   (eq (symbol-package font) *package*))
	      (typep font 'font)
	  )
    (ferror nil "A non-Font found in font list.")
  )
)

(defun cvv-read-font-list (stream)
  "Read in and validate a list of fonts."
  (let* ((*package* (find-package 'fonts))
	 (fonts (read stream))
	)
        (if (not (consp fonts))
	    (ferror nil "List of fonts is required.")
	)
	(mapcar #'validate-font fonts)
	(mapcar #'symbol-value fonts)
  )
)

;-------------------------------------------------------------------------------

(defmacro defprofile-class (name var-name menu-string doc-string)
"Defines a class of profile variable (a class is the named thing that appears
in the profiler's menu across the bottom, and has a set of variables associated
with it.  Name is the name of the class, var-name is the name of the variable
that holds the list of symbols in that class.  Menu-string is the string to put
in the profiler's menu and doc-string is the mouse dics for that menu.
"
  `(progn (defparameter ,var-name nil
	    "A list of all of the variables defined for the :SCD profile type."
          )
	  (pushnew  '(,name ,var-name ,menu-string ,doc-string)
		    profile:*variable-classes* :test #'equalp
	  )
	  ',name
   )
)

(defmacro defprofile (name variables-classes source-tool &rest args)
"Defines a profile variable called Name in the (maybe list) of profile classes
Variable-Classes.  Source-Tool is the name of the KSL tool which must be
loaded in order for this variable to be meaningful.  Args are standard args
to profile:define-profile-variable.
"
  (if (not source-tool)
      (format t "~&No source tool provided for ~A" name)
      nil
  )
 `(progn (profile:define-profile-variable
	   ,name (:ksl ,@(if (consp variables-classes)
			    variables-classes
			    (if variables-classes
				(list variables-classes)
				variables-classes
			    )
			)
		)
	   :declare-special-p nil
	   ,@args
	 )
	 (putprop ',name ',(if (equal :None source-tool) nil source-tool)
		  :Source-Tool
	 )
	 (si:record-source-file-name ',name 'defvar t)
	 ',name
  )
)


;;; Define the class of KSL variables.
(Defprofile-Class :Ksl *KSL-Environment-Extensions*
  "KSL Variables" "Variables controlling KSL Environment Extensions"
)

;-------------------------------------------------------------------------------

(export 'defprofile 'profile)
(export 'defprofile-class 'profile)