;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

(defun parse-headers (header cache-entry)
  (if (is-present (cache-parsed-headers cache-entry))
      (cache-parsed-headers cache-entry)
      (progn (setf (cache-parsed-headers cache-entry)
		   (mapcar #'mail:parse-header
			   (Merge-Tabbed-Lines
			     (Discard-Blank-Lines
			       (split-into-lines header 0)
			     )
			   )
		   )
	     )
	     (cache-parsed-headers cache-entry)
      )
  )
)

(defun header-type (header)
  (send header :Type)
)

(defun skip-content-type-whitespace (stream)
  (loop for char = (peek-char nil stream nil nil)
	while (and char (sys:whitespacep char))
	do (read-char stream)
  )
)

(Defun parse-content-type-token (stream &optional (upcase-p nil))
  (Skip-Content-Type-Whitespace stream)
  (let ((char (peek-char nil stream nil nil)))
       (if (and char (char= char #\"))
	   (intern (read stream) 'keyword)
	   (let ((chars
		   (loop for char = (read-char stream nil nil)
			 while (and char
				    (not (find char *Tspecials* :Test #'char=))
			       )
			 collect (if upcase-p (char-upcase char) char)
			 finally (if char (unread-char char stream) nil)
				 (Skip-Content-Type-Whitespace stream)
		   )
		 )
		)
		(if chars
		    (intern (make-array (length chars)
					:Element-Type 'string-char
					:Initial-Contents chars
			    )
			    'keyword
		    )
		    nil
		)
	   )
       )
  )
)

(defun boundary-char-p (char)
  (or (alpha-char-p char)
      (digit-char-p char)
      (and (find char *Bcharsnospace* :Test #'char=)
	   (Not (find char *Tspecials* :Test #'char=))
      )
  )
)

(defun boundary-subset-of (string)
  (let ((chars (loop for char being the array-elements of string
		     when (boundary-char-p char)
		     collect char
	       )
	)
       )
       (make-array (length chars) :Element-Type 'string-char
		   :Initial-Contents chars
       )
  )
)

(defun parse-content-type-parameter (stream)
  (let ((attribute (parse-content-type-token stream)))
       (let ((next-char (peek-char t stream nil nil)))
	    (if next-char
		(if (equal #\= next-char)
		    (progn (read-char stream)
			   (skip-content-type-whitespace stream)
			   (list attribute
				 (if (equal #\" (peek-char t stream nil nil))
				     (read stream)
				     (Parse-Content-Type-Token stream)
				 )
			   )
		    )
		    attribute
		)
		attribute
	    )
       )
  )
)

(Defun parse-content-type (header &optional (force-p nil))
  (or (and (not force-p) (get header :Type))
      (let ((body (send header :Body)))
	   (with-input-from-string (stream body)
	     (let ((type (parse-content-type-token stream t)))
		  (setf (get header :Type) (net:intern-as-keyword type))
		  (setf (get header :SubType)
			(if (equal #\/ (peek-char nil stream nil :eof))
			    (progn (read-char stream)
				   (parse-content-type-token stream t)
			    )
			    :default
		        )
		  )
		  (setf (get header :Parameters)
			(loop while (equal #\; (peek-char nil stream nil :eof))
			      do (read-char stream)
			      collect (parse-content-type-parameter stream)
			)
		  )
	     )
	     (get header :Type)
	   )
      )
  )
)

(defun nsubstring-with-fill-pointer
       (string from &optional to (area nil) &aux length arraytype)
  "Return a displaced array whose data is part of STRING, from FROM to TO.
If you modify the contents of the displaced array, the original string changes.
If TO is omitted or NIL, the substring runs up to the end of the string.
If AREA is specified, the displaced array is made in that area."
  (sys:coerce-string-arg string)
  (or to (setq to (array-active-length string)))
  (setq length (- to from))
  (or (and (>= length 0) (>= from 0) (<= to (array-active-length string)))
     (ferror () "Args ~S and ~S out of range for ~S." from to string))
  (setq arraytype (array-type string))
  (cond
    ((not (array-indexed-p string))
     (make-array length :area area :type arraytype :displaced-to string
		 :fill-pointer t
		 :displaced-index-offset from));INDEX OFFSET
    ;; OTHERWISE, PROBABLY A SUBSTRING OF A SUBSTRING
    (t
     (make-array length :area area :type arraytype
		 :displaced-to (sys:array-indirect-to string)
		 :fill-pointer t
		 ;;POINT TO ARRAY POINTED TO ORIGINALLY
		 :displaced-index-offset
		   (+ from (sys:array-index-offset string))))))

(defun boundary-string-of (parameters)
  (loop for parameter in parameters
	when (and (consp parameter)
		  (string-equal (first parameter) "BOUNDARY")
	     )
	return (string (second parameter))
	when (not (consp parameter)) return (string parameter)
  )
)

(defun Maybe-Find-Body-Parts
       (content-type-header cache-entry mailstream &optional (force-p nil))
  (if (and (is-present (cache-rfc822text cache-entry))
	   (or (not (is-present (cache-body-parts cache-entry)))
	       (not (cache-body-parts cache-entry))
	       force-p
	   )
      )
      (let ((separator
	      (string-append
		"--" (string (Boundary-String-Of
			       (get content-type-header :Parameters)
			     )
		     )
	      )
	    )
	    (text (cache-rfc822text cache-entry))
	   )
	   (let ((indices
		   (loop with start = 0
			 for index = (String-Search separator text start)
			 until (or (not index)
				   (string-equal "--" text :Start2
						 (+ index (length separator))
				   )
			       )
			 collect (or index (length text))
			 do (setq start (+ index (length separator)))
		   )
		 )
		)
	        (setf (cache-body-parts cache-entry)
		      (loop for this in indices
			    for next in (rest indices)
			    collect (nsubstring-with-fill-pointer
				      text (+ 1 this (length separator)) next
				    )
		      )
		)
	   )
      )
      nil
  )
  (if (consp (cache-body-parts cache-entry))
      (setf (cache-body-parts cache-entry)
	    (let ((*default-content-type*
		    (if (equal :Digest (cache-content-subtype cache-entry))
			:Message
			:Text
		    )
		  )
		  (*default-content-subtype*
		    (if (equal :Digest (cache-content-subtype cache-entry))
			:RFC822
			:default
		    )
		  )
		 )
		 (turn-multi-part-message-into-mailstream
		   cache-entry mailstream
		 )
	    )
      )
      nil
  )
)

(defmethod mailbox-name-from-superior-cache-entry ((cache-entry cache))
  (maybe-preempt-envelopes (cache-mailstream cache-entry) cache-entry)
  (get-and-format-header-display-string cache-entry)
  (princ cache-entry 'sys:null-stream)
  (without-tabs
    (format-header-display-string cache-entry
       *sub-mailbox-header-display-specification*
    )
    1000
  )
)

(defun turn-multi-part-message-into-mailstream (cache-entry mailstream)
  (if (consp (cache-body-parts cache-entry))
      (progn (print-self cache-entry 'sys:null-stream 0 nil)
	     (setf (cache-body-parts cache-entry)
		   (let ((*mailer* (find-mail-window mailstream t t)))
		        (Make-Stream-And-Headers-From-Digest
			  (mailbox-name-from-superior-cache-entry cache-entry)
			  (cache-body-parts cache-entry) mailstream
			  cache-entry nil 'multipart-stream
			)
		   )
	     )
      )
      (cache-body-parts cache-entry)
  )
)
	  
(defmethod Can-Handle-Content-Type-P-Internal
	   ((type (eql :Multipart)) (subtype t))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Multipart)) (subtype t) content-type-header
	    cache-entry mailstream force-p
	   )
  (maybe-find-body-parts content-type-header cache-entry mailstream force-p)
  (let ((substream (cache-body-parts cache-entry)))
       (if (and (typep substream 'imap-stream-mixin) (is-present substream))
	   (progn (send substream :Maybe-Initialize)
		  (loop for index from 1 to (send substream :Messagecnt)
			do (Maybe-Parse-Multi-Part-Stuff
			     (cache-entry-of index substream) substream
			   )
		  )
	   )
	   ;; message hasn't been parsed yet.
	   nil
       )
  )
)


;-------------------------------------------------------------------------------

(defun fix-up-cache-entry-to-be-contained-message
       (content-type-header cache mailstream)
  (ignore content-type-header)
  (if (is-present (cache-decoded-body cache))
      nil
      (let ((text (cache-rfc822text cache)))
	   (parse-message-into-cache-entry
	     cache mailstream text
	     (loop for start from 0 below (length text)
		   until (not (sys:whitespacep (aref text start)))
		   finally (return start)
	     )
	   )
	   ;;; Relabel as :message.  This may be overridden by the maybe-parse
           (setf (cache-content-type cache) :message)
	   (setf (cache-decoded-body cache) (cache-rfc822text cache))
	   (maybe-filter-header (cache-rfc822header cache) cache)
	   (maybe-parse-multi-part-stuff cache mailstream)
      )
  )
)

(defmethod can-handle-content-type-p-internal
	   ((type (eql :Message)) (subtype (eql :default)))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Message)) (subtype (eql :Default)) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore force-p)
  (fix-up-cache-entry-to-be-contained-message
    content-type-header cache-entry mailstream
  )
)

(defmethod can-handle-content-type-p-internal
	   ((type (eql :Message)) (subtype (eql :Rfc822)))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Message)) (subtype (eql :Rfc822)) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore force-p)
  (fix-up-cache-entry-to-be-contained-message
    content-type-header cache-entry mailstream
  )
)

;-------------------------------------------------------------------------------

(defun font-for-environment (environment)
  (if (member :Bold environment :Test #'eq)
      (if (member :italic environment :Test #'eq)
	  3
	  1
      )
      (if (member :italic environment :Test #'eq)
	  2
	  0
      )
  )
)

(defun read-richtext-command (string start)
  (let ((end-index (loop for ind from start below (length string)
			 when (equal #\> (aref string ind))
			 return ind
			 finally (return nil)
		   )
	)
       )
       (if end-index
	   (values (intern (string-upcase (subseq string start end-index))
			   :Keyword
		   )
		   (+ end-index 1)
	   )
	   (values nil (+ start 1))
       )
  )
)

(defmethod Process-Richtext-Command
	   ((command (eql :nl)) environment start-index body new-string)
  (ignore body)
  (Newline-And-Indent-For-Current-Environment environment new-string)
;  (yw-zwei:add-chars-to-output new-string #\newline)
  (values environment start-index)
)

(defun newline-and-indent-for-current-environment (environment new-string)
  (let ((strings
	  (loop for index from 0 below (count :Excerpt environment :Test #'eq)
		for str
		  = (nth (mod index
			     (length yw-zwei:*mail-reply-indent-prefix-strings*)
			 )
			 yw-zwei:*mail-reply-indent-prefix-strings*
		    )
	  )
	)
       )
       (yw-zwei:add-chars-to-output
	 new-string (apply 'string-append #\newline strings)
       )
  )
)

(defmethod Process-Richtext-Command
	   ((command (eql :excerpt)) environment start-index body new-string)
  (ignore new-string body)
  (let ((new-environment (cons :Excerpt environment)))
       (newline-and-indent-for-current-environment new-environment new-string)
       (values new-environment start-index)
  )
)

(defmethod Process-Richtext-Command
	   ((command (eql :/Excerpt)) environment start-index body new-string)
  (ignore new-string body)
  (let ((new-environment (remove :Excerpt environment :Count 1)))
       (newline-and-indent-for-current-environment new-environment new-string)
       (values new-environment start-index)
  )
)

(defmethod Process-Richtext-Command
	   ((command (eql :comment)) environment start-index body new-string)
  (ignore new-string)
  (let ((end-index
	  (search "</comment>" body :Test #'char-equal :Start2 start-index)
	)
       )
       (if end-index
	   (values environment (+ (length "</comment>") end-index))
	   (values environment start-index)
       )
  )
)

(defmethod Process-Richtext-Command
	   ((command (eql :np)) environment start-index body new-string)
  (ignore body)
  (yw-zwei:add-chars-to-output new-string #\page)
  (values environment start-index)
)

(defmethod Process-Richtext-Command
	   ((command (eql :italic)) environment start-index body new-string)
  (ignore body new-string)
  (values (pushnew :Italic environment) start-index)
)

(defmethod Process-Richtext-Command
	   ((command (eql :/Italic)) environment start-index body new-string)
  (ignore body new-string)
  (values (remove :Italic environment :Count 1) start-index)
)

(defmethod Process-Richtext-Command
	   ((command (eql :lt)) environment start-index body new-string)
  (ignore body)
  (yw-zwei:add-chars-to-output new-string #\<)
  (values environment start-index)
)

(defun parse-richtext (message body)
  (let ((new-string (make-array (length body) :Element-Type 'sys:fat-char
				:Fill-Pointer t
		    )
	)
	(environment nil)
       )
       (setf (fill-pointer new-string) 0)
       (loop for index from 0 below (length body)
	     for char = (aref body index)
	     do (case char
		  (#\newline nil) ;; Do nothing
		  (#\< (multiple-value-bind (command new-index)
			   (read-richtext-command body (+ index 1))
			 (setq index new-index)
			 (multiple-value-setq
			   (environment index)
			   (process-richtext-command command environment index
						     body new-string
                           )
			 )
			 (decf index)
		       )
		  )
		  (otherwise
		   (vector-push-extend
		     (make-char (char-code char) 0
				(font-for-environment environment)
		     )
		     new-string
		   )
		  )
	        )
       )
       (setf (cache-decoded-body message) new-string)
  )
)

(defmethod can-handle-content-type-p-internal
	   ((type (eql :Text)) (subtype (eql :default)))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Text)) (subtype (eql :default)) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore content-type-header mailstream force-p cache-entry)
  nil ;;; This is just fine.
)

(defmethod can-handle-content-type-p-internal
	   ((type (eql :Text)) (subtype (eql :richtext)))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Text)) (subtype (eql :richtext)) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore content-type-header mailstream force-p cache-entry)
  (or (is-present (cache-decoded-body cache-entry))
      (progn (map-fetch-message mailstream cache-entry)
	     (parse-richtext cache-entry (cache-rfc822text cache-entry))
      )
  )
  nil ;;; This is just fine.
)

(defmethod can-handle-content-type-p-internal
	   ((type (eql :Text)) (subtype (eql :us-ascii)))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Text)) (subtype (eql :us-ascii)) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore content-type-header mailstream force-p cache-entry)
  nil ;;; This is just fine.
)

(defmethod can-handle-content-type-p-internal
	   ((type (eql :Text)) (subtype (eql :plain)))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Text)) (subtype (eql :plain)) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore content-type-header mailstream force-p cache-entry)
  nil ;;; This is just fine.
)

;-------------------------------------------------------------------------------

(defmethod can-handle-content-type-p-internal ((type (eql :audio)) (subtype t))
  nil
)

(defmethod Process-Content-Type
	   ((type (eql :Audio)) (subtype t) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore content-type-header mailstream force-p)
  (setf (cache-content-type cache-entry) :text)
  (setf (cache-content-subtype cache-entry) :default)
  (setf (cache-rfc822text cache-entry)
        "
   <<Unplayable Audio>>
"
  )
)

;-------------------------------------------------------------------------------

;(net:parse-host (get (get-printer-device (get-default-printer)) :HOST))

(defmethod can-handle-content-type-p-internal
	   ((type (eql :Image)) (subtype t))
  t
)

(defmethod Process-Content-Type
	   ((type (eql :Image)) (subtype t) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore content-type-header mailstream force-p)
  (if *show-images-if-you-can-p*
      nil
      (progn (setf (cache-content-type cache-entry) :text)
	     (setf (cache-content-subtype cache-entry) :default)
	     (setf (cache-rfc822text cache-entry)
        "
   <<Undisplayable Image>>
"
             )
      )
  )
)

(defun xloadimage (cache-entry)
  (with-open-file (str *xloadimage-temp-file-path* :direction :output
		       :if-exists :overwrite
		       :if-does-not-exist :create
		       :characters nil
		       :byte-size 8
		  )
    (send str :string-out (cache-rfc822text cache-entry))
  )
  (let ((fs:user-id (string-downcase fs:user-id)))
       (net:rshell *xloadimage-server-host-name*
		   (format nil "xloadimage -quiet -name \"~A\" -display ~A ~A &"
			   (cache-subjecttext cache-entry)
			   *x-server-display*
			   (send (pathname *xloadimage-temp-file-path*)
				 :string-for-host
			   )
		   )
		   :Stdout 'sys:null-stream
       )
  )
)

;-------------------------------------------------------------------------------

(defmethod can-handle-content-type-p-internal ((type symbol) (subtype t))
  nil
)

(defmethod Process-Content-Type
	   ((type symbol) (subtype t) content-type-header
	    cache-entry mailstream force-p
	   )
  (ignore content-type-header mailstream force-p)
  (setf (cache-content-type cache-entry) :text)
  (setf (cache-content-subtype cache-entry) :default)
  (setf (cache-rfc822text cache-entry)
        (format nil "~2% <<Undisplayable Unsupported ~A content type ~A, ~A>>"
	   (if (string-equal "X-" (string type) :End2 2) "experimental " "")
	   type subtype
        )
  )
)


;-------------------------------------------------------------------------------

(defun string-stream-to-suitably-encoded (cache-entry)
  (let ((message (Chase-Superiors-Until-You-Find-A-Message
		   cache-entry
		   (cache-content-type cache-entry)
		   (cache-content-subtype cache-entry)
		 )
        )
       )
       (map-fetch-message (cache-mailstream message) message)
       (make-concatenated-stream
	 (make-string-input-stream (cache-rfc822header message))
	 (make-string-input-stream "
"
	 )
	 (if (is-present (cache-old-content-transfer-encoding message))
	     (make-encoding-stream (cache-rfc822text message)
				   (cache-old-content-transfer-encoding message)
	     )
	     (make-string-input-stream (cache-rfc822text message))
	 )
       )
  )
)

(defmethod chase-superiors-until-you-find-a-message
	   ((message cache) (type (eql :message)) (subtype t))
  message
)

(defun chase-to-superior-if-defined (message type subtype)
  (let ((super (superior-cache-of message)))
       (if (eq message super)
	   message
	   (Chase-Superiors-Until-You-Find-A-Message super type subtype)
       )
  )
)

(defmethod chase-superiors-until-you-find-a-message
	   :Around ((message cache) (type t) (subtype t))
  (let ((super (superior-cache-of message)))
       (if (eq message super)
	   (clos:call-next-method)
	   (if (eq :Multipart (cache-content-type super))
	       (if (eq :Digest (cache-content-subtype super))
		   message
		   (clos:call-next-method)
	       )
	       (clos:call-next-method)
	   )
       )
  )
)

(defmethod chase-superiors-until-you-find-a-message
	   ((message cache) (type t) (subtype t))
  (chase-to-superior-if-defined message type subtype)
)

(defun Maybe-Parse-Multi-Part-Stuff
       (cache-entry mailstream &optional (force-p nil))
  (declare (special *edit-server*))
  (let ((content-type-header
	  (find :Content-Type (cache-parsed-headers cache-entry)
		:Key 'Header-Type
	  )
	)
	(encoding-header
	  (find :Content-Transfer-Encoding (cache-parsed-headers cache-entry)
		:Key 'Header-Type
	  )
	)
	(content-id-header
	  (find :Content-Id (cache-parsed-headers cache-entry)
		:Key 'Header-Type
	  )
	)
       )
       (if content-id-header
	   (setf (cache-content-id cache-entry) (send content-id-header :body))
	   nil
       )
       (if encoding-header
	   (let ((body (string-trim *whitespace-chars*
				    (send encoding-header :body)
		       )
		 )
		)
	        (setf (cache-content-transfer-encoding cache-entry)
		      (if (equal body "")
			  :7bit
			  (intern (string-upcase body) 'keyword)
		      )
	        )
	        (setf (cache-content-encoded-p cache-entry)
		      (not (equal (cache-content-transfer-encoding
				    cache-entry
				  )
				  :7bit
			   )
		      )
		)
	   )
	   nil
       )
       (if content-type-header
	   (let ((*default-content-type* :Text)
		 (*default-content-subtype* :Default)
		)
	        (parse-content-type content-type-header)
	        (let ((type (get content-type-header :Type))
		      (subtype (get content-type-header :SubType))
		      (params (get content-type-header :Parameters))
		     )
		     (setf (cache-content-type cache-entry) type)
		     (setf (cache-content-subtype cache-entry) subtype)
		     (setf (cache-content-type-parameters cache-entry)
			   params
		     )
		     (send *edit-server* :Put-Task :Imap-Parse-Data-Flags
			   (list :Flush-Display-Cache-For cache-entry t)
		     )
		     (Process-Content-Type type subtype
		       content-type-header cache-entry mailstream force-p
		     )
		)
	   )
	   nil
       )
  )
)

(defun can-handle-content-type-p (message)
  (can-handle-content-type-p-internal
    (cache-content-type message)
    (cache-content-subtype message)
  )
)

;===============================================================================

(defmethod maybe-decode-message
	   ((encoding-type t) cache-entry mailstream)
  (ignore cache-entry mailstream)
  nil
)

(defmethod maybe-decode-message
	   ((encoding-type (eql nil)) cache-entry mailstream)
  (ignore cache-entry mailstream)
  nil
)

(defmethod (mail:basic-header :Set-Body) (to)
  (setq mail:body to)
)

(defmethod maybe-decode-message
	   :Around ((encoding-type t) cache-entry mailstream)
  (ignore cache-entry mailstream)
  (or (not (cache-content-encoded-p cache-entry))
      (let ((encoding-header
	       (find :Content-Transfer-Encoding
		     (cache-parsed-headers cache-entry)
		     :Key 'Header-Type
	       )
	     )
	    )
	    (multiple-value-bind (changed-p surgically-p)
		(clos:call-next-method)
	      (if changed-p
		  (progn (setf (cache-content-encoded-p cache-entry) nil)
			 (setf (cache-old-content-transfer-encoding cache-entry)
			       (cache-content-transfer-encoding cache-entry)
	                 )
			 (setf (cache-content-transfer-encoding cache-entry)
			       nil
			 )
		  )
		  nil
	      )
	      (if surgically-p
		  (progn (setf (cache-surgically-modified-p cache-entry) t)
			 (send encoding-header :Set-Body "")
		  )
		  nil
	      )
	    )
      )
  )
)

(defmethod maybe-decode-message
	   ((encoding-type (eql :base64)) cache-entry mailstream)
  (ignore mailstream)
  (ndecode-base64 (cache-rfc822text cache-entry))
  (values t t)
)

(defmethod maybe-decode-message
	   ((encoding-type (eql :Quoted-Printable)) cache-entry mailstream)
  (ignore mailstream)
  (ndecode-quoted-printable (cache-rfc822text cache-entry))
  (values t t)
)

(defmethod maybe-decode-message
	   ((encoding-type (eql :7bit)) cache-entry mailstream)
  (ignore cache-entry mailstream)
  nil
)

;===============================================================================

;;; Content Transfer Encoding stuff.

(defun encode-base64 (from-string &optional
		      (into-string
			(make-array (ceiling (* 1.5 (length from-string)))
				    :Element-Type 'string-char :Fill-Pointer t
			)
		      )
		      (start-index 0)
		     )
  (loop with output-index = 0
	with in-length = (length from-string)
	for char-index from start-index below in-length by 3
	for char0 = (char-code (aref from-string char-index))
	for char1 = (if (>= (+ 1 char-index) in-length)
			0
			(char-code (aref from-string (+ 1 char-index)))
		    )
	for char2 = (if (>= (+ 2 char-index) in-length)
			0
			(char-code (aref from-string (+ 2 char-index)))
		    )
	do (setf (aref into-string output-index)
		 (aref *base64-mapping-table* (ldb #o0206 char0))
	   )
	   (incf output-index)
	   (setf (aref into-string output-index)
		 (aref *base64-mapping-table*
		       (+ (ash (ldb #o0002 char0) 4) (ldb #o0404 char1))
		 )
	   )
	   (incf output-index)
	   (setf (aref into-string output-index)
		 (if (>= (+ 1 char-index) in-length)
		     #\=
		     (aref *base64-mapping-table*
			   (+ (ash (ldb #o0004 char1) 2) (ldb #o0602 char2))
		     )
		 )
	   )
	   (incf output-index)
	   (setf (aref into-string output-index)
		 (if (>= (+ 2 char-index) in-length)
		     #\=
		     (aref *base64-mapping-table* (ldb #o0006 char2))
		 )
	   )
	   (incf output-index)
	   (if (>= (mod output-index 70) 65)
	       (progn (setf (aref into-string output-index) #\newline)
		      (incf output-index)
	       )
	       nil
	   )
	finally (setf (fill-pointer into-string) output-index)
  )
  into-string
)

(defun get-next-base64-char (from-string char-index in-length)
  (loop until (>= char-index in-length)
	for char = (aref from-string char-index)
	do (incf char-index)
	until (or (and (char>= char #\A)
		       (char<= char #\Z)
		  )
		  (and (char>= char #\a)
		       (char<= char #\z)
		  )
		  (and (char>= char #\0)
		       (char<= char #\9)
		  )
		  (member char '(#\+ #\/ #\=) :Test #'char=)
	      )
	finally
	  (return (values (if (> char-index in-length) nil char) char-index))
  )
)

(defun shuffle-array-portion (string from-start from-end to-start)
  (if (< to-start from-end)
      (loop for index1 from (- from-end 1) downto from-start
	    for index2 from (- (+ to-start (- from-end from-start)) 1) by -1
	    do (setf (aref string index2) (aref string index1))
      )
      (copy-array-portion
	string from-start from-end
	string to-start (+ to-start (- from-end from-start))
      )
  )
)

(defun nencode-base64 (string)
  (if (and (> (array-total-size string) (* 1.35 (array-active-length string)))
	   (>= (array-leader-length string) 1)
      )
      (let ((new-start
	      (- (array-total-size string) (array-active-length string))
	    )
	   )
	   (shuffle-array-portion
	     string 0 (array-active-length string)
	     new-start
	   )
	   (setf (fill-pointer string) (array-total-size string))
	   (Encode-Base64 string string new-start)
      )
      (ferror nil "Cannot encode string into itself.")
  )
)

(defun decode-base64 (from-string &optional
		      (into-string
			(make-array (ceiling (* 0.8 (length from-string)))
				    :Element-Type 'string-char :Fill-Pointer t
			)
		      )
		     )
  (loop with char0 = 0
	with char1 = 0
	with char2 = 0
	with char3 = 0
	with output-index = 0
	with in-length = (length from-string)
	with char-index = 0
	do (setq char0 nil char1 nil char2 nil char3 nil)
	   (multiple-value-setq (char0 char-index)
	     (Get-Next-Base64-Char from-string char-index in-length)
	   )
	   (multiple-value-setq (char1 char-index)
	     (Get-Next-Base64-Char from-string char-index in-length)
	   )
	   (multiple-value-setq (char2 char-index)
	     (Get-Next-Base64-Char from-string char-index in-length)
	   )
	   (multiple-value-setq (char3 char-index)
	     (Get-Next-Base64-Char from-string char-index in-length)
	   )
	when (not (and char0 char1 char2 char3))
	do (setf (fill-pointer into-string) output-index)
	   (return nil)
	do (let ((mapped-char0
		   (aref *inverse-base64-mapping-table* (char-code char0))
		 )
		 (mapped-char1
		   (aref *inverse-base64-mapping-table* (char-code char1))
		 )
		 (mapped-char2
		   (aref *inverse-base64-mapping-table* (char-code char2))
		 )
		 (mapped-char3
		   (aref *inverse-base64-mapping-table* (char-code char3))
		 )
		)
	        (setf (aref into-string output-index)
		      (+ (ash mapped-char0 2) (ldb #o0402 mapped-char1))
		)
		(incf output-index)
		(setf (aref into-string output-index)
		      (+ (ash (ldb #o0004 mapped-char1) 4)
			 (ldb #o0204 (or mapped-char2 0))
		      )
		)
		(if mapped-char2 (incf output-index) nil)
		(setf (aref into-string output-index)
		      (+ (ash (ldb #o0002 (or mapped-char2 0)) 6)
			 (or mapped-char3 0)
		      )
		)
		(if mapped-char3 (incf output-index) nil)
	   )
	finally (setf (fill-pointer into-string) output-index)
  )
  into-string
)

(defun ndecode-base64 (string)
  (if (and (array-leader-length string) (>= (array-leader-length string) 1))
      (Decode-Base64 string string)
      (ferror nil "String has no fill pointer.")
  )
)


;===============================================================================

(defun encode-quoted-printable
       (from-string &optional
	(into-string
	  (make-array (ceiling (* 1.5 (length from-string)))
		      :Element-Type 'string-char :Fill-Pointer t
		      :Adjustable t
	  )
	)
	(start 0)
       )
  (loop with out-index = 0
	with chars-this-line = 0
	for from-index from start below (length from-string)
	for char = (aref from-string from-index)
	for code = (char-code char)
	do (incf chars-this-line)
	   (cond ((>= chars-this-line 76)
		  (setf (aref into-string out-index) #\=)
		  (incf out-index)
		  (setf (aref into-string out-index) #\newline)
		  (incf out-index)
		 )
		 ((and (or (= char (char-code #\space))
			   ;;; Maybe should be 32 and 9
			   (= char (char-code #\tab))
		       )
		       (or (= from-index (- (length from-string) 1))
			   (not (char= (aref from-string (+ 1 from-index))
				       #\newline
				)
			   )
		       )
		  )
		  (setf (aref into-string out-index) char)
		  (incf out-index)
		 )
		 ((or (and (>= code 33) (<= code 57))
		      (and (>= code 59) (<= code 126))
		      (char= #\newline char)
		  )
		  (setf (aref into-string out-index) char)
		  (incf out-index)
		 )
		 (t (setf (aref into-string out-index) #\=)
		    (incf out-index)
		    (let ((translated-ascii
			    (case char
			      (#\Backspace 8)
			      (#\Tab 9)
			      (#\Linefeed 10)
			      (#\Page 12)
			      (#\Rubout 127)
			      (otherwise code)
			    )
			  )
			 )
		    (setf (aref into-string out-index)
			  (aref *Hex-Char-Table* (ldb #o0404 translated-ascii))
		    )
		    (incf out-index)
		    (setf (aref into-string out-index)
			  (aref *Hex-Char-Table* (ldb #o0004 translated-ascii))
		    )
		    (incf out-index)
		    )
		 )
	   )
	finally (setf (fill-pointer into-string) out-index)
  )
  into-string
)

(defun hexchar-to-nibble (char)
  (cond ((and (char>= char #\A) (char<= char #\F))
	 (+ 10 (- (char-code char) (char-code #\A)))
	)
	((and (char>= char #\a) (char<= char #\f))
	 (+ 10 (- (char-code char) (char-code #\a)))
	)
	(t (- (char-code char) (char-code #\0)))
  )
)

(defun decode-quoted-printable
       (from-string &optional
	(into-string
	  (make-array (length from-string)
		      :Element-Type 'string-char :Fill-Pointer t
		      :Adjustable t
	  )
	)
	(start 0)
       )
  (loop with out-index = 0
	with from-index = start
	until (>= from-index (length from-string))
	for char = (aref from-string from-index)
	for new-char =
	   (cond ((char= char #\=)
		  (if (char= (aref from-string (+ 1 from-index)) #\newline)
		      (progn (incf from-index) #\newline)
		      ;;; Must be hex.
		      (let ((code (+ (ash (Hexchar-To-Nibble
					    (aref from-string (+ 1 from-index))
					  )
					  4
				     )
				     (Hexchar-To-Nibble
				       (aref from-string (+ 2 from-index))
				     )
				  )
			    )
			   )
			   (incf from-index 3)
			   (case code
			     (8 #\Backspace)
			     (9 #\Tab)
			     (10 #\Linefeed)
			     (12 #\Page)
			     (127 #\Rubout)
			     (otherwise (code-char code))
			   )
		      )
		  )
		 )
		 (t (incf from-index) char)
	   )
	do (setf (aref into-string out-index) new-char)
	   (incf out-index)
	finally (setf (fill-pointer into-string) out-index)
  )
  into-string
)

(defun ndecode-quoted-printable (string)
  (if (and (array-leader-length string) (>= (array-leader-length string) 1))
      (Decode-Quoted-Printable string string)
      (ferror nil "String has no fill pointer.")
  )
)