;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

(defun process-status-pane-default-number-of-lines ()
"A functional hook for the selection of the number of lines in the status pane."
  *process-status-pane-default-number-of-lines*
)

(defun process-status-window-inits ()
"A functional hook for the selection of init plist items for the status window."
  *user-process-status-window-inits*
)

(defflavor process-status-pane
	   ((old-states nil))  ;;; A cache of the process states last time we
	                       ;;; displayed.
	   (tv:process-mixin tv:window)
  (:Default-Init-Plist
    :Font-Map *process-status-window-font-map*
    :Label nil
    :Blinker-P nil
    :Process '(process-status-top-level)
    :More-P nil
  )
  (:Documentation
"A flavor of windows that shows the status of interesting things
like background processes in a little display in the mailer window.
"
  )
)

(defmethod (Process-Status-Pane :After :Init) (ignore)
"Make sure we're running."
  (tv:process-reset-and-enable tv:process)
  (pushnew self *all-daemons*)
)

(defmethod (Process-Status-Pane :After :Kill) (&rest ignore)
  (setq *all-daemons* (remove self *all-daemons*))
)

(defun coerce-to-process (x)
"Coerces something to a process."
  (etypecase x
    (si:process x)
    (symbol (if (not (equal (symbol-value x) x))
		(coerce-to-process (symbol-value x))
		(yw-error "~S cannot be coerced into a process." x)
	    )
    )
    (tv:process-mixin (coerce-to-process (send x :Process)))
  )
)

(defmethod (Process-Status-Pane :compute-statuses) ()
"Returns a list of the process statuses of the processes that we're interested
in.
"
  (loop for (process) in *processes-to-show-in-status-window*
	collect (tv:peek-whostate (coerce-to-process process))
  )
)

(defwhopper (Process-Status-Pane :end-of-page-exception) ()
"Makes sure we throw out when we hit the end of page."
  ;;; Catch-error just in case we aren't inside out normal redisplay catch.
  (catch-error (throw 'process-status 'end-of-page) nil)
  (continue-whopper)
)

(defmethod (Process-Status-Pane :maybe-display-statuses)
	   (&optional (force-p nil))
"Displays the statuses of processes for the mailer in the status window if
they have changed or if force-p is true.  First it processes the things to do
in *other-things-to-do-in-status-window*, then it deals with the processes in
*processes-to-show-in-status-window*.
"
  (catch 'process-status
    (let ((new (send self :Compute-Statuses)))
	 (if (or force-p (not (equalp new old-states)))
	     (progn (setq old-states new)
		    (send self :Clear-Window)
		    (loop for fun in *other-things-to-do-in-status-window* do
			  (funcall fun self)
		    )
		    (loop for (process format-string)
			  in *processes-to-show-in-status-window*
			  for state on new
			  do (format self format-string)
			     (format self "~A"
				     (tv:fontify-string (first state) 1)
			     )
			  when (rest state) do (format self "; ")
		    )
	     )
	     nil
	 )
    )
  )
)

(defmethod (Process-Status-Pane :string-out)
	   (string &optional (start 0) end color)
"Prints a string on the status pane.  Does it char by char to make sure we're
getting the right fonts.
"
  (loop for i
	from start
	to (or (and end (min end (- (length string) 1)))
	       (- (length string) 1)
	   )
	do (if (> (+ (+ (tv:sheet-cursor-x self) tv:width)
		     (- (tv:sheet-inside-right self)
			(tv:sheet-char-width self))))
	       (tv:sheet-tyo self (aref string i) nil color)
	       nil
	   )
  )
)

(defmethod (Process-Status-Pane :After :Refresh) (&rest ignore)
"Makes sure we redisplay properly."
  (send self :maybe-display-statuses t)
)

(defun process-status-loop (window)
"The top level loop for the status wundow process.  Only does anything if we are
exposed.
"
  (loop do
	(process-wait "Wait for exposure" #'(lambda (win) (send win :Exposed-P))
		      window
        )
	(send window :Maybe-Display-Statuses)
	(sleep *process-staus-pane-sleep-interval* "Process status check sleep")
  )
)

(defun process-status-top-level (window)
"The top level for the process status window."
  (loop (error-restart ((abort error) "Return to process status top level.")
	  (process-status-loop window)
	)
	(tv:deselect-and-maybe-bury-window window :first)
  )
)

;-------------------------------------------------------------------------------

(defun system-output-pane-default-number-of-lines ()
"A functional hook for the selection of the number of lines in the output pane."
  *system-output-pane-default-number-of-lines*
)

(defun system-output-window-inits ()
"A functional hook for the selection of init plist items for the output window."
  *user-system-output-window-inits*
)

(defflavor system-output-pane
	   ()
	   (w:scroll-bar-mixin
	    tv:text-scroll-window
	    tv:window
	   )
  (:Default-Init-Plist
    :Font-Map *system-output-window-font-map*
    :Label nil
  )
  (:Documentation "A scrolling outpu window for status messages.")
)

(defmethod (system-output-pane :print-item) (item line-no item-no)
"Just princs the item into the window."
  (ignore line-no item-no)
  (princ item self)
)

(defun split-string-into-lines-internal
       (window string start-index result)
"Given a string and a window to print the string into, splits the string into
lines that can be displayed without wrapping on that window.  Start-index is
the start point in the string from which we work.
"
  (declare (optimize (speed 3) (safety 0)))
  (if (>= start-index (- (length string) 1))
      (nreverse result)
      (loop for end from start-index to (- (length string) 1)
	    for (new-x new-y)
	    = (multiple-value-list
		(tv:sheet-compute-motion window 0 0 string start-index end)
	      )
	    while (= 0 new-y)
	    finally (return (Split-String-Into-Lines-internal
			      window string
			      (if (eql #\newline (aref string (- end 1)))
				  end
				  (- end (if (equal new-y 1) 2 1))
			      )
			      (cons (subseq string start-index
					    (- end (if (> new-y 0) 1 0))
			            )
				    result
			      )
			    )
		    )
       )
  )
)

(defun split-string-into-lines (window string &optional (start-index 0))
"Given a string and a window to print the string into, splits the string into
lines that can be displayed without wrapping on that window.  Start-index is
the start point in the string from which we work.
"
  (split-string-into-lines-internal window string start-index nil)
)

(defmethod (System-Output-Pane :Set-Number-Of-Items) (to)
"Sets the number of items in the window to To."
  (loop with items = (symeval-in-instance self 'tv:displayed-items)
	for i from 0 below (array-active-length items)
	do (setf (aref items i) nil)
  )
  (loop with items = (send self :Items)
	for i from 0 below (array-active-length items)
	do (setf (aref items i) nil)
  )
  (Setf (array-leader tv:items 0) to)
)

(defmethod (System-Output-Pane :Reset) ()
"Clears the window and resets the items in it."
  (send self :Set-Number-Of-Items 0)
  (send self :Clear-Window)
)

(defmethod (System-Output-Pane :add-new-item) (control-string &rest args)
"Adds a new format control string + args as a new item in the window
at the bottom and scrolls to it.
"
  (let ((string (apply #'format nil control-string args)))
       (let ((strings (split-string-into-lines self string)))
	    (loop for str in strings do
		  (send self :Append-Item str)
	    )
       )
  )
  (send self :Scroll-To
	(max 0
	     (- (send self :Number-Of-Items) (tv:sheet-number-of-inside-lines))
	)
	:Absolute
  )
)

(defun find-mail-window (from &optional (error-p t) (create-p nil))
"Finds a mailer window from the starting point From.  If no-error-p is true then
if it fails to find a mailer window then it returns nil.
"
  (or (typecase from
	(tv:screen (any-mailer (not error-p)))
	(mail-control-window from)
	(tv:sheet (Find-Mail-Window (tv:sheet-superior from) error-p create-p))
	(imap-stream-mixin
	 (find-mail-window (send from :Owning-Window) error-p create-p)
	)
	(otherwise (any-mailer (not error-p)))
      )
      (and create-p (get-mail-control-window))
      (and error-p (ferror nil "Cannot find mailer window from ~S." from))
  )
)

(defun format-scroll-window (using control-string &rest args)
"Formats a string into the scrolling output window of the MCW denoted by Using."
  (let ((frame (Find-Mail-Window using nil)))
       (if frame
	   (let ((pane (send frame :system-output-pane)))
		(lexpr-send pane :add-new-item control-string args)
	   )
	   (apply 'tv:notify tv:selected-window control-string args)
       )
  )
)


;-------------------------------------------------------------------------------


(defun mailbox-selector-font-map ()
"A functional hook that delivers the font map for the mailbox selector window."
  *mailbox-selector-font-map*
)

(defun mailbox-selector-default-number-of-lines ()
"A functional hook that controlls the number of lines in the mailbox selector."
  *mailbox-selector-default-number-of-lines* 
)


(defflavor mailbox-selector
	   (owner (item-mappings nil)) ;;; Our superior.
	   (tv:top-label-mixin
	    w:scroll-bar-mixin
	    tv:scroll-mouse-mixin
	    tv:scroll-window
	    tv:dont-select-with-mouse-mixin
	    tv:Essential-Mouse
	   )
  (:Default-Init-Plist
    :Label nil
    :Font-Map (mailbox-selector-font-map)
    :Activate-p t
    :Scroll-Bar 2
    :Scroll-bar-always-displayed nil
    :Expose-P nil
    :Activate-P t
    :Save-Bits t
    :Truncation t
    :Deexposed-Typeout-Action :Permit
  )
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  (:Documentation
"A scrolling window with mouse sensitive items each one of which represents a
summary window.  Clicking on an item in this window has the same effect as
clicking on the background of a summary window.
"
  )
)

(defmethod (Mailbox-Selector :forget-all) ()
"Forgets all of the summary windows associated with self."
  (send self :Expose)
  (send self :set-display-item (list nil))
  (setq item-mappings nil)
  (send self :Clear-Window)
)

(defmethod (mailbox-selector :selection-substitute) ()
"Points to our superior."
  (setq tv:selection-substitute (send tv:superior :selection-substitute))
  tv:selection-substitute
)

(defmethod (Mailbox-Selector :New-Summary-Window) (window)
"Adds a new summary window to our list, putting in a new item in the
window for it at the top.
"
  (let ((item (list nil
		    (tv:scroll-parse-item
		      (list :Mouse (list :Summary-Window window)
			    :String (send window :Make-Label)
			    nil nil '("~A")
		      )
		    )
	      )
	)
       )
       (push (list window item) item-mappings)
       (send self :Set-Display-Item
	     (cons nil (cons item (rest tv:display-item)))
       )
       (send self :Scroll-To 0)
       (send self :Redisplay)
  )
)


(defmethod (Mailbox-Selector :make-current) (spec)
"Makes the item for the window spec the current item by moving it to the top of
the item list and scrolling there.
"
  (let ((entry (assoc spec item-mappings :Test #'equalp)))
       (if entry
	   (progn (send self :set-display-item
			(cons nil
			      (cons (second entry)
				    (remove (second entry)
					    (rest tv:display-item)
				    )
			      )
			)
		  )
		  (send self :Scroll-To 0)
	   )
	   (send self :New-Summary-Window spec)
       )
       (send self :Redisplay)
  )
)

(defmethod (Mailbox-Selector :Update-Label-For) (spec)
"Updates the label for a window spec.  This involves recomputing the item
in the window and refreshing.  We don't need to be terribly smart here because
there aren't ever a really huge number of items in this window.
"
  (let ((entry (assoc spec item-mappings :Test #'equalp)))
       (if entry
	   (let ((new-item
		   (list nil
			 (tv:scroll-parse-item
			   (list :Mouse (list :Summary-Window spec)
				 :String (send spec :Make-Label)
				 nil nil '("~A")
	   
			   )
			 )
		   )
		 )
		)
		(send self :set-display-item
		      (subst new-item (second entry) tv:display-item)
		)
	        (setf (second entry) new-item)
		(send self :Redisplay)
	   )
	   (yw-warn "~&!!! Summary window not found for spec ~S" spec)
       )
  )
)

(defmethod (Mailbox-Selector :update-all-labels) ()
"Flushes and updates the items for all of the items in the window."
  (loop for spec in (mapcar #'first item-mappings) do
	(send self :Update-Label-For spec)
  )
)

(defmethod (Mailbox-Selector :After :Remove-Summary-Window) (&rest ignore)
"When we remove a summary window, maybe saves the address database."
  (if (not (rest tv:display-item))
      (maybe-save-address-database)
      nil
  )
)

(defmethod (Mailbox-Selector :Remove-Summary-Window)
      (window &optional (from-streams (send window :Mailstreams)) (quiet-p nil))
"Removes a summary window from our item list.  If quiet-p is true then if the
window is not there then we get no barfage.
"
  (let ((entry (assoc window item-mappings :Test #'equalp)))
       (if entry
	   (progn (setq item-mappings (remove entry item-mappings))
		  (send self :Set-Display-Item
			(remove (second entry) tv:display-item)
		  )
		  (send self :Scroll-To 0)
		  (send self :Redisplay)
		  (if *close-mailbox-when-all-summary-windows-closed*
		      (progn (loop for stream in from-streams
				   do (send stream :Remove-Associated-Window
					    window
				      )
			     )
			     (loop for stream in from-streams
				   when
				    (and stream
					 (not (send stream :Associated-Windows))
				    )
				   do
				    (with-daemons-reset-and-arrested ()
				      (close stream)
				    )
			     )
		      )
		      nil ;;; Do nothing whatever.
		  )
	   )
	   (if quiet-p
	       nil
	       (yw-warn "~&!!! Summary window not found for window ~S" window)
	   )
       )
  )
)

(defmethod (mailbox-selector :mouse-click) (button x y &aux op)
"A mouse click method for the summary window.  Uses the
*background-mouse-blip-methods* list to decide what to do when
we get a blip on an item, i.e. do the same thing that we would do if the
user blipped on the background of a summary window.  This uses the normal mouse
sensitive item handling.  If the user blips on our background then use
the *mailbox-selector-background-mouse-blip-methods* to decide what to do.
"
  (declare (special *background-mouse-blip-methods*
		    *mailbox-selector-background-mouse-blip-methods*
           )
  )
  (multiple-value-bind (item type)
      (send self :mouse-sensitive-item x y)
    (cond ((null item)
	   (let ((entry
		   (find-if #'(lambda (X)
				(and (char-equal button
						 (background-blip-Mouse-Char X)
				     )
				     (or (not (background-blip-applicable-if x))
					 (funcall
					   (background-blip-applicable-if x)
					   (send self :Owner)
					 )
				     )
				)
			      )
			      *mailbox-selector-background-mouse-blip-methods*
		   )
		 )
		)
	        (if entry
		    (process-run-function
		      '(:Name "Mailbox Selector Background" :Priority 1)
		      #'(lambda (window)
			  (send window (background-blip-method entry))
			)
		      self
		    )
		    nil
		)
	   )
	  )
	  (t
	   (cond
	     ((null item) nil)
	     ((or (null type)
		  (setq op (first (cdr (assoc type tv:type-alist :test #'eq)))))
	      ;; psych out :BUTTONS --- Copy of code in
	      ;; (TV:BASIC-MENU :MOUSE-BUTTONS)
	      (cond
		((and (consp item)
		      (>= (length item) 3)
		      (eq (second item) :buttons))
		 (setq item (nth (char-mouse-button button) (third item)))))
	      (tv:blinker-set-visibility tv:item-blinker nil)
	      (send self :execute (if op
				      (list* nil op item)
				      item))
	      t)
	     (t (send self :force-kbd-input (list type item self button))
		t)))
    )
  )
)

(defmethod (mailbox-selector :Who-line-documentation-string) ()
"A mouse doc method for the summary window.  Uses the
*background-mouse-blip-methods* list to decide what to do when
we are over an item, i.e. do the same thing that we would do if the
user was over the background of a summary window.  If the user is over the
background then use the *mailbox-selector-background-mouse-blip-methods*
to decide what to show.
"
  (declare (special *background-mouse-blip-methods*
		    *mailbox-selector-background-mouse-blip-methods*
           )
  )
  (multiple-value-bind (item ignore) (Get-Any-Mouse-Sensitive-Item self)
    (if item
        (nconc (loop for X in *background-mouse-blip-methods*
		     when (or (not (background-blip-applicable-if x))
			      (funcall (background-blip-applicable-if x)
				       (send self :Owner) (third item)
			      )
			  )
		     nconc (List (Background-Blip-Mouse-Key X)
				 (Background-Blip-Doc-String X)
			   )
	       )
	      '(:Mouse-R-2 "Bring up the system menu"
		:Sort ""
		:Smart-Newlines ""
	       )
	)
        (nconc (loop for X in *mailbox-selector-background-mouse-blip-methods*
		     when (or (not (background-blip-applicable-if x))
			      (funcall (background-blip-applicable-if x)
				       (send self :Owner)
			      )
			  )
		     nconc (List (background-Blip-Mouse-Key X)
				 (background-Blip-Doc-String X)
			   )
	       )
	      '(:Mouse-R-2 "Bring up the system menu"
		:Sort ""
		:Smart-Newlines ""
	       )
	)
    )
  )
)

;-------------------------------------------------------------------------------

(defflavor yw-prompt-window
	   ()
	   (Ucl:selective-features-mixin
	    ucl:basic-command-loop
	    tv:auto-scrolling-mixin tv:notification-mixin tv:window)
  (:Default-Init-Plist
    :Remove-Features '(:Lisp-Typein)
    :More-P nil
    :Label *no-mailbox-open-text*
    :Prompt 'compute-prompt-from-superior
    :Save-Bits t
    :Deexposed-Typeout-Action :permit
  )
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  (:Documentation "The prompt window for the mailer.")
)

(defun compute-prompt-from-superior ()
"Asks our superior to get a prompt for us."
  (send (send self :Superior) :Eval-Inside-Yourself '(compute-prompt))
)

(defmethod (Yw-prompt-window :After :Clear-Screen) (&rest ignore)
"Reprompts when we clear screen."
  (send self :send-if-handles :handle-prompt)
)

(defwhopper (yw:Yw-prompt-window :any-tyi) (&rest args)
"A magic whopper on reading chars so that we can deal with mouse blips coming
in as arguments, pretending they are typein.
"
  (declare (special *expecting-blip-from-mailbox-selector*
		    *expecting-blip-from-message-window*
		    *expecting-blip-for-message-number*
           )
  )
  (let ((result (lexpr-continue-whopper args)))
       (if (consp result)
	   (cond ((or (and (boundp '*expecting-blip-from-mailbox-selector*)
			    *expecting-blip-from-mailbox-selector*
		       )
		       (and (boundp '*expecting-blip-from-message-window*)
			    *expecting-blip-from-message-window*
		       )
		  )
		  (if (and (boundp '*expecting-blip-from-mailbox-selector*)
			   *expecting-blip-from-mailbox-selector*
		      )
		      (send self :process-expected-selector-blip result)
		      (send self :process-expected-header-blip result)
		  )
		  (lexpr-send self :Any-Tyi args)
		 )
		 ((and (boundp '*expecting-blip-for-message-number*)
		       *expecting-blip-for-message-number*
		  )
		  (send self :process-expected-message-blip result)
		  (lexpr-send self :Any-Tyi args)
		 )
		 ((send tv:superior :can-interpret-input result)
		  (send tv:superior :Interpret-Input nil result)
		  (lexpr-send self :Any-Tyi args)
		 )
		 ((member (first result)
			  '(tv:redisplay-rubout-handler) :Test #'eq
		  )
		  (lexpr-send self :any-tyi args)
		 )
		 (t (beep))
	   )
	   result
       )
  )
)

(defun spacey-p (string)
"True if String contains any whitespace chars."
  (loop for char in *whitespace-chars*
	when (position char string :Test #'char=)
	return t
	finally (return nil)
  )
)

(defun Parallel-Force-String-Input
       (window text &optional (string-quotes-p t) (trailing-space-p nil))
"Forces kbd input for the text into the window Window.  If string-quotes-p
is true then it wraps the text in strng quotes.  If trailing-space-p is true
then it puts in a space too.  It does this all in a process that it spins
off for the purpose.
"
  (process-run-function "Insert-text"
     #'(lambda (window text)
	 (if (and string-quotes-p (spacey-p text))
	     (send window :Force-Kbd-Input #\")
	     nil
	 )
	 (send window :Force-Kbd-Input text)
	 (if (and string-quotes-p (spacey-p text))
	     (send window :Force-Kbd-Input #\")
	     nil
	 )
	 (if trailing-space-p (send window :Force-Kbd-Input #\space) nil)
	)
	window text
  )
)

(defmethod (Yw-prompt-window :Process-Expected-Message-Blip) (blip)
"Processes a blip that we were expecting of type :message."
  (if (equal (first blip) :Message)
      (Parallel-Force-String-Input
	self (format nil "~D" (number-of (third (second blip)))) nil
      )
      (beep)
  )
)

;;; Modified TI code.
;;; Copied from stream-mixin.
(defmethod (yw-prompt-window :any-tyi) (&optional ignore &aux ch idx)
"Just like the normal ANY-TYI method for windows only hacks on the rhb
to set the cursor as we go.
"
  ;; If there are characters to be read located in the rubout handler's buffer
  ;; then get the input from there.  Otherwise get input from the window's
  ;; IO buffer.
  (cond ((> (tv:rhb-fill-pointer) (setq idx (tv:rhb-scan-pointer)))
	 (setf (tv:rhb-scan-pointer) (1+ idx))
	 ;;; change here.  JPR.
	 (tv:rh-set-position (tv:rh-scan-pointer))
	 (or (setq ch (aref tv:rubout-handler-buffer idx))
	     (ferror nil "EOF on input from a window.")))
	((not tv:rubout-handler)
	 (when (null (setq ch (tv:kbd-io-buffer-get tv:io-buffer t)))
	   ;; No character was waiting for us.
           ;; We need to wait until there is one.
	   (send self :notice :input-wait)
	   (setq ch (tv:kbd-io-buffer-get tv:io-buffer)))
	 ;; When we have a keypad character and the keypad bit for the window is
	 ;; off then make it a non-keypad character.  By default, all characters
	 ;; generated from keys on the keypad have their keypad bit turned on.
	 (when (and (or (characterp ch) (integerp ch))
		    (zerop (tv:sheet-keypad-enable))
		    (plusp (char-keypad ch)))
	   (setq ch (set-char-bit ch :keypad nil))
	   ;; The following trick is to make sure that if someone does an untyi
	   ;; there won't be an error.  One can only untyi something from a
	   ;; buffer if it the same as was gotten from the buffer.
	   (tv:io-buffer-put tv:io-buffer ch)
	   (tv:io-buffer-get tv:io-buffer ch)))
	(t (or (setq ch (funcall tv:stream-mixin-rubout-handler))
	       (ferror nil "EOF on input from a window."))))
  (tv:char-int-mouse-button-blip ch)
  ch)


(defmethod (yw-prompt-window :Process-Expected-Selector-Blip) (blip)
"Processes a blip on the mailbox selector that we were expecting."
  (declare (special *reading-a-filter*))
  (case (first blip)
    (:Summary-Window
     (cond ((and (boundp '*reading-a-filter*) *reading-a-filter*)
	    (let ((string (let ((filter (send (second (second blip)) :Filter)))
			       (if filter
				   (princ-to-string filter)
				   nil
			       )
			  )
		  )
		 )
		 (if (and string (not (equal string "")))
		     (Parallel-Force-String-Input self string nil t)
		     (beep)
		)
	    )
	   )
	   (t (let ((string (print-short-mailbox-name
			      (send (second (second blip)) :Mailstreams) t
			    )
		    )
		   )
		   (if (and string (not (equal string "")))
		       (Parallel-Force-String-Input self string)
		       (beep)
		   )
	      )
	   )
     )
    )
    (otherwise (beep))
  )
)

(defmethod (yw-prompt-window :Process-Expected-Header-Blip) (blip)
"Processes a blip on a message header that we were expecting."
  (declare (special *expecting-blip-from-message-window*))
  (if (equal :Message (first blip))
      (let ((field-extractor *expecting-blip-from-message-window*)
	    (message (first (send tv:superior :Interpret-Input t blip)))
	   )
	   (let ((string
		   (funcall field-extractor (cache-mailstream message) message)
		 )
		)
		(if (and string (not (equal string "")))
		    (Parallel-Force-String-Input self string)
		    (beep)
		)
	   )
      )
      (beep)
  )
)

(defmethod (Yw-prompt-window :process) ()
"Returns our superior's process."
  (send tv:superior :Process)
)

(defmethod (Yw-prompt-window :insert-text) (text)
"Inserts some text in our rhb."
  (tv:rh-insert-string text 0 nil t nil)
)

(defun remove-quoted-quotes (string)
  "Changes literal quotes (backslash quotes) into just quotes in the string." 
  (if (stringp string)
      (let ((index (search "\\\"" string :Test #'char-equal)))
	   (if index
	       (string-append (subseq string 0 index)
			      (remove-quoted-quotes (subseq string (+ 1 index)))
	       )
	       string
	   )
      )
      string
  )
)


;-------------------------------------------------------------------------------
