;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;;; Message descriptors...

(defun Reset-Display-Items-For
       (cache-entry &optional (new-message-number (cache-msg# cache-entry)))
  (setf (cache-header-display-string cache-entry) nil)
  (loop for item in (cache-all-items cache-entry)
	for window = (window item)
	for filter = (send window :Filter)
	for value-array-index =
	    (+ 1
	       (if filter
		   (position cache-entry
			     (send filter :computed-order-safe)
			     :Test #'eq
		   )
		   (loop for stream in (send window :Mailstreams)
			 until (eq stream (cache-mailstream cache-entry))
			 sum (send stream :Messagecnt) into sum
			 finally (return (+ sum (- new-message-number 1)))
		   )
	       )
	    )
	do 
;	   (setf (left item) nil)
;	   (setf (top item) nil)
;	   (setf (right item) nil)
;	   (setf (bottom item) nil)
	   (setf (object item) cache-entry)
	   ;;; !!!!{!!!!}  What about the mouse entry?
  )
)

(defmethod add-item ((me cache) item window)
"Adds a new item that is in the window to our list of items for self."
  (let ((items (remove-if #'(lambda (x) (equal window (window x)))
			  (cache-all-items me)
	       )
	)
       )
       (setf (cache-all-items me) (cons item items))
  )
)

(defmethod remove-item ((me cache) item &optional (display-p t))
"Removes an item that prints on Window from our list of items."
  (setf (cache-all-items me)
	(remove item (cache-all-items me) :Test #'eq)
  )
  (loop with window = (window item)
	with image = (send window :Screen-Image)
	for index from 0 below (array-dimension image 0)
	for this-item = (aref image index)
	while this-item
	when (eq me (object this-item))
	do (if display-p
	       (send window :Pop-Up-By-N item 1)
	       (setf (aref image index) nil)
	   )
  )
)

(defmethod remove-items-at-end
	   (items &optional (display-p t))
"Removes a bunch of items fro mthe end of the items for a window."
  (loop for item in items
	for cache = (Message-From-Display-Item item)
	do (setf (cache-all-items cache)
		 (remove item (cache-all-items cache) :Test #'eq)
	   )
  )
  (if (and items display-p)
      (loop with window = (window (first items))
	    with image = (send window :Screen-Image)
	    with top = (send window :Top-Item)
	    for index from 0 below (send window :Screen-Lines)
	    for this-item = (aref image index)
	    while this-item
	    for found-p =
		  (loop for item in items
			when (eq item this-item)
			do (return t)
		  )
	    when found-p
	    do (if (< top (- (send window :screen-lines) index))
		   (progn (send window :Scroll-To (- top) :Relative)
			  (send window :Flush-Lines (- index top)
				(- (send window :Screen-Lines))
			  )
		   )
		   (send window :Scroll-To (- index (send window :screen-lines))
			 :Relative
		   )
	       )
	       (return nil)
      )
      nil
  )
)

(defmethod get-and-format-header-display-string ((me cache))
"Causes the header string to be computed.  If we have no envelope then
we return :wait, which is an indication to the caller to get the envelope
and wait for the response before retrying.  If we have the envelope then
if calls format-header to compute the header string.
"
  (let ((maybe-subject (Cache-SubjectText me))
	(maybe-msgfrom (Cache-FromText me))
	(maybe-msgto   (Cache-ToText me))
	(MailStream    (cache-mailstream me))
	(number        (cache-msg# me))
       )
       (if (is-present (cache-envelope me))
	   (let ((MsgSubject 
		   (or (and (is-present maybe-subject) maybe-subject)
		       (MAP-Fetch-Subject MailStream number)
		   )
		 )
		 (Message-number (Cache-Msg# me))
		 (MsgFlags (copy-tree (Cache-Flags me)))
		 (MsgChars (Cache-RFC822Size me))
		 (msgfrom
		   (or (and (is-present maybe-msgfrom) maybe-msgfrom)
		       (MAP-Fetch-From MailStream (Cache-Msg# me))
		   )
		 )
		 (msgto
		   (or (and (is-present maybe-msgto) maybe-msgto)
		       (MAP-Fetch-To MailStream (Cache-Msg# me))
		   )
		 )
		)
		(setf (cache-flags-used-for-display me) (copy-list msgflags))
		(format-header-display-string me nil message-number MsgFlags
		  (Cache-InternalDate me) msgfrom msgto msgsubject msgchars
		)
	   )
	   :Wait
       )
  )
)

(defmethod prepare-for-display ((me cache))
"Prepares a descriptor for display.  Makes sure that we have already got and
cached the subject and from fields.
"
  (let ((val (Cache-SubjectText me)))
       (or (and (not (equal val :Unbound)) val)
	   (MAP-Fetch-Subject (cache-mailstream me) (cache-msg# me))
       )
  )
  (let ((val (Cache-FromText me)))
       (or (and (not (equal val :Unbound)) val)
	   (MAP-Fetch-From (cache-mailstream me) (cache-msg# me))
       )
  )
  (setf (cache-ready-to-recompute me) t)
)

(defmethod On-Screen-P ((me cache) item)
"Is true if the item is visible on the screen in window (some items may have
scrolled off the top of the window.
"
  (let ((window (window item)))
       (let ((image (send window :screen-image)))
	    (send window :Resynch nil)
	    (loop for i from 0 below (length image)
		  when (equal item (aref image i))
		  do (return i)
		  finally (return nil)
	    )
       )
  )
)

(defmethod redisplay ((me cache) item)
"Redisplays self on all interested windows."
  (let ((*force-redisplay* t))
       (declare (special *force-redisplay*))
       (send (window item) :Redisplay-Selected-Items (list item))
  )
)

(defmethod selected-p ((me cache))
  (cache-selected-p me)
)

(defmethod (setf selected-p) (new-value (me cache))
  (setf (cache-selected-p me) new-value)
)

(defmethod really-set-selected-p ((me cache) to)
"Sets the selected-p slot to true."
  (setf (cache-selected-p me) to)
  to
)

(defmethod (setf selected-p) :around (new-value (me cache))
"Makes sure that we redisplay ourselves when our selected status changes."
  (let ((old (cache-selected-p me)))
       (call-next-method)
       (if (not (equal old (selected-p me)))
	   (Refresh-On-All-Windows me)
	   nil
       )
  )
)

(defmethod Add-Selected-Type ((me cache) type)
"Adds a selected type to the set of selected types in selected-p."
  (setf (Selected-P me) (fs:cons-new type (Selected-P me)))
)

(defmethod refresh-on-all-windows ((me cache))
"Refreshes self on all of the windows that have an item for it, being clever
about only redisplaying on the windows that actually have us visible.
"
  (let ((items (all-items-for-message me)))
       (loop for item in items
	     when (and (On-Screen-P me item) (equal me (object item)))
	     do (Really-Set-Selected-P (object item) (cache-selected-p me))
		(redisplay me item)
       )
  )
)

(defmethod Remove-Selected-Type ((me cache) type)
"Removes a selected type from the set of selected types in selected-p."
  (setf (selected-p me) (remove type (selected-p me)))
)

(defmethod Address-Object
	   ((me cache) &optional (accessor 'envelope-from))
  (let ((env (cache-envelope me)))
       (and (is-present env)
	    (let ((from (funcall accessor env)))
	         (and (is-present from)
		      from
		      (progn (parse-address-component (first from))
			     (address-address-object (first from))
		      )
		 )
	    )
       )
  )
)

(defun all-items-for-message (cache-entry)
"Returns the list of acceptable items for Message."
   (let ((result nil))
        (loop for item in (cache-all-Items cache-entry)
	      for window = (window item)
	      for not-ok-p = (or (not window) (not (send window :Mailstreams)))
	      when not-ok-p
	      do (remove-item cache-entry item)
	      unless (or not-ok-p
			 (member window result :Key 'window :Test #'eq)
		     )
	      do (push item result)
	)
	result
   )
)

(defmethod Print-Self ((me cache) stream depth slashify)
"A print method for message descriptors.  If we are princing then we want
to produce a header-display-string for a header window.  If we don't
have an envelope yet then get-and-format-header-display-string will return
:wait.  This tells us to print out the *waiting-message-text* instead and
tell the task server to wait and recompute the header-display-string later.
"
  (declare (special *edit-server*))
  (ignore depth)
  (multiple-value-bind (result error-p)
    (catch-error
      (let ((*nopoint t))
	     (if (numberp (cache-msg# me))
		 (if (or slashify *dbg*)
		     (if (cache-header-display-string me)
			 (format stream "#<Msg ~A, ~>" (cache-msg# me)
				 (list (cache-header-display-string me) t
				       (Without-Tabs
					 (cache-header-display-string me)
				       )
				 )
			 )
			 (format stream "#<Msg ~A, ~>" (cache-msg# me))
		     )
		     (if (and (cache-header-display-string me)
			      (not (cache-ready-to-recompute me))
			 )
			 (format stream "~A" (cache-header-display-string me))
			 (let ((result
				 (Get-And-Format-Header-Display-String me)
			       )
			      )
			      (if (equal result :Wait)
				  (progn (setf (cache-header-display-string me)
					       (string-append
						 *waiting-message-text*
						 (format nil "~D"
							 (cache-msg# me))
					       )
					 )
					 (format stream "~A"
					    (cache-header-display-string me)
					 )
					 (send *edit-server* :Put-Task
					       :print-self
					       (list :Preempt-Header me)
					 )
				  )
				  (progn (setf (cache-header-display-string me)
					       result
					 )
					 (format stream "~A"
					    (cache-header-display-string me)
					 )
				  )
			      )
			      (setf (cache-ready-to-recompute me) nil)
			 )
		     )
		 )
		 (format stream "#<Msg {Uninitialised}>")
	     )
	   nil
      )
      nil
    )
    (if error-p
	(format stream "#<Error printing header-display-string>")
	result
    )
  )
)

;===============================================================================

(defmethod format-header-date
	   ((value (eql :Date)) message-number message-flags message-date
	    message-from message-to message-subject message-length-in-chars
	    day month year date-mode-args
	   )
"A method for formatting the date of a message in a header display
for the :date display type.
"
  (ignore message-number message-flags message-from message-to
	  message-subject message-length-in-chars message-date date-mode-args
  )
  (format t "~?"
	  (get *yw-date-print-mode* 'time:date-format)
	  (list day month (time:month-string month :short)
		nil (mod year 100.)
	  )
  )
)

(defmethod format-header-date
	   ((value (eql :Date-And-Time))
	    message-number message-flags message-date
	    message-from message-to message-subject message-length-in-chars
	    day month year date-mode-args
	   )
"A method for formatting the date of a message in a header display
for the :date-and-time display type.
"
  (ignore message-number message-flags message-from message-to
	  message-subject message-length-in-chars message-date date-mode-args
	  day month year
  )
  (time:print-universal-time (time:parse-universal-time message-date))
)

(defmethod format-header-date
	   ((value (eql :Brief-Date))
	    message-number message-flags message-date
	    message-from message-to message-subject message-length-in-chars
	    day month year date-mode-args
	   )
"A method for formatting the date of a message in a header display
for the :brief-date display type.
"
  (ignore message-number message-flags message-from message-to
	  message-subject message-length-in-chars message-date date-mode-args
	  day month year
  )
  (format t "~?" (get *yw-date-print-mode* 'time:date-format) date-mode-args)
)

(defmethod format-header-date
	   ((value t)
	    message-number message-flags message-date
	    message-from message-to message-subject message-length-in-chars
	    day month year date-mode-args
	   )
"A method for formatting the date of a message in a header display
for a display type that we don't understand.
"
  (ignore message-number message-flags message-from message-to
	  message-subject message-length-in-chars message-date date-mode-args
	  day month year
  )
  (format t "???~S" value)
)

(defmethod format-header-date
	   ((value (eql :Date-or-Time))
	    message-number message-flags message-date
	    message-from message-to message-subject message-length-in-chars
	    day month year date-mode-args
	   )
"A method for formatting the date of a message in a header display
for the :date-or-time display type.
"
  (ignore message-number message-flags message-from message-to
	  message-subject message-length-in-chars message-date date-mode-args
	  day month year
  )
  (multiple-value-bind (ignore ignore ignore d m y)
      (time:decode-universal-time (time:get-universal-time))
    (if (and (= d day) (= m month) (= y year))
	(multiple-value-bind (ignore mins hours)
	    (time:decode-universal-time
	      (safe-parse-universal-time message-date)
	    )
	  (format t "~2,'0,D:~2,'0,D " hours mins)
	)
	(format t "~?" (get *yw-date-print-mode* 'time:date-format)
		date-mode-args
	)
    )
  )
)

(defmethod yw-header-display-spec
       ((type (eql :Date))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of dates in message displays."
  (ignore message-number cache-entry message-flags message-from message-to
	  message-subject message-length-in-chars
  )
  (typecase value
    (keyword
     (format-header-date value message-number message-flags message-date
			 message-from message-to message-subject
			 message-length-in-chars day month year date-mode-args
     )
    )
    ((or cons number)
     (maybe-max-width "~V?" value (get *yw-date-print-mode* 'time:date-format)
			date-mode-args
     )
    )
    (otherwise (format t "???~S" value))
  )
)

(defun coerce-to-thin-string (string)
"Given a string makes sure that it is a thin string."
  (let ((thin-string (make-string (length string))))
       (loop for i from 0 below (length string) do
	     (setf (aref thin-string i) (int-char (char-code (aref string i))))
       )
       thin-string
  )
)

(defun extract-tabs (string start)
"Starting at the start index Start, the string has all absolute tab denotations
removed.  Thus \"foo200bar\" -> \"foobar\".
"
  (let ((start-index (string-search-set '(#\) string start)))
       (if start-index
	   (let ((stop-index
		   (string-search-set '(#\) string (+ 1 start-index))
		 )
		)
		(if start-index
		    (string-append (subseq string start-index (+ 1 stop-index))
				   (extract-tabs string (+ 1 stop-index))
		    )
		    ""
		)
	   )
	   ""
       )
  )
)

(defun without-tabs-1 (string &optional (start 0))
"Starting at the start index Start, the string has all absolute tab denotations
removed.  Thus \"foo200bar\" -> \"foobar\".
"
  (let ((start-index (string-search-set '(#\) string start)))
       (if start-index
	   (let ((stop-index
		   (string-search-set '(#\) string (+ 1 start-index))
		 )
		)
		(if stop-index
		    (string-append (subseq string start start-index)
				   (without-tabs-1 string (+ 1 stop-index))
		    )
		    (subseq string start)
		)
	   )
	   (subseq string start)
       )
  )
)

(defun without-tabs (string &optional (length 20) (trailer "..."))
"Returns a string like String only with any tabbing removed.  It
abbreviates the string to max-length Length and sticks trailer at the
end if it was too long.
"
  (let ((string (Without-Tabs-1 string 0)))
       (if (> (length string) length)
	   (string-append (subseq string 0 length) trailer)
	   string
       )
  )
)

(defun parse-out-tabs (string stop-point)
"If string has any tabs between the beginning and stop-point then we strip
them out and return the original string and the stripped string, otherwise we
return the original string.
"
  (declare (values original-string untabified-string-or-nil-if-no-tabs))
  (if (string-search-set '(#\) string stop-point)
      (values string (extract-tabs string 0))
      (values string nil)
  )
)

(defun maybe-max-width (format-string arg &rest format-args)
"Formats the format string with the format args to standard-output using
arg.  If arg is not a cons then it is used as the first arg to format.  If it is
a cons then it is of the form:
   (arg-for-format &optional (max-length) :font <font-specifier>)
"
  (if (consp arg)
      (let ((font (second (member :Font arg :Test #'eq))))
	   (let ((string
		   (apply #'format nil format-string (first arg) format-args)
		 )
		)
	        (let ((stop-point (if (numberp (second arg))
				      (min (second arg) (length string))
				      (length string)
				  )
		      )
		     )
		     (multiple-value-bind (string tabs)
			 (Parse-Out-Tabs string stop-point)
		       (send *standard-output* :String-Out
			     (if font (tv:fontify-string string font) string)  0
			     stop-point
		       )
		       (if tabs (send *standard-output* :String-Out tabs) nil)
		     )
		)
		
	   )
      )
      (apply #'format t format-string arg format-args)
  )
)

(defmethod Yw-Header-Display-Spec
       ((type (eql :Number))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of message numbers
in message displays.
"
  (ignore cache-entry message-from message-flags message-date message-subject
	  message-to message-length-in-chars day month year date-mode-args
  )
  (maybe-max-width *message-header-number-format-string* value message-number)
)

(defmethod Yw-Header-Display-Spec
       ((type (eql :From))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of from fields
in message displays.
"
  (ignore message-number cache-entry message-flags message-date message-subject
	  message-length-in-chars day month year date-mode-args message-to
  )
  (if (and *show-to-address-if-from-me-p*
	   (Address-Object-For-User-Address)
	   (let ((address (Address-Object cache-entry)))
	        (yw-zwei:address-equal (Address-Object-For-User-Address)
				       address
		)
	   )
      )
      (maybe-max-width (string-append "-> " *message-header-from-format-string*)
		       value message-to
      )
      (maybe-max-width *message-header-from-format-string* value message-from)
  )
)

(defmethod Yw-Header-Display-Spec
       ((type (eql :Subject))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of subject fields
in message displays.
"
  (ignore message-number cache-entry message-flags message-date message-from
	  message-length-in-chars day month year date-mode-args message-to
  )
  (maybe-max-width *message-header-subject-format-string* value message-subject)
)

(defmethod Yw-Header-Display-Spec
       ((type (eql :Keywords))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of message keywords
in message displays.
"
  (ignore message-number cache-entry message-flags message-date message-from
	  message-subject message-length-in-chars day month year date-mode-args
	  message-to
  )
  (maybe-max-width *message-header-keywords-format-string*
		   value
		   (header-keywords-string
		     (loop for x in Message-Flags
			   unless (member x *system-flags* :Test #'eq)
			   collect x
		     )
;		     (set-difference Message-Flags *System-Flags*)
		   )
  )
)

(defmethod Content-Type-String
	   ((content-type (eql :Text)) (content-subtype t) stream)
  (format stream "")
)

(defmethod Content-Type-String
	   ((content-type (eql :Message)) (content-subtype t) stream)
  (format stream "Msg")
)

(defmethod Content-Type-String
	   ((content-type (eql :Text)) (content-subtype (eql :richtext)) stream)
  (format stream "Rch")
)

(defmethod Content-Type-String
	   ((content-type (eql :Multipart)) (content-subtype t) stream)
  (format stream "MPt")
)

(defmethod Content-Type-String
  ((content-type (eql :Multipart)) (content-subtype (eql :Alternative)) stream)
  (format stream "Alt")
)

(defmethod Content-Type-String
   ((content-type (eql :Multipart)) (content-subtype (eql :Digest)) stream)
  (format stream "Dig")
)

(defmethod Content-Type-String
	   ((content-type (eql :Image)) (content-subtype t) stream)
  (format stream "Img")
)

(defmethod content-type-string ((content-type t) (content-subtype t) stream)
  (format stream "~A" content-type)
)
  
(defmethod Yw-Header-Display-Spec
       ((type (eql :Content-Type))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of the message flags
in message displays.
"
  (ignore message-number message-flags message-date message-from
	  message-subject message-length-in-chars day month year
	  date-mode-args message-to
  )
  (Maybe-Max-Width *message-content-type-format-string* value
		   (with-output-to-string (stream)
		     (let ((*print-case* :Capitalize))
		          (content-type-string
			    (cache-content-type cache-entry)
			    (cache-content-subtype cache-entry) stream
			  )
		     )
		   )
  )
)

(defmethod Yw-Header-Display-Spec
       ((type (eql :flags))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of the message flags
in message displays.
"
  (ignore message-number cache-entry message-flags message-date message-from
	  message-subject message-length-in-chars day month year
	  date-mode-args message-to
  )
  (Maybe-Max-Width *message-header-flags-format-string*
		   value (with-output-to-string (stream)
			   (header-flags-string message-flags stream
						(cache-mailstream cache-entry)
			   )
			 )
  )
)

(defmethod Yw-Header-Display-Spec
       ((type (eql :Length))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of the message length
in message displays.
"
  (ignore message-number cache-entry message-flags message-date message-from
	  message-subject message-length-in-chars day month year
	  date-mode-args message-to value
  )
  (Maybe-Max-Width *message-header-length-format-string*
		   (if (keywordp value)
		       *Message-Header-Message-Length-Field-Width*
		       value
		   )
		   Message-length-in-chars
  )
)

(defmethod Yw-Header-Display-Spec
       ((type (eql :Space))
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The function that deals with the printing of a simple space
in message displays.  The value of Value must be either a number for the
number of spaces, T for one space or Nil for no space.
"
  (ignore message-number cache-entry message-flags message-date message-from
	  message-subject message-length-in-chars day month year
	  date-mode-args message-to
  )
  (if value
      (if (numberp value)
	  (format t "~VA" value " ")
	  (princ " ")
	  nil
      )
  )
)

(defmethod Yw-Header-Display-Spec
       ((type t)
        value message-number cache-entry message-flags message-date
	message-from message-to message-subject message-length-in-chars day
	month year date-mode-args
       )
"The default method for header item display."
  (ignore message-number cache-entry message-flags message-date message-from
	  message-subject message-length-in-chars day month year
	  date-mode-args message-to
  )
  (format t "?~S, ~S" type value)
)

(defun format-header-display-string
       (cache-entry &optional
	(print-spec *message-header-display-specification*)
	(message-number (cache-msg# cache-entry))
	(message-flags (copy-tree (cache-flags cache-entry)))
	(message-date (cache-internaldate cache-entry))
	(message-from (Cache-FromText cache-entry))
	(message-to (Cache-ToText cache-entry))
	(message-subject (Cache-SubjectText cache-entry))
	(message-length-in-chars (Cache-RFC822Size cache-entry))
       )
"Formats the string for the display of a header in a summary window.  It does
this by using the specs in *Message-Header-Display-Specification*.  It puts a
space between each element unless it finds a :space spec entry.  If it finds
one of these it does whatever the :space specs says instead.
"
  (let ((print-spec (or print-spec *message-header-display-specification*)))
       (tv:with-output-to-fat-string (*standard-output*)
	 (multiple-value-bind (ignore ignore ignore day month year)
	     (time:decode-universal-time
	       (safe-parse-universal-time message-date)
	     )
	   (multiple-value-bind (ignore ignore ignore ignore ignore this-year)
	       (time:decode-universal-time (time:parse-universal-time "today"))
	     (with-stack-list (date-mode-args
				day month (time:month-string month :short)
				(= year this-year) (mod year 100.)
			      )
	       (loop for (key value) on print-spec By #'cddr
		     for rest on print-spec by #'cddr
		     Do (Yw-Header-Display-Spec key
			  value message-number cache-entry
			  message-flags message-date message-from
			  message-to message-subject
			  message-length-in-chars day
			  month year date-mode-args
			  )
			  (if (or (equal key :Space)
				  (equal (third rest) :Space)
			      )
			      nil
			      (princ " ")
			  )
	       )
	     )
	   )
	 )
       )
  )
)

(defun safe-parse-universal-time (string)
"Parses a universal time out of String in a manner that won't result in
barfage.  If it really cannot compute a UT from the string it picks today.
"
  (or (loop for start from 0 below (length string)
	    for result = (loop for end from (length string) downto 0
			       for results = (catch-error
					       (multiple-value-list
						 (time:parse-universal-time
						   string start end
						 )
					       )
					       nil
					     )
			       when results
			       return results
			 )
	    when result
	    return (values-list result)
      )
      (time:parse-universal-time "Today")
  )
)

(defun address-object-for-user-address ()
  (or *address-object-for-user-address*
      (and mail:*user-mail-address*
	   (progn (setq *address-object-for-user-address*
			(mail:parse-address mail:*user-mail-address*)
		  )
		  *address-object-for-user-address*
	   )
      )
  )
)
