;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;;; This file contains a client side parser for the IMAP protocol.
;;; The syntax for the interaction over the IMAP stream is specified
;;; in RFC 1064 on pages 24-26.

(defvar *Stream* :Unbound
"The current IMAP stream."
)

(defun-method imap-stream-error imap-stream-mixin (message &rest format-args)
"IMAP stream error signaler."
  (case *catch-imap-stream-errors-p*
    (:catch
      (apply 'format-scroll-window *Stream* message format-args)
      (setq current-line nil)
      (signal 'sys:abort 'sys:abort)
    )
    (:Close (apply 'format-scroll-window *Stream* message format-args)
	    (send self :close)
    )
    (otherwise (apply 'ferror nil message format-args))
  )
)

(Defun-method Parse-Atom imap-stream-mixin (&rest one-of-these)
"Parses an atom from the IMAP stream.  If one-of-these is provided then we get
an error if it the atom is not one of those specified.
"
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (let ((start nil)
	(line current-line)
	(length (length current-line))
       )
       (Skip-Imap-Whitespace line)
       (setq start line-index)
       (if (= start length)
	   (progn (Ensure-Line-Read-In t)
		  (Skip-Imap-Whitespace)
		  (setq line current-line)
		  (setq start line-index)
		  (setq length (length current-line))
	   )
	   nil
       )
       (loop do (setq line-index (+ 1 line-index))
	     until (or (>= line-index length)
		       (let ((char (aref line line-index)))
			    (or (char= char #\space)
				(char= char #\))
			    )
		       )
		   )
       )
; (if (= start line-index) (ferror nil "???"))
       (let ((string (nsubstring line start line-index)))
	    (Maybe-Discard-Line nil length)
; (if (or (= start line-index) (equal "" string)) (ferror nil "???"))
	    (let ((atom (intern (nstring-upcase string) 'keyword)))
		 (if (or (not one-of-these)
			 (member atom one-of-these :Test #'eq)
		     )
		     atom
		     (imap-stream-error "Atom ~S read in and not expected."
					atom
		     )
		 )
	    )
       )
  )
)

(Defun-method Parse-Atom-or-number imap-stream-mixin ()
"Parses an atom or a number from the IMAP stream."
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (let ((start nil)
	(line current-line)
	(length (length current-line))
       )
       (Skip-Imap-Whitespace line)
       (setq start line-index)
       (if (= start length)
	   (progn (Ensure-Line-Read-In t)
		  (Skip-Imap-Whitespace)
		  (setq line current-line)
		  (setq start line-index)
		  (setq length (length current-line))
	   )
	   nil
       )
       (loop do (setq line-index (+ 1 line-index))
	     until (or (>= line-index length)
		       (let ((char (aref line line-index)))
			    (or (char= char #\space)
				(char= char #\))
			    )
		       )
		   )
       )
       (let ((string (nsubstring line start line-index)))
	    (Maybe-Discard-Line nil length)
	    (let ((numberp
		    (loop for char being the array-elements of string
			  always (Numberchar-P char)
		    )
		  )
		 )
	         (if numberp
		     (parse-integer string)
		     (intern (nstring-upcase string) 'keyword)
	         )
	    )
       )
  )
)

(defun-method parse-text-line imap-stream-mixin ()
"Reads a text line from the current stream."
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (skip-imap-whitespace)
  (let ((length (length current-line)))
       (let ((string (make-string (- length line-index))))
	    (copy-array-portion current-line line-index length
				string 0 (- length line-index)
	    )
	    (Maybe-Discard-Line t)
	    string
       )
  )
)

(defun parse-ready ()
"Parses an IMAP Ready statement."
  (declare (values :+ status-string))
  (Declare (optimize (speed 3) (safety 0)))
  (values :+ (Parse-Text-Line))
)

(defun parse-tagged (tag)
"Parses an IMAP tagged command line."
  (declare (values tag ok-bad-or-no status-string)) 
  (declare (optimize (speed 3) (safety 0)))
  (let ((atom (parse-atom-or-number)))
       (case atom
	 ((:Ok :No :Bad) (values tag atom (parse-text-line)))
	 (otherwise (Parse-Solicited tag atom))
       )
  )
)

(defun numberchar-p (char)
  "Is true if char is a legal decimal numeral character."
  (declare (optimize (speed 3) (safety 0)))
  (sys:%string-search-char char "0123456789" 0 10)
)

(defun-method parse-imap-number-1 imap-stream-mixin (so-far length)
"An internal function used in the parsing of IMAP numbers.  So-Far is the
accumulated number that we've parsed so far.  Length is the length of the input
line, which saves us from making repeated IV lookups on this value.
"
  (declare (optimize (speed 3) (safety 0)))
  (let ((char nil))
       (if (>= line-index length)
	   so-far
	   (if (numberchar-p (setq char (Current-Char t)))
	       (let ((new-val (+ (* 10 so-far) (as-number char))))
		    (setq line-index (+ 1 line-index)) ;(Inc-Char length)
		    (parse-imap-number-1 new-val length)
	       )
	       (if (char= #\. (Current-Char t))
		   (let ((decimal-part (progn (inc-char) (Parse-Imap-Number))))
		        (+ so-far
			   (/ decimal-part
			      (float
				(expt
				  10 (length (format nil "~D" decimal-part))
				)
			      )
			   )
			)
		   )
		   so-far
	       )
	   )
       )
  )
)

(defun-method parse-imap-number imap-stream-mixin ()
"Parses a number from the IMAP stream."
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (skip-imap-whitespace)
  (parse-imap-number-1 0 (length current-line))
)

(defun-method skip-char imap-stream-mixin (char)
"Skips over the character Char, which we expect to be the next non-whitespace
 char.  If it isn't then we signal an error.
"
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (skip-imap-whitespace)
  (if (char= char (Current-Char))
      (Inc-Char)
      (imap-stream-error
	  "Illegal char ~S found at index ~D when ~S expected on line \"~A\""
	  (Current-Char) line-index char current-line
      )
  )
  (Current-Char)
)

(defun-method parse-literal imap-stream-mixin (into-string length-remaining)
"Parses a literal string i.e. {nnn}\newline text.  Into-String is a string that
we have already consed of at least the right length using the number in the
braces.  Length-Remaining is a counter that tells us how many chars we have
yet to read.  Note:  there's a complication here because IMAP specifies that
these lines end in CRLF.  When we get it it has had the CRs stripped out.
This means that we have to increment our length by 2 for each nl that we read
in.  We end up by setting the fill pointer of the into-string so that it has
a length that we can really deal with.  This is guaranteed to be <= than the
length provided by IMAP because of CRLF behaviour.
"
  (declare (optimize (speed 3) (safety 0)))
  (let ((line-length (length current-line))
	(fill (fill-pointer into-string))
       )
       (if (> line-length length-remaining)
	   (progn (copy-array-portion current-line 0 length-remaining
				      into-string fill (+ fill length-remaining)
                  )
		  (setf (fill-pointer into-string) (+ length-remaining fill))
		  (setq line-index length-remaining)
		  (Maybe-Discard-Line)
		  into-string
	   )
	   (progn (copy-array-portion current-line 0 line-length
				      into-string fill (+ fill line-length)
                  )
		  (setf (fill-pointer into-string) (+ line-length fill 1))
		  (setf (aref into-string (- (fill-pointer into-string) 1))
			#\newline
		  )
		  (ensure-line-read-in t)
		  (parse-literal into-string (- length-remaining line-length 2))
	   )
       )
  )
)

(defun-method parse-string imap-stream-mixin ()
"Parses a string from the IMAP stream.  This could be one of three types:
  Quote delimited string: We detect this by the first char being a \"
  Literal: We detect this by the first char being a {
  Atom: This is used if we ever get anything other than \" or {

We can read atoms in the normal manner.  Quote delimited strings are a
little hairy, but we are guaranteed that the next \" will close the string
and that quote is on the same line.  Literals a more hairy, since they
generally span more than one line.  We read in the length of the literal
and then read the string itself elsewhere.
"
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (skip-imap-whitespace)
  (let ((char (Current-Char)))
       (cond ((char= char #\")
	      (let ((start (+ 1 line-index))
		    (line current-line)
		   )
		   (loop do (setq line-index (+ 1 line-index))
			 until (char= (aref line line-index) #\")
		   )
		   (let ((string (make-string (- line-index start))))
		        (copy-array-portion line start line-index
					    string 0 (length string)
			)
			(Inc-Char) ;; skip closing quote
			string
		   )
	      )
	     )
	     ((char= char #\{)
	      (Skip-Char #\{)
	      (let ((length (Parse-Imap-Number)))
		   (Skip-Char #\})
		   (Ensure-Line-Read-In)
		   (let ((string (make-array length :Element-Type 'string-char
					     :Fill-Pointer t
				 )
			 )
			)
		        (setf (fill-pointer string) 0)
		        (parse-literal string length)
			string
		   )
	      )
	     )
	     (t (string (Parse-Atom)))
       )
  )
)

(defun intern-address (name routelist mailbox host comment)
  (declare (optimize (speed 3) (safety 0)))
  (let ((key (list name routelist mailbox host comment)))
       (or (gethash key *interned-address-table*)
	   (setf (gethash key *interned-address-table*)
		 (Make-Address :Personalname name
			       :Routelist    routelist
			       :Mailbox      mailbox
			       :Host         host
			       :Comment      comment
		 )
	   )
       )
  )
)

(defun parse-address ()
"Parses an address from an IMAP stream.  The result of parsing it is an
YW address object.
"
  (declare (optimize (speed 3) (safety 0)))
  (Skip-Char #\()
  (let ((name      (parse-string-or-nil))
	(routelist (parse-string-or-nil))
	(mailbox   (parse-string-or-nil))
	(host      (parse-string-or-nil))
	(comment   (if (equal #\) (Peek-Next-Char)) nil (parse-string-or-nil)))
       )
       (let ((address (intern-address name routelist mailbox host comment)))
	    (Skip-Char #\))
	    address
       )
  )
)

(defun-method Parse-Address-List imap-stream-mixin ()
"Parses a list of addresses from the IMAP stream.  This is a number of addresses
encapsulated within a pair of ()s.
"
  (declare (optimize (speed 3) (safety 0)))
  (Skip-Imap-Whitespace)
  (let ((char (Current-Char)))
       (if (char= char #\()
	   (progn (Skip-Char #\()
		  (let ((addresses (loop until (char= (Current-Char t) #\))
					 collect (parse-address)
				   )
			)
		       )
		       (Skip-Char #\))
		       addresses
		  )
	   )
	   (progn (Parse-Atom :Nil)
		  nil
	   )
       )
  )
)

(defun parse-string-or-nil ()
"Parses an IMAP grammar item which can be either a string or NIL.  Returns
either the string or NIL, as opposed to the string \"NIL\".
"
  (declare (optimize (speed 3) (safety 0)))
  (let ((string (Parse-String)))
       (if (string-equal "NIL" string)
	   nil
	   string
       )
  )
)

(defun Parse-envelope (cache-entry)
"Parses an envelope from the IMAP stream.  Cache-Entry is the cache
object in our message cache that we are going to use to represent this message.
Returns the envelope of the message.  As a side effect it gets the daemons to
take note of the newly arrived envelope data.
"
  (declare (optimize (speed 3) (safety 0)))
  (declare (special *edit-server* *address-server*))
  (Skip-Char #\()
  (let ((date (parse-string)) ;date
	(subject (or (parse-string-or-nil) "")) ;subject
	(from (parse-address-list)) ;from
	(sender (parse-address-list)) ;sender
	(reply-to (parse-address-list)) ;reply-to
	(to (parse-address-list)) ;to
	(cc (parse-address-list)) ;cc
	(bcc (parse-address-list)) ;bcc
	(in-reply-to (parse-string-or-nil)) ;in-reply-to
	(message-id (parse-string-or-nil)) ;message-id
       )
       (let ((envelope (make-envelope date subject from sender reply-to to cc
				      bcc in-reply-to message-id
		       )
	     )
	    )
	    (Skip-Char #\))
	    (setf (Cache-Envelope cache-entry) envelope)
	    (send *edit-server* :Put-Task :IMAP-Parse-Data-envelope
		  (list :flush-display-cache-for cache-entry t)
	    )
	    (send *address-server* :Put-Task
		  :IMAP-Parse-Addresses-In-Envelope-From-Parse-Envelope
		  (list :Parse-Envelope
			(Cache-Envelope cache-entry) cache-entry *stream*
		  )
	    )
	    envelope
       )
  )
)

(defun-method Parse-Flags imap-stream-mixin ()
"Parses a set of flags from the IMAP stream.  These are a list of
atoms delimited by a pair of ()s.
"
  (declare (optimize (speed 3) (safety 0)))
  (Skip-Char #\()
  (Parse-Keywords)
)

(defun-method Parse-keywords imap-stream-mixin ()
"Parses a list of keywords until it finds a closing paren."
  (let ((line current-line))
       (let ((flags (loop do (skip-imap-whitespace line)
			  until (char= (current-char t) #\))
			  collect (Parse-Atom)
		    )
	     )
	    )
	    (inc-char (length line))
	    flags
       )
  )
)

(defun reinitialise-cache-entry-for-envelope (cache-entry)
"Given a cache entry clears out all of the slots that it has computed
from data in the envelope because we have received another envelope.
"
  (setf (cache-subjecttext cache-entry) :unbound)
  (setf (cache-totext cache-entry) :unbound)
)

(defun parse-fetch-attribute (message-number cache-entry fetch/store)
"Parses an item from the fetch_att (fetch attribute) production in the IMAP
grammar.  Message-number is the number of the message that we have already
read in.  Cache-Entry is the message cache object that we are using to
represent this message.  Fetch-Store is the :FETCH or :STORE atom that has
already been read in.  Both fetch and store are processed in much the same
way.  As a side effect of parsing these attributes we notify the daemons
of the change of status of the message.
"
  (declare (optimize (speed 3) (safety 0)))
  (declare (special *edit-server* *address-server*))
; (print (let ((index (send *Stream* :line-index))
;	      (line (Send *Stream* :Current-Line))
;	     )
;	  (list index (aref line index) line)))
  (Let ((atom (Parse-Atom :Envelope :Flags :Internaldate :Rfc822 :Rfc822.Header
			  :Rfc822.Size :Rfc822.Text :Rfc822* :Rfc822.Text*
	      )
	)
       )
       (Ecase atom
	 (:Envelope
	  (let ((envelope (Parse-Envelope cache-entry)))
	       (reinitialise-cache-entry-for-envelope cache-entry)
	       envelope
	  )
	 )
	 (:Flags (setf (cache-flags cache-entry) (Parse-Flags))
		 (send *edit-server* :put-task :change-of-flags
		       (list :Maybe-Flush-Search-Cache-for-flags
			     *stream* cache-entry fetch/store
		       )
		 )
		 (send *edit-server* :Put-Task :Imap-Parse-Data-Flags
		       ;;; !!!! I'm not quite sure what to do about this T  !
		       (list :flags-changed cache-entry *Stream*
			     (if (equal :Store fetch/store) :Flags-Changed nil)
		       )
                 )
		 (Values :Flags (cache-flags cache-entry))
         )
	 (:Internaldate
	  (values :Internaldate
		  (setf (Cache-InternalDate cache-entry) (parse-string))
	  )
	 )
	 (:Rfc822.Header
	  (action-for-new-header cache-entry (Parse-String))
	 )
	 ((:Rfc822 :Rfc822*)
	  (values atom
		  (Setf (cache-rfc822-all-text cache-entry) (parse-string))
	  )
	 )
	 ((:Rfc822.Text :Rfc822.Text*)
	  (values atom (Setf (Cache-RFC822Text cache-entry) (parse-string)))
	 )
	 (:Rfc822.Size
	  (values atom
		  (setf (Cache-RFC822Size cache-entry) (parse-imap-number))
	  )
	 )
       )
  )
)

(defun action-for-new-header (cache-entry header-string)
  (declare (special *edit-server* *address-server*))
  (setf (Cache-RFC822Header cache-entry) header-string)
  ;;; Force the parsing of all addresses in the background for
  ;;; the completion stuff.
  (send *address-server* :Put-Task :IMAP-Parse-Data-rfc822.header
	(list :maybe-filter-header cache-entry header-string *stream*)
  )
  (send *edit-server* :Put-Task :Imap-Parse-Data-Flags
	(list :Flush-Display-Cache-For cache-entry t)
  )
  (values :Header header-string)
)

(defun-method Parse-Fetch/Store imap-stream-mixin
	      (message-number cache-entry fetch/store)
"Parses a FETCH or STORE command from the IMAP stream."
  (declare (optimize (speed 3) (safety 0)))
  (skip-char #\()
  (let ((results (loop until (char= (Current-Char) #\))
		       do (skip-imap-whitespace)
		       append (if (char= (Current-Char) #\()
				  (progn (Skip-Char #\()
					 (loop until (char= (Current-Char) #\))
					       collect
						 (Parse-fetch-attribute
						   message-number cache-entry
						   fetch/store
						 )
						 ;;; {!!!!}  I don't understand
						 ;;; why I occasionally
						 ;;; get quote chars here.
						 do (if (char=
							  #\" (Current-Char))
							(skip-char #\")
							nil
						    )
					 )
				   )
			           (list (Parse-fetch-attribute
					   message-number cache-entry
					   fetch/store
					 )
				   )
			       )
		 )
        )
       )
       (Inc-Char)
       results
  )
)

(defun process-fetch/store (fetch/store message-number)
"Parses a FETCH or STORE command from the IMAP stream."
  (declare (optimize (speed 3) (safety 0)))
  (let ((cache-entry (cache-entry-of message-number *Stream*)))
       (check-type cache-entry cache)
       (let ((data (Parse-Fetch/Store message-number cache-entry fetch/store)))
	    (if (send *Stream* :MessageArray)
		(send *stream* :Maybe-Flush-Search-Cache fetch/store data)
		(format *error-output*
			"Internal error no message array."
		)
	    )
       )
  )
)

(defun parse-msg-data (message-number)
"Parses a msg_data production from the IMAP stream, having already read in the
message-number.  This could be an EXISTS, a RECENT, and EXPUNGE a COPY, a STORE
or a FETCH.
"
  (declare (values :* message-number exists-recent-store-expunge-fetch-or-copy))
  (Declare (optimize (speed 3) (safety 0)))
  (let ((atom (Parse-Atom :Exists :Recent :Store :Expunge :Fetch :Copy)))
       (ecase atom
	 (:Exists (if (not (equal message-number (send *stream* :Messagecnt)))
		      ;;; Then our unseen search cache will be invalid.
		      (send *stream* :maybe-flush-search-cache :Exists
			    message-number
		      )
		      nil
		  )
		  (IMAP-Exists *Stream* message-number)
	 )
	 (:Recent (if (> message-number 0)
		      ;;; Then our unseen search cache will be invalid.
		      (send *stream* :Maybe-Flush-Search-Cache :Recent)
		      nil
		  )
		  (IMAP-Recent *Stream* message-number)
	 )
	 (:Expunge
	  (IMAP-Expunged *Stream* (cache-entry-of message-number *Stream*))
	 )
	 (:Copy nil) 
	 ((:Fetch :Store) (process-fetch/store atom message-number))
       )
       (values :* message-number atom)
  )
)

(defun-method Parse-Flag-List imap-stream-mixin ()
"Parses a flag list from the IMAP stream.  A flag list is a set
of atoms delimited by a pair of ()s.
"
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (skip-imap-whitespace)
  (let ((char (Current-Char)))
       (if (char= char #\()
	   (progn (Skip-Char #\()
		  (let ((atoms (loop until (char= (Current-Char) #\))
				     collect (Parse-Atom)
			       )
			)
		       )
		       (Skip-Char #\))
		       atoms
		  )
	   )
	   (list (Parse-Atom))
       )
  )
)

(defun-method Parse-Number-List imap-stream-mixin ()
"Parses a set of numbers from a search.  These numbers are all on the line and
are not delimited.  Note that there may be no numbers in which case we
return nil.
"
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (skip-imap-whitespace)
  (loop until (>= line-index (length current-line))
	collect (Parse-Imap-Number)
  )
)

(defun-method Parse-Number-List-In-Parens imap-stream-mixin ()
"Parses a set of numbers from a search.  These numbers are all on the line and
are not delimited.  Note that there may be no numbers in which case we
return nil.
"
  (declare (optimize (speed 3) (safety 0)))
  (Skip-Char #\()
  (let ((numbers (loop until (char= (Current-Char) #\))
		       collect (Parse-Imap-Number)
		 )
	)
       )
       (Skip-Char #\))
       numbers
  )
)

(defun-method Parse-message-sequence imap-stream-mixin ()
"Parses an IMAP message sequence from the stream.  The sequence is atomic
on the line. 
"
  (declare (optimize (speed 3) (safety 0)))
  (loop for char = (Peek-Next-Char)
	while (number-char-p char)
	collect (let ((number (Parse-Imap-Number)))
		     (let ((next-char (Peek-Next-Char)))
		          (case next-char
			    (#\, (Skip-Char next-char) number)
			    (#\: (Skip-Char next-char)
				 (list number (Parse-Imap-Number))
			    )
			    (otherwise number)
			  )
		     )
		)
  )
)

(defun-method Parse-Version-Spec imap-stream-mixin ()
"Parses a version spec of the form (real_number &rest features)."
  (Skip-Char #\()
  (let ((number (parse-imap-number))
	(the-keywords (Parse-keywords))
       )
       (if the-keywords
	   (let ((number? (read-from-string (string (first the-keywords)))))
	        (if (numberp number?)
		    (cons number (cons number? (rest the-keywords)))
		    (cons number the-keywords)  ;;; This skips the closing paren
		)
	   )
	   (list number)
       )
  )
)

(defun-method Parse-Version-Specs imap-stream-mixin ()
"Parses a list of version specs from the mailstream."
  (Skip-Char #\()
  (let ((specs (loop until (char= (Current-Char t) #\))
		     collect (parse-version-spec)
	       )
	)
       )
       (Skip-Char #\))
       specs
  )
)

(defun parse-search-criterion ()
"Parses a search criterion such as FROM acuff from the stream."
  (let ((keyword (Parse-Atom)))
       (let ((argument
	       (letf ((#'imap:read-string #'(lambda (ignore) (Parse-String))))
		     (imap:read-search-keyword keyword *Stream*)
	       )
	     )
	    )
	    (list keyword argument)
       )
  )
)

(Defun-method parse-search-criteria imap-stream-mixin ()
"Parses a set of search criteria from the mailstream.  Returns them as a list
of the form ((from acuff) (seen)).
"
  (Skip-Char #\()
  (let ((criteria (loop until (char= (Current-Char) #\))
			collect (parse-search-criterion)
		  )
	)
       )
       (Skip-Char #\))
       criteria
  )
)

(defun-method process-search-reply imap-stream-mixin ()
  "Processes the result of a search."
  (if (At-Least-Imap-3-P *stream*)
      (let ((selected (Parse-Number-List-In-Parens))
	    (criteria (parse-search-criteria))
	   )
	   (ignore criteria)
	   (send *Stream* :set-SelectedMsgs selected)
	   (send *stream* :Maybe-Flush-Search-Cache
		 :Search selected
	   )
      )
      (let ((selected (if current-line (parse-number-list) nil)))
	   (send *Stream* :set-SelectedMsgs selected)
	   (send *stream* :Maybe-Flush-Search-Cache :Search selected)
      )
  )
)

(defun do-reset-of-message-sequence ()
"Flushes any caches for the messages in the sequence that we read from the
 stream."
  (let ((sequence (parse-message-sequence)))
       (let ((windows
	       (remove-duplicates
		 (apply #'append
		        (map-over-sequence 'flush-all-caches-containing
					   sequence *Stream*
			)
		 )
	       )
	     )
	    )
	    (if windows
		(process-run-function
		  "Refresh Windows."
		  #'(lambda (wins mailstream)
		      (loop for win in wins do
			    (send win :Set-Up (send win :Owner)
				  (send win :Filter)
				  (send mailstream :Mailbox-Name)
				  (list mailstream)
			    )
		      )
		    )
		  windows *stream*
		)
		nil
	    )
       )
  )
)

(defun-method parse-data imap-stream-mixin (&optional (first-token nil))
"Parses the DATA production from the IMAP grammar.  This could be any of:
flags, search, bye, ok, no or bad.  It does the appropriate thing
depending on what it finds.
"
  (declare (optimize (speed 3) (safety 0)))
  (let ((legal-tokens '(:Flags :Search :Bye :Ok :No
			:Bad :Supported.Versions :Reset
			:Readonly :Readwrite :Bboard :Mailbox
		       )
        )
       )
       (let ((atom (if first-token
		       (if (member first-token legal-tokens :Test #'eq)
			   first-token
			   (Imap-Stream-Error
			     "Atom ~S read in and not expected."
			     first-token
			   )
		       )
		       (apply 'Parse-Atom legal-tokens)
		   )
	     )
	    )
	    (let ((arg
		    (case atom
		      (:Readonly  (send *Stream* :Set-Read-Only-P t))
		      (:Readwrite (send *Stream* :Set-Read-Only-P nil))
		      (:Bboard (let ((name (Parse-String)))
				    (send *Stream* :add-bboard name)
				    name
			       )
		      )
		      (:Mailbox (let ((name (Parse-String)))
				     (send *Stream* :add-mailbox name)
				     name
				)
		      )
		      (:Flags (let ((flag-list (parse-flag-list)))
				   (send *Stream* :set-FlagList flag-list)
				   (send
				     *Stream* :set-Keywords
				     (set-difference (send *Stream* :FlagList)
						     *system-flags*
				     )
				   )
				   flag-list
			      )
		      )
		      (:Search (Process-Search-Reply))
		      (:Reset (Do-Reset-Of-Message-Sequence))
;		      (:Select.Version
;		       (format-scroll-window
;			 (any-mailer) "~&Version ~A for ~A~%"
;			 (Parse-Text-Line) (send *stream* :pretty-string)
;		       )
;		      )
		      (:Supported.Versions
		       (let ((versions (parse-version-specs)))
			    (send self :Set-Supported-Versions versions)
			    versions
		       )
		      )
		      (:Bye (format-scroll-window (any-mailer) "~&~A for ~A~%"
			      (Parse-Text-Line) (send *stream* :pretty-string)
			    )
		      )
		      (:Ok (Parse-Text-Line))
		      (:No (format-scroll-window (any-mailer)
			      "~&Command not completed by server ~A for ~A~%"
			      (Parse-Text-Line) (send *stream* :pretty-string)
			   )
		      )
		      (:Bad
		       (format-scroll-window (any-mailer)
		         "~&Protocol error ~A on stream ~A  please report.~%"
			 (Parse-Text-Line) (send *stream* :pretty-string)
		       )
		      )
		      (otherwise
			(format-scroll-window (any-mailer)
			  "~&Unknown reply ~S from server for stream ~A, ~
			  the rest of the message was ~S"
			   atom (send *stream* :pretty-string) (Parse-Text-Line)
			)
		      )
		    )
		  )
		 )
		 (values :* atom arg)
	    )
       )
  )
)

(defun-method peek-next-char imap-stream-mixin ()
"Peeks at the next char on the current line of IMAP stream.  In general this
does not involve reading in a new line, but if there is no current new line
then one is read in.
"
  (declare (optimize (speed 3) (safety 0)))
  (ensure-line-read-in)
  (skip-imap-whitespace)
  (Current-Char)
)

(Defun parse-solicited (tag &optional (first-token nil))
"Parses the solicited production in the IMAP grammar or the unsolicited
production with tag beoing :*.
"
  (declare (optimize (speed 3) (safety 0)))
  (if first-token
      (If (numberp first-token)
	  (parse-msg-data first-token)
	  (Parse-Data first-token)
      )
      (let ((char (peek-next-char)))
	   (If (numberchar-p char)
	       (let ((message-number (parse-imap-number)))
		    (parse-msg-data message-number)
	       )
	       (parse-data)
	   )
      )
  )
)

(Defun parse-unsolicited ()
"Parses the unsolicited production in the IMAP grammar."
  (declare (optimize (speed 3) (safety 0)))
  (Parse-Solicited :*)
)

;;; Debug.
;(defwhopper (imap-stream-mixin :parse-something-from-imap-stream) ()
;  (print (list :>>>> current-line line-index))
;  (unwind-protect (continue-whopper)
;    (print (list :<<<< Current-line line-index))
;    nil
;  )
;)

;(defwhopper (imap-stream-mixin :line-in) (&rest args)
;  (let ((result (lexpr-continue-whopper args)))
;       (print (list (length result) result))
;       result
;  )
;)
;(undefmethod (imap-stream-mixin :around :line-in))

(Defmethod (imap-stream-mixin :parse-something-from-imap-stream) ()
"Parses a statement from the IMAP stream.  At this point we have no idea what
sort of a thing it might be.
"
  (declare (optimize (speed 3) (safety 0)))
  (if (send self :closed-P)
      (ferror 'close-worthy-error
	      "~&IMAP stream is closed."
      )
      nil
  )
  (let ((*Stream* self))
       (Maybe-Discard-Line)
       (Ensure-Line-Read-In)
       (if (not (equal 0 line-index))
	   (progn (format-scroll-window
		    nil "Imap command line corrupt.  Line skipped:"
		  )
		  (format-scroll-window nil "~A" current-line)
		  (push (copy current-line) *all-rejected-imap-command-lines*)
		  (Ensure-Line-Read-In t)
;		  (setq line-index 0)
	   )
	   nil
       )
       (Skip-Imap-Whitespace)
       (Let ((atom (Parse-Atom)))
	    (case atom
	      (:+ (Parse-ready))
	      (:* (parse-unsolicited))
	      (Otherwise (Parse-Tagged atom))
	    )
       )
  )
)

(defun parse-something-from-imap-stream (stream)
"Parses an item from the IMAP stream, taking whatever action needs to be taken."
  (declare (optimize (speed 3) (safety 0)))
  (send stream :parse-something-from-imap-stream)
)

;-------------------------------------------------------------------------------

(defun mailbox-name-for-select (mailbox)
"Returns a string for a Select command given the host:mbox-name spec Mailbox."
  (format NIL "~A~A"
	  (directory-namestring MailBox)
	  (let ((fs:*merge-unix-types* nil))
	       ;;; Fix here because this screws
	       ;;; up the file names if the
	       ;;; above is set to T.
	       (file-namestring MailBox)
	  )
  )
)

(defun ideal-imap-version-to-select (mailstream)
"Computes a good version number of the server and list of features to use
on that server version.
"
  (declare (values version-command version-number features))
  (let ((entry (assoc (send (send mailstream :Host) :System-Type)
		      *ideal-imap-version-to-select*
	       )
	)
	(default (assoc :Default *ideal-imap-version-to-select*))
       )
       (if entry
	   (values (second entry) (third entry)
		   (let ((features (rest (rest (rest entry)))))
		        (if (not *wildcard-match-in-searches-if-possible*)
			    (remove :Wildcard.Searches features)
			    features
			)
		   )
	   )
	   (values (second default) (third default)
		   (rest (rest (rest default)))
           )
       )
  )
)

(defun MAP-Open (MailBox NewUser read-only-p)
  "Establish a host connection, log the user in, and open the MailBox."
  ;;;We know the MailBox format is ok and we also add it to the list.
  (let ((MailHost (or (subseq MailBox 0 (position #\: MailBox))
		      *user-host*
		  )
	)
       )
       (let ((MailStream (IMAP-Open MailHost))
	     (completed-ok-p nil)
	    )
	    (unwind-protect
	      (if (and MailStream
		       (eq :Ok (second (multiple-value-list
					 (Parse-Something-From-Imap-Stream
					   MailStream
					 )
				       )
			       )
		       )
		       (IMAP-Login MailStream MailHost NewUser)
		       (IMAP-Select MailStream
				    (mailbox-name-for-select mailbox)
		       )
		       (or (> (send MailStream :MessageCnt) 0)
			   (format *error-output* "~&Mailbox is empty~%")
		       )
		  )
		  (progn (send MailStream :set-Mailbox MailBox)
			 (if read-only-p
			     (send mailstream :Mark-As-Read-Only)
			     nil
			 )
			 (multiple-value-bind (version-command version features)
			     (ideal-imap-version-to-select mailstream)
			   (Imap-Version mailstream
					 version-command version features
                           )
			 )
			 (setq completed-ok-p t)
			 MailStream
		  )
		  (progn (MAP-Close MailStream)		  ;;Else
			 nil
		  )
	      )
	      (if completed-ok-p
		  nil
		  (progn ;;; Panic cleanup.
			 (setq *all-open-imap-streams*
			       (remove mailstream *all-open-imap-streams*)
			 )
			 (send mailstream :Close)
		  )
	      )
	    )
       )
  )
)

;-------------------------------------------------------------------------------

(defmethod imap-send :Before ((stream t) (Command (eql 'yw:store))
			      &Optional Args (reply-type nil)
			      (non-ok-reply-ok nil)
		             )
  (ignore non-ok-reply-ok reply-type)
  (destructuring-bind (message-numbers type &rest flags) args
    (ignore message-numbers)
    (if (member type '(yw:flags yw:+flags yw:-flags) :Test #'eq)
	(let ((new-flags (set-difference flags (send stream :Flaglist))))
	     (if (and new-flags (at-least-imap-3-p stream))
		 (imap-send stream 'yw:set.flags (append (send stream :Flaglist)
							 new-flags
						 )
		 )
		 nil
	     )
	)
	nil
    )
  )
)

(defmethod imap-send ((stream Imap-Stream) Command
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
 "Sends specified command/arguments to IMAP server and then waits for response."
  (if (and (send stream :Read-Only-P)
	   (member command '(expunge) :Test #'string-equal)
      )
      nil ;;; Ignore these commands, since you can't do them.
      (if (output-stream-p Stream)
	  (let ((Tag (gentemp "TAG-" 'KEYWORD)))
	       (with-mailbox-locked (stream)
		 (let ((cmdstring
			 (build-imap-command-arglist tag command args)
		       )
		      )
		      (format Stream "~A~%" CmdString)
		      (stream-safely (stream) (force-output Stream))
		      (Fetch-And-Process-Synchronous-Imap-Reply
			stream tag reply-type non-ok-reply-ok
		      )
		 )
	       )
	  )
	  ;;Else
	  (format *error-output* "~&TCP Stream is not opened for output.~%")
	  (list :* :Bye "IMAP connection went away!")
      )
  )
)

(defun build-imap-command-arglist (tag command args)
"Builds the string used as the args to a request to the IMAP server.  This
involves transforming sundry strings into quoted strings and/or literals.  We
have to be careful here.  We cannot just use Prin1.
"
  (with-output-to-string (*standard-output*)
    (format t "~A ~A" tag command)
    (build-imap-arglist command args)
  )
)

(defun build-imap-arglist (command args)
"Builds the string used as the args to a request to the IMAP server.  This
involves transforming sundry strings into quoted strings and/or literals.  We
have to be careful here.  We cannot just use Prin1.
"
  (loop for arg in args do
	(if (and (eq arg (first args))
		 (member command '(copy fetch store) :Test #'string-equal)
	    )
	    (progn (princ " ") (format t "~A" arg))
	    (if (stringp arg)
		(if (IMAP-Break-Chars Arg)
		    (format t " {~D}~%~A" (length arg) arg)
		    (case command
		      (select (format t " ~A" arg))
		      (otherwise
		       (if (find arg *simple-term-specifiers* :Key #'second)
			   (format t " ~A" arg)
			   (format t " \"~A\"" arg)
		       )
		      )
		    )
		)
		(format t " ~A" arg)
	    )
	)
  )
)

(defun Fetch-And-Process-Synchronous-Imap-Reply
       (stream tag reply-type non-ok-reply-ok)
"Fetches a reply from the IMAP stream which should be a reply to a tagged
 command labelled with Tag.  Reply-type is a kludgy way to get solicited
 results back.  This should be reworked when proper tagging and searching
 gets fixed up.
"
  (ignore non-ok-reply-ok)
  (let ((expected-reply nil)
	(tag-or-* nil)
	(number-or-reply-type nil)
	(command-or-reply-type-arg nil)
       )
       (loop do (multiple-value-setq
		  (tag-or-* number-or-reply-type command-or-reply-type-arg)
		  (parse-something-from-imap-stream stream)
		)
	     when (eq number-or-reply-type reply-type)
	     do (setq expected-reply command-or-reply-type-arg)
	     until (eq tag tag-or-*)
	     do nil
       )
       (list tag-or-* number-or-reply-type command-or-reply-type-arg)
  )
)