;;; -*- Mode:Common-Lisp; Package:EH; Fonts:(TVFONT); Base:10 -*-

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986, 1987.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Much of this file is derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

;;; This file contains the definition of a number of new commands for the window
;;; debugger.

(load-tools '(Development-Tool-Consistency-Enhancements))

(require 'Development-Tool-Consistency-Enhancements)

(defvar *Window-debugger-Enhancements-to-add* :All
"Can have the value :All, in which case all commands are loaded,
 :Menu in which case the user is prompted, or a list of commands to load.
")

(defparameter *Window-debugger-Enhancements-Commands*
  '(("Debug Stack Group" :Value (nil nil ((Comw-Debug-Stack-Group-Cmd t))))
    ("Modify Inspect" :Value (nil nil ((Comw-Modify-Inspect t))))
   )
"This is a list of the commands which can be added to the Window-debugger.  The
 structure of this list is as follows :-
 It is made of items.  Each item is a list of the form (Menustring :Value spec)
 the spec is used to determine which commands for which tools is represented
 by the menu item.  The spec is a list of the form
 (nil nil window-debugger-commands)
 Each element of window-debugger-commands has the form
 (command-name put-in-frames-menu-p).
"
)


(defun install-Window-debugger-commands ()
"Installs all of the commands that the user wants.
"
  (tv:select-and-install-commands *Window-debugger-Enhancements-Commands*
			          *Window-debugger-Enhancements-to-add*
  )
)


;-------------------------------------------------------------------------------

;;; The Debug Stack Group Command.

;-------------------------------------------------------------------------------



(defvar *type-to-stack-group-extractor-alist*
	`((Si:Process Process-Stack-Group)
	  (stack-group Identity)
	  (tv:process-mixin
	    ,#'(lambda (x) (process-stack-group (send x :Process)))
	  )
	 )
)

(defun extract-a-stack-group (something)
"This function is passed something, which contains a stack group.  It extracts
 the stack group and returns it.  The way that it does it is to look on the
 property list of the type of the argument.  If there is a property called
 :Stack-Group-Accessor then this is used as an access function, by which the
 stack group can be extracted.  The value of this property can be either a
 symbol or a function.
"
  (let ((the-accessor (loop for (type accessor)
			    in *Type-To-Stack-Group-Extractor-Alist*
			    when (typep something type)
			    return accessor
			    finally (return nil)
		      )
	)
       )
       (If (equal nil the-accessor)
	   :Error
	   (funcall (if (symbolp the-accessor)
			(symbol-function the-accessor)
			the-accessor
		    )
		    something
	   )
       )
  )
)


(defun redefined-sg-funcall-in-frame (sg frame function &Rest args)
"This function is a redefined version of sg-funcall-in-frame.  For some reason
 that I don't understand something goes wrong with evaluating expressions in the
 lisp listener pane and nasty things happen.  This change makes sure that things
 are evaluated in the context of the stack group as a whole but not in the
 context of the current frame - a bit of a restriction, I know but not that bad
 really and still worth it to have this feature.
"
  (ignore frame)
  (sg-apply sg function args)
)

;(defun debug-a-stack-group (something)
;"This procedure is passed something that contains a stack group.  It extacts
; the stack group and calls the window debugger to debug it.  In order to do this
; it binds the special variables error-sg and current-frame, which are known to
; the debugger.  Error-Sg is the stack group which is to be debugged and
; current-frame is the stack frame pointer.  This is set to function at the
; bottom of the stack.
;"
;  (let ((*error-sg* (extract-a-stack-group something)))
;       (if (equal *error-sg* :Error)
;	   (progn (beep)
;		  (Format tv:selected-window "~&~S cannot be debugged.~%"
;			  something
;		  )
;	   )
;	   (let ((*current-frame* (sg-initial-function-index error-sg))
;		 (* nil)
;		 (** nil)
;		 (*** nil)
;		)        
;		(Declare (Special *error-sg* *current-frame* * ** ***))
;		(letf ((#'sg-funcall-in-frame #'redefined-sg-funcall-in-frame))
;		      (com-window-debugger *error-sg*
;			  (make-instance 'Ferror :Condition-Names nil
;					  :Format-String "Debug a Stack Group"
;					  :Format-Args nil
;					  :Initial-Error-Message-Printed-P t
;			  )
;		      )
;		)
;	   )
;       )
;  )
;)

(defun debug-a-stack-group (something)
"This procedure is passed something that contains a stack group.  It extacts
 the stack group and calls the window debugger to debug it.  In order to do this
 it binds the special variables error-sg and current-frame, which are known to
 the debugger.  Error-Sg is the stack group which is to be debugged and
 current-frame is the stack frame pointer.  This is set to function at the
 bottom of the stack.
"
  (let ((sg (extract-a-stack-group something)))
       (if (equal sg :Error)
	   (progn (beep)
		  (Format tv:selected-window "~&~S cannot be debugged.~%"
			  something
		  )
	   )
	   (using-resource (stream tv:background-lisp-interactors)
	     (let ((*terminal-io* stream)
		   (*debug-io* stream)
		   (*error-output* stream)
		   (*trace-output* stream)
		   (action (send stream :deexposed-typeout-action))
		  )
	          (declare (special *terminal-io* *debug-io*
				    *error-output* *trace-output*
			   )
		  )
		  (unwind-protect
		      (progn (send stream :set-deexposed-typeout-action :expose)
			     (eh sg)
		      )
		    (send stream :set-deexposed-typeout-action action)
		    (send stream :deactivate)
		  )
	     )
	   )
       )
  )
)

(defun Comw-Debug-Stack-Group (ignore ignore &aux thing)
"This is the command procedure for the window debugger, which causes the user to
 be prompted to point at a stack group or stack group containing data structure.
"
  (and (setq thing (window-read-thing
			"~&Type or mouse a process or stack group to debug:~%"
		   )
       )
       (Process-Run-Function "Window Debugger from Window Debugger"
			     #'debug-a-stack-group thing
       )
  )
)


;;; This window debugger command was copied and modified from the command, which
;;; controls the inspect option.  It defines a command called
;;; Comw-Debug-Stack-Group-Cmd, which is activated either from a menu or by the
;;; keystroke h-S.  It causes the user to be prompted for a stack group
;;; containing data structure and starts off a new debugger frame, which debugs
;;; that stack group.

(defcommand Comw-Debug-Stack-Group-Cmd nil
  '(:Description
     "Run a new debugger in a selected stack group" :Names "Dbg SG" 
    :Keys (#\h-S)
   )
   (send *window-debugger* :Set-Who-Line-Doc-String-Overide
	 "Select a stack group, or something containing one, to debug."
   )
   (unwind-protect
      (progn
	(Comw-Debug-Stack-Group *error-sg* *error-object*)
	(tv:delaying-screen-management
	    (when (equal (send *window-debugger* :configuration)
			 'step-configuration
		  )
	          (send *window-debugger* :set-configuration
			'debugger-configuration
		  )
	    )
	)
      )
      (progn (send *window-debugger* :set-who-line-doc-string-overide nil)
             (send *window-debugger* :handle-prompt)
      )
   )
)



;-------------------------------------------------------------------------------

;;; The Modify Inspect command.

;-------------------------------------------------------------------------------


(defmethod (debugger-frame :inspect-history-window) ()
"Makes the inspect-history-window instance variable visible."
  inspect-history-window
)


(defmethod (debugger-frame :inspect-window) ()
"Makes the inspect-window instance variable visible."
  inspect-window
)


(defmethod (debugger-frame :update-*)  ()
"A dummy method, because * is updated by explicit typein or by the middle
 button.
"
  nil
)


(defcommand Comw-Modify-Inspect ()
            '(:description "Modify something in the inspect pane."
              :names "ModInsp" :keys (#\c-sh-M))
  (let ((tv:history (send *window-debugger* :inspect-history-window))
	(tv:inspectors (list (send *window-debugger* :inspect-window)))
	(tv:frame *window-debugger*)
       )
       (declare (special tv:top-item tv:history tv:inspectors tv:frame))
       (setq tv:top-item
	     (tv:inspect-modify-object *terminal-io* tv:history tv:inspectors)
       )
       (tv:update-panes)
  )
)


;-------------------------------------------------------------------------------

;;; The following fix lets the value of Self be printed in the locals window
;;; for methods which would not otherwise have this happen.

;;; This facility is switched on by the variable
;;; eh:*show-self-in-locals-window-anyway* .  Set it to non-nil to
;;; enable it.  If it is set to :Always then Self is printed even for
;;; non-methods.

;;; Code by JPR.
(defvar *show-self-in-locals-window-anyway* nil
"Controls how Self is displayed in the window debugger.  Can have the values;
 Nil (default), which causes the default behaviour, :Always, which causes
 Self to be displayed in the Locals window for all functions, and any other
 non-nil value, which causes Self to be displayed in the locals window for
 all method functions.
"
)

;;; Code by JPR.
(defun is-a-method (function)
  (and (consp (function-name function))
       (equal :method (first (function-name function)))
  )
)

(defun has-a-special-value (symbol sg frame)
  (let ((sp (sg-special-pdl sg)))
       (multiple-value-bind (start end)
	   (sg-frame-special-pdl-range sg frame)
	 (ignore end)
	 (if (and start end)
;	 (if start
;	     (do ((i start (- i 2)))
;		 ((<= i 0)
;		  nil
	     (do ((i start (+ i 2)))
		 ((>= i end)
		  nil
		 )
	       (if (eq symbol
		       (symbol-from-value-cell-location (aref sp (1+ i)))
		   )
		   (multiple-value-bind (val error)
		       (catch-error (aref sp i) nil)
		     (if error
			 (return nil)
			 (return (list symbol val))
		     )
		   )
		   nil
	       )
	     )
	     nil
	 )
       )
  )
)


;;; Copied and modified by JPR from TI code for compiler:DISASSEMBLE-INSTRUCTION
compiler:
(DEFUN compiler:DISASSEMBLE-INSTRUCTION-without-printing (FEF PC &OPTIONAL VERBOSE)
  "Print on STANDARD-OUTPUT the disassembly of the instruction at PC in FEF.
Returns the length of that instruction."
  (ignore verbose)
  (LET (WD ILEN SECOND-WORD)
    (SETQ ILEN (DISASSEMBLE-INSTRUCTION-LENGTH FEF PC))
    (SETQ WD (DISASSEMBLE-FETCH FEF PC))
    (WHEN (>= ILEN 2)
      (INCF PC)
      (SETQ SECOND-WORD (DISASSEMBLE-FETCH FEF PC)))
    (DISASSEMBLE-ONE-INSTRUCTION WD SECOND-WORD FEF PC)
    ILEN))


;;; Copied and modified by JPR from TI code for compiler:DISASSEMBLE-ONE-INSTRUCTION
compiler:
(DEFUN compiler:DISASSEMBLE-ONE-INSTRUCTION-without-printing (WD &OPTIONAL SECOND-WORD FEF PC)
  "Print on STANDARD-OUTPUT the disassembly of the instruction."
  (LET ( OP SUBOP DEST DISP REG )
    (BLOCK NIL
      (SETQ OP    (LDB (BYTE 4  9) WD)
	    SUBOP (LDB (BYTE 3 13) WD)
	    DEST  (LDB (BYTE 2 14) WD)
	    DISP  (LDB (BYTE 9  0) WD)
	    REG   (LDB (BYTE 3  6) WD))
      (WHEN (< OP #o11)
	(SETQ OP (LDB (BYTE 5 9) WD)))
      (SETQ OP (ASH WD -9))
      (LET* ((NAME (AREF (INSTRUCTION-DECODE-TABLE) OP))
	     (NO-REG (GET NAME 'NO-REG)))
	   (COND
	     ((EQ NO-REG 'MISC)
	      (LET ((MISC-NAME (AREF (MISC-OP-NAME-TABLE) DISP)))
		(WHEN (AND (MEMBER MISC-NAME '(LOAD-FROM-HIGHER-CONTEXT LOCATE-IN-HIGHER-CONTEXT))
			   PC)
		  (LET (( NUM (PUSH-NUMBER-VALUE FEF (1- PC)) ))
		    (UNLESS (NULL NUM)
		      (DISASSEMBLE-LEXICAL-VAR-COMMENT
			FEF
			(LDB SI:%%CONTEXT-DESC-REL-LEVEL NUM)
			(LDB SI:%%CONTEXT-DESC-SLOT NUM)
			T) )))))
	     ((EQ NO-REG 'AREFI))
	     ((NULL NAME))
	     ((EQ NO-REG 'AUX)
	      (IF (OR (= REG 4)
		      (= REG 5))
		  nil		  
		(LET ((AUX-NAME (AREF (AUX-OP-NAME-TABLE) DISP)))
		  (COND ((<= #O160 DISP #O177)
			 (UNLESS (NULL SECOND-WORD)
			   (SETQ DISP SECOND-WORD)
			   (WHEN (>= DISP #O100000)
			     (SETQ DISP (LOGIOR #O-100000 DISP)))
			   nil ))
			((EQ AUX-NAME 'STORE-IN-HIGHER-CONTEXT)
			 (UNLESS (NULL PC)
			   (LET (( NUM (PUSH-NUMBER-VALUE FEF (1- PC)) ))
			     (UNLESS (NULL NUM)
			       (DISASSEMBLE-LEXICAL-VAR-COMMENT
				 FEF
				 (LDB SI:%%CONTEXT-DESC-REL-LEVEL NUM)
				 (LDB SI:%%CONTEXT-DESC-SLOT NUM)
				 T) ))))
			((<= #O100 DISP #O103)	   ; complex call
			 (UNLESS
			   (OR (NULL PC)
			       (LET* ((TEM (DISASSEMBLE-FETCH FEF (- PC 1)))
				      (NAME (AREF (INSTRUCTION-DECODE-TABLE)
						  (LDB (SYMEVAL-FOR-TARGET '%%QMI-FULL-OPCODE)
						       TEM))))
				 (NOT (OR (MEMBER NAME '(PUSH PUSH-LONG-FEF))
					  (EQL TEM '#.(+ (LAP-VALUE 'PUSH-MISC-GROUP)
							 (MISC-OP-EVAL '%FUNCTION-INSIDE-SELF)))
					  (AND (EQ (GET NAME 'DEST) 'D-PDL)
					       (EQ (GET NAME 'NO-REG) 'NIL)
					       (NOT (EQL (LDB (BYTE 9 0) TEM) (LAP-VALUE 'PDL-POP))))
					  ))))
			   (LET (( CALL-INFO (PUSH-NUMBER-VALUE FEF (- PC 2)) ))
			     (UNLESS (NULL CALL-INFO)
			       (DISASSEMBLE-CALL-INFO-WORD CALL-INFO)
			       ))))
			))  ) )
	     ((EQ NO-REG 'MODULE)
	      (LET ((TEM (MODULE-OP-NAME-TABLE)))
		(IF (AND TEM
			 (SETQ TEM
			       (AREF TEM
				     (LDB (SYMEVAL-FOR-TARGET '%%QMI-EXTERNAL-MODULE-NUMBER)
					  DISP))))
		    (LET ((OPNUM (LDB (SYMEVAL-FOR-TARGET '%%QMI-MODULE-OP)
				      DISP)))
		      opnum
		      nil)
		  nil)))
	     ((EQ NO-REG 'CALL)
	      (DISASSEMBLE-ADDRESS FEF REG DISP SECOND-WORD))
	     ((SYMBOLP NAME)
	      (COND
		((EQ NO-REG 'NIL) ; does use register
		 ;;12/09/85 CLM now prints PDL-PUSH instead of PDL-POP
		 (IF (AND (EQ (GET NAME 'DEST) 'D-STORE)
			  (EQl DISP (LAP-VALUE 'PDL-PUSH)))
		     nil
		   (DISASSEMBLE-ADDRESS FEF REG DISP SECOND-WORD)))
		((EQ NO-REG 'BRANCH)
		 nil
		 (WHEN (> DISP #o400)
		   (SETQ DISP (LOGIOR #o-400 DISP)))	   ;Sign-extend
		 (IF (NULL PC)
		     nil
		     (+ PC DISP 1)))
		((EQ NO-REG 'IMMED)
		 nil)
		((EQ NO-REG 'NOTHING)
		 (COND
		   ((EQ NAME 'DISPATCH)
		    (DISASSEMBLE-DISPATCH-TABLE FEF DISP))
		   ((EQ NAME 'PUSH-LONG-FEF)
		    (DISASSEMBLE-ADDRESS FEF 0 DISP NIL))
		   ((EQ NAME 'SELECT)
		    (DISASSEMBLE-SELECT-TABLE FEF DISP))
		   (T
		    (COND
		      ((EQ NAME 'LDB-IMMED)
		       (DISASSEMBLY-COMMENT)
		       nil)
		      ((EQ NAME 'LEXICAL-UNSHARE)
		       (UNLESS (NULL FEF)
			 (LET ((VARNAME
				(NTH DISP
				     (SI:GET-DEBUG-INFO-FIELD
				       (FUNCTION-DEBUGGING-INFO FEF)
				       :VARIABLES-USED-IN-LEXICAL-CLOSURES))))
			   (UNLESS (NULL VARNAME)
			     (DISASSEMBLY-COMMENT VARNAME)))))))))
		(T nil)))
	     (T nil))))
    (VALUES) ))


compiler:
(DEFUN compiler:DISASSEMBLE-ADDRESS-without-printing
       (FEF REG DISP &OPTIONAL SECOND-WORD PC
	&AUX TEM)
  "Print out the disassembly of an instruction source address.
REG is the register number of the address, and DISP is the displacement.
SECOND-WORD should be the instruction's second word if it has two.
PC should be where the instruction was found in the FEF."
  (WHEN (AND (>= REG 4) (NOT SECOND-WORD))
    (SETQ DISP (LOGAND #o77 DISP)))
  (COND
    ((= REG 5)
     (UNLESS (NULL FEF)
       (SETQ TEM (DISASSEMBLE-LOCAL-NAME FEF DISP))
       (UNLESS (NULL TEM)
	 (DISASSEMBLY-COMMENT TEM))))
    ((= REG 6)
     (UNLESS (NULL FEF)
       (SETQ TEM (DISASSEMBLE-ARG-NAME FEF DISP))
       (UNLESS (NULL TEM)
	 (DISASSEMBLY-COMMENT TEM))))
    ((EQ REG (SYMEVAL-FOR-TARGET '%QMI-REG-LEX))
     (LET ((LEVEL (LDB (BYTE 1 5) DISP))
	   (OFFSET (LDB (BYTE 5 0) DISP)))
       (DISASSEMBLE-LEXICAL-VAR-COMMENT FEF LEVEL OFFSET NIL) ))
    ((< REG 4)
     (UNLESS (NULL FEF)
       (DISASSEMBLY-COMMENT)
       (DISASSEMBLE-POINTER FEF DISP PC)))
    ((AND (= REG 7) (NOT SECOND-WORD) (= DISP 63))
     nil)
    ((EQ REG (SYMEVAL-FOR-TARGET '%QMI-REG-IVAR))
     (IF (< DISP 32)
	 (PROGN
	   (UNLESS (NULL FEF)
	     (SETQ TEM (DISASSEMBLE-INSTANCE-VAR-NAME FEF DISP))
	     (UNLESS (NULL TEM)
	       (DISASSEMBLY-COMMENT TEM " in SELF"))))
       (PROGN
	 (UNLESS (NULL FEF)
	   (SETQ TEM (DISASSEMBLE-MAPPED-INSTANCE-VAR-NAME FEF (- DISP 32)))
	   (UNLESS (NULL TEM)
	     (DISASSEMBLY-COMMENT TEM " in SELF"))))))
    (T nil))
  NIL)

compiler:
(DEFUN compiler:DISASSEMBLE-CALL-INFO-WORD-without-printing (CALL-INFO)
  ;;  9/06/86 DNG - Original.
  ;;  8/09/88 DNG - Added recognition of the CLOS bit.
  (WHEN (PLUSP (LDB (SYMEVAL-FOR-TARGET 'SI:%%call-info-lexpr-funcall-flag) CALL-INFO))
     nil)
  
  (WHEN (PLUSP (LDB (SYMEVAL-FOR-TARGET 'SI:%%call-info-self-map-table-provided) CALL-INFO))
    nil)
  (WHEN (PLUSP (LDB (SYMEVAL-FOR-TARGET 'si:%%call-info-clos-info-provided) CALL-INFO))
    nil)
  (LET ((RETURN-TYPE (LDB (SYMEVAL-FOR-TARGET 'SI:%%call-info-return-type) CALL-INFO)))
    (DECLARE (FIXNUM RETURN-TYPE))
    (COND ((EQL RETURN-TYPE (SYMEVAL-FOR-TARGET 'SI:%only-one-result-needed))
	   nil)
	  ((EQL RETURN-TYPE (SYMEVAL-FOR-TARGET 'SI:%normal-return))
	   nil)
	  ((EQL RETURN-TYPE (SYMEVAL-FOR-TARGET 'SI:%multiple-value-list-return))
	   nil)
	  ((EQL RETURN-TYPE (SYMEVAL-FOR-TARGET 'SI:%return-all-values-with-count-on-stack))
	   nil))))


compiler:
(DEFUN compiler:DISASSEMBLE-DISPATCH-TABLE-without-printing (FEF DISP)
  (DISASSEMBLY-COMMENT)
  (LET ((MAX-INDEX (%P-CONTENTS-OFFSET FEF DISP)))
    (DO ((INDEX 0 (1+ INDEX))
	 (TAB-DISP (+ DISP 2) (1+ TAB-DISP)))
	((> INDEX MAX-INDEX))))
)


(defun specials-used-by-fef (fef &optional (sg nil) (frame nil))
  (let ((specials nil)
	(old-disassemble-pointer #'compiler:disassemble-pointer)
       )
       (letf ((#'compiler:disassemble-pointer
	       #'(lambda (&rest args)
		   (let ((result (apply old-disassemble-pointer args)))
		        (if (and result
				 (symbolp result)
				 
			    )
			    (pushnew result specials)
			    nil
			)
			result
		   )
		 )
	      )
	      (#'compiler:DISASSEMBLE-INSTRUCTION
	       'compiler:DISASSEMBLE-INSTRUCTION-without-printing
	      )
	      (#'compiler:DISASSEMBLE-ONE-INSTRUCTION
	       'compiler:DISASSEMBLE-ONE-INSTRUCTION-without-printing
	      )
	      (#'compiler:DISASSEMBLE-ADDRESS
	       'compiler:DISASSEMBLE-ADDRESS-without-printing
	      )
	      (#'compiler:disassembly-comment
	       #'(lambda (&rest ignore) nil)
	      )
	      (#'compiler:DISASSEMBLE-CALL-INFO-WORD
	       'compiler:DISASSEMBLE-CALL-INFO-WORD-without-printing
	      )
	      (#'compiler:DISASSEMBLE-DISPATCH-TABLE
	       'compiler:DISASSEMBLE-DISPATCH-TABLE-without-printing
	      )
	     )
	     (do ((i 0 (1+ i))
		  (pc (fef-initial-pc fef)
		      (+ pc (compiler:disassemble-instruction-length fef pc))
		  )
		  (lim-pc (compiler:disassemble-lim-pc fef))
		 )
		 ((>= pc lim-pc))
	       (let ((*standard-output* 'si:null-stream))
		    (compiler:disassemble-instruction fef pc)
	       )
	     )
       )
       (remove nil (mapcar #'(lambda (x)
			       (if sg
				   (has-a-special-value x sg frame)
				   (if (fboundp x) nil x)
			       )
			     )
			     (nreverse specials)
		   )
       )
  )
)

(defvar *show-referenced-specials-in-locals-window* t
"When true specials that are read are shown in the debugger's locals window as
 well as those that are bound."
)

(defun boundp-in-sg (sym sg)
  (multiple-value-bind (value location boundp) (symeval-in-stack-group sym sg)
    (ignore value boundp)
    location ;; !!
  )
)

(defun value-for-locals-window (spec sg frame)
  (let ((value (has-a-special-value spec sg frame)))
       (cond (value value)
	     ((boundp-in-sg spec sg)
	      (multiple-value-bind (value location)
		  (symeval-in-stack-group spec sg)
		(list spec value location)
	      )
	     )
	     ((boundp spec)
	      (list spec (symbol-value spec) (value-cell-location spec))
	     )
	     (t (list spec '--unbound--))
       )
  )
)

(defun compute-specials-for-locals-window
       (function sg frame sp start end
	&aux (list nil) (title-p nil) (self-value nil)
	(self-mapping-table-value nil) (local-specials nil)
       )
  ;;; list is the specials list
  (if (and *show-self-in-locals-window-anyway*
	   (not self-value)
	   (or (equal :Always *show-self-in-locals-window-anyway*)
	       (is-a-method function)))
      (progn (push "" list)
	     (push "Specials bound:" list)
	     (setq title-p t)
	     (push (list 'self
			 (catch-error
			   (symeval-in-stack-group 'self sg frame) nil))
		   list)
	     (if (not self-mapping-table-value)
		 (push (list 'sys:self-mapping-table
			     (catch-error
			       (symeval-in-stack-group
				 'sys:self-mapping-table sg frame) nil))
		       list))))
  (WHEN START
    (DO ((I START (+ I 2)))
	((>= I END))
      (IF (EQ 'SELF (SYMBOL-FROM-VALUE-CELL-LOCATION (AREF SP (1+ I))))
	  (SETQ SELF-VALUE (AREF SP I)))
      (IF (EQ 'sys:self-mapping-table
	      (SYMBOL-FROM-VALUE-CELL-LOCATION (AREF SP (1+ I))))
	  (SETQ self-mapping-table-value (AREF SP I)))
      (pushnew (SYMBOL-FROM-VALUE-CELL-LOCATION (AREF SP (1+ I)))
	       local-specials :Test #'eq)
      (if (not title-p)
	  (progn (PUSH "" LIST)
		 (PUSH "Specials bound:" LIST)
		 (setq title-p t)))
      (pushnew (LIST (SYMBOL-FROM-VALUE-CELL-LOCATION (AREF SP (1+ I)))  ;Name
		  (MULTIPLE-VALUE-BIND (VAL ERROR)  ;Value
		      (CATCH-ERROR (AREF SP I) NIL)
		    (IF ERROR "unbound" VAL)))
	       LIST :Test #'equal)
    )
  )
  (multiple-value-bind
     (referenced-ivars referenced-keywords problem
      referenced-functions referenced-generic-functions args returned-values
      locals specials-referenced specials-bound
     )
      (tv:ivars-and-messages-in-method function)
    (ignore referenced-ivars referenced-keywords problem
	    referenced-functions referenced-generic-functions args
	    returned-values locals specials-bound
    )
    (let ((unbound-specials
	    (loop for spec in specials-bound
		  unless (member spec local-specials)
		  collect spec
	    )
	  )
	 )
         (if unbound-specials
	     (progn (push "" list)
		    (push "Specials not bound in execution path:" list)
		    (loop for spec in unbound-specials
			  do (push (value-for-locals-window spec sg frame) list)
		    )
	     )
	 )
    )
    (let ((specials (if (and *show-referenced-specials-in-locals-window*
			     (TYPEP function 'compiled-function)
			)
			(loop for spec in specials-referenced
			      collect (value-for-locals-window spec sg frame)
			)
			nil
		    )
	  )
	 )
         (if specials
	     (progn (push "" list)
		    (push "Referenced Specials:" list)
		    (loop for (name value) in specials
			  do (push (list name value) list)
	            )
	     )
	     nil
	 )
    )
  )
  (values list self-value)
)

;;; TI code (fixed by JPR).
(defun setup-locals-window (window sg frame rest-arg-p)
  (let* (list
         (rp (sg-regular-pdl sg))
         (function (rp-function-word rp frame))
         (sp (sg-special-pdl sg))
         start end self-value)
    ;; Print the locals if this is a fef
    (if (typep function 'compiled-function)
        (do ((n-locals (fef-number-of-locals function))
             (local-idx (sys:rp-local-offset sg rp frame))
             (i 0 (1+ i)))
;;	   (J (+ FRAME (RP-LOCAL-BLOCK-ORIGIN RP FRAME)) (1+ J)))
	    ;; * old way to find locals *
            ((>= i n-locals))
          (cond ((not (and rest-arg-p (zerop i)))
		 ;;;Don't show rest arg (local 0) twice
                 (push (list (local-name function i)   ; Local Name
;;			   (AREF RP J)
                             (aref rp (+ i local-idx)) ; Local Value
                             i)                        ; Local Number
                       list)))))
    (let ((closure-variables
	    (catch-error (get-lexical-environment-for-this-fef
			   sg frame start end function rp sp
			 )
			 nil
	    )
	  )
	 )
         (if closure-variables
	     (progn (push "" list)
		    (push "Closure Variables Visible in this Function:" list)
		    (loop for (var val loc) in (nreverse closure-variables)
			  do (push (list var val) list)
		    )
	     )
	     nil
	 )
    )
    ;; Print the specials if any
    (multiple-value-setq (start end)
                         (sg-frame-special-pdl-range sg frame))

    (multiple-value-bind (specials-list self-val)
	(compute-specials-for-locals-window function sg frame sp start end)
      (setq self-value self-val)
      (setq list (append specials-list list))
    )
    ;; if SELF is mentioned in this frame, include its instance variables:
    (if (and self-value
             (typep self-value 'instance))
        (let* ((self-flavor 
                 (si:instance-flavor self-value))
               (self-vars (si:flavor-all-instance-variables-slow self-flavor)))
          (push "" list)
          (push "Non-special instance variables of SELF:" list)
          (do ((sv self-vars (cdr sv))
               (i 1 (1+ i)))
              ((null sv))
            (cond ((not (si:assq-careful (car sv) list))
                   (push (list (car sv)
                               (multiple-value-bind (val error)       ;Value
                                  (catch-error (%instance-ref self-value i) nil)
                                 (if error "unbound" val)))
                         list))))))
    (send window :setup (list 'print-arg-or-local '(local "Local")
			      (nreverse list)))))


;-------------------------------------------------------------------------------

;;; Special searching...

(defun specials-bound-by-frame (sg frame)
  (let ((result nil))
       (multiple-value-bind (start end) (sg-frame-special-pdl-range sg frame)
	 (if start
	     (do ((i start (+ i 2))) ((>= i end))
	       (push (symbol-from-value-cell-location
		       (aref (sg-special-pdl sg) (1+ i))
		     )
		     result
	       )
	     )
	     nil
	 )
       )
       result
  )
)

(defun special-bound-in-frame (symbol sg frame)
  (let ((syms (specials-bound-by-frame sg frame)))
       (member symbol syms :test
	 #'(lambda (a b)
	     (search (symbol-name a) (symbol-name b)
		     :test #'string-equal
	     )
	   )
       )
  )
)

(defun special-used-in-frame (symbol sg frame)
  (let ((syms (specials-used-by-fef
		(rp-function-word (sg-regular-pdl sg) frame) sg frame
	      )
	)
       )
       (member symbol syms :test
	 #'(lambda (a b)
	     (search (symbol-name a) (symbol-name (ucl:first-if-list b))
		     :test #'string-equal
	     )
	   )
       )
  )
)

(defun get-something-and-search-for-it (sg from-frame)
  (let ((thing (Window-Read-thing-dont-eval
		 "Type or mouse on a special to search for: "
	       )
	)
       )
       (let ((frame (do ((frame from-frame (sg-next-frame sg frame)))
			((null frame) nil)
		      (if (if (equal 2 *numeric-arg*)
			      (special-used-in-frame  thing sg frame)
			      (special-bound-in-frame thing sg frame)
			  )
			  (return frame)
			  nil
		      )
		    )
	     )
	    )
	    frame
       )
  )
)

(defun search-for-special (sg frame)
   (send *window-debugger* :Set-Who-Line-Doc-String-Overide
	 "Select a special to search for."
   )
   (unwind-protect (get-something-and-search-for-it sg frame)
      (send *window-debugger* :set-who-line-doc-string-overide nil)
      (send *window-debugger* :handle-prompt)
   )

)


(DEFCOMMAND COMW-SEARCH-CMD (*numeric-arg*)
            '(:DESCRIPTION  
              "Search stack for a frame whose function's name contains a specified string.  With numeric arg, search stack for frame that binds [arg = 2 -> uses] a special."  
              :NAMES "Search"
	      :arguments (ucl:numeric-argument)
              :KEYS (#\c-S))
            (UNWIND-PROTECT (COMW-SEARCH *ERROR-SG* *ERROR-OBJECT*)
                            (SEND *WINDOW-DEBUGGER* :HANDLE-PROMPT)))

(DEFUN COMW-SEARCH (SG IGNORE)
  (if (and *numeric-arg* (plusp *numeric-arg*))
      (let ((frame (get-something-and-search-for-it sg *current-frame*)))
	(COND ((NULL FRAME)
	       (FORMAT T "Search failed.~%"))
	      (T
	       (SETQ *CURRENT-FRAME* FRAME)
	       (setf ucl:top-level-self
		     (symeval-in-stack-group
		       'self ucl:*stack-group* *current-frame*
		     )
	       )
	       (SEND *WINDOW-DEBUGGER* :SETUP-FRAME SG *CURRENT-FRAME*))))
	(LET (KEY FRAME)
	  (FORMAT T "String to search for (end with RETURN):~%")
	  (SETQ KEY (READ-LINE))
	  (SETQ FRAME 
		(DO ((FRAME *CURRENT-FRAME* (SG-NEXT-FRAME SG FRAME))
		     (RP (SG-REGULAR-PDL SG))
		     (NAME))
		    ((NULL FRAME) NIL)
		  (SETQ NAME (FUNCTION-NAME (RP-FUNCTION-WORD RP FRAME)))
		  (SETQ NAME (COND ((STRINGP NAME) NAME)
				   ((SYMBOLP NAME) (STRING NAME))
				   (T (FORMAT NIL "~S" NAME))))
		  (AND (SEARCH KEY NAME :TEST #'string-equal) (RETURN FRAME))))
	  (COND ((NULL FRAME)
		 (FORMAT T "Search failed.~%"))
		(T
		 (SETQ *CURRENT-FRAME* FRAME)
		 (setf ucl:top-level-self
		       (symeval-in-stack-group
			 'self ucl:*stack-group* *current-frame*))
		 (SEND *WINDOW-DEBUGGER* :SETUP-FRAME SG *CURRENT-FRAME*))))))
      

(defun com-search (sg error-object &optional numeric-arg flag)
  "Searches down the stack for a frame whose function name contains a specified string.  With numeric arg, search stack for frame that binds [arg = 2 -> uses] a special."
  (declare (special *current-frame*)
	   (ignore error-object))
  (if numeric-arg
      (let ((frame (get-something-and-search-for-it sg *current-frame*)))
	(cond ((null frame)
	       (format t "Search failed.~%"))
	      (t
	       (setq *current-frame* frame)
	       (if flag
		   (show-all sg)
		   (show-function-and-args sg)))))
      (progn
	(format t "~%String to search for (end with RETURN):~%")
	(let* ((key (read-line))
	       ;; Look for frame whose function name contains KEY.
	       ;; If found print function and its args.
	       (frame (do ((frame *current-frame* (sg-next-frame sg frame))
			   (rp (sg-regular-pdl sg))
			   (name))
			  ((null frame) nil)
			(setq name (function-name (rp-function-word rp frame)))
			(setq name (cond ((stringp name) name)
					 ((symbolp name) (string name))
					 (t (format nil "~S" name))))
			(if (search key (the string name) :test #'char-equal) 
			    (return frame)))))
	  (cond ((null frame)
		 (format t "Search failed.~%"))
		(t
		 (setq *current-frame* frame)
		 (if flag
		     (show-all sg)
		     (show-function-and-args sg))))))))


;-------------------------------------------------------------------------------


(defun show-open-catches-for-frame (sg rp frame &optional (print-p t))
   (let* ((frame-address (locf (aref rp frame)))
	  (next-frame (sg-next-frame sg frame))
	  (next-frame-address (if next-frame
				  (locf (aref rp next-frame))
				  0))
	  (open-catches
	    (loop as open-catch = (sg-catch-pointer sg)
		  then (%p-contents-offset
			 open-catch %catch-block-saved-catch-pointer)
		  until (or (null open-catch)
			    (< (%pointer-difference
				 open-catch next-frame-address) 0))
		  when (< (%pointer-difference open-catch frame-address) 0)
		  collect (%p-contents-offset
			    open-catch %catch-block-catch-tag))))
     (if print-p
	 (let ((all-open (remove t (get-all-open-catches sg rp frame nil nil))))
	      (if all-open
		  (format t
		   "~%Open catches above this frame for tag~P:  ~S~{, ~S~}."
		   (length all-open) (first all-open) (cdr all-open)
		  )
		  nil
	      )
	 )
	 nil
     )
     (if print-p
	 (let ((all-open (get-all-open-catches sg rp frame nil)))
	      (if all-open
		  (format t
		   "~%Open catches below this frame for tag~P:  ~S~{, ~S~}."
		   (length all-open) (first all-open) (cdr all-open)
		  )
		  nil
	      )
	 )
	 nil
     )
     (if print-p
	 (when open-catches
	   (format t "~%Open catches in this frame for tag~P:  ~S~{, ~S~}."
		   (length open-catches) (first open-catches)
		   (cdr open-catches)
	   )
	 )
	 open-catches
     )
   )
)

(defun get-all-open-catches (sg rp frame &optional (this-p t) (down-p t))
  (let ((previous (sg-previous-nth-frame sg frame (if down-p -1 1))))
       (if (= frame previous)
	   nil
	   (append (if this-p (show-open-catches-for-frame sg rp frame nil) nil)
		   (get-all-open-catches sg rp previous t down-p)
	   )
       )
  )
)

;-------------------------------------------------------------------------------

;;; The following fix added by JPR on 02/15/90 15:01:04 to
;;; repair the bogus/broken non-resourcifying of dbg frames.



(defvar *resourcify-window-debugger-frames* t)

(defun recycle-debugger (window)
  (process-wait "Await Dexposure"
		#'(lambda (win) (not (send win :Exposed-P)))
		window
  )
  (if (and *resourcify-window-debugger-frames*
	   (sys:resource-object-in-use window 'debugger-frame)
      )
      (if (get 'debugger-frame 'sys:no-memory)
	  (letf (((get 'debugger-frame 'sys:no-memory) nil))
	        (deallocate-resource 'debugger-frame window)
	  )
	  (deallocate-resource 'debugger-frame window)
      )
      nil
  )
)

;;;TI Code: RDA: don't kill windows that might be reused.
(DEFUN 4WINDOW-COMMAND-LOOP* (*ERROR-SG* *ERROR-OBJECT* *WINDOW-DEBUGGER*)
  "2This function calls the UCL command loop.*"
  (LET ((*evalhook* nil)
        (*NOPOINT NIL)
	(*print-pretty* nil)
	(*print-array*  *PRINT-ARRAY*)     ;1!*
	(*print-circle* *print-circle*)    ;1!*
	(*print-radix*  *print-radix*)     ;1!*
	(*print-base*   *print-base*)      ;1!*
	(*read-base*    *read-base*)       ;1!*
	(*print-level*  *print-level*)     ;1!*
	(*print-length* *print-length*)    ;1!*
        (*PACKAGE* *PACKAGE*)   
        (*window-debugger-old-window* T)
        (*TERMINAL-IO* (SEND *WINDOW-DEBUGGER* :LISP-WINDOW))
	(ucl:*default-top-level-function-execute-specials* '(*terminal-io*))
	(ucl:*default-top-level-symbols-execute-specials* '(*terminal-io*))
	(ucl:top-level-self (symeval-in-stack-group 'self ucl:*stack-group*))
        (tv-cur-win tv:selected-window)
        (tv-old-sub (AND TV:SELECTED-WINDOW (SEND TV:SELECTED-WINDOW :SELECTION-SUBSTITUTE)))
        sexp pkg)
    (DECLARE (SPECIAL pkg sexp))
    (DECLARE (SPECIAL *PRINT-ARRAY* *print-circle* *print-radix* *nopoint *print-base* *read-base* *print-level* *print-length*))
    (SEND *WINDOW-DEBUGGER* :SETUP-SG *ERROR-SG* *CURRENT-FRAME*)     ;1this used to be inside the tv:window-call ???*
    ;1;We have to do our own selection substitute stuff here because <abort> throws out of the stack-group,*
    ;1;leaving the clean-up forms undone. CLEAN-UP-WINDOW-DEBUGGER will be called by LEAVING-ERROR-HANDLER in*
    ;1;that case.*
    (SEND *window-debugger* :set-TV-CURRENT-WINDOW tv-cur-win)
    (SEND *window-debugger* :set-TV-OLD-SUBST tv-old-sub)
    (UNWIND-PROTECT (PROGN (SEND *WINDOW-DEBUGGER* :SELECT)
                           (WHEN TV-CUR-WIN
                             (SEND TV-CUR-WIN :SET-SELECTION-SUBSTITUTE *WINDOW-DEBUGGER*))
                           (SETQ *window-debugger-old-window* (OR TV-CUR-WIN T))
                           (SEND *terminal-io* :clear-screen)
                           (SEND *terminal-io* :clear-input)
                           (LET ((+ (SYMEVAL-IN-STACK-GROUP '- *ERROR-SG*))
                                 (* (SYMEVAL-IN-STACK-GROUP '* *ERROR-SG*)))
                             (SEND *window-debugger* :command-loop)))
      (TV:DELAYING-SCREEN-MANAGEMENT (WHEN TV-CUR-WIN
                                       (SEND TV-CUR-WIN :SET-SELECTION-SUBSTITUTE TV-OLD-SUB))
                                     (LET ((TV::.FLAG. (TV:SHEET-ME-OR-MY-KID-P TV:SELECTED-WINDOW
                                                                                *WINDOW-DEBUGGER*)))
				       ;;RDA: If the window might be reused, don't kill it
				       (if *resourcify-window-debugger-frames*
					   (SEND *WINDOW-DEBUGGER* :deactivate)
					   (SEND *WINDOW-DEBUGGER* :KILL))
                                       (AND TV-CUR-WIN
                                            TV::.FLAG.
                                            (SEND TV-CUR-WIN :SELECT NIL)))))))

(advise com-window-debugger :Around :check-resourcify nil
  (if (and *resourcify-window-debugger-frames*
	   (get 'debugger-frame 'sys:no-memory)
      )
      (letf (((get 'debugger-frame 'sys:no-memory) nil)) :Do-It)
      :Do-It
  )
)

(defmacro defpreserve-dbg-windows (name)
 `(advise ,name :Around :maybe-deallocate-resource nil
    (let ((window (if (boundp '*window-debugger*) *window-debugger* nil)))
	 (if window
	     (process-run-function '(:Name "Recycle Dbg" :Priority -1)
				   'recycle-debugger window
	     )
	     nil
	 )
	 :Do-It
    )
  )
)

(Defpreserve-Dbg-Windows com-top-level-throw)

(Defpreserve-Dbg-Windows sg-unwind-to-frame-and-reinvoke)


;;; Modified TI code.
(defun com-proceed (*error-sg* error-object &optional numeric-arg)
  "Proceeds from this error if possible."
  (declare (special *error-sg* *error-handler-running* *special-commands* error-object)
	   (ignore numeric-arg))
  (when (not *error-handler-running*)
    (throw 'exit t))				; There is a CATCH 'EXIT in function DEBUG
  (format t "~&")
  (let ((proceed-types (append (send error-object :user-proceed-types
				    (sg-condition-proceed-types *error-sg* error-object))
			       *special-commands*))
	proceed-type)
    (cond ((null proceed-types)
	   (format t "~&There is no way to proceed from this error.~%"))
	  ;; Resume key is assigned to the first proceed type
	  ;; if it is atomic (and not a resume handler)
	  ((and (atom (first proceed-types))
		(not (and (find-package 'cleh)
			  (type-specifier-p 'cleh:restart)
			  (typep (first proceed-types) 'cleh:restart))))
	   (setq proceed-type (first proceed-types)))
	  ;; Resume key could also be assigned to a CONTINUE restart
	  ;; Look for a Common LISP restart with name CONTINUE
	  ;; The first continue restart found is assigned to the resume key
	  ;; - mjf/clm 7/20/88
	  ((and (find-package 'cleh)
		(fboundp 'cleh:restartp)
		(setq proceed-type 
		      (car (member :continue proceed-types :test #'cleh:restartp)))))
	  (:else
	    (format t "~&There are no atomic proceed types.  There are no Continue restarts. 
There are only ways to restart command loops.~%")))
    (if proceed-type
	;;; Modified here by JPR.
	(go-ahead-and-proceed error-object proceed-type))
    nil))

(defun go-ahead-and-proceed (error-object proceed-type)
  (send error-object :proceed-asking-user proceed-type
	'proceed-error-sg
	'read-object
  )
)


(Defpreserve-Dbg-Windows go-ahead-and-proceed)

;-------------------------------------------------------------------------------

(defwhopper (debugger-frame :set-sensitive-item-types) (val)
  (let ((new-val (loop for type in val
		       for spec = (get type 'special-item-type)
		       append (if spec (list type spec) (list type))
		 )
	)
       )
       (continue-whopper new-val)
  )
)

(defvar *prefered-arg-and-local-print-case* :Upcase)

(DEFUN PRINT-ARG-OR-LOCAL (ITEM TYPE STREAM IGNORE)
  (LET (NUMBER NAME VALUE TYPE-NAME NOVALUE ERROR)
    (IF (STRINGP ITEM)
        (SEND STREAM :STRING-OUT ITEM)
	(1if* (1consp* type)
	    (PROGN
	      (SETQ TYPE-NAME (SECOND TYPE)
		    TYPE (FIRST TYPE)
		    NAME (FIRST ITEM)
		    VALUE (SECOND ITEM)
		    NUMBER (THIRD ITEM))
	      (AND (CONSP NUMBER)
		   (SETQ NOVALUE (FIRST NUMBER)
			 NUMBER (SECOND NUMBER)))
	      (COND ((NULL NUMBER))
		    ((STRINGP NUMBER)
		     (SEND STREAM :STRING-OUT NUMBER))
		    (T (FORMAT STREAM "~A ~D" TYPE-NAME NUMBER)))
	      (AND NAME (COND (NUMBER
			       (SEND STREAM :STRING-OUT " (")
			       (SEND STREAM
				     :ITEM1 (LIST NAME NUMBER)
				     (or (and (get name 'special)
						(get TYPE 'special-item-type))
					   type)
				     (FUNCTION (LAMBDA (X STREAM)
						 (IF (CONSP (CAR X))
						     (SEND STREAM :String-Out
							   (let ((*Print-Case* *Prefered-arg-and-local-print-case*))
							     (FORMAT NIL "~A" (CAR X))))
						     (SEND STREAM :String-Out
							   (1string-capitalize*
							     (SYMBOL-NAME (CAR X))))))))
			       (WRITE-CHAR #\) stream))   ;(SEND STREAM :TYO #\)))
			      (T (SEND STREAM :ITEM1 NAME
				       (or (and (1symbolp* name)
						(get name 'special)
						(1symbolp* type)
						(get TYPE 'special-item-type))
					   type)
				       #'(lambda (x stream)
					   (let ((*Print-Case* *Prefered-arg-and-local-print-case*))
					     (1prin1* x stream)))))))
	      (COND ((NEQ NOVALUE :NOVALUE)
		     (SEND STREAM :STRING-OUT ": ")
		     (if (eq '--unbound-- value)
			 (SEND STREAM :STRING-OUT "Unbound")
			 (MULTIPLE-VALUE-SETQ (NIL ERROR)
			   (CATCH-ERROR (SEND STREAM :ITEM1 VALUE
					      :VALUE (FUNCTION TV::PRINT-ITEM-CONCISELY))
					NIL))
		     )
		     (IF ERROR (SEND STREAM :STRING-OUT "<<unprintable>>")))))))))


(defvar *Default-Dbg-Mouse-Sensitive-Item-Types*
	'(:Value :FUNCTION STACK-FRAME special-arg special-local)
)

(setf (get 'local 'special-item-type) 'special-local)
(setf (get 'arg   'special-item-type) 'special-arg)

(DEFUN COM-WINDOW-DEBUGGER (SG ETE) 
  "Use a window-based debugger to debug the stack."
  (IF (EQ *TERMINAL-IO* sys:COLD-LOAD-STREAM) ;sys: is temp
   (FORMAT T
    "~&The window-based debugger cannot be invoked since we are using the cold load stream.")
   (TV:REMOVE-SYSTEM-KEY (char-int #\W))
   (TV:ADD-SYSTEM-KEY (char-int #\W) 'DEBUGGER-FRAME
                      "Window-based Debugger - debug a program with menus and mouse interaction."
                      NIL)
   (TV::DELETE-FROM-SYSTEM-MENU-COLUMN :DEBUG "Debugger")	;In case one is already out there
   (TV::ADD-TO-SYSTEM-MENU-COLUMN :DEBUG "Debugger"
     '(LET ((W (TV:FIND-WINDOW-OF-FLAVOR 'DEBUGGER-FRAME)))
        (COND (W
               (SEND W :SELECT)
               (SEND w :update-print-info))   ;!
              (t (BEEP))))
     "Select a Window-based Debugger Frame.")
;!   (tv:set-print-info)      ;Make sure *inspect-print-base*, etc. is correct
   (USING-RESOURCE (WINDOW DEBUGGER-FRAME TV:DEFAULT-SCREEN)
     (SEND WINDOW :SET-SENSITIVE-ITEM-TYPES *default-dbg-mouse-sensitive-item-types*)
     (UNWIND-PROTECT 
	 (WINDOW-COMMAND-LOOP SG ETE WINDOW)
       (UNLESS (more-window-debuggers-p)
    (TV:REMOVE-SYSTEM-KEY (char-int #\W))
    (TV::DELETE-FROM-SYSTEM-MENU-COLUMN :DEBUG "Debugger"))))))


(DEFUN WINDOW-READ-OBJECT (KEYWORD &REST FORMAT-STRING-AND-ARGS)
  (LET (SPECIAL SEXP ASK-P OLD-SI-TYPES)
    (COND
      ((EQ KEYWORD :EVAL-READ)
       (SETQ OLD-SI-TYPES (SEND *WINDOW-DEBUGGER* :SENSITIVE-ITEM-TYPES))
       (UNWIND-PROTECT
           (PROG NIL
                 (SEND *WINDOW-DEBUGGER* :SET-SENSITIVE-ITEM-TYPES  *default-dbg-mouse-sensitive-item-types*)
              RETRY
                 (APPLY 'FORMAT T FORMAT-STRING-AND-ARGS)
                 (MULTIPLE-VALUE-SETQ (SPECIAL SEXP) (WINDOW-COMMAND-LOOP-READ))
                 (COND ((CONSP SPECIAL)
                        (IF (SEND *WINDOW-DEBUGGER* :INSPECT-WINDOW-P (THIRD SPECIAL))
                            (SETQ SEXP (TV::INSPECT-REAL-VALUE SPECIAL)
                                  ASK-P T)
                            (LET ((TYPE (FIRST SPECIAL)))
                              (COND ((EQ TYPE :VALUE)
                                     (SETQ SEXP (SECOND SPECIAL)
                                           ASK-P T))
                                    ((AND (EQ TYPE :MENU)
                                          (MEMBER (SETQ SEXP (SEND (FOURTH SPECIAL) :EXECUTE (SECOND SPECIAL)))
                                                  '(T NIL-VALUE)
                                                  :TEST (FUNCTION EQ)))
                                     (SETQ SEXP  (EQ SEXP T)
                                           ASK-P NIL))
                                    (T (BEEP) (GO RETRY))))))
                       (T (SETQ ASK-P (SI:TRIVIAL-FORM-P SEXP)
                                SEXP (LET ((OTOC (SG-FLAGS-TRAP-ON-CALL *ERROR-SG*)))
                                       (SETF (SG-FLAGS-TRAP-ON-CALL *ERROR-SG*) 0)
                                       (PROG1 (CAR (SG-EVAL *ERROR-SG* SEXP T))
                                              (SETF (SG-FLAGS-TRAP-ON-CALL *ERROR-SG*) OTOC))))))
                 (AND ASK-P (COND ((NOT (WINDOW-Y-OR-N-P "The object is ~S, ok? (Y or N) " SEXP))     ;.
                                   (TERPRI)
                                   (GO RETRY))))
                 (RETURN SEXP))
         (SEND *WINDOW-DEBUGGER* :SET-SENSITIVE-ITEM-TYPES OLD-SI-TYPES)))
      ((EQUAL KEYWORD '(:FQUERY)) (APPLY 'WINDOW-Y-OR-N-P FORMAT-STRING-AND-ARGS))
      (T (APPLY 'PROMPT-AND-READ KEYWORD FORMAT-STRING-AND-ARGS)))))


(DEFUN WINDOW-READ-FUNCTION (ACTION &OPTIONAL ALLOW-T RETURN-STACK-FRAMES)
  (LET (SPECIAL FUNCTION)
    (FORMAT T
            "~&Type or mouse a function ~A (NIL aborts~:[, T ~A nothing~]):~%"
            ACTION
            (NOT ALLOW-T)
            ACTION)
    (MULTIPLE-VALUE-SETQ (SPECIAL FUNCTION) (WINDOW-COMMAND-LOOP-READ))
    (AND SPECIAL (SETQ FUNCTION (IF (SEND *WINDOW-DEBUGGER* :INSPECT-WINDOW-P (THIRD SPECIAL))
                                    (TV::INSPECT-REAL-VALUE SPECIAL)
                                    (CASE (FIRST SPECIAL)
                                      (:MENU  (AND (EQ (SEND (FOURTH SPECIAL)
                                                             :EXECUTE
                                                             (SECOND SPECIAL))
                                                       T)
                                                   ALLOW-T))
                                      (STACK-FRAME  (IF
                                                      RETURN-STACK-FRAMES
                                                      SPECIAL
                                                      (STACK-FRAME-FUNCTION-AND-ARGS
                                                        *ERROR-SG*
                                                        (SECOND SPECIAL))))
                                      (:LINE-AREA  (IF
                                                     RETURN-STACK-FRAMES
                                                     (LIST 'STACK-FRAME (SECOND SPECIAL))
                                                     (STACK-FRAME-FUNCTION-AND-ARGS
                                                       *ERROR-SG*
                                                       (SECOND SPECIAL))))
                                      ((SPECIAL ARG special-arg LOCAL special-local)  (FIRST (SECOND SPECIAL)))
                                      ((:VALUE :FUNCTION SPECIAL)  (SECOND SPECIAL))))))
    (WHEN (CLOSUREP FUNCTION)
      (SETQ FUNCTION (CLOSURE-FUNCTION FUNCTION)))
    (COND ((TYPEP FUNCTION 'INSTANCE)
           (SETQ SPECIAL (WINDOW-READ-THING "~&Type or mouse a message name for ~S:~%" FUNCTION))
           (LET ((HANDLER (GET-HANDLER-FOR FUNCTION SPECIAL)))
             (OR HANDLER (FORMAT T "~&~S does not handle the ~S message.~%" FUNCTION SPECIAL))
             (SETQ FUNCTION HANDLER)))
          ((NULL FUNCTION)
           (FORMAT T "~&Aborted.~%")))
    FUNCTION))

(DEFUN WINDOW-READ-THING (PROMPT &REST FORMAT-ARGS)
  (LET (SPECIAL THING)
    (APPLY (FUNCTION FORMAT) T PROMPT FORMAT-ARGS)
    (MULTIPLE-VALUE-SETQ (SPECIAL THING) (WINDOW-COMMAND-LOOP-READ))
    (SETQ THING (IF SPECIAL
                    (IF (SEND *WINDOW-DEBUGGER* :INSPECT-WINDOW-P (THIRD SPECIAL))
                        (TV::INSPECT-REAL-VALUE SPECIAL)
                        (CASE (FIRST SPECIAL)
                          (:MENU  (EQ (SEND (FOURTH SPECIAL) :EXECUTE (SECOND SPECIAL)) T))
                          (STACK-FRAME  (LIST-STACK-FRAME-FUNCTION-AND-ARGS *ERROR-SG*
                                                                            (SECOND SPECIAL)))
                          (:LINE-AREA  (LIST-STACK-FRAME-FUNCTION-AND-ARGS *ERROR-SG*
                                                                           (SECOND SPECIAL)))
                          ((SPECIAL ARG special-arg LOCAL special-local)  (FIRST (SECOND SPECIAL)))
                          ((:VALUE :FUNCTION SPECIAL)  (SECOND SPECIAL))))
                    (car (SG-EVAL-in-frame *ERROR-SG* THING *current-frame* T))))       ;.. take frame into consideration
    (IF (NULL THING) (FORMAT T "~&Aborted.~%"))
    THING))

;-------------------------------------------------------------------------------

;;; Redefine print-carefully so that it catches any errors that might
;;; occur during the printing of the error message.  Otherwise lossage can
;;; occur pretty badly.

eh:
(defmacro eh:print-carefully (message &body body)
  2"If BODY gets an error, print a message saying error printing."*
  `(condition-case (condition)
       (progn . ,body)
     (error (1catch-error* (format t "... Error while printing ~A: ~A" ,message condition) nil))))


eh:
(defun eh:print-initial-error-message (sg error-object)
  2"Print error message and backtrace for error-object when in sg."*
  (print-carefully "error message"
    (send *standard-output* :fresh-line)
    (send error-object :print-error-message sg nil *standard-output*))
  (print-brief-error-backtrace sg error-object)
  (send error-object :maybe-clear-input *standard-input*))


;-------------------------------------------------------------------------------

(provide 'Window-Debugger-Enhancements)

(install-Window-debugger-commands)