;;; -*- Mode:Common-Lisp; Package:SYSTEM-INTERNALS; Base:10 -*-


;;; NOTE:  We really should also cater for slot/iv use in methods.


(eval-when (compile load eval) (defvar *all-who-calls-report-types* nil))

(defmacro defwho-calls-report-type (keyword &body string)
  `(progn (pushnew ',keyword *all-who-calls-report-types*)
	  (sys:record-source-file-name ',keyword 'who-calls-type)
	  (defmethod Who-Calls-Format-String ((how (eql ',keyword)))
	    (declare (sys:function-parent ,keyword))
	    ,@string
	  )
	  ',keyword
   )
)

(eval-when (compile load eval) (defvar *all-who-calls-sub-functions* nil))

(defmacro def-who-calls-function (name (&rest arglist) &body body)
 `(progn (pushnew ',name *All-Who-Calls-Sub-Functions*)
	 (defun ,name ,arglist ,@body)
  )
)

;-------------------------------------------------------------------------------

(defmethod Who-Calls-Format-String ((how t)) " uses ~S somehow.") ;; Stub
(defwho-calls-report-type :Variable " uses ~S as a variable.")
(defwho-calls-report-type :function " calls ~S as a function.")
(defwho-calls-report-type :constant " uses ~S as a constant.")
(defwho-calls-report-type :flavor " uses ~S's flavor definition.")
(defwho-calls-report-type :macro " calls ~S as a macro.")
(defwho-calls-report-type setf " calls function (SETF ~S).")
(defwho-calls-report-type locf " calls function (LOCF ~S).")
(defwho-calls-report-type nil ", an interpreted function, uses ~S somehow.")
(defwho-calls-report-type :Instruction
  " uses an instruction for the ~S function."
)
(defwho-calls-report-type :Unbound-Function
  " calls ~S, which is an undefined function."
)

;-------------------------------------------------------------------------------

(defvar *who-calls-cache* (make-hash-table :Test #'eq))

(defun who-calls (symbol-or-symbols &optional pkg (inheritors t) (inherited t)
		  (inhibit-previous-definitions-p t) (types-to-ignore nil)
		  (display-only-these-types nil only-these-supplied-p))
 "Find all symbols in package PKG whose values, definitions or properties
 flavors, classes plists, etc. use SYMBOL.
 
 SYMBOL-OR-SYMBOLS can be a symbol or a list of symbols, each of which is
 looked for.
 
 PKG defaults to NIL, which means search all packages.
 
 The packages which inherit from PKG are processed also,
 unless INHERITORS is NIL.
 
 The packages PKG inherits from are processed also, unless INHERITED is NIL.
 \(Other packages which merely inherit from the same ones are NOT processed.)
 
 When INHIBIT-PREVIOUS-DEFINITIONS-P is true then
 (:property foo :previous-definition) type callers are skipped.
 
 TYPES-TO-IGNORE is a list of classes of call that should be skipped.
 For example, if you are looking for uncompiled calls to typep then you will
 get a lot of output because of all of the compiled calls to typep and the
 instructions they turn into.  If you specify :Instruction as a type to
 ignore then these will be skipped.  The list of supported caller types
 is kept in *all-who-calls-report-types*, which is currently:
    :CLOS-SLOT
    :FLAVOR-IV
    :PLIST
    :DBIS-PLIST
    :LOCAL
    :ARGLIST
    :UNBOUND-FUNCTION
    :INSTRUCTION
    NIL
    LOCF
    SETF
    :MACRO
    :FLAVOR
    :CONSTANT
    :FUNCTION
    :VARIABLE) 

 DISPLAY-ONLY-THESE-TYPES is a list of classes that are the set that you
 are explicitly interested in.  For example, if you were only interested
 in the uses of foo as a constant, then this arg would be '(:constant).
 
 Callers are printed and a list of them is returned.
 
 The symbol :UNBOUND-FUNCTION is special:  (WHO-CALLS :UNBOUND-FUNCTION)
 will find all functions that are used but not currently defined."
  (let ((types-to-ignore (if (listp types-to-ignore) types-to-ignore (list types-to-ignore)))
	(display-only-these-types
	  (if (listp display-only-these-types) display-only-these-types (list display-only-these-types)))
	(return-list nil))
    (declare (special return-list))
    (unwind-protect
      (find-callers-of-symbols symbol-or-symbols pkg
			       #'(lambda (caller callee how)
				   (cond ((and inhibit-previous-definitions-p
					       (consp caller)
					       (equal :Property (first caller))
					       (equal :Previous-Definition
						      (third caller)
					       )
					  )
					  nil
					 )
					 ((and only-these-supplied-p (not (member how display-only-these-types))) nil)
					 ((member how types-to-ignore) nil)
					 (t (format t "~&~S" caller)
					    (format
					      t (who-calls-format-string how)
					      callee
					    )
					    (push caller return-list)
					 )
				   )
				 )
			       inheritors inherited)
      (clrhash *who-calls-cache*))
    return-list))

(defmethod documentation ((symbol (eql 'who-calls)) &optional (type 'function))
  (case type
    (function
     (string-append
   " Find all symbols in package PKG whose values, definitions or properties
 flavors, classes plists, etc. use SYMBOL.
 
 SYMBOL-OR-SYMBOLS can be a symbol or a list of symbols, each of which is
 looked for.
 
 PKG defaults to NIL, which means search all packages.
 
 The packages which inherit from PKG are processed also,
 unless INHERITORS is NIL.
 
 The packages PKG inherits from are processed also, unless INHERITED is NIL.
 \(Other packages which merely inherit from the same ones are NOT processed.)
 
 When INHIBIT-PREVIOUS-DEFINITIONS-P is true then
 (:property foo :previous-definition) type callers are skipped.
 
 TYPES-TO-IGNORE is an &rest arg of classes of call that should be skipped.
 For example, if you are looking for uncompiled calls to typep then you will
 get a lot of output because of all of the compiled calls to typep and the
 instructions they turn into.  If you specify :Instruction as a type to
 ignore then these will be skipped.  The currently supported set of
 Who-Calls types is:
"
   (with-output-to-string (*standard-output*)
     (loop for type in (sort (copy-list *All-Who-Calls-Report-Types*)
			     #'string<
		       )
	   For string = (Who-Calls-Format-String type)
	   do (format t "~&  ~S~20TWhen something~A"
		      type (format nil string "it")
	      )
     )
   )
 "
 The symbols are printed and a list of them is returned.
 
 The symbol :UNBOUND-FUNCTION is special:  (WHO-CALLS :UNBOUND-FUNCTION)
 will find all functions that are used but not currently defined."
 ))
  (otherwise (clos:call-next-method))))

(defun find-callers-of-symbols (symbol pkg function &optional (inheritors t) (inherited t))
  "This is the main driving function for WHO-CALLS and friends.
Looks at all symbols in PKG and USErs (if INHERITORS is T)
and the ones it USEs (if INHERITED is T).
If PKG is NIL, looks at all packages.
Looks at each symbol's function definition and if it
refers to SYMBOL calls FUNCTION with the function name, the symbol used,
and the type of use (:VARIABLE, :FUNCTION, :MISC-FUNCTION,
 :CONSTANT, :UNBOUND-FUNCTION, :FLAVOR,
 or NIL if used in an unknown way in an interpreted function.)
SYMBOL can be a single symbol or a list of symbols.
The symbol :UNBOUND-FUNCTION is treated specially."
  
  ;; Sorting first, in order of function definitions, didn't help much when
  ;; tried in the previous generation of this function.
  (when pkg
    (setq pkg (find-package pkg)))
;  (check-arg symbol
;	     (or (symbolp symbol)
;		 (loop for sym in symbol always (symbolp sym)))
;	     "a symbol or a list of symbols")
  (if (not (consp symbol))
      (setq symbol (list symbol)))
  (dolist (sym symbol)
    (if (symbolp sym)
	(setq symbol (add-symbols-optimized-into sym symbol))))
  ;;  Since we can't find everything.
  (loop for sym in symbol
	when (and (symbolp sym) (get sym 'compiler:post-optimizers))
	do (format t "~&~S has a COMPILER:POST-OPTIMIZERS property:  some uses may actually use another symbol." sym))
  ;; If one of the symbols is :PUTPROP, say, make sure we look for GLOBAL:PUTPROP too.
  (let (tem)
    (dolist (sym symbol)
      (when (and (symbolp sym)
		 (eq (symbol-package sym)  *keyword-package*)
		 (setq tem (find-symbol sym  *lisp-package* )))
	(push tem symbol))))
  (cond (pkg
	 (if inherited
	     (do-symbols (s pkg)
	       (find-callers-of-symbols-aux s symbol function))
	     (do-local-symbols (s pkg)
	       (find-callers-of-symbols-aux s symbol function)))
	 (when inheritors
	   (dolist (p (package-used-by-list pkg))
	     (do-local-symbols (s p)
	       (find-callers-of-symbols-aux s symbol function)))))
	(t
	 (dolist (p (list-all-packages))
	   (when (neq p *global-package*)
	     (do-local-symbols (s p)
	       (find-callers-of-symbols-aux s symbol function))))))
  nil)

;-------------------------------------------------------------------------------

(defun find-callers-of-symbols-aux-list1
       (caller defn symbol function &optional (how nil))
  (declare (special suppress))
  (loop for l on defn
	until (atom l)
	finally (if (not (null l))
		    (find-callers-of-symbols-aux1 caller l symbol function))
	as carl = (car l)
	doing (cond ((and ;(symbolp carl)  ;;; JPR
			  (not (member carl (the list suppress) :test #'eq))
			  (if (atom symbol)
			      (eq carl symbol)
			      (member carl (the list symbol) :test #'eq)))
		     (push carl suppress)
		     (funcall function caller carl how))
		    ((listp carl)
		     (Find-Callers-Of-Symbols-Aux-List1
		       caller carl symbol function how))
		    (t
		     (Find-Callers-Of-Symbols-Aux1
		       caller carl symbol function)))))


;-------------------------------------------------------------------------------

(Defun ARGLIST-and-locals (function &OPTIONAL real-flag)
  (DECLARE (VALUES ARGLIST locals))
  (eTYPECASE function
    (symbol 
     (IF (FBOUNDP function)
	 (ARGLIST-AND-LOCALS
	   (fdefinition (unencapsulate-function-spec function)) real-flag)
	 (LET ((arglist-property (GET function 'arglist ':default))) ;; for things like %call,%push, etc.
	   (IF (listp arglist-property)
	       arglist-property
	       (FERROR NIL "~S is neither a function nor a function spec" function)))))
    (compiled-function
     (LET ((debug-info (GET-DEBUG-INFO-STRUCT function)))
       (IF real-flag
	   (VALUES
	     (GET-DEBUG-INFO-FIELD debug-info :ARGLIST) ; first value -- the argument list
	     (GET-DEBUG-INFO-FIELD debug-info :local-map))
	   (values
	     ;;arglist connot be a non null symbol so it is safe to use a symbol as
	     ;;an empty marker.
	     (let ((val (GET-DEBUG-INFO-FIELD debug-info :DESCRIPTIVE-ARGLIST 'empty)))
		 (if (eq 'empty val) (GET-DEBUG-INFO-FIELD debug-info :ARGLIST) val))
	     (GET-DEBUG-INFO-FIELD debug-info :local-map)))))
    (cons
     (CASE (CAR function)
	   ((CLI:LAMBDA GLOBAL:LAMBDA) (ALL-ARGLIST-BEFORE-&AUX (CADR function)))
	   ((CLI:SUBST GLOBAL:SUBST) (VALUES (CADR function) NIL))
	   ((NAMED-SUBST NAMED-LAMBDA CLOSURE-NAMED-LAMBDA GLOBAL:NAMED-SUBST GLOBAL:NAMED-LAMBDA)
	    (LET ((debug-info (GET-DEBUG-INFO-STRUCT function)))   ;; this is meaningful for encapsulations
	      ;; remember here that debug-info for interpreted functions is a list
		(IF real-flag
		    (values (ALL-ARGLIST-BEFORE-&AUX (CADDR function))
			    ())
		    (values
		      (let ((val (GET-DEBUG-INFO-FIELD debug-info :DESCRIPTIVE-ARGLIST 'empty)))
			(if (eq 'empty val)
			    (let ((val (getf debug-info :ARGLIST 'empty)))
			      (if (eq 'empty val)
				  (ALL-ARGLIST-BEFORE-&AUX (CADDR function))
				  val))
			    val))
		      (getf  debug-info :VALUES)))))
	   (MACRO
	    (LET ((macro-function (CDR function)))
	      (multiple-value-bind (argl values)
		  (ARGLIST-AND-LOCALS macro-function real-flag)
		(values argl values))))
	   (T
	    (IF (VALIDATE-FUNCTION-SPEC function)
		(ARGLIST-AND-LOCALS (FDEFINITION (unencapsulate-function-spec function)) real-flag)
		(FERROR NIL "~S not a recognized function" function)))
	   ))
    (stack-group '(STACK-GROUP-ARG))
    (array
     (DO ((I (%P-LDB %%ARRAY-NUMBER-DIMENSIONS function) (1- I))
	  (L NIL))
	 ((<= I 0) L)
       (SETQ L (CONS (INTERN (FORMAT NIL "DIM-~D" I) PKG-SYSTEM-INTERNALS-PACKAGE) L))))
    ((OR closure lexical-closure) (ARGLIST-AND-LOCALS (CLOSURE-FUNCTION function) real-flag))
    (instance '(OP &REST METHOD-ARGS-VARY))   ;; Can't tell arglist, shouldn't give error though
    (microcode-function 
     (GET-DEBUG-INFO-FIELD (EXTRACT-DEBUG-INFO-STRUCT-FROM-UCODE function) :ARGLIST))))


(defwho-calls-report-type :arglist " mentions ~S in its arglist.")

(defwho-calls-report-type :local " uses ~S as a local variable.")

(let ((compiler:compile-encapsulations-flag t))
  (advise find-callers-of-symbols-aux-fef :around :look-at-arglist nil
	  :do-it
	  (destructuring-bind (caller defn symbol function) arglist
	    (let ((suppress nil))
	      (declare (special suppress))
	      (multiple-value-bind (args locals) (arglist-and-locals defn)
		(Find-Callers-Of-Symbols-Aux-List1
		  caller args symbol function :arglist)
		(Find-Callers-Of-Symbols-Aux-List1
		  caller locals symbol function :local))))))

;-------------------------------------------------------------------------------

(defwho-calls-report-type :Dbis-Plist
  " mentions ~S in its debug-info-struct plist."
)

(defun Find-Callers-Of-Symbols-Aux1
       (caller defn symbol function &optional (known-to-be-function-p nil))
  ;; Don't be fooled by macros, interpreted or compiled.
  (when (or known-to-be-function-p (functionp defn t))
    (when (and (consp defn) (eq (car defn) 'macro))
      (setq defn (cdr defn)))
    (typecase defn
      (compiled-function (find-callers-of-symbols-aux-fef caller defn symbol function))
      (list (find-callers-of-symbols-aux-lambda caller defn symbol function))
      (closure
       (when (eql (%data-type defn) dtp-closure)
	 (dolist (sym (closure-variables defn))
	   (when (if (atom symbol)
		     (eq sym symbol)
		   (member sym (the list symbol) :test #'eq))
	     (funcall function caller sym :variable))))
       (find-callers-of-symbols-aux1 caller (closure-function defn) symbol
				     function t)))
    ;; If this function is traced, advised, etc.
    ;; then look through the actual definition.
    (when (or (listp defn) (typep defn 'compiled-function))
      (let* ((debug-info  (get-debug-info-struct defn))
	     (inner  (car (get-debug-info-field debug-info 'si:encapsulated-definition))))
	(when inner
	  (find-callers-of-symbols-aux inner symbol function))))
    (locally
      (declare (notinline ticlos:generic-function-p ticlos:generic-function-methods ticlos:method-function))
      (when (and (ticlos:generic-function-p defn)
		 (not (symbolp defn)))
	(dolist (method (ticlos:generic-function-methods defn))
	  (let ((fef (ticlos:method-function method)))
	    (find-callers-of-symbols-aux1 (function-name fef) fef symbol
					  function t)))))
    (when (consp (function-name defn)) ;; Added by JPR.
      (let ((suppress nil))
	(declare (special suppress))
	(Find-Callers-Of-Symbols-Aux-List1
	  caller (dbis-plist (get-debug-info-struct defn))
	  symbol function :dbis-plist)))
    (values)))

;-------------------------------------------------------------------------------

(def-who-calls-function who-calls-for-fef (caller symbol function)
  (loop for (prop value) on (symbol-plist caller) by #'cddr
	if (= (%data-type value) dtp-function)
	do (find-callers-of-symbols-aux-fef (list :property caller prop)
					    value symbol function)
	else if (and (consp value)
		     (consp (car value))
		     (consp (cdr (car value)))
		     (eq caller (second (car value)))
		     (si:validate-function-spec (car value)))
	do (let ((defn (fdefinition-safe (car value) nil)))
	     (when (and defn (member defn (cdr value) :test #'eq))
	       ;; here for SETF and LOCF functions
	       (find-callers-of-symbols-aux1 (car value)
					     (fdefinition-safe (car value) t)
					     symbol
					     function)))))

;-------------------------------------------------------------------------------

(defwho-calls-report-type :plist " mentions ~S in its plist.")

(Def-Who-Calls-Function who-calls-for-plist (caller symbol function)
  (let ((suppress nil))
    (declare (special suppress))
    (Find-Callers-Of-Symbols-Aux-List1
      caller (lambda-exp-args-and-body (symbol-plist caller))
      symbol function :plist)))

;-------------------------------------------------------------------------------

(defwho-calls-report-type :Flavor-Iv
  "'s flavor definition uses ~S as an instance variable name."
)

(defun find-callers-of-symbols-ivs (caller fl symbols function)
  (let ((ivs (flavor-all-instance-variables fl)))
       (loop for sym in symbols do
	     (loop for iv? in ivs
		   for iv = (ucl:first-if-list iv?)
		   when (eq sym iv)
		   do (funcall function caller sym :flavor-iv)))))

(Def-Who-Calls-Function who-calls-for-flavor (caller symbol function)
  ;; Also look for flavor methods
  (let (fl)
    (when (and (setq fl (get caller 'flavor))
	       (arrayp fl))			;Could be T
      (find-callers-of-symbols-ivs caller fl symbol function)
      (dolist (mte (flavor-method-table fl))
	(dolist (meth (cdddr mte))
	  (if (meth-definedp meth)
	      (find-callers-of-symbols-aux1 (meth-function-spec meth)
					    (meth-definition meth)
					    symbol function t)))))))

;-------------------------------------------------------------------------------

(defwho-calls-report-type :Clos-Slot
  "'s class definition uses ~S as a slot name."
)

(defun find-callers-of-symbols-slots (caller cl symbols function)
  (let ((slots (tv:class-local-slots-safe cl)))
    (loop for sym in symbols do
	  (loop for slotd in slots
		for name = (clos:slot-definition-name slotd)
		when (eq sym name)
		do (funcall function caller sym :clos-slot)))))

(defun get-class-direct-gfs (a-class &optional (top-class a-class))
  (multiple-value-bind (gfs found-p) (gethash a-class *Who-Calls-Cache*)
    (if found-p
	gfs
	(let ((gfs (tv:Ticlos-Class-Direct-Generic-Functions a-class top-class)
	      )
	     )
	     (Setf (gethash a-class *Who-Calls-Cache*) gfs)
	     gfs
	)
    )
  )
)


(defun my-generic-function-methods-safe (generic-function)
  "Returns the methods associated with a generic function."
  (let ((list (clos:slot-value generic-function 'ticlos:method-list)))
       ;;; Put in catch error to protect from During Transport
       ;;; of Self-Ref-Pointer error.
       (apply #'append (mapcar 'tv:get-method-from-spec list))
  )
)

(defun get-gf-methods (gf)
  (multiple-value-bind (methods found-p) (gethash gf *Who-Calls-Cache*)
    (if found-p
	methods
	(let ((methods (my-generic-function-methods-safe gf)))
	     (Setf (gethash gf *Who-Calls-Cache*) methods)
	     methods
	)
    )
  )
)

(Def-Who-Calls-Function who-calls-for-class (caller symbol function)
  ;; Also look for flavor methods
  (let ((cl (clos:find-class caller nil)))
    (letf ((#'tv:generic-function-methods-safe #'Get-Gf-Methods)
	   (#'tv:pcl-p #'(lambda () nil))
	   (#'tv:ticlos-p #'(lambda () t))
	   (#'tv:class-p-safe #'(lambda (x) (typep x 'clos:class)))
	   (#'tv:method-parameter-specializers-safe
	    #'(lambda (method)
		(mapcar #'tv:coerce-to-class
		  (ticlos:method-parameter-specializers method)
	        )
	      )
	   )
	  )
	  (When (and cl
		     (not (typep cl 'ticlos:flavor-class))
		     (not (and (symbolp caller) (tv:class-p-safe caller))))
	    (find-callers-of-symbols-slots caller cl symbol function)
	    (loop for class in (tv:class-precedence-list-safe cl)
		  when (and (equal (symbol-package (clos:class-name class))
				   (symbol-package (clos:class-name cl)))
			    (not (gethash class *Who-Calls-Cache*)))
		  Do (loop for gf in (get-class-direct-gfs class)
			   do (loop for meth in (get-gf-methods gf)
				    do (find-callers-of-symbols-aux1
					 (function-name
					   (tv:method-function-safe meth))
					 (tv:method-function-safe meth)
					 symbol function t)))
		  (Setf (gethash class *Who-Calls-Cache*) :processed))))))

(Def-Who-Calls-Function who-calls-for-initialization (caller symbol function)
  ;; Also look for initializations
  (when (get caller 'initialization-list)
    ;; It is an initialization list.
    (dolist (init-list-entry (symbol-value caller))
      (find-callers-of-symbols-aux-list
	caller (init-form init-list-entry) symbol
	function))))

(Def-Who-Calls-Function who-calls-for-generic-function (caller symbol function)
  (when (and (fboundp caller)
	     (ticlos:generic-function-p (symbol-function caller)))
    (loop for method in (tv:generic-function-methods-safe
			  (tv:function-generic-function-safe
			    (symbol-function caller)))
	  do (find-callers-of-symbols-aux1
	       (function-name (tv:method-function-safe method))
	       (tv:method-function-safe method) symbol function t))))

(defun sys:find-callers-of-symbols-aux (caller symbol function)
  ;; Ignore all symbols which are forwarded to others, to avoid duplication.
  (when (and (/= (%p-data-type-offset caller 2) dtp-one-q-forward)
	     (fboundp caller))
    (find-callers-of-symbols-aux1
      caller (symbol-function caller) symbol function))
  (when (/= (%p-data-type-offset caller 3) dtp-one-q-forward)
    ;; Also look for properties
    (loop for fun in *All-Who-Calls-Sub-Functions* do
	  (funcall fun caller symbol function))))

