;;; -*- Mode:Common-Lisp; Package:W; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;;  Richard Acuff, Stanford KSL, Dec '87

;;; This file, and those it depends on, sets up a windowing environment
;;; on the Explorer that is more a "west coast" or "desk top" style than
;;; what is normally used on the Explorer (the "east coast",
;;; "application stack", or "ring of Ann Arbor Ambassadors" approach).

#|
(defun set-up-windows ()
  "Function to put windows where I like them, sort of."

  ;; Fast ways to Bury, Shrink, or Zoom a window.
  (add-soft-key #\keypad-equal 'soft-key-bury nil)
  (add-soft-key #\keypad-0 'soft-key-shrink nil)
  (add-soft-key #\keypad-7 'soft-key-zoom nil)

  ;; Setup background
  (create-background)
  (set-background (read-bit-array-file "IMAGES:IMAGES;DOUBLE-GRIFFON#>"))
  ;;Make sure it's the right color on the mX
  (when (sys:mx-p)
    (when (send *background* :reverse-video-p)
      (kbd-complement-with-background 1)))
  (clean-up-windows 1)

  ;; Start a round clock
  (push (analog-clock nil `(,(- (sheet-width main-screen) 71)
			    0
			    ,(sheet-width main-screen)
			    71)
		      )
	*windows-to-leave-exposed*)
  ;; Don't use the first Lisp or Zmacs
  (setf *windows-not-to-pick*
	(list initial-lisp-listener (find-window "Zmacs Frame 1")))

  ;; Main working windows
  ;; Little Listener in the middle of the screen
  (setup-application 'lisp-listener "Listener" #\keypad-period
		     (if (sys:mx-p)
			 `(35 116 700 520)
			 `(160 225 825 630)
			 )
		     )
  ;; Editor on the left-hand side of the screen
  (setup-application 'zwei:zmacs-frame "Editor" #\keypad-enter
		     `(0 71 ,(if (sys:mx-p) 560 655) 0))
  (if (sys:mx-p)
      (setup-application 'zwei:zmacs-frame "Mail" #\keypad-plus `(303 71 0 0))
      (setup-application 'zwei:zmacs-frame "Mail" #\keypad-tab `(303 71 0 0))
      )
  (setup-application 'telnet:single-window-VT100 "VT100"
		     #\keypad-minus `(280 50 0 631))
  (setup-application 'tv:peek-frame "Peek" #\keypad-space `(50 71 770 0))
  (setup-application 'tv:general-inspector "Inspector" #\keypad-1)
  )

(set-up-windows)

|#

;;;----------------------------------------------------------------------
;;; Support modules

;;; For the right button menu, Snap, background window, transparent
;;; window, clock, rubber banding, and drag moving
(require 'window-system-additions)
(require 'backgrounds)				;Really in W-S-A
(require 'analog-clock)				;Really in W-S-A
;;; For attaching function (and applications) to keys
(require 'soft-keys)
;;; Vt100 emulater that doesn't take over the world
(require 'single-window-vt100)
;;; Shrinkable windows
(require 'window-icons)
;;; Zooming, Drag-shaping, and label accelerators
(require 'window-accelerators)
;(sys:load-if "X1:ACUFF.HACKS;WINDOW-ACCELERATORS")

;;;----------------------------------------------------------------------
;;;  Control variable settings

(setf *mouse-warp-gratuitously?* nil)
(setf tv::*inspector-configuration* :one-pane)
(setf tv::*flavor-inspector-configuration* :one-pane)
(setf beep :beep)
(setf w:*dragging-enabled?* t)
(setf w:*use-window-manager-system-menu* t)
(setf w:*use-rubber-bands?* t)
(setf w:scroll-bar-max-exit-speed 1)
(setq *window-manager-system-menu-items*
      '(("Move" :window-op move-window-op
	 :documentation "Move this window")
	("Reshape" :buttons ((nil :window-op reshape-window-op)
			     (nil :window-op move-edge-window-op)
			     (nil :window-op move-multi-edge-window-op))
	 :documentation
	 "L: New edges from mouse, M: Move one edge; R: Move multiple edges")

	("Create" :window-op create-window-op
	 :documentation "Create a new window")
	("Snap" :window-op snap-window-op
	 :documentation "Take a snapshop copy of a region of the screen")
	("Kill" :window-op kill-window-op
	 :documentation "Kill this window")
	("Bury" :window-op bury-window-op
	 :documentation "Bury this window")
	("Shrink" :window-op shrink-window-op
	 :documentation "Shrink this window to an Icon")
	("Zoom" :window-op zoom-window-op
	 :documentation "Zoom this window to it's other size")	
	("Sys Menu" :funcall mouse-call-standard-system-menu
	 :documentation "Pop up the Explorer System Menu")))

(clear-resource 'window-manager-system-menu)

;;;----------------------------------------------------------------------
;;; Make many windows smaller by default

(defresource tv:background-lisp-interactors ()
  :constructor
  ;;RDA: This mess is to get a good size, even on weird screens
  (let* ((sh (sheet-height default-screen))
	 (h (truncate sh 1.5))
	 (sw (tv:sheet-width tv:default-screen))
	 (w (min (if (sys:mx-p)
		     560
		     653
		     )
		 (- sh 10)))
	 x
	 y
	 )
    (setf h (if (< h 340) sh h))
    (setf w (if (< w 500) sw w))
    (setf x (max 100 (- (tv:sheet-width tv:default-screen) w)))
    (setf y (max 71 (- (tv:sheet-height tv:default-screen) h)))
    (make-window 'tv:background-lisp-interactor
		 :process current-process	       ;will be set later
		 :superior default-screen	       ;always on this screen
		 ;;RDA: Tall but not too tall.  More complex?
		 :height h
		 :width w
		 :x x
		 :y y
		 )
    )
  )
(clear-resource 'tv:background-lisp-interactors)

(setf eh:*debugger-frame-default-edges*
      `(20 80
	   ,(- (tv:sheet-width tv:default-screen) 20)
	   ,(- (tv:sheet-height tv:default-screen) 10)))

(clear-resource 'eh:DEBUGGER-FRAME)

;;;RDA: Remove this so it doesn't clobber the general inspector.  It's now
;;;done on a :INIT method of debugger frames.
#||
eh:
(DEFWINDOW-RESOURCE DEBUGGER-FRAME
                    NIL
                    :MAKE-WINDOW
                    (DEBUGGER-FRAME
		      :edges `(20 80
			       ,(- (tv:sheet-width tv:default-screen) 20)
			       ,(- (tv:sheet-height tv:default-screen) 10)))
                    :REUSABLE-WHEN
                    :DEACTIVATED)
||# 

(DEFWINDOW-RESOURCE POP-UP-FINGER-WINDOW ()
  :MAKE-WINDOW (TRUNCATING-POP-UP-TEXT-WINDOW-WITH-RESET
		 :edges `(20 72
			  ,(min 900 (- (tv:sheet-width tv:default-screen) 20))
			  ,(min 750 (- (tv:sheet-height tv:default-screen) 10)))
                 :RIGHT-SHADOW-WIDTH 0
                 :BOTTOM-SHADOW-WIDTH 0)
  :REUSABLE-WHEN :DEACTIVATED
  :INITIAL-COPIES 0)

;;;----------------------------------------------------------------------
;;;  Setup support code

(defvar *windows-not-to-pick* nil)
;; These default edges leave some room at the top of the screen.
(defvar *default-edges* `(0 71 ,(sheet-width default-screen)
			    ,(sheet-height default-screen)))
(advise set-number-of-who-line-documentation-lines
	:after
	"Reset Default Edges"
	nil
  (setf *default-edges* `(0 71 1024 ,(sheet-height default-screen)))
  )

(defun setup-application (flavor name soft-key
			  &optional (edges *default-edges*))
  "Find or create a window of FLAVOR with EDGES.  If it's icon is a text
   icon, set it's text to NAME.  Place the icon at ICON-X, ICON-Y.  Make
   SOFT-KEY be a soft select key for the new window.  A right or bottom
   edge that is zero or negative means use that distance from the edge
   of the superior."
  (setf edges (canonicalize-edges edges))
  (let* ((window (get-window flavor edges))
	 (icon (icon-for-window window)))
    (push window *windows-not-to-pick*)
    (add-soft-select-key soft-key window)
    (when (typep icon 'text-icon)
      (send icon :set-text name))
    (position-window-next-to-window
      icon (if (eq icon (first *windows-to-leave-exposed*))
		   (second *windows-to-leave-exposed*)
		   (first *windows-to-leave-exposed*))
      :left)
    (shrink-window window)
    window
    )
  )

(defun position-window-next-to-window (new old position)
  "Put the window NEW next to the window OLD according to POSITION as
   used in position-window-next-to-rectangle."
  (when old
    (apply #'position-window-next-to-rectangle new position
	   (multiple-value-list (send old :edges)))
    )
  )

(defun canonicalize-edges (edges &optional (sup default-screen))
  "Check that all the edges in EDGES are within SUP, trimming them back
   if necessary.  If any edge is non-positive, set it to be SUP's edge minus
   the old value.  Returns the new edges."
  (let ((left (first edges))
	(top (second edges))
	(right (third edges))
	(bottom (fourth edges))
	(width (sheet-width sup))
	(height (sheet-height sup))
	)
    ;;First take care of negative values
    (when (< left 0)
      (setf left (+ left width))		       ;From right edge
      )
    (when (< top 0)
      (setf top (+ top height))			       ;From bottom edge
      )
    ;;0 means 'on the edge' for right and bottom
    (when (<= right 0)
      (setf right (+ right width))
      )
    (when (<= bottom 0)
      (setf bottom (+ bottom height))
      )
    ;;Now trim
    (multiple-value-list (trim-edges left top right bottom width height))
    )
  )

(defun trim-edges (left top right bottom sup-width sup-height)
  "Make sure the edges LEFT, RIGHT, TOP, and BOTTOM fit within SUP-WIDTH and
   SUP-HEIGHT.  If LEFT or TOP won't fit, set them to 0.  If RIGHT or BOTTOM
   won't fit set them to SUP-WIDTH or SUP-HEIGHT respectively.  Returns the
   (possibly modified) edges."
  (declare (values left right top bottom))
  (unless (<= 0 left sup-width)
    (setf left 0)
    )
  (unless (<= 0 top sup-height)
    (setf top 0)
    )
  (unless (<= left right sup-width)
    (setf right sup-width)
    )
  (unless (<= top bottom sup-height)
    (setf bottom sup-height)
    )
  (values left top right bottom)
  )

;;;Edited by Acuff                 4 Dec 87  17:59
(defun soft-key-bury (key window)
  "Bury window"
  (declare (ignore key))
  (when window (send (send window :alias-for-selected-windows) :bury))
  )

;;;Edited by Acuff                 30 Nov 87  19:12
;;;Edited by Acuff                 4 Dec 87  17:59
(defun soft-key-shrink (key window)
  "Call SHRINK-WINDOW on WINDOW."
  (declare (ignore key))
  (when window (shrink-window (send window :alias-for-selected-windows)))
  )

;;;Edited by Acuff                 30 Nov 87  19:12
;;;Edited by Acuff                 4 Dec 87  17:59
(defun soft-key-zoom (key window)
  "Call SHRINK-WINDOW on WINDOW."
  (declare (ignore key))
  (when window (send (send window :alias-for-selected-windows)
		     :send-if-handles :zoom-size))
  )

;;;Edited by Acuff                 4 Dec 87  17:59
(defun get-window (flavor &optional (edges *default-edges*))
  (let ((w (or (find-or-make-window-of-flavor flavor)
	       (make-instance flavor :edges edges))))
    (push w *windows-not-to-pick*)
    (apply w :set-edges edges)
    w)
  )

;;;Edited by Acuff                 4 Dec 87  17:59
(defun find-or-make-window-of-flavor (flavor)
  (dotimes (i (length previously-selected-windows))
    (let ((w (aref previously-selected-windows i)))
      (when w
	(when (and (typep w flavor)
		   (send w :name-for-selection)
		   (not (member w *windows-not-to-pick*)))
	  (return w))
	(let ((wss (send w :selection-substitute)))
	  (when (and wss (typep wss flavor)
		     (send wss :name-for-selection)
		     (not (member wss *windows-not-to-pick*)))
	    (return wss)))))))

;;;----------------------------------------------------------------------

(provide 'west-coast-windows)
