;;; -*- Mode:Common-Lisp; Package:Sys; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	Rich Acuff
;;; at the Stanford University Knowledge Systems Lab in May '86, Dec '87.
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15

;;;  This file contains sundry functions of utility that, while useful,
;;;  aren't significant enough to be tools or systems in their own
;;;  right.  If you add to this file, be sure to update the
;;;  documentation in DEFSYSTEMS.LISP.

;;;----------------------------------------------------------------------
;;; Functions for mapping between key objects and data objects.  I'd
;;; like to extend this to use a-lists, hash tables, etc., as
;;; appropriate.

(defun make-map ()
  "Make a data structure that can be used to map from key objects to
   associated objects."
  (cons nil nil)
  )

(defun map-lookup (map key)
  "Find the object associated with KEY in MAP.  Returns <object> T on
   success, and NIL NIL on failure."
  (let ((cell (assoc key map)))
    (if cell
	(values (rest cell) t)
	(values nil nil)
	)
    )
  )

(defun map-set (map key object)
  "Associate OBJECT with KEY in MAP so that calling MAP-LOOKUP on MAP
   and KEY will find OBJECT.  This overwrites any previous mapping."
  (let ((cell (assoc key map)))
    (if cell
	;; Then it's already there, so change it
	(setf (rest cell) object)
	;; Not there before, so add it
	(setf (rest map) (cons `(,key . ,object) (rest map)))
	)
    )
  )

(defun map-remove (map key)
  "Remove any association between KEY and any object in MAP, returning
   the object KEY used to map to, and T, or NIL, NIL if there was none."
  (let ((cell (assoc key map)))
    (when cell
      ;;; fixed by JPR.
      (setq cell (delete cell map))
      )
    (if cell
	(values (rest cell) t)
	(values nil nil)
	)
    )
  )

(defun map-map (map fn &rest args)
  "Do (APPLY FN KEY VALUE ARGS) for all KEY, VALUE pairs in MAP."
  (loop for entry in map
	as (key . obj) = entry
	when entry
	doing (apply fn key obj args)
	)
  )

(import '(make-map map-lookup map-set map-remove map-map) 'ticl)
(export '(make-map map-lookup map-set map-remove map-map) 'ticl)

;;;----------------------------------------------------------------------
;;; Disk updating help

;;;TI Code: Original from KSL-EXP-7: DISK-IO; DISK-PARTITION.LISP#88
(DEFUN COPY-DISK-PARTITION-noconf (FROM-UNIT FROM-PART TO-UNIT TO-PART &OPTIONAL (N-blocks-AT-A-TIME 85.) (DELAY NIL)
			    (STARTING-HUNDRED 0.) (WHOLE-THING-P NIL) &AUX FROM-PART-BASE FROM-PART-SIZE TO-PART-BASE
			    TO-PART-SIZE RQB PART-COMMENT to-partition-name-string  from-partition-name-string)	;03.23.87 DAB
  "Copy partition FROM-PART on FROM-UNIT to partition TO-PART on TO-UNIT.
FROM-PART and TO-PART can be partition names or partition-name-strings, such as \"PART.Explorer\", where \"Explorer\" is
the user/cpu type.
While names of other machines can be specified as units, this is not very fast 
for copying between machines.  Use SI:RECEIVE-BAND or SI:TRANSMIT-BAND for that."
  (SETQ FROM-UNIT (DECODE-UNIT-ARGUMENT FROM-UNIT (FORMAT () "reading ~A partition" FROM-PART))
	TO-UNIT (DECODE-UNIT-ARGUMENT TO-UNIT (FORMAT () "writing ~A partition" TO-PART) () T))
  (UNWIND-PROTECT (PROGN
		    (SETQ RQB (GET-DISK-RQB N-blocks-AT-A-TIME))
		    (MULTIPLE-VALUE-SETQ
		      (FROM-PART-BASE FROM-PART-SIZE nil from-part nil from-partition-name-string)	;03.23.87 DAB
		      (FIND-DISK-PARTITION-FOR-READ FROM-PART () FROM-UNIT))	;CONFIRM-read is T, prompt for duplicates.
		    (MULTIPLE-VALUE-SETQ
		      (TO-PART-BASE TO-PART-SIZE nil to-part nil to-partition-name-string)	;03.23.87 DAB
		      ;;RDA: Don't confirm
		      (FIND-DISK-PARTITION-FOR-WRITE TO-PART () TO-UNIT nil "LOD" nil))	;CONFIRM-write is NIL, prompt for duplicates.
		    (WHEN TO-PART-BASE
		      (SETQ PART-COMMENT (PARTITION-COMMENT  from-partition-name-string FROM-UNIT))	;03.23.87 DAB
		      (FORMAT T "~&Copying ~S" PART-COMMENT)
		      (AND
			(OR (NUMBERP FROM-PART)
			    (STRING-EQUAL FROM-PART "LOD" :START1 0. :END1 3. :START2 0. :END2 3.))
			(NOT WHOLE-THING-P)
			(NOT
			  (AND (CLOSUREP FROM-UNIT)
			       (EQ (CLOSURE-FUNCTION FROM-UNIT) 'FS::BAND-MAGTAPE-HANDLER)))
			(LET (RQB size)
			  (UNWIND-PROTECT
			      (PROGN
				(SETQ rqb (get-disk-rqb disk-blocks-per-page))
				(SETQ size (lod-partition-info rqb from-unit from-part-base))
				(COND
				  ((AND (> SIZE 8.) (<= SIZE FROM-PART-SIZE))
				   (SETQ FROM-PART-SIZE SIZE)
				   (FORMAT T
					   "... using measured size of ~D. blocks."
					   SIZE))))
			    (RETURN-DISK-RQB RQB))))
		      (WHEN (> FROM-PART-SIZE TO-PART-SIZE)
			(FERROR ()
				"Source partition length, ~D. blocks, is larger than destination length, ~D."
				FROM-PART-SIZE TO-PART-SIZE))
		      (FORMAT T "~%")
		      (UPDATE-PARTITION-COMMENT  to-partition-name-string "Incomplete Copy" TO-UNIT)	;03.23.87 DAB
		      (COND
			((AND (CLOSUREP TO-UNIT)	;magtape needs to know this stuff before
			      (FUNCALL TO-UNIT :HANDLES-LABEL))	;writing file.
			 (FUNCALL TO-UNIT :PUT PART-COMMENT :COMMENT)
			 (FUNCALL TO-UNIT :PUT FROM-PART-SIZE :SIZE)))
		      (DO ((FROM-ADR (+ FROM-PART-BASE (* 100. STARTING-HUNDRED))
				     (+ FROM-ADR AMT))
			   (TO-ADR (+ TO-PART-BASE (* 100. STARTING-HUNDRED)) (+ TO-ADR AMT))
			   (FROM-HIGH (+ FROM-PART-BASE FROM-PART-SIZE))
			   (TO-HIGH (+ TO-PART-BASE TO-PART-SIZE))
			   (N-BLOCKS (* 100. STARTING-HUNDRED) (+ N-BLOCKS AMT))
			   (N-HUNDRED STARTING-HUNDRED)
			   (AMT))
			  ((OR (>= FROM-ADR FROM-HIGH) (>= TO-ADR TO-HIGH)))
			(SETQ AMT
			      (MIN (- FROM-HIGH FROM-ADR) (- TO-HIGH TO-ADR) N-blocks-AT-A-TIME))
			(COND
			  ((NOT (= AMT N-blocks-AT-A-TIME)) (RETURN-DISK-RQB RQB)
							    (SETQ RQB (GET-DISK-RQB AMT))))
			(DISK-READ RQB FROM-UNIT FROM-ADR)
			(DISK-WRITE RQB TO-UNIT TO-ADR)
			(COND
			  ((NOT (= (FLOOR (+ N-BLOCKS AMT) 100.) N-HUNDRED))
			   (SETQ N-HUNDRED (1+ N-HUNDRED)) (FORMAT T "~D " N-HUNDRED)))
			(IF DELAY
			    (PROCESS-SLEEP DELAY)
			    (PROCESS-ALLOW-SCHEDULE)))	;kludge
		      (UPDATE-PARTITION-COMMENT  to-partition-name-string PART-COMMENT TO-UNIT)))	;03.23.87 DAB
    ;;Unwind-protect forms
    (RETURN-DISK-RQB RQB))
  (DISPOSE-OF-UNIT FROM-UNIT)
  (UNLESS (NUMBERP to-unit) (PROCESS-ALLOW-SCHEDULE))	;don't release TO-UNIT until we're done
  (DISPOSE-OF-UNIT TO-UNIT))

;;;RDA: Original cribbed from SYS:CHAOSNET;BAND.LISP#36's RECEIVE-BAND
(defun receive-band-noconf (FROM-MACHINE FROM-PART TO-UNIT TO-PART
				  &OPTIONAL (SUBSET-START 0) SUBSET-N-BLOCKS
				  &AUX CONN PKT STR REMOTE-PART-SIZE (WINDOW (if (si:mx-p)
										 *mx-receive-band-window-size*
										 *exp-receive-band-window-size* ))
				  FROM-UNIT PART-COMMENT PART-BASE PART-SIZE
				  user-type partition-name-string);03.13.87 DAB
  
  "Read the FROM-PART partition  from FROM-MACHINE into local partition TO-PART.
   If SUBSET-START or SUBSET-N-BLOCKS is specified, they say which part of the partition 
   to transfer.  SUBSET-START and SUBSET-N-BLOCKS are in hundreds of blocks.  If a 
   transfer dies, use the last number it printed  as SUBSET-START, to
   resume where it left off.  FROM-MACHINE is a string consisting of the name of the
   remote machine or the remote machine, colon, remote unit number.  If no unit number
   is specified the default unit of the REMOTE machine is used."
  (SETQ SUBSET-START (* SUBSET-START 100))
  (WHEN SUBSET-N-BLOCKS
	(SETQ SUBSET-N-BLOCKS (* SUBSET-N-BLOCKS 100)))
  (UNWIND-PROTECT
      (PROGN
	(SETQ TO-UNIT (DECODE-UNIT-ARGUMENT TO-UNIT "receive-band"))
	(when (closurep to-unit)   ;03-02-88 DAB Check to make sure it is a local unit.
	  (ferror () "TO-UNIT must be a local unit number."))
	(MULTIPLE-VALUE-SETQ (PART-BASE PART-SIZE NIL to-part nil partition-name-string)  ;03.19.87 DAB      
			     (FIND-DISK-PARTITION-FOR-WRITE TO-PART () TO-UNIT
							    nil "LOD" nil))
	(MULTIPLE-VALUE-SETQ (nil nil nil nil nil from-part) ;12-13-88 DAB
			     (FIND-DISK-PARTITION-FOR-read from-PART () from-machine))
	;; Parse machine and unit number.
	
	(LET ((TEM (POSITION #\: (THE STRING (STRING FROM-MACHINE)) :TEST #'CHAR-EQUAL)))
	  (UNLESS (NULL TEM)
		  (SETQ FROM-UNIT (READ-FROM-STRING FROM-MACHINE () () :START (1+ TEM)))
		  (SETQ FROM-MACHINE (SUBSEQ (STRING FROM-MACHINE) 0 TEM) )))
	
	;; If the partition was found and the user ok'd writing to it . . .
        (setf (values from-part user-type) ;03.13.87 DAB
	      (parse-partition-name from-part :keyword))
	(WHEN PART-BASE	      
	      ;; Connect to remote machine.
	      (SETQ CONN
		    (CHAOS:CONNECT FROM-MACHINE
				   (FORMAT () "BAND-TRANSFER READ ~a ~d ~d ~a ~a ~a" FROM-PART
					   FROM-UNIT 
					   (LIST SUBSET-START
						 (IF SUBSET-N-BLOCKS
						     (MIN SUBSET-N-BLOCKS PART-SIZE)
						     (- PART-SIZE SUBSET-START)))
					   nil nil user-type)   ;03.12.87 DAB
				   WINDOW))
	      
	      ;; Receive packet containing size and comment.
	      
	      (SETQ PKT (CHAOS:GET-NEXT-PKT CONN))
	      
	      ;; Get the data in the form of a string, fetch the partition size, then
	      ;; return the packet.
	      
	      (SETQ STR (CHAOS:PKT-STRING PKT))
	      (SETQ REMOTE-PART-SIZE (READ-FROM-STRING STR T))
	      (CHAOS:RETURN-PKT PKT)
	      
	      ;; Verify that the local partition is large enough.
	      
	      (WHEN (> REMOTE-PART-SIZE PART-SIZE)
		    (RETURN-FROM RECEIVE-BAND-noconf
		      (FORMAT T "~% Does not fit in local partition, ~d > ~d"
			      REMOTE-PART-SIZE PART-SIZE)))
	      
	      ;; Revise partition size to be the actual size, then fetch
	      ;; the partition comment.
	      
	      (SETQ PART-SIZE REMOTE-PART-SIZE)
	      (SETQ PART-COMMENT
		    (READ-FROM-STRING STR () () :START
				      (1+
					(POSITION #\SPACE (THE STRING (STRING STR))
						  :TEST #'CHAR-EQUAL))))
	      
	      ;; Everything looks good.  Let's transfer the partition.
	      
	      (FORMAT T
		      "~& Receiving partition ~s on unit ~d from ~a ~
                   into partition ~s unit ~d:~%  ~d blocks, ~s~%"
		      FROM-PART 
		      (or FROM-UNIT "*DEFAULT-DISK-UNIT*")  ; DAB 05-03-89
		      FROM-MACHINE partition-name-string TO-UNIT PART-SIZE ;03.19.87 DAB
		      PART-COMMENT)
	      
	      ;; Put "Incomplete Copy" into the comment field so that you can
	      ;; tell that the data is incomplete.
	      
	      (UPDATE-PARTITION-COMMENT  partition-name-string "Incomplete Copy" TO-UNIT ) ;03.19.87 DAB 
	      
	      ;; Transfer the data from Chaos to the partition.
	      
	      (PARTITION-NET-TRANSFER TO-UNIT PART-BASE PART-SIZE CONN T () T
				      SUBSET-START SUBSET-N-BLOCKS)
	      
	      ;; Transfer complete, close net connection.
	      
	      (CHAOS::CLOSE-CONN CONN "Done")
	      
	      ;; Set partition comment.
	      
	      (UNLESS SUBSET-N-BLOCKS
		      (UPDATE-PARTITION-COMMENT partition-name-string  PART-COMMENT TO-UNIT ))))
    
    ;; Unwind-protect-forms
    
    (DISPOSE-OF-UNIT TO-UNIT)
    (WHEN CONN
	  (CHAOS:REMOVE-CONN CONN)))
  (RETURN-FROM RECEIVE-BAND-noconf
    (IF PART-BASE
	(PROGN
	  (FORMAT T "~%") T)))
  ())

(defparameter *numbered-partitions* '("MCR" "LOD" "CFG"))

(defun maybe-add-number (raw-part to-unit from-unit)
  (if (member raw-part *numbered-partitions* :test #'equalp)
      (values (format nil "~A~D" raw-part to-unit)
	      (format nil "~A~D" raw-part from-unit))
      (values raw-part raw-part)
      )
  )

(defparameter *partition-host-defaults*
	      (let ((map (make-map)))
		(map-set map :explorer-i "X15")
		(map-set map :explorer-ii "X18")
		map))

(defun default-from-host ()
  (or (map-lookup *partition-host-defaults*
		  (sys:processor-type sys:microcode-type-code))
      (error "Unrecognized processor type")))

(defparameter *default-partitions*
	      (let ((map (make-map)))
		(map-set map :explorer-i
			 '("EXPT" "GDOS" "DIAG" "BOOT" "MCR" "LOD"))
		(map-set map :explorer-ii
			 '("EXP2" "GDOS" "DIAG" "BOOT" "MCR" "LOD"))
		map))

(defun default-partitions ()
  (or (map-lookup *default-partitions*
		  (if (explorer-ii?) :explorer-ii :explorer-i))
      (error "Unrecognized processor type")))

(defparameter *max-n-pages-at-a-time* 238)

(defun copy-partitions (&optional (to-unit *default-disk-unit*)
			(from-host (default-from-host))
			(from-unit 0)
			(parts (default-partitions)))
  (setf from-host (net:parse-host from-host))
  (loop for raw-part in parts
	with from-part and to-part
	do
	(setf (values to-part from-part)
	      (maybe-add-number raw-part to-unit from-unit))
	(if (eq from-host local-host)
	    (copy-disk-partition-noconf from-unit from-part to-unit to-part *max-n-pages-at-a-time*)
	    (receive-band-noconf (format nil "~a:~d" from-host from-unit) from-part to-unit to-part)
	    )
	)
  )

;;; A simple function for running things in the background

(defun bg (form &optional (priority -5))
  "Run FORM in a background process at priority PRIORITY."
  (apply #'process-run-function `(:name ,(format nil "Background running ~S" form)
				      :priority ,priority)
	 `(eval ,form)
	 )
  )

;;;----------------------------------------------------------------------
;;;  Function to reboot quickly and safely

(defun reboot (&optional (partition *Loaded-Band*) (unit (Hunt-For-Band partition)))
;;; Changed by JPR on 11/16/89 14:27:25 to look on different devices.
  "Reload the current world after shutting down the system.
If Partition is specified then that partition is restored.
Unit need only be specified if there are multiple units with the same parition
name.
"
  (system-shutdown :type :user
		   :reason-string (format nil "Reboot by ~A" user-id)
		   :return t
		   )
  (if (not (equal (disk-restore-decode partition) *loaded-band*))
      (format t "~&Restoring partition ~A on unit ~A." partition unit))
  (Really-Disk-Restore partition unit)
  )

;;; Modified from TI code SYS:MEMORY-MANAGEMENT;DISK-SAVE-RESTORE.LISP#12's DISK-RESTORE
(defun hunt-for-band (partition &optional (error-p t)&aux rqb foundp)
"Hunts for a partition called Partition, returning the unit number that the
partition is on if it is there.  If it is not then an error is signalled unless
error-p is nil.
"
;;; By JPR.
  (loop for unit in (cons *default-disk-unit* (all-disk-units)) do
	;; look on *Default-Disk-Unit* first, then look elsewhere.
	(multiple-value-bind (name)
	    ;; Decode partition argument
	    (disk-restore-decode partition)
	  ;; Verify valid partition & get its desired Ucode.
	  (unwind-protect
	      (setq rqb (read-disk-label unit)
		    name (if partition name (current-band unit))
		    foundp (find-disk-partition name rqb unit t () t)
	      )
	    (return-disk-rqb rqb))
	  (if foundp (return unit) nil)
	)
	finally (if error-p
		    (ferror nil "There is no partition called ~S" partition)
		    nil
		)
  )
)

;;;TI Code: cribbed from SYS:MEMORY-MANAGEMENT;DISK-SAVE-RESTORE.LISP#12's DISK-RESTORE
(Defun really-Disk-Restore (&optional (partition *Loaded-Band*) (unit *Default-Disk-Unit*))
  "Reboot partition PARTITION on unit UNIT as a saved Lisp world.
  PARTITION can be either a string naming a partition, or a number
which signifies a partition whose name starts with LOD.  The default is
to reboot the current Lisp world.  NIL means boot the default load partition for UNIT.
  Note that this does not change the running microcode.  You cannot 
successfully DISK-RESTORE a world that will not work with the microcode that 
is currently running."
  (let (rqb block name name-hi-16-bits name-lo-16-bits comment desired-ucode)

    ;; Decode partition argument
    (multiple-value-setq (name name-hi-16-bits name-lo-16-bits)
			 (disk-restore-decode partition))

    ;; Verify valid partition & get its desired Ucode.
    (unwind-protect
       (setq rqb (read-disk-label unit)
	     name (if partition
		      name
		      (current-band unit))    
	     block (find-disk-partition-for-read name rqb unit t)
	     comment (partition-comment name unit)
	     desired-ucode (get-ucode-version-of-band name unit))
      (return-disk-rqb rqb))

    ;; Verify Ucode level.
    (and (/= desired-ucode %Microcode-Version-Number)
	 (not (zerop desired-ucode))		;; Not stored yet
	 (format *Query-IO*
		 "~&That band prefers microcode ~D but the running microcode is ~D.~%"
		 desired-ucode %Microcode-Version-Number))   

    ;; Verify with user.
    (cond (T					;RDA:  DON'T ASK
	   (and (fboundp 'tv:close-all-servers)
		(tv:close-all-servers "Disk-Restoring"))

	   (%disk-restore name-hi-16-bits name-lo-16-bits (get-real-unit unit))))
    ))

;;;----------------------------------------------------------------------
;;; This software developed by:
;;;	Rich Acuff
;;; at the Stanford University Knowledge Systems Lab in Jun '90
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15
;;;	DARPA Grant N00039-86- C-0033

;;;----------------------------------------------------------------------
;;; Portions of this code indicated by the comment line:
;;;	;TI Code:
;;; are derived from code licensed from Texas Instruments Inc.
;;; KSL's changes are noted by comment lines beginning with:
;;;	;RDA:
;;;  The following restrictions apply to the TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

;;; This file contains enhancements to the TI function SYS:EDIT-DISK-LABEL.
;;; To wit:
;;;   - Change to c-O command to default starting address to end of previous
;;;     partition in all cases
;;;   - Change to bind special vars for the start, length, and end of the
;;;     current partition, previous partition, and next partition (if any) in
;;;     c-E command.  These are S-, S, S+, L-, L, L+, E-, E, and E+ ("S" for
;;;     start, "L" for length, "E" for end", "-" for previous, and "+" for
;;;     next).  For non-number edits the old value is placed in the input
;;;     history ring.
;;;   - Lots of utility functions and functional interfaces to commands.
;;;   - New commands to make installing disks easier

;;;----------------------------------------------------------------------

;;; This section makes c-O command default the starting address to end of
;;; previous partition in all cases

;;;TI Code: from KSL-EXP-7: DISK-IO; DISK-LABEL-EDITOR.LISP#49
(Defun LE-Com-Control-O ()
  (Setq LE-Something-Changed t)
  (Let ((ploc (LE-Current-Partition))
	(nparts
	  (1+ (Get-disk-Fixnum LE-Disk-Label
			       (+ %pt-base %PT-Number-of-Partitions))))
	(nwords (Get-disk-Fixnum LE-Disk-Label
				 (+ %pt-base %PT-Size-of-Partition-Entries)))
	(buf (RQB-Buffer LE-Disk-Label)))
    (When (> (+ (* nparts nwords) %PT-Partition-Descriptors)
	     (* (Get-disk-Fixnum LE-Disk-Label %DL-Partition-Table-Length)
		disk-block-word-size))
      (When (y-or-n-p "Partition table full. Expand it?")
	(Format user:*terminal-io* "~&Sorry, unable to expand partition table.")
	))
    ;; Check if partition table is now large enough.
    (When (<= (+ (* nparts nwords) %PT-Partition-Descriptors)
	      (* (Get-disk-Fixnum LE-Disk-Label %DL-Partition-Table-Length)
		 disk-block-word-size))
      (Put-disk-Fixnum LE-Disk-Label nparts
		       (+ %pt-base %PT-Number-of-Partitions))
      ;;used to be 400 for 1 block partition table
      (Let ((foo (Make-Array #o12000 ':Type 'Art-16b)))	
	;;with 3 blocks lengths 12000 seems to to work
        (Copy-Array-Portion buf (* ploc 2) (Array-total-size buf)
                            foo (* nwords 2) #o12000)
        (Copy-Array-Portion foo 0 #o12000
                            buf (* ploc 2) (Array-total-size buf))
        ;; Initialize new partition.
        (Put-disk-String LE-Disk-Label "????" (+ ploc %PD-Name) 4)
        (Put-disk-Fixnum LE-Disk-Label 0 (+ ploc %PD-Length))
        (Put-disk-Fixnum LE-Disk-Label (dpb %cpu-explorer %%cpu-type-code 0)
			 (+ ploc %PD-Attributes))
        (Put-disk-Fixnum LE-Disk-Label
			 ;;RDA: Always use end of previous partition as start
			 ;;of inserted partition: add "t ;".  This should not
			 ;;fail on the first partition since the first
			 ;;partitions should never be edited from Lisp
			 (If t ;(= LE-Item-Number (Length LE-Structure))
			     (+ (Get-disk-Fixnum LE-Disk-Label
						 (+ %PD-Length (- ploc nwords)))
				(Get-disk-Fixnum LE-Disk-Label
						 (+ %PD-Start (- ploc nwords))))
			     (Get-disk-Fixnum LE-Disk-Label
					      (+ ploc nwords %PD-Start)))
			 (+ ploc %PD-Start))
        (Return-Array foo))
      (LE-Display-Label LE-Disk-Label LE-Unit)))
  )

;;;----------------------------------------------------------------------

;;; Utility functions

(defun start-of-partition (&optional (offset 0))
  "Returns the starting address of a partition.  If OFFSET is 0, returns for
   the current partition; if OFFSET is -1 then the previous partition; if
   OFFSET is 1 then the next partition, etc.  If OFFSET causes reference to a
   non-existant partition then results are undefined.  If there is no current
   partition then results are undefined."
  (let ((ploc (LE-Current-Partition))
	(nwords (Get-disk-Fixnum LE-Disk-Label
				 (+ %pt-base %PT-Size-of-Partition-Entries))))
    (Get-disk-Fixnum LE-Disk-Label (+ %PD-Start ploc (* offset nwords)))))

(defun length-of-partition (&optional (offset 0))
  "Returns the length of a partition.  If OFFSET is 0, returns for the
   current partition; if OFFSET is -1 then the previous partition; if OFFSET
   is 1 then the next partition, etc.  If OFFSET causes reference to a
   non-existant partition then results are undefined.  If there is no current
   partition then results are undefined."
  (let ((ploc (LE-Current-Partition))
	(nwords (Get-disk-Fixnum LE-Disk-Label
				 (+ %pt-base %PT-Size-of-Partition-Entries))))
    (Get-disk-Fixnum LE-Disk-Label (+ %PD-Length ploc (* offset nwords)))))

(defun end-of-partition (&optional (offset 0))
  "Returns the end address of a partition.  If OFFSET is 0, returns for
   the current partition; if OFFSET is -1 then the previous partition; if
   OFFSET is 1 then the next partition, etc.  If OFFSET causes reference to a
   non-existant partition then results are undefined.  If there is no current
   partition then results are undefined."
  (+ (start-of-partition offset) (length-of-partition offset)))

;;; Arrange to bind *REMOTE-TYPE* around edits, and set it when necessary
(advise EDIT-DISK-LABEL :around "Bind *REMOTE-TYPE*" nil
  (let ((*remote-type* nil))			;set to type of remote CPU
    (declare (special *remote-type*))
    :do-it))

(defun pretend-to-be-explorer-ii ()
  (advise explorer-ii? :around :explorer-ii-pretense nil t))

(defun pretend-to-be-explorer-i ()
  (advise explorer-ii? :around :explorer-i-pretense nil nil))

(defun explorer-ii? ()
  (declare (special *remote-type*))
  (if (and (boundp '*LE-remote-edit*) *LE-remote-edit*)
      (if *remote-type*
	  (eq *remote-type* :explorer-ii)
	  (eq (setf *remote-type*
		    (if (y-or-n-p "Is the remote machine an Explorer II?")
			:explorer-ii
			:explorer-i))
	      :explorer-ii))
      (= (cpu-type) %cpu-ti-explorer-II)))

(defun unit-number (unit)
  "Return the unit number of the disk label being edited with UNIT."
  (if (numberp unit)
      unit
      (funcall unit :unit-number)))

(defun le-compute-starting-address (start len)
  (etypecase start
    (number start)
    (symbol
     (ecase start
       (:against-prev (end-of-partition -1))
       (:against-next (- (start-of-partition 1) len))))))

(defun le-down-line ()
  (le-com-control-n))

(defun le-up-line ()
  (le-com-control-p))

(defun add-to-input-history (string stream)
  (let ((rhb (symeval-in-instance stream 'tv:rubout-handler-buffer)))
    (zwei:push-on-history string (tv:rhb-input-ring rhb))))

;;;----------------------------------------------------------------------

;;; Higher level utility functions

;;;TI Code: Taken from LE-COM-CONTROL-O
(defun le-insert-new-partition ()
  (Let ((ploc (LE-Current-Partition))
	(nparts
	  (1+ (Get-disk-Fixnum LE-Disk-Label
			       (+ %pt-base %PT-Number-of-Partitions))))
	(nwords (Get-disk-Fixnum LE-Disk-Label
				 (+ %pt-base %PT-Size-of-Partition-Entries)))
	(buf (RQB-Buffer LE-Disk-Label)))
    (When (> (+ (* nparts nwords) %PT-Partition-Descriptors)
	     (* (Get-disk-Fixnum LE-Disk-Label %DL-Partition-Table-Length)
		disk-block-word-size))
      (When (y-or-n-p "Partition table full. Expand it?")
	(format user:*terminal-io*
		"~&Sorry, unable to expand partition table.")))
    ;; Check if partition table is now large enough.
    (When (<= (+ (* nparts nwords) %PT-Partition-Descriptors)
	      (* (Get-disk-Fixnum LE-Disk-Label %DL-Partition-Table-Length)
		 disk-block-word-size))
      (Setq LE-Something-Changed t)
      (Put-disk-Fixnum LE-Disk-Label nparts
		       (+ %pt-base %PT-Number-of-Partitions))
      ;;used to be 400 for 1 block partition table
      (Let ((foo (Make-Array #o12000 ':Type 'Art-16b)))	
	;;with 3 blocks lengths 12000 seems to to work
        (Copy-Array-Portion buf (* ploc 2) (Array-total-size buf)
                            foo (* nwords 2) #o12000)
        (Copy-Array-Portion foo 0 #o12000
                            buf (* ploc 2) (Array-total-size buf))
        ;; Initialize new partition.
        (Put-disk-String LE-Disk-Label "????" (+ ploc %PD-Name) 4)
        (Put-disk-Fixnum LE-Disk-Label 0 (+ ploc %PD-Length))
        (Put-disk-Fixnum LE-Disk-Label (dpb %cpu-explorer %%cpu-type-code 0)
			 (+ ploc %PD-Attributes))
        (Put-disk-Fixnum LE-Disk-Label
			 ;;RDA: Always use end of previous partition as start
			 ;;of inserted partition.  This should not
			 ;;fail on the first partition since the first
			 ;;partitions should never be edited from Lisp
			 (+ (Get-disk-Fixnum LE-Disk-Label
					     (+ %PD-Length (- ploc nwords)))
			    (Get-disk-Fixnum LE-Disk-Label
					     (+ %PD-Start (- ploc nwords))))
			 (+ ploc %PD-Start))
;	(update-le-structure LE-Disk-Label)
        (Return-Array foo)))))

(defun le-add-partition (&optional (name "????") (start :against-prev)
			 (length 0) (type %BT-LOAD-BAND)
			 (cpu-type %cpu-explorer) (attributes ()))
  "Inserts a new partition into the label editor with fields setup according
   to arguments.  For START arg a numerical address, :AGAINST-PREV (start of
   new = end of one before), or :AGAINST-NEXT (start of new = start of next -
   length of new) are acceptable."
  (le-insert-new-partition)
  (le-set-partition-name name)
  (le-set-partition-start (le-compute-starting-address start length))
  (le-set-partition-length length)
  (le-set-partition-type type)
  (le-set-partition-cpu-type cpu-type)
  (loop for attr in attributes
	do (le-set-partition-attribute attr)))

(defun le-set-partition-name (new-name)
  (let ((ploc (le-current-partition)))
    (when *le-prim-style-edit*
      (let ((att (get-disk-fixnum le-disk-label (+ ploc %pd-attributes))))
	(and (= (cpu-type) (ldb %%cpu-type-code att))
	     (or (= %bt-microload (ldb %%band-type-code att))
		 (= %bt-load-band (ldb %%band-type-code att)))))
      (setq *le-mcr-or-lod-changed* t))	
    (SETF new-name (PAD-NAME-FIELD new-name 4))
    (PUT-DISK-STRING LE-DISK-LABEL new-name (+ PLOC %PD-NAME) 4)))

(defun le-set-partition-start (new-start)
  (let ((ploc (le-current-partition)))
    (IF (RESERVED-BLOCKS-OVERWRITE-PROTECTION PLOC 'partition-start new-start)
	(PUT-DISK-FIXNUM LE-DISK-LABEL new-start (+ PLOC %PD-start)))))

(defun le-set-partition-length (new-length)
  (let ((ploc (le-current-partition)))
    (IF (RESERVED-BLOCKS-OVERWRITE-PROTECTION PLOC 'partition-size new-length)
	(PUT-DISK-FIXNUM LE-DISK-LABEL new-length (+ PLOC %PD-LENGTH)))))

(defun le-set-partition-type (new-type)
  (let* ((ploc (LE-Current-Partition))
	 (temp (dpb new-type
		    %%band-type-code
		    (get-disk-fixnum le-disk-label 
				     (+ ploc %pd-attributes)))))
    (Put-disk-Fixnum le-disk-label temp (+ ploc %PD-Attributes))))

(defun le-set-partition-cpu-type (new-cpu-type)
  (Let* ((ploc (LE-Current-Partition))
	 (temp (dpb new-cpu-type
		    %%CPU-type-code
		    (get-disk-fixnum le-disk-label 
				     (+ ploc %pd-attributes)))))
    (Put-disk-Fixnum le-disk-label temp (+ ploc %PD-Attributes))))

(defun le-set-partition-attribute (new-attribute)
  (Let* ((ploc (LE-Current-Partition))
	 (temp (get-disk-fixnum le-disk-label (+ ploc %PD-Attributes))))
    (Cond((and *le-prim-style-edit*
	       (= (cpu-type) (ldb %%cpu-type-code temp))
	       (or (= %bt-microload (ldb %%band-type-code temp))
		   (= %bt-load-band (ldb %%band-type-code temp)))
	       (= new-attribute 0))
	  (let* ((item (nth le-item-number le-structure))
		 (value (second item)))
	    (cond ((= %bt-microload (ldb %%band-type-code temp))
		   (setq *le-mcr-name* value *le-mcr-unit* le-unit)
		   (setq *le-mcr-or-lod-changed* t))
		  ((= %bt-load-band (ldb %%band-type-code temp))
		   (setq *le-lod-name* value *le-lod-unit* le-unit)
		   (setq *le-mcr-or-lod-changed* t)))))
	 ((zerop new-attribute)			    
	  (format *terminal-io*
		  "~&Can't select this partition as a System Load Default. (Press space to continue)")
	  (read-char))
	 (t (if (not (ldb-test  new-attribute temp))
		(setq temp (dpb 1 new-attribute temp))
		(setq temp (dpb 0 new-attribute temp)))
	    (Put-disk-Fixnum le-disk-label temp (+ ploc %PD-Attributes))))))

;;;----------------------------------------------------------------------

;;; Changes to c-E command to bind vars for 'S'tart, 'L'ength, and 'E'nd of
;;; current, previous '-', and next '+' partitions.

;;;TI Code: from KSL-EXP-7: DISK-IO; DISK-LABEL-EDITOR.LISP#49
(DEFUN LE-COM-CONTROL-E ()
  (SETQ LE-SOMETHING-CHANGED T)
  ;; something probably will...
  (CONDITION-CASE ()
      (IF (< LE-ITEM-NUMBER (LENGTH LE-STRUCTURE))
	  (LET ((ITEM (NTH LE-ITEM-NUMBER LE-STRUCTURE)))
	    (LET ((NAME (FIRST ITEM))
		  (VALUE (SECOND ITEM)))
	      
	      (LET ((*print-base* 10.))
		(FORMAT USER:*TERMINAL-IO* "Change ~a from ~s to: " NAME VALUE))
	      (SETQ VALUE
		    (IF (NUMBERP VALUE)
			;;RDA: Bind vars for easier editing.  This should not
			;;fail on the first partition since the first
			;;partitions should never be edited from Lisp
			(LET ((*read-base* 10.)
			      (user:s- (start-of-partition -1))
			      (user:s  (start-of-partition  0))
			      (user:s+ (start-of-partition  1))
			      (user:l- (length-of-partition -1))
			      (user:l  (length-of-partition  0))
			      (user:l+ (length-of-partition  1)) 
			      (user:e- (end-of-partition -1))
			      (user:e  (end-of-partition  0))
			      (user:e+ (end-of-partition  1))
			      )
			  (declare (special user:s- user:s user:s+
					    user:l- user:l user:l+
					    user:e- user:e user:e+))
			  (EVAL
			    (WITH-INPUT-EDITING
			      (USER:*TERMINAL-IO* '((:NO-INPUT-SAVE T)))
			      (READ USER:*TERMINAL-IO*))))
			(progn
			  ;;RDA: Make it possible to use c-C to get old value
			  (add-to-input-history
			    (string value) USER:*TERMINAL-IO*)
			  (zlc:READLINE USER:*TERMINAL-IO* ()
					'((:NO-INPUT-SAVE T))))))
		    ;; Avoid lossage in lowercase partition names.
		    (IF (MEMBER NAME
				'(PARTITION-NAME
				   CURRENT-BAND CURRENT-MICROLOAD
				   system-band system-microload) :TEST #'EQ)
			(SETQ VALUE
			      (IF *PARTITION-NAME-CASE-SENSITIVE*
				  (STRING VALUE)
				  (STRING-UPCASE VALUE))))
		    (CASE NAME
		      (VOLUME-NAME
		       (PUT-DISK-STRING LE-DISK-LABEL VALUE %DL-VOLUME-NAME 16.))
		      (DRIVE-NAME
		       (PUT-DISK-STRING LE-DISK-LABEL VALUE %DL-DEVICE-NAME 12.))
		      (COMMENT
		       (PUT-DISK-STRING LE-DISK-LABEL VALUE %DL-COMMENT 96.))
		      (N-BYTES-PER-BLOCK
		       (PUT-DISK-FIXNUM
			 LE-DISK-LABEL
			 (DPB VALUE %%BYTES-PER-BLOCK
			      (GET-DISK-FIXNUM LE-DISK-LABEL %BYTES-PER))
			 %BYTES-PER))
		      (N-BYTES-PER-SECTOR
		       (PUT-DISK-FIXNUM
			 LE-DISK-LABEL
			 (DPB VALUE %%BYTES-PER-SECTOR
			      (GET-DISK-FIXNUM LE-DISK-LABEL %BYTES-PER))
			 %BYTES-PER))
		      (N-SECTORS-PER-TRACK
		       (PUT-DISK-FIXNUM
			 LE-DISK-LABEL
			 (DPB VALUE %%SECTORS-PER-TRACK
			      (GET-DISK-FIXNUM LE-DISK-LABEL %SECTOR-HEADS))
			 %SECTOR-HEADS))
		      (N-HEADS
		       (PUT-DISK-FIXNUM
			 LE-DISK-LABEL
			 (DPB VALUE %%NUMBER-OF-HEADS
			      (GET-DISK-FIXNUM LE-DISK-LABEL %SECTOR-HEADS))
			 %SECTOR-HEADS))
		      (N-CYLINDERS
		       (PUT-DISK-FIXNUM
			 LE-DISK-LABEL
			 (DPB VALUE %%NUMBER-OF-CYLINDERS
			      (GET-DISK-FIXNUM LE-DISK-LABEL %CYLINDERS))
			 %CYLINDERS))
		      (N-SECTORS-FOR-DEFECTS
		       (PUT-DISK-FIXNUM
			 LE-DISK-LABEL
			 (DPB VALUE %%NUMBER-OF-SECTORS-FOR-DEFECTS
			      (GET-DISK-FIXNUM LE-DISK-LABEL %CYLINDERS))
			 %CYLINDERS))
		      (CURRENT-MICROLOAD
		       (cond (*le-prim-style-edit*
			      (FORMAT USER:*TERMINAL-IO*
				      "Editing this field only changes the disk label, not the System Load defaults.")
			      (when (fquery '(:beep t :stream user:*terminal-io*)
					    "Change ~a to ~a anyway ? " name value)
				(setq *le-mcr-or-lod-changed* t)
				(SETF VALUE (PAD-NAME-FIELD VALUE 4))
				(SET-DEFAULT-MICROLOAD LE-DISK-LABEL VALUE)))
			     (t
			      (SETF VALUE (PAD-NAME-FIELD VALUE 4))
			      (SET-DEFAULT-MICROLOAD LE-DISK-LABEL VALUE))))
		      (CURRENT-BAND
		       (cond (*le-prim-style-edit*
			      (FORMAT USER:*TERMINAL-IO*
				      "Editing this field only changes the disk label, not the System Load defaults.")
			      (when (fquery '(:beep t :stream user:*terminal-io*)
					    "Change ~a to ~a anyway ? " name value)
				(setq *le-mcr-or-lod-changed* t)
				(SETF VALUE (PAD-NAME-FIELD VALUE 4))
				(SET-DEFAULT-LOAD-BAND LE-DISK-LABEL VALUE)))
			     (t
			      (SETF VALUE (PAD-NAME-FIELD VALUE 4))
			      (SET-DEFAULT-LOAD-BAND LE-DISK-LABEL VALUE))))
		      
						;mrr 3.14.87
		      (system-microload (setq *le-mcr-or-lod-changed* t
					      *le-mcr-name* value))
		      (system-microload-unit (setq *le-mcr-or-lod-changed* t
						   *le-mcr-unit* value))
		      (system-band (setq *le-mcr-or-lod-changed* t
					 *le-lod-name* value))
		      (system-band-unit (setq *le-mcr-or-lod-changed* t
					      *le-lod-unit* value)) 
		      
		      (N-PARTITIONS
		       (PUT-DISK-FIXNUM LE-DISK-LABEL VALUE
					(+ %PT-BASE %PT-NUMBER-OF-PARTITIONS)))
		      (WORDS-PER-PART (CHANGE-PARTITION-MAP LE-DISK-LABEL VALUE))
		      (PARTITION-TABLE-NAME
		       (SETF VALUE (PAD-NAME-FIELD VALUE 4))
		       
		       
		       (PUT-DISK-STRING LE-DISK-LABEL VALUE %DL-PARTITION-TABLE-NAME 4))
		      (PARTITION-TABLE-START
		       (PUT-DISK-FIXNUM LE-DISK-LABEL VALUE %DL-PARTITION-TABLE-START))
		      (PARTITION-TABLE-LENGTH
		       (IF (>= *MAX-PTBL-SIZE* VALUE)
			   (PUT-DISK-FIXNUM
			     LE-DISK-LABEL VALUE %DL-PARTITION-TABLE-LENGTH)
			   (PROGN
			     (FORMAT USER:*TERMINAL-IO*
				     "Partition table length cannot exceed SI:*MAX-PTBL-SIZE*, ~d."
				     *MAX-PTBL-SIZE*)
			     (BEEP)
			     (zlc:READLINE USER:*TERMINAL-IO* () '((:NO-INPUT-SAVE T))))))
		      (SAVE-AREA-NAME
		       (SETF VALUE (PAD-NAME-FIELD VALUE 4))
		       (PUT-DISK-STRING LE-DISK-LABEL VALUE %DL-SAVE-AREA-NAME 4))
		      (SAVE-AREA-START
		       (PUT-DISK-FIXNUM LE-DISK-LABEL VALUE %DL-SAVE-AREA-START))
		      (SAVE-AREA-LENGTH
		       (PUT-DISK-FIXNUM LE-DISK-LABEL VALUE %DL-SAVE-AREA-LENGTH))
		      ;; These occur in multiple instances
		      ((PARTITION-NAME PARTITION-START
				       PARTITION-SIZE PARTITION-COMMENT PARTITION-TYPE)
		       (LET ((PLOC (LE-CURRENT-PARTITION)))
			 (CASE NAME
			   (PARTITION-NAME
						;if prim-style and an explorer-type mcr or load name change, set flag. mrr 3.13.87
			    (when  *le-prim-style-edit*
			      (let* ((ploc (le-current-partition))
				     (att (get-disk-fixnum le-disk-label (+ ploc %pd-attributes))))
				(and (= (cpu-type) (ldb %%cpu-type-code att))
				     (or (= %bt-microload (ldb %%band-type-code att))
					 (= %bt-load-band (ldb %%band-type-code att)))))
			      (setq *le-mcr-or-lod-changed* t))	
			    (SETF VALUE (PAD-NAME-FIELD VALUE 4))
			    (PUT-DISK-STRING LE-DISK-LABEL VALUE (+ PLOC %PD-NAME) 4))
			   (PARTITION-TYPE (PUT-DISK-FIXNUM LE-DISK-LABEL VALUE (+ PLOC %PD-ATTRIBUTES)))
			   (PARTITION-START
			    (IF (RESERVED-BLOCKS-OVERWRITE-PROTECTION PLOC NAME VALUE)
				(PUT-DISK-FIXNUM LE-DISK-LABEL VALUE (+ PLOC %PD-START))))
			   (PARTITION-SIZE
			    (IF (RESERVED-BLOCKS-OVERWRITE-PROTECTION PLOC NAME VALUE)
				(PUT-DISK-FIXNUM LE-DISK-LABEL VALUE (+ PLOC %PD-LENGTH))))
			   (PARTITION-COMMENT
			    (PUT-DISK-STRING LE-DISK-LABEL VALUE (+ PLOC %PD-COMMENT)
					     (* 4
						(-
						  (GET-DISK-FIXNUM LE-DISK-LABEL
								   (+ %PT-BASE
								      %PT-SIZE-OF-PARTITION-ENTRIES))
						  %PD-COMMENT)))))))
		      (OTHERWISE (FERROR () "No editor for ~S" NAME)))))
	    (BEEP))
	  (ABORT (VALUES () T)))
    (LE-DISPLAY-LABEL LE-DISK-LABEL LE-UNIT))

;;;----------------------------------------------------------------------

;;; Updated Help command

;;;TI Code: from KSL-EXP-7: DISK-IO; DISK-LABEL-EDITOR.LISP#49
;;;RDA: Note changes
(DEFUN LE-COM-? ()				;2.1 fix changed listing to be alphabetic by control
  (SEND USER:*TERMINAL-IO* :CLEAR-SCREEN)		;    letter. Seperated cursor control commands.
						; Uppercased key words for readability.
  (FORMAT USER:*TERMINAL-IO*
	  "~&Commands are as follows:   
CTRL-A set partition ATTRIBUTES (Load Band, File Band, etc.).
CTRL-C set partition CPU or Operating System type (Explorer,TI Lisp, etc.).
CTRL-D DESCRIBE the current partition.
CTRL-E EDIT selected item with S, L, E (with - and +) bound to Start, Length, End.
CTRL-I INITIALIZE the label from defaults.
CTRL-K DELETE partition.
CTRL-L, SPACE, RETURN clear the screen and redisplay the label.
CTRL-M set/remove partition PROPERTIES (Diagnostic, Default, etc.).
CTRL-O ADD partition.
CTRL-R READ label from disk.
CTRL-S SORT partitions by starting block address.
CRTL-T toggles case sensitivity.
CTRL-W WRITE label to disk.
META-~~ mark buffer unmodified.

KSL added commands:
SUPER-E Insert EXPT and SAV2 or EXP2 partition(s)
SUPER-G Insert GDOS partition
SUPER-D Insert DIAG partition
SUPER-B Insert BOOT partition
SUPER-M Insert MCRn partition (n is the disk unit number)
SUPER-F Insert FILE partition
SUPER-0 Insert LOD0 type partition (actually uses disk unit number)
SUPER-END Insert END partition
SUPER-L Insert LOG partition (at end of open space)
SUPER-? Insert METR partition (at end of open space)
SUPER-1 Insert LOD1 type partition (at end of open space) (uses disk unit number)
SUPER-P Insert PAGE partition to fill available space
HYPER-0 Insert all standard unit 0 partitions
HYPER-1 Insert all standard unit 1 partitions

CONTROLS:
  CTRL-B back
  CTRL-F forward
  CTRL-N down 
  CTRL-P up.
 Or use arrow keys.

~:c-exit~%"
	  #\END)
  (FORMAT USER:*TERMINAL-IO* "~&Label Edit Command: "))

;;;----------------------------------------------------------------------

;;; Functional interfaces for commands

(defparameter *EXP2-length* 200)
(defparameter *EXPT-length* 150)
(defparameter *GDOS-length* 1024)
(defparameter *DIAG-length* 5120)
(defparameter *BOOT-length-1* 64)
(defparameter *BOOT-length-2* 128)
(defparameter *MCR-length* 300)
(defparameter *FILE-length* 10000)
(defparameter *LOD0-length* 38000)
(defparameter *LOD1-length* 40000)
(defparameter *LOG-length* 100)
(defparameter *METR-length* 0)

(defun le-insert-expt-partition ()
  "Add both EXPT and SAV2 or an EXP2 partition.  SAV2 is the difference
   between an EXP2 size partition and EXPT to make it easy to convert to
   Explorer II later."
  (if (explorer-ii?)
      (le-add-partition "EXP2" :against-prev *EXP2-length* %BT-MICROLOAD
			%cpu-TI-Explorer-II (list %%DIAGNOSTIC-INDICATOR))
      ;;Explorer I
      (progn
	(le-add-partition "EXPT" :against-prev *EXPT-length* %BT-MICROLOAD
			  %CPU-EXPLORER (list %%DIAGNOSTIC-INDICATOR))
	(LE-Display-Label LE-Disk-Label LE-Unit t)
	(le-down-line)			;Next line
	(le-add-partition "SAV2" :against-prev (- *EXP2-length* *EXPT-length*)
			  %BT-EMPTY-BAND %CPU-GENERIC-BAND))))

(defun le-insert-GDOS-partition ()
  "Add GDOS partition."
  (le-add-partition "GDOS" :against-prev *GDOS-length* %BT-MICROLOAD
		    (if (explorer-ii?)
			%cpu-TI-Explorer-II
			%CPU-EXPLORER)
		    (list %%DIAGNOSTIC-INDICATOR)))

(defun le-insert-DIAG-partition ()
  "Add DIAG partition."
  (le-add-partition "DIAG" :against-prev *DIAG-length* %BT-FILE-BAND
		    %CPU-SYSTEM5 (list %%DIAGNOSTIC-INDICATOR)))

(defun le-insert-BOOT-partition ()
  "Add BOOT partition."
  (if (explorer-ii?)
      (le-add-partition "BOOT" :against-prev *BOOT-length-2* %BT-MICROLOAD
			(if (explorer-ii?) %cpu-TI-Explorer-II %CPU-EXPLORER))
      ;;Explorer I
      (progn
	(le-add-partition "BOOT" :against-prev *BOOT-length-1* %BT-MICROLOAD
			  (if (explorer-ii?) %cpu-TI-Explorer-II %CPU-EXPLORER)) 
	(LE-Display-Label LE-Disk-Label LE-Unit t)
	(le-down-line)				       ;Next line
	(le-add-partition "SAV3" :against-prev
			  (- *BOOT-length-2* *BOOT-length-1*)
			  %BT-EMPTY-BAND %CPU-GENERIC-BAND)))
  )

(defun le-insert-MCR-partition ()
  "Add MCR partition.  This follows the KSL convention of naming MCR
   partitions with the disk unit number such that, for instance, the MCR
   partition on unit 0 will be MCR0."
  (le-add-partition (format nil "MCR~D" (unit-number le-unit))
		    :against-prev *MCR-length* %BT-MICROLOAD
		    (if (explorer-ii?) %cpu-TI-Explorer-II %CPU-EXPLORER)
		    (list %%DEFAULT-INDICATOR)))

(defun le-insert-FILE-partition ()
  "Add FILE partition."
  (le-add-partition "FILE" :against-prev *FILE-length*
		    %BT-FILE-BAND %CPU-EXPLORER))

(defun le-insert-LOD0-type-partition ()
  "An LOD0-type partition is one that comes just after the diags, etc, before
   paging, and is sized to hold a 'virgin' world.  In the KSL this is usually
   on unit 0.  This function inserts a load partition called LODn where n is
   the unit number."
  (le-add-partition (format nil "LOD~D" (unit-number le-unit))
		    :against-prev *LOD0-length* %BT-LOAD-BAND
		    %CPU-EXPLORER (list %%DEFAULT-INDICATOR)))

(defun le-insert-END-type-partition ()
  "Put a 0 length END partition at the end of the disk."
  ;;The following will set *WORKING-DISK-SIZE*
  (CHECK-FOR-END-OF-DISK LE-Disk-Label (LE-Current-Partition))
  (le-add-partition "END" *working-disk-size* 0
		    %BT-EMPTY-BAND %CPU-GENERIC-BAND))

(defun le-insert-LOG-type-partition ()
  (le-add-partition "LOG" :against-next *LOG-length* %BT-LOG-BAND))

(defun le-insert-METR-type-partition ()
  (le-add-partition "METR" :against-next *METR-length* %BT-METER-BAND))

(defun le-insert-LOD1-type-partition ()
  "An LOD1-type partition is one that comes after paging to allow a file
   partition before paging easy expansion.  It is typically sized a little
   large to hold customized worlds.  This function inserts a load partition
   called LODn where n is the unit number butted up against the FOLLOWING
   partition to make for easy insertion of PAGE."
  (le-add-partition (format nil "LOD~D" (unit-number le-unit))
		    :against-next *LOD1-length* %BT-LOAD-BAND %CPU-EXPLORER
		    (list %%DEFAULT-INDICATOR)))

(defun le-insert-PAGE-type-partition ()
  "Insert a PAGE partition to fill the space between the current partition
   and the following partition."
  (le-add-partition "PAGE" :against-prev
		    (- (start-of-partition 0) (end-of-partition -1))
		    %BT-PAGE-BAND))

(defun le-insert-diagnostic-partitions ()
  (le-com-super-e)
  (le-com-super-g)
  (le-com-super-d))

(defun le-insert-microcode-partitions ()
  (le-com-super-b)
  (le-com-super-m))

(defun le-insert-unit-0-partitions ()
  (le-com-super-0)
  (le-com-super-end)
  (le-com-super-l)
  (le-com-super-?)
  (le-com-super-p)
  )

(defun le-insert-unit-1-partitions ()
  (le-com-super-f)
  (le-com-super-end)
  (le-com-super-1)
  (le-com-super-p)
  )

(defun le-insert-all-unit-0-partitions ()
  (le-insert-diagnostic-partitions)
  (le-insert-microcode-partitions)
  (le-insert-unit-0-partitions)
  )

(defun le-insert-all-unit-1-partitions ()
  (le-insert-diagnostic-partitions)
  (le-insert-microcode-partitions)
  (le-insert-unit-1-partitions)
  )

;;;----------------------------------------------------------------------

;;; New commands

(defun le-com-super-e ()
  (le-insert-EXPT-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit)
  (le-down-line))

(defun le-com-super-g ()
  (le-insert-GDOS-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit)
  (le-down-line))

(defun le-com-super-d ()
  (le-insert-DIAG-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit)
  (le-down-line))

(defun le-com-super-b ()
  (le-insert-BOOT-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit)
  (le-down-line))

(defun le-com-super-m ()
  (le-insert-MCR-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit)
  (le-down-line))

(defun le-com-super-f ()
  (le-insert-FILE-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit)
  (le-down-line))

(defun le-com-super-0 ()
  (le-insert-LOD0-type-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit)
  (le-down-line))

(defun le-com-super-end ()
  (le-insert-END-type-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

(defun le-com-super-l ()
  (le-insert-LOG-type-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

(defun le-com-super-? ()
  (le-insert-METR-type-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

(defun le-com-super-1 ()
  (le-insert-LOD1-type-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

(defun le-com-super-p ()
  (le-insert-PAGE-type-partition)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

(defun le-com-hyper-d ()
  (le-insert-diagnostic-partitions)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

(defun le-com-hyper-m ()
  (le-insert-microcode-partitions)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

(defun le-com-hyper-0 ()
  (le-insert-all-unit-0-partitions)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))


(defun le-com-hyper-1 ()
  (le-insert-all-unit-1-partitions)
  (setq le-something-changed t)
  (LE-Display-Label LE-Disk-Label LE-Unit))

;;; Insert commands
(setf *EDIT-DISK-LABEL-COMMAND-ALIST*
      (append *EDIT-DISK-LABEL-COMMAND-ALIST*
	      '((#\super-e . le-com-super-e)
		(#\super-g . le-com-super-g)
		(#\super-d . le-com-super-d)
		(#\super-b . le-com-super-b)
		(#\super-m . le-com-super-m)
		(#\super-f . le-com-super-f)
		(#\super-0 . le-com-super-0)
		(#\super-end . le-com-super-end)
		(#\super-l . le-com-super-l)
		(#\super-? . le-com-super-?)
		(#\super-1 . le-com-super-1)
		(#\super-p . le-com-super-p)
		(#\hyper-d . le-com-hyper-d)
		(#\hyper-m . le-com-hyper-m)
		(#\hyper-0 . le-com-hyper-0)
		(#\hyper-1 . le-com-hyper-1)
		)))
;;;----------------------------------------------------------------------

;;; Date and time functions

time:
(defun dd-mmm-yy-string (&optional (univ-time (time:get-universal-time)))
  (multiple-value-bind (sec min hr day mon yr)
      (time:decode-universal-time univ-time)
    (declare (ignore sec min hr))
    (format nil "~2,'0D-~A-~2D"
	    day
	    (time:month-string mon :short)
	    (mod yr 100)
	    )
    )
  )

time:(import '(dd-mmm-yy-string) 'ticl)
time:(export '(dd-mmm-yy-string))

;;;----------------------------------------------------------------------

;;; SETF for ASSOC

(defun assoc-setf (item alist assoc-options newval)
  "If (ASSOC ITEM ALIST) finds an entry, set CDR of it to (CDR NEWVAL), else
   add (CONS ITEM (CDR NEWVAL)) to the end of ALIST.  Returns the (possibly
   new) entry.  Signals an error if ALIST isn't a list or if (NEQ (CAR NEWVAL)
   ITEM)."
   (check-arg alist consp "a list")
   (unless (eq (car newval) item)
     (error "Attempt to make ASSOC of ~S return ~S" item newval))
   (let ((item (apply #'assoc item alist assoc-options)))
     (if item
	 (setf (rest item) (cdr newval))
	 (nconc alist (list (setf item (cons item (cdr newval)))))
	 )
     item
     )
  )

(defsetf assoc (item alist &rest options) (newval)
  `(assoc-setf ,item ,alist (list ,@options) ,newval)
  )

;;;----------------------------------------------------------------------

;;; This bit of code put in by JPR.  The reason for this is that
;;; there are lots of flavors that are created/loaded by the tools,
;;; but, for whatever reasons, these are not all compile-flavor-methodsed.
;;; What we want to do, therefore is to try to c-f-m the flavors if we
;;; can so we try to do this for each flavor in si:*all-flavor-names*
;;; and put a catch-error around it, so that we do all the ones that we can.
;;; As an extra complication we also want to Finalize all classes that
;;; we know about, so this is done for each flavor if reasonable too.
;;; The trouble is that a) there is no global list of all classes so
;;; we can only do it for the classes that also have flavor definitions
;;; and b) not all classes are finalizable, for instance, Flavor-Classes
;;; are not finalizable for some reason, so we don't try to finalize these.
;;; Ok?  The whole TICLOS bit is conditionalised, just in case we ship
;;; to some unfortunate soul without TICLOS.

(defvar *compile-flavors-and-classes-before-full-gc-p* nil)
(defvar *full-gc-compile-flavor-methods-notify-stream* 'si:null-stream)

(defvar *all-named-class-names* nil)

(let ((compiler:compile-encapsulations-flag t))
     (advise ticlos:add-named-class-internal :around :record-named-classes nil
       (let ((results (multiple-value-list :do-it)))
	    (pushnew (second arglist) *all-named-class-names*)
	    (values-list results)
       )
     )
)
  
#+CLOS
(defun build-gf-hash-tables (a-class)
  (mapc #'(lambda (class)
	      (unless (and (eq class ticlos:*t-class*)
			   (not (eq class a-class)))
		(ticlos:map-specializer-direct-generic-functions
		  #'ticlos:prepare-generic-function class)))
	  (ticlos:class-precedence-list a-class))
  (values)
;  (let ((any-built nil))
;       (mapcar
;	 #'(lambda (class)
;	    (mapcar #'(lambda (gf)
;			(if (not (ticlos:generic-function-method-hash-table gf))
;			    (progn (ticlos:%build-method-hash-table
;				     (ticlos:generic-function-discriminator-code
;				       gf
;				     )
;				   )
;				   (setq any-built t)
;			    )
;			    nil
;			)
;		      )
;		      (ticlos:specializer-direct-generic-functions class)
;	    )
;	   )
;	   (ticlos:class-precedence-list a-class)
;       )
;       (if any-built :built t)
;  )
)


(defun try-to-compile-flavor-methods (flav)
  (if (or (getf (flavor-plist (get flav 'flavor)) :Required-Flavors)
	  (getf (flavor-plist (get flav 'flavor)) :Alias-Flavor)
	  (getf (flavor-plist (get flav 'flavor)) :required-methods)
	  (getf (flavor-plist (get flav 'flavor))
		:Required-Instance-Variables
	  )
      )
      :cant
      (progn (or (flavor-depends-on-all (get flav 'flavor))
		 (compose-flavor-combination (get flav 'flavor))
	     )
	     (if (flavor-method-hash-table (get flav 'flavor))
		 :already
		 (progn
		   (si:compose-method-combination (get flav 'flavor))
		   t
		 )
	     )
      )
  )
)

(defun compile-flavor-and-or-class
       (flav &optional (notify-stream *standard-output*))
  (if (get flav 'flavor)
      (let ((compiled? (catch-error (try-to-compile-flavor-methods flav) nil)))
	   (case compiled?
	     (:cant (format notify-stream "Cannot Compile Flavor Methods for ~S"
			    flav
		    )
	     )
	     (:already
	      (format notify-stream
		      "Already Compiled Flavor Methods for ~S" flav
	      )
	     )
	     (nil (format notify-stream
			  "***** Failed Compile Flavor Methods for ~S"
			  flav
		  )
	     )
	     (t (format notify-stream "Compiled Flavor Methods ok for ~S" flav))
	   )
      )
      nil
  )
;  #+CLOS
;  (if (and (ticlos:find-class flav nil)
;	   (not (typep (ticlos:find-class flav nil)
;		       'ticlos:flavor-class
;		)
;	   )
;      )
;      (let ((compiled?
;	      (catch-error
;	       (progn (ticlos:finalize-inheritance (ticlos:find-class flav))
;		      (build-gf-hash-tables (ticlos:find-class flav))
;	       )
;	       nil
;	      )
;	    )
;	   )
;	   (if (get flav 'flavor)
;	       (format notify-stream ", ")
;	       nil
;	   )
;	   (case compiled?
;	     (:built
;	      (format notify-stream "Finalized & Built Inheritance for ~S"
;		      flav
;	      )
;	     )
;	     (nil (format notify-stream
;			  "***** Failed Finalized Inheritance for ~S"
;			  flav
;		  )
;	     )
;	     (t (format notify-stream "Finalized Inheritance for ~S" flav))
;	   )
;      )
;      nil
;  )
)

(defun compile-flavors-and-classes (&optional (notify-stream *standard-output*))
  (timeit (:interrupts :print notify-stream :units :seconds :time)
;    #+CLOS
;    (format notify-stream "~&Compiling all known named classes:")
;    #+CLOS
;    (loop for i from (length *all-named-class-names*) downto 1
;	  for class in *all-named-class-names* do
;	  (format notify-stream "~&~D:	" i)
;	  (compile-flavor-and-or-class class notify-stream)
;    )
;    #+CLOS
;    (format notify-stream "~%~%~&Compiling all flavors:")
    (loop for i from (length *all-flavor-names*) downto 1
	  for flavor in *all-flavor-names* do
	  (format notify-stream "~&~D:	" i)
	  (compile-flavor-and-or-class flavor notify-stream)
    )
  )

  nil ;;; We don't want to return this list.  It's very large.
)

(Add-Initialization
   "Compile Flavors and Classes"
  '(if *compile-flavors-and-classes-before-full-gc-p*
       (compile-flavors-and-classes
	 *full-gc-compile-flavor-methods-notify-stream*
       )
       nil
   )
  '(:Full-GC :Normal)
)

;;;----------------------------------------------------------------------

;;; Way to size a Mac screen from Lisp

(defun mac:resize-a-screen (screen new-height new-width)
  "SCREEN is an object of class TV:SCREEN (eg. from TV:DEFAULT-SCREEN).
   NEW-HEIGHT and NEW-WIDTH are the number of pixels high and wide you want
   SCREEN to be."
  (declare (special mac:*mac-resident-explorer-screens*))
  (let ((id
	  (loop for i from 0 to (1- (length mac:*mac-resident-explorer-screens*))
		as entry = (aref mac:*mac-resident-explorer-screens* i)
		when (and (consp entry)
			  (eq (mac:the-screen entry) screen))
		do (return i))))
    (mac:resize-screen id new-height new-width)
    )
  )

;;;----------------------------------------------------------------------

(defun call-many (fn process-args error-value &rest args)
  "Simple way to do many runs of FN in parallel.  ARGS should be a set of
   lists, one for each arguement to FN.  FN is called with all the first
   elements, again with all the second elements, and so on.  Each call to FN
   happens in its own process.  CALL-MANY returns an unordered list
   containing the first values of the calls to FN.  PROCESS-ARGS is passed to
   PROCESS-RUN-FUNCTION.  ERROR-VALUE is placed into the result list when a
   call to FN results in an error.  For example, (CALL-MANY #'CONS NIL NIL
   '(A B C) '(X Y Z)) might return ((A .  X) (B .  Y) (C .  Z)).  Only as
   many evaluations as the length of the shortest arg list are made.  While
   waiting for the results to accumulate the process running CALL-MANY will
   display ''N Calls'' where N is the number of calls yet to produce
   results."
  (let ((Count 0)
	(Results nil))
    (flet ((call-many-top-level (fn my-args)
				(multiple-value-bind (res error?)
				    (catch-error (apply fn my-args) nil)
				  (if error? (setf res error-value))
				  (without-interrupts
				    (push res Results)
				    (decf Count)))))
      ;;This array gives us a pointer to the arg lists that we can modify
      (let ((args-lists (make-array (length args) :initial-contents args)))
	(do ((done? nil)
	      (these-args nil nil))		       ;Reset this each time
	    (done?)
	  ;;Collect args for one call, going backwards to get the order right
	  (do ((idx (1- (length args-lists)) (1- idx)))	       
	      ((minusp idx))
	    (declare (type fixnum idx))
	    ;;Remove arg from list and save it
	    (push (pop (aref args-lists idx)) these-args)
	    (setq done? (or done? (not (aref args-lists idx)))))
	  (process-run-function
	    (or process-args (format nil "~A" (first these-args)))
	    #'call-many-top-level fn these-args)
	  (without-interrupts
	    (incf Count))))
      (loop until (zerop Count) doing
	    (process-wait (format nil "~D Calls" Count)
			  #'(lambda (old-count)
			      (not (= Count old-count)))
			  Count)))
    Results))
;;;----------------------------------------------------------------------

(defun is-in-tree (symbol tree)
"Is true if Symbol occurs in Tree.  Special attention is taken to watch for
 Setfs so that the symbol is not found in the first component of the thing
 being set.
"
  (declare (optimize (safety 0)))
  (typecase tree
    (symbol (eq symbol tree))
    (cons (if (and (equal (first tree) 'setf)
		   (consp (second tree))
	      )
	      (or (is-in-tree symbol (rest (second tree)))
		  (is-in-tree symbol (third tree))
		  (is-in-tree symbol (cons 'setf (rest (rest (rest tree)))))
	      )
	      (or (is-in-tree symbol (first tree))
		  (is-in-tree symbol (rest tree))
	      )
	  )
    )
    (otherwise nil)
  )
)

;-------------------------------------------------------------------------------

(provide 'utilities)


