1;;; -*- Mode: Common-Lisp; Package: SPELLER; Fonts: cptfont,cptfontb; Base: 10 -*-


;;;
;;; Copyright (c) 1987 Douglas Johnson.  All rights reserved. 
;;; Copyright (c) 1985 Douglas Johnson.  All rights reserved. 
;;;

;;;************************************************************************************************
;;;                                                                                               *
;;;                                                                                               *
;;;                    The following routines deal with spelling correction.                      *
;;;                                                                                               *
;;;                                                                                               *
;;;************************************************************************************************             *

(DEFVAR *PROMPT-IN-MINIBUFFER* NIL
  "2If NIL, corrections are asked for by menu, otherwise the 
   mini-buffer is used.*")


(DEFUN EDITOR-CORRECT-WORD (WORD)
  "2Returns a word that has been corrected by the user.*"
  (1if* (word-checkable-p word)
       (IF *PROMPT-IN-MINIBUFFER*
	   (EDITOR-CORRECT-WORD-MINIBUFFER WORD)
	   (EDITOR-CORRECT-WORD-MENU WORD))
       word))

(DEFUN EDITOR-CORRECT-WORD-MENU (WORD)
  "Returns a word that has been corrected by the user.  Variant
   spellings of WORD are found and menu is popped up to let the 
   user select one."
  (LET* ((WORD (STRING-REMOVE-FONTS (STRING-DOWNCASE WORD)))
	 (WORD-LIST (FIND-VARIATIONS WORD))
	 (CHOOSE-LIST NIL)
	 (RWORD ""))
	 (DOLIST (ITEM WORD-LIST)
	   (SETQ CHOOSE-LIST (NCONC CHOOSE-LIST (LIST (LIST ITEM
							    :VALUE ITEM 
							    :DOCUMENTATION "Replace error with this word"
							    :FONT FONTS:HL12B)))))
	 (SETQ CHOOSE-LIST (NCONC CHOOSE-LIST `(("Skip" :VALUE ,WORD
						 :DOCUMENTATION ,(STRING-APPEND "Do not correct \""
									        WORD "\"")
						 :FONT FONTS:HL12I)
						("Accept" :EVAL (ADD-LOCAL-WORD ,WORD NIL)
						 :DOCUMENTATION ,(STRING-APPEND "Accept \""
									        WORD 
									       "\" in this document.")
						 :FONT FONTS:HL12I)
						("Abort" :EVAL (THROW 'SPELLER:ABORT NIL)
						 :DOCUMENTATION "Stop spelling correction."
						 :FONT FONTS:HL12I)
						("Type-in" :EVAL (ZWEI:TYPEIN-LINE-READLINE-WITH-DEFAULT
								   ,WORD
								   (STRING-APPEND "Replacement for \"" ,WORD
										  "\" - Press END to continue"))
						 :DOCUMENTATION ,(STRING-APPEND "Type in replacement for \""
									        WORD "\"")
						 :FONT FONTS:HL12I))))	
										  	       
	 (1prog1*
	   (IF (SETQ RWORD
#+elroy		   (W:MENU-CHOOSE CHOOSE-LIST
							  :LABEL
					(STRING-APPEND "Correcting " (STRING #\") WORD (string #\")))
#-elroy            (TV:MENU-CHOOSE CHOOSE-LIST
				   (STRING-APPEND "Correcting " (STRING #\") WORD (string #\")))
                   )
	     RWORD
	     WORD)
	   (1send* *query-io* :Mouse-Select)))) ;; JPR.  Menu tends to deselect us.


(DEFUN EDITOR-CORRECT-WORD-MINIBUFFER (WORD)
  "Returns a word that has been corrected by the user.  Variant
   spellings of WORD are found and displayed in the minibuffer so a 
   user can select one."
  (LET* ((WORD (STRING-REMOVE-FONTS (STRING-DOWNCASE WORD)))
	 (WORD-LIST (FIND-VARIATIONS WORD))
	 (RWORD WORD)
	 REPLY)
    (SEND *QUERY-IO* :CLEAR-SCREEN)
    (LOOP DOING
       (LOOP WITH SIZE = (SEND *QUERY-IO* :SIZE-IN-CHARACTERS) WITH TOTAL = 0 FOR N FROM 0 BELOW
	  9
	  ;1; Display a maximum of 9 words*
	  FOR WORD IN WORD-LIST FOR LEN = (+ (LENGTH WORD) 5) DO
	  (WHEN (> (+ TOTAL LEN) SIZE);1Fold line on word boundaries*
	    (TERPRI *QUERY-IO*)
	    (SETQ TOTAL 0))
	  (FORMAT *QUERY-IO* "~d ~a   " (1+ N) WORD) (INCF TOTAL LEN))
       (FORMAT *QUERY-IO* "~&Correcting \"~a\" (Skip, Accept, Type-in, or <abort>) " WORD)
       (WHEN (AND REPLY (CHAR= REPLY #\HELP))
	 (SETQ REPLY (CORRECT-WORD-HELP WORD WORD-LIST)))
       (SETQ RWORD
	     (CASE (SETQ REPLY (OR REPLY (CHARACTER (SEND *QUERY-IO* :TYI))))
	       ((#\s #\S #\SPACE) WORD)
	       ((#\a #\A) (ADD-LOCAL-WORD WORD) WORD)
	       ((#\t #\T)
		(ZWEI::TYPEIN-LINE-READLINE-WITH-DEFAULT WORD
							 (STRING-APPEND "Replacement for \""
									WORD
									"\" - Hit END to continue")))
	       ((#\w) (THROW 'ABORT
			     ()))
	       (OTHERWISE
		(IF (AND WORD-LIST (< #\0 REPLY (+ #\1 (LENGTH WORD-LIST))))
		  (NTH (- REPLY #\1) WORD-LIST)
		  (SETQ REPLY #\HELP)))))
       WHILE (CHAR= REPLY #\HELP))
    RWORD)) 

(DEFUN CORRECT-WORD-HELP (WORD word-list)
  "2Display the help menu for editor-correct-word*"
  (LET ((choose-list
	  (LOOP for ITEM in WORD-LIST
		for i upfrom #\1 collecting
		`(,(FORMAT nil "~c - ~a" i item) :value ,i
		  :DOCUMENTATION "Replace error with this word" :FONT FONTS:HL12B))))
    (SETQ choose-list
	  (APPEND `(("<space> or S - Skip" :VALUE #\a
		     :DOCUMENTATION ,(STRING-APPEND "Do not correct \"" WORD "\"")
		     :FONT FONTS:HL12I)
		    ("A - Accept" :value #\i
		     :Documentation ,(STRING-APPEND "Accept \"" WORD "\" in this document.")
		     :FONT FONTS:HL12I)
		    ("T - Type-in" :value #\r
		     :DOCUMENTATION ,(STRING-APPEND "Type in replacement for \"" WORD "\".")
		     :FONT FONTS:HL12I)
		    ("<Abort>" :value #\w
		     :DOCUMENTATION "Stop spelling correction."
		     :FONT FONTS:HL12I)
		    ) CHOOSE-LIST))
    (prog1
    #-elroy(TV:MENU-CHOOSE CHOOSE-LIST
		    (FORMAT nil "These are your options for correcting \"~a\".~@
				 Move the mouse off this window, or~@
				 Click on an option." word))
    #+elroy (W:MENU-CHOOSE CHOOSE-LIST
		    :LABEL
		    (FORMAT nil "These are your options for correcting \"~a\".~@
				 Move the mouse off this window, or~@
				 Click on an option." word))
      (1send* zwei:*window* :Mouse-Select)))) ;; JPR.  Menu tends to deselect us.
 


(DEFUN FIND-VARIATIONS (WORD)
  "Returns a list of words that are correctly spelled variants of WORD."
  (DELETE-DUPLICATE-WORDS 
    (NCONC (DELETE-VARIANTS WORD)
	   (TRANSPOSE-VARIANTS WORD)
	   (INSERT-VARIANTS WORD)
	   (REPLACE-VARIANTS WORD)
	   (SPLIT-VARIANTS WORD))))



(DEFUN DELETE-DUPLICATE-WORDS (LIST)
  "Returns LIST with the duplicate words deleted."
  (COND
    ((NULL LIST) NIL)
    ((MEMBER (CAR LIST) (CDR LIST) :TEST #'STRING-EQUAL) (DELETE-DUPLICATE-WORDS (CDR LIST)))
    (T (APPEND (LIST (CAR LIST)) (DELETE-DUPLICATE-WORDS (CDR LIST)))))) 

(DEFUN SPLIT-VARIANTS (WORD)
  "Returns a list of the correctly spelled variants of
   WORD created by spliting WORD into two."
  (LET ((WORD-LIST NIL)
	(LENGTH (1- (LENGTH WORD))))
    (DOTIMES (N LENGTH WORD-LIST)
      (LET ((FIRST (SUBSEQ (STRING WORD) 0 (1+ N)))
	    (REST (SUBSEQ (STRING WORD) (1+ N))))
	(IF (AND (CHECK-WORD-P FIRST) (CHECK-WORD-P REST))
	  (SETQ WORD-LIST (NCONC WORD-LIST (LIST (STRING-APPEND FIRST " " REST))))))))) 


(DEFSUBST NDEL-CHAR (WORD NUMBER)
  "Returns WORD with the NUMBERth character deleted from it 
   destructively.  WORD must have a fill pointer."
  (LET ((LENGTH (LENGTH WORD)))
    (UNLESS (OR (MINUSP NUMBER) (>= NUMBER LENGTH))
      (DECF (FILL-POINTER WORD))
      (LOOP FOR I FROM NUMBER TO (1- LENGTH) DO
	  (SETF (AREF WORD I) (AREF WORD (1+ I)))))
    WORD))

(DEFUN DELETE-VARIANTS (WORD)
  "Returns a list of the correctly spelled variants of
   WORD created by deleting single letters out of WORD."
  (LET ((WORD-LIST NIL)
	(LENGTH (LENGTH WORD)))
    (DOTIMES (NUMBER LENGTH WORD-LIST)
      (LET ((DWORD (NDEL-CHAR (NSTRING-MOVE WORD *STRING*) NUMBER)))
	(IF (CHECK-WORD-P DWORD)
	  (PUSH (STRING-COPY DWORD) WORD-LIST)))))) 

(DEFSUBST NTRANS-CHAR (WORD N)
  "Distructively modifies WORD to have the Nth 
   and N+1 characters swapped in it."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((MINUSP N) WORD)
      ((>= N (1- LENGTH)) WORD)
      (T
       (LET ((C1 (GET-CHAR WORD N))
	     (C2 (GET-CHAR WORD (1+ N))))
	 (PUT-CHAR WORD N C2)
	 (PUT-CHAR WORD (1+ N) C1)
	 WORD))))) 

(DEFUN TRANSPOSE-VARIANTS (WORD)
  "Returns a list of the correctly spelled variants of
   created by transposing pairs of letters in WORD."
  (LET ((WORD-LIST NIL)
	(LENGTH (LENGTH WORD))
	(TWORD (STRING-COPY WORD)))
    (DOTIMES (NUMBER (1- LENGTH) WORD-LIST)
      (NTRANS-CHAR TWORD NUMBER);transpose characters in TWORD 
      (IF (CHECK-WORD-P TWORD)
	(UNLESS (STRING-EQUAL WORD TWORD)
	  (SETQ WORD-LIST (NCONC WORD-LIST (LIST (STRING-COPY TWORD))))))
      (NTRANS-CHAR TWORD NUMBER);restore characters in TWORD
))) 

(DEFSUBST INS-CHAR (WORD N CHAR)
  "Returns a string that is WORD with CHAR
   inserted before the Nth character."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((MINUSP N) WORD)
      ((>= N LENGTH) (STRING-APPEND WORD (STRING CHAR)))
      ((ZEROP N) (STRING-APPEND (STRING CHAR) WORD))
      (T (STRING-APPEND (SUBSEQ (STRING WORD) 0 N) (STRING CHAR) (SUBSEQ (STRING WORD) N)))))) 

(DEFSUBST NINS-CHAR (WORD NUMBER CHAR)
  "Returns WORD with CHAR destructively inserted 
   before the Nth character.  WORD must have a fill pointer."
  (LET ((LENGTH (LENGTH WORD)))
    ;; Change by JPR?
    (UNLESS (OR (>= (+ 1 LENGTH) (ARRAY-TOTAL-SIZE WORD)) (MINUSP NUMBER) (> NUMBER LENGTH))
      (INCF (FILL-POINTER WORD))
      (DO ((I LENGTH (1- I)))
	  ((< I NUMBER))
	 (SETF (AREF WORD (1+ I)) (AREF
				   WORD I)))
      (SETF (AREF WORD NUMBER) CHAR))
    WORD)) 

(DEFUN INSERT-VARIANTS (WORD)
  "Returns a list of the correctly spelled variants of
   WORD created by inserting single letters into WORD."
  (LET* ((WORD-LIST NIL)
	 (LENGTH (1+ (LENGTH WORD)))
	 (ALPHAS "abcdefghijklmnopqrstuvwxyz'")
	 (LENGTH-ALPHAS (LENGTH ALPHAS)))
    (DOTIMES (NUMBER LENGTH WORD-LIST)
      (LET ((DWORD (NINS-CHAR (NSTRING-MOVE WORD *STRING*) NUMBER #\SPACE)))
	(DOTIMES (N LENGTH-ALPHAS)
	   (SETF (AREF DWORD NUMBER) (AREF ALPHAS N))
	  (WHEN (CHECK-WORD-P DWORD)
	    (PUSH (STRING-COPY DWORD) WORD-LIST)))))))  


(DEFUN REPLACE-VARIANTS (WORD)
  "Returns a list of the correctly spelled variants of
   WORD created by replacing single letters in WORD."
  (LET* ((WORD-LIST NIL)
	 (LENGTH (LENGTH WORD))
	 (ALPHAS "abcdefghijklmnopqrstuvwxyz")
	 (LENGTH-ALPHAS (LENGTH ALPHAS))
	 (CHAR ()))
    (NSTRING-MOVE WORD *STRING*)
    (DOTIMES (NUMBER LENGTH WORD-LIST)
      (SETQ CHAR (GET-CHAR *STRING* NUMBER))1;save the character in the spot*
      (DOTIMES (N LENGTH-ALPHAS)
	(PUT-CHAR *STRING* NUMBER (GET-CHAR ALPHAS N))1;try the other characters*
	(IF (CHECK-WORD-P *STRING*)1;if it is a word, remember it*
	  (UNLESS (OR (MEMBER *STRING* WORD-LIST :TEST #'STRING-EQUAL) (STRING-EQUAL *STRING* WORD))
	    (PUSH (STRING-COPY *STRING*) WORD-LIST))))
      (PUT-CHAR *STRING* NUMBER CHAR)1;restore the character that was there*
))) 
