;;; -*- Mode:Common-Lisp; Package:TELNET; Base:10 -*-

;;; This software developed by:
;;;	Rich Acuff
;;; at the Stanford University Knowledge Systems Lab in Mar '87.
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15

;;;----------------------------------------------------------------------
;;; Portions of this code indicated by the comment line:
;;;	;TI Code
;;; are derived from code licensed from Texas Instruments Inc.
;;; KSL's changes are noted by comment lines beginning with:
;;;	;RDA:
;;;  The following restrictions apply to the TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

;;;  A version of the VT100 emulator that runs happily in one window,
;;;  and allows one to specify fonts.

;;;----------------------------------------------------------------------

;;; A little support to help with different versions
(unless (fboundp 'sys:mx-p)
  sys:(defun mx-p ()
	(eq sys:(processor-type microcode-type-code) :micro-Explorer)
	)
  )

;;;  Constants and parameters

(progn
  (defvar *VT100-default-standard-font* 'fonts:wider-medfnt
    "The default display font for the VT100."
  )
  (defvar *VT100-default-bold-font* 'fonts:higher-medfnb
    "The default bold font for the VT100."
  )
  (defvar *VT100-default-double-wide-font* 'fonts:wider-font
    "The default double width font for the VT100."
  )
  (defvar *VT100-default-double-wide-graphics-font*
	  'fonts:vt-graphics-wider-font
    "The default double width graphics font for the VT100."
  )
  (defvar *VT100-default-double-height-top-font* 'fonts:top
    "The default top font for double height display for the VT100."
  )
  (defvar *VT100-default-double-height-graphics-top-font*
	  'fonts:vt-graphics-top
    "The default top graphics font for double height display for the VT100."
  )
  (defvar *VT100-default-double-height-bottom-font* 'fonts:bottom
    "The default bottom font for double height display for the VT100."
  )
  (defvar *VT100-default-double-height-graphics-bottom-font*
	  'fonts:vt-graphics-bottom
    "The default bottom graphics font for double height display for the VT100."
  )
  (defvar *VT100-default-graphics-font* 'fonts:vt-graphics
    "The default graphics font for the VT100."
  )
  (defvar *VT100-default-narrow-font* 'fonts:higher-tr8
    "The default narrow font for the VT100."
  )
)

;; Only one pane
(defparameter single-VT100-constraints
   '((landscape-no-keypad . ((VT100-pane)
			     ((VT100-pane :even))))
     )
  )

;;  VT100-pane is the only one used, but the others keep VT100-frame
;;  happy.
(if (sys:mx-p)
    ;; Then don't include LED and MENU panes
    (defparameter single-VT100-panes
		  `((VT100-pane VT100-screen
				:font-map	       ;default init to regular fonts
				(FONTS:WIDER-MEDFNT
				  FONTS:HIGHER-MEDFNB
				  FONTS:TOP
				  FONTS:BOTTOM
				  FONTS:VT-GRAPHICS
				  FONTS:WIDER-FONT
				  FONTS:HIGHER-TR8
				  FONTS:VT-GRAPHICS-WIDER-FONT
				  FONTS:VT-GRAPHICS-TOP
				  FONTS:VT-GRAPHICS-BOTTOM)
				:borders 0	       ;get borders from frame
				:border-margin-width 3
				)
;     (led-pane led)
;     (menu-telnet VT100-telnet-menu)
;     (menu-keypad auxiliary-keypad)
		    )
      )
    ;; Else do include LED and MENU panes for old-style VT100-FRAME
    (defparameter single-VT100-panes
		  `((VT100-pane VT100-screen
				:font-map	       ;default init to regular fonts
				(FONTS:WIDER-MEDFNT
				  FONTS:HIGHER-MEDFNB
				  FONTS:TOP
				  FONTS:BOTTOM
				  FONTS:VT-GRAPHICS
				  FONTS:WIDER-FONT
				  FONTS:HIGHER-TR8
				  FONTS:VT-GRAPHICS-WIDER-FONT
				  FONTS:VT-GRAPHICS-TOP
				  FONTS:VT-GRAPHICS-BOTTOM)
				:borders 0	       ;get borders from frame
				:border-margin-width 3
				)
		    (led-pane led)
		    (menu-telnet VT100-telnet-menu)
		    (menu-keypad auxiliary-keypad)
		    )
      )
    )

;;;----------------------------------------------------------------------

;;; The flavor that does most of the overlay work

(defflavor single-window-VT100
	   ((standard-font *VT100-default-standard-font*)
	    (graphics-font *VT100-default-graphics-font*)
	    (double-wide-font *VT100-default-double-wide-font*)
	    (double-wide-graphics-font *VT100-default-double-wide-graphics-font*)
	    (double-height-top-font *VT100-default-double-height-top-font*)
	    (double-height-bottom-font *VT100-default-double-height-bottom-font*)
	    (double-height-graphics-top-font *VT100-default-double-height-graphics-top-font*)
	    (double-height-graphics-bottom-font *VT100-default-double-height-graphics-bottom-font*)
	    (bold-font *VT100-default-bold-font*)
	    (narrow-font *VT100-default-narrow-font*)
	    )
	   (tv:box-label-mixin VT100-frame)
  (:default-init-plist
    :borders 1
    :panes single-VT100-panes
    :constraints single-VT100-constraints
    :label-box-p t
    )
  (:initable-instance-variables)
  (:documentation "A version of VT100-FRAME that uses only one big pane,
and allows setting of fonts with the :SET-VT100-FONT method.")
  )

;;; Font stuff

(defmethod (single-window-VT100 :get-VT100-font) (type)
  (select type
    (:graphics graphics-font)
    (:double-wide double-wide-font)
    (:double-height-top double-height-top-font)
    (:double-height-bottom double-height-bottom-font)
    (:double-wide-graphics double-wide-graphics-font)
    (:double-height-graphics-top double-height-graphics-top-font)
    (:double-height-graphics-bottom double-height-graphics-bottom-font)
    (:standard 
     (if column-mode-80-p
	 (if bold-on-p bold-font standard-font)
	 narrow-font))
    (otherwise standard-font))
  )

(defmethod (single-window-VT100 :set-VT100-font) (type font)
  "Sets the font type TYPE to be FONT."
  (flet ((set-font (idx)
		   (let* ((map (send VT100-pane :font-map))
			  (old-font (aref map idx)))
		     (setf (aref map idx) font)
		     (send vt100-pane :set-font-map map)
		     (when (eq type :standard)	;do all extra fonts too
		       (do ((i 10 (1+ i)))
			   ((>= i (array-active-length map)))
			 (setf (aref map i) font)))
		     (when (eq old-font (send VT100-pane :current-font))
		       (send VT100-pane :set-current-font font)
		       )
		     )
		   )
	 )
    (case type
      (:standard (setq standard-font font)
		 (set-font 0))
      (:bold (setq bold-font font)
	     (set-font 1))
      (:double-height-top (setq double-height-top-font font)
			  (set-font 2))
      (:double-height-bottom (setq double-height-bottom-font font)
			     (set-font 3))
      (:graphics (setq graphics-font font)
		 (set-font 4))
      (:double-wide (setq double-wide-font font)
		    (set-font 5))
      (:narrow (setq narrow-font font)
	       (set-font 6))
      (:double-wide-graphics (setq double-wide-graphics-font font)
			     (set-font 6))
      (:double-height-graphics-top (setq
				     double-height-graphics-top-font
				     font)
				   (set-font 7))
      (:double-height-graphics-bottom (setq
					double-height-graphics-bottom-font
					font)
				      (set-font 8))    
      (otherwise (error nil "Unrecognized font type, ~S" type))
      )
    )
  (send self :update-new-line)
  )

(defparameter Idx-To-Keyword #(:standard :bold :double-height-top
			   :double-height-bottom :graphics
			   :double-wide :narrow :double-wide-graphics
			   :double-height-graphics-top
			   :double-height-graphics-bottom))

(defmethod (single-window-VT100 :get-font-type) (font)
  (let ((map (send VT100-pane :font-map)))
    (dotimes (i (length map))
      (when (eq font (aref map i))
	(return (aref Idx-To-Keyword i))
	)
      )
    )
		
)

;;;TI Code:
(defmethod (single-window-VT100 :get-bold-font) (font)
  ;; only one bold font implemented now
  (declare (ignore font))
  ;;RDA: Don't hardwire
  (send self :get-VT100-font :bold))

(DEFMETHOD (vt100-escape-sequence-mixin :INITIAL-SETUP) ()
  (setq bottom-of-scroll (tv:sheet-number-of-inside-lines vt100-pane))
  (setq save-font (send self :get-vt100-font :standard))
  (send vt100-pane :set-current-font save-font t)
  (setq shift-in save-font
	shift-out save-font
	long-term-save-font save-font))

(defmethod (single-window-VT100 :update-new-line) ()
  "For some reason, VT100-ESCAPE-SEQUENCE-MIXIN uses the NEW-LINE iv
   instead of LINE-HEIGHT.  This is here to keep it up-to-date."
  (setf new-line (send vt100-pane :line-height))
  )

;;;  Get the terminal width and height into the label

(defmethod (single-window-VT100 :after :change-of-size-or-margins)
	   (&rest ignore)
  "Add the width and height, in chars, to the label"
  (declare (ignore ignore))
  (send self :add-window-size-to-label)
  )

(defmethod (single-window-VT100 :after :new-connection) (&rest ignore)
  "Add the width and height, in chars, to the label"
  (declare (ignore ignore))
  (send self :add-window-size-to-label)
  )

(defmethod (single-window-VT100 :after :init) (&rest ignore)
  "Add the width and height, in chars, to the label, and setup fonts."
  (declare (ignore ignore))
  (send vt100-pane :set-font-map
	(list standard-font
	      bold-font
	      double-height-top-font
	      double-height-bottom-font
	      graphics-font
	      double-wide-font
	      narrow-font)
	)
  (send vt100-pane :set-current-font standard-font)
  (send self :update-new-line)			;what a hack
  (send self :add-window-size-to-label)
  )

(defmethod (single-window-VT100 :after :disconnect) (&rest ignore)
  "Add the width and height, in chars, to the label"
  (declare (ignore ignore))
  (send self :add-window-size-to-label)
  )

(defmethod (single-window-VT100 :add-window-size-to-label) ()
  "Add the width and height, in chars, to the label"
  (when (and tv:label (variable-boundp vt100-pane))
    (let* ((label-string (nth 5 tv:label))
	   (idx (find #\( label-string :test #'char= :from-end t))
	   new-string
	   )
      (multiple-value-bind (width height)
	  (send VT100-pane :size-in-characters)
	(setf new-string (format nil " (~D x ~D)" width height))
	(setq label-string (string-append
			     (if idx
				 (subseq label-string 0 (1- idx))
				 label-string)
			     new-string))
	(send self :set-label `(:font ,(nth 4 tv:label)
				    :string ,label-string))
	)
      )
    )
  )

;;;----------------------------------------------------------------------

;;; Pop up menu for commands

(defcommand (single-window-VT100 :set-VT100-size) ()
  '(:description
     "Set the number of lines and character width for the VT100 screen"
    :names ("Set Size")
    :keys ((#\NETWORK #\l) (#\NETWORK #\L)))
  (multiple-value-bind (*width* *height*)
      (send vt100-pane :size-in-characters)
    (declare (special *width* *height*))
    (catch 'aborting
      (tv:choose-variable-values
	`((*width* "Width"
		   :documentation
		   "Number of characters wide the VT100 screen should be"
		   :number)
	  (*height* "Height"
		    :documentation
		    "Number of lines hight the VT100 screen should be"
		    :number
		    )
	  )
	:label "VT100 Screen Size"
	:margin-choices '("Done"
			  ("Abort" (throw nil aborting)))
	)
      (send self :set-inside-size
	    (send vt100-pane :decode-character-width *width*)
	    (send vt100-pane :decode-character-height *height*))
      (setq *number-of-vt100-lines* *height*)
      (setq bottom-of-scroll
	    (tv:sheet-number-of-inside-lines vt100-pane))
      )
    )
  (when (and (null stream) (not ucl::preempting?))
    (send self :handle-prompt))
  )

(defmethod (vt100-screen :decode-character-width) (tv:spec)
  tv:(+ (* spec char-width) left-margin-size right-margin-size)
  )

(defmethod (vt100-screen :decode-character-height) (tv:spec)
  tv:(+ (* spec line-height) top-margin-size bottom-margin-size)
  )

(build-menu 'ucl-pop-up-VT100-telnet-menu 'single-window-VT100
  :item-list-order  
    '((:method telnet-frame :exit-command)		
      (:method telnet-frame :disconnect-command)
      (:method telnet-frame :interrupt-process-command)
      :send-answerback-command
      :reverse-video-command
      :reset-command
      :escape-processing-command
      (:method telnet-frame :quit-and-disconnect-command)
      (:method telnet-frame :status-command)
      (:method telnet-frame :abort-output-command)
      :column-command
      :truncate-command
      :set-VT100-size
      :network-help-command
     ))

(make-command command-menu-or-drag
  '(:description "Drag move this window if W:*DRAGGING-ENABLED?* is non-NIL, else pop up menu of VT100 commands"
    :keys #\mouse-M-1
    :definition drag-or-menu
    :arguments ('ucl-pop-up-VT100-telnet-menu)))

;;;Edited by Acuff                 31 Oct 87  11:43
;;;Edited by Acuff                 31 Oct 87  17:32
(defun drag-or-menu (menu)
  (if (and w:*dragging-enabled?* (functionp 'w:mouse-drag-window))
      (w:mouse-drag-window (send w:selected-window :superior))
      (ucl:pop-up-command-menu menu)
      )
  )

(make-command command-menu
  '(:description "Pop up menu of VT100 commands"
    :keys #\mouse-M-2
    :definition ucl:pop-up-command-menu
    :arguments ('ucl-pop-up-VT100-telnet-menu)))

;;;Edited by Acuff                 31 Oct 87  11:43
;;;Edited by Acuff                 31 Oct 87  17:32
(defmethod (single-window-vt100 :who-line-documentation-string) (&rest ignore)
  (if (and (boundp 'w:*dragging-enabled?*) w:*dragging-enabled?*)
      '(:mouse-r-1 "System Menu"
	:mouse-m-1 "Drag the VT100 window"
	:mouse-m-2 "Pop up menu of VT100 commands")
      '(:mouse-r-1 "System Menu"
	:mouse-m-1 "Pop up menu of VT100 commands" )
      )
  )

;;; Should make sure we're in a single, but life's too short...
;;;Edited by Acuff                 31 Oct 87  11:43
;;;Edited by Acuff                 31 Oct 87  17:32
(defmethod (vt100-screen :who-line-documentation-string) (&rest ignore)
  (if (and (boundp 'w:*dragging-enabled?*) w:*dragging-enabled?*)
      '(:mouse-r-1 "System Menu"
	:mouse-m-1 "Drag the window"
	:mouse-m-2 "Pop up menu of VT100 commands")
      '(:mouse-r-1 "System Menu"
	:mouse-m-1 "Pop up menu of VT100 commands" )
      )
  )

(build-command-table 'VT100-telnet-cmd-table 'single-window-VT100
  '((:method telnet-frame :exit-command)		
    (:method telnet-frame :disconnect-command)
    (:method telnet-frame :interrupt-process-command)
    :send-answerback-command
    :reverse-video-command
    :reset-command
    :escape-processing-command
    (:method telnet-frame :quit-and-disconnect-command)
    (:method telnet-frame :status-command)
    (:method telnet-frame :abort-output-command)
    :column-command
    :truncate-command
    :set-VT100-size
    :network-help-command
    (:method telnet-frame :clear-input-command)
    command-menu
    command-menu-or-drag
    )
  :init-options
  '(:name "VT100 & Telnet Commands"
    :documentation "The VT100 & Telnet commands."))

;;;----------------------------------------------------------------------

;;; Fixes to the LED flavor so that it works when it isn't exposed.

(unless (sys:mx-p)
;;;TI Code:
  (DEFMETHOD (led :DISPLAY-LED-SCREEN) ()
    "This function will set-up the LED lights to the initial SET-UP state"
    
    ;;RDA: Check for exposure first
    (when (send self :exposed-p)
      (send self ':string-out-explicit "ON LINE" 90. 20. nil nil
	    fonts:tr8b tv:alu-xor)
      (send self ':string-out-explicit "L1" 302. 20. nil nil
	    fonts:tr8b tv:alu-xor)
      (send self ':string-out-explicit "L2" 402. 20. nil nil
	    fonts:tr8b tv:alu-xor)
      (send self ':string-out-explicit "L3" 502. 20. nil nil
	    fonts:tr8b tv:alu-xor)
      (send self ':string-out-explicit "L4" 602. 20. nil nil
	    fonts:tr8b tv:alu-xor)
      (send self ':draw-filled-in-circle 110. 30. 5.)
      (send self ':draw-circle 305. 30. 5.)
      (send self ':draw-circle 405. 30. 5.)
      (send self ':draw-circle 505. 30. 5.)
      (send self ':draw-circle 605. 30. 5.)))
  
;;;TI Code:
  (DEFMETHOD (led :DRAW-CIRCLE) (x y radius)
    ;;RDA: Check for exposure first
    (when (send self :exposed-p)
      (send self ':draw-filled-in-circle x y radius)
      (send self ':draw-filled-in-circle x y (- radius 2) tv:alu-xor)))
  
;;;TI Code:
  (DEFMETHOD (led :RESET-LED-SCREEN) ()
    ;;RDA: Check for exposure first
    (when (send self :exposed-p)
      (send self ':draw-filled-in-circle 110. 30. 5.)
      (send self ':draw-circle 305. 30. 5.)
      (send self ':draw-circle 405. 30. 5.)
      (send self ':draw-circle 505. 30. 5.)
      (send self ':draw-circle 605. 30. 5.)))
  
;;;TI Code:
  (DEFMETHOD (led :CHANGE-LED) (led-list)
    ;;RDA: Check for exposure first
    (when (send self :exposed-p)
      (loop for led in led-list
	    do
	    (case led
	      (#\0
	       (send self ':reset-led-screen))
	      (#\1				       ;The first led light
	       (send self ':draw-filled-in-circle 305. 30. 5.))
	      (#\2				       ;The second led light, etc.
	       (send self ':draw-filled-in-circle 405. 30. 5.))
	      (#\3
	       (send self ':draw-filled-in-circle 505. 30. 5.))
	      (#\4
	       (send self ':draw-filled-in-circle 605. 30. 5.))))))
  )

;;;  Patches to VT100-FRAME to make this work

(DEFMETHOD (vt100-frame :AFTER :DISCONNECT) (&rest ignore)
  "After disconnect, change font and reset"
  (send vt100-pane ':set-current-font 
	(send self :get-vt100-font :standard))
  (send self ':reset) (send self ':setup-reset))

;;;----------------------------------------------------------------------

(tv:add-system-key #\v 'single-window-VT100
		   "A VT100 emulator in a single pane")

(TV::ADD-TO-SYSTEM-MENU-COLUMN :PROGRAMS "VT100 emulator"
			       '(TV:SELECT-OR-CREATE-WINDOW-OF-FLAVOR
				  'single-window-VT100)
			       "Emulate operation of a VT100 terminal.") 

(pushnew '("Vt100" :value single-window-VT100
	   :documentation "Login to a remote host using a Vt100 emulator")
	 tv:default-window-types-item-list
	 :test #'(lambda (x y) (string-equal (car x) (car y)))
	 )

(compile-flavor-methods single-window-VT100)

(provide 'single-window-vt100)
