;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-

(defvar *mouse-following-display-window-width* 400)
(defvar *Mouse-Following-Display-Window-Height*
	(+ 6 (* 3 (let ((font :Default))
		       (coerce-font font tv:default-screen)
		       (font-char-height font)
		  )
	     )
        )
)

(defflavor mouse-following-display-window
	   ()
	   (tv:window)
  (:Default-Init-Plist
    :Width  (min *mouse-following-display-window-width*
		 (send default-screen :Width)
	    )
    :Height (min *mouse-following-display-window-height*
		 (send tv:default-screen :Height)
	    )
    :Label nil
    :Save-Bits t
    :Deexposed-Typeout-Action :Permit
    :Blinker-P nil
    :More-P nil
    :Border-Margin-Width 4
  )
  (:Documentation "A window used by value-displaying-hollow-rectangular-blinker
 type blinkers onto which to print values.")
)

(defmethod (Mouse-Following-Display-Window :End-Of-Page-Exception)
	   (&rest ignore)
  "Just throws out when we hit the bottom of the page."
  (catch-error (throw :End-Of-Page :End-Of-Page) nil)
)

(defmethod (Mouse-Following-Display-Window :Size-To-Display) ()
  "Returns values for the size to pretend that the window is, given that we've
displayed something into it."
  (declare (values width height))
  (values (if (> cursor-y (sheet-inside-top))
	      (sheet-width self)
	      (+ (sheet-inside-left) cursor-x)
	  )
	  (+ (sheet-inside-top) cursor-y
	     (- (sheet-line-height self) (send self :Vsp))
	  )
  )
)

(defun draw-borders-for-different-width-and-height (alu the-width the-height)
  "Like draw borders only it parameterizes the width and height of the window.
   Lets us draw borders inside of the real edge of the window."
  (declare (:self-flavor borders-mixin))
  (prepare-color (self (send self :border-color))
    (sheet-force-access (self)
      (prepare-sheet (self)
        (dolist (border borders)
	  (and border
	     (neq border :zero)
	     (let ((left (second border))
		   (top (third border))
		   (right (fourth border))
		   (bottom (fifth border)))
	       (send (first border) self alu
			(if (minusp left) (+ left the-width) left)
			(if (minusp top) (+ top the-height) top)
			(if (plusp right) right (+ right the-width))
			(if (plusp bottom) bottom (+ bottom the-height))))))))))

(defmethod (Mouse-Following-Display-Window :Draw-In-Pseudo-Margins) ()
  "Draws in borders at the points that it wants you to see.  Returns the width
   and height of the window that wou should display."
  (declare (values new-width new-height))
  (multiple-value-bind (right bottom) (send self :Size-To-Display)
    (Draw-Borders-For-Different-Width-And-Height alu-seta right bottom)
    (values right bottom)
  )
)

;-------------------------------------------------------------------------------

(defflavor value-displaying-hollow-rectangular-blinker
	   ((get-value-function 'Show-Value-If-Bound-Symbol-P)
	    (display-window (make-instance 'Mouse-Following-Display-Window))
	    (drawn-p nil)
	    (bit-array nil)
	    (drawn-args nil)
	   )
	   (hollow-rectangular-blinker)
  :Initable-Instance-Variables
  :Gettable-Instance-Variables
  :Settable-Instance-Variables
  (:Documentation "A flavor of blinker that, as it tracks the mouse, uses
 get-value-function to determine whether it should pop up a little window
 showing a value.")
)

(defmethod (Value-Displaying-Hollow-Rectangular-Blinker :After :Init) (ignore)
  "Sets the bit array to be the same size and shape as that of
   the display window."
  (setq bit-array
	(make-array (array-dimensions (send display-window :screen-array))
		    :Type (array-type (send display-window :screen-array))
	)
  )
)

(defmethod (Value-Displaying-Hollow-Rectangular-Blinker :After :Blink) ()
  "Draw/Undraw self if we get a blink.  This happens if someone else goes and
prepares the sheet."
  (if phase
      (if drawn-p
	  nil
	  (send self :Draw-Self)
      )
      (if drawn-p
	  (send self :Restore-Picture-At-Point)
	  nil
      )
  )
)

(defmethod (Value-Displaying-Hollow-Rectangular-Blinker :Draw-Self) ()
  (multiple-value-bind (show-p value format-string)
      (funcall get-value-function sheet)
    (cond (drawn-p
	   (destructuring-bind
	     (the-x-pos the-y-pos the-left the-top
	      the-value the-width the-height)
	       drawn-args
	     (ignore the-x-pos the-y-pos the-left
		     the-top the-width the-height
	     )
	     (if (eq value the-value)
		 nil
		 (progn
		  (send self :restore-picture-at-point)
		  (if show-p
		      (send self :Show-Value-At-Point
			    value format-string
		      )
		      nil
		  )
		 )
	     )
	   )
	  )
	  ((not show-p) nil)
	  (t (send self :show-value-at-point value format-string))
    )
  )
)

(defwhopper (Value-Displaying-Hollow-Rectangular-Blinker :set-visibility) (to)
"Checks when we get our visibility set so that we expose the
window as appropriate."
  (let ((result (continue-whopper to)))
       (cond ((member to '(nil :off))
	      (if drawn-p (send self :restore-picture-at-point) nil)
	     )
	     ((member to '(:on t)) (send self :Draw-Self))
	     (t nil)
       )
       result
  )
)

(defmethod (Value-Displaying-Hollow-Rectangular-Blinker :restore-picture-at-point)
	   ()
"Restores the oroginal bitmap to underneath where we have displayed the window."
  (destructuring-bind (the-x-pos the-y-pos left top value the-width the-height)
		      drawn-args
    (ignore the-x-pos the-y-pos value)
    (prepare-sheet (sheet)
      (bitblt alu-seta
	      the-width the-height;(+ 12 the-height)
	      bit-array 0 0
	      (or (send sheet :screen-Array)
		  (send sheet :bit-Array)
	      )
	      left top
      )
    )
  )
  (setq drawn-p nil)
  (setq drawn-args nil)
)

(defmethod (Value-Displaying-Hollow-Rectangular-Blinker :show-value-at-point)
	   (value format-string)
  "Shoes a value in a little window near the current point."
  (if drawn-p (send self :restore-picture-at-point) nil)
  (sheet-clear display-window t) ;; resets cursor to home too.  Erase margins.
  (prepare-sheet (sheet)
    (multiple-value-bind (the-x-pos the-y-pos the-width the-height)
	(without-interrupts
	  ;;; Read these once so no confusion results.
	  (values x-pos y-pos (min (send display-window :width)
				   (array-dimension bit-array 1)
			      )
		  (min (send display-window :Height)
		       (array-dimension bit-array 0)
		  )
	  )
	)
      ;;; Draw text onto display window safely.
      (catch :End-Of-Page
	(catch-error (format display-window
			     (or (and (stringp format-string) format-string)
				 "~S"
			     )
			     value
		     )
		     nil
        )
      )
      (multiple-value-bind (max-width max-height)
	  ;;; Size of little window.
	  (send display-window :Draw-In-Pseudo-Margins)
	(setq max-width (min max-width (send sheet :Width)))
	(setq max-height (min max-height (send sheet :Height)))
	(multiple-value-bind (left top)
	    ;;; left and top of place to display window
	    (send self :Good-Position max-width max-height)
	  ;;; Save stuff on sheet into bit-array
	  ;;; Don't understand these +12 fudges.
	  (bitblt alu-seta the-width the-height ;(+ 12 the-height)
		  (or (send sheet :screen-Array) (send sheet :bit-Array))
		  left top bit-array 0 0
	  )
	  ;;; Write display-window contents to sheet
	  (bitblt alu-seta max-width max-height
		  (send display-window :bit-Array) 0 0
		  (or (send sheet :screen-Array) (send sheet :bit-Array))
		  left top
	  )
	  ;;; Save what we drew so we can undo it easily.
	  (setq drawn-args (list the-x-pos the-y-pos left top value
				 max-width max-height
			   )
	  )
	  (setq drawn-p t)
	)
      )
    )
  )
)

(defmethod (Value-Displaying-Hollow-Rectangular-Blinker :good-position)
	   (the-width the-height)
  "Returns left and top positions for us to display the little window so
   as not to bump into anything."
  (declare (values left top))
  (Position-Rectangle-Next-To-Rectangle
    (list 0 0 the-width the-height)
    (list 0 0 (send sheet :Width) (send sheet :Height))
    '(:Above :Below :Left :Right)
    x-pos y-pos (+ x-pos width) (+ y-pos height)
  )
)

(defvar *use-show-value-blinker-in-inspector-p* nil)

(defun symbol-and-boundp (thing)
  "Is true if thing is a bound non-keyword symbol."
  (if (and *use-show-value-blinker-in-inspector-p*
	   thing (symbolp thing) (not (keywordp thing)) (boundp thing)
      )
      (values t (symbol-value thing))
      (values nil nil)
  )
)

(defun-method show-value-if-thing
	      Value-Displaying-Hollow-Rectangular-Blinker
	      (window predicate)
  "Shows the value in the window if the predicate is satisfied.
   The predicate must return (values show-p value-to-show-if-show-p)."
  (declare (values show-p value))
  (if (typep window 'instance)
      (multiple-value-bind (item type)
	(send window :Send-If-Handles :Mouse-Sensitive-Item
	      (+ x-pos (floor width 2)) (+ y-pos (floor height 2))
        )
	(ignore type)
	(let ((value (if (and (consp item) (equal :Item1 (first item)))
			 (third item)
			 item
		     )
	      )
	     )
	     (let ((value (if (typep value 'inspection-data)
			      (or (send value :Send-If-Handles
					:Middle-Button-Result
			          )
				  nil
			      )
			      value
			  )
		   )
		  )
	          (let ((value (if (typep value 'locative)
				   (or (catch-error (first value) nil) value)
				   value
			       )
			)
		       )
		       (multiple-value-bind (results error-p)
			   (values
			     (multiple-value-list (funcall predicate value))
			     nil
			   )
			 (if error-p
			     (values nil nil)
			     (values-list results)
			 )
		       )
		  )
	     )
	)
      )
      nil
  )
)

(defun-method show-value-if-bound-symbol-p
	      Value-Displaying-Hollow-Rectangular-Blinker (window)
  "Trackingly show values on this window if they are bound symbols."
  (declare (values show-p value))
  (show-value-if-thing window 'symbol-and-boundp)
)


(defun position-rectangle-next-to-rectangle
       (rectangle superior position left top right bottom)
"Move Rectangle near the specified left top right bottom.
  Try hard to put it at POSITION.
POSITION is a keyword which indicates where to put the window.
	It can be one of :ABOVE, :BELOW, :LEFT or :RIGHT or a
	list of any these keywords.  If position is a single keyword
	and that position fails then the possibilities in the
	window-positioning-preference-list are also tried.
LEFT TOP RIGHT BOTTOM specify a rectangle using the inside coordinates
	of WINDOW's superior."
  (let* ((rectangle-width (- (third rectangle) (first rectangle)))
	 (rectangle-height (- (fourth rectangle) (second rectangle)))
         (it-fits nil)
         (superior-width  (- (third superior) (first superior)))
         (superior-height (- (fourth superior) (second superior)))
	 (vsp 1)
         ;; Compute the amount of room around the rectangle.  All values use
         ;; inside coordinates.
         (room-left   left)
         (room-top    top )
         (room-right  (- superior-width  right))
         (room-bottom (- superior-height bottom))
	 new-rectangle-x new-rectangle-y)
  (incf rectangle-height vsp)
  (incf rectangle-width  vsp)
  ;; Try each possibility in the preference list.  The first element in the
  ;; preference list is the same as the position argument.
  (loop for possible-position
	in (if (listp position)
	       position
	       ;;ELSE
	       (cadr (assoc position w:window-positioning-preference-list
			    :test #'eq)))
        do 
        (case possible-position
              (:above (when (<= rectangle-height room-top)
                        ;; The rectangle fits above the rectangle.
                        (setq it-fits :above
                              new-rectangle-y (- top rectangle-height)
                              new-rectangle-x
			        (cond ((<= (+ left rectangle-width)
					   superior-width) left)
				      ((not (minusp (- right rectangle-width)))
				       (+ (- right rectangle-width) vsp))
				      (t
				       0)))))
              (:left (when (<= rectangle-width room-left)
                       (setq it-fits :left
                             new-rectangle-x (- left rectangle-width)
                             new-rectangle-y
			     (cond ((<= (+ top rectangle-height)
					superior-height) top)
				   ((not (minusp (- bottom rectangle-height)))
				    (+ (- bottom rectangle-height) vsp))
				   (t
				    0)))))
              (:right (when (<= rectangle-width room-right)
                        (setq it-fits :right
                              new-rectangle-x (+ right vsp)
                              new-rectangle-y
			      (cond ((<= (+ top rectangle-height)
					 superior-height) top)
				    ((not (minusp (- bottom rectangle-height)))
				     (+ (- bottom rectangle-height) vsp))
				    (t
				     0)))))
              (:below (when (<= rectangle-height room-bottom)
                        ;; The rectangle fits below the rectangle.
                        (setq it-fits :below
                              new-rectangle-y (+ bottom vsp)
                              new-rectangle-x
			      (cond ((<= (+ left rectangle-width)
					 superior-width) left)
				    ((not (minusp (- right rectangle-width)))
				     (+ (- right rectangle-width) vsp))
				    (t
				     0))))))
        until it-fits)
  (when (not it-fits)
    (if (or (eq position :below) (eq position :right))
	(setq new-rectangle-x 0
	      new-rectangle-y 0))
        (setq new-rectangle-x 0
              new-rectangle-y
	        (max bottom (- superior-height rectangle-height))))
  (Make-Sure-Rectangle-Is-Inside-Superior
    superior rectangle new-rectangle-x new-rectangle-y)))


(defun make-sure-rectangle-is-inside-superior (superior rectangle x y)
  "Makes dead sure that the rectangle is inside the bounds of the
   superior's rectangle."
  (destructuring-bind (sl st sr sb) superior
    (destructuring-bind (rl rt rr rb) rectangle
      (if (< (+ rl x) sl)
	  (setq x sl)
	  nil
      )
      (if (> (+ rr x) sr)
	  (setq x (- sr (- rr rl)))
	  nil
      )
      (if (< (+ rt y) st)
	  (setq y st)
	  nil
      )
      (if (> (+ rb y) sb)
	  (setq y (- sb (- rb rt)))
	  nil
      )
      (values (max x 0) (max 0 y))
    )
  )
)

;-------------------------------------------------------------------------------

(defun debug-source-for-function (function)
"Returns the source code debugging source for Function if it can."
  (catch-error
    (let ((name (function-name function)))
         (if (and (consp name)
		  (equal (first name) :internal)
	     )
	     (debug-source-for-function (second name))
	     (let ((dbis (sys:get-debug-info-struct function t)))
	          (let ((encapsulated-p
			  (not (eq (sys:get-debug-info-struct function) dbis))
			)
		       )
		       (or (if encapsulated-p
			       (getf (sys:dbis-plist dbis)
				     :Source-Code-Debugger-Numbered-Source
			       )
			       (sys:function-spec-get
				 name :Source-Code-Debugger-Numbered-Source
			       )
			   )
			   ;;; Added by JPR to support eh Source command stuff.
			   (if encapsulated-p
			       (getf (sys:dbis-plist dbis)
				     :zmacs-source
			       )
			       (sys:function-spec-get
				 name :zmacs-source
			       )
			   )
		       )
		  )
	     )
	 )
    )
    nil
  )
)


(defun read-source-file-and-read-definition (function source)
  "Reads the source definition named Source for function Function into
   zmacs and then reads the definition form out again, stashing the form
   in the :zmacs-source property of the function.
  "
  (let ((form nil)
	(error-p nil)
       )
       (catch 'zwei:zwei-command-loop
	 (let ((frame (if Zwei::*All-Zmacs-Windows*
			  (Send (First Zwei::*All-Zmacs-Windows*)
				:Superior
			  )
			  (zwei:find-or-create-idle-zmacs-window)
		      )
	       )
	      )
	      (funcall (send frame :editor-closure)
		#'(lambda ()
		    (zwei:edit-definition-1 (function-name function) t ""
					    (first source) 'defun
		    )
		    (with-open-stream
		      (stream (zwei:interval-stream
				(zwei:point)
				(zwei:forward-sexp (zwei:point))
			      )
		      )
		      (multiple-value-bind (f err)
			  (catch-error (values (read stream)) nil)
			(setq form f)
			(setq error-p err)
			(close stream)
		      )
		    )
		  )
	      )
	 )
	 (if error-p
	     (beep)
	     (sys:function-spec-putprop
	       (function-name function) form :zmacs-source
	     )
	 )
       )
  )
)

(defun eh:comw-source (sg object frame)
  "Reads the source definition of the current fef into zmacs and, using that,
   displays it in the inspect pane."
  (ignore object)
  (eh:with-frame-environment (sg frame)
    (ignore eh:start eh:end eh:sp)
    (let ((source (zwei:source-file-names (function-name function) 'defun)))
         (if source
	     (progn (if (sys:function-spec-get
			  (function-name function) :zmacs-source
			)
			nil
			(read-source-file-and-read-definition function source)
		    )
		    (send eh:*window-debugger* :inspect-object
			  (tv:allocate-data 'eh:show-source-code function)
		    )
	     )
	     (beep)
	 )
    )
  )
)

(defcommand eh:Comw-Edit-Source nil
  '(:Description
     "Show the source of the current fef in the inspect pane if possible."
    :names "Source"
    :keys (#\S-Sh-S)
   )
   (unwind-protect
       (eh:comw-source eh:*error-sg* eh:*error-object* eh:*current-frame*)
     (progn (send eh:*window-debugger* :set-who-line-doc-string-overide nil)
	    (send eh:*window-debugger* :handle-prompt)
     )
   )
)

(defun eh:provide-edit-source-command ()
  "Install the Source command."
  (tv:install-command '(nil nil ((eh:Comw-Edit-Source t))))
)

;-------------------------------------------------------------------------------

(defvar eh:*use-show-value-blinker-in-debugger-p* nil)

(defun bound-in-fef-p (thing)
  "Is true if thing is the name of some sort of variable bound in the current
   fef in the debugger.  Thing can be any sort of local, special, lexical or
   instance variable reference."
  (declare (values show-p value))
  (if (and eh:*use-show-value-blinker-in-debugger-p*
	   thing (symbolp thing) (not (keywordp thing))
      )
      (let ((process (send (send (tv:sheet-superior (send self :sheet))
				 :tv-current-window
			   )
			   :process
		     )
	    )
	   )
	   (let ((sg (send process :initial-stack-group))
		 (eh-sg (send process :stack-group))
		)
		(let ((frame
			(symeval-in-stack-group 'eh:*current-frame* eh-sg)
		      )
		     )
		     (if (and eh-sg frame sg (numberp frame))
			 (let ((bindings
				 (eh:get-all-bindings-for-fef sg frame)
			       )
			      )
			      (let ((entry (assoc thing bindings :test #'eq)))
				   (if entry
				       (values t (second entry))
				       (values nil nil)
				   )
			      )
			 )
			 (values nil nil)
		     )
		)
	   )
      )
      (values nil nil)
  )
)

(defun-method show-value-if-bound-in-fef-p
	      Value-Displaying-Hollow-Rectangular-Blinker (window)
  "Is true if thing is the name of some sort of variable bound in the current
   fef in the debugger.  Thing can be any sort of local, special, lexical or
   instance variable reference."
  (declare (values show-p value))
  (show-value-if-thing window 'bound-in-fef-p)
)

;-------------------------------------------------------------------------------
;===============================================================================
;-------------------------------------------------------------------------------

;;; Patches.

(defvar *use-value-displaying-blinker-in-text-scroll-windows-p* nil)

(defmethod (mouse-sensitive-text-scroll-window-without-click :after :init)
           (ignore)
  "Makes text scroll windows have value-displaying blinkers."
  (setq item-blinker
	(if *Use-Value-Displaying-Blinker-In-Text-Scroll-Windows-P*
	    (make-blinker self 'value-displaying-hollow-rectangular-blinker
			     :visibility nil
	    )
	    (make-blinker self 'hollow-rectangular-blinker
			  :visibility nil
	    )
	)
  )
)

(defmethod (eh:debugger-inspect-pane :after :init) (ignore)
  "Makes debugger inspectors have value-displaying blinkers."
  (if (and eh:*use-show-value-blinker-in-debugger-p*
	   (typep item-blinker 'Value-Displaying-Hollow-Rectangular-Blinker)
      )
      (send item-blinker :Set-Get-Value-Function 'show-value-if-bound-in-fef-p)
      nil
  )
)

(defmethod (eh:new-debugger-inspect-pane :after :init) (ignore)
  "Makes debugger inspectors have value-displaying blinkers."
  (if (and eh:*use-show-value-blinker-in-debugger-p*
	   (typep item-blinker 'Value-Displaying-Hollow-Rectangular-Blinker)
      )
      (send item-blinker :Set-Get-Value-Function 'show-value-if-bound-in-fef-p)
      nil
  )
)


;-------------------------------------------------------------------------------

(export '(value-displaying-hollow-rectangular-blinker
	  show-value-if-thing
	 )
	'tv
)