;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(CPTFONT HL10B TR10BI TR10I); Patch-file:T -*-
;;;
;;;===============================================================================
;;;                                    RESTRICTED RIGHTS LEGEND 
;;; Use,  duplication, or  disclosure  by  the  Government is subject to restrictions
;;; as set forth in subdivision (c)(1)(ii) of the Rights in Technical Data and
;;; Computer Software clause at 52.227-7013. 
;;;
;;; TEXAS INSTRUMENTS INCORPORATED, P.O. BOX 2909 AUSTIN, TEXAS 78769  
;;; Copyright (C) 1984, Texas Instruments Incorporated. All rights reserved.
;;;===============================================================================

;; Blank the screen after SCREEN-SAVER-TIME-DELAY (default 20) minutes

;;;
;;; Change history:
;;;
;;;  Date	Author	Description
;;; -------------------------------------------------------------------------------------
;;;;    9/06/88	MAY	Fixed problem when user ended rotate-sphere before (build) done and ary's completed
;;; 08-30-88         MAY	moved :clear-input in screen-saver  Fixes SPR 8675.  Deleted (listen) from BALLS.
;;;  2/09/88    TWE	Changed twinkle to use circles instead of rectangles.
;;;  1/25/88    HS      Modified attraction code so that is doesn't cons as profusely as
;;;                     it used to (mainly by using short floats).
;;;  1/14/88    TWE	Hacked up the attraction code so that it works even though we
;;;			don't have the balls-16 font.
;;; 09/25/87	LGO	Set inhibit-who-line instead of deexposing the who-line
;;; 06/29/87   PMH    Many changes were made for production use:
;;;                   Cooperating process made: one to run a background activity
;;;                     function and another to poll, via a wait function, 
;;;                     for foreground activity.
;;;                   :OUTPUT-HOLD-EXCEPTION was added to the blackout window to
;;;                     arrest the activity process/function.
;;;                   Removed queueing messages to the LISP-Listener; they are now
;;;                     on the pending-notifications list.
;;;                   The variable *screen-saver-hacks* controls what activity
;;;                     function will be executed.  The default is the QIX function;
;;;                     setting this variable to NIL leaves the screen blank.
;;; 05/12/87	LGO	Modified to ensure two blackout process aren't started
;;; 05/11/87	LGO	Modified to queue notifications, and print them on the 
;;;			initial-lisp-listener AFTER un-blanking the screen.
;;;			This gets around MORE processing problems in the Lisp Listener.
;;; 04/30/87	LGO	Modified to move the who-line documentation back and forth
;;; 04/27/87	LGO	Modified to forward notifications to the initial-lisp-listener
;;; 03/24/87	LGO	Converted for Explorer release 3.0
;;; 09/17/84	LGO	Created

#|
        The SCREEN-SAVER utility
     
	When the monitor is left unattended for long periods of
	time, like overnight, it is best to turn down the brightness
	so the phosphors on the screen don't burn needlessly.
	The screen saver utility is an automatic tool which also helps
	to preserve the phosphors.  This feature makes the screen
	go black if there has not been any keyboard or mouse interaction
	after a designated period of time.

	After a certain period of time with no user interaction the screen
	saver utility is invoked.  A special blackout screen is exposed
	And the utility waits either for some other application to expose
	their window or for the user to move the mouse or press a key.
	If either case is true the screen saver utility terminates and the
	previous application is resumed.  As an additional feature, while
	the screen saver is awaiting one of these two events, a separate
	process is started which can be programmed to do some user specified
	task; the default is to draw the QIX pattern.  When the screen saver
	utility is terminated it automatically kills this associated process.

	While the screen saver utiltiy is running notifications are queued
	up and not displayed until the user resumes interaction with the
	keyboard or mouse.  If a notification has been queued, you will
	receive a message that instructs you to press TERM-N to see the
	messages.

	To restore the screen and resume your application, just move the
	mouse or press any key.  If you resume by pressing a key, the
	keystroke will be processed by the screen saver utility and is
	not passed on to the original application.  This includes
	asynchronous event keys like SYSTEM or TERMINAL.  Note that if the
        application screen did not have a bit-save array, then when its
        screen is re-exposed it will be blank and that information is lost.


	TV:*SCREEN-SAVER-TIME-DELAY*				VARIABLE

	This is a positive integer that says how long the interval should
	be, in minutes, between any keyboard/mouse interaction and invoking
	the screen save utility.  If the value is NIL then the screen saver
	will never be invoked.  The default value is 20.  This variable is
	acessible through the PROFILE utility.

	This variable is also checked while the utility is running and
	terminates if it is set to NIL.  This allows programs to do a
	LET-GLOBALLY if they want to temporarily inhibit this facility.

	TV:*SCREEN-SAVER-HACKS*					VARIABLE

	The value of this variable is T, NIL, or a function that takes one
	argument, the blackout screen window.  The function is run in a separate
	process while the screen saver utility is active.  It is run at
	a very low priority with a small quantum so it won't interfere with
	any background activity.  The default for this variable is T,
	which runs the QIX function.  Setting this variable to NIL means
	that no special function is run, and thus the screen remains
	completely black.

	Should you decide to write your own function, please keep in mind
	this utilities original intent, keeping the phosphors from excesive
	burning.  Other things to keep in mind are that consing should be
	kept to an absolute minimum; also since the process may be killed
	without notice, unwind-protects may be needed to clean up your
	calculations.
|#

;;; If you want to run an application and guarantee that the SS utility
;;; does not interrupt it, do a let globally of the following variable to NIL.
(DEFVAR *screen-saver-time-delay* 20
  "The number of minutes to wait before blacking out the screen.
 NIL means disable screen-saver feature.")
;;; For compatibility let the old time delay name be recognized
(forward-value-cell 'black-screen-time-delay '*screen-saver-time-delay*)

(DEFVAR *screen-saver-time-delay-saved* nil)

;;; Make sure screen saver is shut down at critical times.
(Add-initialization "Inhibit Screen Saver"
		    '(setq *screen-saver-time-delay-saved* *screen-saver-time-delay*
			   *screen-saver-time-delay* nil)
		    :before-cold)
(Add-initialization "Inhibit Screen Saver"
		    '(setq *screen-saver-time-delay-saved* *screen-saver-time-delay*
			   *screen-saver-time-delay* nil)
		    :full-gc)
;;; Make sure screen saver gets re-enabled.
(Add-initialization "Enable Screen Saver"
		    '(setq  *screen-saver-time-delay* *screen-saver-time-delay-saved*
			    *the-screen-is-black* nil)
		    :cold)
(Add-initialization "Enable Screen Saver"
		    '(setq  *screen-saver-time-delay* *screen-saver-time-delay-saved*
			    *the-screen-is-black* nil)
		    :after-full-gc)

;;; The following profile form should be added to SYS:PROFILE;VARIABLES.LISP
(PROFILE:DEFINE-PROFILE-VARIABLE *SCREEN-SAVER-TIME-DELAY* (:IMPORTANT)
     :CVV-TYPE :POSITIVE-FIXNUM-OR-NIL)

(DEFVAR *screen-saver-hacks* 'random-screen-save-hack
  "This variable identifies a function to run while the screen is black.
   The function takes one argument which is the blackout-screen-window.
   A NIL value means the screen is left totally black. A value of T means run QIX.")

(defvar *screen-saver-hacks-list* ()
  "List of screen-saver functions that could be called.  Used by Random-Screen-Save-Hack.")

(DEFVAR *the-screen-is-black* nil
  "This variable is used as a lock to determine if a
   screen-saver process is already running.")

(defun random-screen-save-hack (window)
  "Selects a screen saver hack at random from the list of available ones."
  (funcall (nth (random (length *screen-saver-hacks-list*))
		 *screen-saver-hacks-list*)
	   window))


;;;
;;; Hacking to get safe location in memory to impersonate as the run-bar location
;;; 

(defvar *fake-run-bar-location*
	(UNLESS (si:mx-p)
	  (flet ((find-area (name) (position name sys:area-list))
		 (find-static-area ()
				   (loop for area-num from 0 below (length sys:area-list)
					 when (= 9. (LDB %%region-space-type
							 (AREF #'area-region-bits area-num)))
					 return area-num)))
	    (let ((fake-run-bar-area (or (find-area 'fake-run-bar-area)
					 (find-static-area)
					 (progn (make-area :name 'fake-run-bar-area :gc :static)
						(find-area 'fake-run-bar-area)))))
	      (%pointer (locf (aref (make-array 1 :area fake-run-bar-area) 0))))))
      "We use this as the run bar location when we don't wish to illuminate the real thing.")

(defparameter real-run-bar-location
	      (AND (BOUNDP 'sys:really-run-light) sys:really-run-light)
  "This is the real location of the run bar.")

;;  Use the execution environment's value for SYS:REALLY-RUN-LIGHT, not the compile environment's...
;;(EVAL-WHEN (EVAL LOAD)
;;  (SETF real-run-bar-location sys:really-run-light))

(defun erase-real-run-bar ()
  "Erase the real run bar.  To be used while the real run bar is being shadowed with the dummy location *fake-run-bar-location*."
  (UNLESS (si:mx-p)
    (LET ((VAL 0))
      (IF (AND (TYPEP real-run-bar-location 'fixnum) (/= real-run-bar-location 0))
	  (PROGN 
	    (%P-DPB VAL #o0020 real-run-bar-location)
	    (%P-DPB VAL #o2020 real-run-bar-location))
	;; else...
	(CERROR "forget it" "run bar at zero")))))

(defun set-fake-run-light ()
  "Impersonate the real run bar so the real thing doesn't get lit up."
  (UNLESS (si:mx-p)
    (setq sys:really-run-light *fake-run-bar-location*)
    (erase-real-run-bar)))

(defun maybe-start-screen-saver ()
  "Starts the screen saver if it's appropriate."
  (WHEN (AND *screen-saver-time-delay* tv:kbd-last-activity-time
	     (>= (TRUNCATE
		   (TIME-DIFFERENCE (TIME) tv:kbd-last-activity-time) 3600.)
		 *screen-saver-time-delay*)
	     (NOT *the-screen-is-black*))
    (SETQ *the-screen-is-black* t) ;; Set here to ensure two blackout processes aren't started
    (PROCESS-RUN-FUNCTION "Screen-Saver" 'screen-saver)))

;; This could run any place that executes once a minute.
;; We don't have a seperate process, because that's too much overhead...
(DEFMETHOD (who-line-file-sheet :before :update) ()
  "Makes the screen black after a period if inactivity"
  (maybe-start-screen-saver))

;;;Use the timer-queue facility 
(when (fboundp 'si:add-timer-queue-entry)
  (si:add-timer-queue-entry "now" '(:forever 60.) "Screen Saver" 'maybe-start-screen-saver)
  (when (fdefinedp '(:method who-line-file-sheet :after :update))
    (undefmethod (who-line-file-sheet :after :update))))
  
;; Full screen window with no save array for covering up the screen
(DEFFLAVOR blackout-screen-window
	   ()
	   (tv:not-externally-selectable-mixin
	    tv:kbd-mouse-buttons-mixin
	    tv:stream-mixin
	    tv:graphics-mixin
	    tv:select-mixin
	    tv:minimum-window)
  (:default-init-plist
    :reverse-video-p nil
    :blinker-p nil
    :save-bits nil)
  (:documentation :combination "Window to use while blanking the screen"))

(DEFMETHOD (blackout-screen-window :mouse-moves) (&rest IGNORE)
  (SEND SELF :FORCE-KBD-INPUT 0))

;; Stop the screen saver hack process.
(DEFMETHOD (blackout-screen-window :output-hold-exception)()
  (send current-process :arrest-reason :output-hold))

;; Save notifications for later display.
;; Press Term-N to see these messages.
(DEFMETHOD (blackout-screen-window :print-notification) (time string window-of-interest)
  (PUSH (LIST TIME STRING WINDOW-OF-INTEREST) PENDING-NOTIFICATIONS))

(COMPILE-FLAVOR-METHODS blackout-screen-window who-line-file-sheet)

;; Make the screen the first time it is needed.
(DEFVAR blackout-screen-window nil)

;;; This is the function that controls the screen saving activity
;;; 1) blank the screen [expose the blackout window]
;;; 2) deexpose the who line
;;; 3) kick off the activity process
;;; 4) wait for user interaction
;;; 5) kill off the activity process
;;; 6) deexpose the blackout window
;;; 7) notify user if notifications are pending.
(DEFUN screen-saver (&aux bow-mode screen-saver-hack-process)
  "Black out the screen until a character is typed."
  (unless blackout-screen-window
    (setq blackout-screen-window (MAKE-INSTANCE 'blackout-screen-window)))
  ;; Disable the abort, system and other such keys
  (LET ((save-kbd-intercepted-characters kbd-intercepted-characters)
	(save-kbd-global-asynchronous-characters kbd-global-asynchronous-characters)
	direct-drawing-mode-function)
    (UNWIND-PROTECT
	(PROGN
	  (SETQ inhibit-who-line (NOT (si:mx-p)))
	  (set-fake-run-light)		   ;;Lie to the scheduler about where the run-bar is
	  (SETQ *the-screen-is-black* t)
	  (SETQ kbd-intercepted-characters nil)
	  (SETQ kbd-global-asynchronous-characters nil)
	  (SETQ bow-mode *current-screen-color*)
	  (WHEN (si:mx-p)	    
	    (SETF direct-drawing-mode-function
		  (SYMBOL-FUNCTION (FIND-SYMBOL "SET-DIRECT-DRAWING-MODE" "MAC")))
	    (FUNCALL direct-drawing-mode-function :off)
	    (FUNCALL direct-drawing-mode-function :on))
	  (send blackout-screen-window :select)
;;;	  (SEND blackout-screen-window :clear-input)  ;; may 08-30-88
	  (unless (send blackout-screen-window :lock)
	    (UNLESS (si:mx-p)
	      (send WHO-LINE-SCREEN :clear-screen))
	    (setf screen-saver-hack-process
		(process-run-function  '(:name "Screen-Saver-Hack" :priority -200 :quantum 10)
				      #'(lambda (w &aux (*terminal-io* w))
					  (send w :select)
					  (white-on-black) 
					  (mouse-set-blinker-definition
					    :character 0 0 :off
					    :set-character mouse-glyph-small-dot)
					  (if *screen-saver-hacks*
					      (funcall (if (eq t *screen-saver-hacks*)
							   #'screen-saver-hacks
							    *screen-saver-hacks*)
						       w)
					      (process-wait "Blackout"
							    #'(lambda()
								*screen-saver-hacks*))
						))
				      blackout-screen-window))
	    (sleep 1)
	    ;; clear out io-buffer before waiting so that mouse-wakeup in
	    ;; select-mixin :select which causes a :mouse-moves, which
	    ;; we respond to by stuffing a 0 in keybd buffer, does NOT
	    ;; get noticed here. We only want USER mouse or keyboard blips
	    ;; to stop the screen-saver process.
	    (SEND blackout-screen-window :clear-input) ;; may 08-30-88
	    (process-wait "Screen-Saver"
			  #'(lambda (w p)
			      (erase-real-run-bar) ;;keep erasing the run bar: someone is writing it!
			      (or (SEND w :mouse-or-kbd-tyi-no-hang)
				  ;; test to see if process is arrested
				  ;; this happens on the MORE exception
				  (not (send p :runnable-p))
				  ;; test to see if Screen Saver has been turned off
				  (not *screen-saver-time-delay*)
				  ;; If our default activity runs into trouble
				  ;;   (wait funtion is not true) then exit
				  (and (eq t *screen-saver-hacks*)
				       (not (eq #'true (send p :wait-function))))
					   ))
			  blackout-screen-window screen-saver-hack-process))
	  )
      (UNLESS (si:mx-p)
	(SETQ inhibit-who-line nil)
	(send TV:WHO-LINE-SCREEN :refresh))
      (SETQ kbd-intercepted-characters save-kbd-intercepted-characters
	    kbd-global-asynchronous-characters save-kbd-global-asynchronous-characters
	    tv:kbd-last-activity-time (TIME))
      (when screen-saver-hack-process
	(WHEN bow-mode (black-on-white))
	(unless (send screen-saver-hack-process :arrest-reasons)
	  (send screen-saver-hack-process :arrest-reason :output-hold))
	(send blackout-screen-window :deselect)
	(send screen-saver-hack-process :kill)
	(send screen-saver-hack-process :set-priority 1)
	(send screen-saver-hack-process :revoke-arrest-reason :output-hold)
	)
      (setq mouse-reconsider t)
      (send blackout-screen-window :deexpose)
      (SETQ *the-screen-is-black* nil)
      (setq sys:really-run-light real-run-bar-location)
      (WHEN (and pending-notifications selected-window)
	(tv:notify nil "There ~[are~;is~:;are~] ~:*~d notification~:p pending.  ~
                          Press TERM N to see notifications." 
		   (length pending-notifications)))
      )))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; QIX - the default screen saver function
;;; 

(DEFUN screen-saver-hacks (window)
  "This function runs the QIX display in the blackout window."
  (qix 100 window))

(pushnew 'screen-saver-hacks *screen-saver-hacks-list*)
  
;;; I changed QIX to not CONS but I needed to add these two variables
;;; The QIX list is made into a circular list so I need to track the
;;; length separately.
(defvar qix-list nil)
(defvar qix-list-length nil)

(defun qix (&optional (length 100) (stream *terminal-io*) (times NIL))
  "Non-consing QIX."
  (let* ((list (if (and qix-list (<= (1+ length) qix-list-length))
		    qix-list
		    (progn
		      (setq qix-list-length length
			    qix-list (make-list (1+ qix-list-length)))
		      ;; Make history a circular list.
		      (si:%p-store-cdr-code (cdr (nthcdr (1- length) qix-list)) cdr-error)
		      (si:%p-store-cdr-code (nthcdr (1- length) qix-list) cdr-normal)
		      (rplacd (nthcdr (1- length) qix-list) qix-list)
		      (loop repeat length
			    for h =(nthcdr (1- length) qix-list) then (cdr h)
			    do (setf (car h) (make-list 4)))
		      qix-list)))
	  (history (nthcdr (1- length) list)))
     (send stream :clear-screen)
     (multiple-value-bind (xlim ylim)
	 (send stream :inside-size)
       (loop with x1 = (random xlim)
	     and y1 = (random ylim)
	     and x2 = (random xlim)
	     and y2 = (random ylim)
	     and dx1 = 5
	     and dy1 = 12
	     and dx2 = 12
	     and dy2 = 5
	     with tem
	     until (if times (= (setf times (1- times)) 0) NIL)
	     when (caar history)
	     do (prepare-sheet (stream)
			      (sys:%draw-shaded-triangle
			        (first (car history))(second (car history))
			        (third (car history))(fourth (car history))
			        (third (car history))(fourth (car history))
				tv:alu-xor t t t nil stream))
	     do
	     (setf (first (car history)) x1)
	     (setf (second (car history)) y1)
	     (setf (third (car history)) x2)
	     (setf (fourth (car history)) y2)
	     (setf history (cdr history))
	     (prepare-sheet (stream)
			      (sys:%draw-shaded-triangle
			        x1 y1 x2 y2 x2 y2
				tv:alu-xor t t t nil stream))
	     (setf dx1 (1- (+ dx1 (random 3)))
		   dy1 (1- (+ dy1 (random 3)))
		   dx2 (1- (+ dx2 (random 3)))
		   dy2 (1- (+ dy2 (random 3))))
	     (cond ((> dx1 12) (setf dx1 12))
		   ((< dx1 -12) (setf dx1 -12)))
	     (cond ((> dy1 12) (setf dy1 12))
		   ((< dy1 -12) (setf dy1 -12)))
	     (cond ((> dx2 12) (setf dx2 12))	
		   ((< dx2 -12) (setf dx2 -12)))
	     (cond ((> dy2 12) (setf dy2 12))
		   ((< dy2 -12) (setf dy2 -12)))
	     (cond ((or (>= (setf tem (+ x1 dx1)) xlim)
			(minusp tem))
		    (setf dx1 (- dx1))))
	     (cond ((or (>= (setf tem (+ x2 dx2)) xlim)
			(minusp tem))
		    (setf dx2 (- dx2))))
	     (cond ((or (>= (setf tem (+ y1 dy1)) ylim)
			(minusp tem))
		    (setf dy1 (- dy1))))
	     (cond ((or (>= (setf tem (+ y2 dy2)) ylim)
			(minusp tem))
		    (setf dy2 (- dy2))))
	     (setf x1 (+ x1 dx1)
		     y1 (+ y1 dy1)
		     x2 (+ x2 dx2)
		     y2 (+ y2 dy2))
	  finally (loop repeat length
			when (caar history)
			 do (prepare-sheet (stream)
			      (sys:%draw-shaded-triangle
			        (first (car history))(second (car history))
			        (third (car history))(fourth (car history))
			        (third (car history))(fourth (car history))
				tv:alu-xor t t t nil stream))
			do (setf history (cdr history))))))
    )

;; Debug - Execute this to invoke the blackout
;  (SETQ tv:kbd-last-activity-time (- (TIME) (* 3600 *screen-saver-time-delay*))) 

; (setq *screen-saver-hacks* 'tv:screen-saver-hacks)
; (setq *screen-saver-hacks* 'tv:balls)
; (setq *screen-saver-hacks* 'tv:twinkle)
; (setq *screen-saver-hacks* 'tv:rotate-sphere)
; (setq *screen-saver-hacks* 'tv:attraction)
;;; ; (setq *screen-saver-hacks* 'user:FIREWORKS-DISPLAY)





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; SAVER-BALLS - Bouncing Balls
;;;

;;; Change History
;;;
;;;  Date	Author	Description
;;; -------------------------------------------------------------------------------------
;;; 10/30/87	LGO	Modified to allow setting ball radius as a parameter.
;;; 10/20/87    TWE	Updated to handle different sized balls in a more generic manner.
;;;			Also fixed up the calculation for *mini-raster-offset* to make
;;;			sure that the size of the ball image array is calculated
;;;			correctly.  Previously it was `off by one'.  Made the drawing of
;;;			the balls appear smoother by displaying the results of a movement
;;;			instead of doing two XORs.  This makes large balls appear much
;;;			better by removing much of the flicker.  Fixed up the top-level
;;;			function (i.e. balls) to use its ball count argument properly.

(DEFPARAMETER maxvel 1500)
(DEFPARAMETER elastic 800)             ; elasticity coefficient
(defparameter *number-of-balls* 5)
(DEFVAR ball-array (make-array *number-of-balls*))       ; The universe of balls.
(defvar number-balls 0)          ; current number of balls.
(defvar ball-image nil)          ; bitblt array
(defvar ball-temp-image nil)     ; temporary bitblt array
(defvar combined-ball-image nil) ; combination of old and new balls
(DEFVAR deactivated-window nil)  ;  the above 3 arrays all indirect into this screen-array.
(proclaim '(special MinX MaxX MinY MaxY))	; window dimensions, actual
(proclaim '(special VMaxX VMaxY))		; window dimensions, virtual

;;; Macros to translate between internal (virtual) coordinates and
;;;   screen coordinates:

(defconstant virt-coordinates-scale-factor 64)
(defmacro virt-to-screen(q) `(floor ,q virt-coordinates-scale-factor))
(defmacro screen-to-virt(q) `(floor (* ,q virt-coordinates-scale-factor)))

(defparameter *ball-radius* 32)			; radius of ball
(defvar *ball-diameter*); diameter
(defvar *ball-v-diameter*); diameter
(defvar *ball-dsq*)
(defvar *mini-raster-size*)      ; size of miniraster
(defvar *mini-raster-offset*)    ; distance to miniraster center
(defvar *mini-raster-slack*)

;;; Initialize window, various globals.

(defun SetupCrapolla(number-of-balls radius)  ; in honor of you-know-who

  (LET (the-width)
  (SETQ *ball-radius* radius)
  (SETQ *ball-diameter* (* 2 *ball-radius*)); diameter
  (SETQ *ball-v-diameter* (screen-to-virt (* 2 *ball-radius*))); diameter
  (SETQ *ball-dsq* (* (screen-to-virt *ball-diameter*)
			   (screen-to-virt *ball-diameter*)))
  (SETQ *mini-raster-size* (truncate (* *ball-radius* 2.5)))      ; size of miniraster
  (SETQ *mini-raster-offset*		; distance to miniraster center
	     (ceiling *mini-raster-size* 2))
  (SETQ *mini-raster-slack* (- *mini-raster-offset* *ball-radius*))

  (setq ball-array (make-array number-of-balls))
  (setq number-balls 0)
  (multiple-value-setq (MinX MinY MaxX MaxY) (send *terminal-io* ':inside-edges))
  (incf MinX *ball-radius*)
  (incf MinY *ball-radius*)

  ;;; Virtual coordinates q range:  0 <= q < VMaxQ...
  (setq VMaxX (screen-to-virt (- MaxX MinX *mini-raster-size*)))
  (setq VMaxY (screen-to-virt (- MaxY MinY *mini-raster-size*)))

  (send *terminal-io* :clear-screen)

  ;;;  Build the ball image mini-raster directly on the screen:
  (UNLESS deactivated-window
    (SETF deactivated-window
	  (MAKE-INSTANCE 'w:window-without-label
			 :save-bits t :borders nil
			 :deexposed-typeout-action :permit
			 :height (* 6 *mini-raster-offset*)
			 :width (* 10 *mini-raster-offset*)))
    (SETF the-width (ARRAY-DIMENSION (tv:sheet-screen-array deactivated-window) 1)))
  (setq ball-image
	(tv:make-sheet-bit-array *terminal-io*
				 the-width (* 2 *mini-raster-offset*)
				 :displaced-to (tv:sheet-screen-array deactivated-window)))
  (setq ball-temp-image
	(tv:make-sheet-bit-array *terminal-io*
				 the-width (* 2 *mini-raster-offset*)
				 :displaced-to (tv:sheet-screen-array deactivated-window)
				 :displaced-index-offset (* 2 *mini-raster-offset*)))
  (setq combined-ball-image
	(tv:make-sheet-bit-array *terminal-io*
				 the-width (* 6 *mini-raster-offset*)
				 :displaced-to (tv:sheet-screen-array deactivated-window)
				 :displaced-index-offset (* 4 *mini-raster-offset*)))
  (CREATE-BALL-IMAGE)

  ;;; Now paint the initial screen:
  (send *terminal-io* :clear-screen)
  #+comment
  (do ((i 0 (+ 1 i)))
      ((= i 5))
    (let ((dd (- *mini-raster-offset* *ball-radius* i i)))
      (send *terminal-io* :draw-lines tv:alu-xor
	  (+ MinX dd) (+ MinY dd)
	  (+ MinX dd) (- MaxY dd)
	  (- MaxX dd) (- MaxY dd)
	  (- MaxX dd) (+ MinY dd)
	  (+ MinX dd) (+ MinY dd) )))
  ))

(DEFUN CREATE-BALL-IMAGE ()
;;;  The old code to create the ball is commented out.  It left a tiny opening in the
;;;  center of the ball.  The new code does not have this problem.
;;;  (if (get-handler-for *terminal-io* :draw-filled-in-circle)
;;;      (send *terminal-io* :draw-filled-in-circle *mini-raster-offset* *mini-raster-offset*
;;;            *Ball-radius* tv:alu-xor)
;;;      (send *terminal-io* :draw-filled-circle *mini-raster-offset* *mini-raster-offset*
;;;            *Ball-radius* w:black w:alu-xor))
;;;  (send *terminal-io* :bitblt-from-sheet tv:alu-seta
;;;		   (* 2 *mini-raster-offset*) (* 2 *mini-raster-offset*)
;;;		   0 0 ball-image 0 0)
  (CREATE-BALL *mini-raster-offset* *mini-raster-offset* BALL-IMAGE))

(defun CREATE-BALL (center-x center-y array)
  (LET ((ALU W:ALU-XOR)
        (radius *ball-radius*))
    (IF (si:mx-p)
	  (SEND deactivated-window :draw-filled-circle center-x center-y radius)
      ;; else...
      (DO ((Y 0)
	   (F 0)				       ; F is just Y squared without any multiplies
	   (X RADIUS)
	   (last-x radius))
	  (NIL)
	(si:%draw-shaded-raster-line (- CENTER-X X) (+ CENTER-X X) (- CENTER-Y Y) alu t nil array)
	;; Draw the middle line only once.
	(when (not (zerop y))
	  (si:%draw-shaded-raster-line (- CENTER-X X) (+ CENTER-X X) (+ CENTER-Y Y) alu t nil array))
	;; Handle pixel errors by only drawing this upper/lower part once.
	(when (not (= x last-x))
	  (si:%draw-shaded-raster-line (- CENTER-X y) (+ CENTER-X y) (- CENTER-Y x) alu t nil array)
	  (si:%draw-shaded-raster-line (- CENTER-X y) (+ CENTER-X y) (+ CENTER-Y x) alu t nil array))
	(setq last-x x)
	(SETQ F (+ F Y Y 1)
	      Y (1+ Y))
	(COND ((>= F X) (SETQ F (- F X X -1)
			      X (- X 1))))
	(COND ((> Y X) (RETURN)))
	(COND ((= Y X) (RETURN)))))))

(defflavor Ball ((x-position 100)		; Positions of CENTER of ball...
		 (y-position 100)		;   ... in virt coordinates.
		 (x-velocity 0)			; 1st differences, virt coords.
		 (y-velocity 0)
		 ball-number
		 (OnScreen nil))		; T iff image on screen.
	   ()
  :gettable-instance-variables
  :settable-instance-variables
  #+3600
  :initable-instance-variables
  #+Explorer
  :inittable-instance-variables
  (:outside-accessible-instance-variables x-position y-position)
  )

;;; XOR a ball on the screen at center position x, y (screen)
(defun XOR-ball(x y)
  (bitblt tv:alu-xor *mini-raster-size* *mini-raster-size*
          ball-image 0 0
          (w:sheet-screen-array *terminal-io*) (max 0 x) (max y 0)))

(defun MOVE-ball(x y dx dy)
  (bitblt tv:alu-seta *mini-raster-size* *mini-raster-size*
	  ball-image 0 0
	  ball-temp-image 0 0)
  (bitblt tv:alu-xor *ball-diameter* *ball-diameter*
	  ball-image *mini-raster-slack* *mini-raster-slack*
	  ball-temp-image
	  (+ *mini-raster-slack* dx)
	  (+ *mini-raster-slack* dy) )

  (send *terminal-io* :bitblt tv:alu-xor
		*mini-raster-size* *mini-raster-size* ball-temp-image 0 0
		(virt-to-screen x)
		(virt-to-screen y)))

;;; Combine the two balls together and display the change.
;;; In this way we reduce the flicker due to XOR when the
;;; size of a ball is large.  Care is taken to only bitblt
;;; what we need to and no more.
(defun move-balls (old-x old-y new-x new-y)
  (let ((delta-x (- new-x old-x))
        (delta-y (- new-y old-y))
        (inside-left   (w:sheet-inside-left   *terminal-io*))
        (inside-top    (w:sheet-inside-top    *terminal-io*))
        (inside-right  (w:sheet-inside-right  *terminal-io*))
        (inside-bottom (w:sheet-inside-bottom *terminal-io*))
        (dest-x (- old-x *mini-raster-size*))
        (dest-y (- old-y *mini-raster-size*))
        (source-x 0)
        (source-y 0)
        (combined-image-width  (* 3 *mini-raster-size*))
        (combined-image-height (* 3 *mini-raster-size*)))
    ;; Erase the combined image.
    (bitblt tv:alu-setz combined-image-width combined-image-height
            combined-ball-image 0 0
            combined-ball-image 0 0)
    ;; Copy a ball to the center.
    (bitblt tv:alu-seta  *mini-raster-size*  *mini-raster-size*
            ball-image 0 0
            combined-ball-image  *mini-raster-size* *mini-raster-size*)
    ;; Combine with the new ball.
    (bitblt tv:alu-xor *mini-raster-size* *mini-raster-size*
            ball-image 0 0
            combined-ball-image (+ delta-x *mini-raster-size*) (+ delta-y *mini-raster-size*))

    ;; Make the image width/height and source-x/y be just big enough for the image.
    (setq combined-image-width (+ *mini-raster-size* (abs delta-x)))
    (if (plusp delta-x)
        (setq source-x *mini-raster-size*
              dest-x (+ dest-x *mini-raster-size*))
        ;;ELSE
        (setq source-x (+ *mini-raster-size* delta-x)
              dest-x (+ dest-x *mini-raster-size* delta-x)))
    (setq combined-image-height (+ *mini-raster-size* (abs delta-y)))
    (if (plusp delta-y)
        (setq source-y *mini-raster-size*
              dest-y (+ dest-y *mini-raster-size*))
        ;;ELSE
        (setq source-y (+ *mini-raster-size* delta-y)
              dest-y (+ dest-y *mini-raster-size* delta-y)))

    ;; Copy the result to the screen.
    ;;  Clip on the left and top.
    (when (< dest-x inside-left)
      (incf combined-image-width (- dest-x inside-left))
      (setq source-x (+ source-x (- dest-x) inside-left))
      (setq dest-x inside-left))
    (when (< dest-y inside-top)
      (incf combined-image-height (- dest-y inside-top))
      (setq source-y (+ source-y (- dest-y) inside-top))
      (setq dest-y inside-top))
    ;; Clip on the right and bottom.
    (when (> (+ combined-image-width dest-x) inside-right)
      (setq combined-image-width (- inside-right dest-x)))
    (when (> (+ combined-image-height dest-y) inside-bottom)
      (setq combined-image-height (- inside-bottom dest-y)))
    (bitblt tv:alu-xor combined-image-width combined-image-height
            combined-ball-image source-x source-y
            (w:sheet-screen-array *terminal-io*) dest-x dest-y)))

(defmethod (ball :move-to)(vx vy)
  (let ((dx (- vx x-position))
	(dy (- vy y-position))
	(x (virt-to-screen vx))
	(y (virt-to-screen vy)))
    (if OnScreen
	(if (and nil (<= (abs dx) *mini-raster-slack*)
		 (<= (abs dy) *mini-raster-slack*))
	    (move-ball x y dx dy)
	    (progn
;;              (xor-ball (virt-to-screen x-position) (virt-to-screen y-position))
;;              (xor-ball x y)
              (move-balls x y (virt-to-screen x-position) (virt-to-screen y-position))
                   ))
        (xor-ball x y))
;;        (xor-ball (+ x *mini-raster-offset*) (+ y *mini-raster-offset*)))
    (setq x-position vx)
    (setq y-position vy))
  (setq OnScreen t))

;;; Apply a force vector to a ball:
(defmethod (ball :force)(dx dy)				; The force vector.
 	   (setq x-velocity (+ (floor (* elastic dx) 1024) x-velocity))
	   (setq y-velocity (+ (floor (* elastic dy) 1024) y-velocity)))

;;; Generate a force (DDQ) for bouncing:
(defun bounce-force(displacement) (* displacement 1))

;;; Update a coordinate:
;;; Adds in velocity after checking for wall bounce.

(defmacro update-ball-coord(Q DQ QMAX)`(progn
	   (if (< ,Q 0)(setq ,DQ (+ (bounce-force (- ,Q)) ,DQ)))
	   (if (>= ,Q ,QMAX)(setq ,DQ (- ,DQ (bounce-force (- ,Q ,QMAX)))))
	   (if (> ,DQ maxvel)(setq ,DQ maxvel))
	   (if (< ,DQ (- maxvel))(setq ,DQ (- maxvel)))
	   (setq ,Q (+ ,Q ,DQ)) ))

;;; Update the position of a ball:
(defmethod (ball :update)
	   (&aux (oldx x-position)(oldy y-position))
	   (update-ball-coord oldx x-velocity VMaxX)
	   (update-ball-coord oldy y-velocity VMaxY)
	   (send self :move-to oldx oldy))

#+comment
(defmethod (ball :dump) ()
;  (send *terminal-io* :set-cursorpos 0 (* 16 (send self :ball-number)))
  (format *terminal-io* "~& ~d: ~s: xy = (~d, ~d);  vxy = (~d, ~d)    "
	  (send self :ball-number)
	  self
	  (send self :x-position)
	  (send self :y-position)
	  (send self :x-velocity)
	  (send self :y-velocity)))

;;; Handle ball-ball collisions:
(defun collide(b1 b2)
  (let ((dx (- (ball-x-position b1)
	       (ball-x-position b2)))
	(dy (- (ball-y-position b1)
	       (ball-y-position b2)))
	dsq ddsq)
    
    (setq dsq (+ (* dx dx) (* dy dy)))
    
    (if (< dsq *ball-dsq*)
	(progn (setq dsq (isqrt dsq))
	       (setq ddsq (- *ball-v-diameter* dsq))
	       (setq dx (floor (* ddsq dx) dsq))
	       (setq dy (floor (* ddsq dy) dsq))
;	       (print (list dx dy dsq))
	       (send b1 :force dx dy)
	       (send b2 :force (- dx)(- dy))
	       ))))

;;; Update the world:
(defun update-step()
	   (do* ((i 0 (+ 1 i))
		 (b1 nil))
		((>= i number-balls))
		(send (setq b1 (aref ball-array i)) :update)
		(do ((j (+ 1 i) (+ 1 j)))
		    ((>= j number-balls))
		    (collide b1 (aref ball-array j)) )))

;; Make a new ball:
(defun new-ball(x y &optional (vx 0) (vy 0))
  (setf (aref ball-array number-balls)
        (make-instance 'Ball :x-position x :y-position y :x-velocity vx :y-velocity vy))
  (send (aref ball-array number-balls) :set-ball-number number-balls)
  (setq number-balls (+ 1 number-balls)))

;;; Make a random ball:
(defun rand-ball(&aux (x (random VMaxX))
		      (y (random VMaxY))
		      (vx (srandom (FLOOR maxvel 2)))
		      (vy (srandom (FLOOR maxvel 2))))
  (new-ball x y vx vy))

;;; generate a signed, random fixnum:
(defun srandom(mag)
  (- (random (+ 1 mag mag)) 1 mag))

;;; Try some stuff:

(defun balls (IGNORE &optional (n *number-of-balls*) (radius *ball-radius*))
  (IF (and (= number-balls n) (= n (length ball-array)) (= (* 2 radius) *ball-diameter*))
      (PROGN
	(send *terminal-io* :clear-screen)
	(DOTIMES (i n)
	  (SEND (AREF ball-array i) :set-onscreen nil)))
      ;;ELSE
    (SetupCrapolla n radius)
    (DOTIMES (i n)
      (rand-ball)))
  
  (do ()
      (nil ;(LISTEN) ;; may-9-1-88
       )
;;    (PROCESS-ALLOW-SCHEDULE) ;; Do process-allow-schedule without updating the who-line-run-state
    (setf (si:process-quantum-remaining current-process) -1)
    (funcall si:scheduler-stack-group)
    
    (update-step))
  )

(pushnew 'balls *screen-saver-hacks-list*)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; TWINKLE
;;; This shows a screen of 1000 twinkling stars.
;;; 28 September 1987 pz.  The structure follows along the lines of QIX.

(defvar star-list nil)
(defvar star-list-length nil)

(defconstant exp1  (exp 1))
(defconstant exp2  (exp 2))
(defconstant exp3  (exp 3))
(defconstant exp4  (exp 4))
(defconstant exp5  (exp 5))
(defconstant exp6  (exp 6))
(defconstant exp7  (exp 7))
(defconstant exp8  (exp 8))
(defconstant exp9  (exp 9))
(defconstant exp10 (exp 10))
(defconstant exp11 (exp 11))
(defconstant exp12 (exp 12))
(defconstant exp13 (exp 13))
(defconstant exp14 (exp 14))
(defconstant exp15 (exp 15))

;;; Array of circle images.
(defvar twinkles nil)

;;; Temporary window used to create twinkles.
(defvar junk-twinkle-window nil)

;;; Must be 15 or less.  If you want a larger size then you need to add more constants, and change
;;; twinkle to use them.  Of course, a better way would be to make it independent of the size, but
;;; that would require more thought.
(defvar *max-twinkle-size* 7)

(pushnew 'twinkle *screen-saver-hacks-list*)

(defun setup-twinkles ()
  ;; Initialize the twinkles array with circles of various sizes.
  (when (or (null twinkles) (not (= (length twinkles) *max-twinkle-size*)))
    (when (null junk-twinkle-window)
      (setq junk-twinkle-window (make-instance 'w:window-without-label
					       :height (* *max-twinkle-size* 3)
					       :save-bits t :borders nil
                                               :deexposed-typeout-action :permit))
      (SETF (tv:sheet-blinker-list junk-twinkle-window) nil))
    (setq twinkles (make-array *max-twinkle-size*))
    (loop for index from 0 below *max-twinkle-size*
	  for offset first 0 then (+ offset box-size)
          for radius = index
          for diameter = (* radius 2)
	  for the-width = (ARRAY-DIMENSION (tv:sheet-screen-array junk-twinkle-window) 1)
          ;; For some reason, the width of the radius is too low, resulting in a clipped
          ;; image.  Make the box size more than large enough to compensate.
          for box-size = (max 1 (* radius 3))
          for array = (tv:make-sheet-bit-array tv:default-screen the-width box-size
					       :displaced-to (tv:sheet-screen-array junk-twinkle-window)
					       :displaced-index-offset offset)
          do (progn
               ;; Draw a circle onto the junk window and copy it from there to the twinkle array element.
               (send junk-twinkle-window :draw-filled-circle (+ offset radius) radius radius)
               (setf (aref twinkles index) array)))))

(defun draw-twinkle (width height x y alu window)
  (declare (ignore height))
  ;; Be nice to the window by not writing into its margins.
  (when (>= (+ x width) (tv:sheet-inside-right window))
    (setq x (- (tv:sheet-inside-right window) width 1)))
  (when (>= (+ y width) (tv:sheet-inside-bottom window))
    (setq y (- (tv:sheet-inside-bottom window) width 1)))
  (when (<= x (tv:sheet-inside-left window))
    (setq x (tv:sheet-inside-left window)))
  (when (<= y (tv:sheet-inside-top window))
    (setq y (tv:sheet-inside-top window)))
  ;; Copy the selected circle to the window.
  (bitblt alu (+ 1 width) (+ 1 width)
          (aref twinkles (truncate width 2)) 0 0 (w:sheet-screen-array window) x y))

(defun twinkle (&optional (stream *terminal-io*) (length 1000) (times NIL) (max-size *max-twinkle-size*) (lifetime 100))
  "Non-consing TWINKLE."
  (setup-twinkles)
  (let* ((list (if (and star-list (<= (1+ length) star-list-length))
		   star-list
		   (progn
		     (setq star-list-length length
			   star-list (make-list (1+ star-list-length)))
		     ;; Make history a circular list.
		     (si:%p-store-cdr-code (cdr (nthcdr (1- length) star-list)) cdr-error)
		     (si:%p-store-cdr-code (nthcdr (1- length) star-list) cdr-normal)
		     (rplacd (nthcdr (1- length) star-list) star-list)
		     (loop repeat length
			   for h =(nthcdr (1- length) star-list) then (cdr h)
			   do (setf (car h) (list 0 0 1 0)))
		     star-list)))
	 (history (nthcdr (1- length) list)))
    (send stream :clear-screen)
    (multiple-value-bind (xlim ylim)
	(send stream :inside-size)
      (loop with this = nil
	    and  exp-max-size = (floor (exp max-size))
	    until (if times (= (setf times (1- times)) 0) NIL)
;	    when (caar history)
	    do
	    (setf this (car history))
	    (setf (fourth this) (- (fourth this) 1))
	    (if (<= (fourth this) 0)
		(progn
		  ;; Erase current placement
		  (tv:prepare-sheet (stream)
		    (draw-twinkle
		      (third this) (third this)	; width, height
		      (first this) (second this)	; x-pos, y-pos
		      tv:alu-setz stream))
		  
		  ;; Create a new position
		  (setf (first  this) (random (- xlim (third this))))
		  (setf (second this) (random (- ylim (third this))))
		  (setf (third  this) (- max-size
					 (let ((s (+ 1 (random exp-max-size))))
					   (cond ((> s exp15) 15) ((> s exp14) 14)
						 ((> s exp13) 13) ((> s exp12) 12)
						 ((> s exp11) 11) ((> s exp10) 10)
						 ((> s  exp9)  9) ((> s  exp8)  8)
						 ((> s  exp7)  7) ((> s  exp6)  6)
						 ((> s  exp5)  5) ((> s  exp4)  4)
						 ((> s  exp3)  3) ((> s  exp2)  2)
						 ((> s  exp1)  1) (t 1)))))
		  (setf (fourth this) (random lifetime))
		  
		  ;; And draw it
		  (tv:prepare-sheet (stream)
		    (draw-twinkle
		      (third this) (third this)	; width, height
		      (first this) (second this)	; x-pos, y-pos
		      tv:alu-seta stream))))
	    
	    ;; Next!
	    (setf history (cdr history))
	    
	    ;; Last time through, clean up.
	    finally (loop repeat length
			  when (caar history)
			  do (tv:prepare-sheet (stream)
                               (draw-twinkle
				 (third (car history)) (third (car history))	; width, height
				 (first (car history)) (second (car history))	; x-pos, y-pos
				 tv:alu-setz stream))
			  do (setf history (cdr history)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ROTATING SPHERES - [implemented by Steve Ward?]
;;;

;;; Screen dimensions, centering, etc:


(defvar xc 400)					; center, in screen coordinates.
(defvar yc 400)
(defvar rad 100)				; radius of globe.
(setq rad 100)
(defvar globe-margin 32)
(defvar globe-size (* 32 (truncate (+ globe-margin 31 (+ rad rad 1)) 32)))
(defvar globe-offset (truncate (- globe-size 1 rad rad) 2))

(defvar globe-pix nil)

(defvar globe-window tv:selected-window)
;(defvar globe-window tv:main-screen)
(DEFVAR globe-pix-window nil)
(setq globe-window tv:main-screen)
(defvar Globe-Matrix (make-array '(3 3)))

(defconstant scale-factor 4096.)

(defun Unity-Matrix()
  (loop for i from 0 below 3 do
	(loop for j from 0 below 3 do
	      (setf (aref Globe-Matrix i j)
		    (if (= i j) scale-factor 0)))))

(defun Set-Matrix(lat long
		  &aux (s1 (sin lat))(c1 (cos lat))(s2 (sin long))(c2 (cos long)))
  (setf (aref Globe-Matrix 0 0)(truncate (* c1 c2 scale-factor scale-factor) scale-factor))
  (setf (aref Globe-Matrix 1 0)(truncate (* -1 s1 scale-factor scale-factor) scale-factor))
  (setf (aref Globe-Matrix 2 0)(truncate (* -1 c1 s2 scale-factor scale-factor) scale-factor))
  (setf (aref Globe-Matrix 0 1)(truncate (* s1 c2 scale-factor scale-factor) scale-factor))
  (setf (aref Globe-Matrix 1 1)(truncate (* c1 scale-factor scale-factor) scale-factor))
  (setf (aref Globe-Matrix 2 1)(truncate (* -1 s1 s2 scale-factor scale-factor) scale-factor))
  (setf (aref Globe-Matrix 0 2)(truncate (* s2 scale-factor scale-factor) scale-factor))
  (setf (aref Globe-Matrix 1 2) 0)
  (setf (aref Globe-Matrix 2 2)(truncate (* c2 scale-factor scale-factor) scale-factor))
  )

;;; Do a scaled, fixed-point multiply:
(defsubst scaled-product(number fraction)
  (truncate (* number fraction) scale-factor))

;;; Perform the matrix multiplication: (x, y, z) => (x, y).
;;; Takes 3-vector in, produces 2-vector.
;;; Presumes 12-bit fixnums everywhere.

(defun map-point(x y z &aux xp yp zp)

  (setq xp (+ (scaled-product x (aref Globe-Matrix 0 0))
	      (scaled-product y (aref Globe-Matrix 1 0))
	      (scaled-product z (aref Globe-Matrix 2 0))))
  (setq yp (+ (scaled-product x (aref Globe-Matrix 0 1))
	      (scaled-product y (aref Globe-Matrix 1 1))
	      (scaled-product z (aref Globe-Matrix 2 1))))
  (setq zp (+ (scaled-product x (aref Globe-Matrix 0 2))
	      (scaled-product y (aref Globe-Matrix 1 2))
	      (scaled-product z (aref Globe-Matrix 2 2))))

 (if (< zp 0)
       nil
      (setq xp (+ xc (scaled-product xp rad)))
      (setq yp (+ yc (scaled-product yp rad)))
      (cons xp yp)
      )
  )

(defun globe(xlat xlong &aux (latstep 10)(longstep 10) lat long)

  (loop for rlat from 0 below 360 by latstep do
	(setq lat (rem (+ xlat rlat) 360))
	(globe-line lat longstep xlat xlong))

  (loop for rlong from longstep below 180 by longstep do
	(setq long (rem (+ xlong rlong) 360))
	(loop for rlat from 0 below 360 by latstep do
	      (setq lat (rem (+ xlat rlat) 360))
	      (globe-line lat long (+ lat latstep) long)
	      (globe-line lat long lat (- long longstep))))

  (loop for rlat from 0 below 360 by latstep do
	(setq lat (rem (+ xlat rlat) 360))
	(globe-line lat (- (+ xlong 180) longstep) xlat (+ xlong 180)))

  )


;;; Sin/cos tables, in 1-degree increments.
;;; Scaled to 12 bits signed.
(defvar sintab (make-array 360))
(defvar costab (make-array 360))

(defun sincostables(&aux (twopi (* 2.0 pi)) radians)
  (loop for phi from 0 below 360 do
	(setq radians (/ (* phi twopi) 360.0))
	(setf (aref sintab phi)(truncate (* 4096 (sin radians))))
	(setf (aref costab phi)(truncate (* 4096 (cos radians))))))

;;; Compute the screen coordinates of a (lat, long) point.
;;; Returns (x . y), or NIL if invisible.

(defun globe-point(lat long &aux x y z r)
  (setq lat (rem (+ lat 360) 360))
  (setq long (rem (+ long 360) 360))
      (setq x (aref costab lat))
      (setq y (aref sintab lat))
      (setq z (aref costab long))
      (setq r (aref sintab long))
      (setq x (truncate (* x r) 4096))
      (setq y (truncate (* y r) 4096))

      (map-point x y z)
      )


(defun globe-line (lat1 long1 lat2 long2 &aux p1 p2)
; (print (list 'globe-line lat1 long1 lat2 long2)) 
  (when (and (setq p1 (globe-point lat1 long1))
	     (setq p2 (globe-point lat2 long2)))
    (send globe-window :draw-line (car p1)(cdr p1)(car p2)(cdr p2))))

(defun SetupCrappola()
  (sincostables)
  )


(defun build(&optional (lat -1.5)(long 2.0)
	     &aux arr (row 0) (col 0) the-width)
  (setq globe-pix (make-array 10))
  (UNLESS globe-pix-window
    (SETF globe-pix-window
	  (MAKE-INSTANCE 'w:window-without-label
			 :save-bits t
			 :deexposed-typeout-action :permit
			 :borders nil
			 :height 32
			 :width 32))
    (SEND globe-pix-window :set-size (+ globe-offset (* 4 globe-size))
	  (+ globe-offset (* 3 globe-size))))
  (SETF the-width (ARRAY-DIMENSION (tv:sheet-screen-array globe-pix-window) 1))
  (SetupCrappola)
  (setq globe-window tv:selected-window)
  (loop for delta from 0 below 10 do
	(SETF arr (tv:make-sheet-bit-array *terminal-io* the-width (+ globe-size globe-offset)
		   :displaced-to (tv:sheet-screen-array globe-pix-window)
		   :displaced-index-offset (+ (* row globe-size the-width)
					      (* col globe-size))))
	(loop for i from 0 below globe-size do
	     (loop for j from 0 below globe-size do (setf (aref arr i j) 0)))
	(send globe-window :clear-screen)
	(set-matrix lat long)
	(globe delta 0)
	(send globe-window :bitblt-from-sheet tv:alu-ior
	      globe-size globe-size (- xc rad) (- yc rad) arr globe-offset globe-offset)
	(setf (aref globe-pix delta) arr)
	(WHEN (= 4 (INCF col))
	  (INCF row)
	  (SETF col 0))
	)
  )


(defun show(&aux (x 0)(y 0)
	    (dx 1)(dy 1)
	    xmax ymax)

  (setq xmax (- (send globe-window :width) globe-size))
  (setq ymax (- (send globe-window :height) globe-size))

  (loop do
	(loop for delta from 0 below 10 do
	      (send globe-window :bitblt tv:alu-seta globe-size globe-size (aref globe-pix delta)
		    0 0 x y)
	      (when (or (>= (setq x (+ x dx)) xmax)
			(< x 0))
		(setq x (- x dx))
		(setq dx (- dx)))
	      (when (or (>= (setq y (+ y dy)) ymax)
			(< y 0))
		(setq y (- y dy))
		(setq dy (- dy))) ))
  )

;(defmacro globe-with-real-time body
;  `(let ((old-sb-state (si:sb-on)))
;     (unwind-protect
;       (progn
;	 (si:sb-on '(:keyboard))
;	 . ,body)
;       (send tv:who-line-screen :refresh)
;       (send tv:main-screen :refresh)
;       (time:set-local-time)
;       (si:sb-on old-sb-state))))

(defun raw-show (&optional (window *terminal-io*)
		 &aux
		(x 0)(y 0)(spin t) phi
;		(dx 6500)(dy 0)
		(dx 1701)(dy 0)
;		(grav 650)
		(grav 11)
		(scr (send window :screen-array))
		xmax ymax
		(max-pos (* 4 (MAX (array-dimension scr 1)(array-dimension scr 0)))))
  
  (setq xmax (* max-pos (- (array-dimension scr 1) globe-size)))
  (setq ymax (* max-pos (- (array-dimension scr 0) globe-size)))
  
;  (send tv:who-line-screen :clear-screen)
;  (send tv:main-screen :clear-screen)
;  (globe-with-real-time
;    (send tv:who-line-screen :clear-screen)

;	  (SETQ tv:the-screen-is-black t)
;	  (SETQ tv:kbd-intercepted-characters nil)
;	  (SETQ tv:kbd-global-asynchronous-characters nil)

    (loop ;until (SEND tv:selected-window ':mouse-or-kbd-tyi-no-hang) do
	  (loop for delta from 0 below 10 do

		(tv:prepare-sheet (tv:main-screen)
		  (if spin (setq phi delta)(setq phi (- 9 delta)))
		  (bitblt tv:alu-seta
			  globe-size globe-size
			  (aref globe-pix phi) 0 0
			  scr (truncate x max-pos) (truncate y max-pos))
		  )

		(when (>= (setq x (+ x dx)) xmax)
		  (setq x (- x dx dx))
		  (setq spin (>= dy 0))
		  (setq dx (- dx))
		  ;(send tv:who-line-screen :clear-screen))
		  )

		(when (< x 0)
		  (setq x (- x dx dx))
		  (setq spin (< dy 0))
		  (setq dx (- dx))
		  ;(send tv:main-screen :clear-screen)
		 )
		
		(when (>= (setq y (+ y dy)) ymax)
		  (setq y (- y (* 2 dy)))
		  (setq dy (- dy)))

		(setq dy (+ dy grav))
		
		(when (< y 0)
		  (setq y 0)
		  (setq dy 0))
		)))

(defun rotate-sphere (&optional (window *terminal-io*) (lat -1.0)(long 2.5))
  (when (OR (null globe-pix) (NULL (AREF globe-pix (1- 10)))) ;; may 9-6-88 in case user aborts out of initial globe build
    (build lat long)
    (send window :clear-screen))
  (raw-show window)
  )

(pushnew 'rotate-sphere *screen-saver-hacks-list*)

;(defun try() (screen-save))

;(defun tv:blackout-screen-hack() (screen-save))
;;(screen-save)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ATTRACTION
;;;

;; ATTRACTION
;; A number of attractive/repulsive balls.  Needs to have a special font loaded.

(when (or (not (boundp 'fonts:balls-16)) (null fonts:balls-16))
  (IF (PROBE-FILE "sys:fonts;balls-16.XLD")
    (load "sys:fonts;balls-16" :verbose nil)
    ;;ELSE
    (SETQ FONTS:BALLS-16 fonts:mouse)))

(defvar orb-list nil)
(defvar orb-list-length nil)
(defparameter default-orb-length 5)
(defparameter orb-font fonts:balls-16)
(defparameter orb-char #o143)
(when (and (boundp 'fonts:balls-16) fonts:balls-16)
  (pushnew 'attraction *screen-saver-hacks-list*))

(defun attraction (&optional (stream *terminal-io*) (length default-orb-length) (times nil)
		   (max-size 16))
  "Non-consing ATTRACTION"

  (macrolet
    ((x-acc (orb) `(first  ,orb))
     (y-acc (orb) `(second ,orb))
     (x-vel (orb) `(third  ,orb))
     (y-vel (orb) `(fourth ,orb))
     (x-pos (orb) `(fifth  ,orb))
     (y-pos (orb) `(sixth  ,orb))
     (mass  (orb) `(seventh ,orb))
     (size  (orb) `(eighth  ,orb))
     (x-pix (orb xlim) `(min ,xlim (max 0 (floor (+ (/ ,xlim 2.0s0) (x-pos ,orb))))))
     (y-pix (orb ylim) `(min ,ylim (max 0 (floor (+ (/ ,ylim 2.0s0) (y-pos ,orb))))))
     (increment (place value)
		"increment a value"
		(declare (arglist place value &key test test-not key))
		(let ((pl (gensym))
		      (val (gensym)))
		  (si:sublis-eval-once `((,val . ,value))
				       (si:sublis-eval-once `((,pl . ,place))
							    `(values (setf ,place (+ ,pl ,val))))))))

  
  (multiple-value-bind (xlim ylim)
      (send stream :inside-size)
    
    (let* ((xlim/2 (ash xlim -1))
	   (ylim/2 (ash ylim -1))
	   (-xlim/2 (* -1.0s0 xlim/2))
	   (-ylim/2 (* -1.0s0 ylim/2))
	   (2pi (* 2.0s0 (float pi 1.0s0))))
      
      (let* ((diameter (- (min xlim/2 ylim/2) 50))
	     (list (progn
		     (if (not (and orb-list (= length orb-list-length)))
			 (progn
			   (setq orb-list-length length
				 orb-list (make-list orb-list-length))
			   (dotimes (n length)
			     (setf (nth n orb-list) (list 0.0s0 0.0s0 0.0s0 0.0s0 0.0s0 0.0s0 0.0s0 0)))))
		     (dotimes (n length)
		       (let ((new-size (min 16 (+ 8 (random (- max-size 9)))))
			     (orb (nth n orb-list)))
			 (setf (x-acc orb) 0.0s0)
			 (setf (y-acc orb) 0.0s0)
			 (setf (x-vel orb) (/ (- 6.0s0 (random 11)) 8.0s0))
			 (setf (y-vel orb) (/ (- 6.0s0 (random 11)) 8.0s0))
			 (setf (x-pos orb) (* diameter (cos (* n (/ 2pi length)))))
			 (setf (y-pos orb) (* diameter (sin (* n (/ 2pi length)))))
			 (setf (mass orb) (* new-size new-size 10.0s0))
			 (setf (size orb) new-size)))
		     orb-list)))
	
	(send stream :clear-screen)
 
	(loop until (if times (= (setf times (1- times)) 0) NIL)
	      do
	      
	      (dotimes (l length)
		;; calculate attraction of this orb to the other orbs, this will set the x-acc and y-acc.
		(let ((new-x-acc 0.0s0) (new-y-acc 0.0s0)
		      (orb (nth l list)))
		  (dotimes (ll length)
		    (let ((other-orb (nth ll list)))
		      (if (not (eq orb other-orb))	; don't do it to yourself !!!
			  (let* ((x-dist (- (x-pos other-orb) (x-pos orb)))
				 (y-dist (- (y-pos other-orb) (y-pos orb)))
				 (dist^2 (+ (* x-dist x-dist)
					    (* y-dist y-dist)))
				 (dist (sqrt dist^2))
				 (new-acc 0.0s0)
				 (new-acc/dist 0.0s0))
			    (if (> dist 0.1s0)
				(progn
				  (setq new-acc (* (/ (mass other-orb) dist^2)
						   (cond ((< dist 100.0s0) -1.0s0)
;							 ((< dist 3.0s0)   -10.0s0)
;							 ((< dist 1.0s0)   -100.0s0)
							 (t 1.0s0))))
				  (setq new-acc/dist (/ new-acc dist))
				  (increment new-x-acc (* new-acc/dist x-dist))
				  (increment new-y-acc (* new-acc/dist y-dist)))
				(progn
				  (increment new-x-acc (- 5.0s0 (random 10.0s0)))
				  (increment new-y-acc (- 5.0s0 (random 10.0s0)))))))))
		  (setf (x-acc orb) new-x-acc)
		  (setf (y-acc orb) new-y-acc)))
	      
	      (dotimes (l length)
		(let* ((orb (nth l list))
		       (old-x-pix (x-pix orb xlim))
		       (old-y-pix (y-pix orb ylim))
		       (new-x-pix 0)
		       (new-y-pix 0)
		       (orb-size (size orb)))
		  
		  ;; set the new velocities
		  (increment (x-vel orb) (x-acc orb))
		  (increment (y-vel orb) (y-acc orb))

		  ;; make sure that if things get too fast, they slow down!
		  (if (< 10.0s0 (abs (x-vel orb)))
		      (progn
			(setf (x-vel orb) (* (x-vel orb) 0.9s0))
			(setf (x-acc orb) 0.0s0)))

		  (if (< 10.0s0 (abs (y-vel orb)))
		      (progn
			(setf (y-vel orb) (* (y-vel orb) 0.9s0))
			(setf (y-acc orb) 0.0s0)))
		  
		  ;; set the new positions
		  (increment (x-pos orb) (x-vel orb))
		  (increment (y-pos orb) (y-vel orb))
		  
		  ;; check for maximum position in x direction
		  (if (<= (- xlim/2 max-size 4) (x-pos orb))
		      (progn
			(setf (x-pos orb) (- xlim/2 max-size 5))
			(setf (x-vel orb) (- (max 0.1s0 (abs (x-vel orb))))))
		      (if (>= (+ -xlim/2 1) (x-pos orb))
			  (progn
			    (setf (x-pos orb) (+ -xlim/2 2))
			    (setf (x-vel orb) (max 0.1s0 (abs (x-vel orb)))))))
		  
		  ;; check for maximum position in y direction
		  (if (<= (- ylim/2 max-size 4) (y-pos orb))
		      (progn
			(setf (y-pos orb) (- ylim/2 max-size 5))
			(setf (y-vel orb) (- (max 0.1s0 (abs (y-vel orb))))))
		      (if (>= (+ -ylim/2 1) (y-pos orb))
			  (progn
			    (setf (y-pos orb) (+ -ylim/2 2))
			    (setf (y-vel orb) (max 0.1s0 (abs (y-vel orb)))))))
		  
		  ;; calculate the new pixel position
		  (setq new-x-pix (x-pix orb xlim))
		  (setq new-y-pix (y-pix orb ylim))
		  
		  (tv:prepare-sheet (stream)
		    (sys:%draw-character orb-font orb-char	; orb-size
                                         (+ 4 orb-size)		; erase
					 old-x-pix old-y-pix
					 tv:alu-setz stream)
		    (sys:%draw-character orb-font orb-char	; orb-size
                                         (+ 4 orb-size)		; draw new orb
					 new-x-pix new-y-pix
					 tv:alu-seta stream))))))))))

(DEFFLAVOR screen-saver-blinker ((counter 120)) (blinker))

(WHEN (si:mx-p)
  
  (DEFMETHOD (screen-saver-blinker :blink) ()
    (WHEN (ZEROP (DECF counter))
      (SETF counter 120)
      (maybe-start-screen-saver)))

  (DEFMETHOD (screen-saver-blinker :size) ()
    (VALUES 1 1))
  
  (make-blinker tv:default-screen 'screen-saver-blinker))

