;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-

(defflavor fat-char-tyoing-mixin
	   ()
	   ()
  (:Documentation "Knows how to print out fat chars as fonified chars.")
)

(defwhopper (fat-char-tyoing-mixin :around :tyo) (char &rest args)
"Prints any fontified chars specially, otherwise just prints as normal."
  (if (> (char-font char) 0)
      (let ((old-font (send self :current-font)))
           (unwind-protect
	       (progn (if (equal (aref (send self :font-map) (char-font char))
				 (aref (send self :font-map) 0)
		          )
			  (set-font-map-if-you-must self)
			  nil
		      )
		      (send self :set-current-font (char-font char))
		      (lexpr-continue-whopper char args)
	       )
	     (send self :set-current-font old-font)
	   )
      )
      (lexpr-continue-whopper char args)
  )
)

(defmethod (fat-char-tyoing-mixin :string-out)
	   (string &optional (start 0) (end nil))
"Prints out the string with any fonts in it."
  (let ((tem (if (typep string 'array) string (string string))))
       (loop for i
	     from start
	     to (if end (- end 1) (- (array-active-length tem) 1))
	     do (send self :tyo (aref tem i))
       )
  )
)

(defmethod (fat-char-tyoing-mixin :line-out)
	   (string &optional (start 0) (end nil))
"Prints out the string with any fonts in it."
  (let ((tem (if (typep string 'array) string (string string))))
       (loop for i from start to (if end end (- (array-active-length tem) 1))
	     do (send self :tyo (aref tem i))
       )
       (send self :tyo #\newline)
  )
)


(DEFFLAVOR BASIC-PEEK ((NEEDS-REDISPLAY NIL))
   (Fat-Char-Tyoing-Mixin  ;;;; !!! JPR
    scroll-MOUSE-MIXIN
    BORDERS-MIXIN
    scroll-bar-mixin
    SCROLL-WINDOW-WITH-TYPEOUT FULL-SCREEN-HACK-MIXIN)
  :SETTABLE-INSTANCE-VARIABLES
  :GETTABLE-INSTANCE-VARIABLES
  (:DEFAULT-INIT-PLIST :LABEL "Peek"
		       :TRUNCATION t)
  (:DOCUMENTATION :Special-Purpose
		  "The actual peek window.  This has the capability
                   to display in a PEEK display mode."))

(defun describe-sheet (sheet)
  (let ((more (sheet-more-flag sheet))
	(o/h  (sheet-output-hold-flag sheet))
	(d/t/a (send sheet :Deexposed-Typeout-Action))
	(s/b-p (send sheet :Save-Bits))
	(*print-case* :Downcase)
       )
       (apply 'string-append
	      (mapcar 'fontify-string
		      (list "More: "
			    (format nil "~3a" (eql more 1))
			    ", O/H: "
			    (format nil "~3a" (eql o/h 1))
			    ", DTA: "
			    (format nil "~6a" d/t/a)
			    ", S/Bs "
			    (format nil "~3a" s/b-p)
		      )
		      (list 1
			    2
			    1
			    2
			    1
			    2
			    1
			    2
		      )
	      )
       )
  )
)


(defvar *peek-extra-data-column* 50)

;; Modified TI code.
(DEFUN PEEK-WINDOW-INFERIORS (WINDOW INDENT)
  (DECLARE (SPECIAL WINDOW INDENT))
  (scroll-maintain-list
    (CLOSURE '(WINDOW) #'(LAMBDA () (SHEET-INFERIORS WINDOW)))
    (CLOSURE '(INDENT)
	     #'(LAMBDA (SHEET)
		 (LIST ()
		       (SCROLL-PARSE-ITEM 
			 (FORMAT NIL "~V@t" INDENT)
			 `(:MOUSE
			    (NIL :EVAL (PEEK-WINDOW-MENU ',SHEET)
				 :DOCUMENTATION
				 "Menu of useful things to do to this window.")
			    :STRING
			    ,(string (SEND SHEET :NAME)))
			 (FORMAT NIL "~V@t"
			   (max 0 (- *Peek-Extra-Data-Column*
				     Indent
				     (length (string (SEND SHEET :NAME))))))
			 " "
			 (describe-sheet sheet)
			 (fontify-string ", Lock: " 1)
			 (cond ((typep (sheet-lock sheet) 'sys:process)
				`(:mouse
				   (nil :Eval
					(peek-process-menu ',(sheet-lock sheet))
					:documentation
				 "Menu of useful things to do to this process.")
				   :string
				   ,(string (send (sheet-lock sheet) :name))))
			       ((sheet-lock sheet)
				(format nil "~S" (sheet-lock sheet))
			       )
			       (t (fontify-string "Not Locked" 1))
			 )
			 (if (typep sheet 'process-mixin)
			     (fontify-string ", Process: " 1)
			     ""
			 )
			 (if (typep sheet 'process-mixin)
			     `(:mouse
				(nil :Eval
				     (peek-process-menu ',(send sheet :process))
				     :documentation
				 "Menu of useful things to do to this process.")
				:string
				,(string (send (send sheet :Process) :name)))
			     ""
			 )
		       )
		       (PEEK-WINDOW-INFERIORS SHEET (+ INDENT 4)))))))


;;; Define how to show interval streams in the peek file display.
(defun peek-zwei-stream (&optional (indent 0) &aux direction)
  (declare (:self-flavor zwei:interval-stream))
  (tv:scroll-parse-item
    :mouse
    `(nil :eval (fs:peek-file-system-stream-menu ',self) :documentation
	  "Menu of useful things to do to this open file.")
    (and (/= indent 0)
	 (format () "~VT" indent))
    (case (setq direction (send self :direction))
      (:input "Input ")
      (:output "Output ")
      (:Bidirectional "BiDirectional ")
      (otherwise "Direction? "))
    (let ((path (catch-error (send self :pathname) nil)))
         (or (and path (send path :string-for-printing))
	     "Error.... Cannot find pathname."
	 )
    )
    (if (send self :characters)
	", Character, "
	", Binary, ")
    `(:function
       ,(function
	  (lambda (stream)
	    (setf (tv::value 0)
		  (or (catch-error (send stream :read-pointer) nil) 0))
	    (tv::value 0)))
       (,self) nil ("~D"))
    (and (eq direction :input)
	 `(:function
	    ,(function
	       (lambda (stream)
		 (let ((length (or (catch-error (send stream :length) nil) 0)))
		   (and length (not (zerop length))
			(truncate (* 100 (tv::value 0)) length)))))
	    (,self) nil ("~@[ (~D%)~]")))
    " bytes"))


;;; Augment the peek file system display to show open files on the
;;; local machine.
(defwhopper (host:host :peek-file-system) (&rest args)
  (let ((result (lexpr-continue-whopper args))
	(others (remove nil
		 (loop for str
		       in (send tv:who-line-file-state-sheet :Open-Streams)
		       collect
		       (if (typep str 'zwei:interval-stream)
			   (if (or (typep self 'zwei:zwei-buffer-host)
				   (equal self net:local-host)
			       )
			       (send str :Eval-Inside-Yourself
				     '(peek-zwei-stream 0)
			       )
			       nil
			   )
			   (let ((path (catch-error (send str :Pathname) nil)))
			        (if (and path (equal (pathname-host path) self))
				    (send str :Send-If-Handles
					  :Peek-File-System
				    )
				    nil
				)
			   )
		       )
		 )
		)
	)
       )
       (if others
	   (append result others)
	   result
       )
  )
)

;-------------------------------------------------------------------------------

(defvar *sort-processes-in-peek-p* nil
  "When true causes the peek process display to be sorted by process name.")

(DEFUN PEEK-PROCESSES (peek-pane &rest IGNORE)
  "Shows state of all active processes."
  (LIST ()
	;; 30 of process name, 25 of state, 5 of priority, 10 of quantum left/quantum,
	;; 8 of percentage, followed by idle time (11 columns)
	(SCROLL-PARSE-ITEM
	 (FORMAT () "~30A~21A~10A~10A~8A~8A" "Process Name" "State" "Priority" "Quantum" " %"
		 "Idle"))
	(SCROLL-PARSE-ITEM "")
	(SCROLL-MAINTAIN-LIST #'(LAMBDA ()
				  ;;; JPR.
				  (if *sort-processes-in-peek-p*
				      (sort (copy-list ALL-PROCESSES)
					    #'string-lessp
					    :key #'si:process-name)
				      ALL-PROCESSES))
			      #'(LAMBDA (PROCESS)
				  (SCROLL-PARSE-ITEM
				   `(:MOUSE-ITEM
				     (NIL :EVAL (PEEK-PROCESS-MENU ',PROCESS ',peek-pane 'ITEM 0)
				      :DOCUMENTATION
				      "Menu of useful things to do to this process.")
				     :STRING ,(PROCESS-NAME PROCESS) 36)
				   `(:FUNCTION ,(FUNCTION PEEK-WHOSTATE) ,(CONS PROCESS ()) 31)
				   `(:FUNCTION ,PROCESS (:PRIORITY) 5 ("~D."))
				   `(:FUNCTION ,PROCESS (:QUANTUM-REMAINING) 5 ("~4D/"))
				   `(:FUNCTION ,PROCESS (:QUANTUM) 5 ("~D."))
				   `(:FUNCTION ,PROCESS (:PERCENT-UTILIZATION) 10
                                               ("~1,1,4$%"))
				   `(:FUNCTION ,PROCESS (:IDLE-TIME) NIL
                                               ("~\\PEEK-PROCESS-IDLE-TIME\\"))))
			      () ())
	(SCROLL-PARSE-ITEM "")
        (SCROLL-PARSE-ITEM "Clock Function List")
	(SCROLL-MAINTAIN-LIST #'(LAMBDA () CLOCK-FUNCTION-LIST)
			      #'(LAMBDA (FUNC)
				  (SCROLL-PARSE-ITEM
				   `(:STRING ,(WITH-OUTPUT-TO-STRING (STR)
						(PRINC FUNC STR)))))))
	  )