;;; -*- Mode:Common-Lisp; Package:File-System; Base:10; Patch-File: t -*-

;;; This software developed/modified by:
;;;	James Rice
;;;     Rich Acuff
;;; at the Stanford University Knowledge Systems Lab in 1986, 1987, 1988, 1989.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Much of this file is derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND


;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (c)(1)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1985- 1989 Texas Instruments Incorporated. All rights reserved.
;;;  ** (c) Copyright 1980 Massachusetts Institute of Technology **
;;;----------------------------------------------------------------------


;;; This file contains extensions to the Explorer pathname system which add the
;;; following functionality:
;;; a) :Wild-Inferiors - This is like the Symbolics :Wild-Inferiors directory
;;;    component.  It is expressed either as :Wild-Inferiors as a directory
;;;    item to Make-Pathname etc. or as "**" in a namestring.  Thus the path
;;;    "Foo:Bar.**.Baz;Frob.Lisp" will match to any Frob.Lisps on Foo, which
;;;    are under the top level directory Bar (or logical directory) and have
;;;    Baz as their last directory component, e.g.
;;;    "Foo:Bar.Bax.Frob.Baz;Frob.Lisp".
;;;
;;; b) Relativised wild pathname versions (for want of a better name).  These
;;;    allow you to express wild pathnames such as "foo.lisp#>-*", i.e.
;;;    everything other than the most recent version, or "foo.lisp#-2+*, i.e.
;;;    everything after the -2th version (in fact the most recent two versions.
;;;    Any version can be relativised like this, be it an absolute version
;;;    (e.g. 42) or a relative one (e.g. -2, :Oldest, :Newest, >, <, 0).
;;;    A + sign denotes everything after and a - sign denotes everything before.
;;;
;;; c) Logical expressions in directory, name, types and versions.  These
;;;    allow you to express such things as: all the files whose names begin
;;;    with "L" but not "LOGIN-INIT".lisp.  Such a pathname would be expressed
;;;    as "lm:foo;(and l* (not login-init)).lisp".  Any depth of nesting of
;;;    expressions is allowed.  Any function/special form name which is fbound
;;;    can be used, e.g. And, Or, Not.  The function must be a function which
;;;    takes boolean args and returns a boolean value.  The args for a call
;;;    to such a function are T if the expression denotes a match and Nil
;;;    if not.
;;;
;;; As an example of the above:
;;;
;;; x6:rice.**.(not frob);*.(or lisp grammar)#-1-*
;;;
;;; will match to all of the files in the Rice directory tree, except for
;;; those in the Frob directory, which are either of the type Lisp or Grammar
;;; and whose version is lower than the most recent but one.

;-------------------------------------------------------------------------------

(defmethod (pathname :Match-up-relative-versions) (pathname)
  (and (consp version)
       (equal :version (first version))
       (let ((relative-path
	       (fs:make-pathname :defaults pathname :version (second version))
	     )
	    )
	    (let ((actual-relative-path (probe-file relative-path)))
		 (and actual-relative-path
		      (file-versions-match
			version
			(and actual-relative-path
			     (pathname-version actual-relative-path)
			)
			(pathname-version pathname)
		      )
		 )
	    )
       )
  )
)

;-------------------------------------------------------------------------------
;;; Fix up new-lm-parsing-mixin

(DEFMETHOD (PATHNAME :new-PATHNAME-MATCH)
	   (PATHNAME &OPTIONAL (MATCH-HOST T))
  ;;; Implemented by JPR, copied from (PATHNAME :PATHNAME-MATCH)
  (MULTIPLE-VALUE-BIND (W* W1)
    (SEND SELF :INTERNAL-WILD-CHARACTERS)
    (AND (OR (NOT MATCH-HOST) (EQ HOST (PATHNAME-HOST PATHNAME)))
       (PATHNAME-COMPONENT-MATCH DEVICE (PATHNAME-DEVICE PATHNAME) W* W1)
       (PATHNAME-COMPONENT-MATCH DIRECTORY (PATHNAME-DIRECTORY PATHNAME) W* W1)
       (PATHNAME-COMPONENT-MATCH NAME (PATHNAME-NAME PATHNAME) W* W1)
       (PATHNAME-COMPONENT-MATCH TYPE (PATHNAME-TYPE PATHNAME) W* W1)
       (or (send self :Match-up-relative-versions pathname)
	   (and (not (consp version))
		(PATHNAME-COMPONENT-MATCH
		  VERSION (PATHNAME-VERSION PATHNAME) W* W1))))))

(DEFMETHOD (NEW-LM-PARSING-MIXIN :PATHNAME-MATCH)
	   (PATHNAME &OPTIONAL (MATCH-HOST T))
  (send self :New-Pathname-Match pathname match-host)
)

(DEFMETHOD (PATHNAME :New-PATHNAME-MATCH-NO-DEVICE)
	   (PATHNAME &OPTIONAL (MATCH-HOST T))
  ;;; Implemented by JPR, copied from (PATHNAME :PATHNAME-MATCH-NO-DEVICE)
  (MULTIPLE-VALUE-BIND (W* W1)
    (SEND SELF :INTERNAL-WILD-CHARACTERS)
    (AND (OR (NOT MATCH-HOST) (EQ HOST (PATHNAME-HOST PATHNAME)))
       (PATHNAME-COMPONENT-MATCH DIRECTORY (PATHNAME-DIRECTORY PATHNAME) W* W1)
       (PATHNAME-COMPONENT-MATCH NAME (PATHNAME-NAME PATHNAME) W* W1)
       (PATHNAME-COMPONENT-MATCH TYPE (PATHNAME-TYPE PATHNAME) W* W1)
       (OR ;;; Added by JPR.
	   (send self :Match-up-relative-versions pathname)
	   (and (not (consp version))
		(or (PATHNAME-COMPONENT-MATCH
		      VERSION (PATHNAME-VERSION PATHNAME) W* W1)
		    (MEMBER VERSION '(:NEWEST :OLDEST) :TEST #'EQ)))))))

(DEFMETHOD (NEW-LM-PARSING-MIXIN :PATHNAME-MATCH-NO-DEVICE)
	   (PATHNAME &OPTIONAL (MATCH-HOST T))
  (send self :New-Pathname-Match-No-Device pathname MATCH-HOST)
)

(DEFMETHOD (PATHNAME :new-PARSE-TYPE-SPEC) (SPEC)
  (COND
    ((STRINGP SPEC) (SEND SELF :PARSE-COMPONENT-SPEC SPEC))
    ((MEMBER SPEC '(NIL :UNSPECIFIC :WILD) :TEST #'EQ) SPEC)
    ((consp spec) spec) ;;; allows things like (or foo bar)
    ((KEYWORDP spec)
     (string (DECODE-CANONICAL-TYPE spec (SEND HOST :SYSTEM-TYPE))))
    ;allow unknown :TYPEs
    (T (DECODE-CANONICAL-TYPE :LISP (SEND HOST :SYSTEM-TYPE)))))

(DEFMETHOD (NEW-LM-PARSING-MIXIN :PARSE-TYPE-SPEC) (SPEC)
  (send self :New-Parse-Type-Spec spec)
)

(DEFMETHOD (PATHNAME :new-PARSE-DIRECTORY-COMPONENT-SPEC) (SPEC)
  (COND
    ((STRINGP SPEC) SPEC)
    ;;; Wild-Inferiors added by JPR.
    ((MEMBER SPEC '(NIL :UNSPECIFIC :WILD :ROOT :Wild-Inferiors) :TEST #'EQ)
     SPEC)
    ((consp spec) spec) ;;; Allows things like (or foo bar) added by JPR on
    ;;; 28/10/88
    (T "FOO")))

(DEFMETHOD (NEW-LM-PARSING-MIXIN :PARSE-DIRECTORY-COMPONENT-SPEC) (SPEC)
  (send self :New-Parse-Directory-Component-Spec spec)
)


(DEFMETHOD (pathname :new-PARSE-VERSION-SPEC) (SPEC)
  (COND
    ((STRINGP SPEC) (LM-PARSE-VERSION SPEC SELF T))
    ((OR (FIXNUMP SPEC)
	 (MEMBER SPEC '(NIL :UNSPECIFIC :WILD :NEWEST :OLDEST) :TEST #'EQ))
     SPEC)
    ((and (consp spec)
          (equal (length spec) 3)
	  (equal :Version (first spec))
	  (or (keywordp (second spec))
	      (integerp (second spec))
	  )
	  (member (third spec) '(:Before :After))
     )
     spec
    )
    ((and (consp spec)
          (equal (length spec) 3)
	  (or (keywordp (first spec))
	      (integerp (first spec))
	  )
	  (member (second spec) '(:Before :After))
     )
     (cons :Version spec)
    )
    ((and (consp spec)
	  (fboundp (first spec))
     )
     (cons (first spec)
	   (loop for s in (rest spec) collect
		 (send self :New-Parse-Version-Spec s)
	   )
     )
    )
    (T :NEWEST)))

(DEFMETHOD (NEW-LM-PARSING-MIXIN :PARSE-VERSION-SPEC) (SPEC)
  (send self :New-Parse-Version-Spec spec)
)

;-------------------------------------------------------------------------------
;;; Fix up logical pathnames.

;;; By Jpr.
(defun get-physical-host (path)
"Gets the physical host object for a path."
  (typecase path
    (logical-pathname
      (get-physical-host
	(net:parse-host (send (send path :host) :host-translation))
      )
    )
    (pathname (send path :host))
    (otherwise path)
  )
)


(defun is-a-lispm (path)
"Is true if Path is a pathname for a LM file system."
  (equal :lispm (send (get-physical-host path) :system-type))
)

(defwhopper (logical-pathname :PARSE-VERSION-SPEC) (SPEC)
  (if (is-a-lispm self)
      (send self :New-Parse-Version-Spec spec)
      (continue-whopper spec)
  )
)

(defwhopper (logical-pathname :parse-directory-component-spec) (SPEC)
  (if (is-a-lispm self)
      (send self :New-Parse-directory-component-Spec spec)
      (continue-whopper spec)
  )
)

(DEFwhopper (logical-pathname :PARSE-TYPE-SPEC) (SPEC)
  (if (is-a-lispm self)
      (send self :New-Parse-Type-Spec spec)
      (continue-whopper spec)
  )
)

(defwhopper (logical-pathname :PATHNAME-MATCH-NO-DEVICE) (&rest args)
  (if (and (is-a-lispm self) (is-a-lispm (first args)))
      (lexpr-send self :New-Pathname-Match-No-Device args)
      (lexpr-continue-whopper args)
  )
)

(defwhopper (logical-pathname :PATHNAME-MATCH) (&rest args)
  (if (and (is-a-lispm self) (is-a-lispm (first args)))
      (lexpr-send self :New-Pathname-Match args)
      (lexpr-continue-whopper args)
  )
)

;-------------------------------------------------------------------------------


(DEFUN LM-PARSE-DIRECTORY-COMPONENT (STRING SELF &OPTIONAL JUNK-ALLOWED UNQUOTED-IS-OKAY)
  (COND
    ((ZEROP (LENGTH STRING)) NIL)
    ((= (LENGTH STRING) 1.)
     (LET ((CHR (AREF STRING 0.)))
       (COND
	 ((CHAR= CHR #\*) :WILD)
	 ((CHAR= CHR #\~) :ROOT)
	 (T
	  (NAME-VALIDATION STRING (SEND SELF :NAME-LENGTH) () SELF JUNK-ALLOWED UNQUOTED-IS-OKAY)))))
    ;;; Added by JPR.
    ((STRING= STRING "**") :WILD-INFERIORS)
    (T (NAME-VALIDATION STRING (SEND SELF :NAME-LENGTH) () SELF JUNK-ALLOWED UNQUOTED-IS-OKAY))))



(DEFUN LM-PRINT-COMPONENT (SPEC STREAM &OPTIONAL VERSION-P &AUX TEM)
  (COND
    ((EQ SPEC :WILD) (FUNCALL STREAM :TYO #\*))
    ;;; Added by JPR.
    ((EQ SPEC :WILD-INFERIORS)
     (FUNCALL STREAM :TYO #\*)
     (FUNCALL STREAM :TYO #\*))
    ((NUMBERP SPEC) (FORMAT STREAM "~D" SPEC))
    ;;; Added by JPR.
    ((and (consp spec) (fboundp (first spec))
	  (get (first spec) :how-to-print-pathname)
     )
     (funcall (get (first spec) :how-to-print-pathname)
	      spec stream version-p
     )
    )
    ((and (consp spec) (fboundp (first spec)))
     (format stream "(~A" (first spec))
     (mapc #'(lambda (item)
	       (send stream :tyo #\space)
	       (lm-print-component item stream version-p)
	     )
	     (rest spec)
     )
     (format stream ")")
    )
    (VERSION-P
     (COND
       ((SETQ TEM (CDR (ASSOC SPEC '((:NEWEST . #\>) (:OLDEST . #\<)) :TEST #'EQ)))
	(FUNCALL STREAM :TYO TEM))
       ;;; Added by JPR
       ((and (consp spec) (equal (first spec) :version))
	(LM-PRINT-COMPONENT (second SPEC) STREAM VERSION-P)
	(funcall stream :tyo (if (equal (third spec) :before) #\- #\+))
	(FUNCALL STREAM :TYO #\*)
       )
       (T (FERROR "Attempt to print ~S, which is not a valid version." SPEC))))
    ((STRINGP SPEC)
     (DOTIMES (I (ARRAY-ACTIVE-LENGTH SPEC))
       (AND (MEMBER (SETQ TEM (AREF SPEC I)) LM-DELIMITERS :TEST #'EQ)
	  (FUNCALL STREAM :TYO LM-QUOTE-CHARACTER))
       (FUNCALL STREAM :TYO TEM)))
    (T (FERROR () "Attempt to print ~S, which is not a valid component." SPEC))))


;;; By JPR, taken from LM-PARSE-VERSION
(defun maybe-junk-version (string junk-allowed)
  (cond (JUNK-ALLOWED
	 (LM-PARSE-VERSION
	   (NSUBSTRING STRING 0. (- (LENGTH STRING) 1.)) JUNK-ALLOWED))
	(T (FERROR 'PATHNAME-PARSE-ERROR "Invalid version spec ~S" STRING))))


(DEFUN LM-PARSE-VERSION (STRING ignore-this &OPTIONAL JUNK-ALLOWED)
  (IGNORE ignore-this)
  (COND
    ((NUMERIC-P STRING () T))
    ((ZEROP (LENGTH STRING)) NIL)
    ((= (LENGTH STRING) 1.)
     (LET ((CHR (AREF STRING 0.)))
       (COND
	 ((CHAR= CHR #\*) :WILD)
	 ((CHAR= CHR #\>) :NEWEST)
	 ((CHAR= CHR #\<) :OLDEST)
	 ((CHAR= CHR #\) NIL)
	 ((CHAR= CHR #\) :UNSPECIFIC)
	 (JUNK-ALLOWED NIL)
	 (T (FERROR 'PATHNAME-PARSE-ERROR "Invalid version spec ~S" STRING)))))
    ;;; This bit added here by JPR
    ((let ((form (catch-error (read-from-string string) nil)))
          (if (consp form)
	      (parse-logical-expression-for-version form junk-allowed)
	      nil
	  )
     )
    )
    ((search "*" string :test #'string-equal)
     (let ((index (string-reverse-search-set '(#\- #\+) string)))
          (if index
	      (let ((token (read-from-string string nil :eof :end index)))
		   (if (or (equal '> token)
			   (equal '< token)
			   (integerp token)
		       )
		       (if (string-equal "*"
				 (subseq (the string string) (+ index 1))
			   )
			   (list :Version
				 (if (or (equal token '>) (equal token 0))
				     :newest
				     (if (equal token '<)
					 :oldest
					 token
				     )
				 )
				 (if (string-reverse-search-set '(#\+) string)
				     :after
				     :before
				 )
			   )
			   (maybe-junk-version string junk-allowed)
		       )
		       (maybe-junk-version string junk-allowed)
		   )
	      )
	      (maybe-junk-version string junk-allowed)
	  )
     )
    )
    (JUNK-ALLOWED
      (LM-PARSE-VERSION
	(NSUBSTRING STRING 0. (- (LENGTH STRING) 1.)) JUNK-ALLOWED))
    (T (FERROR 'PATHNAME-PARSE-ERROR "Invalid version spec ~S" STRING))))


(DEFMETHOD (NEW-LM-PARSING-MIXIN :PARSE-VERSION-SPEC) (SPEC)
  (COND
    ((STRINGP SPEC) (LM-PARSE-VERSION SPEC SELF T))
    ((OR (FIXNUMP SPEC) (MEMBER SPEC '(NIL :UNSPECIFIC :WILD :NEWEST :OLDEST) :TEST #'EQ)) SPEC)
    ;;; Added by JPR.
    ((consp version) version)
    (T :NEWEST)))

(defun match-to-subdirs (pattern sample wild-any wild-one)
"Backtracks over pattern and sample (if there are :wild-inferiors specs)
 trying to match them."
;;; Written by JPR.
  (if pattern
      (if (equal (first pattern) :wild)
	  (equal (length sample) 1)
	  (if (equal (first pattern) :wild-inferiors)
	      (or (and (not (rest pattern)) ;;; No other subdirs to check
		       sample)
		  (match-to-subdirs (rest pattern) sample wild-any wild-one)
		  (let ((this-one-wild
			  (match-to-subdirs pattern (rest sample) wild-any
					    wild-one)))
		    (if this-one-wild
			(cons (first sample)
			      (if (listp this-one-wild) this-one-wild nil))
			nil)))
	      (let ((specs (pathname-component-match
			     (first pattern) (first sample)
			     wild-any wild-one t)))
		(if specs
		    (let ((rest (match-to-subdirs
				  (rest pattern) (rest sample) wild-any
				  wild-one)))
		      (if rest (append specs rest) nil))
		    nil))))
      (equal nil sample)))

(DEFUN PATHNAME-COMPONENT-MATCH (PATTERN SAMPLE WILD-ANY WILD-ONE &OPTIONAL RETURN-SPECS-FLAG &AUX SPECS)
 ;; If RETURN-SPECS-FLAG, we return a list of the chars or strings
 ;; that matched the wildcards, in the order they appeared,
 ;; or T if no wildcards but the pattern does match.
  (COND
    ((or (EQ PATTERN :WILD) (EQ PATTERN :WILD-inferiors))
     (IF RETURN-SPECS-FLAG
	 (IF (CONSP SAMPLE) SAMPLE (LIST SAMPLE))
	 T))
    ((SYMBOLP PATTERN) (EQ PATTERN SAMPLE))
    ((NUMBERP PATTERN) (EQ PATTERN SAMPLE))
    ;;; Added by JPR.
    ((and (consp pattern) (equal (first pattern) :version))
     (ferror nil "Cannot pathname match with relative version."))
    ((CONSP PATTERN)
     ;;; Modded here by JPR. used to be the commented out bit.
;     (IF (AND (CONSP SAMPLE) (= (LENGTH PATTERN) (LENGTH SAMPLE)))
;       (LOOP FOR P IN PATTERN FOR S IN SAMPLE DO
;	  (LET ((TEM (PATHNAME-COMPONENT-MATCH P S WILD-ANY WILD-ONE RETURN-SPECS-FLAG)))
;	    (IF (NULL TEM)
;	      (RETURN ()))
;	    (UNLESS (EQ TEM T)
;	      (SETQ SPECS (APPEND SPECS TEM))))
;	  FINALLY (RETURN (OR SPECS T)))
;       (WHEN (STRINGP SAMPLE)
;	 (LET (RESULT)
;	   (SETQ RESULT
;		 (PATHNAME-COMPONENT-MATCH (CAR PATTERN) SAMPLE WILD-ANY WILD-ONE
;					   RETURN-SPECS-FLAG))
;	   (IF (AND RESULT (CDR PATTERN))
;	     (DOLIST (COMPONENT (CDR PATTERN) RESULT)
;	       (IF (NEQ COMPONENT :WILD)
;		 (RETURN ())
;		 (SETQ RESULT T)))
;	     RESULT))))
     (if (consp sample)
	 (let ((specs (or (match-to-subdirs
			    pattern sample wild-any wild-one
			  )
			  (match-to-logical-expr
			    pattern sample wild-any wild-one
			  )
		      )
	       )
	      )
	      (if RETURN-SPECS-FLAG specs t))
	 (or (match-to-logical-expr pattern sample wild-any wild-one)
       (WHEN (STRINGP SAMPLE)
	 (LET (RESULT)
	   (SETQ RESULT
		 (PATHNAME-COMPONENT-MATCH
		   (CAR PATTERN) SAMPLE WILD-ANY WILD-ONE
		   RETURN-SPECS-FLAG)
	   )
	   (IF (AND RESULT (CDR PATTERN))
	     (DOLIST (COMPONENT (CDR PATTERN) RESULT)
	       (IF (NEQ COMPONENT :WILD)
		 (RETURN ())
		 (SETQ RESULT T)))
	     RESULT))))))
    ((CONSP SAMPLE)
     (IF (EQl (LENGTH SAMPLE) 1.)
	 (PATHNAME-COMPONENT-MATCH
	   PATTERN (CAR SAMPLE) WILD-ANY WILD-ONE RETURN-SPECS-FLAG)
	 (match-to-logical-expr pattern sample wild-any wild-one)
	 ;;; used to be nil
     )
    )
    ((SYMBOLP SAMPLE) NIL)
    (T
     (DO ((P-PTR 0.)
	  (P-NEXT)
	  (P-CHAR WILD-ONE)
	  (S-PTR -1.)
	  (SET (LIST WILD-ANY WILD-ONE)))
	 (NIL)
       (SETQ P-NEXT (STRING-SEARCH-SET SET PATTERN P-PTR))
       (COND
	 ((>= S-PTR (LENGTH SAMPLE))
	  (LET ((OLD-S-PTR S-PTR))
	    (SETQ S-PTR
		  (SEARCH (THE STRING PATTERN) (THE STRING (STRING SAMPLE))
			  :FROM-END T :START1 P-PTR :START2 S-PTR :TEST #'CHAR-EQUAL))
	    (WHEN RETURN-SPECS-FLAG
	      (PUSH (SUBSEQ SAMPLE OLD-S-PTR S-PTR) SPECS))))
	 ((EQ P-CHAR WILD-ONE)
	  (AND RETURN-SPECS-FLAG (>= S-PTR 0.) (PUSH (AREF SAMPLE S-PTR) SPECS))
	  (SETQ S-PTR
		(AND
;;; rla 8/12/86 - note that this is an obsolete positional calling sequence start1 start2 end1 end2 
;;;		 (STRING-EQUAL SAMPLE PATTERN (1+ S-PTR) P-PTR
;;;			       (+ 1. S-PTR (- (OR P-NEXT (LENGTH PATTERN)) P-PTR)) P-NEXT)
		 (STRING-EQUAL SAMPLE PATTERN :start1 (1+ S-PTR) :start2 P-PTR
			       :end1 (+ 1. S-PTR (- (OR P-NEXT (LENGTH PATTERN)) P-PTR)) :end2 P-NEXT)
		 (1+ S-PTR))))
	 ((NULL P-NEXT)
	  ;; Stuff at end following a star =>
	  ;;  win if tail of rest of string matches that stuff.
	  (LET ((OLD-S-PTR S-PTR))
	    (SETQ S-PTR
		  (SEARCH (THE STRING PATTERN) (THE STRING (STRING SAMPLE)) :START1 P-PTR
			  :START2 S-PTR :TEST #'CHAR-EQUAL :FROM-END T))
	    (WHEN RETURN-SPECS-FLAG
	      (PUSH (SUBSEQ SAMPLE OLD-S-PTR S-PTR) SPECS))))
	 (T
	  (LET ((OLD-S-PTR S-PTR))
	    (SETQ S-PTR
		  (SEARCH (THE STRING PATTERN) (THE STRING (STRING SAMPLE)) :START2 S-PTR
			  :START1 P-PTR :END1 P-NEXT :TEST #'CHAR-EQUAL))	;MBC 8-19-86
	    (WHEN RETURN-SPECS-FLAG
	      (PUSH (SUBSEQ SAMPLE OLD-S-PTR S-PTR) SPECS)))))
       (UNLESS S-PTR
	 (RETURN ()))
       (INCF S-PTR (- (OR P-NEXT (LENGTH PATTERN)) P-PTR))
       (UNLESS P-NEXT
	 (RETURN (AND (= S-PTR (LENGTH SAMPLE)) (OR (NREVERSE SPECS) T))))
       (SETQ P-CHAR (AREF PATTERN P-NEXT))
       (SETQ P-PTR (1+ P-NEXT))))))


;;; By JPR
(defun cons-new (x list &key (test #'eql))
  (if (member x list :test test)
      list
      (cons x list)))

;;; By JPR
(defun wild-inferiors-lookup (node existing-dirs path)
  (if (rest path)
      (union existing-dirs
	     (lookup-subdirectories node (rest path)))
      (cons-new node (lookup-subdirectories node path))))


;;; By JPR
(defun dirs-or-maybe-error (directories ok-if-not-there)
  (if directories directories
      (if (not ok-if-not-there)
	  (lm-signal-error 'directory-not-found))))


(defun file-versions-match (wild-version relative-version file-version)
  (or (and (equal (third wild-version) :after)
	   relative-version
	   (> file-version relative-version)
      )
      (and (equal (third wild-version) :before)
	   relative-version
	   (< file-version relative-version)
      )
  )
)


(defun wild-versions-match (dir file new-file version)
"True if relativised wild versions match between New-File and Version or if
 New-File matches any logical expression syntax specified in Version.
"
  ;;; By JPR.
  (if (equal :version (first version))
      (let ((relative-file
	      (lookup-file
		dir (file-name file) (file-type file) (second version)
	      )
	    )
	   )
	   (and new-file
		(file-versions-match
		  version (and relative-file (file-version relative-file))
		  (file-version file)
		)
	   )
      )
      (match-to-logical-version-expr dir file new-file version)
  )
)

;;;Added by RDA 21-Jun-91 to fix problem with dual allocations after expunges.
(DEFUN LMFS-EXPUNGE-DIRECTORY (DIRECTORY NAME TYPE VERSION)
  (%STORE-CONDITIONAL (LOCF DIRECTORY) :UNSPECIFIC (QUOTE NIL))
  (%STORE-CONDITIONAL (LOCF NAME) :UNSPECIFIC :WILD)
  (%STORE-CONDITIONAL (LOCF TYPE) :UNSPECIFIC :WILD)
  (%STORE-CONDITIONAL (LOCF VERSION) :UNSPECIFIC :WILD)
  (LET ((RESULTING-BLOCKS-FREED 0.)
	DIRECTORY-FILEs)			;RDA
    (DOLIST (FILE (LOOKUP-FILES DIRECTORY NAME TYPE VERSION :DELETED))
      (WHEN (FILE-DELETED? FILE)
	;;RDA: Keep a list, not just one since LOOKUP-FILES may return
	;;more than one directory.
;	(SETQ DIRECTORY-FILE (FILE-DIRECTORY FILE))
	(pushnew (FILE-DIRECTORY FILE) DIRECTORY-FILEs)
	(LMFS-EXPUNGE-FILE FILE)
	(INCF RESULTING-BLOCKS-FREED (FILE-NPAGES FILE))))
    (dolist (DIRECTORY-FILE DIRECTORY-FILEs)	;RDA: Make this a loop
      (WRITE-DIRECTORY-FILES DIRECTORY-FILE))
    RESULTING-BLOCKS-FREED))

(DEFUN LOOKUP-DIRECTORY-FILES (DIR NAME TYPE VERSION DELETED?)
  (LET ((FILES (QUOTE NIL)))
    (DOLIST (FILE (READ-DIRECTORY-FILES DIR))
      (IF (AND (PATHNAME-COMPONENT-MATCH NAME (FILE-NAME FILE) #\* #\? ())
	       (PATHNAME-COMPONENT-MATCH TYPE (FILE-TYPE FILE) #\* #\? ())
	       (OR (NOT (NUMBERP VERSION))
		   (NOT (PLUSP VERSION))
		   (= VERSION (FILE-VERSION FILE))
		   
	       )
	  )
	  (IF (OR DELETED? (LMFS-CLOSED-FILE? FILE))
	      (PUSH FILE FILES))))
    (SETQ FILES (NREVERSE FILES))
    (IF (OR (EQ VERSION :WILD) (AND (NUMBERP VERSION) (> VERSION 0.)))
      FILES
      ;;; If and Then part added by JPR.
      (if (and (consp version)
	       (or (fboundp (first version))
		   (equal :version (first version))
	       )
	  )
	  (LOOP FOR FILE IN FILES AS NEW-FILE =
		(LOOKUP-FILE DIR (FILE-NAME FILE) (FILE-TYPE FILE)
			     (file-version file)
			     () () () DELETED?)
		WHEN (wild-versions-match dir file new-file version)
		COLLECT NEW-FILE)
	  (LOOP FOR FILE IN FILES BY #'NEXT-GENERIC-FILE AS NEW-FILE =
		(LOOKUP-FILE DIR (FILE-NAME FILE) (FILE-TYPE FILE) VERSION
			     () () () DELETED?)
		WHEN NEW-FILE
		COLLECT NEW-FILE)))))


(DEFUN LOOKUP-SUBDIRECTORY (NODE SUBPATH OK-IF-NOT-THERE)
  (if (member :wild-inferiors subpath)
      (lookup-subdirectories node subpath ok-if-not-there)
      (IF (NULL (CDR SUBPATH))
	  (LOOKUP-SUBDIRECTORY-STEP NODE (CAR SUBPATH) OK-IF-NOT-THERE)
	  (LOOKUP-SUBDIRECTORY (LOOKUP-SUBDIRECTORY-STEP NODE (CAR SUBPATH) OK-IF-NOT-THERE)
			       (CDR SUBPATH) OK-IF-NOT-THERE))))

(defun lookup-subdirectories (node path &optional (ok-if-not-there t))
  (let ((directories
	  (mapcan #'(lambda (file)
		      (cond
			((not (directory? file)) (quote nil))
			((eq path :wild) (list file))
			((pathname-component-match
			   (car path) (directory-name file) #\* #\? nil)
			 (if (and (null (rest path))
				  ;;; Modded here by JPR.
				  (not (equal :wild-inferiors (first path))))
			     (list file)
			     ;;; Modded here by JPR.
			     (if (equal :wild-inferiors (first path))
				 (wild-inferiors-lookup
				   file (lookup-subdirectories file path) path)
				 (lookup-subdirectories file (rest path)))))
			(t (quote nil))))
		  (read-directory-files node))))
    ;;; Modded here by JPR.
    (if (equal :wild-inferiors (first path))
	(let ((maybe-match (pathname-component-match
			     (second path) (directory-name node) #\* #\? nil)))
	  (if maybe-match
	      (wild-inferiors-lookup node directories (rest path))
	      (if (second path) ;; dir specialisations
		  (dirs-or-maybe-error directories ok-if-not-there)
		  (cons-new node directories))))
	(dirs-or-maybe-error directories ok-if-not-there))))


;-------------------------------------------------------------------------------

;;; By JPR.
(defun get-function-name (spec)
"Given a spec, which could be something like (AND ...), in which case it
 returns AND, or \"(and ....)\" in which case again it returns AND.  If the
 it can't get an fbound symbol out of a list like this it returns nil.
"
  (typecase (first spec)
    (cons (first spec))
    (string (let ((form (catch-error (read-from-string (first spec)) nil)))
		 (and (symbolp form) (fboundp form))
	    )
    )
    (symbol (if (fboundp (first spec)) (first spec) nil))
    (otherwise nil)
  )
)

(DEFMETHOD (PATHNAME :PARSE-NAME-SPEC) (SPEC)
  (COND
    ((STRINGP SPEC) (SEND SELF :PARSE-COMPONENT-SPEC SPEC))
    ((AND (CONSP SPEC) (STRINGP (CAR SPEC)) (NULL (CDR SPEC)))
     (SEND SELF :PARSE-COMPONENT-SPEC (CAR SPEC)))
    ;;; Added by JPR to support logical expressions.
    ((and (consp spec)
	  (let ((fn (get-function-name spec)))
	       (if fn
		   (cons fn
			 (mapcar #'(lambda (x) (send self :parse-name-spec x))
				 (rest spec)
			 )
		   )
		   nil
	       )
	  )
     )
    )
    ((MEMBER SPEC '(NIL :UNSPECIFIC :WILD) :TEST #'EQ) SPEC)
    (T "FOO") ;;; I think that this is bogus, but it is what was there before.
    ;;; JPR.
  )
)

;;; By JPR.
(defun match-to-logical-expr (a b wild-any wild-one)
"This is called from Pathname-Component-Match.  A and B are sample/pattern.
 Given a sample such as \"FOO\" it tries to match it to a logical expression
 such as (And \"F*\" (Not \"Frob\")).  It returns T if it matches, otherwise
 nil.
"
  (if (consp a)
      (if (typep b 'string)
	  (eval (cons (first a) ;;; Eval because of special forms....
		      (mapcar #'(lambda (item)
				  (pathname-component-match
				    item b wild-any wild-one
				  )
				)
				(rest a)
		      )
		)
	  )
	  nil
      )
      (eval (cons (first b) ;;; Eval because of special forms.. bletch...
		  (mapcar #'(lambda (item)
			      (pathname-component-match
				a item wild-any wild-one
			      )
			    )
			    (rest b)
		  )
	    )
      )
  )
)


;;; By JPR.
(defun parse-logical-expression
       (form charset self junk-allowed unquoted-is-okay)
"Parses a logical expression out of a pathname component.  Is called by
 Name-Validation.  Given an expression of the form (f x y) it parses all
 expressions x, y, which are conses (other logical expressions) and
 Name-Validates any x, y, which are symbols.  Thus (And foo bar) ->
 (And \"FOO\" \"BAR\").
"
  (typecase form
    (cons
      (if (symbolp (first form))
	  (cons (first form)
		(mapcar #'(lambda (x)
			    (parse-logical-expression
			      x charset self junk-allowed unquoted-is-okay
			    )
			  )
			  (rest form)
		)
	  )
	  (ferror nil "Cannot parse the expression ~S" form)
      )
    )
    (symbol (name-validation (symbol-name form) nil charset self junk-allowed
			     unquoted-is-okay
	    )
    )
    (otherwise (ferror nil "Cannot parse the expression ~S" form))
  )
)


(DEFUN NAME-VALIDATION (STRING VALID-LENGTH CHARSET SELF &OPTIONAL JUNK-ALLOWED UNQUOTED-IS-OKAY)
  (let ((error-p (not (catch-error (UNQUOTE-DELIMITERS-OR-ERROR SELF STRING UNQUOTED-IS-OKAY) nil))))
    ;;; watch out for unreasonable chars due to logical expression.
  (SETQ STRING (UNQUOTE-DELIMITERS-OR-ERROR SELF STRING t))
  (when VALID-LENGTH
    (WHEN (AND (STRINGP STRING) (> (LENGTH STRING) VALID-LENGTH))
      (SETQ STRING (NSUBSTRING STRING 0. VALID-LENGTH))))
  (UNLESS (or JUNK-ALLOWED (not error-p) (not (catch-error (read-from-string string) nil)))
    (WHEN CHARSET
      (WHEN (AND (STRINGP STRING) (STRING-SEARCH-NOT-SET CHARSET STRING))
	(FERROR 'PATHNAME-PARSE-ERROR "Invalid character in spec ~s" STRING))))
;;; The following expression added by JPR instead of just STRING.
  (let ((form (catch-error (read-from-string string) nil)))
       (if (consp form)
	   (parse-logical-expression
	     form charset self junk-allowed unquoted-is-okay
	   )
	   STRING))))


;;; By JPR.
(defun parse-logical-expression-for-version (form junk-allowed)
"Parses a logical expression out of a pathname component.  Is called by
 Name-Validation.  Given an expression of the form (f x y) it parses all
 expressions x, y, which are conses (other logical expressions) and
 lm-parse-versions any x, y, which are not.  Thus (And >-* (Not 42)) ->
 (And (:version :before :newest) (not 42)).
"
  (typecase form
    (cons
      (if (symbolp (first form))
	  (cons (first form)
		(mapcar #'(lambda (x)
			    (parse-logical-expression-for-version
			      x junk-allowed
			    )
			  )
			  (rest form)
		)
	  )
	  (ferror nil "Cannot parse the expression ~S" form)
      )
    )
    (otherwise (lm-parse-version (format nil "~S" form) nil junk-allowed))
  )
)

;;; By JPR.
(defun match-to-logical-version-expr (dir file new-file version)
"Called by Wild-Versions-Match.  When files are being matched by the file
 system, If a logical expression is found in the version then this function
 is invoked.  It checks the file denoted by New-File's version against the
 expression denoted by Version.
"
  (let ((body (mapcar #'(lambda (item)
			  (if (consp item)
			      (wild-versions-match dir file new-file item)
			      (or (equal :wild item)
				  (let ((denoted-file
					  (catch-error
					    (lookup-file
					      dir (file-name file)
					      (file-type file)
					      item
					    )
					    nil
					  )
					)
				       )
				       (and denoted-file
					    (equalp (file-version file)
						    (file-version denoted-file)
					    )
				       )
				  )
			      )
			  )
			)
			(rest version)
	      )
	)
       )
       (eval (cons (first version) body))
       ;;; Eval because of special forms.. bletch...
  )
)


;;; Put here so that Logical pathnames are able to understand the new logical
;;; expression pathname stuff.  Without this change Logical pathnames cannot
;;; understand the logical expression syntax.

(setq LOGICAL-TABLE
   '(((#\:) NIL (#\SPACE) LM-PARSE-DEVICE); Device
     ((#\;) NIL (#\SPACE #\:) LM-PARSE-DIRECTORY); Directory
     ((#\. #\#) T (#\SPACE #\: #\;) LM-PARSE-NAME-COMPONENT); Name
     ((#\. #\#) T (#\. #\SPACE) LM-PARSE-NAME-COMPONENT); Type
     (NIL T (#\. #\#) LM-PARSE-VERSION)))

(setq LM-TABLE
   '(((#\:) NIL (#\SPACE) LM-PARSE-DEVICE); Device
     ((#\<) NIL (#\SPACE #\:) NIL NIL T)
     ((#\>) NIL (#\<) LM-PARSE-DIRECTORY); Twenex style directory
     ((#\;) NIL (#\SPACE #\: #\<) LM-PARSE-DIRECTORY); Directory
     ((#\. #\#) T (#\SPACE #\: #\; #\>) LM-PARSE-NAME-COMPONENT); Name
     ((#\. #\#) T (#\.) LM-PARSE-NAME-COMPONENT); Type
;     ((#\SPACE) T (#\. #\#) LM-PARSE-VERSION)))   ;;; JPR
     (NIL T (#\. #\#) LM-PARSE-VERSION)))  ;;; JPR

;;; Patch Make-Pathname-1

(defun list-ok-string-equal (a b)
  (cond ((consp a)
	 (if (consp b)
	     (and (equal (first a) (first b))
		  (list-ok-string-equal (rest a) (rest b))
	     )
	     nil
	 )
	)
        ((consp b) nil)
	(t (string-equal a b))
  )
)

(defun list-ok-string= (a b)
  (cond ((consp a)
	 (if (consp b)
	     (and (equal (first a) (first b))
		  (list-ok-string= (rest a) (rest b))
	     )
	     nil
	 )
	)
        ((consp b) nil)
	(t (string= a b))
  )
)

(DEFUN MAKE-PATHNAME-1 (&REST OPTIONS &KEY &ALLOW-OTHER-KEYS STARTING-PATHNAME
			(PARSING-PATHNAME STARTING-PATHNAME)     ;12-17-87 DAB Removed &optional - not needed.
			(HOST NIL HOST-P) (VERSION NIL VERSION-P)
			(ORIGINAL-TYPE NIL ORIGINAL-TYPE-P)
			&AUX DEVICE-P DIRECTORY-P NAME-P TYPE-P DEVICE DIRECTORY
  NAME TYPE CANONICAL-TYPE)
  (LOOP FOR (KEYWORD VALUE) ON OPTIONS BY 'CDDR DO
     (CASE KEYWORD
       (:NAME (UNLESS NAME-P
		(SETQ NAME VALUE
		      NAME-P T)))
       (:RAW-NAME (UNLESS NAME-P
		    (SETQ NAME VALUE
			  NAME-P :RAW)))
       (:DIRECTORY (UNLESS DIRECTORY-P
		     (SETQ DIRECTORY VALUE
			   DIRECTORY-P T)))
       (:RAW-DIRECTORY (UNLESS DIRECTORY-P
			 (SETQ DIRECTORY VALUE
			       DIRECTORY-P :RAW)))
       (:DEVICE (UNLESS DEVICE-P
		  (SETQ DEVICE VALUE
			DEVICE-P T)))
       (:RAW-DEVICE (UNLESS DEVICE-P
		      (SETQ DEVICE VALUE
			    DEVICE-P :RAW)))
       (:TYPE
	(UNLESS TYPE-P
	  (IF (AND (SYMBOLP VALUE) (NOT (MEMBER VALUE '(NIL :UNSPECIFIC) :TEST #'EQ)))
	    (SETQ CANONICAL-TYPE VALUE
		  TYPE-P :CANONICAL)
	    (SETQ TYPE VALUE
		  TYPE-P T))))
       (:RAW-TYPE (UNLESS TYPE-P
		    (SETQ TYPE VALUE
			  TYPE-P :RAW)))
       (:CANONICAL-TYPE (UNLESS TYPE-P
			  (SETQ CANONICAL-TYPE VALUE
				TYPE-P :CANONICAL)))
       ;; All keywords that do NOT require special decoding must go here.
       ;; All keywords that do NOT require special decoding must go here.
       ((:HOST :VERSION :STARTING-PATHNAME :PARSING-PATHNAME :ORIGINAL-TYPE :DEFAULTS NIL) NIL)
       (T (FERROR () "Unknown keyword ~S to MAKE-PATHNAME or :NEW-PATHNAME." KEYWORD))))
  (OR HOST-P (SETQ HOST (PATHNAME-HOST STARTING-PATHNAME)))
  (SETQ HOST (GET-PATHNAME-HOST HOST))
  ;; Turn a specified canonical type into a string (in standard case).
  (IF (EQ TYPE-P :CANONICAL)
    (MULTIPLE-VALUE-BIND (PREFERRED ALL)
      (DECODE-CANONICAL-TYPE CANONICAL-TYPE (SEND HOST :SYSTEM-TYPE))
      (SETQ TYPE
	      (LET ((case-sensitive (send host :case-sensitive-when-hashing)))	;3.18.87
		(UNLESS ORIGINAL-TYPE-P
		  (SETQ ORIGINAL-TYPE (PATHNAME-TYPE STARTING-PATHNAME)))
		(IF ;;; Changed by JPR from just (member...
		    ;;; so as to allow for (and "foo" "bar"...
		    (MEMBER (convert-solid-case ORIGINAL-TYPE) ALL
			    :TEST (if case-sensitive #'list-ok-string= #'list-ok-string-equal))
		    (convert-solid-case ORIGINAL-TYPE)	;3.18.87
		    PREFERRED)))))
  (IF (EQ (PATHNAME-HOST STARTING-PATHNAME) HOST)
    (PROGN
      (OR DEVICE-P (SETQ DEVICE (PATHNAME-RAW-DEVICE STARTING-PATHNAME)
			 DEVICE-P :RAW))
      (OR DIRECTORY-P
	 (SETQ DIRECTORY (PATHNAME-RAW-DIRECTORY STARTING-PATHNAME)
	       DIRECTORY-P :RAW))
      (OR NAME-P (SETQ NAME (PATHNAME-RAW-NAME STARTING-PATHNAME)
		       NAME-P :RAW))
      (OR TYPE-P (SETQ TYPE (PATHNAME-RAW-TYPE STARTING-PATHNAME)
		       TYPE-P :RAW)))
    ;; Hosts don't match; must convert to standard syntax and reparse.
    (PROGN
      (OR DEVICE-P (SETQ DEVICE (PATHNAME-DEVICE STARTING-PATHNAME)))
      (OR DIRECTORY-P (SETQ DIRECTORY (PATHNAME-DIRECTORY STARTING-PATHNAME)))
      (OR NAME-P (SETQ NAME (PATHNAME-NAME STARTING-PATHNAME)))
      (OR TYPE-P (SETQ TYPE (PATHNAME-TYPE STARTING-PATHNAME)))))
  (OR VERSION-P (SETQ VERSION (PATHNAME-RAW-VERSION STARTING-PATHNAME)))
  ;; The new fields are parsed only once to save time, consing, and possible errors
  ;; due to incompatible fields in different types of pathnames.
    (AND (NEQ DEVICE-P :RAW) (SETQ DEVICE (FUNCALL PARSING-PATHNAME :PARSE-DEVICE-SPEC DEVICE)))
    (AND (NEQ DIRECTORY-P :RAW)
       (SETQ DIRECTORY (FUNCALL PARSING-PATHNAME :PARSE-DIRECTORY-SPEC DIRECTORY)))
    (AND (NEQ NAME-P :RAW) (SETQ NAME (FUNCALL PARSING-PATHNAME :PARSE-NAME-SPEC NAME)))
    (AND (NEQ TYPE-P :RAW) (SETQ TYPE (FUNCALL PARSING-PATHNAME :PARSE-TYPE-SPEC TYPE)))
    (SETQ VERSION (FUNCALL PARSING-PATHNAME :PARSE-VERSION-SPEC VERSION))
    (if (stringp directory) (setf directory (list directory)))	;3.08.88
  (MAKE-PATHNAME-INTERNAL (SEND STARTING-PATHNAME :RAW-QUOTED-STRING) HOST DEVICE DIRECTORY NAME
			  TYPE VERSION))


;-------------------------------------------------------------------------------

;;; Zwei changes...

(defun dir-string (path)
  (let ((str (send (fs:make-pathname
		     :host (send path :host) :directory (send path :directory)
		     :name nil :type nil :version nil)
		   :string-for-printing)))
       (string-trim '(#\space #\tab)
		    (subseq (the string str)
			    (+ 1 (position #\: (the string str)
					   :test #'char-equal))))
  )
)


(defwrapper (fs:pathname :string-for-dired) (ignore . body)
 `(locally (declare (special *current-dired-interval*))
    (if (boundp '*current-dired-interval*)
	(let ((path (first (get *current-dired-interval* 'zwei:pathname-list))))
	     (if (and path
		      (or (member (send path :Directory) '(:wild :root))
			  (member :wild (send path :directory))
			  (member :wild-inferiors (send path :directory))
		      )
		 )
		 (string-append (dir-string self) (progn ,@body))
		 (progn ,@body)
	     )
	)
	,@body
    )
  )
)

(DEFWRAPPER (HOST-PATHNAME :STRING-FOR-DIRED) (IGNORE . BODY)
   `(locally (declare (special *current-dired-interval* ))
	     (if (boundp '*current-dired-interval*)
		 (progn ,@body)
		 (CACHE-IN-VARIABLE STRING-FOR-DIRED ,@BODY))))


(advise zwei:directory-edit-revert :around :better-printer nil
  (let ((*current-dired-interval* (first arglist)))
       (declare (special *current-dired-interval*))
       :do-it
  )
)

sys:
(defun sys:number-into-array
       (array n &optional (radix *print-base*) (at-index 0) (min-columns 0))
  "Store a printed representation of the fixnum N into ARRAY starting at AT-INDEX.
The index of the first element of ARRAY not stored into and
the new value of ARRAY are returned. (ARRAY is munged)
RADIX, which defaults to *PRINT-BASE*, is used for conversion.
Leading spaces are used to fill up at least MIN-COLUMNS columns."
  (declare (values new-index array))
  (multiple-value-bind (quot digit)
    (truncate n radix)
    (if (zerop quot)
      (dotimes (i (1- min-columns))
	(if (>= at-index (length (the string array)))
	    (vector-push-extend #\space array 10)
	    nil)
	(setf (aref array at-index) #\SPACE)
	(setq at-index (1+ at-index)))
      (setq at-index
	    (number-into-array array quot radix at-index (1- min-columns))))
    (if (>= at-index (length (the string array)))
	(vector-push-extend #\space array 10)
	nil)
    (setf (aref array at-index) (if (< digit 10)
				  (+ digit #\0)
				  (+ digit #\A -10)))
    (values (1+ at-index) array)))

;-------------------------------------------------------------------------------

(defun stringise (something)
  ;;RDA: Add NULL check
  (if something
      (with-output-to-string (stream)
	(fs:lm-print-component something stream)
	)
      ""
      )
  )

(DEFUN LMFS-COMPLETE-PATH (DIR NAME TYPE DEFAULT-NAME DEFAULT-TYPE OPTIONS)
  "Implements the :COMPLETE-STRING message for pathnames."
;  (%STORE-CONDITIONAL (LOCF DIR) :WILD "*")	;Can't ever get :WILD here.  2.18.87 MBC
  (MULTIPLE-VALUE-BIND (NEW-DIR NEW-NAME NEW-TYPE NIL COMPLETION)
      ;;; Changed here by JPR.
      (COMPLETE-PATH DIR (stringise NAME) (stringise TYPE)
		     (stringise DEFAULT-NAME) (stringise DEFAULT-TYPE))
    (COND
      ((or (MEMBER :WRITE OPTIONS :TEST #'EQ)
	   (MEMBER :NEW-OK OPTIONS :TEST #'EQ))
       (unless new-name (SETQ NEW-NAME (OR NAME DEFAULT-NAME)))	;2.18.87
       (unless new-type (SETQ NEW-TYPE (OR TYPE DEFAULT-TYPE "LISP")))	;don't allow NIL 2.18.87
       (unless completion (setf completion :NEW)))	;Can't fail looking for NEW-OK
      ((eq completion :NEW)			;must be a :READ or :OLD request
       (setf new-type nil)			;assume partial completion on NAME, so no TYPE
       (setf completion NIL)))			;So :NEW is failure
    (VALUES NEW-DIR NEW-NAME NEW-TYPE COMPLETION)))


(DEFUN TRY-COMPLETE-DIRECTORY (DIRECTORY &AUX COMPLETION-SO-FAR)
  (DO ((DIRLEFT DIRECTORY (CDR DIRLEFT)))
      ((NULL DIRLEFT)
       (VALUES COMPLETION-SO-FAR T))
    (LET* ((DIR-COMPONENT (CAR DIRLEFT))
	   (COMPLETED-DIRECTORY
	    (OR (LOOKUP-DIRECTORY (APPEND COMPLETION-SO-FAR (LIST DIR-COMPONENT)) T)
	       (MULTIPLE-VALUE-BIND (TEM NIL DIRECTORY-COMPLETED)
		 (ZWEI::COMPLETE-STRING DIR-COMPONENT
					(IF COMPLETION-SO-FAR
					  (MAKE-SUBDIRECTORY-ALIST COMPLETION-SO-FAR)
					  (MAKE-DIRECTORY-ALIST))
					'(#\-) T)
		 (AND DIRECTORY-COMPLETED
		    (LOOKUP-DIRECTORY (APPEND COMPLETION-SO-FAR (LIST TEM))))))))
      (IF COMPLETED-DIRECTORY
	(SETQ COMPLETION-SO-FAR
	      (APPEND COMPLETION-SO-FAR
		      ;;; Changed here by JPR.
		      (LIST (if (consp COMPLETED-DIRECTORY)
				;;; I don't think this is the best thing.
				(DIRECTORY-NAME
				  (second COMPLETED-DIRECTORY))
;				(mapcar #'directory-name COMPLETED-DIRECTORY)
				(DIRECTORY-NAME COMPLETED-DIRECTORY)))))
	(RETURN
	 (VALUES (APPEND COMPLETION-SO-FAR DIRLEFT)
		 (NOT (EQUAL (APPEND COMPLETION-SO-FAR DIRLEFT) DIRECTORY))))))))

(DEFUN COMPLETE-STRINGS-OR-KEYWORDS (SPEC DEFAULT-SPEC ALIST-OR-ARRAY
				     &AUX COMPLETED-SPEC SPEC-COMPLETIONS SPEC-COMPLETED 
				     Ambiguous)
  (DECLARE (VALUES COMPLETED-SPEC ALL-COMPLETIONS SPEC-COMPLETED-FLAG))
  
  (BLOCK ()
    (LET ((A-LIST (IF (TYPEP ALIST-OR-ARRAY 'ARRAY)	;MAKE SURE ALIST IS A LIST NOT AN ARRAY!
		      (CONCATENATE 'LIST ALIST-OR-ARRAY)
		      ALIST-OR-ARRAY)))
      (UNLESS SPEC
	(IF (EVERY #'(LAMBDA (X) (STRING-EQUAL (CAR X) (CAAR A-LIST)))
		   (CDR A-LIST))
	    (RETURN (VALUES (CAAR A-LIST) A-LIST ()))	; NO COMPLETION NECCESSARY IF ALL THE SAME
	    (SETQ SPEC DEFAULT-SPEC)))		; USE THE DEFAULT-SPEC, BECAUSE SPEC IS NIL.
      
      (COND
	((EQ SPEC :WILD)
	 (IF (EQl (LENGTH A-LIST) 1)		;ONLY ONE THING TO MATCH
	     (SETQ COMPLETED-SPEC (CAAR A-LIST))	;THEN USE THAT STRING.
	     (SETQ COMPLETED-SPEC "*"))		;OTHERWISE USE "*".
	 (SETQ SPEC-COMPLETIONS A-LIST		; USE THE WHOLE ALIST,EVERY THING MATCHES A :WILD
	       
	       SPEC-COMPLETED T))
	((EQ SPEC :UNSPECIFIC)
	 (SETQ COMPLETED-SPEC SPEC
	       SPEC-COMPLETIONS
	       (REMOVE-IF-NOT			; RETURN ALIST WITH KEY = ""
		 (FUNCTION (LAMBDA (X)
			     (STRING-EQUAL (CAR X) ""))) A-LIST)
	       SPEC-COMPLETED ()))
	((NULL SPEC))				; DEFAULT-SPEC IS NIL, NO MATCHES
	(T
	 (MULTIPLE-VALUE-SETQ (COMPLETED-SPEC SPEC-COMPLETIONS SPEC-COMPLETED nil Ambiguous)	;2.18.87 MBC
	   ;;; Changed by JPR
	   (my-COMPLETE-STRING SPEC ALIST-OR-ARRAY '(#\-)))
	 (UNLESS SPEC-COMPLETIONS		; NO MATCHES,SO FAR.
	   (IF (OR (STRING-EQUAL SPEC DEFAULT-SPEC) (NULL DEFAULT-SPEC))	;SAME OR NIL
	       (SETQ COMPLETED-SPEC SPEC)
	       (MULTIPLE-VALUE-SETQ (COMPLETED-SPEC SPEC-COMPLETIONS SPEC-COMPLETED)
		 (COMPLETE-STRINGS-OR-KEYWORDS "" () ALIST-OR-ARRAY)))))))
    (RETURN (VALUES COMPLETED-SPEC SPEC-COMPLETIONS (if Ambiguous :ambiguous SPEC-COMPLETED)))	;2.18.87 MBC
    ()))

(defun parse-out-a-component (component)
  (multiple-value-bind (result error-p)
      (multiple-value-list (catch-error (read-from-string component) nil))
    (if error-p
	component
	(if (consp (first result))
	    (cons (first (first result))
		  (mapcar #'symbol-name (rest (first result)))
	    )
	    component
	)
    )
  )
)

(defun my-complete-string (spec alist delims)
  (let ((component (parse-out-a-component spec)))
       (if (consp component)
	   (loop for spec in (rest component)
		 do (multiple-value-bind
		      (completed-spec spec-completions spec-completed
		       nil Ambiguous
		      )
			(zwei::complete-string spec alist delims)
		      (if completed-spec
			  (return completed-spec spec-completions spec-completed
				  nil Ambiguous
			  )
			  nil
		      )
		    )
	   )
	   (zwei::complete-string spec alist delims)
       )
  )
)