;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-


;-------------------------------------------------------------------------------

(defun structurify-listarray (x &optional (depth (array-rank x))
			      (indices nil) (dimensions (array-dimensions x))
			     )
"Given a multi-dimensionsl array returns a list of its elements that could be
used to initialize it."
  (if (equal depth 1)
      (loop for i from 0 below (first dimensions)
	    collect (apply #'aref x (reverse (cons i indices)))
      )
      (loop for j from 0 below (first dimensions)
	    collect (structurify-listarray x (- depth 1) (cons j indices)
					   (rest dimensions)
		    )
      )
  )
)

;-------------------------------------------------------------------------------

(defflavor show-structured-listarray
	   ()
	   (generic-middle-button-mixin auxiliary-data-mixin inspection-data)
  (:documentation
"A perspective to view arrays as structured lists.
"
  )
)

(defmethod (show-structured-listarray :format-concisely) (stream)
"Just prints it out but notes that it is a listarray perspective."
  (format stream "~ as the structured list: ~" (list data t data)
	  (list aux-data t aux-data)
  )
)

(defmethod (show-structured-listarray :generate-item-specialized) (window)
"Makes the inspector items for an array as a structured list."
  (multiple-value-bind (string-list type printer)
      (send window :Object-List aux-data)
    (values string-list type printer 0
	   `(:font fonts:hl12bi
		   :string ,(format nil "~S as a structured list" data)
	    )
    )
  )
)

(defmethod (show-structured-listarray :help) ()
"Gives help when you middle button on an array."
  (format nil "
The inspection pane you just selected is currently displaying the
array ~S as a structured list.  The different dimensions in the array are
shown as nested lists.  The list structure is the same as would be required
to initialize a new instance of the array.  Mousing L2 on it should
show it to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------

(defflavor show-listarray
	   ()
	   (show-structured-listarray)
  (:documentation
"A perspective to view arrays as flattened lists.
"
  )
)

(defmethod (show-listarray :format-concisely) (stream)
"Just prints it out but notes that it is a listarray perspective."
  (if (> (array-rank data) 1)
      (format stream "~ as the flattened list: ~" (list data t data)
	      (list aux-data t aux-data)
      )
      (format stream "~ as the list: ~" (list data t data)
	      (list aux-data t aux-data)
      )
  )
)

(defmethod (show-listarray :generate-item-specialized) (window)
"Makes the inspector items for an array as a list."
  (multiple-value-bind (string-list type printer)
      (send window :Object-List aux-data)
    (values string-list type printer 0
	   `(:font fonts:hl12bi
		   :string ,(format nil "~S as a ~Alist" data
				    (if (> (array-rank data) 1) "flattened " "")
			    )
	    )
    )
  )
)

(defmethod (show-listarray :help) ()
"Gives help when you middle button on an array."
  (format nil "
The inspection pane you just selected is currently displaying the
array ~S as a ~Alist.  Mousing L2 on it should
show it to you in some other way.
"
	  data (if (> (array-rank data) 1) "flattened " "")
  )
)

;-------------------------------------------------------------------------------

(defun fast-parse-host (string)
  (with-timeout (30 nil) (net:parse-host string t))
)

;(Defperspective :host (x show-x)
;  :show-x-type-for-perspective Show-Generic-Object-thing
;  :This-Perspective-Applicable-Function
;    (and (or (symbolp x) (stringp x))
;	 (Fast-Parse-Host x)
;    )
;  :menu-item-name "Host"
;  :Menu-Who-Line-Doc-String
;    "Inspect this symbol/string as the Host object it names."
;  :New-Inspect-Function
;    (allocate-data 'show-generic-object-thing (net:parse-host (string x) t ))
;)

(Defperspective :glp (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-thing
  :This-Perspective-Applicable-Function
    (and (not (hash-table-p x)) (arrayp x) (equal (array-type x) 'art-q-list))
  :menu-item-name "Art-Q-list as a list"
  :Menu-Who-Line-Doc-String
    "Inspect this art-q-list as a list instead of as an array."
  :New-Inspect-Function
    (allocate-data 'show-generic-object-thing (g-l-p x))
)

(Defperspective :Art-Q-List (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :This-Perspective-Applicable-Function
    (and (listp x) (typep (sys:%find-structure-header x) 'array)) 
  :menu-item-name "Art-Q-List"
  :Menu-Who-Line-Doc-String "Show this list as an Art-Q-List."
  :New-Inspect-Function (allocate-data 'show-generic-object-thing
				       (sys:%find-structure-header x)
			)
)

(Defperspective :Listarray (x show-x)
  :show-x-type-for-perspective Show-listarray
  :This-Perspective-Applicable-Function
    (and (arrayp x) (not (equal (array-type x) 'art-q-list)))
  :menu-item-name "Listarray"
  :Menu-Who-Line-Doc-String
    "Inspect this array as (listarray <<array>>)."
  :New-Inspect-Function
    (allocate-data 'Show-Listarray x (listarray x))
  :Priority -1
)

(Defperspective :Structured-Listarray (x show-x)
  :show-x-type-for-perspective show-structured-listarray
  :This-Perspective-Applicable-Function
    (and (arrayp x) (> (array-rank x) 1))
  :menu-item-name "Structured Listarray"
  :Menu-Who-Line-Doc-String
    "Inspect this array as a structured list with array elements represented as nested lists."
  :New-Inspect-Function
    (allocate-data 'Show-Structured-Listarray x (structurify-listarray x))
  :Priority -1
)

;(Defperspective :Pathname (x show-x)
;  :show-x-type-for-perspective show-generic-object-thing
;  :This-Perspective-Applicable-Function
;    (and (stringp x)
;	 (let ((index (position #\: x :Test #'char=))
;	       (length (length x))
;	      )
;	      (and index (> index 0) (not (equal index (- length 1)))
;		   (not (equal #\: (aref x (+ index 1))))
;		   (Fast-Parse-Host (subseq x 0 index))
;		   (catch-error (pathname x) nil)
;	      )
;	 )
;    )
;  :menu-item-name "Pathname"
;  :Menu-Who-Line-Doc-String
;    "Inspect this string as the pathname it names."
;  :New-Inspect-Function
;    (allocate-data 'Show-generic-object-thing (pathname x))
;)

;-------------------------------------------------------------------------------

(defflavor show-string
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:Documentation "A perspective to view strings as lines of text.")
)

(defmethod (show-string :format-concisely) (stream)
"Just prints it out but notes that it is a string perspective."
  (format stream "~ as text"
	  (list data nil (with-output-to-string (str)
			   (print-item-concisely data str)
		         )
          )
  )
)

(defun item-for-leader (item data name indent)
  (let ((pntr (locf (array-leader data item))))
      `((:item1 array-leader-slot ,(list data item)
	  ,(if name
	       #'(lambda (x stream) (ignore x) (princ name stream))
	       #'(lambda (x stream) (format stream "leader ~d" (second x)))
	   )
	)
	(:colon ,indent)
	,(if (%p-contents-safe-p pntr)
	    `(:item1 t ,(array-leader data item))
	    `(:font 1 ,(format nil "#<~A ~O>"
			       (or (nth (%p-data-type pntr) q-data-types)
				   (%p-data-type pntr)
			       )
			       (%p-pointer pntr)
		       )
	     )
	 )
       )
  )
)

(defun (:property array-leader-slot set-function) (item new-value object)
  (store-array-leader new-value object (second (third (second item))))
)

(defprop array-leader-slot t only-when-modify) 

(defun leader-items-of (data &optional names)
  (let ((leader-items (if (array-has-leader-p data)
			  (let ((indent (if names
					    (loop for name in names
						  maximize
						    (length (string name))
					    )
					    8
					)
				)
			       )
			       (loop for item from 0
				     below (array-leader-length data)
				     for name
				     in (or names
					    (make-list
					      (array-leader-length data)
					    )
					)
				     collect (Item-For-Leader
					       item data name (+ 5 indent)
					     )
			       )
			  )
			  nil
		      )
        )
       )
       (if leader-items
	   (append leader-items (list *blank-line-item*))
	   nil
       )
  )
)

(defun show-string-title (data)
  (with-output-to-string (str)
    (catch-error
      (if (stringp data)
	  (print-item-concisely data str)
	  (let ((index (%string-search-char
			 #\newline data 0 (array-active-length data)
		       )
		)
		(max-print-length
		  (min (array-active-length data)
		       (floor (send tv:default-screen :Width)
			      (font-char-width
				(let ((font :Default))
				     (tv:coerce-font font tv:default-screen)
				)
			      )
		       )
		  )
		)
	       )
	       (loop for i from 0 below (min index max-print-length) do
		     (format str "~C" (aref data i))
	       )
	       (if index (format str "...") nil)
	  )
      )
      nil
    )
  )
)

(defmethod (show-string :generate-item) ()
"Makes the inspector items for a string."
  (values (cons *blank-line-item*
		(append (leader-items-of data)
			(mapcar #'list (Split-Into-Lines data))
		)
	  )
     `(:font fonts:hl12bi :String ,(show-string-title data))
  )
)

(defmethod (show-string :help) ()
"Gives help when you middle button on a String."
  (format nil "
The inspection pane you just selected is currently displaying a
String/ART-8B. Mousing L2 on it should show it to you in some other way.
"
  )
)

(Defperspective :String (x show-x)
  :show-x-type-for-perspective show-string
  :This-Perspective-Applicable-Function
    (and (stringp x) (not (typep show-x 'Show-String)))
  :menu-item-name "String as text"
  :Menu-Who-Line-Doc-String
    "Inspect this string as text (as opposed to as an array)."
  :New-Inspect-Function (allocate-data 'Show-String x)
  :Priority 11
)

(Defperspective :ART-8B-as-String (x show-x)
  :show-x-type-for-perspective show-string
  :This-Perspective-Applicable-Function
    (and (arrayp x) (equal (array-type x) 'sys:art-8b)
	 (not (typep show-x 'Show-String))
    )
  :menu-item-name "ART-8B as a string"
  :Menu-Who-Line-Doc-String
    "Inspect this ART-8B as a string (as opposed to as an array)."
  :New-Inspect-Function (allocate-data 'Show-String x)
  :Priority 0
)

;-------------------------------------------------------------------------------

(defflavor show-zmacs-line
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:Documentation "A perspective to view zmacs-lines as lines of text.")
)

(defmethod (show-zmacs-line :format-concisely) (stream)
"Just prints it out but notes that it is a zmacs line perspective."
  (format stream "Zmacs Line ~"
	  (list data nil (with-output-to-string (str)
			   (print-item-concisely data str)
		         )
          )
  )
)

(defmethod (show-zmacs-line :generate-item) ()
"Makes the inspector items for a zmacs line."
  (values (cons *blank-line-item*
		(append (Leader-Items-Of
			  data
			  '(length next previous bp-list tick node
			    contents-plist plist
			   )
			)
			(mapcar #'list (Split-Into-Lines data))
		)
	  )
     `(:font fonts:hl12bi :String ,(show-string-title data))
  )
)

(defmethod (show-zmacs-line :help) ()
"Gives help when you middle button on a Zmacs-line."
  (format nil "
The inspection pane you just selected is currently displaying a
Zmacs-line.  Mousing L2 on it should show it to you in some other way.
"
  )
)

(Defperspective :Zmacs-line (x show-x)
  :show-x-type-for-perspective show-zmacs-line
  :This-Perspective-Applicable-Function
    (and (stringp x)
	 (array-has-leader-p x)
	 (= (array-leader-length x) 8)
	 (not (typep show-x 'Show-Zmacs-line))
    )
  :menu-item-name "Zmacs Line"
  :Menu-Who-Line-Doc-String
    "Inspect this zmacs line as text (as opposed to as an array)."
  :New-Inspect-Function (allocate-data 'Show-Zmacs-line x)
  :Priority 12
)

;-------------------------------------------------------------------------------

(defflavor show-call-graph
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A perspective to view call graphs.
"
  )
)

(defmethod (show-call-graph :format-concisely) (stream)
"Just prints it out but notes that it is a call-graph perspective."
  (format stream "~ as a call graph." (list data t data))
)

(defmethod (Show-Call-Graph :setup-for-window) (window display-list)
  (destructuring-bind
    (printer arg display-obj the-top-item the-label the-item-generator flavor)
    display-list
    (ignore printer arg the-top-item display-obj the-item-generator flavor)
    (let ((*force-exposure* t))
	 (declare (special *force-exposure*))
	 (Graph-Calls data window the-label)
    )
  )
)

(defmethod (show-call-graph :cache-info) (window)
  (send window :Cache-Info)
)

(defmethod (show-call-graph :generate-item) ()
"Makes the inspector items for a call graph."
  (values self
	 `(:font fonts:hl12bi
	   :string ,(format nil "~S as a call graph." data)
	   :Top
	  )
	  'tv:stand-alone-grapher
  )
)

(defmethod (show-call-graph :help) ()
"Gives help when you middle button on a call graph."
  (format nil "
The inspection pane you just selected is currently displaying the
call graph of ~S.
"
	  data
  )
)

;-------------------------------------------------------------------------------


(defflavor show-inheritance-graph
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A perspective to view inheritance graphs.
"
  )
)

(defmethod (show-inheritance-graph :format-concisely) (stream)
"Just prints it out but notes that it is a inheritance-graph perspective."
  (format stream "~ as an inheritance graph." (list data t data))
)

(defmethod (Show-Inheritance-graph :setup-for-window) (window display-list)
  (destructuring-bind
    (printer arg display-obj the-top-item the-label the-item-generator flavor)
    display-list
    (ignore printer arg the-top-item display-obj the-item-generator flavor)
    (let ((*force-exposure* t))
	 (declare (special *force-exposure*))
	 (graph-class data window the-label)
    )
  )
)

(defmethod (show-inheritance-graph :cache-info) (window)
  (send window :Cache-Info)
)

(defmethod (show-inheritance-graph :generate-item) ()
"Makes the inspector items for an inheritance graph."
  (values self
	 `(:font fonts:hl12bi
	   :String
	    ,(if (class-p-safe data)
		 (format nil "~A's superclasses and supclasses"
			 (class-name-safe data)
		 )
		 (format nil "~A's components and dependants"
			 (typecase data
			   (si:flavor (si:flavor-name data))
			   (symbol (si:flavor-name (get data 'si:flavor)))
			   (otherwise (ferror "~S is not a class or flavor."
					      data
				      )
			   )
			 )
		 )
	     )
	   :Top
	  )
	  'tv:stand-alone-grapher
  )
)

(defmethod (show-inheritance-graph :help) ()
"Gives help when you middle button on a call graph."
  (format nil "
The inspection pane you just selected is currently displaying the
inheritance graph of ~S.
"
	  data
  )
)

;-------------------------------------------------------------------------------

(defflavor show-font
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:Documentation "A perspective to view fonts.")
)

(defmethod (show-font :format-concisely) (stream)
"Just prints it out but notes that it is a font perspective."
  (format stream "~ as an Font's characters" (list data t data))
)

(defmethod (show-font :generate-item-specialized) (window)
"Makes the inspector items for an font."
  (values
    (cons `((:Font 2 "Char-Int") 10 (:Font 2 "Char") 30
	    (:Font 2 "Printed") 50 (:Font 2 "Width")
	    ,@(if (font-left-kern-table data) '(60 (:Font 2 "Left Kern")) nil)
	   )
	   (loop for index from 0
		 for exists being the array-elements
		     of (font-chars-exist-table data)
		 for width = (if (font-char-width-table data)
				 (aref (font-char-width-table data) index)
				 (font-char-width data)
			     )
		 when (and exists width)
		 Append
		 (let ((number-of-lines
			 (ceiling (font-char-height data)
				  (send window :Line-Height)
			 )
		       )
		      )
		     `(,@(loop for i from 0 below (- number-of-lines 1) collect
			       *blank-line-item*
			 )
		       ((:Font 1 ,(format nil "~D" index))
			10
			(:Font ,fonts:cptfont
			       ,(format nil "~C" (int-char index))
			)
			30
			(:Font ,data
			       ,(format nil "~A" (string (int-char index)))
			)
			50
			(:Font 1 ,(format nil "~D" width))
			,@(if (font-left-kern-table data)
			     `(60 (:Font 1
				   ,(format nil "~D"
				     (aref (font-left-kern-table data) index)
				    )
				  )
			      )
			      nil
			  )
		       )
		      )
		 )
	    )
     )
     data
     'inspect-printer
     nil
    `(:font fonts:hl12bi
	    :string ,(format nil "Font ~S" (font-name data))
     )
  )
)

(defmethod (show-font :help) ()
"Gives help when you middle button on an Font."
  (format nil "
The inspection pane you just selected is currently displaying the
Font ~S as the characters it represewnts. Mousing L2 on it should
show it to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------


(defun rhb-p (x)
"Is true if X is a rubout handler buffer."
  (and (typep x 'array)
       (equal (length (array-dimensions x)) 1)
       (equal (array-leader-length x) 6)
       (typep (rhb-input-ring x) 'zwei:history)
  )
)

(defflavor Show-Rhb
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:Documentation "A simple inspection data flavor for rbout handler buffers.")
)

(defmethod (show-rhb :middle-button-result) ()
"Just returns the data slot (the rhb)."
  data
)

(defun rhb-string (rhb &optional (remove-newlines-p nil))
"Given a rubout handler buffer returns the values:
 - The substring from the start up to the fill pointer
 - The substring from the start up to the typein pointer
 - The substring from the start up to the scan pointer
 - The substring from the start up to the end of printable chars in the rhb
If remove-newlines-p is true then any newline chars get changed into #\newline
strings.
"
  (declare (values fill-string typein-string scan-string full-string))
  (let ((string (make-string (rhb-fill-pointer rhb)))
	(tstring (make-string (rhb-typein-pointer rhb)))
	(sstring (make-string (rhb-scan-pointer rhb)))
	(end (loop for i from 0 below (array-total-size rhb)
		   when (not (fixnump (aref rhb i)))
		   return i
		   finally (return nil)
	     )
	)
       )
       (loop for i from 0 below (rhb-fill-pointer rhb) do
	     (setf (aref string i) (int-char (aref rhb i)))
       )
       (loop for i from 0 below (rhb-typein-pointer rhb) do
	     (setf (aref tstring i) (int-char (aref rhb i)))
       )
       (loop for i from 0 below (rhb-scan-pointer rhb) do
	     (setf (aref sstring i) (int-char (aref rhb i)))
       )
       (let ((fstring (make-string end)))
	    (loop for i from 0 below end do
		  (setf (aref fstring i) (int-char (aref rhb i)))
	    )
	    (if remove-newlines-p
		(values (Remove-Newlines string)  (remove-newlines tstring)
			(Remove-Newlines sstring) (remove-newlines fstring)
		)
		(values string tstring sstring fstring)
	    )
       )
  )
)

(defmethod (show-rhb :format-concisely) (stream)
"Shows an RHB simply."
  (multiple-value-bind (fill-string typein-string scan-string full-string)
      (Rhb-String data t)
    (format stream "RHB ~, ~, ~, ~"
	    (list fill-string t) (list typein-string t)
	    (list scan-string t) (list full-string t)
    )
  )
)

(defun Remove-Newlines
       (string &optional (start 0) (newline-string "#\\NEWLINE"))
"Given a string that might contain newline chars returns a string that has
the string newline-string substituted instead of the newline char.
"
  (let ((index
	  (position #\newline (the string string) :Test #'char= :Start start)
	)
       )
       (if index
	   (string-append (subseq (the string string) start index)
			  newline-string
			  (remove-newlines string (+ 1 index) newline-string)
	   )
	   (subseq string start)
       )
  )
)
				  
(defmethod (show-rhb :generate-item) ()
"Generates the display item for show-rhbs."
  (multiple-value-bind (fill-string typein-string scan-string full-string)
      (Rhb-String data t)
    (values
      `(,*blank-line-item*
	((:font 1 "Details of ")
	 (:item1 instance ,(allocate-data 'show-generic-object-thing data)))
	,*blank-line-item*
	((:font 1 "Fill Pointer") (:Colon 16)
	 ,(format nil "~D" (rhb-fill-pointer data))
	)
	((:font 1 "Fill Text") (:Colon 16)
	 (:item1 t ,(allocate-data 'show-generic-object-thing fill-string))
	)
	((:font 1 "Scan Pointer") (:Colon 16)
	 ,(format nil "~D" (rhb-scan-pointer data))
	)
	((:font 1 "Scan Text") (:Colon 16)
	 (:item1 t ,(allocate-data 'show-generic-object-thing scan-string))
	)
	((:font 1 "Typein Pointer") (:Colon 16)
	 ,(format nil "~D" (rhb-typein-pointer data))
	)
	((:font 1 "Typein Text") (:Colon 16)
	 (:item1 t ,(allocate-data 'show-generic-object-thing typein-string))
	)
	((:font 1 "Full Text") (:Colon 16)
	 (:item1 t ,(allocate-data 'show-generic-object-thing full-string))
	)
       )
      `(:font fonts:hl12bi :string ,(format nil "RHB ~A" data))
    )
  )
)

(defmethod (show-rhb :help) ()
"Simple help method."
  (format nil "You're currently looking at a rubout handler buffer: ~S"
	  (send data :name)
  )
)


;-------------------------------------------------------------------------------

;;; A simple perspective for rubout handler buffers.
(Defperspective :Rubout-Handler-Buffer (x show-x)
  :show-x-type-for-perspective Show-RHB
  :This-Perspective-Applicable-Function
    (and (rhb-p x) (not (typep show-x 'show-rhb)))
  :menu-item-name "Rubout Handler Buffer"
  :Menu-Who-Line-Doc-String
    "Rubout Handler Buffer."
  :New-Inspect-Function
    (allocate-data 'Show-RHB x)
  :Priority 11
)

;-------------------------------------------------------------------------------

(Defperspective :font-characters (x show-x)
  :show-x-type-for-perspective Show-font
  :This-Perspective-Applicable-Function
    (or (and (typep x 'font) (not (typep show-x 'Show-Font)))
	(let ((name (typecase x
		      (symbol (symbol-name x))
		      (string x)
		      (otherwise nil)
		    )
	      )
	     )
	     (and name
		  (let ((font-name (find-symbol (string-upcase name) 'fonts)))
		       (and font-name
			    (boundp font-name)
			    (typep (symbol-value font-name) 'font)
		       )
		  )
	     )
	)
    )
  :menu-item-name "Font Characters"
  :Menu-Who-Line-Doc-String
    "Font, showing all of the chars."
  :New-Inspect-Function
    (allocate-data 'Show-Font
      (Etypecase x
	(font x)
	((or string symbol)
	 (symbol-value (find-symbol (string-upcase (string x)) 'fonts))
	)
      )
    )
)

;-------------------------------------------------------------------------------


(defflavor show-expanded-flavors-instance ()
	   (show-generic-object-thing)
)


(defwhopper (show-expanded-flavors-instance :generate-item-specialized) (window)
  (let ((*show-plists-and-alists-for-show-slots-p* t))
       (declare (special *show-plists-and-alists-for-show-slots-p*))
       (continue-whopper window)
  )
)


(defmethod (show-expanded-flavors-instance :format-concisely) (stream)
"Just prints it."
  (format stream "~ with expanded instance variables." (list data t data))
)


(defun flavor-is-expandable-p (instance flavor)
  (and (typep flavor 'sys:flavor)
       (loop for slot in (sys:flavor-all-instance-variables flavor)
	     for slot-name = (ucl:first-if-list slot)
	     when (loop for (test) in *expand-slot-alist*
			when (and (boundp-in-instance instance slot-name)
				  (funcall test instance slot-name
				    (symeval-in-instance instance slot-name)
				  )
			     )
			return t
			finally (return nil)
		  )
	     return t
	     finally (return nil)
       )
  )
)

(Defperspective :Expanded-Flavors-Instance (x show-x)
  :show-x-type-for-perspective Show-Expanded-Flavors-Instance
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-Expanded-Flavors-Instance))
	 (or (typep x 'sys:property-list-mixin)
	     (and (typep x 'sys:vanilla-flavor)
		  (Flavor-Is-Expandable-P
		    x
		    (let ((flavor (if (symbolp (type-of x))
				      (type-of x)
				      (clos:class-name (type-of x))
				  )
			  )
			 )
		         (get flavor 'sys:flavor)
		    )
		  )
	     )
	 )
    )
  :menu-item-name "Expanded Flavors Instance"
  :Menu-Who-Line-Doc-String
    "Flavors Instance with expanded IVs (e.g. PList/AList)"
  :New-Inspect-Function
    (allocate-data 'Show-Expanded-Flavors-Instance x)
  :Priority -1
)

;-------------------------------------------------------------------------------


(defflavor show-expanded-clos-instance ()
	   (show-generic-object-thing)
)


(defwhopper (show-expanded-clos-instance :generate-item-specialized) (window)
  (let ((*show-plists-and-alists-for-show-slots-p* t))
       (declare (special *show-plists-and-alists-for-show-slots-p*))
       (continue-whopper window)
  )
)


(defmethod (show-expanded-clos-instance :format-concisely) (stream)
"Just prints it."
  (format stream "~ with expanded slots." (list data t data))
)

(defun class-is-expandable-p (instance class)
  (and (class-p-safe class)
       (loop for slot in (class-all-slots-safe class)
	     for slot-name = (slotd-name-safe slot)
	     when (loop for (test) in *expand-slot-alist*
			when (and (catch-error
				    (slot-boundp-safe instance slot-name) nil
				  )
				  (catch-error
				    (funcall test instance slot-name
				      (slot-value-safe instance slot-name)
				    )
				    nil
				  )
			     )
			return t
			finally (return nil)
		  )
	     return t
	     finally (return nil)
       )
  )
)

(Defperspective :Expanded-Clos-Instance (x show-x)
  :show-x-type-for-perspective Show-Expanded-Clos-Instance
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-Expanded-Clos-Instance))
	 (and (any-sort-of-clos-instance-p x)
	      (class-is-expandable-p x (class-of-safe x))
	 )
    )
  :menu-item-name "Expanded CLOS Instance"
  :Menu-Who-Line-Doc-String
    "CLOS Instance with expanded slots (e.g. PList/AList)"
  :New-Inspect-Function
    (allocate-data 'Show-Expanded-Clos-Instance x)
  :Priority -1
)

;-------------------------------------------------------------------------------


(defflavor show-expanded-defstruct ()
	   (show-generic-defstruct)
)


(defwhopper (show-expanded-defstruct :generate-item-specialized) (window)
  (let ((*show-plists-and-alists-for-show-slots-p* t))
       (declare (special *show-plists-and-alists-for-show-slots-p*))
       (continue-whopper window)
  )
)


(defmethod (show-expanded-defstruct :format-concisely) (stream)
"Just prints it."
  (format stream "~ with expanded slots." (list data t data))
)

(defun defstruct-is-expandable-p (instance defstruct)
  (let ((descr (get defstruct 'sys:defstruct-description)))
       (and descr
	    (loop for entry in (si::defstruct-description-slot-alist descr)
		  for slot-name = (first entry)
		  for slot-value
		      = (catch-error
			  (funcall
			    (si:defstruct-slot-description-ref-macro-name
			      (second entry)
			    )
			    instance
			  )
			  nil
			)
		  when (loop for (test) in *expand-slot-alist*
			     when (catch-error
				    (funcall test instance slot-name slot-value)
				    nil
				  )
			     return t
			     finally (return nil)
		       )
		  return t
		  finally (return nil)
	    )
       )
  )
)

(Defperspective :Expanded-Defstruct (x show-x)
  :show-x-type-for-perspective Show-Expanded-Defstruct
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-Expanded-Defstruct))
	 (let ((type (named-structure-p x)))
	      (and type (defstruct-is-expandable-p x type))
	 )
    )
  :menu-item-name "Expanded Defstruct Instance"
  :Menu-Who-Line-Doc-String
    "Defstruct Instance with expanded slots (e.g. PList/AList)"
  :New-Inspect-Function
    (allocate-data 'Show-Expanded-Defstruct x)
  :Priority -1
)

;-------------------------------------------------------------------------------

(defflavor show-method-list ()
	   (generic-middle-button-mixin auxiliary-data-mixin inspection-data)
)


(defmethod (show-method-list :generate-item) ()
  (values
    (loop for method in aux-data collect
	  `("    " (:item1 instance
			   ,(apply 'allocate-data 'show-method-details
				   (Data-From-Method (third method))
			    )
		   )
	   )
    )
    `(:font fonts:hl12bi :String ,(format nil "Method List for ~S" data))
  )
)

(defmethod (show-method-details :format-concisely) (stream)
  (let ((flavor (allocate-data 'show-flavor
			       (get (second (first aux-data)) 'sys:flavor)
		)
	)
       )
       (format stream "Method ~~{ ~s~}" (list flavor t flavor)
	       (rest (rest (first aux-data)))
       )
  )
)

(defmethod (show-method-list :format-concisely) (stream)
  "Just prints it."
  (format stream "Methods named ~" (list data t data))
)


(Defperspective :Method-List (x show-x)
  :show-x-type-for-perspective Show-Method-List
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-Method-List))
	 (self-evaluating-form-p x)
	 (gethash x *all-method-names*)
    )
  :menu-item-name "Method List"
  :Menu-Who-Line-Doc-String
    "List of methods with this key"
  :New-Inspect-Function
    (allocate-data 'Show-Method-List x
		   (sort-method-list (copy-list (gethash x *all-method-names*)))
    )
  :Priority -1
)

;-------------------------------------------------------------------------------

(defflavor show-method-list-no-combined-methods ()
	   (generic-middle-button-mixin auxiliary-data-mixin inspection-data)
)


(defmethod (show-method-list-no-combined-methods :generate-item) ()
  (values
    (loop for method in aux-data collect
	  `("    " (:item1 instance
			   ,(apply 'allocate-data 'show-method-details
				   (Data-From-Method (third method))
			    )
		   )
	   )
    )
    `(:font fonts:hl12bi :String
	    ,(format nil "Non Combined Method List for ~S" data)
     )
  )
)

(defmethod (show-method-details :format-concisely) (stream)
  (let ((flavor (allocate-data 'show-flavor
			       (get (second (first aux-data)) 'sys:flavor)
		)
	)
       )
       (format stream "Method ~~{ ~s~}" (list flavor t flavor)
	       (rest (rest (first aux-data)))
       )
  )
)

(defmethod (show-method-list-no-combined-methods :format-concisely) (stream)
  "Just prints it."
  (format stream "Methods named ~" (list data t data))
)

(Defperspective :Method-List-no-combined-methods (x show-x)
  :show-x-type-for-perspective Show-method-list-no-combined-methods
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-method-list-no-combined-methods))
	 (self-evaluating-form-p x)
	 (gethash x *all-method-names*)
    )
  :menu-item-name "Method List (ignoring combined methods)"
  :Menu-Who-Line-Doc-String
    "List of methods with this key except for combined methods"
  :New-Inspect-Function
    (allocate-data 'Show-method-list-no-combined-methods x
		   (sort-method-list
		     (remove-if #'(lambda (x)
				    (member (third (third x))
					    '(:Combined sys:fasload-combined)
				    )
				  )
				  (copy-list (gethash x *all-method-names*))
		     )
		   )
    )
  :Priority -1
)

;-------------------------------------------------------------------------------

(defflavor grapher-perspective () (basic-perspective)
  (:Documentation "Perspectives for graphing.")
)

;;; Make sure that we know how to graph functions named by symbols.
(defperspective :call-graph (x show-x)
  :This-Perspective-Applicable-Function
    (or (and (or (symbolp x) (consp x)) (si:fdefinition-safe x))
	(compiled-function-p x)
    )
  :Menu-Item-Name "Call Graph"
  :Side-Effect-Function
    (Inspect-Graph-Function
      (if (compiled-function-p x) x (si:fdefinition-safe x))
    )
  :Priority -1
  :Flavor grapher-perspective 
)

;;; Make sure that we know how to graph functions named by symbols.
(defperspective :graph-of-callers (x show-x)
  :This-Perspective-Applicable-Function
    (or (symbolp x)
	(compiled-function-p x)
    )
  :Menu-Item-Name "Graph of Callers"
  :Side-Effect-Function
    (Inspect-Graph-Callers x)
  :Priority -1
  :Flavor grapher-perspective 
)

(defperspective :inheritance-graph (x show-x)
  :This-Perspective-Applicable-Function (Inspect-Graph-Class-P x)
  :Menu-Item-Name "Inheritance Graph"
  :Side-Effect-Function (Inspect-Graph-Class x)
  :Priority -1
  :Flavor grapher-perspective 
)
