;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-


(defflavor inspector-framifying-mixin
	   ()
	   (full-screen-hack-mixin
	    frame-dont-select-inferiors-with-mouse-mixin
	    bordered-constraint-frame-with-shared-io-buffer
	    constraint-frame-forwarding-mixin))

(defmethod (Inspector-Framifying-Mixin :Before :Set-Panes) (to)
  (setq panes to)
)
	    
(defmethod (Inspector-Framifying-Mixin :Before :Set-Constraints) (to)
  (setq constraints to)
)
	    
(DEFFLAVOR basic-inspect-frame                        ;fi
	   ((inspectors nil)
	    (typeout-window nil)
	    (menu nil)
	    (user nil)
	    (frame nil)
	    (history nil)
            (INSPECTOR-TYPEIN-P NIL)
	    (inspection-data-active? NIL)) ;;See documentation string below.
	   (ucl:basic-command-loop
 	    PROCESS-MIXIN
	    BORDERS-MIXIN LABEL-MIXIN BASIC-FRAME)
  (:DEFAULT-INIT-PLIST
    :active-command-tables '(inspector-menu-cmd-table inspector-other-cmd-table)
    :all-command-tables '(inspector-menu-cmd-table inspector-other-cmd-table)
    :menu-panes '((menu ucl-inspector-menu))
    :typein-handler :handle-typein-input
    :prompt "Inspect: "
    ;;This predicate tells UCL to print command execution results only
    ;;on typed Lisp forms during our Lisp Evaluation mode.  It also inspects results when not in that mode.
    :print-results? 'inspector-print-values?
    :basic-help '(documentation-cmd)
    :SAVE-BITS :DELAYED
    :PROCESS '(INSPECT-TOP-LEVEL
		:SPECIAL-PDL-SIZE 4000
		:REGULAR-PDL-SIZE 10000))
  :GETTABLE-INSTANCE-VARIABLES
  :SETTABLE-INSTANCE-VARIABLES
  :INITTABLE-INSTANCE-VARIABLES
  :SPECIAL-INSTANCE-VARIABLES
  (:INIT-KEYWORDS :NUMBER-OF-INSPECTORS)
  (:DOCUMENTATION :MIXIN
    "Basic flavor used to build the inspector and other inspector-like applications.
This flavor may also be used as a mixin to construct special-purpose inspection windows,
such as the Flavor Inspector (TV:FLAVOR-INSPECTOR).  Most any information can be presented using 
it, as long as the Inspector paradigm is appropriate for your application.  Use the code for flavor
TV:FLAVOR-INSPECTOR as a guide for constructing your own inspector.  Basic points to keep in mind
are the following:

1. The inspector command interface makes use of the Universal Command Loop.  Therefore, UCL instance
   variables control much of the command interaction.  Important initializations are:

   :ACTIVE-COMMAND-TABLES --the set of commands your inspector accepts.  Can be NIL.
   :ALL-COMMAND-TABLES    --probably EQ to :ACTIVE-COMMAND-TABLES.  The set of all command tables used in your inspector.
   :MENU-PANES            --an alist which should be ((TV:MENU <your symbol>)).  <your symbol> is a menu symbol used
                            to build your permanent command menu using UCL's BUILD-MENU function.  If for some reason
                            you have arranged your constraint frame to not include a permanent command menu, ommitt this option.
   :TYPEIN-MODES          --the list of symbols set to UCL:TYPEIN-MODE instances which control the processing of typed
                            expressions in the interaction window.   The default just interprets Lisp forms and typed
                            command names; in a special inspector, you might want to design special typein-modes which
                            allow the user to type an expression of some kind to inspect some data.  For instance, in
                            the Flavor Inspector, the user can type flavor names and method specs to inspect them.
 2.  As with the Flavor Inspector, you'll be defining flavors built on TV:INSPECTION-DATA for displaying your various types of data
     in the inspection panes.  Pattern them off of the flavor inspector's.  Use method :INSPECT-THING to create and explicitly inspect 
     a TV:INSPECTION-DATA instance.
 3.  To specify your own window panes and constraints, give your flavor a :BEFORE :INIT method
     similar to (:METHOD TV:FLAVOR-INSPECTOR :BEFORE :INIT).
 3.  If you don't want to handle typed expressions in your inspector, initialize :TYPEIN-HANDLER to NIL.  
 4.  Your inspector should initialize :INSPECTION-DATA-ACTIVE? to T.  Since the regular inspector just inspects Lisp,
     it turns this off so that programmers are able to inspect instances of TV:INSPECTION-DATA in the normal manner.
     Any special inspectors such as the Flavor Inspector have to set this to T in order to activate the special inspection
     features of TV:INSPECTION-DATA instances.  If you have an inspector which mixes the function of Lisp inspection with
     special data inspection, you can flip this variable on and off when switching between Lisp inspection and
     TV:INSPECTION-DATA inspection.
Constructing your own inspector isn't automatic, but if you use the flavor inspector as a model, it'll be pretty easy."))



(DEFMETHOD (BASIC-INSPECT-FRAME :BEFORE :INIT) (PLIST)        ;fi
  ;;Unless a higher-level :BEFORE :INIT demon has already initialized INSPECTORS, PANES, and CONSTRAINTS, initialize them.
  ;;This condition allows a superior flavor's :BEFORE :INIT demon to set them the way it wants--we don't interfere.
  (UNLESS INSPECTORS
    (LET ((NOI (OR (GET PLIST :NUMBER-OF-INSPECTORS) 3))
          (names nil)
          (iobuff (make-default-io-buffer)))
      (setf (send self :Panes)
	    (LIST `(INTERACTOR INSPECTOR-INTERACTION-PANE
                               :LABEL NIL
                               :FONT-MAP ,(LIST (FIRST *inspector-Font-Map*))  ;(,inspect-standard-font)
                               :IO-BUFFER ,IOBUFF
                               :MORE-P NIL)
		  `(HISTORY INSPECT-HISTORY-WINDOW ;-PANE-WITH-MARGIN-SCROLLING
                            :FONT-MAP ,(LIST (FIRST *inspector-Font-Map*))
                            :IO-BUFFER ,IOBUFF)
;                            :SCROLL-BAR 3)
		  `(MENU INSPECTOR-MENU-PANE
                         :FONT-MAP ,(LIST (FIRST *inspector-Font-Map*))  ;(,inspect-standard-font)
                         :IO-BUFFER ,IOBUFF)))
      ;;Add an inspector to PANES, taking into account the number of inspector panes requested.  The first
      ;;inspector is given a typeout pane.  Also initialize INSPECTORS.
      (DOTIMES (I NOI)
	(LET ((NAME1 (INTERN (FORMAT NIL "INSPECTOR-~D" I) "TV")))
	  (PUSH
	   `(,NAME1 ,(IF (= I (1- NOI))
		       'INSPECT-PANE-WITH-TYPEOUT
		       'INSPECT-PANE)
;	     :SCROLL-BAR 2 :SCROLL-BAR-ALWAYS-DISPLAYED T
             :FONT-MAP ,(LIST (FIRST *inspector-Font-Map*)) :IO-BUFFER ,IOBUFF)
	   (send self :PANES))
          (PUSH NAME1 NAMES)))  ;?
      (SETQ INSPECTORS NAMES)   ;?
	  ;(PUSH NAME1 INSPECTORS)))
      (setf (send self :Constraints)
	    `((:THREE-PANES ,(REVERSE `(INTERACTOR MENU-HISTORY ,@INSPECTORS))
                    ((INTERACTOR 4 :LINES))
                    ((MENU-HISTORY :HORIZONTAL (4 :LINES HISTORY) (MENU HISTORY)
                                   ((MENU :ASK :PANE-SIZE))
                                   ((HISTORY :EVEN))))
                    ,(MAPCAR
                       #'(LAMBDA (NAME1)
                           `(,NAME1 :LIMIT (1 36 :LINES)
                             ,(/ 0.3s0 (1- NOI)) :LINES))
                       (CDR INSPECTORS))
                    ((,(CAR INSPECTORS) :EVEN)))
	      (:ONE-PANE (,(CAR INSPECTORS) MENU-HISTORY INTERACTOR)
                             ((INTERACTOR 4 :LINES))
                             ((MENU-HISTORY
                                :HORIZONTAL (4 :LINES HISTORY)
                                (MENU HISTORY)
                                ((MENU :ASK :PANE-SIZE))
                                ((HISTORY :EVEN))))
                             ((,(CAR INSPECTORS) :EVEN)))
              (:TWO-HORIZONTAL-PANES ,(REVERSE `(INTERACTOR MENU-HISTORY INSPECTOR-2 INSPECTOR-1))
                    ((INTERACTOR 4 :LINES))
                    ((MENU-HISTORY :HORIZONTAL (4 :LINES HISTORY) (MENU HISTORY)
                                   ((MENU :ASK :PANE-SIZE))
                                   ((HISTORY :EVEN))))
                    ((INSPECTOR-1 0.5))
                    ((INSPECTOR-2 :even)))
              (:TWO-VERTICAL-PANES ,(REVERSE `(INTERACTOR MENU-HISTORY SIDE-BY-SIDE))
                    ((INTERACTOR 4 :LINES))
                    ((MENU-HISTORY :HORIZONTAL (4 :LINES HISTORY) (MENU HISTORY)
                                   ((MENU :ASK :PANE-SIZE))
                                   ((HISTORY :EVEN))))
                    ((SIDE-BY-SIDE :HORIZONTAL (:even)
                                   (INSPECTOR-2 INSPECTOR-1)
                                   ((INSPECTOR-1 0.5))
                                   ((INSPECTOR-2 :even)))))
              ;;This configuration is for debugging purposes. Set *inspector-configuration* to debug. It should
              ;;not show up in profile or in the pop up config menu as a selectable configuration.
	      (:DEBUG (,(CAR INSPECTORS) MENU-HISTORY INTERACTOR)
                             ((INTERACTOR 40 :LINES))
                             ((MENU-HISTORY
                                :HORIZONTAL (4 :LINES HISTORY)
                                (MENU HISTORY)
                                ((MENU :ASK :PANE-SIZE))
                                ((HISTORY :EVEN))))
                             ((,(CAR INSPECTORS) :EVEN)))
              )))))


(DEFMETHOD (BASIC-INSPECT-FRAME :AFTER :INIT) (IGNORE)      ;!fi
;  (SETQ *inspect-print-base* *print-base*
;        *inspect-read-base*  *read-base*
;        *inspect-print-array*  nil
;        *inspect-print-circle* t      	; default of T so recursive structures print
;        *inspect-print-radix*  *print-radix*
;        *inspect-nopoint       *nopoint
;        *inspect-print-level*  8.
;        *inspect-print-length* 1000.)
  ;;Bind the pane variables and select the interaction pane.
  (DO ((IS INSPECTORS (CDR IS)))
      ((NULL IS))
    (RPLACA IS (SEND SELF :GET-PANE (CAR IS))))
  (SETQ TYPEOUT-WINDOW (SEND (CAR INSPECTORS) :TYPEOUT-WINDOW) USER
        (SEND SELF :GET-PANE 'INTERACTOR) FRAME SELF HISTORY
        (SEND SELF :GET-PANE 'HISTORY))
  (SEND TYPEOUT-WINDOW :SET-IO-BUFFER (SEND USER :IO-BUFFER))
  (SEND SELF :SELECT-PANE USER)
  ;;Necessary for the :ask :pane-size constraint for the menu.
  (SEND SELF :SET-CONFIGURATION (SEND SELF :CONFIGURATION)))


(DEFMETHOD (BASIC-INSPECT-FRAME :DESIGNATE-IO-STREAMS) ()     ;fi
  ;;Redefine this UCL method to set up the correct io bindings.
  (SETQ *TERMINAL-IO* (SEND SELF :GET-PANE 'INTERACTOR) *STANDARD-INPUT* *TERMINAL-IO*
        *STANDARD-OUTPUT* *TERMINAL-IO* *DEBUG-IO*
        (SEND (CAR INSPECTORS) :TYPEOUT-WINDOW))) 

(DEFMETHOD (BASIC-INSPECT-FRAME :AROUND :HANDLE-TYPEIN-INPUT) ;fi
           (CONT MT IGNORE &OPTIONAL (UNTYI-FIRST-CHAR? T)) 
           "Make sure io for typein is bound to the interactor pane, and use UCL who line documentation for typein."
           (LET ((*TERMINAL-IO* USER))
             (SETQ INSPECTOR-TYPEIN-P T)
             (UNWIND-PROTECT (FUNCALL-WITH-MAPPING-TABLE
                              CONT
                              MT
                              :HANDLE-TYPEIN-INPUT
                              UNTYI-FIRST-CHAR?)
                             (SETQ INSPECTOR-TYPEIN-P NIL))))

(DEFMETHOD (BASIC-INSPECT-FRAME :AROUND :HANDLE-PROMPT) (CONT MT IGNORE)      ;fi
  ;;Make sure io for the prompt is bound to the interactor pane.
  (LET ((*TERMINAL-IO* USER))
    (FUNCALL-WITH-MAPPING-TABLE CONT MT :HANDLE-PROMPT))) 

(DEFMETHOD (BASIC-INSPECT-FRAME :BEFORE :LOOP) ()     ;fi
  ;;Do an intitial update panes.
  (UPDATE-PANES)) 


(DEFMETHOD (BASIC-INSPECT-FRAME :AROUND :FETCH-AND-EXECUTE) (CONT MT IGNORE)  ;fi
  ;;Check for typeout.
  (SEND (CAR INSPECTORS) :FLUSH-TYPEOUT) (SEND FRAME :SELECT-PANE USER)
  (FUNCALL-WITH-MAPPING-TABLE CONT MT :FETCH-AND-EXECUTE)) 

(DEFMETHOD (BASIC-INSPECT-FRAME :AROUND :HANDLE-UNKNOWN-INPUT) (CONT MT IGNORE)       ;fi
  (LET (inspection-data)
    (COND
      ;;first see if they toggled a pane's locked status
      ((AND (CONSP ucl:kbd-input)
            (EQ (FIRST ucl::kbd-input) :MOUSE-BUTTON)
            (= (SECOND ucl::kbd-input) #\MOUSE-M-2))
       (SEND (THIRD ucl::kbd-input) :send-if-handles :toggle-lock))   
      ((AND (CONSP ucl:kbd-input)
            (= (fourth ucl::kbd-input) #\MOUSE-M-2))
       (SEND (THIRD ucl::kbd-input) :send-if-handles :toggle-lock))   
      ;;If not a blip, let UCL's method handle unknown input
      ((NEQ UCL::INPUT-MECHANISM 'UCL::UNKNOWN)
      ;?((OR (EQ UCL::INPUT-MECHANISM 'UCL::TYPEIN) (not (LISTP THING)))
       (FUNCALL-WITH-MAPPING-TABLE CONT MT :HANDLE-UNKNOWN-INPUT))
      ;;Blip contains an inspection-data instance and we are currently inspecting treating them specially.
      ((AND INSPECTION-DATA-ACTIVE?
            (OR
              ;;Blip in form (INSTANCE (:ITEM1 INSTANCE <inspection-data instance>) <window> <mouse button>).
              ;;These are the standard inspection-data blips from the inspection panes.
              (AND (EQ (FIRST UCL::KBD-INPUT) 'INSTANCE)
                   (EQ (FIRST (SECOND UCL::KBD-INPUT)) :ITEM1)
                   (TYPEP (THIRD (SECOND UCL::KBD-INPUT)) 'INSPECTION-DATA)
                   (SETQ INSPECTION-DATA (THIRD (SECOND UCL::KBD-INPUT))))
              ;;Blip in form (:VALUE <inspection-data instance> <window> <mouse button>).  These blips come from
              ;;the inspection history and always have flavor information in them.
              (AND (EQ (FIRST UCL::KBD-INPUT) :VALUE)
                   (TYPEP (SECOND UCL::KBD-INPUT) 'INSPECTION-DATA)
                   (SETQ INSPECTION-DATA (SECOND UCL::KBD-INPUT)))))
       ;;Have the INSPECTION-DATA handle the mouse blip.  (Each type of info handles the various mouse buttons differently.)
       (SEND INSPECTION-DATA :HANDLE-MOUSE-CLICK UCL::KBD-INPUT SELF))
      ((EQ (FIRST UCL::KBD-INPUT) :LINE-AREA)
       (SELECTOR (FOURTH UCL::KBD-INPUT) =
         (#\MOUSE-L (SEND SELF :INSPECT-INFO-LEFT-CLICK))
         (#\MOUSE-M
          ;; Delete from line area
          (SEND HISTORY :FLUSH-OBJECT (INSPECT-REAL-VALUE UCL::KBD-INPUT))
          (SEND HISTORY :SET-CACHE NIL)
          ;;make sure the pane is unlocked if they deleted that item
          (LOOP for iw in inspectors
                when (EQ (INSPECT-REAL-VALUE UCL::KBD-INPUT) (SEND iw :current-object))
                do (SEND iw :set-locked-p nil))
          (UPDATE-PANES))
         (T
          (SEND SELF :INSPECT-INFO-RIGHT-CLICK))))
      ;;Middle click on inspected Lisp object--inspect it, leaving source in one of the windows
      ((AND (= (FOURTH UCL::KBD-INPUT) #\MOUSE-M)
            (MEMBER (THIRD UCL::KBD-INPUT) INSPECTORS :TEST #'EQ))
       (SEND SELF :INSPECT-INFO-MIDDLE-CLICK))
      ;; Right Click on inspected Lisp Object-- inspect its function definition, or itself if no function.
      ((= (FOURTH UCL::KBD-INPUT) #\MOUSE-R)
       (SEND SELF :INSPECT-INFO-RIGHT-CLICK))
      ((KEY-STATE :HYPER)
       ;; Hyper means modify the slot we are pointing at.
       (IF (OR (NULL (FIRST UCL::KBD-INPUT)) (NULL (GET (FIRST UCL::KBD-INPUT) 'SET-FUNCTION)))
           (FORMAT USER "~&Cannot set this component.")
           (PROGN
             (INSPECT-SET-SLOT UCL::KBD-INPUT USER HISTORY INSPECTORS)
             (UPDATE-PANES)))
       (SEND SELF :HANDLE-PROMPT))
      (T ;; Otherwise inspect UCL:KBD-INPUT.
       (SEND SELF :INSPECT-INFO-LEFT-CLICK)))))

;;;Handles middle clicks on mouse sensitive items in inspection and history panes.
;;;Middle clicks in inspection panes inspect the item, but put (or leave) the current item in the
;;;middle pane.  Middle cliks in the history pane just inspect the item.  (The item comes from UCL:KBD-INPUT.)
(DEFMETHOD (BASIC-INSPECT-FRAME :INSPECT-INFO-MIDDLE-CLICK) ()        ;fi
  (LET ((1ST-THING (INSPECT-REAL-VALUE UCL::KBD-INPUT))
	(2ND-THING
	 (WHEN (MEMBER (THIRD UCL::KBD-INPUT) INSPECTORS :TEST #'EQ)
	   (SEND (THIRD UCL::KBD-INPUT) :CURRENT-OBJECT))))
	;; First flush item we will be inspecting
    (INSPECT-FLUSH-FROM-HISTORY 1ST-THING HISTORY)
    (WHEN 2ND-THING
      (INSPECT-FLUSH-FROM-HISTORY 2ND-THING HISTORY))
    (WHEN 2ND-THING
      (SEND HISTORY :APPEND-ITEM 2ND-THING))
    (SEND HISTORY :APPEND-ITEM 1ST-THING)
    (UPDATE-PANES))) 

;;;Handles right clicks on mouse sensitive items in inspection panes.
;;;On right clicks we try to find the function definition, if the item is a defined function,
;;;otherwise we just inspect it.  The item comes from UCL:KBD-INPUT.
(DEFMETHOD (BASIC-INSPECT-FRAME :inspect-info-right-click) () ;fi
  (LET ((thing (inspect-find-function (inspect-real-value UCL:KBD-INPUT))))
    (inspect-flush-from-history thing history)
    (SEND history :append-item thing)
    (update-panes)))

(defmethod (basic-inspect-frame :update-*)  ()
  (LET* ((ITEMS (SEND HISTORY :ITEMS))
	 (NITEMS (IF ITEMS (ARRAY-ACTIVE-LENGTH ITEMS) 0)))
    (AND (>= NITEMS 1) (SETQ * (AREF ITEMS (- NITEMS 1))))
    (AND (>= NITEMS 2) (SETQ ** (AREF ITEMS (- NITEMS 2))))
    (AND (>= NITEMS 3) (SETQ *** (AREF ITEMS (- NITEMS 3))))))

(DEFMETHOD (BASIC-INSPECT-FRAME :INSPECT-THING)       ;fi
           (TYPE THING &OPTIONAL (AUX-DATA NIL AUX-SUPPLIED?))
  (LET ((INSPECTED-THING
	 (INSPECT-REAL-VALUE
	  `(:VALUE
	    ,(IF AUX-SUPPLIED?
	       (ALLOCATE-DATA TYPE THING AUX-DATA)
	       (ALLOCATE-DATA TYPE THING))
	    ,HISTORY))))
    (INSPECT-FLUSH-FROM-HISTORY INSPECTED-THING HISTORY)
    (SEND HISTORY :APPEND-ITEM INSPECTED-THING)
    (UPDATE-PANES)))

(DEFMETHOD (BASIC-INSPECT-FRAME :pretty-print-thing) (thing)  ;fi
  (FORMAT typeout-window "~%")
  (GRIND-TOP-LEVEL thing NIL typeout-window)
  (FORMAT typeout-window "~2%~a" tv:*REMOVE-TYPEOUT-STANDARD-MESSAGE*)
  (LET ((char (SEND typeout-window :ANY-TYI)))
    (UNLESS (= char #\SPACE)
      (SEND *STANDARD-INPUT* :FORCE-KBD-INPUT char))
    (SEND (CAR inspectors) :FLUSH-TYPEOUT)))

(DEFMETHOD (BASIC-INSPECT-FRAME :NAME-FOR-SELECTION) () NAME) ;fi

(DEFMETHOD (BASIC-INSPECT-FRAME :PREPARE-FOR-USE) (OBJECT OBJP NEW-LABEL)     ;fi
;  (set-print-info)
  (SEND SELF :SET-LABEL NEW-LABEL)
  (LET ((HW (SEND SELF :GET-PANE 'HISTORY)))
    (SEND HW :FLUSH-CONTENTS)
    (dolist (iw (send self :inspectors))
	  (SEND iw :set-locked-p nil))
    (COND
      (OBJP                                ;!as long as something was passed to inspect
       (WITH-SHEET-DEEXPOSED (SELF) (SEND HW :FLUSH-CONTENTS)  
			     (SEND HW :APPEND-ITEM OBJECT)
			     (DOLIST (IW (SEND SELF :INSPECTORS))
			       (SEND IW :SET-CURRENT-DISPLAY
				     (SEND IW :SETUP
					   `(INSPECT-PRINTER NIL NIL NIL
							     (NIL NIL NIL NIL
								  ,(LABEL-FONT (SEND IW :LABEL))
								  "Empty"))))
			       (SEND IW :SET-CURRENT-OBJECT (LIST NIL))))))
    (SEND (SEND SELF :TYPEOUT-WINDOW) :MAKE-COMPLETE)
    (SEND HW :CLEAR-INPUT)))

;-------------------------------------------------------------------------------

(DEFFLAVOR inspect-frame
           ()
           (basic-inspect-frame Inspector-Framifying-Mixin)
  (:DOCUMENTATION :MIXIN
                  "Flavor for the actual Inspector.")
  )

;-------------------------------------------------------------------------------

(DEFMETHOD (INSPECTOR-INTERACTION-PANE :AFTER :REFRESH-HELP)  ;fi
           (&OPTIONAL (option T) application (current-package *package*))
  (let ((frame (Find-Inspector-Window self)))
    (DOLIST (inspector (SEND frame :INSPECTORS))
      (SEND inspector :REFRESH-HELP option application current-package))
    (SEND (SEND frame :history)
	  :REFRESH-HELP option application current-package)))


;-------------------------------------------------------------------------------

;;; Make the label light up.

(defwhopper (basic-inspect :Mouse-Sensitive-Item) (x y)
  (let ((label (send self :Label)))
       (if (and (>= x (label-left   label))
		(>= y (label-top    label))
		(<= y (label-bottom label))
		(<= x (+ width (label-right  label)))
	   )
	   (let ((string-width
		   (sheet-string-length
		     self (label-string label) 0 nil nil (label-font label)
		   )
		 )
		 (font (label-font label))
		)
	        (coerce-font font self)
		(if (<= x string-width)
		    (values current-object
;			    (if (typep current-object 'inspection-data)
;			       `(:Item1 instance ,current-object)
;			        current-object
;			    )
			    :Value
			    (label-left label)
			    (+ (label-left label) string-width)
			    (label-top label)
			    (font-char-height font)
			    t
		    )
		    (continue-whopper x y)
		)
	   )
	   (continue-whopper x y)
       )
  )
)


(DEFMETHOD (BASIC-INSPECT :MOUSE-MOVES)
	   (X Y &AUX ITEM TYPE LEFT TOP the-height literal-p BWIDTH BHEIGHT)
           (MOUSE-SET-BLINKER-CURSORPOS)
           (MULTIPLE-VALUE-SETQ (ITEM TYPE LEFT BWIDTH TOP the-height literal-p)
             (SEND SELF :MOUSE-SENSITIVE-ITEM X Y))
           (COND
             ((CLI:MEMBER TYPE '(:LIST-STRUCTURE :LIST-STRUCTURE-TOP-LEVEL)
			  :TEST #'EQ)
              (SETQ SENSITIVE-INSPECT-ITEM T)
	      (BLINKER-SET-VISIBILITY ITEM-BLINKER NIL)
	      ;;;LEFT, BWIDTH, TOP are invalid
              (SEND LIST-BLINKER :SET-LIST-ITEM ITEM)
	      (BLINKER-SET-VISIBILITY LIST-BLINKER T))
             (TYPE
	      (SETQ SENSITIVE-INSPECT-ITEM T)
	      (BLINKER-SET-VISIBILITY LIST-BLINKER NIL)
              (SETQ BWIDTH (- BWIDTH LEFT)
		    BHEIGHT
		      (or the-height (+ (FONT-BLINKER-HEIGHT CURRENT-FONT) 1)))
	      (if literal-p
		  (send item-blinker :Set-Cursorpos-Literally left top)
		  (blinker-set-cursorpos
		    ITEM-BLINKER (- LEFT (SHEET-INSIDE-LEFT))
		    (- TOP (SHEET-INSIDE-TOP)))
	      )
              (BLINKER-SET-SIZE ITEM-BLINKER BWIDTH BHEIGHT)
              (BLINKER-SET-VISIBILITY ITEM-BLINKER T))
             (T (BLINKER-SET-VISIBILITY LIST-BLINKER NIL)
              (BLINKER-SET-VISIBILITY ITEM-BLINKER NIL)
	      (SETQ SENSITIVE-INSPECT-ITEM NIL))))

(DEFMETHOD (BLINKER :SET-CURSORPOS-literally) (X Y &AUX (OLD-PHASE PHASE))
  "Set the position of a blinker relative to the sheet it is on.  Args in
terms of raster units.  If blinker was following cursor, it will no longer
be doing so."
  (WITH-BLINKER-READY T
    ;; Handle range checking for X and Y.
    (SETQ X (min (MAX (FLOOR X) 0) (SHEET-width SHEET))
	  Y (MIN (MAX (FLOOR Y) 0) (sheet-height sheet)))
    (COND (
           ;; Don't open if visibility NIL (especially the mouse cursor!)
           (NULL VISIBILITY)
	   (SETQ X-POS X
                 Y-POS Y
                 FOLLOW-P NIL))
	  ((OR (NEQ X X-POS)		;Only blink if actually moving blinker
	       (NEQ Y Y-POS))
	   (OPEN-BLINKER SELF)
	   (SETQ X-POS X
                 Y-POS Y
                 FOLLOW-P NIL
                 TIME-UNTIL-BLINK 0)
	   ;; If this is the mouse blinker, and it is not being tracked
           ;; by microcode, then it is important to turn it back on
           ;; immediately.
	   (AND (NEQ VISIBILITY :BLINK)
		OLD-PHASE
		(BLINK SELF))))))

;-------------------------------------------------------------------------------

(defun get-items-for-plist (object plist window &optional (prefix "    "))
  (ignore object)
  (let ((show-plist (allocate-data 'show-plist plist)))
       (let ((the-items (send show-plist :Generate-Item-Specialized window)))
	    (loop for line in (rest (rest the-items))
		  collect (cons prefix line)
	    )
       )
  )
)

(defun get-items-for-alist (object plist window &optional (prefix "    "))
  (ignore object)
  (let ((show-plist (allocate-data 'show-alist plist)))
       (let ((the-items (send show-plist :Generate-Item-Specialized window)))
	    (loop for line in (rest (rest the-items))
		  collect (cons prefix line)
	    )
       )
  )
)

(DEFMETHOD (BASIC-INSPECT :OBJECT-SYMBOL) (OBJ)	    ;!
  (declare (special eh:current-frame))
  `(((:ITEM1 SYMBOL-VALUE-CELL "Value: " PRINC)
     ; to inspect symbols from the window debugger in the error stack group.
     ,(if  eh:*error-sg*
	   (MULTIPLE-VALUE-BIND (obj-val obj-bound-in-sg?)
               (symeval-in-stack-group  obj eh:*error-sg*  eh:current-frame)
             (IF obj-bound-in-sg?
                 `(:ITEM1 SYMBOL-VALUE ,obj-val)
                 "unbound"))
	   (if (boundp obj)
	       `(:ITEM1 SYMBOL-VALUE ,(SYMBOL-VALUE OBJ))
	       "unbound")))
     ((:ITEM1 SYMBOL-FUNCTION-CELL "Function: " PRINC)
     ,(IF (FBOUNDP OBJ)
	  `(:ITEM1 SYMBOL-FUNCTION ,(SYMBOL-FUNCTION OBJ))
	  "unbound"))
    ("Package: "
     (:ITEM1 SYMBOL-PACKAGE ,(CAR (PACKAGE-CELL-LOCATION OBJ))))
    ((:ITEM1 SYMBOL-PROPERTY-CELL "Property list: " PRINC)
     (:ITEM1 SYMBOL-PROPERTY-LIST ,(SYMBOL-PLIST OBJ)))
    ,@(Get-Items-For-Plist obj (symbol-plist obj) self)))


(defun Make-Window-Items-For-Hash-Table
       (hash-table &optional (extractor 'identity) (sort-p t) &Aux maxlength)
"Takes the elements in a hash table and turns tham into a set of inspector
 items, sorted so that it's easy to find the ones you want.
"
   (Setq maxlength 0)
   (sys:maphash #'(Lambda (key &Rest Ignore)
                 (Setq maxlength (max (flatsize key) maxlength)))
             hash-table)
   (let ((list 
	   (sys:MapHash-Return
	     #'(Lambda (key &Rest values)
		 (Append
		   `((:Item1 Named-Structure-Value ,Key
			     ,#'(lambda (key stream)
				  (format stream "~S" key)))
		     (:colon ,(+ 2 maxlength)))
		   (Mapcar #'(Lambda (value)
			       (let ((value (funcall extractor value)))
				 `(:Item1 Named-Structure-Value
					  ,value
					  ,#'(lambda (value stream)
					       (format stream "~S " value)))))
			   values)))
	     hash-table)))
     (if sort-p
	 (sort list #'string-lessp :key 'item-key)
	 list
     )
   )
)