;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-

;;;  RDA: Note to Rice: I've been hacking on this file.  I'll leave it in
;;;  CPTFONT as flag.  Change it back after you check my chagnes.

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986 and 1987.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Most of this file derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

;;; This file implements a set of commands which add consistency between the
;;; tools already provided oon the Lispm.  For instance and inspect command
;;; is added to Zmacs, just there is one in the other tools.

;;; Also defined is a facility whereby edit buffer streams appear in the who
;;; line.

;-------------------------------------------------------------------------------
zwei:
(defvar zwei:*Zmacs-Buffer-Streams-Displayed-In-Who-Line-P* nil
"When true any open Zmacs buffer streams wil be displayed in the who line
 along with ordinary files.
"
)


(defvar *background-process-display-font* fonts:tiny
"The font in which to display background processes."
)

(defvar *who-line-run-state-toggle-time* 0.5
"The time after which the run state sheet toggles into showing
 background processes.
"
)

(defvar *show-background-processes-in-who-line* nil
"When true background processes that are interesting are shown in the who-line
 run-state part of the who line.
"
)

;(setq *show-background-processes-in-who-line* t)

(defvar *Chars-for-who-line-package* 7.
  "The number of chars in the who-line for the package name."
)
(export '*Chars-for-who-line-package* 'tv)

(defvar *Chars-for-who-line-process-state* 20.
  "The number of chars in the who-line for the process-state."
)
(export '*Chars-for-who-line-process-state* 'tv)

(defvar *Show-Two-Files-In-Who-Line* nil
  "When non-nil, two files will be shown in the who line when appropriate."
)
(export '*Show-Two-Files-In-Who-Line* 'tv)

 ;; End of microExplorer conditionalization

;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

;;; Commands to add.

;-------------------------------------------------------------------------------


(defparameter *all-consistancy-commands*
  '(("Inspect" :Value
      (((#\c-sh-i zwei:Com-Inspect-Region)
	(#\m-sh-i zwei:Com-Inspect-Region)
       )
       nil
       nil
      )
    )
    ("Eval" :Value
      (nil
       ((Inspect-Eval-Cmd t))
       nil
      )
    )
    ("MacroExpand" :Value
      (nil
       ((Inspect-MacroExpand-Cmd t))
       nil
      )
    )
    ("Compile" :Value
      (nil
       ((Inspect-Compile-Cmd t))
       ((eh:Comw-Compile-Cmd t))
      )
    )
    ("Save Region" :Value
      (nil
       ((save-region-Cmd nil))
       ((eh:Comw-Save-Region nil))
      )
    )
    ("ISearch" :Value
      (nil
       ((Search-Cmd nil))
       ((eh:Comw-ISearch nil))
      )
    )
    ("Reverse ISearch" :Value
      (nil
       ((reverse-Search-Cmd nil))
       ((eh:Comw-Reverse-ISearch nil))
      )
    )
    ("Arglist" :Value
      (nil
       ((Arglist-Cmd t))
       ((eh:Comw-Arglist-Cmd t))
      ) 
    )
    ("Trace" :Value
      (((#\s-sh-t zwei:com-untrace)
	(#\m-sh-t zwei:com-just-trace)
	(#\c-sh-t zwei:com-trace)
       )
       ((Trace-Cmd t)
	(Just-Trace-Cmd nil)
	(Untrace-Cmd nil)
       )
       ((eh:Comw-Trace t)
	(eh:Comw-Just-Trace nil)
	(eh:Comw-UnTrace nil)
       )
      )
    )
    ("Document" :Value
      (nil
       ((Document-Something-Cmd t)
	(Documentation-Cmd t)
       )
       ((eh:Comw-Debug-Document-Something t)
	(eh:Window-Debugger-Help-Cmd t)
       )
      )
    )
    ("Flavor Inspect" :Value
      (((#\c-sh-f zwei:com-flavor-inspect)
	(#\m-sh-f zwei:com-flavor-inspect)
	(#\h-f    zwei:com-flavor-inspect)
       )
       nil
       ((eh:Flavor-Inspect-Cmd t))
      )
    )
   )
"This is a list of the commands which can be added to the system.  The structure
 of this list is as follows :-
 It is made of items.  Each item is a list of the form (Menustring :Value spec)
 the spec is used to determine which commands for which tools is represented
 by the menu item.  The spec is a list of the form
 (zmacs-commands inspector-commands debugger-commands)
 Each element of zmacs-commands has the form (key command-name).
 Each element of inspector-commands and debugger-commands has the form
 (command-name put-in-frames-menu-p).
"
)



(defun convert-to-non-menu-item (item)
"Given a menu item which might generate a menuable command, this one returns
 a command which will certainly not appear in the frame's command menu.
"
  (append (list (first item) (second item))
	  (list
	  (cons (first (third item))
		(mapcar #'(lambda (component)
			    (mapcar #'(lambda (element)
					(list (first element) nil)
				      )
				      component
			    )
			  )
			  (rest (third item))
		)
	  )
	  )
  )
)


(defun convert-to-column-list (list)
"Given a list of items to put in the menu of commands, this turns it into
 a list of menu items suitable to be in two columns, one for commands to go
 into the frame's command menu and one which is not.
"
  (Append '(("Put in Menu" :NoSelect nil :Font Fonts:Hl12b))
	  '(("Don't put in Menu" :NoSelect nil :Font Fonts:Hl12b))
	  (apply #'append
		 (mapcar #'list list (mapcar 'convert-to-non-menu-item list))
	  )
  )
)



(defun install-zmacs-commmand (command)
"Installs a zmacs command."
  (zwei:set-comtab zwei:*standard-comtab* command
		   (zwei:make-command-alist (rest command))
  )
)


(defun reinstall-inspector-commands ()
"Reinstalls all inspector commands just in case they have changed."
  (build-command-table
    'inspector-menu-cmd-table
    'inspect-frame inspector-menu-cmds :init-options	
    '(:name "Inspector menu commands")
  )
  (build-command-table
    'inspector-other-cmd-table
    'inspect-frame inspector-non-menu-cmds :init-options	
    '(:name "Other Inspector commands")
  )
  (build-menu 'ucl-inspector-menu 'inspect-frame
    :default-item-options `(:font ,inspect-standard-font)	
    :item-list-order inspector-menu-cmds
  )
)

(defun install-inspector-command (command)
"Installs an inspector command."
  (let ((command-name (first command))
	(menu-p (second command))
       )
       (if menu-p
	   (if (member command-name inspector-menu-cmds)
	       nil
	       (progn (setq inspector-menu-cmds
			    (cons command-name inspector-menu-cmds)
		      )
		      (reinstall-inspector-commands)
	       )
	   )
	   (if (member command-name inspector-non-menu-cmds)
	       nil
	       (progn (setq inspector-non-menu-cmds
			    (cons command-name inspector-non-menu-cmds)
		      )
		      (reinstall-inspector-commands)
	       )
	   )
       )
  )
)


eh:
(defparameter eh:*debugger-examine-menu-commands*
  '(comw-inspect-cmd
    window-debugger-help-cmd 
    comw-edit-cmd
    comw-search-cmd
    bug-report-cmd
    comw-what-error-cmd 
    comw-arglist-cmd
    stay-cmd
    comw-set-arg-cmd
   )
"The commands for the window debugger extracted from the source code where they
 are a literal constant (groan).
"
)


eh:
(defparameter eh:*debugger-general-commands*
  '(comw-what-error-cmd
    comw-arglist-cmd
    comw-exit-cmd
    comw-inspect-cmd 
    com-top-level-throw-cmd
    comw-edit-cmd
    window-debugger-help-cmd 
    clear-screen-cmd
    bug-report-cmd
    all-commands-menu-cmd
   )
"The commands for the window debugger extracted from the source code where they
 are a literal constant (groan).
"
)

eh:
(defparameter eh:*Window-Debugger-All-Commands-Menu-Commands*
   '((COMW-WHAT-ERROR-CMD :COLUMN "General")
     (COMW-ARGLIST-CMD :COLUMN "General") 
     (COMW-EXIT-CMD :COLUMN "General")
     (COMW-INSPECT-CMD :COLUMN "General") 
     (COM-TOP-LEVEL-THROW-CMD :COLUMN "General")
     (COMW-EDIT-CMD :COLUMN "General") 
     (WINDOW-DEBUGGER-HELP-CMD :COLUMN "General")
     (CLEAR-SCREEN-CMD :COLUMN "General")
     (BUG-REPORT-CMD :COLUMN "General")
     (COMW-SEARCH-CMD :COLUMN "Stack") 
     (UP-STACK-CMD :COLUMN "Stack")
     (DOWN-STACK-CMD :COLUMN "Stack") 
     (PAGE-UP-STACK-CMD :COLUMN "Stack")
     (PAGE-DOWN-STACK-CMD :COLUMN "Stack") 
     (TOP-STACK-CMD :COLUMN "Stack")
     (BOTTOM-STACK-CMD :COLUMN "Stack") 
     (STEP-CMD :COLUMN "Step")
     (COM-TOGGLE-TRAP-ON-CALL-CMD :COLUMN "Step") 
     (COM-TOGGLE-FRAME-TRAP-ON-EXIT-CMD :COLUMN "Step") 
     (COM-SET-ALL-FRAMES-TRAP-ON-EXIT-CMD :COLUMN "Step") 
     (COM-CLEAR-ALL-FRAMES-TRAP-ON-EXIT-CMD :COLUMN "Step") 
     (COM-TOGGLE-ALL-FRAMES-TRAP-ON-EXIT-CMD :COLUMN "Step") 
     (TOGGLE-CONFIG-CMD :COLUMN "Step") (STAY-CMD :COLUMN "Step") 
     (COM-RETURN-REINVOCATION-CMD :COLUMN "Resume") 
     (COMW-SET-ARG-CMD :COLUMN "Resume")
     (COM-RETURN-A-VALUE-CMD :COLUMN "Resume") 
     (COM-PROCEED-CMD :COLUMN "Resume")
     (COM-THROW-CMD :COLUMN "Resume"))
"The commands for the window debugger extracted from the source code where they
 are a literal constant (groan).
"
)


(defvar eh:*resume-menu-commands*
       '(eh:com-toggle-trap-on-call-cmd 
	 eh:com-return-reinvocation-cmd
	 eh:com-toggle-frame-trap-on-exit-cmd
	 eh:com-proceed-cmd 
	 eh:com-toggle-all-frames-trap-on-exit-cmd
	 eh:com-return-a-value-cmd
	 eh:step-cmd 
	 eh:com-top-level-throw-cmd
	 eh:comw-exit-cmd
	)
)
(defvar eh:*window-debugger-menu-item-font* :Default)
(defvar eh:*window-debugger-menu-bold-font* 'fonts:hl12b)

eh:
(defun tv:reinstall-debugger-commands ()
"Reinstalls all debugger commands just in case they have changed."
  (build-command-table 'window-debugger-general-cmd-table
		       'debugger-frame
    *debugger-general-commands*
    :init-options
    '(:name "General window-based debugger commands")
  )
  (build-menu 'ucl-window-debugger-examine-menu
	      'debugger-frame
    :default-item-options `(:font ,eh:*window-debugger-menu-item-font*)
    :item-list-order
    (cons `("Examine" :font ,eh:*window-debugger-menu-bold-font*)
	  *debugger-examine-menu-commands*
    )
  )
  (build-menu '*window-debugger-all-commands-menu*
	      'debugger-frame
    :default-item-options `(:font ,eh:*window-debugger-menu-item-font*)
    :item-list-order *window-debugger-all-commands-menu-commands*
    :column-list-order
    `(("General" :FONT ,eh:*window-debugger-menu-bold-font*)
      ("Stack" :FONT ,eh:*window-debugger-menu-bold-font*) 
      ("Step" :FONT ,eh:*window-debugger-menu-bold-font*)
      ("Resume" :FONT ,eh:*window-debugger-menu-bold-font*)
     )
  )
  (build-menu 'ucl-window-debugger-resume-menu
            'debugger-frame
            :default-item-options
            `(:font ,eh:*window-debugger-menu-item-font*)
            :item-list-order
            `(("Resume" :font ,eh:*window-debugger-menu-bold-font*)
	      ,@eh:*resume-menu-commands*)
  )
)


(defun install-debugger-command (command)
"Installs a new debugger command."
  (let ((command-name (first command))
	(menu-p (second command))
       )
       (if menu-p
	   (if (member command-name eh:*debugger-examine-menu-commands*)
	       nil
	       (progn (setq eh:*debugger-examine-menu-commands*
			    (cons command-name
				  eh:*debugger-examine-menu-commands*
			    )
		      )
		      (reinstall-debugger-commands)
	       )
	   )
       )
       (if (member command-name eh:*debugger-general-commands*)
	   nil
	   (progn (setq eh:*debugger-general-commands*
			(cons command-name eh:*debugger-general-commands*)
		  )
		  (reinstall-debugger-commands)
	   )
       )
       (let ((general-command-spec (list command-name :Column "General")))
	    (if (member general-command-spec
			eh:*window-debugger-all-commands-menu-commands*
		)
		nil
		(progn (Setq eh:*window-debugger-all-commands-menu-commands*
			 (append eh:*window-debugger-all-commands-menu-commands*
				 (list general-command-spec)
			 )
		       )
		       (reinstall-debugger-commands)
		)
	    )
       )
  )
)



(defun install-command (command-spec)
"Given a command spec from the menu of commands to install, install the commands
 in all of the tools that are interested in them.
"
  (let ((zmacs-component     (first  command-spec))
	(inspector-component (second command-spec))
	(debugger-component  (third  command-spec))
       )
       (mapcar 'install-zmacs-commmand    zmacs-component)
       (mapcar 'install-inspector-command inspector-component)
       (mapcar 'install-debugger-command  debugger-component)
  )
)



(defvar *selected-consistancy-commands* :All
"Can have the value :All, in which case all consistancy commands are loaded,
 :Menu in which case the user is prompted, or a list of commands to load.
"
)



(defun select-and-install-commands (commands switch)
"Is passed a list of all of the commands that can be added and a switch
 which tells the system what commands to add.  This switch can have the values
 :Menu, :All or it can be a list of items.  Each item can be an element from
 Commands, a subitem from Commands (the bit following the :Value) or a string
 denoting the name of one of the menu items.
" 
  (let ((selected-items
	  (if (equal switch :Menu)
	      (w:menu-choose (convert-to-column-list commands)
			     :Highlighting t
			     :Highlighted-Items nil
			     :Near-Mode '(:Mouse)
			     :Label "Select commands"
			     :Menu-Margin-Choices '(:Doit)
			     :Geometry '(2)
			     :Superior Mouse-Sheet
	      )
	      (if (equal switch :All)
		  (mapcar #'Third commands)
		  (mapcar #'(lambda (command)
			      (if (consp command)
				  (if (stringp (first command))
				      (third command)
				      command
				  )
				  (if (stringp command)
				      (let ((entry (assoc command commands
							  :Test #'String-Equal
						   )
					    )
					   )
					   (if entry
					       (third entry)
					       (ferror nil
 "~S is not a defined command to add." command)
					   )
				      )
				      (ferror nil 
 "~S is not a valid command specifier." command)
				  )
			      )
			    )
			    (if (consp switch)
				switch
				(ferror nil
 "~S is not :Menu :All or a list of commands" switch)
			    )
		  )
	      )
	  )
        )
       )
       (mapcar 'install-command selected-items)
  )
)



(defun install-consistency-commands ()
"Installs all of the commands necessary to induce consistency in the system
 that the user wants.
"
  (select-and-install-commands *all-consistancy-commands*
			       *selected-consistancy-commands*
  )
)

;-------------------------------------------------------------------------------

;;; Support for fat strings.

format:
(defun tv:format-fat-string-stream (op &rest args)
  (case op
    (:tyo
     (or format-string (setq format-string (make-array 64. :type 'art-fat-string :fill-pointer 0.)))
      (vector-push-extend (first args) format-string))
    (:string-out
     (let ((string (first args))
	   (first (or (second args) 0.))
	   (last (third args))
	   new-length)
       (or format-string (setq format-string (make-array 64. :type 'art-fat-string :fill-pointer 0.)))
       (setq last (or last (length string)))
       (setq new-length (+ (array-leader format-string 0.) (- last first)))
       (and (< (array-total-size format-string) new-length)
	  (adjust-array format-string (+ (array-total-size format-string) new-length)))
       (copy-array-portion string first last format-string (array-leader format-string 0.)
			   new-length)
       (store-array-leader new-length format-string 0.)))
    (:read-cursorpos
     (let ((mode (or (first args) :character))
	   pos)
       (or format-string (setq format-string (make-array 64. :type 'art-fat-string :fill-pointer 0.)))
       (or (eq mode :character) (ferror () "String cannot have :PIXEL"))
       (setq pos
	     (position #\NEWLINE (the string (string format-string)) :from-end t :test
		       #'char-equal))
       (values (- (length format-string) (if pos
					   (+ pos 1.)
					   0.)) 0.)))
    (:increment-cursorpos
     (let ((dx (first args))
	   (dy (second args))
	   (mode (or (third args) :character))
	   newlen)
       (or format-string (setq format-string (make-array 64. :type 'art-fat-string :fill-pointer 0.)))
       (or (eq mode :character) (ferror () "String cannot have :PIXEL"))
       (or (and (zerop dy) (not (minusp dx))) (ferror () "Cannot do this :INCREMENT-CURSORPOS"))
       (setq newlen (+ (length format-string) dx))
       (and (< (array-total-size format-string) newlen)
	  (adjust-array format-string (+ (array-total-size format-string) newlen)))
       (do ((i (length format-string) (1+ i)))
	   ((>= i newlen))
	 (setf (aref format-string i) #\SPACE))
       (store-array-leader newlen format-string 0.)))
    (:set-cursorpos
     (let ((x (first args))
	   (y (second args))
	   (mode (or (third args) :character))
	   pos
	   delta
	   newlen)
       (or format-string (setq format-string (make-array 64. :type 'art-fat-string :fill-pointer 0.)))
       (or (eq mode :character) (ferror () "String cannot have :PIXEL"))
       (setq pos (string-reverse-search-set '(#\NEWLINE #\LINEFEED #\PAGE) format-string)
	     delta (- x (- (length format-string) (if pos
						    (+ pos 1.)
						    0.))))
       (or (and (zerop y) (plusp delta)) (ferror () "Cannot do this :SET-CURSORPOS"))
       (setq newlen (+ (length format-string) delta))
       (and (< (array-total-size format-string) newlen)
	  (adjust-array format-string (+ (array-total-size format-string) newlen)))
       (do ((i (length format-string) (1+ i)))
	   ((>= i newlen))
	 (setf (aref format-string i) #\SPACE))
       (store-array-leader newlen format-string 0.)))
    (:untyo-mark (fill-pointer format-string))
    (:untyo (let ((mark (first args)))
	      (setf (fill-pointer format-string) mark)))
    (extract-string (prog1
		      format-string
		      (setq format-string ())))
    (:get-string format-string)
    (:fresh-line
     (when (not
       (or (null format-string) (zerop (length format-string))
	  (= (aref format-string (1- (length format-string))) #\NEWLINE)))
       (vector-push-extend #\NEWLINE format-string)
       t))
    (:which-operations nil
     '(:tyo :string-out :read-cursorpos :increment-cursorpos :set-cursorpos :untyo-mark :untyo
       extract-string :fresh-line))
    (t (stream-default-handler 'format-string-stream op (car args) (cdr args)))))


(defun make-fat-string-output-stream (&optional string start-index extra-arg)
  (if (stringp start-index)
    (let ((string start-index)
	  (start-index extra-arg))
      (let-closed
       ((format-string
	  (or string
	      (make-array 64. :type 'art-fat-string :fill-pointer 0.))))
       (if start-index
	 (setf (fill-pointer format-string) start-index))
       'format-string-stream))
    (let-closed
      ((format-string
	 (or string
	     (make-array 64. :type 'art-fat-string :fill-pointer 0.))))
       (if start-index
	 (setf (fill-pointer format-string) start-index))
       'format-fat-string-stream)))

(defmacro with-output-to-fat-string
	  ((stream string index) &body body &aux (string-symbol string))
  (multiple-value-bind (realbody decls)
      (parse-body body nil nil)
    (let ((doc (and  decls `((declare . ,(sys:flatten-declarations decls))))))
      (if index
	  `(let* (,@(and (not (symbolp string))
			 `((,(setf string-symbol (gensym))  ,string)))
		  (,stream
		   (make-fat-string-output-stream ,string-symbol ,index)))
	     ,@doc 
	     (unwind-protect
		     (progn
		       ,@realbody)
		   (setf ,index (length ,string-symbol))))
	      `(let ((,stream (make-fat-string-output-stream
				,@(if string `(,string)))))
		 ,@doc
		 ,@realbody
		 ,@(if (null string )
		       `((get-output-stream-string ,stream))))))))


;;;TI Code:
sys:
(defun sys:make-string-input-stream (string &optional (start 0) end)
  "Return a stream from which one can read the characters of STRING, or some substring of it.
START and END are indices specifying a substring of STRING;
they default to 0 and NIL (NIL for END means the end of STRING)."
  ;;RDA: jpr seems to have added the (IF (TYPEP STRING 'ARRAY) STRING part
  (setq string (if (typep string 'array) string (string string)))
  (let-closed ((*ioch start) (*ioend (or end (length string))) (*iolst string))
     'read-from-string-stream))

;-------------------------------------------------------------------------------

;;; Support for searching and grovling over items in inspectors and such-like.


;;; A defstruct with which to remember the saved points in the searches.
(defstruct (saved-pos :named) line char printed string failed-p)

(defun (:property saved-pos Named-Structure-Invoke)
       (message-name pos &Rest arguments)
  (case message-name
    (:Print-Self
      (let ((stream   (first arguments)))
	   (format stream "#<Saved Pos: ~A, ~A>" (saved-pos-line pos)
		   (saved-pos-char pos)
	   )
      )
    )
    (:Which-Operations '(:Which-Operations :Print-Self))
    (otherwise (ferror "Illegal message ~A sent to a saved pos."))
  )
)

(defparameter *last-search-string* ""
"The last string used by the user in searching."
)

;-------------------------------------------------------------------------------

(defparameter *fontify-this-region* nil
"When true tells the pretty printer to fontify the region.  If it is not source
 code debugging then its value is the font number to fontify to.
"
)

(defparameter *font-for-selected-region* 1
"The font number to use for the selected region in source code debugging."
)

(defun shifted-font ()
"Returns the font number to which to shift."
  (declare (special eh::*grinding-debugged-source-code* eh::*current-pc*))
  (if (and (boundp 'eh::*grinding-debugged-source-code*)
	   eh::*grinding-debugged-source-code*
      )
      ;;; Hook for Source code debugger.
      (if (equal (first (funcall 'eh::numbered-component-numbers
			  (first eh::*grinding-debugged-source-code*)
			)
		 )
		 eh::*current-pc*
	  )
	  *font-for-selected-region*
	  (+ 1 *font-for-selected-region*)
      )
      (if (numberp *fontify-this-region*)
	  *fontify-this-region*
	  0
      )
  )
)

(defun fontify-char (char)
"Given a char returns the char fontified to the (shifted-font)."
  (code-char (typecase char
	       (integer (code-char char))
	       (character (code-char (char-code char)))
	       (otherwise (beep) char)
	     )
	     0 (or (shifted-font) 0)
  )
)

(defflavor pseudo-itemising-stream
	   (stream  ;;; Actual stream to print to.
	    (current-font 0) ;;; Printing in this font.
	    superior ;;; The superior of the current window, which created me.
	    (width 0) ;;; The current printed width in pixels.
	    max-width ;;; The max width of the window.
	    font-map  ;;; The font map of the window.
	   )
	   (si:output-stream)
  :Initable-instance-variables
  :settable-instance-variables
  :gettable-instance-variables
  (:Documentation "A stream which font-shifts all of its characters if
 necessary.")
)

(defmethod (Pseudo-Itemising-Stream :Set-Current-Font) (to &rest ignore)
  (setq current-font to)
)

(defmethod (pseudo-itemising-stream :tyo) (char &rest args)
"Prints char to the stream, fontifying it to the stream's current font."
  (let ((the-char (code-char (char-code (fontify-char char)) 0 current-font)))
       ;;; Record the width of the char.
       (setq width (+ (font-char-width (aref font-map current-font)) width))
       ;;; Throw if we've written too much, otherwise print it.
       (if (>= width max-width)
	   (throw 'truncate nil)
	   (lexpr-send stream :tyo the-char args)
       )
  )
)

(defmethod (pseudo-itemising-stream :item1)
    (item type &optional (function #'prin1) &rest print-args)
"A dummy version of the :item1 method."
  (ignore type)
  (apply function item self print-args)
)

(defmethod (pseudo-itemising-stream :compound) (some-items)
  (loop for item in some-items do
	(if (consp item)
	    (lexpr-send self item)
	    (format self "~A" item)
	)
  )
)

(defmethod (pseudo-itemising-stream :read-cursorpos) (&rest args)
"Passes this message on so that tabbing works ok."
  (lexpr-send stream :read-cursorpos args)
)

(defmethod (pseudo-itemising-stream :increment-cursorpos)
	   (x-increment y-increment &optional (mode :pixel))
"Passes this message on so that tabbing works ok."
  ;;; Increment width as approriate.
  (setq width (+ (if (equal mode :pixel)
		     x-increment
		     (* (font-char-width (aref font-map 0)) x-increment)
		 )
		 width
	      )
  )
  ;;; Do this because sometimes we get negative increments on 0 length strings.
  (catch-error (send stream :increment-cursorpos x-increment y-increment mode)
	       nil
  )
)

;-------------------------------------------------------------------------------

(defun print-out-item
       (print-function print-function-arg item item-number width)
"Prints out a window item for an inspect window into a fat string."
  (let ((result
	  (if (equal (send self :print-function-arg)
		     :list-structure
	      )
	      (let ((item (aref (send self :items) item-number)))
		   (if (and (consp item) (stringp (second item)))
		       (second item)
		       ""
		   )
	      )
	      (with-output-to-fat-string (string-stream)
		(let ((*standard-output*
			(make-instance
			  'pseudo-itemising-stream
			  :stream    string-stream
			  :superior  (send self :superior)
			  :max-width (or width (send self :width))
			  :font-map  (send self :font-map)
			)
		      )
		     )
		     (catch 'truncate
			    (funcall print-function item print-function-arg
				     *standard-output* item-number
			    )
		     )
		)
	      )
	  )
	)
       )
       (if result result "")
  )
)

(defmethod (function-text-scroll-window :stringise-item)
	   (item-number cache &optional (max-width (send self :width)))
"Given an item number from self's items, finds a fat-string printed
 representation of it in the cache or prints it to a fat string and caches it
 as needed.  Returns the string.
"
  (or (aref cache item-number)
      (let ((str (print-out-item print-function print-function-arg
				 (send self :item-of-number item-number)
				 item-number max-width
		 )
	    )
	   )
	   (setf (aref cache item-number) str)
	   str
      )
  )
)

(defun search-for-item
    (cache string item-number from-line from-char reverse-p)
"Searches forwards or backwards, according to reverse-p for String in the item
 of self, whose index is Item-Number.  It starts looking from the beginning or
 end of the item as approriate, using from-line and from-char to tell it where
 to start from if the search is on the line that we're already on - we have to
 make sure that we don't find the same match again unless we want to.
"
  (let ((printed
	  (send self :stringise-item item-number cache most-positive-fixnum)
	)
       )
       (let ((index
	        (if (equal from-line item-number)
		    (search string printed :test #'char-equal
			    :start2 (if reverse-p nil from-char)
			    :end2   (if reverse-p
					(+ (length string) from-char)
					nil
				    )
			    :from-end reverse-p
		    )
		    (search string printed :test #'char-equal
			    :from-end reverse-p
		    )
		)
	     )
	    )
	    (if index
		(values index printed)
		(values nil nil)
	    )
       )
  )
)


(defmethod (function-text-scroll-window :search)
	   (cache for &optional (from-line 0) (from-char 0) (printed-line "")
	    (reverse-p nil)
	   )
"Searches for the string For in the window items self either forwarrds or
 backwards, as defined by reverse-p, starting from From-line, From-Char.
"
  (ignore printed-line)
  (if reverse-p
      (loop for item downfrom from-line to 0
	    do (multiple-value-bind (char line)
		   (search-for-item cache for item from-line from-char t)
		 (if char (return item char line) nil)
	       )
	    finally (return nil nil nil)
      )
      (loop for item from from-line to (- (send self :number-of-items) 1)
	    do (multiple-value-bind (char line)
		   (search-for-item cache for item from-line from-char nil)
		 (if char (return item char line) nil)
	       )
	    finally (return nil nil nil)
      )
  )
)

(defmethod (function-text-scroll-window :move-to) (pos reverse-p)
"Moves Self to the position described by Pos, scrolling the window so that the
 item is at the top if necessary.  The blinker is moved to point at the
 beginning of the string and sized to fit the char under it.
"
  (multiple-value-bind (bottom line) (send self :bottom-item-no-on-screen)
    (ignore line)
    (if (or (< (saved-pos-line pos) top-item) (> (saved-pos-line pos) bottom))
	(send self :scroll-to (saved-pos-line pos) :absolute)
	nil
    )
    (if (string-equal "" (saved-pos-printed pos))
	nil
	(let ((char (+ (aref (saved-pos-printed pos) (saved-pos-char pos))
		       (if reverse-p 0 (length (saved-pos-string pos)))
		    )
	      )
	     )
	     (send (first blinker-list) :set-visibility :blink)
	     (let ((font (aref font-map (char-font char))))
	          (let ((bl-width (if (and (font-char-width-table font)
					   (> (aref (font-char-width-table font)
						    (char-code char)
					      )
					      0
					   )
				      )
				      (aref (font-char-width-table font)
					    (char-code char)
				      )
				      (font-char-width font)
				  )
			)
		       )
		       (multiple-value-bind (final-x final-y)
			   (sheet-compute-motion self 0 0
			       (saved-pos-printed pos) 0
			       (+ (if reverse-p
				      0
				      (length (saved-pos-string pos))
				  )
				  (saved-pos-char pos)
			       )
			   )
			 (send (first blinker-list) :set-size-and-cursorpos
			       bl-width
			       (font-char-height font)
			       (if (> final-y 0)
				   (- (sheet-inside-right self) bl-width)
				   final-x
			       )
			       (* (send self :line-height)
				  (- (saved-pos-line pos) top-item)
			       )
			 )
		       )
		  )
	     )
	)
    )
  )
)


(defmacro with-saved-pos (&body body)
"Binds the names Line, Char and Printed to position at the top of the
 saved position stack.
"
  `(let ((pos (first saved-positions)))
        (declare (unspecial pos))
	(check-type pos saved-pos)
        (let ((line    (saved-pos-line    pos))
	      (char    (saved-pos-char    pos))
	      (printed (saved-pos-printed pos))
	      (string  (saved-pos-string  pos))
	     )
	     (declare (unspecial line char printed string))
	     (check-type line    integer)
	     (check-type char    integer)
	     (check-type printed string)
	     (check-type string  string)
	     ,@body
	)
   )
)

(defun new (new-string line char printed failed-p)
"Create a new saved position record for new-string, line, char.  If the last one
 was a failed one then do nothing.  If this one is failed then use the line char
 etc of the last one.  Beep if this is the first failure.
"
  (declare (special saved-positions))
  (if failed-p
      (if (and (saved-pos-failed-p (first saved-positions))
	       (string-equal (saved-pos-string (first saved-positions))
			     new-string
	       )
	  )
	  nil ;;; We've already failed so do nothing.
	  (with-saved-pos
	    (if (saved-pos-failed-p (first saved-positions)) nil (beep))
	    (push (make-saved-pos :line line
				  :char char
				  :printed printed
				  :string new-string
				  :failed-p failed-p
		  )
		  saved-positions
	    )
	  )
      )
      (push (make-saved-pos :line line
			    :char char
			    :printed printed
			    :string new-string
			    :failed-p failed-p
	    )
	    saved-positions
      )
  )
)

(defmethod (function-text-scroll-window :process-other-char)
	   (cache char reverse-p)
"Processes a typed in char which is not a special one, such as c-s.  Continues
 the search.
"
  (declare (special *last-search-string* saved-positions))
  (let ((new-string
	  (string-append (saved-pos-string (first saved-positions)) char)
	)
       )
       (multiple-value-bind (line char printed)
	   (with-saved-pos (ignore string)
	     (send self :search cache new-string line char printed reverse-p)
	   )
	 (if line
	     (progn (new new-string line char printed nil)
		    (setq *last-search-string* new-string)
		    (send self :move-to (first saved-positions) reverse-p)
	     )
	     (with-saved-pos (ignore string)
	       (new new-string line char printed t)
	     )
	 )
       )
  )
)

(defmethod (function-text-scroll-window :process-next-search) (cache reverse-p)
"Processes a request to continue the search, i.e. c-s or c-r.  Reverse-p defines
 whether to search forwards or backwards.
"
  (declare (special saved-positions))
  (multiple-value-bind (line char printed)
      (with-saved-pos
	(let ((string (if (equalp string "") *last-search-string* string)))
	     (declare (unspecial string))
	     (send self :search cache string line (+ (if reverse-p -1 1) char)
		   printed reverse-p
	     )
	)
      )
      (let ((string (if (equalp (saved-pos-string (first saved-positions)) "")
			*last-search-string*
			(saved-pos-string (first saved-positions))
		    )
	    )
	   )
	   (declare (unspecial string))
	   (if line
	       (progn (new string line char printed nil)
		      (send self :move-to (first saved-positions) reverse-p)
	       )
	       (new string line char printed t)
	   )
      )
  )
)

(defmethod (function-text-scroll-window :show-search-string) (reverse-p)
"Print out the search string in the prompt window i nthe form \"ISearch: foo\".
 Also prints out Reverse and Failing, as appropriate.
"
  (declare (special saved-positions))
  (send *query-io* :set-cursorpos 0
       (second (multiple-value-list (send *query-io* :read-cursorpos)))
  )
  (send *query-io* :clear-eol)
  (if (saved-pos-failed-p (first saved-positions))
      (format *query-io* "Failing ")
      nil
  )
  (if reverse-p (format *query-io* "Reverse ") nil)
  (format *query-io* "I-Search: ~A" (saved-pos-string (first saved-positions)))
)


(defmethod (function-text-scroll-window :get-a-char) ()
"Reads a char from the user, coercing it into a char if it can.  It does this so
 that it can spot mouse blips and yet still get char objects.
"
  (let ((char (send *query-io* :any-tyi)))
       (typecase char
	 (cons char)
	 (integer (int-char char))
	 (character char)
	 (otherwise (beep))
       )
  )
)

(defmethod (function-text-scroll-window :control-s-internal) (reverse-p cache)
"Loops reading chars from the user and searches for the designated thing in the
 window, like c-s/c-r in ZMacs.
"
  (declare (special saved-positions))
  (loop for char = (send self :get-a-char)
	do (case char
	     (#\ (format *query-io* "~&") (return nil))
	     (#\c-s
	      (setq reverse-p nil)
	      (send self :process-next-search cache reverse-p)
	     )
	     (#\c-r
	      (setq reverse-p :backwards)
	      (send self :process-next-search cache reverse-p)
	     )
	     (#\rubout
	       (if (rest saved-positions)
		   (progn (pop saved-positions)
			  (send self :move-to (first saved-positions) reverse-p)
		   )
		   (beep)
	       )
	     )
	     (otherwise
	      (if (consp char)
		  (progn (format *query-io* "~&") (return nil))
		  (send self :process-other-char cache char reverse-p)
	      )
	     )
	   )
	   (send self :show-search-string reverse-p)
  )
)


(defmethod (function-text-scroll-window :bottom-item-no-on-screen) ()
"Returns values of the item# for the bottom item on the screen and the number
 of lines on the screen.
"
  (let ((lines-on-screen (floor (/ (send self :inside-height)
				   (send self :line-height)
				)
			 )
        )
       )
       (values (+ top-item lines-on-screen) lines-on-screen)
  )
)


(defmethod (function-text-scroll-window :start-saved-position) (reverse-p cache)
"Determines an initial position from which to start the search and makes a
 saved position do hold it.  It has to stringise a line to do this so the cache
 is passed so that it can be saved.
"
  (if reverse-p
      (let ((line (- (min (send self :bottom-item-no-on-screen)
		          (send self :number-of-items)
		     )
		     1
		  )
	    )
	   )
	   ;;; End of last line on screen.
	   (let ((string (send self :stringise-item line cache)))
	       `(,(make-saved-pos :line line :printed string :string ""
			:char (max 0 (- (array-active-length string) 1))
			:failed-p nil
		  )
		     
		)
	   )
      )
      ;;; Beginning of first line on screen.
     `(,(make-saved-pos :line (send self :top-item) :char 0 :string "" :printed
           (send self :stringise-item 0 cache) :failed-p nil
	)
      )
  )
)

(defmethod (function-text-scroll-window :reset-blinkers) ()
"Cleans up blinkers.  Sometimes these can get confused.  This is bit kludgy."
  (let ((blinkers (butlast (tv:sheet-blinker-list self) 2)))
       (mapcar #'(lambda (x)
		   (send x :set-visibility nil)
		   (setf (tv:sheet-blinker-list self)
			 (remove x (tv:sheet-blinker-list self))
		   )
		 )
		 blinkers
       )
  )
)

(defmethod (function-text-scroll-window :make-a-blinker) (reverse-p)
"Makes a blinker and sets its initial position.  The initial position is
 determined by whether we are going forwards or backwards.
"
  (make-blinker self 'rectangular-blinker
		;;; These positions are a bit random.
		:x-pos (if reverse-p (- (send self :inside-width)  10) 10)
		:y-pos (if reverse-p (- (send self :inside-height) 10)  0)
  )
)

(defmethod (function-text-scroll-window :control-s) (reverse-p)
"Searches over the items in an inspect window (optionally backwards) in such
 a manner that it looks like I-Search in ZMacs.
"
  (send self :reset-blinkers)
  (format *query-io* (if reverse-p "~&Reverse I-Search: " "~&I-Search: "))
  (if (> (send self :number-of-items) 0)
      (let ((cache (make-array (send self :number-of-items))))
	   (let ((saved-positions
		   (send self :start-saved-position reverse-p cache)
		 )
		 (old-visibilities (get-visibility-of-all-sheets-blinkers self))
		 (blinker (send self :make-a-blinker reverse-p))
		)
	        (declare (special saved-positions)
			 (unspecial old-visibilities blinker)
		)
	        ;;; These functions would be IVs if this was to be a mixin.
	        ;;; Mixing it in would be a good idea in principle but a drag
	        ;;; in practice.
	        (unwind-protect
		    (progn (set-visibility-of-all-sheets-blinkers
			     self
			     (cons :blink
				   (make-list (length old-visibilities))
			     )
			   )
			   (send self :control-s-internal reverse-p cache)
		    )
		  (open-blinker blinker)
		  (set-visibility-of-all-sheets-blinkers
		    self (cons nil old-visibilities)
		  )
		  (setf (sheet-blinker-list self)
			(remove blinker (sheet-blinker-list self))
		  )
		)
	   )
      )
      (progn (format *query-io* "Nothing in window.") (beep))
  )
)

(defun get-a-pane (frame numeric-arg)
"Given a frame and the numeric arg typed by the user, if any, it selects a pane
 in the frame to use.  This should be an instance of
 Function-Text-Scroll-Window.  If there is a numeric arg, then it picks the
 window under the mouse.  If not then it picks the first
 Inspect-Pane-With-Typeout instance in the exposed-inferiors, failing that it
 picks the first Basic-Inspect.
"

  (if numeric-arg
      (if (typep (window-under-mouse) 'function-text-scroll-window)
	  (window-under-mouse)
	  (progn (beep)
		 (format *query-io*
			 "~&Cannot search in window under mouse.~&"
		 )
		 nil
	  )
      )
      (or (find-if
	    #'(lambda (x) (typep x 'inspect-pane-with-typeout))
	    (send frame :exposed-inferiors)
	  )
	  (find-if
	    #'(lambda (x) (typep x 'inspect-window-with-typeout))
	    (send frame :exposed-inferiors)
	  )
	  (find-if
	    #'(lambda (x) (typep x 'basic-inspect))
	    (send frame :exposed-inferiors)
	  )
      )
  )
)

(defun process-control-s (frame reverse-p numeric-arg)
"Searches over the items in the main inspect window (optionally backwards) of
 the frame in such a manner that it looks like I-Search in ZMacs.
"
  (let ((pane (get-a-pane frame numeric-arg)))
       (if pane (send pane :control-s reverse-p) nil)
  )
)

;-------------------------------------------------------------------------------


(defmethod (function-text-scroll-window :meta-w) ()
"Like m-w in Zmacs, copies the text in the window into the kill ring."
  (if (> (send self :number-of-items) 0)
      (let ((cache (make-array (send self :number-of-items))))
	   (multiple-value-bind (bottom ignore)
	       (send self :bottom-item-no-on-screen)
	     (let ((result nil))
	          (loop for i
			from top-item
			to (- (min bottom (send self :number-of-items)) 1)
			do (push (send self :stringise-item i cache) result)
			   (push #\newline result)
		  )
		  (zwei:kill-string
		    (apply #'string-append
			   (cons (format nil "~A~&" (sixth (send self :label)))
				 (reverse result)
			   )
		    )
		    nil nil
		  )
	     )
	   )
      )
      (progn (format *query-io* "Nothing in window.") (beep))
  )
)

(defun process-meta-w (frame numeric-arg)
"Called when the user does a m-w in the inspector/debugger.  Copies the text
 in the window into the kill ring as appropriate.
"
  (let ((pane (get-a-pane frame numeric-arg)))
       (if pane (send pane :meta-w) nil)
  )
)

;-------------------------------------------------------------------------------

;;; The Meta-W command (save-region).

(defcommand save-region-cmd (numeric-arg)
  '(:description
     "Puts the contents of the window into the kill ring, numeric arg for window under mouse."
    :names ("Save Region")
    :keys (#\m-w)
    :arguments (ucl:numeric-argument)
   )
  (declare (special frame))
  (process-meta-w frame numeric-arg)
  (send frame :handle-prompt)
)

eh:
(defcommand eh:Comw-Save-Region (numeric-arg)
  '(:description
     "Puts the contents of the window into the kill ring, numeric arg for window under mouse."
     :names ("Save Region")
     :keys (#\m-w)
     :arguments (ucl:numeric-argument)
   )
  (declare (special *window-debugger*))
  (tv:process-meta-w *window-debugger* numeric-arg)
  (send *window-debugger* :handle-prompt)
)

;-------------------------------------------------------------------------------

;;; The ISearch Command.

(defcommand search-cmd (numeric-arg)
  '(:description
     "I-Search for text in window, numeric arg for window under mouse."
    :names ("Search")
    :keys (#\c-s #\c-sh-s)
    :arguments (ucl:numeric-argument)
   )
  (declare (special frame))
  (process-control-s frame nil numeric-arg)
  (send frame :handle-prompt)
)


eh:
(defcommand eh:Comw-ISearch (numeric-arg)
  '(:description
     "I-Search for text in window, numeric arg for window under mouse."
     :names ("Search")
     :keys (#\c-sh-s)
     :arguments (ucl:numeric-argument)
   )
  (declare (special *window-debugger*))
  (tv:process-control-s *window-debugger* nil numeric-arg)
  (send *window-debugger* :handle-prompt)
)

;-------------------------------------------------------------------------------

;;; The Reverse ISearch command.

(defcommand refresh-cmd nil
  '(:description 
"Redisplay the inspected objects, updating any fields that have changed values."
;;; Changed from Control r by JPR.
    :names ("Refresh" "Decache") :keys (#\m-R))
  (declare (special history frame))
  (send history :set-cache nil)
  (update-panes)
)

(defcommand reverse-search-cmd (numeric-arg)
  '(:description
     "Reverse I-Search for text in window, numeric arg for window under mouse."
    :names ("Reverse Search")
    :keys (#\c-r #\c-sh-r)
    :arguments (ucl:numeric-argument)
   )
  (declare (special frame))
  (process-control-s frame :backwards numeric-arg)
  (send frame :handle-prompt)
)

eh:
(defcommand eh:Comw-Reverse-ISearch (numeric-arg)
  '(:description
     "Reverse-I-Search for text in window, numeric arg for window under mouse."
     :names ("Reverse Search")
     :keys (#\c-sh-r)
     :arguments (ucl:numeric-argument)
   )
  (declare (special *window-debugger*))
  (tv:process-control-s *window-debugger* :backwards numeric-arg)
  (send *window-debugger* :handle-prompt)
)

;-------------------------------------------------------------------------------

;;; The Inspect command

;-------------------------------------------------------------------------------


;;; Code for the implementation of an Inspect Region command in Zmacs.

Zwei:
(defun zwei:read-until-eof (stream)
"Given a stream this function returns a list of all of the forms that it reads
from the stream until it gets to the end of file.
"
  (let ((sexpr (read stream nil :Eof)))
       (if (equal sexpr :Eof) nil (Cons sexpr (read-until-eof stream)))
  )
)


Zwei:
(defun zwei:read-form-or-forms-from-buffer (&optional (section-p nil))
"This function reads either a form or a collection of forms from the current
buffer.  If a section has been marked out then forms are read from this,
otherwise a form is read from after the cursor.  If only one form is found
then this is returned.  If more than one form is found then a list containing
these forms is returned.
"
  (if (or (Window-Mark-P *Window*) section-p)
      (let ((bp1 (mark))
	    (bp2 (point))
	    (defun-name nil)
	   )
	   (or (bp-< bp1 bp2) (psetq bp1 bp2 bp2 bp1))
	   (if (bp-= (forward-over *whitespace-chars* (mark))
		     (forward-over *whitespace-chars* (point)) )
	       (setq *mark-stays* nil)
	       (setq defun-name "Region")
	   )
	   (cond (defun-name)
		 ((setq bp1 (defun-interval (beg-line (point)) 1 () ()))
		  (setq bp2 (interval-last-bp bp1) bp1 (interval-first-bp bp1))
		  (setq si:*force-defvar-init* t)
		 )
		 (t (barf "Unbalanced parentheses"))
	   )
	   (let ((stream (Interval-Stream bp1 bp2 t)))
	        (unwind-protect
		  (let ((all-sexprs (read-until-eof stream)))
		       (values (if (equal (length all-sexprs) 1)
				   (first all-sexprs)
				   all-sexprs
			       )
			       (if defun-name
				   defun-name
				   (send (array-leader (bp-line bp1) 5) :Name)
			       )
		       )
		  )
		  (close stream)
		)
	   )
      )
      (let ((stream (rest-of-interval-stream (point))))
	   (unwind-protect
	     (let ((sexpr (read stream nil :Eof)))
	          (values (if (equal :Eof sexpr) :Nothing-found sexpr)
			  "Expression"
		  )
	     )
	     (close stream)
	   )
      )
  )
)


Zwei:
(Defcom zwei:Com-Inspect-Region "Call the inspector on a region." ()
  (let ((sexprs (read-form-or-forms-from-buffer)))
       (if (equal :Nothing-found sexprs)
	   (beep)
	   (inspect sexprs)
       )
  )
  Dis-None
)

;-------------------------------------------------------------------------------

;;; Add key assignments to the inspect command in the debugger.

eh:
(DEFCOMMAND eh:COMW-INSPECT-CMD NIL
 '(:DESCRIPTION  "Inspect an object specified with keyboard or mouse."
   :NAMES "Inspect" 
   :KEYS (#\c-sh-I #\M-sh-I #\c-I))
 (SEND *WINDOW-DEBUGGER* :SET-WHO-LINE-DOC-STRING-OVERIDE "Select an object to inspect.")
 (UNWIND-PROTECT
  (PROGN
   (COMW-INSPECT *ERROR-SG* *ERROR-OBJECT*)
   (TV:DELAYING-SCREEN-MANAGEMENT
    (WHEN (EQUAL (SEND *WINDOW-DEBUGGER* :CONFIGURATION) 'STEP-CONFIGURATION)
      (SEND *WINDOW-DEBUGGER* :SET-CONFIGURATION 'DEBUGGER-CONFIGURATION))))
  (PROGN (SEND *WINDOW-DEBUGGER* :SET-WHO-LINE-DOC-STRING-OVERIDE NIL)
         (SEND *WINDOW-DEBUGGER* :HANDLE-PROMPT))))


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

;;; Modifications to the macroexpand command and additions of macroexpand
;;; to tools.

;-------------------------------------------------------------------------------

;;; Code for the modification of the macro-expand-expression (c-sh-m) command
;;; in ZMacs.

zwei:
(defun zwei:map-with-args (a-function over-a-list &Rest other-arguments)
"This function is much like mapcar only more useful.  It takes a function and a
 list to map the function over and an &Rest arguments feature.  It applies the
 function to each element in the list, with the element being the first
 argument and any subsequent arguments being taken from the &Rest paremeter. 
 The value of a call to this function is a list of values from this function
 call, one element for each element in the source list.
"
  (Loop For element In over-a-list
        Collect (Apply a-function element other-arguments)
  )
)


zwei:
(defun zwei:expand-n-times (form n)
"Takes a form and a depth count.  It returns a new form, which has macros in
 form expanded to a depth of n."
  (if (< n 1)
      form
      (if (consp form)
	  (let ((expanded-form (sys:macroexpand-1 form)))
	       (If (not (equalp expanded-form form))
		   (expand-n-times expanded-form (- n 1))
		   (if (consp expanded-form)
		       (if (equal form expanded-form)
			   (map-with-args 'expand-n-times form n)
			   (map-with-args 'expand-n-times expanded-form (- n 1))
		       )
		       (values form expanded-form)
		   )
	       )
	  )
          form
      )
  )
)



;;TI Code:
zwei:
(DEFCOM zwei:COM-MACRO-EXPAND-EXPRESSION 
"Print macroexpansion of next s-expression or marked s-expression(s).
The result is printed on the screen with PPrint.
If a numeric arg is supplied then that number of macroexpansions is applied." ()
  (let ((form (read-form-or-forms-from-buffer)))
    (and (eq form :Nothing-found) (barf))
    (if *numeric-arg-p*
	(pprint
	    (expand-n-times (sys:macroexpand-1 form) (- *numeric-arg* 1))
	)
	(pprint (sys:macroexpand-1 form))
    )
  )
  dis-none
)


;-------------------------------------------------------------------------------

;;; Code for the implementation of a MacroExpand command in the Inspector.

;-------------------------------------------------------------------------------



(defun macroexpand-and-inspect-expression (expr)
"Is passed an expression which it macroexpands and then inspects."
  (declare (special frame))
  (unwind-protect
      (let ((value (sys:macroexpand-1 expr)))
	   (send frame :look-at-this-object value)
      )
    nil
  )
)

(defun data-from-inspection-data (thing)
  (or (send thing :Send-If-Handles :Middle-Button-Result)
      (send thing :Send-If-Handles :Aux-Data)
      (send thing :Data)
  )
)

(defun maybe-data-from-inspection-data (thing)
  (if (typep thing 'inspection-data)
      (data-from-inspection-data thing)
      thing
  )
)

     
(defun do-something-and-inspect (string action)
"Takes a prompt string and a function argument.  It prompts the user with the
 string, rewads in a value, perhaps with the mouse, and calls the function
 with the value returned.
"
   (declare (special user history = inspectors frame))
;   (send user :clear-screen)
   (format user string)
   (multiple-value-bind (value punt-p)
       (inspect-get-value-from-user user history inspectors)
     (or punt-p (funcall action value))
   )
   (send frame :handle-prompt)
)



(Defcommand Inspect-Macroexpand-Cmd nil			
  '(:description "MacroExpand and inspect something."
    :names ("MacExp")
    :keys (#\c-sh-M)
   )
   (do-something-and-inspect "~&Expression to macroexpand:"
			     'macroexpand-and-inspect-expression
   )
)


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------


;-------------------------------------------------------------------------------

;;; Code for the Trace command in ZMacs.

;-------------------------------------------------------------------------------

zwei:
(defun zwei:just-trace-name (name)
"Given a name traces it without a menu."
  (si:eval-abort-trivial-errors
    (if (atom name) `(trace (,name)) `(trace (:function ,name)))
  )
)


zwei:
(defcom zwei:com-just-trace "Trace a function without any menu.
Reads the name of the function from the mini-buffer (the top of the kill
ring has the 'current' function from the buffer) and then traces it." ()
  (let ((fcn (zwei:read-function-name "Trace"
				 (relevant-function-name (point) nil t t t)
	     )
	)
	(*print-case* :Capitalize)
       )
       (just-trace-name fcn)
       (format *query-io* "Traced ~A." fcn)
  )
  dis-none
)

zwei:
(set-comtab *standard-comtab* nil
	    (make-command-alist '(COM-Just-Trace))
)

;-------------------------------------------------------------------------------

zwei:
(defun zwei:untrace-name (name)
"Given a name untraces it without a menu."
  (si:eval-abort-trivial-errors
    (if (atom name) `(untrace ,name) `(untrace ,name))
  )
)


zwei:
(defcom zwei:com-untrace "Untrace a function without any menu.
Reads the name of the function from the mini-buffer (the top of the kill
ring has the 'current' function from the buffer) and then untraces it." ()
  (let ((fcn (read-function-name "Untrace"
				 (relevant-function-name (point) nil t t t) t
	     )
	)
	(*print-case* :Capitalize)
       )
       (untrace-name fcn)
       (format *query-io* "Untraced ~A." fcn)
  )
  dis-none
)

zwei:
(set-comtab *standard-comtab* nil
	    (make-command-alist '(Com-Untrace))
)

;-------------------------------------------------------------------------------

;;; Code required to implement a Trace command in the Inspector.

;-------------------------------------------------------------------------------



(defun function-if-defined (spec)
"Nil if it is passed an invalid function spec or a spec for an undefined
 function, otherwise the function object for the spec."
  (if (function-spec-p spec)
      (let ((function (if (consp spec)
			  (funcall (get (first spec) 'function-spec-handler)
			           'si:fdefinedp spec
			  )
			  (if (fboundp spec) (symbol-function spec) nil)
		      )
	    )
	   )
	   (cond ((equal t function) (fdefinition spec))
		 ((and (symbolp function) (fboundp function))
		  (symbol-function function)
		 )
		 ((or (functionp function) (consp function)) function)
		 (t nil)
	   )
      )
      (if (functionp spec)
	  spec
	  (if (and (fboundp 'tv:any-sort-of-clos-method-p)
		   (tv:any-sort-of-clos-method-p spec)
	      )
	      (tv:method-function-safe spec)
	      nil
	  )
      )
  )
)

(defun get-real-inspect-thing (thing)
  (if (fboundp 'map-into-show-x)
      (let ((map (funcall 'map-into-show-x thing)))
	   (if map
	       map
	       (if (typep thing 'inspection-data)
		    (if (not (symbolp (send thing :send-if-handles :aux-data)))
			(send thing :aux-data)
			(send thing :data)
		    )
		    thing
		)
	   )
      )
      thing
  )
)


eh:
(DEFUN tv:Get-method-spec-if-instance (FUNCTION prompter &Aux message)
"If it is passed an instance then it prompts for a message name to be clicked
 on and then returns a method spec for that method, otherwise it returns the
 function that it was passed.
"
  (AND (CLOSUREP FUNCTION)
       (SETQ FUNCTION (CAR (%MAKE-POINTER DTP-LIST FUNCTION))))
  (COND ((and (MEMBER (DATA-TYPE FUNCTION)
		      '(DTP-ENTITY DTP-INSTANCE DTP-SELECT-METHOD))
	      (typep function 'tv:inspection-data)
	 )
	 (tv:get-real-inspect-thing function)
	)
	((and (fboundp 'tv:any-sort-of-clos-method-p)
	      (tv:any-sort-of-clos-method-p function)
	 )
	 (tv:method-function-safe function)
	)
	((MEMBER (DATA-TYPE FUNCTION)
		 '(DTP-ENTITY DTP-INSTANCE DTP-SELECT-METHOD))
	 (SETQ MESSAGE
	       (funcall prompter
			"~&Type or mouse a message name for ~S:~%" FUNCTION
	       )
	 )
	 (LET ((HANDLER (GET-HANDLER-FOR FUNCTION MESSAGE)))
	      (OR HANDLER
		  (FORMAT T "~&~S does not handle the ~S message.~%"
			  FUNCTION MESSAGE
		  )
	      )
	      (SETQ FUNCTION HANDLER)
	 )
	)
	(t function)
  )
)



(defun trace-something (thing prompter action)
"Prompts the user for something to trace and traces it with a menu, just traces
 it or untraces it according to Action, which can be :Menu, :Just-Trace,
 :Untrace.
"
  (let ((thing-to-trace
	    (Get-method-spec-if-instance thing prompter)
        )
       )
       (let ((function (if (functionp thing-to-trace)
			   thing-to-trace
			   (function-if-defined thing-to-trace)
		       )
	     )
	    )
	    (if function
		(let ((name (function-name function)))
		     (case action
		       (:Menu
			(eval (trace-via-menus (function-name function)))
		       )
		       (:Just-Trace (zwei:just-trace-name name))
		       (:Untrace (zwei:untrace-name name))
		     )
		)
		(progn (beep)
		       (format t "~S cannot be traced." thing)
		)
	    )
       )
  )
)



(defun read-value-from-user (&Rest format-args)
"Is passed a set of format args.  It prompts the user in the Inspector user
interaction pane for something, using the format args as the prompt.
"
  (declare (special user history inspectors frame))
  (send user :clear-screen)
  (apply #'format user format-args)
  (inspect-get-value-from-user user history inspectors)
)



(defun trace-command-body (action string)
"The body of all trace related commands for the inspector."
   (declare (special user history = inspectors frame))
   (send user :clear-SCREEN)
   (format user "~&Function to ~A:" string)
   (multiple-value-bind (value punt-p)
       (inspect-get-value-from-user user history inspectors)
     (or punt-p (trace-something value 'read-value-from-user action))
   )
   (send frame :handle-prompt)
)


(defcommand Trace-Cmd nil			
  '(:description "Trace a function using trace menu."
    :Names ("Trace")
    :Keys (#\c-sh-T)
   )
   (trace-command-body :Menu "trace with menus")
)



(defcommand Just-Trace-Cmd nil			
  '(:description "Trace a function without using trace menu."
    :Names ("Just Trace")
    :Keys (#\m-sh-T)
   )
   (trace-command-body :Just-Trace "trace")
)



(defcommand UnTrace-Cmd nil			
  '(:description "Untrace a function without using trace menu."
    :Names ("Untrace")
    :Keys (#\s-sh-T)
   )
   (trace-command-body :Untrace "untrace")
)

;-------------------------------------------------------------------------------

;;; Trace command in the window debugger.

;-------------------------------------------------------------------------------


(defun eh:window-read-thing-dont-eval (prompt &rest format-args)
  (letf ((#'eh:sg-eval-in-frame
	  #'(lambda (ignore thing &rest ignore) (list thing))
	 )
	)
	(apply 'eh:window-read-thing prompt format-args)
  )
)

(defun eh:get-something-and-trace-it (action string)
"Gets something to trace from the user."
  (tv:trace-something
      (eh:window-read-function string)
      #'eh:Window-Read-Thing
      action
  )
)


eh:
(defcommand eh:Comw-Trace ()
            '(:description "Trace a function using trace menu."
              :names "Trace"
	      :keys (#\c-sh-t)
	     ) 
   (send *window-debugger* :Set-Who-Line-Doc-String-Overide
	 "Select something to trace using the trace menu."
   )
   (unwind-protect
       (get-something-and-trace-it :Menu "to trace using the trace menu")
      (send *window-debugger* :set-who-line-doc-string-overide nil)
      (send *window-debugger* :handle-prompt)
   )
)


eh:
(defcommand eh:Comw-Just-Trace ()
            '(:description "Trace a function without using the trace menu."
              :names "Just Trace"
	      :keys (#\m-sh-t)
	     ) 
   (send *window-debugger* :Set-Who-Line-Doc-String-Overide
	 "Select something to trace without using the trace menu."
   )
   (unwind-protect (get-something-and-trace-it :Just-Trace "to trace")
      (send *window-debugger* :set-who-line-doc-string-overide nil)
      (send *window-debugger* :handle-prompt)
   )
)


eh:
(defcommand eh:Comw-UnTrace ()
            '(:description "Untrace a function without using the trace menu."
              :names "Untrace"
	      :keys (#\s-sh-t)
	     ) 
   (send *window-debugger* :Set-Who-Line-Doc-String-Overide
	 "Select something to untrace without using the trace menu."
   )
   (unwind-protect (get-something-and-trace-it :UnTrace "to untrace")
      (send *window-debugger* :set-who-line-doc-string-overide nil)
      (send *window-debugger* :handle-prompt)
   )
)


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

;;; The Eval command.

;-------------------------------------------------------------------------------

;;; Code for the implementation of an Eval command in the Inspector.

;-------------------------------------------------------------------------------


(defmethod (basic-inspect-frame :look-at-this-object) (object)
"Is passed something to inspect.  It puts it into the history window and
inspects it.
"
;  (let ((hw (send self :get-pane 'history)))
;       (send hw :append-item object)
;       (update-panes)
;  )
  (send self :Inspect-Object object)
)



(defun eval-and-inspect-expression (expr)
"Is passed an expression which it Evals and then inspects."
  (declare (special frame))
  (unwind-protect
      (let ((value (eval expr)))
	   (send frame :look-at-this-object value)
      )
    nil
  )
)



(Defcommand Inspect-Eval-Cmd nil			
  '(:description "Eval and inspect something."
    :NAMES ("Eval")
    :KEYS (#\c-sh-E)
   )
   (do-something-and-inspect "~&Expression to eval:"
			     'eval-and-inspect-expression
   )
)


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------


;;; The compile command.


;-------------------------------------------------------------------------------

;;; Code for the implementation of a Compile command in the Inspector.

;-------------------------------------------------------------------------------


(defun inspect-compile (something)
  (if (functionp something)
      (compiler:compile (function-name something))
      (compiler:compile-form something)
  )
)


(Defcommand Inspect-Compile-Cmd nil			
  '(:description "Compile something."
    :names ("Compile")
    :keys (#\c-sh-C)
   )
   (do-something-and-inspect "~&Expression to compile:" 'inspect-compile)
)


;-------------------------------------------------------------------------------

;;; The compile command in the window debugger.

eh:
(defcommand eh:Comw-Compile-Cmd nil
  '(:description "Compile something."
    :names "Compile"
    :keys (#\c-sh-C)
   )
   (send *window-debugger* :set-who-line-doc-string-overide
	 "Select a something to compile."
   )
   (unwind-protect
       (tv:inspect-compile (eh:Window-Read-Thing
			     "~&Type or mouse something to compile.~%"
			   )
       )
     (progn (send *window-debugger* :set-who-line-doc-string-overide nil)
	    (send *window-debugger* :handle-prompt)
     )
   )
)


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------


;;; The arglist command.


;------------------------------------------------------------------------------

;;; Code required to implement an Arglist command in the inspector.

;-------------------------------------------------------------------------------



(defun inspect-arglist (something)
"Is passed something that the user wants the arglist for.  If it is a function
name or function object then it prints the arglist.
"
  (let ((function (function-if-defined something)))
       (if (equal nil function)
	   (progn (beep) (format t "~S is not a function." something))
	   (zwei::print-arglist (function-name function))
       )
  )
)



(Defcommand Arglist-Cmd nil			
  '(:description "Display arglist of a function."
    :names ("Arglist")
    :keys (#\c-sh-A)
   )
   (do-something-and-inspect "~&Function whose arglist is to be printed:"
			     'inspect-arglist
   )
)

;-------------------------------------------------------------------------------

;;; The Arglist command in the window debugger.

;-------------------------------------------------------------------------------


;;; Redefine the arglist command so that it will take c-sh-a as a key
;;; assignment for compatibility with ZMacs.

eh:
(defcommand eh:Comw-Arglist-Cmd nil
  '(:description "Display the argument list of a specified function."
    :names "Arglist"
    :keys (#\c-A #\c-sh-A)
   )
   (send *window-debugger* :set-who-line-doc-string-overide
	 "Select a function to apply Arglist to."
   )
   (unwind-protect (comw-arglist *error-sg* *error-object*)
     (progn (send *window-debugger* :set-who-line-doc-string-overide nil)
	    (send *window-debugger* :handle-prompt)
     )
   )
)


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

;;; The document command (with bits of arglist too).

;-------------------------------------------------------------------------------


(defparameter *documentors*
  (list
    (list
      #'(lambda (name) (or (functionp name t) (fdefinedp name)))
      #'(lambda (name &aux doc)
	  (multiple-value-bind (args return macro-p) (arglist name)
	    (if (get name 'si::defstruct-slot)
		(format t "~%~S is a defstruct accessor macro for ~s" name
			(car (get name 'si::defstruct-slot)))
		(format t "~%~s is a ~:[function~;macro~]" name macro-p))
	    (format t "~%arguments  ~A~%" args)
	    (when return
	      (format t "returns  ~{~A  ~}~%" return)))
	  (when (setq doc (documentation name))
	    (format t "~A~%" doc))))
    (list
      #'(lambda (name) (get name 'si:flavor))
      #'(lambda (name &aux doc)
	  (let ((temp (get name 'si:flavor)))
	    (format t "~%~s is a flavor~%" name)
	    (when (and (setq temp (si:flavor-plist temp))
		       (setq doc
			     (second (member :documentation temp :test #'eq))))
	      (format t "~a~%" doc)))))
    (list
      #'(lambda (name) (get name 'defresource))
      #'(lambda (name) (format t "~%~s is the name of a resource." name))
    )
    (list
      #'(lambda (name) (get name 'si::defstruct-description))
      #'(lambda (name &aux doc)
	  (let ((temp (get name 'si::defstruct-description)))
	    (format t "~%~s is a structure~%" name)
	    (when
	      (and (setq temp (si::defstruct-description-property-alist temp))
		   (setq doc (rest (assoc :documentation temp :test #'eq))))
	      (format t "~a~%" doc)))))
    (list
      #'(lambda (name) (and (symbolp name) (boundp name)))
      #'(lambda (name &aux temp)
	  (if (get name 'compiler:system-constant) ;; may 7-11-88
	      (format t "~%~s is a constant~%" name);; may 7-11-88
	      (format t "~%~s is a variable~%" name))
	  (if (consp (setq temp (symbol-value name)))
	      (format t "Value is a ~A~%" (type-of temp))
	      (format t "Value is ~s~%" temp))
	  (format t "~@[~a~%~]"
		  (getf (get name 'sys:documentation-property) 'variable))))
    (list
      #'(lambda (name) (get name 'eh::make-condition-function))
      #'(lambda (name &aux doc)
	  (let ((temp (get name 'eh::make-condition-function)))
	    (format t "~%~S is an ~:[undocumented ~]error condition~%" name
		    (setq doc (documentation temp)))
	    (when doc
	      (format t "~a~%" doc)))))
    (list
      #'(lambda (name)
	  (and (find-package 'ticlos)
	       (funcall (read-from-string "ticlos:class-named") name t)
	       (documentation
		 (funcall (read-from-string "ticlos:class-named") name)
	       )
	  )
	)
      #'(lambda (name)
	  (format t "~&~S is the name of a CLOS class.~&~A"
		  name
		  (documentation
		    (funcall (read-from-string "ticlos:class-named") name)
		  ))))
   )
)

(defun long-document (name stream)
  (let ((*package* nil)
	(docs nil))
     (loop for (pred doc-fn) in *documentors*
	   when (funcall pred name)
	   do (let ((str (with-output-to-string (*standard-output*)
			   (funcall doc-fn name)
			 )
		    )
		   )
		   (push str docs)
		   (princ str stream)
	      )
     )
     (let ((temp (get name :documentation)))
          (if (and temp (not (member temp docs :test #'equal)))
	      (progn (format stream "~%~s is a ~a~%" name (type-of name))
		     (format stream "~a~%" temp)))
	  (if (and (not temp) (not docs))
	      (format *query-io* "~&~s was not found." name)))))

(Defun Long-Documentation (name)
 (with-output-to-string (output-string)
  (long-document name output-string)
 )
)

zwei:
(DEFCOM zwei:COM-LONG-DOCUMENTATION
   "Print long documentation for the specified function.
Reads the name of the function from the mini-buffer (the default is
the \"current\" function from the buffer) and displays the
function's arguments and documentation" () ;;by OREN/STENGER. Patched in by gsl
   (LET ((NAME (READ-FUNCTION-NAME
		 "Document"
		 (RELEVANT-FUNCTION-NAME (POINT)) 'AARRAY-OK)))
        ;;; JPR.
        (princ (tv:long-documentation name))
	dis-none))
	     

(defun document-an-object (thing window)
"Given something it displays the documentation for it."
  (let ((thing-to-document
	   (cond ((or (symbolp thing)
		      (and (consp thing) (function-spec-p thing))
		  )
		  thing
		 )
		 ((functionp thing) (function-name thing))
		 ((and (not (documentation thing))
		       (or (named-structure-p thing) (typep thing 'Instance))
		  )
		  (type-of thing)
		 )
		 (t nil)
	   )
	 )
	)
	(if thing-to-document
	    (let ((docs (Long-Documentation thing-to-document)))
		 (if (equal docs "")
		     (format t "has no documentation.")
		     (si:with-help-stream
		       (window :label "Documentation" :superior window)
		       (format window docs)
		     )
		 )
	    )
	    (progn (beep)
		   (format t "is not documentable.")
	    )
	)
  )
)



(Defcommand Document-Something-Cmd nil			
  '(:description "Document something."
    :names ("Docmnt")
    :keys (#\m-sh-d #\c-sh-d)
   )
   (declare (special frame))
   (do-something-and-inspect "~&Object to show documentation for:"
     #'(lambda (value) (document-an-object value frame))
   )
)

;-------------------------------------------------------------------------------

(defvar *all-method-names* (make-hash-table :test #'eq))

(defun traverse-methods (&optional (method nil))
  (loop for flavor in *all-flavor-names*
	for fl = (get flavor 'si:flavor)
	when fl
	do (loop for method-entry in (si:flavor-method-table fl)
		 for operation = (first method-entry)
		 when (or (not method) (eq method operation))
		 do (let ((method-list (gethash operation *all-method-names*)))
			 (loop for meth in (cdddr method-entry)
			       do (or (eq (si:meth-method-type meth) :combined)
				      (and (si:meth-definedp meth)
					   (pushnew (list flavor
							  (si:meth-method-type
							    meth
							  )
							  (si:meth-function-spec
							    meth
							  )
						    )
						    method-list
						    :test #'equal
					   )
				      )
				  )
			 )
			 (setf (gethash operation *all-method-names*)
			       method-list
			 )
		    )
	   )
  )
  (if method
      (or (gethash method *all-method-names*)
	  (progn (setf (gethash method *all-method-names*) nil) nil)
      )
      nil
  )
)

(defun zwei:list-methods-internal (op &optional (filter-combined-methods-p t))
  (let ((methods
	  (multiple-value-bind (methods found-p) (gethash op *all-method-names*)
	    (if found-p
		methods
		(Traverse-Methods op)
	    )
	  )
	)
       )
       (values (if filter-combined-methods-p
		   (remove-if #'(lambda (method)
				  (equal '(:Combined) (second method))
				)
			      methods
		   )
		   methods
	       )
	       op
       )
  )
)

(advise sys:method-function-spec-handler :Around :Flush-Cache nil
  (destructuring-bind (function function-spec) arglist
    (if (member function '(sys:fundefine sys:fdefine) :test #'eq)
	(let ((cache-spec (list (second function-spec)
				(butlast (rest (rest function-spec)))
				function-spec
			  )
	      )
	      (operation (first (last function-spec)))
	     )
	     (let ((all-existing (gethash operation *all-method-names*)))
		  (let ((match (find cache-spec all-existing :Test #'equal)))
		       (setf (gethash operation *all-method-names*)
			     (case function
			       (sys:fundefine (remove match all-existing))
			       (sys:fdefine
				(if match
				    all-existing
				    (cons cache-spec all-existing)
				)
			       )
			     )
		       )
		  )
	     )
	)
	nil
    )
  )
  :do-it
)

(defun sort-method-list (method-list)
  (sort method-list #'string-lessp
	:key #'(lambda (x) (format nil "~S" (third x)))
  )
)

(defun self-evaluating-form-p (x)
  (typecase x
    ((or keyword fixnum string) t)
    (cons (equal 'quote (first x)))
    (symbol (case x
	      ((t nil) t)
	      (otherwise (and (boundp x) (eq x (symbol-value x))))
	    )
    )
  )
)

(If (= 0 (hash-table-count *all-method-names*))
    (traverse-methods)
    nil
)

;-------------------------------------------------------------------------------

(DEFMETHOD (tv:lisp-help-mixin :rh-send-help-hack) (help-function &AUX error)
  "Special case help for the SEND function"
  (MULTIPLE-VALUE-BIND (flavor method)
      (rh-send-hack)
    (WHEN (AND flavor method)
      ;; Get a value for the flavor instance, check for errors first
      (IF (SETQ error (ucl:check-top-level flavor))
	  (let ((method (catch-error (read-from-string method) nil)))
	    (if method
		(case (function-name help-function)
		  (ucl:get-method-arglist
		   (with-output-to-string (str)
		     (zwei:find-and-print-arglist-for 'send str method)
		   )
		  )
		  (ucl:Get-Method-Documentation
		   (with-output-to-string (str)
		     (zwei:quick-documentation-1 'send str method)
		   )
		  )
		  (otherwise (VALUES NIL error))
		)
		(VALUES NIL error)
	    )
	  )
	  (LET ((flavor-instance (EVAL flavor)))
	    ;; Ensure that we really have a flavor instance
	    (IF (NOT (TYPEP flavor-instance 'INSTANCE))
		(VALUES NIL (string-append
			      "The first argument to SEND must be a flavor instance."
			      #\return (if (or (symbolp flavor) (stringp flavor))
					   flavor
					   (princ-to-string flavor))
			      " is a " (DATA-TYPE flavor-instance)))
		;; Get a value for the method name, check for errors first
		(PROGN (SETQ method (READ-FROM-STRING method NIL nil))
                       (IF (SETQ error (ucl:check-top-level method))
                           (VALUES NIL error)
                           (FUNCALL help-function
				    flavor-instance (EVAL method))))))))))


#!C
; From file COMF.LISP#> ZMACS; Hotel:
#8R ZWEI#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "ZWEI"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* SYS:COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: ZMACS; COMF.#"


(defun find-and-print-arglist-for (symbol stream &optional (method nil))
  (COND ((COND ((MEMBER SYMBOL '(FUNCALL FUNCALL-SELF SEND
				  LEXPR-FUNCALL LEXPR-FUNCALL-SELF LEXPR-SEND <-)
		 :TEST #'EQ)
	 (LET ((TEMP-SYMBOL (or method
				(RELEVANT-METHOD-NAME
				  (POINT)
				  (IF (MEMBER SYMBOL '(FUNCALL-SELF
							LEXPR-FUNCALL-SELF)
					      :TEST #'EQ)
				      1 2)))))
	   (AND TEMP-SYMBOL
		(tv:self-evaluating-form-p temp-symbol)
		(SETQ SYMBOL TEMP-SYMBOL))))
	((and (boundp '*winndow*) (EQ SYMBOL 'DEFMETHOD))
;; may 06/05/90 Start patch ...
	 ;; IF SUBSTITUTE-SECTION-NAME is T, then RELEVANT-FUNCTION-NAME
	 ;; performs the equivalent of RELEVANT-DEFMETHOD-METHOD-NAME for FLAVOR
	 ;; methods and for CLOS methods the portion of (:PROPERTY :LISP GET-SECTION-NAME)
	 ;; which calls READ-FROM-STRING.
	 (setq symbol (relevant-function-name (point) nil t t t))
	 nil)
;;; Don't call RELEVANT-DEFMETHOD-METHOD-NAME - it does NOT understand CLOS methods
;;;			    (LET ((METHOD-SYMBOL (RELEVANT-DEFMETHOD-METHOD-NAME (POINT))))
;;;			      (COND ((and METHOD-SYMBOL (tv:self-evaluating-form-p method-symbol))
;;;				     (SETQ SYMBOL METHOD-SYMBOL)
;;;				     T))))))
	     )
;; may 06/05/90 End patch ...
  (let ((specs (method-arglist-for-quick-arglist symbol)))
    (if specs
	(let ((stream (if (> (length specs) 1)
			  *standard-output*
			  stream)))
	  (loop for (flavor method-name args ret-vals) in specs do
		(COND ((EQ STREAM *QUERY-IO*)
		       (FORMAT *QUERY-IO* "~&")
		       (SEND STREAM :SEND-IF-HANDLES :TYPEOUT-STAYS)))
		(FORMAT STREAM "~&~S: ~:A~@[  ~:A~]"
			(OR method-NAME SYMBOL) (rest ARGs) RET-vals)))
	(format stream "~&No methods found called ~S" symbol))))
 ((and SYMBOL
       (OR (FDEFINEDP SYMBOL)
	   (SI:MEMQ-ALTERNATED 'ARGLIST (SYMBOL-PLIST SYMBOL))))
  (PRINT-ARGLIST SYMBOL STREAM))
 ;; Looked hard but couldn't find a defined function
 ((BARF "~@[~a~]~A" symbol (if symbol " is not defined." "No relevant function.")))))

(DEFUN QUICK-ARGLIST (&OPTIONAL (STREAM *QUERY-IO*))
  (IF *NUMERIC-ARG-P*
      (LET ((NAME (READ-FUNCTION-NAME
		    "Arglist" (RELEVANT-FUNCTION-NAME (POINT)) T)))
	(PRINT-ARGLIST NAME STREAM))
      (LET ((SYMBOL (RELEVANT-FUNCTION-NAME (POINT))))
	(find-and-print-arglist-for symbol stream))))


(defun zwei:method-arglist-for-quick-arglist (message-name)
  (let ((methods (zwei:list-methods-internal message-name)))
       (let ((filtered methods
;		       (remove-if-not
;			 #'(lambda (x) (subtypep (first x) zwei:*base-flavor*))
;			 methods
;		       )
	     )
	    )
	    (let ((all-specs
		    (loop for (flavor quals? method-name) in filtered
			  for (args ret-vals)
			      = (multiple-value-list (arglist method-name))
			  collect (list flavor method-name args ret-vals)
		    )
		  )
		 )
		 (let ((result (zwei:filter-method-specs all-specs)))
		      result
		 )
	    )
       )
  )
)

(defun zwei:is-subsumed-by (spec specs)
  (destructuring-bind (flavor method-name args ret-vals) spec
    (ignore method-name)
    (let ((match (loop for spec? in specs
		       when (subtypep flavor (first spec?))
		       return spec?
		 )
	  )
	 )
	 (and match
	      (equalp args (third match))
	      (equalp ret-vals (fourth match))
	 )
    )
  )
)

(defun zwei:filter-method-specs-1 (spec specs)
  (if (zwei:is-subsumed-by spec specs)
      specs
      (cons spec (remove-if #'(lambda (x) (zwei:is-subsumed-by x (list spec)))
			    specs
		 )
      )
  )
)

(defun zwei:filter-method-specs (specs)
  (if specs
      (zwei:filter-method-specs-1
	(first specs) (zwei:filter-method-specs (rest specs))
      )
      nil
  )
)

(DEFCOM zwei:COM-QUICK-DOCUMENTATION
   "Prints documentation for the function point is at.
Prints the documentation string of the function which point is inside a call to.
With a numeric argument, reads the name of the function to document
from the mini buffer." ()
   (LET ((NAME (RELEVANT-FUNCTION-NAME (POINT))))
     (IF *NUMERIC-ARG-P*
	 (SETQ NAME (READ-FUNCTION-NAME "Brief Document" NAME T)))
     (zwei:quick-documentation-1 name *QUERY-IO*))
   DIS-NONE)

(defun zwei:quick-documentation-1 (name query-stream &optional (method nil))
  (LET ((DOC (DOCUMENTATION NAME 'FUNCTION)))
    (COND ((NULL DOC)
	   (IF (FDEFINEDP NAME)
	       (PRINT-ARGLIST NAME))
	   (FORMAT query-stream "~&~S is not documented." NAME))
	  ((COND ((MEMBER NAME '(FUNCALL FUNCALL-SELF SEND
				    LEXPR-FUNCALL LEXPR-FUNCALL-SELF LEXPR-SEND <-)
			   :TEST #'EQ)
		   (LET ((TEMP-SYMBOL (or method
					  (RELEVANT-METHOD-NAME
					    (POINT)
					    (IF (MEMBER NAME '(FUNCALL-SELF
								LEXPR-FUNCALL-SELF)
							:TEST #'EQ)
						1 2)))))
		     (AND TEMP-SYMBOL (SETQ NAME TEMP-SYMBOL))))
		  ((and (boundp '*window*) (EQ NAME 'DEFMETHOD))
;; may 06/05/90 Start patch ...
		   ;; IF SUBSTITUTE-SECTION-NAME is T, then RELEVANT-FUNCTION-NAME
		   ;; performs the equivalent of RELEVANT-DEFMETHOD-METHOD-NAME for FLAVOR
		   ;; methods and for CLOS methods the portion of (:PROPERTY :LISP GET-SECTION-NAME)
		   ;; which calls READ-FROM-STRING.
		   (setq name (relevant-function-name (point) nil t t t))
		   nil)
		  )
	   (let ((specs (method-arglist-for-quick-arglist name)))
		(loop for (flavor method-name args ret-vals) in specs do
		      (zwei:quick-documentation-1 method-name t)))
	  )
	  (t
	   (IF (or (FDEFINEDP NAME)  
		   (and (symbolp name) 							;; may 04/20/89
			;; for special compiler functions like: (ticlos:call-next-method) 	;; may 04/20/89 
			(listp (get name 'arglist :default))))				;; may 04/20/89 
	       (PROGN
		 (FRESH-LINE)
		 (PRINT-ARGLIST NAME *STANDARD-OUTPUT*))
	       (FORMAT T "~S:" NAME))
	   (FORMAT T "~%~A" DOC)))))

)) ;;; End of patch

(defvar *method-document*
        (list
	  #'(lambda (name) (gethash name *all-method-names*))
	  #'(lambda (name)
	      (let ((specs (zwei:method-arglist-for-quick-arglist name)))
		   (loop for (flavor method-name args ret-vals) in specs do
			 (zwei:quick-documentation-1 method-name t)
		   )
	      )
	    )
	)
)

(pushnew *Method-Document* *documentors*)


(reinstall-inspector-commands)

;-------------------------------------------------------------------------------

;;; The Document command in the window debugger.

;-------------------------------------------------------------------------------


eh:
(defun eh:find-and-document-something ()
"Prompts the user for something to document and displays its doc string if it
 can.
"
  (tv:document-an-object
      (window-read-thing "~&Type or mouse something to document:~%")
      *window-debugger*
  )
)

eh:
(Defcommand eh:Comw-Debug-Document-Something ()
  '(:description  "Show documentation for something."
    :names "Docmnt"
    :keys (#\c-sh-d #\m-sh-d)
   )
   (send *window-debugger* :Set-Who-Line-Doc-String-Overide
	 "Select something to document."
   )
   (unwind-protect (find-and-document-something)
      (send *window-debugger* :set-who-line-doc-string-overide nil)
      (send *window-debugger* :handle-prompt)
   )
)

;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------


;;; The Help command.

;-------------------------------------------------------------------------------

(defun uniquise-commands (commands)
  (if (and commands (rest commands))
      (if (and (equalp (first  (first commands)) (first  (second commands)))
	       (equalp (second (first commands)) (second (second commands)))
	  )
	  (uniquise-commands (rest commands))
	  (cons (first commands) (uniquise-commands (rest commands)))
      )
      commands
  )
)

(defun show-all-commands-for-frame (frame on-window)
  (format on-window "~&The following commands are supported on top of the normal input editor commands.~%")
  (let ((all-commands
	  (mapcar #'(lambda (table)
		      (mapcar #'(lambda (command)
				  (append
				    (firstn 2 (send command :parsed-edit-form))
				    (list (send command :description)
					  (if (string-equal
						(send command :description)
						(send command :documentation)
					      )
					      ""
					      (send command :documentation)
					  )
				    )
				  )
				)
			        (listarray
				  (send (symbol-value table) :commands)
				)
		      )
		    )
		    (uniqueise (send frame :all-command-tables))
	  )
	)
       )
       (let ((sorted (sortcar (apply #'append all-commands) #'string-lessp)))
	    (mapcar #'(lambda (list)
			(apply #'format on-window "~&~A~25T~A~48T~A ~A" list))
		    (uniquise-commands sorted)
	    )
       )
  )
)

(Defcommand Documentation-Cmd nil		
  '(:description 
    "Display some brief documentation about each of the Inspector's panes."
    :names ("Help")
    :keys (#\c-HELP #\m-HELP)
   )
   (declare (special frame))
   (si:with-help-stream (window :label "Documentation for Inspector"
				:superior frame
			)
     (format window
"
  -----------------------------------------------------------------------------------
                    *** Optional Third Inspection Pane ***

    Displays previously inspected item.

 ------------------------------------------------------------------------------------
                    *** Optional Second Inspection Pane ***

    Displays previously inspected item.

 ------------------------------------------------------------------------------------ 
                        *** Main Inspection Pane ***

    This pane displays the structure of the most recently inspected item.
    Specify objects to inspect by

      1) Entering them into the Interaction Pane or,
      2) Clicking left on the mouse sensitive elements of previously inspected items.

    Right click on items here tries to inspect the item's function definition.


 ------------------------------------------------------------------------------------
   * Command * |                  *** History Pane ***
   *  Menu   * |
               |    This pane displays a list of the objects that have been
     For UCL   |  inspected.  To bring an object back into the Main Inspection
     command   |  pane click left on it in this pane.
     display   |
      press    |    To remove an item from the history, click middle in the item's
      HYPER-   |  line area (the area just left of the item where the mouse cursor
     CONTROL-  |  becomes a right pointing arrow).
      HELP.    |
 ------------------------------------------------------------------------------------
                           *** Interaction Pane *** 

      Enter items to inspect in this pane.  This pane may also be used for command
    name typein and for Lisp typein.  For Lisp typein use the Mode command.

      The last three inspected objects are stored in *, ** and ***.
 
 ------------------------------------------------------------------------------------
  ")
     (show-all-commands-for-frame frame window)))

(reinstall-inspector-commands)

;-------------------------------------------------------------------------------

;;; The window debugger Help command.  This used to be the document command.

;-------------------------------------------------------------------------------

eh:
(DEFUN WINDOW-DEBUGGER-HELP (&OPTIONAL IGNORE IGNORE)
  (declare (special *window-debugger*))
  (SI:WITH-HELP-STREAM (WINDOW :LABEL
                               "Help for Window-based debugger"
                               :SUPERIOR
                               tv:default-screen)
    (FORMAT WINDOW "
                                 WINDOW-BASED DEBUGGER HELP

----------------------------------------------------------------------------------------------
                                   *** INSPECTION PANE ***

     This pane displays the structure of the most recently inspected item.  By default
     the item inspected here is the selected stack frame.  To inspect other items here, 
     use the Inspect command or click Mouse-Left on them in this pane or 
     in the Inspection History Pane.

----------------------------------------------------------------------------------------------
             *** ARGS PANE ***                 |             *** LOCALS PANE ***
                                               |
     This pane displays the argument           |     This pane displays the local variable
     values for the currently selected         |     values and the special variable values
     frame if there are any.  Otherwise,       |     for the currently selected frame if
     this pane is gray.                        |     there are any.  Otherwise, this pane
                                               |     is gray.
                                               |
----------------------------------------------------------------------------------------------
                                      *** STACK PANE ***

     This pane displays the execution stack that contained the error.  The contents of the 
     above three panes are determined by the frame that is selected.  The selected frame is
     the frame in this pane with the small arrow pointing to it.

     To select another frame to see its args\/locals\/specials\/code use the up\/down
     commands or move the mouse cursor to where you want the new arrow to be.  Then click 
     Mouse-Left when the cursor changes back to the small arrow.

----------------------------------------------------------------------------------------------
*** MENU PANE ***    |                   *** INSPECTION HISTORY PANE ***
                     |
Click Mouse-Left     |     This pane maintains a history of objects that have appeared 
to select a          |     in the Inspection Pane.  To see these objects in the Inspection
command.             |     Pane again, click Mouse-Left on them here.
                     |
----------------------------------------------------------------------------------------------
                                    *** INTERACTION PANE ***

This pane is used to output messages and to evaluate Lisp forms.  Clicking Mouse-Left on
mouse-sensitive items within the Args, Locals, or Stack panes will print them out here and
will set * to these items.

----------------------------------------------------------------------------------------------



 ")
    (tv:show-all-commands-for-frame *window-debugger* window)))


eh:
(defcommand eh:Window-Debugger-Help-Cmd ()
            '(:description "Show documentation for each of the panes."
			    :names "Help"
			    :keys (#\c-help #\m-help)
	     )
             (window-debugger-help *error-sg* *error-object*)
)



;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

;;; The flavor inspect command.

;-------------------------------------------------------------------------------


;;; For Zmacs

zwei:
(defun zwei:safe-inspect-flavor (object)
"Inspects an object but just beeps if it is not inspectable."
  (let ((flavor (cond ((or (instancep object) (typep object 'si:flavor))
		       object
		      )
		      ((symbolp object)
		       (get object 'si:flavor)
		      )
		      (t nil)
		)
	)
       )
       (if flavor
	   (Process-Run-Function "Flavor Inspector" 'inspect-flavor flavor)
           (progn (beep)
		  (format *query-io* "~S is not a flavor." object)
	   )
       )
  )
)


zwei:
(defcom zwei:com-flavor-inspect "Call the flavor inspector on something." ()
  (let ((flavor (read-function-name "Flavor Inspect"
				    (relevant-function-name (point) nil t t t) t
	        )
	)
	(*print-case* :Capitalize)
       )
       (safe-inspect-flavor flavor)
  )
  dis-none
)

zwei:
(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-flavor-inspect))
)

;-------------------------------------------------------------------------------

;;; For the inspector.


(defcommand Flavor-Inspect-Cmd nil			
  '(:description "Flavor Inspect a Flavor or Method."
    :names ("FlavIns")
    :keys (#\c-sh-f #\m-sh-f #\h-F)
   )
   (declare (special user history = inspectors frame))
   (send user :clear-screen)
   (if (fboundp 'inspect-flavor) 
       (progn
	 (format user "~&Object to Flavor Inspect:")
	 (multiple-value-bind (value punt-p)
	     (inspect-get-value-from-user user history inspectors)
	   (or punt-p
	       (zwei:safe-inspect-flavor value)
	   )
	 )
	 (send frame :handle-prompt)
       )
       (progn
	 (if (y-or-n-p "The Flavor Inspector is not currently loaded. Do you wish to load it?")
	     (progn
	       (load "sys:debug-tools;flavor-inspector")   
	       (send user :clear-screen)
	       (format user "~&Object to Flavor Inspect:")
	       (multiple-value-bind (value punt-p)
		   (inspect-get-value-from-user user history inspectors)
		 (OR PUNT-P
		     (zwei:safe-inspect-flavor value)
		 )
	       )
	       (send frame :handle-prompt)
	     )
	 )
       )
   )
)

;-------------------------------------------------------------------------------

;;; For the window debugger.

eh:
(Defcommand eh:Flavor-Inspect-Cmd nil			
  '(:description "Flavor Inspect a Flavor or Method."
    :names ("FlavIns")
    :keys (#\c-sh-f #\m-sh-f #\h-F)
   )
   (zwei:safe-inspect-flavor (window-read-thing "~&Object to Flavor Inspect:"))
)


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

;;; The edit command.

;-------------------------------------------------------------------------------

;;; For Zmacs
;;; Just add a new key assignment.

Zwei:
(set-comtab *standard-comtab* '(#\h-E Com-Edit-Definition)
	                       (make-command-alist '(Com-Edit-Definition))
)


;-------------------------------------------------------------------------------

;;; For the inspector.
;;; Just add new key assignments.

(defun make-ununspecific (path)
"Takes a path and makes sure that it has an specific type."
  (or (and (not (equal :Unspecific (send path :type)))
	   path
      )
      (let ((new-path (make-pathname :defaults path
				     :type :lisp
				     :version :newest
		      )
	    )
	   )
	   new-path
      )
  )
)


(defun pathname-could-be-editted (path)
"Is true if the pathname represents something that could reasonably be editted.
 This means that it has a name and type.
"
  (let ((name (send path :name))
        (type (send path :type))
       )
       (and name
	    type
	    (or (and (not (equal :Unspecific type))
		     (probe-file path)
		)
		(let ((new-path (make-pathname :defaults path
					       :type :lisp
					       :version :newest
				)
		      )
		     )
		     (and (probe-file new-path)
			  (y-or-n-p "Try and edit Lisp file: \"~A\""
				    (send new-path :string-for-printing)
			  )
		     )
		)
	    )
       )
  )
)


(defun edit-a-path (path)
"Given a pathname edits the file for it.  If the version of the path is
 :UnSpecific then it edits the :Newest
"
  (ed (if (equal :UnSpecific (Send path :Version))
	  (send path :New-Version :Newest)
	  path
      )
  )
)


(defun try-and-edit-flavor-of-object (object)
"Given an instance it tries to edit the source file of the type of the
 instance.
"
  (cond ((and (fboundp 'class-p-safe) (class-p-safe object))
	 (try-and-edit (class-name-safe object))
	)
	((and (fboundp 'any-sort-of-clos-method-p)
	      (any-sort-of-clos-method-p object)
	 )
	 (try-and-edit (method-function-safe object))
        )
	(t (typecase object
	     (si:flavor  (try-and-edit (si:flavor-name  object)))
	     (otherwise
	       (if (get (type-of object) :Source-File-Name)
		   (ed (type-of object))
		   (progn (beep)
			  (format *query-io*
				  "~&No source file can be found for ~S"
				  object
			  )
		   )
	       )
	     )
	   )
       )
  )
)


(defun try-and-edit-object (object)
"Is passed an instance or a defstruct instance and tries to edit it in the most
 intelligent way possible.  If the thing responds to the :Pathname message then
 it tries to edit the file defined by the path.  If it fails to do this then
 it tries to edit the source file of the type of the thing.
"
  (if (if (typep object 'Instance)
	  (send object :Operation-Handled-p :Pathname)
	  (member 'Pathname (send object :Which-Operations) :Test #'Eq)
          ;;; Cannot use :Operation-Handled-P because most defstructs do not
          ;;; support this message.
      )
      (let ((path (send object :pathname)))
	   (if (and (typep path 'pathname)
		    (pathname-could-be-editted path)
		    (probe-file path)
	       )
	       (edit-a-path path)
	       (try-and-edit-flavor-of-object object)
	   )
      )
      (try-and-edit-flavor-of-object object)
  )
)


(defun try-and-edit-string (string)
"Given a string it tries to edit a file which the string denotes.  If it fails
 to find such a file then it beeps and tells you why it failed to edit it.
"
  (let ((path (fs:parse-pathname string nil nil 0 nil t)))
       (if path
	   (if (pathname-could-be-editted path)
	       (if (probe-file path)
		   (edit-a-path path)
		   (progn (beep)
			  (format *Query-IO* "~&File not found for ~S" string)
		   )
	       )
	       (progn (beep)
		      (format *Query-IO*
			     "~&~S cannot be coerced into an edittable pathname"
			     string
		      )
	       )
	   )
	   (progn (beep)
		  (format *Query-IO* "~&~S cannot be coerced into a pathname"
			  string
		  )
	   )
       )
  )
)


(defun remove-internals (name)
"Extracts a function name from any closure specs.
 Thus (:internal (:internal foo 0) 123) -> foo.
"
  (if (and (consp name) (equal (first name) :internal))
      (remove-internals (second name))
      name
  )
)


#-CLOS
(defun try-and-edit (object)
"Takes any object and edits its definition if it can find a
 meaningful definition to edit.  For instance if it is passed a
 named structure then it edits the defstruct definition for
 that type.  If it fails to edit the object it beeps and tells you what it was
 that it failed to edit a definition of.
"
  (cond ((and (typep object 'symbol)
	      (or (get object :Source-File-Name)
		  (get object 'zwei:zmacs-buffers)
	      )
	 )
	 (ed object)
	)
	((and (typep object 'pathname)
	      (pathname-could-be-editted object)
	 )
	 (edit-a-path (make-ununspecific object))
	)
	((and (functionp object)
	      (or (si:function-spec-get
		    (remove-internals (function-name object))
		    :Source-File-Name
		  )
		  (si:function-spec-get
		    (remove-internals (function-name object))
		    'zwei:zmacs-buffers
		  )
	      )
	 )
	 (ed (function-name object))
	)
	((and (or (named-structure-p object) (typep object 'Instance)))
	 (try-and-edit-object object)
	)
	((and (consp object) (function-spec-p object)
	      (or (si:function-spec-get object :Source-File-Name)
		  (si:function-spec-get object 'zwei:zmacs-buffers)
	      )
	 )
	 (ed object)
	)
	((and (consp object) (member (first object) '(Function Quote)))
	 (try-and-edit (second object))
	)
        ((typep object 'string) (try-and-edit-string object))
	(t (beep)
	   (format *Query-IO* "~&Cannot find a definition to edit for ~S"
		   object
	   )
	)
  )
)

#+CLOS
ticlos:
(defgeneric tv:edit (something)
  (:Documentation "Takes any object and edits its definition if it can find a
 meaningful definition to edit.  For instance, if it is passed a
 named structure then it edits the defstruct definition for
 that type.  If it fails to edit the object it beeps and tells you what it was
 that it failed to edit a definition of.
"
  )
)


#+CLOS
(defmethod edit ((me t))
  (if (named-structure-p me)
      (try-and-edit-object me)
      (progn (beep)
	     (format *Query-IO* "~&Cannot find a definition to edit for ~S"
		     me
	     )
      )
  )
)


#+CLOS
(defmethod edit ((me symbol))
  (if (or (get me :Source-File-Name)
	  (get me 'zwei:zmacs-buffers)
      )
      (ed me)
      (if (gethash me *all-method-names*)
	  (let ((method (zwei:list-methods-thru-mouse me)))
	       (if method
		   (ed (first method))
		   (beep)
	       )
	  )
	  (ticlos:call-next-method)
      )
  )
)

#+CLOS
(defmethod edit ((me pathname))
  (if (pathname-could-be-editted me)
      (edit-a-path (make-ununspecific me))
      (ticlos:call-next-method)
  )
)

#+CLOS
(defmethod edit ((me compiled-function))
  (if (or (si:function-spec-get
	     (remove-internals (function-name me))
	     :Source-File-Name
	   )
	   (si:function-spec-get
	     (remove-internals (function-name me))
	     'zwei:zmacs-buffers
	   )
       )
      (ed (function-name me))
      (ticlos:call-next-method)
  )
)

#+CLOS
(defmethod edit ((me ticlos:generic-function))
  (clos:generic-function-name me)
)

(defun menu-of-methods (object)
  (w:menu-choose (mapcar #'(lambda (method)
			     (let ((method-spec 
				     (function-name
				       (tv:method-function-safe
					 method
				       )
				     )
				   )
				  )
				  (list (format nil "~s" method-spec)
					:Value
					method-spec
				  )
			     )
			   )
			   (tv:generic-function-methods-safe
			     (tv:function-generic-function-safe
			       (symbol-function object)
			     )
			   )
		 )
		 :Label
		 (format nil
		      "~s is a generic function.  Edit which method?"
		      object
		 )
  )
)


#+CLOS
(defmethod Edit ((me lexical-closure))
  (let ((fef (first (si:convert-closure-to-list me))))
       (if (and (fboundp 'pcl-p) (pcl-p) (fef-of-gf-p me)) ;;; Allow for PCL.
	   (let ((choice (Menu-Of-Methods (generic-function-name-safe me))))
		(if choice
		    (Edit choice)
		    (beep)
		)
	   )
	   (Edit fef)
       )
  )
)


#+CLOS
(defmethod edit ((me cons))
  (if (and (function-spec-p me)
	   (or (si:function-spec-get me :Source-File-Name)
	       (si:function-spec-get me 'zwei:zmacs-buffers)
	   )
      )
      (ed me)
      (if (member (first me) '(Function Quote))
	  (edit (second me))
	  (ticlos:call-next-method)
      )
  )
)

#+CLOS
(defmethod edit ((me string))
  (try-and-edit-string me)
)

#+CLOS
(defmethod edit ((me array))
  (if (named-structure-p me)
      (try-and-edit-object me)
      (ticlos:call-next-method)
  )
)

#+CLOS
(defmethod edit ((me si:vanilla-flavor))
  (try-and-edit-object me)
)

#+CLOS
(defmethod edit ((me clos:standard-class))
  (try-and-edit-object me)
)

#+CLOS
(defmethod edit ((me ticlos:flavor-class))
  (try-and-edit-object me)
)

#+CLOS
(defmethod edit ((me ticlos:method))
  (Edit (ticlos:method-function me))
)

#+CLOS
(defmethod edit ((me inspection-data))
  (try-and-edit-object (send me :middle-button-result))
)

#+CLOS
(defun try-and-edit (object)
"Takes any object and edits its definition if it can find a
 meaningful definition to edit.  For instance, if it is passed a
 named structure then it edits the defstruct definition for
 that type.  If it fails to edit the object it beeps and tells you what it was
 that it failed to edit a definition of.
"
  (edit object)
)


;;; Just compile this so that M-. works ok.
;;RDA: Commented this out since I don't understand it and it's recursing forever in rel 6.
#+CLOS
(let ((gf (ticlos:get-generic-function-object #'edit)))
     (if (not (ticlos:generic-function-method-hash-table gf))
	 (ticlos:%build-method-hash-table
	   (ticlos:generic-function-discriminator-code
	     gf
	   )
	 )
	 nil
     )
)


;-------------------------------------------------------------------------------

;;; Extensions for reading in things to edit.


(defun try-in-all-packages (action symbols &rest args)
"Performs Action to all of the symbols in Symbols &rest args, within a
 catch error.  If there is no error and a non null result from the action then
 this is returned.
"
  (if symbols
      (let ((result (catch-error (apply action (first symbols) args) nil)))
	   (if result
	       result
	       (try-in-all-packages action (rest symbols))
	   )
      )
      nil
  )
)


(defun try-to-find-function-spec-for-type (spec type)
"Given a spec of the form [eg] (foo :after :bar) and a type of the form [eg]
 :Method or :Property, try to find a function spec of the form
 (:method foo :after :bar), where foo is tried in all reasonable packages.
"
  (if (symbolp (first spec))
      (apply 'try-in-all-packages
	#'(lambda (symbol &rest args)
	    (let ((fspec (cons type (cons symbol args))))
	         (if (fdefinition-location fspec)
		     fspec
		     nil
		 )
	    )
	  )
	(zwei:package-lookalike-symbols (symbol-name (first spec)))
	(rest spec)
      )
      nil
  )
)

(defparameter *function-spec-first-symbols* '(:Method :Property)
"A list of all of the keywords which can start a function spec."
)

(defun try-to-find-function-spec (spec symbols)
"Given a function spec of the form [eg] (foo :after :bar) and a list of type
 symbols of the form (:Method or :Property), try to find a function spec of
 the form (:method foo :after :bar), where foo is tried in all reasonable
 packages.
"
  (if (equal nil symbols)
      nil
      (let ((result (try-to-find-function-spec-for-type spec (first symbols))))
	   (if result
	       result
	       (try-to-find-function-spec-for-type spec (rest symbols))
	   )
      )
  )
)


(defun something-to-edit-error-handler (condition)
"An error handler for the try to edit operation.  If it finds an unbound symbol
 then this is returned, since it might be the name of a function.  If it finds
 a (:method... type spec then it returns this (this would have resulted in an
 undefined function error for :Method.  If it finds a list of the form
 (foo bar baz) then it tries to dwimify a bit looking for a method of plist
 function of that name.
"
  (declare (special *form-read*))
  (cond ((and (condition-typep condition 'sys:unbound-variable)
	      (symbolp *form-read*)
	 )
	 (throw :error? :Proceed-with-this)
	)
	((condition-typep condition 'sys:undefined-function)
	 (if (member (first *form-read*) '(:Method :Property))
	     (throw :error? :Proceed-with-this)
	     (throw :error? :Try-Dwimifying)
	 )
	)
	(t (beep) nil)
  )
  (values)
)

  
(defun read-and-record-a-form (function)
"Reads a form using function, which may prompt, and setqs the form to a special
 which is read outside the function.  The form is then evaled.  Errors which
 happen during the read or the eval will be caught.
"
  (declare (special *form-read*))
  (setq *form-read* (funcall function))
  (eval *form-read*)
)

(defun try-to-read-something-to-edit (function)
"Is passed a function, such as #'read (which may prompt the user) and reads
 in something which might be interpretable as something to edit.  It catches
 sundry errors in order to try to do the Right Thing.  Thus if you type
 (foo :bar) then it will try to look for functions and methods of this name,
 and if it finds one will return it.
"
  (let ((*form-read* nil))
       (declare (special *form-read* *function-spec-first-symbols*))
       (let ((result
	       (catch :error?
		 (condition-bind-if t
		   (((sys:unbound-variable sys:undefined-function)
		     'something-to-edit-error-handler
		    )
		   )
		   (read-and-record-a-form function)
		 )
	       )
	     )
	    )
	    (case result
	      (:Proceed-with-this *form-read*)
	      (:Try-Dwimifying
	       (if (consp *form-read*)
		   (if (equal (first *form-read*) 'Function)
		       *form-read*
		       ;;; Could be a method/prop spec.
		       (try-to-find-function-spec
			 *form-read* *function-spec-first-symbols*
		       )
		   )
		   nil
	       )
	      )
	      (otherwise *form-read*)
	    )
       )
  )
)


(defun abort-handler (&rest ignore)
  (throw :Abort-Tag nil)
)


(defun read-something-to-edit (function message-function)
  (let ((result (try-to-read-something-to-edit function)))
       (if (and (typep result 'si:vanilla-flavor)
	        (or (not (fboundp 'any-sort-of-clos-instance-p))
		    (not (any-sort-of-clos-instance-p result))
		)
	   )
	   (let ((message
		   (catch :Abort-Tag
		     (condition-bind
		       (((sys:abort) #'abort-handler))
		       (try-to-read-something-to-edit message-function)
		     )
		   )
		 )
		)
	        (if message
		    (let ((handler (get-handler-for result message)))
			 (if handler
			     handler
			     (progn
			      (beep)
			      (format t "~&~S does not handle the ~S message.~%"
				      result message
			      )
			      result
			     )
			 )
		    )
		    result
		)
	   )
	   result
       )
  )
)

eh:
(DEFUN tv:my-WINDOW-READ-THING (PROMPT &REST FORMAT-ARGS)
"A modified version of eh:window-read-thing.  This allows the reading of things
 for editting.
"
  (LET (SPECIAL THING)
    (APPLY (FUNCTION FORMAT) T PROMPT FORMAT-ARGS)
    (MULTIPLE-VALUE-SETQ (SPECIAL THING) (WINDOW-COMMAND-LOOP-READ))
    (SETQ THING (IF SPECIAL
                    (IF (SEND *WINDOW-DEBUGGER* :INSPECT-WINDOW-P (THIRD SPECIAL))
                        (TV::INSPECT-REAL-VALUE SPECIAL)
                        (CASE (FIRST SPECIAL)
                          (:MENU  (EQ (SEND (FOURTH SPECIAL) :EXECUTE (SECOND SPECIAL)) T))
                          (STACK-FRAME  (LIST-STACK-FRAME-FUNCTION-AND-ARGS *ERROR-SG*
                                                                            (SECOND SPECIAL)))
                          (:LINE-AREA  (LIST-STACK-FRAME-FUNCTION-AND-ARGS *ERROR-SG*
                                                                           (SECOND SPECIAL)))
                          ((SPECIAL ARG LOCAL)  (FIRST (SECOND SPECIAL)))
                          ((:VALUE :FUNCTION SPECIAL)  (SECOND SPECIAL))))
                    thing))       ;.. take frame into consideration
    (IF (NULL THING) (FORMAT T "~&Aborted.~%"))
    THING))

(let ((compiler:compile-encapsulations-flag t))
     (advise eh:window-read-thing :Around :make-sure-not-inspection-data
	     nil
       (let ((results (multiple-value-list :Do-It)))
	    (if (typep (first results) 'inspection-data)
		(values-list (cons (or (send (first results) :Send-If-Handles
					     :Middle-Button-Result
				       )
				       (send (first results) :Send-If-Handles
					     :Aux-Data
				       )
				       (send (first results) :Data)
				   )
				   (rest results)
			     )
		)
		(values-list results)
	    )
       )
     )
)

(let ((compiler:compile-encapsulations-flag t))
     (advise tv:my-window-read-thing :Around :make-sure-not-inspection-data
	     nil
       (let ((results (multiple-value-list :Do-It)))
	    (if (typep (first results) 'inspection-data)
		(values-list (cons (or (send (first results) :Send-If-Handles
					     :Middle-Button-Result
				       )
				       (send (first results) :Send-If-Handles
					     :Aux-Data
				       )
				       (send (first results) :Data)
				   )
				   (rest results)
			     )
		)
		(values-list results)
	    )
       )
     )
)

(DEFUN INSPECT-GET-VALUE-FROM-USER (*TERMINAL-IO* HISTORY INSPECTORS)	
  "Get a value either by the mouse pointing at it or by read and eval on *TERMINAL-IO*."
  (DECLARE (SPECIAL frame))
  (UNWIND-PROTECT (BLOCK 
                   NIL
                   (DOLIST (I INSPECTORS)
                     (SEND I :SET-SETTING-MODE T))
                   (SEND HISTORY :SET-SETTING-MODE T)
                   (FORMAT *TERMINAL-IO* "~%(type a form to be evaled or select with mouse)~%")
                   (LET ((THING (W:READ-ANY *TERMINAL-IO*))
                         ERROR)
                     (COND
                       ((CONSP THING);; Choose somthing with the mouse -- display it truncated and proceed
                        
                        (COND
                          ((EQ (FIRST THING) :MENU) (BEEP)
                           (FORMAT *TERMINAL-IO* "~&Cannot select a menu item.~%Aborted.~%")
                           (RETURN NIL T))
                          ((EQ (FIRST THING) :MOUSE-BUTTON) (BEEP)
                           (FORMAT *TERMINAL-IO* "~&Did not select anything.~%Aborted.~%")
                           (RETURN NIL T))
                          ((CHAR-EQUAL (FOURTH THING) #\MOUSE-3-1)
			    (FORMAT *TERMINAL-IO* "~&Aborted.~%")
			    (RETURN NIL T)))
                        (LET ((*PRINT-LEVEL* 3)
                              (*PRINT-LENGTH* 5))
			  ;;; JPR.
			  (SETQ THING (INSPECT-REAL-VALUE THING))
			  (setq thing (maybe-data-from-inspection-data thing))
                          (PRIN1 thing *TERMINAL-IO*)))
;                       (T (SEND *TERMINAL-IO* :UNTYI THING)
                       (T (W:UNREAD-ANY THING *TERMINAL-IO*)
                        (MULTIPLE-VALUE-SETQ (THING ERROR)
                          (CATCH-ERROR
                           (EVAL (LET ((*STANDARD-INPUT* *TERMINAL-IO*))
                                   (READ-FOR-TOP-LEVEL)))))
                        (WHEN ERROR;Failed to eval, punt
                         
                          (BEEP)
                          (FORMAT *TERMINAL-IO* "~&Aborted.~%")
                          (RETURN NIL T))))
                     (TERPRI *TERMINAL-IO*)
                     (RETURN THING)) nil)
                  (DOLIST (I INSPECTORS)
     (SEND I :SET-SETTING-MODE NIL))
    (SEND HISTORY :SET-SETTING-MODE NIL)))


(DEFUN my-INSPECT-GET-VALUE-FROM-USER (*TERMINAL-IO* HISTORY INSPECTORS)	
  "Get a value either by the mouse pointing at it or by read and eval on *TERMINAL-IO*.  Modified from Inspect-Get-Value-from-User, so as to read things for
 editting."
  (DECLARE (SPECIAL frame))
  (UNWIND-PROTECT (BLOCK 
                   NIL
                   (DOLIST (I INSPECTORS)
                     (SEND I :SET-SETTING-MODE T))
                   (SEND HISTORY :SET-SETTING-MODE T)
                   (FORMAT *TERMINAL-IO* "~%(type a form to be evaled or select with mouse)~%")
                   (LET ((THING (W:READ-ANY *TERMINAL-IO*))
                         ERROR)
                     (COND
                       ((CONSP THING);; Choose somthing with the mouse -- display it truncated and proceed
                        
                        (COND
                          ((EQ (FIRST THING) :MENU) (BEEP)
                           (FORMAT *TERMINAL-IO* "~&Cannot select a menu item.~%Aborted.~%")
                           (RETURN NIL T))
                          ((EQ (FIRST THING) :MOUSE-BUTTON) (BEEP)
                           (FORMAT *TERMINAL-IO* "~&Did not select anything.~%Aborted.~%")
                           (RETURN NIL T))
                          ((CHAR-EQUAL (FOURTH THING) #\MOUSE-3-1)
			    (FORMAT *TERMINAL-IO* "~&Aborted.~%")
			    (RETURN NIL T)))
                        (LET ((*PRINT-LEVEL* 3)
                              (*PRINT-LENGTH* 5))
			  ;;; JPR
			  (SETQ THING (INSPECT-REAL-VALUE THING))
			  (setq thing (maybe-data-from-inspection-data thing))
                          (PRIN1 thing *TERMINAL-IO*)))
;                       (T (SEND *TERMINAL-IO* :UNTYI THING)
                       (T (W:UNREAD-ANY THING *TERMINAL-IO*)
                        (MULTIPLE-VALUE-SETQ (THING ERROR)
                          (CATCH-ERROR
                           (LET ((*STANDARD-INPUT* *TERMINAL-IO*))
                                   (try-to-read-something-to-edit
				     'READ-FOR-TOP-LEVEL))))
                        (WHEN ERROR;Failed to eval, punt
                         
                          (BEEP)
                          (FORMAT *TERMINAL-IO* "~&Aborted.~%")
                          (RETURN NIL T))))
                     (TERPRI *TERMINAL-IO*)
                     (RETURN THING)) nil)
                  (DOLIST (I INSPECTORS)
     (SEND I :SET-SETTING-MODE NIL))
    (SEND HISTORY :SET-SETTING-MODE NIL)))

;-------------------------------------------------------------------------------

(Defun inspect-try-and-edit (something)
  (declare (special user history = inspectors frame))
  (if (and (typep something 'si:vanilla-flavor)
	   (or (not (fboundp 'any-sort-of-clos-instance-p))
	       (not (any-sort-of-clos-instance-p something))
	   )
      )
      (if (and (boundp '*general-inspector-enabled*)
	       (symbol-value '*general-inspector-enabled*)
	       (typep something 'inspection-data)
	  )
	  ;;; Hook for general inspector.
	  (if (funcall 'map-into-show-x something)
	      (try-and-edit (funcall 'map-into-show-x something))
	      (try-and-edit (send something :send-if-handles :data))
	  )
	  (let ((message
		  (catch :Abort-Tag
		    (condition-bind
		      (((sys:abort) #'abort-handler))
		      (format user 
	     "~&Select a method name [] for flavor definition:"
		      )
		      (multiple-value-bind (value punt-p)
			  (inspect-get-value-from-user user history inspectors)
			(if punt-p
			    (throw :Abort-tag nil)
			    value
			)
		      )
		    )
		  )
		)
	       )
	       (if message
		   (let ((handler (get-handler-for something message)))
			(if handler
			    (try-and-edit handler)
			    (progn
			      (beep)
			      (format t "~&~S does not handle the ~S message.~%"
				      something message
			      )
			      (try-and-edit something)
			    )
			)
		   )
		   (try-and-edit something)
	       )
	  )
      )
      (try-and-edit something)
  )
)

(Defcommand Inspect-Edit-Cmd nil			
  '(:description "Edit the definition of something."
    :names ("Edit")
    :keys (#\m-. #\c-E #\h-E)
   )
   (letf ((#'inspect-get-value-from-user #'my-inspect-get-value-from-user))
	 (do-something-and-inspect "~&Object to edit definition of:"
				   'Inspect-Try-and-Edit
	 )
   )
)

(reinstall-inspector-commands)

;-------------------------------------------------------------------------------

;;; For the window debugger.


;-------------------------------------------------------------------------------

;;; Redefine the edit command so that it will take m-. as a key
;;; assignment for compatibility with ZMacs.
;;; Just add new key assignments.

eh:
(defun eh:comw-edit (ignore ignore)
"This is a redefined version of the debugger's edit command.  It uses the
 consistency enhancements mechanism for finding the source file to edit.
"
  (let ((thing (tv:read-something-to-edit
		 #'(lambda ()
		     (tv:my-window-read-thing
		       "~%Type or mouse on something to edit: " T
		     )
		   )
		 #'(lambda ()
		     (tv:my-window-read-thing
      "~%Type or mouse on a message name [] for flavor definition : " T
		     )
		   )
	       )
	)
       )
       (tv:try-and-edit thing)
  )
)


eh:
(Defcommand eh:Comw-Edit-Cmd nil
  '(:description  "Invoke the Editor on a specified function."
    :names "Edit" 
    :keys (#\m-. #\c-E #\h-e)
   )
  (send *window-debugger* :set-who-line-doc-string-overide
	"Select a function to edit."
  )
  (unwind-protect (comw-edit *error-sg* *error-object*)
    (progn (send *window-debugger* :set-who-line-doc-string-overide nil)
	   (send *window-debugger* :handle-prompt)
    )
  )
)


;-------------------------------------------------------------------------------

#+CLOS
(defvar ticlos:*display-method-docs-for-generic-functions-p* nil
"When t, causes the doc strings for methods of a generic function to be
 printed when (documentation is called on a GF.  If this is :query then
 the user is asked before the method docs are shown.  If it is :never, then
 the method docs are never shown, even when the GF has no docs.
"
)

#+CLOS
ticlos:
(defmethod ticlos:documentation ((dobj ticlos:standard-generic-function) &optional doc-type)
  (when doc-type
    (error "~%A second argument, ~s, was supplied in a call to DOCUMENTATION 
of a generic function object.~%" doc-type))
  (let ((doc-string (if (compiled-function-p dobj)
			(let ((dbi (si:get-debug-info-struct dobj)))
			  (si:get-debug-info-field dbi :documentation))
			(ticlos:generic-function-documentation dobj))))
    (case *display-method-docs-for-generic-functions-p*
      (nil (or doc-string
	       (apply #'string-append
		      (mapcar 'get-method-doc (generic-function-methods dobj))
	       )
	   )
      )
      (:never doc-string)
      (:query (if (y-or-n-p "~&Show doc strings for methods of ~S?" dobj)
		  (apply #'string-append
			 doc-string
			 (mapcar 'get-method-doc
				 (generic-function-methods dobj)
			 )
		  )
		  doc-string
	      )
      )
      (t (apply #'string-append
		doc-string
		(mapcar 'get-method-doc (generic-function-methods dobj))
	 )
      )
    )
  )
)

#+CLOS
ticlos:
(defun ticlos:get-method-doc (method)
  (let ((doc (documentation method)))
       (if doc
	   (format nil "~%From ~A:~%~A~%"
		   (function-name (method-function method)) doc
	   )
	   ""
       )
  )
)
;-------------------------------------------------------------------------------

;;; W A R N I N G.....
;;; The following puts a pathname dependency into this file.
;;; This should really be done by a conditional load in the defsystem,
;;; but Defsystem doesn't have such things....   JPR. 28 Feb 89.

(if (eq sys:(processor-type microcode-type-code) :micro-Explorer)
    (si:load-if "TOOLS:TOOLS;DTCE-MICROEXPLORER-SPECIFIC" :verbose nil)
    (if nil;(= 6 (sys:get-system-version))  ;;; put back in by JPR on 07/18/89 16:13:55.
	nil
	(si:load-if "TOOLS:TOOLS;DTCE-EXPLORER-SPECIFIC" :verbose nil)
	)
)

(provide 'Development-Tool-Consistency-Enhancements)

;;; Code to install the commands...

(install-consistency-commands)

;(setq *selected-consistancy-commands* :Menu)
;(setq *selected-consistancy-commands* :All)