;;; -*- Mode:Common-Lisp; Package:FILE-SYSTEM; Syntax:COMMON-LISP; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990, 1992 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	Rich Acuff
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in Feb '86 and extended
;;; in '87 and Jan '92.
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15
;;;	DARPA Grant N00039-86- C-0033

;;;  File-System to File-System backup facility.  Separate doc file.

;;; Make sure these are where we want them.
(let ((*error-output* 'si:null-stream))
  (export '(full-backup backup-files restore-files)
	  "TICL")
  )

(let ((sym (find-symbol "*DEFAULT-BACKUP-ROOT*" "USER")))
  (declare (special *default-backup-root*))
  (when (and sym (neq (symbol-package sym) (find-package "FS")))
    (unintern sym)
    (export (list sym) "TICL")
    (when (boundp sym) (setf *default-backup-root* (eval sym)))
    )
  )
      
(defvar *default-backup-root* '|You haven't set *DEFAULT-BACKUP-ROOT*!!|
  "A string or pathname denoting the root directory in which to place
   files being backed up.")

(defvar *n-files*)
(defvar *total-size*)

(defun full-backup (&key (from-path (user-homedir-pathname))
		    (to-root *default-backup-root*)
		    (report-file *standard-output*)
		    (do-subdirs t)
		    copy-file-options)
  "  Clear all files out of TO-ROOT and backup all files in FROM-PATH to TO-ROOT
   with BACKUP-FILES."
  (setq to-root (canonicalize-path to-root))
  (setq from-path (canonicalize-path from-path))
  (when (and (equal (pathname-host to-root) (pathname-host from-path))
	     (equal (pathname-directory to-root) (pathname-directory from-path)))
    (ferror "Attempt to backup to same place as that being backed up: from ~A to ~A"
	    from-path to-root))
  (delete-files-and-expunge (send to-root :new-pathname :name :wild
				  :type :wild :version :wild))
  (apply #'backup-files :from-path from-path :to-root to-root
	 :report-file report-file
	 :do-subdirs do-subdirs :all-p t
	 copy-file-options)
  )

;;;Edited by Acuff                 27 Oct 87  8:56
(defun backup-files (&key (from-path (user-homedir-pathname))
		     (to-root *default-backup-root*)
		     (all-p nil)
		     (report-file *standard-output*)
		     (do-subdirs t)
		     (indent-level 0)
		     ;;; Arg added by JPR on 3 Mar 87
		     (ignore-root nil)
		     ;; Added by RDA 13-Feb-92
		     short-report
		     copy-file-options)
  " Backup files specified by FROM-PATH onto the root specified by
   TO-ROOT.  Thus, if FROM-PATH is \"X1:A.B;*\" and TO-ROOT is
   \"SOME-UNIX:/u/\" then files will be put into \"SOME-UNIX:/u/a/b/*\".
   Files thus backed up are marked as backed up.  If ALL-P is non-NIL
   then even files that have already been backed-up are copied.
   COPY-FILE-OPTIONS is a list of options to be given to COPY-FILE when
   the backup is done.  Progress is recorded on REPORT-FILE which can be
   a stream, or a string or pathname acceptable to OPEN.  If it is the
   later, then it is openned for output, and closed when BACKUP-FILES is
   done.  Subdirectories are copied if DO-SUBDIRS is non-NIL.
   If IGNORE-ROOT is non nil then the root of the directory of the
   FROM-PATH is filtered so that, for instance, files in \"X1:TEMP...;*\"
   can be copied to \"X2:...;\" without the \"TEMP\" directory getting
   in the way.  If this argument is T then the directories of the
   FROM-PATH are used.  If it is a pathname then the directory component
   of that pathname will be used.  SHORT-REPORT non-NIL causes an abbreviated
   version of the report output to be used."
  
  (setq to-root (canonicalize-path to-root))
  (setq from-path (merge-pathnames (canonicalize-path from-path)
				   (pathname "lm:~;*.*#*")
				   :newest))
  (unless (pathname-version from-path)		;No version means :NEWEST!
    (setq from-path (send from-path :new-version :newest)))
  (when (and (equal (pathname-host to-root) (pathname-host from-path))
	     (equal (pathname-directory to-root) (pathname-directory from-path))
	     ;;; Added by JPR on 3 Mar 87.
	     (not ignore-root))
    (ferror "Attempt to backup to same place as that being backed up: from ~A to ~A"
	    from-path to-root))
  (let ((close-stream? nil)
	(*n-files* 0)
	(*total-size* 0))
    (unwind-protect
	(progn
	  (unless (streamp report-file)
	    (setq close-stream? t)
	    (setq report-file (open report-file :direction :output)))
	  (backup-files-1 from-path to-root all-p report-file do-subdirs
			  indent-level copy-file-options
			  ;;; Added by JPR on 3 Mar 87
			  (cond ((equal ignore-root nil) nil)
				((equal ignore-root t)
				 (pathname-directory from-path))
				((pathnamep ignore-root)
				 (pathname-directory ignore-root))
				(t (ferror nil "The Ignore-Root argument [~S] is not T, Nil, or a pathname." ignore-root)))
			  short-report)
	  )
      (when close-stream? (close report-file))
      )
    (values *n-files* *total-size*)
    )
  )


(defun directory-with-subdirs (from-path do-subdirs)
;; This function defined by JPR on 27 Mar 87 because when one does a
;; backup-files of foo:bar;*.lisp the subdirs are not included because the
;; fs:directory list call removes them.  A call to this function has been
;; placed wherever there was such a call to fs:directory-list
  (if (or (equal nil (pathname-type from-path))
	  (equal :Wild (pathname-type from-path))
	  (not do-subdirs)
      )
      (fs:directory-list from-path)
      (let ((files (fs:directory-list from-path))
	    (all-files (fs:directory-list (send from-path :New-Type :Wild)))
	   )
	   (append files
		(remove-if-not #'(lambda (dirspec) (get dirspec :directory)) all-files)
	   )
      )
  )
)


(defun backup-files-1 (from-path to-root all-p report-stream do-subdirs
		       indent-level copy-file-options ignore-root short-report)
  (let ((to-path (send (concat-directories to-root from-path ignore-root)
		       :new-pathname :name nil :type nil :version nil)))
    (if *be-quiet-about-already-backed-up-files*    ;Gruber 11/25/89 15:05:54
	(when short-report
	  (if (zerop indent-level)
	      (format report-stream "~&~V@TBacking up ~A to ~A~&"
		      indent-level from-path to-path)
	      (format report-stream "~&~V@T(dir) ~A~&"
		      indent-level
		      (first (last (pathname-directory from-path))))))
	(format report-stream "~&~V@TBacking up ~A to ~A~&"
		indent-level from-path to-path))
    (loop for (one-from-path . from-plist)
	  in (directory-with-subdirs from-path do-subdirs)
	  when (pathnamep one-from-path)
	  do
	  (if (and do-subdirs (getf from-plist :directory))
	      ;;; Backup this subdir
	      (backup-files-1
		(send (send one-from-path :pathname-as-directory)
		      :new-pathname
		      :name (pathname-name from-path)
		      :type (pathname-type from-path)
		      :version (pathname-version from-path))
		to-root all-p report-stream do-subdirs (+ indent-level 1)
		copy-file-options ignore-root short-report)
	      (progn
		(backup-copy-one-file one-from-path to-path
				      (list* :if-exists :overwrite
					     copy-file-options)
				      report-stream indent-level t
				      all-p short-report)
		(when (or all-p (getf from-plist :not-backed-up))
		  (incf *n-files* 1)
		  (incf *total-size*
			(or (getf from-plist :length-in-blocks) 0))))
		
	      ) 
	  )
    )
  )

(defun restore-files (&key (to-path (user-homedir-pathname))
		      (from-root *default-backup-root*)
		      (overwrite :ask)
		      (report-file *standard-output*)
		      (do-subdirs t)
		      (indent-level 0)
		      (copy-tilde-files? nil)
		      ;;; Arg added by JPR on 3 Mar 87
		      (ignore-root nil)
		      copy-file-options)
  " Restore files backed up on FROM-ROOT to TO-PATH.  This is the
   inverse function for BACKUP-FILES.  COPY-FILE-OPTIONS are given to
   COPY-FILE, so that, for instance, the :AFTER option could be used to
   get files written after a certain time.  Files are not overwritten by
   default.  OVERWRITE controls what is done when the file to be
   restored already exists (any version): NIL or :NEVER means never
   overwrite, :ASK means query the user (via *query-io*), and :ALWAYS or
   T means always overwrite.  Report progress to REPORT-FILE.  Descend
   subdirectories if DO-SUBDIRS is non-NIL.  Only if COPY-TILDE-FILES?
   is non-NIL are unix files ending in `~' copied since these files are
   usually old versions.  If IGNORE-ROOT is non nil then the root of the
   directory of the FROM-ROOT is filtered so that, for instance, files
   in \"X1:TEMP...;*\" can be copied to \"X2:...;\" without the \"TEMP\"
   directory getting in the way.  If this argument is T then the
   directories of the FROM-ROOT are used.  If it is a pathname then the
   directory component of that pathname will be used."
  (setq from-root (canonicalize-path from-root))
  (setq to-path (merge-pathnames (canonicalize-path to-path) "*.*" :newest))
  (unless (pathname-version to-path)		;No version means :NEWEST!
    (setq to-path (send to-path :new-version :newest)))
  (unless (not copy-tilde-files?)		;only check if unix host
    (setq copy-tilde-files? (neq (send from-root :system-type) :unix-ucb)))
  (case overwrite
    ((nil :never) (setq copy-file-options
			(list* :if-exists nil copy-file-options)))
    ((t :always :ask)  (setq copy-file-options
			     (list* :if-exists :overwrite copy-file-options)))
    (otherwise (fsignal "Illegal OVERWRITE option for RESTORE-FILES: ~A"
			overwrite))
    )
  (when (and (equal (pathname-host from-root) (pathname-host to-path))
	     (equal (pathname-directory from-root) (pathname-directory to-path))
	     ;;; Added by JPR on 3 Mar 87
	     (not ignore-root))
    (ferror "Attempt to restore from same place being restored to: to ~A from ~A"
	    to-path from-root))
  (let ((close-stream? nil))
    (unwind-protect
	(progn
	  (unless (streamp report-file)
	    (setq report-file (open report-file :direction :output))
	    (setq close-stream? t))
	  (restore-files-1 to-path from-root overwrite report-file
			   do-subdirs indent-level copy-tilde-files? copy-file-options
			   ;;; Added by JPR on 3 Mar 87
			   (cond ((equal ignore-root nil) nil)
				((equal ignore-root t) (pathname-directory from-root))
				((pathnamep ignore-root) (pathname-directory ignore-root))
				(t (ferror nil "The Ignore-Root argument [~S] is not T, Nil, or a pathname." ignore-root))))
	  )
      (when close-stream? (close report-file))
      )
    )
  (values)
  )

(defun restore-files-1 (to-path from-root overwrite report-stream
			do-subdirs indent-level copy-tilde-files? copy-file-options ignore-root)
  (let ((from-paths (concat-directories from-root to-path ignore-root)))
    (format report-stream "~&~V@TRestoring from ~A to ~A~&"
	    indent-level from-paths to-path)
    (loop for (from-path . from-path-plist)
	  in (directory-with-subdirs from-paths do-subdirs) 
	  with one-path
	  when (and (pathnamep from-path)
		    (or copy-tilde-files?
			(not (stringp (pathname-type from-path)))
			(not (char= (elt (string	;For Symbolics...
					   (pathname-type from-path))
					 (1- (length (pathname-type from-path))))
				    #\~))))
	  do
	  (if (and do-subdirs (getf from-path-plist :directory))
	      ;;; Restore this subdir
	      (restore-files-1 (get-new-restore-path to-path from-path)
			       from-root overwrite report-stream do-subdirs
			       (1+ indent-level) copy-tilde-files? copy-file-options ignore-root)
	      ;;; Restore this file, maybe
	      (when (setq one-path (should-restore? to-path from-path
						    from-path-plist overwrite))
		(backup-copy-one-file from-path one-path copy-file-options
				      report-stream indent-level nil t nil)
		)
	      )
	  )
    )
  (values)
  )

(defun nil-for-root (dirspec)
  "returns the dirspec or nil if the dirspec is :Root"
  (if (equal dirspec :Root) nil dirspec))

(defun without-ignored-part (dirspec ignored-dirspec)
  "Returns a dirspec which does not start with the ignored-components."
  (if (or (equal nil ignored-dirspec) (equal nil dirspec))
      dirspec
      (if (string-equal (first dirspec) (first ignored-dirspec))
	  (without-ignored-part (rest dirspec) (rest ignored-dirspec))
	  dirspec
      )
  )
)

(defun concat-directories (root path ignore-root)
  "Add the file name specified by PATH onto ROOT, including directories.  Both args
   must be pathnames.  Do not use the directory levels specified in ignore-root
   in the output pathname."
  (make-pathname :host (pathname-host root)
		 :directory (append (nil-for-root (pathname-directory root))
				    (without-ignored-part (nil-for-root (pathname-directory path)) ignore-root))
		 :name (pathname-name path)
		 :type (pathname-type path)
		 :version (pathname-version path)
		 )
  )

;;; Gruber 8/16/89
(defvar *be-quiet-about-already-backed-up-files* nil
  "If this flag is true, messages about already backed up files are supressed
during backup.")

(defun report-file-copied (stream indent-level from to short?)
  ;;major kludge...
  (if short?
      (format stream "~&~V@T~A~%" indent-level
	      (if (eq :lispm (send (pathname-host from) :system-type))
		  (format nil "~A.~A#~A" (pathname-name from)
			  (pathname-type from)
			  (pathname-version from))
		  from))	  
      (format stream "~&~V@T~A~@3,5T=>~@3,5T~A~%" indent-level from to)))

(defun backup-copy-one-file (from-path
			     to-path
			     copy-file-options
			     report-stream
			     indent-level
			     set-backed-up-flag
			     copy-backed-up?
			     short?)
  "   Copy the file FROM-PATH to TO-PATH, reporting to REPORT-STREAM,
   indenting by INDENT-LEVEL, and giving COPY-FILE-OPTIONS to
   COPY-FILE."
  (when set-backed-up-flag
    (setq copy-file-options
	  (list* :set-backed-up-flag t copy-file-options)))
  (unless copy-backed-up?
    (setq copy-file-options
	  (list* :not-backed-up-only t copy-file-options)))
  (multiple-value-bind (to from result)
      (apply #'copy-file from-path
	     to-path
	     :create-directories t
	     copy-file-options
	     )
    (ignore to)				    ;result is the real truename
    (when (listp result) (setq result (car result)))
    (when (listp from) (setq from (car from)))
    (if (pathnamep result)
	(report-file-copied report-stream indent-level from-path result short?)
	(when (errorp result)
	  (if (member :dont-need-to-copy (send result :condition-names))
	      (if (and (not short?)
		       (not *be-quiet-about-already-backed-up-files*))
		  (report-file-copied report-stream indent-level from-path
				      "[already backed up]" short?))
	      (progn
		(report-file-copied
		  report-stream indent-level from-path "" short?)
		(format report-stream "~&   Reason for failure: ~A" result))
	      )
	  )
	)
    )
  (values)
  )

(defun get-new-restore-path (old-path new-dir-path)
  (send old-path :new-directory
	(if (listp (pathname-directory old-path))
	    (append (pathname-directory old-path)
		    (last (pathname-directory
			    (send new-dir-path :pathname-as-directory))))
	    ;;; Mod added by JPR on 3 Mar 87
	    (if (equal :Root (pathname-directory old-path))
		(last (pathname-directory
			      (send new-dir-path :pathname-as-directory)))
	        (cons (pathname-directory old-path)
		      (last (pathname-directory
			      (send new-dir-path :pathname-as-directory)))))
	    )
	)
  )

(defconstant restore-fquery-options `(((:yes "Yes.")
				       ,(char-int #\Y)
				       ,(char-int #\T)
				       ,(char-int #\SP)
				       ,(char-int #\HAND-UP))
				      ((:no "No.")
				       ,(char-int #\N)
				       ,(char-int #\RUBOUT)
				       ,(char-int #\HAND-DOWN))
				      ((:info "Info")
				       ,(char-int #\I))
				      ((:new-version "New Version")
				       ,(char-int #\>))
				      )
  )

(defconstant restore-query-format-string
	     "~&~:@{~%~V@<~A~>~@2,2T~D(~D)~@2,3T~A~@2T~A~}~2%")

(defun should-restore? (to-path-spec from-path from-path-plist overwrite-flag)
  (let ((to-path (send to-path-spec :new-default-pathname
		       :name (pathname-name from-path)
		       :type (pathname-type from-path)
		       :version (pathname-version from-path))))
    (if (setq to-path (probe-file to-path))
	(case overwrite-flag
	      (:ask
	       (case (fquery `(:choices ,restore-fquery-options)
			     "~&    ~A already exists--overwrite it? "
			     to-path)
		     (:yes to-path)
		     (:no nil)
		     (:info
		      (let ((to-path-plist (cdr (fs:file-properties to-path)))
			    (path-width
			      (max (length (send to-path :string-for-printing))
				   (length (send from-path :string-for-printing))))
			    )
			(format *query-io* restore-query-format-string
				(restore-info-list path-width to-path to-path-plist)
				(restore-info-list path-width
						   from-path from-path-plist)))
		      (should-restore? to-path-spec from-path
				       from-path-plist overwrite-flag)
		      )
		     (:new-version
		      (if (numberp (pathname-version to-path))
			  (send to-path :new-version (1+ (pathname-version to-path)))
			  (progn
			    (format *query-io* 
				    "~QThis file system doesn't have version numbers, so I can't use a new version."
				    'beep)
			    (should-restore? to-path-spec from-path
					     from-path-plist overwrite-flag))
			  )
		      )
		     )
			)
	      ((nil :never) nil)
	      ((t :always) to-path)
	      )
	to-path-spec
	)
    )
  )

;;;Edited by Acuff                 22 Oct 87  8:48
;;;Edited by Acuff                 27 Oct 87  8:47
(defun delete-files-and-expunge (paths)
  "Delete all files, including subdirs, in PATHS and expunge."
  (when (yes-or-no-p "Are you sure you want to delete and expunge ~A? "
		     paths)
    (delete-directory paths :are-you-sure t)
    )
  )


;;; RDA: This is no longer needed as we use DELETE-DIRECTORY
(defun delete-files-and-expunge-1 (paths)
  (condition-case (error)
      (loop for (path . plist) in (fs:directory-list paths)
	    when (pathnamep path) do
	    (when (getf plist :directory)
	      (delete-files-and-expunge-1 (send (send path :pathname-as-directory)
						:new-pathname
						:name :wild
						:type :wild
						:version :wild)))
	    and do
	    (send path :delete)
	    finally (send paths :expunge :error nil)
	    )
    ((fs:file-not-found fs:directory-not-found) nil)
    )
  )

(defun restore-info-list (width path plist)
  (list width
	path
	(or (getf plist :length-in-bytes) "???")
	(or (getf plist :byte-size) "???")
	(if (getf plist :creation-date)
	    (time:print-universal-time (getf plist :creation-date) nil)
	    "Unknown time    ")
	(or (getf plist :author) "???")
	)
  )

;;; We have to have this and use it because of the (*&^*&^%*&^ Symbolics Common
;;; Lisp, since functions like PARSE-NAMESTRING don't understand ZL strings.
(defun canonicalize-path (root)
  (cond
    ((pathnamep root) root)
    ((stringp root) (parse-namestring root))
    (t (fsignal "~A is not a pathname or string that can be used for backup."
		root)
       )
    )
  )
