;;; -*- Mode:Common-Lisp; Package:TV; Base:8; Fonts:(TVFONT HL10B TR10); Patch-file:T -*-
;;; ***** Copyright (c) 1987 Texas Instruments.  All rights reserved.
;;; ***** Portions of this file contain code belonging to TI.

(DEFUN flavor-components (flavor-name &OPTIONAL (top-level? T))
  "1Returns a list (<flavor name> <included flavors> <component flavors>) where
<flavor name> is argument FLAVOR-NAME,
<included flavors> is a list of entries representing FLAVOR-NAME's included flavors, and
<component flavors> is a list of entries representing FLAVOR-NAME's mixins.
Entries in <included flavors> and <component flavors> are of the same form as the value
we are returning; in otherwords, we are returning a recursive data structure.*"
  (LET* ((flavor (GET flavor-name 'SI:FLAVOR))
	 ;1;Test put in by JPR 31 May 86 to stop it from bombing out when a component*
	 ;1;is not defined.*
	 (result (if flavor
		     `(,flavor-name
		       ,(LOOP FOR included IN (SI:FLAVOR-INCLUDES flavor)
			      COLLECT (flavor-components included NIL))
		       ,(LOOP FOR component IN (SI:FLAVOR-DEPENDS-ON flavor)
			      COLLECT (flavor-components component NIL)))
		     (list flavor-name))))
    1;;Add on vanilla flavor at end of top-level flavor, unless it's an abstract flavor.*
    (WHEN (AND top-level? (NOT (SI:FLAVOR-GET flavor :NO-VANILLA-FLAVOR)))
      (SETF (THIRD result) (NCONC (THIRD result) '((SI:VANILLA-FLAVOR NIL NIL)))))
    result))

;-------------------------------------------------------------------------------

(defun coerce-to-flavor (something &optional (error-p t))
  (typecase something
    (sys:flavor something)
    (symbol (if something
		(coerce-to-flavor (get something 'sys:flavor) error-p)
		(if error-p
		    (ferror nil "~S cannot be coerced into a flavor object."
			    something
		    )
		    nil
		)
	    )
    )
    (otherwise
     (if error-p
	 (ferror nil "~S cannot be coerced into a flavor object."
		 something
	 )
	 nil
     )
    )
  )
)

(defun flatten (list)
  (if (consp list)
      (if (consp (first list))
	  (append (flatten (first list)) (flatten (rest list)))
	  (mapcar #'flatten list)
      )
      list
  )
)

(defun all-flavor-components (flavor)
  (let ((flavor (coerce-to-flavor flavor nil)))
       (if flavor
	   (sys:uniqueise
	     (cons flavor
		   (flatten
		     (append
		       (mapcar #'all-flavor-components
			       (si:flavor-includes flavor)
		       )
		       (mapcar #'all-flavor-components
			       (si:flavor-depends-on flavor)
		       )
		     )
		   )
	     )
	   )
	   nil
       )
  )
)

(defun all-flavor-dependents (flavor)
  (let ((flavor (coerce-to-flavor flavor nil)))
       (if flavor
	   (sys:uniqueise
	     (cons flavor
		   (flatten
		     (mapcar #'all-flavor-dependents
			     (si:flavor-depended-on-by flavor)
		     )
		   )
	     )
	   )
	   nil
       )
  )
)

(defun related-flavors (flavor)
  (let ((components (all-flavor-components flavor))
        (dependents (all-flavor-dependents flavor))
       )
       (sys:uniqueise (append components dependents))
  )
)

(defun related-flavor-names (flavor)
  (mapcar #'sys:flavor-name (related-flavors flavor))
)


(DEFFLAVOR show-related-methods () (inspection-data))

(DEFMETHOD (show-related-methods :format-concisely) (stream)
  (FORMAT stream "Related methods for ~S ~S"
	  (second (SI:METH-FUNCTION-SPEC DATA))
	  (first (last (SI:METH-FUNCTION-SPEC DATA)))))

(DEFMETHOD (show-related-methods :aux-data) ()
  (second data))

(DEFMETHOD (show-related-methods :handle-mouse-click) (blip flavor-inspector)
  (SELECTOR (FOURTH blip) =
      (#\MOUSE-L-1 (SEND flavor-inspector :inspect-info-left-click))
      (t (beep))))

(DEFMETHOD (show-related-methods :who-line-doc) (IGNORE &OPTIONAL IGNORE)
  '(:MOUSE-L-1 "Inspect method"))

(DEFMETHOD (show-related-methods :GENERATE-ITEM) ()
  (VALUES
    (LET* ((special-comb?)
	   (message-name (first (last (SI:METH-FUNCTION-SPEC DATA))))
	   (items (LOOP FOR flavor-name IN (related-flavor-names (second (SI:METH-FUNCTION-SPEC DATA)))
			FOR flavor = (GET flavor-name 'SI:FLAVOR)
			FOR method-table = (SI:FLAVOR-METHOD-TABLE flavor)
			NCONC (MULTIPLE-VALUE-BIND (items comb?)
				  (collect-method-items
				    (m tpl method-table)
				    (equal (first m) message-name) NIL T)
				(WHEN comb?
				  (SETQ special-comb? T))
				items))))
      `(,*blank-line-item*
	((:FONT 1 "Related Methods of ")
	 (:ITEM1 INSTANCE
	   ,(ALLOCATE-DATA 'SHOW-method-details
		 (get (second (SI:METH-FUNCTION-SPEC DATA)) 'sys:flavor) DATA))
	 (:FONT 1 ,(IF special-comb? ".  * = special method combination type" ":")))
	,*blank-line-item*
	,*method-display-columns-2*
	,@(SORT items
		#'(LAMBDA (x y)
		    (LET* ((x-method-spec (CAR (SEND (THIRD (THIRD x)) :DATA)))
			   (y-method-spec (CAR (SEND (THIRD (THIRD y)) :DATA)))
			   (x-message (si:message x-method-spec))
			   (y-message (si:message y-method-spec)))
		      (IF (EQ x-message y-message)
			  (LET ((x-method-type (si:method-type x-method-spec)))
			    (IF (AND x-method-type (EQ x-method-type (si:method-type y-method-spec)))
				(string-lessp-nil-wins (si:submessage x-method-spec) (si:submessage y-method-spec))
				(string-lessp-nil-wins x-method-type (si:method-type y-method-spec))))
			  (STRING-LESSP x-message y-message)))))))
    `(:FONT fonts:hl12bi :STRING ,(FORMAT NIL "Related methods for ~s ~S"
					  (second (SI:METH-FUNCTION-SPEC DATA))
					  (first (last (SI:METH-FUNCTION-SPEC DATA)))))))


;-------------------------------------------------------------------------------


(defmethod (show-method :print-self) (stream &rest ignore)
  (catch-error
    (if (and (consp data)
	     (consp (first data))
	     (equal :method (first (first data)))
	)
	(format stream "#<Show-Method ~S>" (first data))
	(format stream "#<Show-Method ???>" (first data)))))


(defmethod (show-flavor :print-self) (stream &rest ignore)
  (catch-error (format stream "#<Show-Flavor ~S>" (sys:flavor-name data))))

;-------------------------------------------------------------------------------

(DEFFLAVOR show-init-keywords () (show-flavor))

(DEFMETHOD (show-init-keywords :format-concisely) (stream)
  (FORMAT stream "Init Keywords for ~s" (SI:FLAVOR-NAME data)))
;(pprint (let ((flavor-name 'user:f)) (COLLECT-DEPENDENT-FLAVORS FLAVOR-NAME 1 (FLAVOR-COMPONENTS FLAVOR-NAME) (FLAVOR-COMPONENTS FLAVOR-NAME))))
(DEFMETHOD (SHOW-INIT-KEYWORDS :GENERATE-ITEM) ()
  (LET* ((FLAVOR-NAME (SI:FLAVOR-NAME DATA))
	 (ALL-COMPONENTS (FLAVOR-COMPONENTS FLAVOR-NAME))
	 TEXT-ITEMS
	 REDUNDANT-INCLUDED-FLAVOR?
	 REDUNDANT-MIXIN-FLAVOR?)
    (DECLARE (SPECIAL REDUNDANT-INCLUDED-FLAVOR? REDUNDANT-MIXIN-FLAVOR?))
    (SETQ TEXT-ITEMS (COLLECT-DEPENDENT-FLAVORS-AND-INIT-KEYWORDS FLAVOR-NAME 1 ALL-COMPONENTS ALL-COMPONENTS))
    (VALUES
     `(,*BLANK-LINE-ITEM*
       ,@(itemise-init-keywords (sys:flavor-name data) 1)
       ,*BLANK-LINE-ITEM*
       ((:FONT 1 "Init Keywords of Flavor ")
        (:ITEM1 INSTANCE ,(ALLOCATE-DATA 'SHOW-FLAVOR DATA))
	(:FONT 1 "'s component flavors.")
	,@(WHEN REDUNDANT-MIXIN-FLAVOR?
	    '((:FONT 2 "  *")
              (:FONT 1 " = redundant mixin  ")))
	,@(WHEN REDUNDANT-INCLUDED-FLAVOR?
	    '((:FONT 2 "  **")
              (:FONT 1 " = redundant included flavor"))))
       ,*BLANK-LINE-ITEM*
	;;For each component method, a mouse sensitive method name (METHOD-NAME).
       ,@(OR TEXT-ITEMS *NO-ITEMS*))
     ;;Make the label display the flavor name.
     `(:FONT FONTS:HL12BI :STRING ,(FORMAT NIL "~s" FLAVOR-NAME)))))  


(defun itemise-init-keywords (flavor print-level)
  (if (get flavor 'si:flavor)
      (let ((keys (sort (append
			  (mapcar
			    #'(lambda (key)
				`((:font 1 ,(format nil "~S" key))
				   (:font 2 " (Init Keyword)")))
			    (sys:flavor-init-keywords
			      (get flavor 'si:flavor)))
			  (mapcar
			    #'(lambda (key)
				`((:font 1 ,(format nil "~S" key))
				  (:font 2 " (IV)")))
			    (mapcar
			      #'first
			      (sys:flavor-inittable-instance-variables
				(get flavor 'si:flavor)))))
			#'(lambda (x y) (string-lessp (third (first x))
						      (third (first y)))))))
	(if keys
	    (loop for key in keys collect
		  `((,*space-format* ,(+ 2 print-level)) ,@key))
	   `(((,*space-format* ,(+ 2 print-level)) (:Font 2 "None")))))
      nil))

(DEFUN collect-dependent-flavors-and-init-keywords (flavor-name print-level all-components pointer)
  1;;Returns a list of text scroll window items which describe in detail FLAVOR-NAME's*
  1;;makeup, including its mixins, its mixins mixins, and any included flavors.  Any mixins or*
  1;;included flavors which occur redundantly are flagged as such.*
  1;; Also sets two specials (redundant-included-flavor? and redundant-mixin-flavor? to non-nil if*
  1;;at least one redundant included flavor or one mixin flavor was encountered, respectively.*
  1;;PRINT-LEVEL determines indentation level*
  1;;for text scroll window items.  ALL-COMPONENTS is the structure of FLAVOR-NAME returned by FLAVOR-COMPONENTS.*
  1;;POINTER is a pointer into ALL-COMPONENTS, indicating the particular entry we are generating text scroll window items for.*
  1;;We do our work by calling ourself recursively; in such calls, ALL-COMPONENTS never changes, while POINTER moves about in*
  1;;ALL-COMPONENTS.*

;1;;This function modded by JPR to make sure that whenever a component is allocated it is*
;1;;done so as a show-flavor, if the flavor is defined and a show-undefined-flavor if it is not.*
  (DECLARE (SPECIAL redundant-included-flavor? redundant-mixin-flavor?))
  (LET* ((flavor (GET flavor-name 'SI:FLAVOR)))
    `(,@(LOOP FOR included-entry IN (SECOND pointer)
	      FOR included = (CAR included-entry)
	      COLLECT
	      1;;If this entry is the last included flavor and is not a mixin...*
	      (IF (AND (EQ included-entry (last-included-occurance included all-components))
		1        ;;We use this to try to find ANY mixin occurance; *
		       (NOT (first-mixin-occurance included all-components)))1 *
		  `((,*space-format* ,print-level)
		    ;1;;Modded here by JPR*
		    (:ITEM1 INSTANCE ,(if (GET included 'SI:FLAVOR)
					  (allocate-data 'show-flavor (GET included 'SI:FLAVOR))
					  (allocate-data 'show-undefined-flavor included)))
		    (:FONT 1 " (included flavor of ")
		    ;1;;Modded here by JPR*
		    (:ITEM1 INSTANCE ,(if flavor
					  (allocate-data 'show-flavor flavor)
					  (allocate-data 'show-undefined-flavor flavor-name)))
		    (:FONT 1 ")"))
		(SETQ redundant-included-flavor? T)
		`((,*space-format* ,print-level)
		  ;1;;Modded here by JPR*
		  (:ITEM1 INSTANCE ,(if (GET included 'SI:FLAVOR)
					(allocate-data 'show-flavor (GET included 'SI:FLAVOR))
					(allocate-data 'show-undefined-flavor included)))
		  (:FONT 2 "**")
		  (:FONT 1 " (included flavor of ")
		  ;1;;Modded here by JPR*
		  (:ITEM1 INSTANCE ,(if flavor
					(allocate-data 'show-flavor flavor)
					(allocate-data 'show-undefined-flavor flavor-name)))
		  (:FONT 1 ")")))
	      Append (itemise-init-keywords included print-level)
	      APPEND (collect-dependent-flavors-and-init-keywords included (+ 2 print-level) all-components included-entry))
      . ,(LOOP FOR mixin-entry IN (THIRD pointer)
	       FOR mixin = (CAR mixin-entry)
	       COLLECT
	       1;;If this entry is the first mixin...*
	       (IF (EQ mixin-entry (first-mixin-occurance mixin all-components))
		   `((,*space-format* ,print-level)
		     ;1;;Modded here by JPR*
		     (:ITEM1 INSTANCE ,(if (GET mixin 'SI:FLAVOR)
					   (allocate-data 'show-flavor (GET mixin 'SI:FLAVOR))
					   (allocate-data 'show-undefined-flavor mixin))))
		(SETQ redundant-mixin-flavor? T)
		`((,*space-format* ,print-level)
		  ;1;;Modded here by JPR*
		  (:ITEM1 INSTANCE ,(if (GET mixin 'SI:FLAVOR)
					(allocate-data 'show-flavor (GET mixin 'SI:FLAVOR))
					(allocate-data 'show-undefined-flavor mixin)))
		  (:FONT 2 "*")))
	       Append (itemise-init-keywords mixin print-level)
	       APPEND (collect-dependent-flavors-and-init-keywords mixin (+ 2 print-level) all-components mixin-entry)))))

;-------------------------------------------------------------------------------



(DEFFLAVOR show-init-keywords-sorted () (show-flavor))

(DEFMETHOD (show-init-keywords-sorted :format-concisely) (stream)
  (FORMAT stream "Init Keywords (sorted) for ~s" (SI:FLAVOR-NAME data)))

(defun make-items-for-init-keywords-1 (flavor)
  (append (mapcar #'(lambda (key) (list key :init flavor))
		  (sys:flavor-init-keywords (get flavor 'si:flavor))
	  )
	  (mapcar #'(lambda (key) (list key :IV flavor))
		  (mapcar #'first
			  (sys:flavor-inittable-instance-variables
			    (get flavor 'si:flavor)
			  )
		  )
	  )
  )
)

(defun make-items-for-init-keywords (flavors)
  (let ((elements (sortcar
		    (apply #'append
			   (mapcar #'make-items-for-init-keywords-1 flavors)
		    )
		    #'string-lessp
		  )
	)
       )
       (mapcar #'(lambda (element)
		   `((:Font 1 ,(format nil "~S" (first element)))
		     (:Font 2 ,(if (equal :IV (second element))
				   " IV of "
				   " Init keyword of "
			       )
		     )
		     (:Item1 Instance
			     ,(allocate-data 'show-flavor
					     (get (third element) 'si:flavor)
			      )
		     )
		    )
		 )
	         elements
       )
  )
)

(DEFMETHOD (SHOW-INIT-KEYWORDS-sorted :GENERATE-ITEM) ()
  (LET* ((FLAVOR-NAME (SI:FLAVOR-NAME DATA))
	 (ALL-COMPONENTS (get-FLAVOR-DEPENDS-ON-ALL data))
	 TEXT-ITEMS
	 REDUNDANT-INCLUDED-FLAVOR?
	 REDUNDANT-MIXIN-FLAVOR?)
    (DECLARE (SPECIAL REDUNDANT-INCLUDED-FLAVOR? REDUNDANT-MIXIN-FLAVOR?))
    (SETQ TEXT-ITEMS
	  (make-items-for-init-keywords (cons flavor-name all-components)))
    (VALUES
     `(,*BLANK-LINE-ITEM*
       ,@(OR TEXT-ITEMS *NO-ITEMS*))
     ;;Make the label display the flavor name.
     `(:FONT FONTS:HL12BI :STRING ,(FORMAT NIL "~s" FLAVOR-NAME)))))  

;-------------------------------------------------------------------------------



(defparameter sys:*method-im-looking-for* nil)
(defparameter sys:*method-composition* nil)

sys:
(defun sys:uniqueise (list)
"Takes a list and returns a list which has no duplicates."
  (let ((temp-list list))
       (loop for element in temp-list
	     do (setq temp-list (rest temp-list))
	     when (not (member element temp-list))
	     collect element into result
	     finally (return result)
       )
  )
)

sys:
(defun sys:unwind-method-order (combination)
  (if combination
      (if (consp combination)
	  (if (equal :method (first combination))
	      combination
	      (let ((result
		      (uniqueise
			(remove nil (mapcar #'unwind-method-order combination)))
		    )
		   )
		   (if (equal 1 (length result))
		       (first result)
		       result
		   )
	      )
	  )
	  nil
      )
  )
)

(defun post-process-combined-flavors-method (form &optional (indent 0))
"Takes a form that (when ground) would appear as follows:
   (progn (sys:%apply-method before method 1)
          (sys:%apply-method before method 2)
          ...
          (sys:%apply-method before method n)
          (multiple-value-prog1
            (sys:%apply-method primary method)
            (sys:%apply-method after method 1)
            (sys:%apply-method after method 2)
            ...
            (sys:%apply-method after method n)))

 and transforms it into an inspector compatible item list that will
 look like:
   (progn <before method 1>
          <before method 2>
          ...
          <before method n>
          (multiple-value-prog1
            <primary method>
            <after method 1>
            <after method 2>
            ...
            <after method n>))

 where the things labeled <...> will appear as mouse-sensitive items of the
 type show-clos-method.m  The grindingf that this function does is very
 primitive.  Sub-lists are started on new lines indented by a few spaces
 (*method-combination-indent-increment*).
"
  (declare (special *this-line* *all-lines*))
  (if (consp form)
      (if (member (first form)
		  '(sys:funcall-with-mapping-table-internal
		    sys:lexpr-funcall-with-mapping-table-internal
		    :Method
		   )
	  )
	  (1multiple-value-bind* (meth form args)
	      (1ecase* (1first* form)
		(sys:funcall-with-mapping-table-internal
		 (1values* (1second* (second form))
			  (1cons* (1second* (second form)) (1nthcdr* 4 form))
			  (1nthcdr* 4 form)
		 )
		)
		(sys:lexpr-funcall-with-mapping-table-internal
		 (1values* (1second* (second form))
		  (1cons* '1apply* (1cons* (1second* (second form)) (1nthcdr* 4 form)))
		  (1nthcdr* 4 form)
		 )
		)
		(:Method (1values* form nil nil))
	      )
	       (push (reverse *this-line*) *all-lines*)
	       (setq *this-line* nil)
	       (push `(,tv:*space-format* ,indent) *this-line*)
	       (1if* meth
		  (1progn* (push '(:font 1 "[") *this-line*)
			  (1push* `(:item1 tv:instance
					 ,(apply #'allocate-data
						 'show-method-details
						 (Data-From-Method meth)
					  )
					  tv:print-unpadded-method
				  )
				 *this-line*
			  )
		   )
	           (1progn* (push '(:font 1 "(") *this-line*)
			   (push `(:font 1
				   ,(format nil "~S"
				     (function-name
				       (fdefinition (second (second form)))
				     )
				    )
				  )
				 *this-line*
			   )
		    )
		 )
		 (1loop* for x in args
		       for tail on args
		       do (push '(:font 1 " ") *this-line*)
		          (post-process-combined-flavors-method
			    x
			    (+ *method-combination-indent-increment* indent)
			  )
		 )
		 (1if* meth
		     (push '(:font 1 "]") *this-line*)
		     (push '(:font 1 ")") *this-line*)
		 )
	  )
	  (1case* (1first* form)
	    (sys:compile-time-remember-mapping-table
	     (2Post-Process-Combined-Flavors-Method*
	       `(1progn* ,@(1rest* (1rest* form)))
	       indent
	     )
	    )
	    (otherwise
	      (progn (push (reverse *this-line*) *all-lines*)
		     (setq *this-line* nil)
		     (push `(,tv:*space-format* ,indent) *this-line*)
		     (push '(:font 1 "(") *this-line*)
		     (1loop* for x in form
			   for tail on form
			   do (post-process-combined-flavors-method
				x
				(+ *method-combination-indent-increment* indent)
			      )
			   when (1rest* tail)
			   do (push '(:font 1 " ") *this-line*)
		     )
		     (push '(:font 1 ")") *this-line*)
	      )
	    )
	  )
      )
      (if form (push `(:font 1 ,(symbol-name form)) *this-line*) nil)
  )
)

sys:
(defun sys:new-make-combined-method (fl magic-list-entry form &aux fspec wrappers wrapper-sxhashes)
  ;;  9/16/85 DNG - Modified to use new function OPTIMIZE-METHOD-BODY-AND-ARGS.
  (if (equal (first magic-list-entry) *method-im-looking-for*)
      (progn
  (setq form `(compile-time-remember-mapping-table ,(flavor-name fl) ,form))
  ;; Get the function spec which will name the combined-method
  (setq fspec `(:method ,(flavor-name fl) :combined ,(car magic-list-entry)))
  ;; Put the wrappers and :AROUND methods around the form.
  ;; The base-flavor wrapper goes on the inside.
  (setq wrappers
	(append (get-specially-combined-methods magic-list-entry fl)
		(get-inverse-specially-combined-methods magic-list-entry fl)))
  (do ((wr wrappers (cdr wr))
       (last-method-type nil))
      ((null wr))
    (let ((method (car wr)))
	  ;; Record sxhash of each wrapper that goes in.
	  ;; This way we can tell if the combined method is obsolete when fasloaded.
      (when (and (member (caddr method) '(:wrapper :inverse-wrapper) :test #'eq)
	  (fboundp 'compiler:expr-sxhash))
	(push (list method (compiler:expr-sxhash method)) wrapper-sxhashes))
      (setq form
	    (funcall
	     (cadr
	      (or (assoc (caddr method) *specially-combined-method-types* :test #'eq)
		 (assoc (caddr method) *inverse-specially-combined-method-types* :test #'eq)))
	     fl last-method-type method form))
      (setq last-method-type (caddr method))))
  ;; Remember that it's going to be there, for HAVE-COMBINED-METHOD
  (flavor-notice-method fspec)
  (when *just-compiling*
    (function-spec-putprop fspec magic-list-entry 'combined-method-derivation))
  ;; Compile the function.  It will be inserted into the flavor's tables either
  ;; now or when the QFASL file is loaded.
 (setq *method-composition* form))
      nil)
  fspec)


sys:
(defun sys:new-compose-method-combination (fl &optional (single-operation nil) &aux tem magic-list order msg elem ffl handlers pl
  (default-cons-area *flavor-area*))
  (if (flavor-get fl :alias-flavor)
    (ferror () "Attempt to compose methods of ~S, an alias flavor." (flavor-name fl)))
  ;; If we are doing wholesale method composition,
  ;; compose the flavor bindings list also.
  ;; This way it is done often enough, but not at every defmethod.
  (or single-operation *just-compiling* (flavor-get fl :abstract-flavor)
     (progn
       (compose-flavor-bindings fl)
       (compose-flavor-initializations fl)))
  ;; Look through all the flavors depended upon and collect the following:
  ;; A list of all the operations handled and all the methods for each, called MAGIC-LIST.
  ;; The default handler for unknown operations.
  ;; The declared order of entries in the select-method alist.
  ;; Also generate any automatically-created methods not already present.
  ;; MAGIC-LIST is roughly the same format as the flavor-method-table, see its comments.
  ;; Each magic-list entry is (message comb-type comb-order (type function-spec...)...)
  (do ((ffls (flavor-depends-on-all fl) (cdr ffls)))
      ((null ffls))
    (setq ffl (compilation-flavor (car ffls))
	  pl (locf (flavor-plist ffl)))
    (cond
      ((not single-operation)
       (and (setq tem (get pl :select-method-order)) (setq order (nconc order (copy-list tem))))))
    ;; Add data from flavor method-table to magic-list
    ;; But skip over combined methods, they are not relevant here
    (dolist (mte (flavor-method-table ffl))
      (setq msg (car mte))
      (cond
	((or (not single-operation) (eq msg single-operation))
	 ;; Well, we're supposed to concern ourselves with this operation
	 (setq elem (assoc msg magic-list :test #'eq));What we already know about it
	 (cond
	   ((dolist (meth (cdddr mte))
	      (or (eq (meth-method-type meth) :combined) (not (meth-definedp meth)) (return t)))
	    ;; OK, this flavor really contributes to handling this operation
	    (or elem (push (setq elem (list* msg () () ())) magic-list))
	    ;; For each non-combined method for this operation, add it to the front
	    ;; of the magic-list element, thus they are in base-flavor-first order.
	    (dolist (meth (cdddr mte))
	      (let ((type (meth-method-type meth)))
		(cond
		  ((eq type :combined))
		  ((not (meth-definedp meth)))
		  ((not (setq tem (assoc type (cdddr elem) :test #'eq)))
		   (push (list type (meth-function-spec meth)) (cdddr elem)))
		  ((not (member (meth-function-spec meth) (cdr tem) :test #'eq))
		   (push (meth-function-spec meth) (cdr tem))))))))
	 ;; Pick up method-combination declarations
	 (and (cadr mte)
	    (cadr elem);If both specify combination-type, check
	    ;;;PHD 2/11/86 Fixed bug about some method-combinations being equal but
	    ;;; not eq, changed neq to not equal .
	    (or (neq (cadr mte) (cadr elem)) (not (equal (caddr mte) (caddr elem))))
	    (ferror () "Method-combination mismatch ~S-~S vs. ~S-~S, check your DEFFLAVOR's"
		    (cadr mte) (caddr mte) (cadr elem) (caddr elem)))
	 (cond
	   ((cadr mte);Save combination-type when specified
	    (or elem (push (setq elem (list* msg () () ())) magic-list))
	    (setf (cadr elem) (cadr mte)) (setf (caddr elem) (caddr mte))))))))
  ;; This NREVERSE tends to put base-flavor methods last
  (dolist (msg (nreverse order))
  ;; Re-order the magic-list according to any declared required order
    (and (setq tem (assoc msg magic-list :test #'eq))
       (setq magic-list (cons tem (delete tem (the list magic-list) :count 1 :test #'eq)))))
  ;; Map over the magic-list.  For each entry call the appropriate method-combining
  ;; routine, which will return a function spec for the handler to use for this operation.
  (dolist (mte magic-list)
   ;; Punt if there are no methods at all (just a method-combination declaration)
    (Cond
      ((cdddr mte)
       ;; Process the :DEFAULT methods; if there are any untyped methods the
       ;; default methods go away, otherwise they become untyped methods.
       (and (setq tem (assoc :default (cdddr mte) :test #'eq))
	  (if (assoc () (cdddr mte) :test #'eq)
	    (setf (cdddr mte) (delete tem (the list (cdddr mte)) :test #'eq))
	    (rplaca tem ())))
       (or (setq tem (get (or (cadr mte) :daemon) 'method-combination))
	  (ferror () "~S unknown method combination type for ~S operation" (cadr mte) (car mte)))
       (push (funcall tem fl mte) handlers))
      (t (setq magic-list (delete mte (the list magic-list) :count 1 :test #'eq))))))

(defparameter sys:*flavor-inspecting* nil)

(let ((compiler:compile-encapsulations-flag t))
     (advise (:property :daemon sys:method-combination) :around
	     :catch-errors nil
       (if sys:*flavor-inspecting*
	   (catch-error :do-it nil)
	   :do-it
       )
     )
)

sys:
(defun sys:flavor-combination-order (flavor op)
  (let ((old-map #'map-over-component-flavors)
	(*method-im-looking-for* op)
	(*method-composition* nil)
	(sys:*flavor-inspecting* t)
	(flavor-object (1if* (1typep* flavor 'flavor)
			   flavor
			   (get flavor 'flavor)
		       )
	)
       )
       (letf ((#'map-over-component-flavors
	       #'(lambda (&rest args)
		   (if (typep (fifth args) 'flavor)
		       (apply old-map args)
		       nil
		   )
		 )
	      )
	      (#'make-combined-method 'new-make-combined-method)
	      (#'have-combined-method #'(lambda (&rest ignore) nil))
	     )
	     (sys:new-compose-method-combination flavor-object)
	     (1values* (sys:unwind-method-order (1copy-list* *method-composition*))
		     (let ((tv:*this-line* nil)
			   (tv:*all-lines* nil)
			  )
			  (declare (special tv:*this-line* tv:*all-lines*))
			  (tv:post-process-combined-flavors-method
			    (1copy-list* *method-composition*)
			  )
			  (push (reverse tv:*this-line*) tv:*all-lines*)
			  (let ((result (reverse (remove nil tv:*all-lines*))))
			       result
			  )
		     )
	     )
       )
  )
)

;-------------------------------------------------------------------------------

(defparameter *method-operations-menu*
	'(("Inspect" :VALUE :INSPECT
	   :DOCUMENTATION "Show information about this method:
instance variables and messages referenced, arglist, documentation, source file")
	  ("Show Combined Methods" :VALUE :SHOW-COMBINED-METHODS
	   :DOCUMENTATION "Show the combined methods used in handling the message for this flavor")
	  ("Disassemble" :VALUE :DISASSEMBLE
	   :DOCUMENTATION "Use a standard Inspect window to show disassembled code.")
	  ("Edit Source" :VALUE :EDIT-SOURCE
	   :DOCUMENTATION "Edit this method in a Zmacs buffer.")
	  ("Show Call Tree" :VALUE :Show-Call-Tree
	   :DOCUMENTATION "Show the method's call tree.")
	  ("Show Related Methods" :VALUE :Show-Related-Methods
	   :DOCUMENTATION "Show method related to this one.")
	  ("Trace" :VALUE :TRACE
	   :DOCUMENTATION "Invoke a trace window to trace this method")))

(DEFUN SELECT-METHOD-OPERATIONS (METHOD-TABLE-ENTRY FLAVOR-INSPECTOR CURRENT-FLAVOR &OPTIONAL SELECTION)
  (CASE (OR SELECTION
      (UCL::SMART-MENU-CHOOSE *METHOD-OPERATIONS-MENU* :LABEL
	 (FORMAT () "~s" (CDAR METHOD-TABLE-ENTRY))))
    (:INSPECT
     (SEND FLAVOR-INSPECTOR :INSPECT-THING 'SHOW-METHOD-DETAILS
	(GET (SECOND (CAR METHOD-TABLE-ENTRY)) 'SI:FLAVOR) METHOD-TABLE-ENTRY))
    (:SHOW-COMBINED-METHODS
     (LET ((FOUND-ENTRY
	    (ASSOC (OR (FOURTH (CAR METHOD-TABLE-ENTRY)) (THIRD (CAR METHOD-TABLE-ENTRY)))
		   (SI:FLAVOR-METHOD-TABLE CURRENT-FLAVOR) :TEST #'EQ)))
	   ;1;Use the inspection pane's currently selected flavor (if there is one), instead of the one we are a method for,*
	   ;1;since that's what we want when we are displayed as an inherited method in SHOW-ALL-METHODS.*
       (IF FOUND-ENTRY
	 (SEND FLAVOR-INSPECTOR :INSPECT-THING 'SHOW-MESSAGE-HANDLERS CURRENT-FLAVOR
	    (CAR (FOURTH FOUND-ENTRY)))
	 (SEND FLAVOR-INSPECTOR :INSPECT-THING 'SHOW-MESSAGE-HANDLERS
	    (GET (SECOND (CAR METHOD-TABLE-ENTRY)) 'SI:FLAVOR) (CAR METHOD-TABLE-ENTRY)))))
    (:DISASSEMBLE (INSPECT (FDEFINITION (CAR METHOD-TABLE-ENTRY))))
    (:show-call-tree (send flavor-inspector :inspect-thing 'show-method-call-tree METHOD-TABLE-ENTRY CURRENT-FLAVOR)) ;;; I.e. wrt current flavor.
    (:show-related-methods (send flavor-inspector :inspect-thing 'show-related-methods METHOD-TABLE-ENTRY))
    (:EDIT-SOURCE (ED (CAR METHOD-TABLE-ENTRY)))
    (:TRACE (TRACE-VIA-MENUS (CAR METHOD-TABLE-ENTRY)))))


(DEFFLAVOR show-method-call-tree () (auxiliary-data-mixin inspection-data))

(DEFMETHOD (show-method-call-tree :format-concisely) (stream)
  (FORMAT stream "Call tree for method key ~ starting at ~"
	  (1let* ((key (first (last (SI:METH-FUNCTION-SPEC DATA)))))
	      (1list* key t key))
	  (1list* aux-data
	       t
	       (1if* (1typep* aux-data 'sys:flavor)
		  (sys:flavor-name aux-data)
		  aux-data
	       )
	  )))


(defun make-call-tree-items (tree print-level)
  (if tree
      (if (equal :method (first tree))
	 `(((,*space-format* ,(+ 2 print-level))
	    (:item1 instance
		   ,(allocate-data
		      'show-method-details
		      (get (second tree) 'sys:flavor)
		      (send (find-flavor-inspectable-object tree) :aux-data)
		    )
	    )
	   )
	  )
	  (apply #'append (mapcar
			    #'(lambda (x)
				(make-call-tree-items x (+ 2 print-level)))
			    tree
			  )
	  )
      )
      nil
  )
)


(DEFMETHOD (show-method-call-tree :handle-mouse-click) (blip flavor-inspector)
  (SELECTOR (FOURTH blip) =
      (#\MOUSE-L-1 (SEND flavor-inspector :inspect-info-left-click))
      (#\MOUSE-L-1 (SEND flavor-inspector :inspect-info-left-click))
      (t (beep))))

(DEFMETHOD (show-method-call-tree :who-line-doc) (IGNORE &OPTIONAL IGNORE)
  '(:MOUSE-L-1 "Inspect method"))


(DEFMETHOD (show-method-call-tree :GENERATE-ITEM) ()
  (LET ((FUN-SPEC (SI:METH-FUNCTION-SPEC DATA)))
       (1let* ((flavor (1if* (1typep* (second fun-spec) 'sys:flavor)
			 (second fun-spec)
			 (get (second fun-spec) 'sys:flavor)
		      )
	      )
             )
	     (1multiple-value-bind* (tree1 tree2)
		 (sys:flavor-combination-order aux-data (first (last fun-spec)))
	       (let ((items (1append* (make-call-tree-items tree1 1)
				     (1list* *blank-line-item*)
				     tree2
			    )
		     )
		    )
		    (VALUES
		     `(,*BLANK-LINE-ITEM*
		       ((:FONT 1 "Method call tree from ")
			(:ITEM1 INSTANCE
				,(ALLOCATE-DATA 'show-method-details
						flavor DATA))
			(:FONT 1 " for flavor ")
			(:ITEM1 INSTANCE ,(ALLOCATE-DATA 'show-flavor aux-data))
		       )
		       ,*BLANK-LINE-ITEM*
		       ,@items)
		     `(:FONT FONTS:HL12BI :String
			     ,(FORMAT () "Method call tree for ~s ~s"
				      (sys:flavor-name flavor)
				      (1first* (1last* fun-spec))))))))))


;-------------------------------------------------------------------------------


(DEFUN collect-dependent-flavors (flavor-name print-level all-components pointer)
  1;;Returns a list of text scroll window items which describe in detail FLAVOR-NAME's*
  1;;makeup, including its mixins, its mixins mixins, and any included flavors.  Any mixins or*
  1;;included flavors which occur redundantly are flagged as such.*
  1;; Also sets two specials (redundant-included-flavor? and redundant-mixin-flavor? to non-nil if*
  1;;at least one redundant included flavor or one mixin flavor was encountered, respectively.*
  1;;PRINT-LEVEL determines indentation level*
  1;;for text scroll window items.  ALL-COMPONENTS is the structure of FLAVOR-NAME returned by FLAVOR-COMPONENTS.*
  1;;POINTER is a pointer into ALL-COMPONENTS, indicating the particular entry we are generating text scroll window items for.*
  1;;We do our work by calling ourself recursively; in such calls, ALL-COMPONENTS never changes, while POINTER moves about in*
  1;;ALL-COMPONENTS.*

;1;;This function modded by JPR to make sure that whenever a component is allocated it is*
;1;;done so as a show-flavor, if the flavor is defined and a show-undefined-flavor if it is not.*
  (DECLARE (SPECIAL redundant-included-flavor? redundant-mixin-flavor?))
  (LET* ((flavor (GET flavor-name 'SI:FLAVOR)))
    `(,@(LOOP FOR included-entry IN (SECOND pointer)
	      FOR included = (CAR included-entry)
	      COLLECT
	      1;;If this entry is the last included flavor and is not a mixin...*
	      (IF (AND (EQ included-entry (last-included-occurance included all-components))
		1        ;;We use this to try to find ANY mixin occurance; *
		       (NOT (first-mixin-occurance included all-components)))1 *
		  `((,*space-format* ,print-level)
		    ;1;;Modded here by JPR*
		    (:ITEM1 INSTANCE ,(if (GET included 'SI:FLAVOR)
					  (allocate-data 'show-flavor (GET included 'SI:FLAVOR))
					  (allocate-data 'show-undefined-flavor included)))
		    (:FONT 1 " (included flavor of ")
		    ;1;;Modded here by JPR*
		    (:ITEM1 INSTANCE ,(if flavor
					  (allocate-data 'show-flavor flavor)
					  (allocate-data 'show-undefined-flavor flavor-name)))
		    (:FONT 1 ")"))
		(SETQ redundant-included-flavor? T)
		`((,*space-format* ,print-level)
		  ;1;;Modded here by JPR*
		  (:ITEM1 INSTANCE ,(if (GET included 'SI:FLAVOR)
					(allocate-data 'show-flavor (GET included 'SI:FLAVOR))
					(allocate-data 'show-undefined-flavor included)))
		  (:FONT 2 "**")
		  (:FONT 1 " (included flavor of ")
		  ;1;;Modded here by JPR*
		  (:ITEM1 INSTANCE ,(if flavor
					(allocate-data 'show-flavor flavor)
					(allocate-data 'show-undefined-flavor flavor-name)))
		  (:FONT 1 ")")))
	      APPEND (collect-dependent-flavors included (+ 2 print-level) all-components included-entry))
      . ,(LOOP FOR mixin-entry IN (THIRD pointer)
	       FOR mixin = (CAR mixin-entry)
	       COLLECT
	       1;;If this entry is the first mixin...*
	       (IF (EQ mixin-entry (first-mixin-occurance mixin all-components))
		   `((,*space-format* ,print-level)
		     ;1;;Modded here by JPR*
		     (:ITEM1 INSTANCE ,(if (GET mixin 'SI:FLAVOR)
					   (allocate-data 'show-flavor (GET mixin 'SI:FLAVOR))
					   (allocate-data 'show-undefined-flavor mixin))))
		(SETQ redundant-mixin-flavor? T)
		`((,*space-format* ,print-level)
		  ;1;;Modded here by JPR*
		  (:ITEM1 INSTANCE ,(if (GET mixin 'SI:FLAVOR)
					(allocate-data 'show-flavor (GET mixin 'SI:FLAVOR))
					(allocate-data 'show-undefined-flavor mixin)))
		  (:FONT 2 "*")))
	       APPEND (collect-dependent-flavors mixin (+ 2 print-level) all-components mixin-entry)))))



;1;;The following code is included/changed from the original to stop the flavor inspector from*
;1;;composing any uncomposed flavors.  If a flavor has any undefined components and the*
;1;;system tries to compose it then it bombs out.*


;;; Pinched from map-over-files.
(defun map-with-args (some-function over-a-list &Rest other-arguments)
"This function is much like mapcar only more useful.  It takes a function and a
 list to map the function over and an &Rest arguments feature.  It applies the
 function to each element in the list, with the element being the first
 argument and any subsequent arguments being taken from the &Rest paremeter. 
 The value of a call to this function is a list of values from this function
 call, one element for each element in the source list.
"
  (Loop For element In over-a-list
        Collect (apply some-function element other-arguments)
  )
)


(defun uniqueise (list)
"2Takes a list and returns a list which has no duplicates.*"
  (let ((temp-list list))
       (loop for element in temp-list
	     do (setq temp-list (rest temp-list))
	     when (not (member element temp-list :Test #'Equalp))
	     collect element into result
	     finally (return result)
       )
  )
)


(defun get-local-defined-components (flavor)
"2Given a flavor structure it returns a list of the flavor structures that
represent its defined local components and locally defined included flavors.*"
  (let ((local-components (append (si:flavor-depends-on flavor) (si:flavor-includes flavor))))
       (let ((defined-components (delete nil (map-with-args #'get local-components 'si:flavor))))
            (values local-components defined-components)
       )
  )
)


(defun get-flavor-depends-on-all (flavor)
"2Given a flavor structure it returns a list of all of the names of defined
components of that flavor, including itself.*"
  (multiple-value-bind (local-components defined-components) (get-local-defined-components flavor)
    (ignore local-components)
    (cons (si:flavor-name flavor)
	  (uniqueise (append (mapcar #'si:flavor-name defined-components)
			     (apply #'append (mapcar #'get-flavor-depends-on-all defined-components))
		     )
	  )
    )
  )
)


(defun get-flavor-depends-on-all-even-undefined (flavor)
"2Given a flavor structure it returns a list of all of the names of all of the
components of that flavor, including itself, whether defined or not.*"
  (multiple-value-bind (local-components defined-components) (get-local-defined-components flavor)
    (cons (si:flavor-name flavor)
	  (uniqueise (append local-components
			     (apply #'append (mapcar #'get-flavor-depends-on-all-even-undefined defined-components))
		     )
	  )
    )
  )
)


(defun get-all-components (flavor)
"2Given a flavor structure it returns a list of the flavor structures of
all of the defined components of that flavor, excluding itself.*"
  (multiple-value-bind (local-components defined-components) (get-local-defined-components flavor)
    (ignore local-components)
    (append defined-components (apply #'append (mapcar #'get-all-components defined-components)))
  )
)


;1---------------------------------------------------------------------------------------------------------------*

(DEFMETHOD (show-instance-variables :generate-item) ()
  1;;This makes sure that the flavor's mixin's instance variables have been included into the flavor.*
  ;1;Modded here by JPR.*
;  (UNLESS (SI:FLAVOR-DEPENDS-ON-ALL data)
;    (SI:COMPOSE-FLAVOR-COMBINATION data))
  (LET* ((flavor-name (SI:FLAVOR-NAME data))
	 (instance-size (SI:FLAVOR-INSTANCE-SIZE data)))
    (VALUES
      `(,*blank-line-item*
	((:FONT 1 "Local to flavor ")
	 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
	 (:FONT 1 ":"))
	,@(IF (SI:FLAVOR-LOCAL-INSTANCE-VARIABLES data)
	      (local-instance-var-item-list (SI:FLAVOR-LOCAL-INSTANCE-VARIABLES data) data data)
	      *no-items*)
	;1;Modded here by JPR.*
	. ,(LOOP FOR fname IN (CDR (get-FLAVOR-DEPENDS-ON-ALL data))
		 FOR fl = (GET fname 'SI:FLAVOR)
		 FOR local-vars = (SI:FLAVOR-LOCAL-INSTANCE-VARIABLES fl)
		 APPEND
		 (WHEN local-vars
		   `(,*blank-line-item*
		     ((:FONT 1 "Inherited from ")
		      (:ITEM1 INSTANCE ,(allocate-data 'show-flavor fl))
		      (:FONT 1 ":"))
		     . ,(local-instance-var-item-list local-vars fl data)))))
      `(:FONT fonts:hl12bi :STRING ,(FORMAT NIL "~S's ~@[~s ~]instance variables" flavor-name instance-size)))))



;1---------------------------------------------------------------------------------------------------------------*

(DEFMETHOD (show-component-flavors :generate-item) ()
  1;;This differs from SHOW-FLAVOR's in that it just displays a flat list of the component flavors.*
  1;;Included and redundant flavors are not tagged.  Doesn't show the flavor heirarchy--just the result.*
  1;;This is easier to read, but less informational.*
  ;1;Modded here by JPR.*
;  (UNLESS (SI:FLAVOR-DEPENDS-ON-ALL data)
;    (SI:COMPOSE-FLAVOR-COMBINATION data))
  (LET* ((flavor-name (SI:FLAVOR-NAME data))
  ;1;Modded here by JPR.*
	 (component-flavors (CDR (get-flavor-depends-on-all-even-undefined data))))
    (VALUES
      `(,*blank-line-item*
	("Component flavors of flavor "
	 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
	 ":")
	,*blank-line-item*
	. ,(OR (LOOP FOR component IN component-flavors
		     ;1;Modded here by JPR.*
		     COLLECT `(,*one-space-item* (:ITEM1 INSTANCE ,(if (GET component 'SI:FLAVOR)
								       (allocate-data 'show-flavor (GET component 'SI:FLAVOR))
								       (allocate-data 'show-undefined-flavor component)))))
	     *no-items*))
      `(:FONT fonts:hl12bi :STRING ,(FORMAT NIL "~s's component flavors" flavor-name)))))



;1---------------------------------------------------------------------------------------------------------------*

(DEFMETHOD (show-all-methods :generate-item) ()
  (VALUES
    1;;This method borrows alot from SHOW-LOCAL-METHODS :GENERATE-ITEM.*
    1;;NCONC for efficiency; copy-list to protect last element, which is often *NO-ITEMS*, which must not be destroyed.*
    (NCONC
      (COPY-LIST (SEND (allocate-data 'show-local-methods data) :generate-item))
      (LOOP WITH top-flavor-method-table = (SI:FLAVOR-METHOD-TABLE data)
	    ;1;Modded here by JPR.*
	    FOR flavor-name IN (CDR (get-FLAVOR-DEPENDS-ON-ALL data))
	    FOR flavor = (GET flavor-name 'SI:FLAVOR)
	    FOR method-table = (SORTCAR (COPY-LIST (SI:FLAVOR-METHOD-TABLE flavor)) #'STRING-LESSP)
	    FOR get-methods = (get-methods flavor)
	    FOR set-methods = (set-methods flavor)
	    NCONC (MULTIPLE-VALUE-BIND (items special-com?)
		      (collect-method-items
			(m tpl method-table)
			(AND (NOT (OR (MEMber (CAR m) get-methods :Test #'Eq)
				      (MEMber (CAR m) set-methods :Test #'Eq)))
				 (OR (FOURTH (CAAR tpl))
				     (NOT (ASSoc (THIRD (CAAR tpl))
						 top-flavor-method-table
						 :Test #'Eq))
				     (fourth
				       (first (fourth (ASSoc (THIRD (CAAR tpl))
						 top-flavor-method-table
						 :Test #'Eq)))))))
		    `(,*blank-line-item*
		      ,*blank-line-item*
		      ((:FONT 1 "Methods inherited from flavor ")
		       (:ITEM1 INSTANCE ,(allocate-data 'show-flavor flavor))
		       (:FONT 1 ,(IF special-com? ".  * = special method combination type" ":")))
		      1;;Collect the methods, excluding any GET and SET methods, which we want to list separately (for readability)*
		      1;;and any base (non-demonic) methods which have been defined explicitly for DATA.  (If DATA has it's own*
		      1;;version of method :FOO, it doesn't really inherit it's mixins versions of :FOO).
2;;The ASS**oc2 isn't enough--need to check previous mixins too and take into account
;;the method combination type and order (:base-flavor-first/:base-flavor-first).*
		      ,@(IF items (CONS *method-display-columns* items) *no-items*)
		      1;;Separate the display of :GET and :SET methods from the main body of defined methods,*
		      1;;since most of these are trivial and automatically generated by the flavor system.*
		      ,@(WHEN get-methods
			  `(,*blank-line-item*
			    ((:FONT 1 " GET methods inherited from ")
			     (:ITEM1 INSTANCE ,(allocate-data 'show-flavor flavor))
			     (:FONT 1 ":"))
			    . ,(collect-method-items (m tpl method-table) (AND (MEMber (CAR m) get-methods :Test #'Eq)
									       (OR (FOURTH (CAAR tpl))
										   (NOT (ASSoc (THIRD (CAAR tpl))
											      top-flavor-method-table :Test #'Eq)))))))
		      ,@(WHEN (> (LENGTH set-methods) 1) 1;;There's always at least a :SET entry.*
			  `(,*blank-line-item*
			    ((:FONT 1 " SET methods inherited from ")
			     (:ITEM1 INSTANCE ,(allocate-data 'show-flavor flavor))
			     (:FONT 1 ":"))
			    . ,(collect-method-items (m tpl method-table) (AND (MEMber (CAR m) set-methods :Test #'Eq)
									       (OR (FOURTH (CAAR tpl))
										   (NOT (ASSoc (THIRD (CAAR tpl))
											      top-flavor-method-table :Test #'Eq)))))))))))
      `(:FONT fonts:hl12bi :STRING ,(FORMAT NIL "~s's methods (all)" (SI:FLAVOR-NAME data)))))

;1---------------------------------------------------------------------------------------------------------------*

(DEFMETHOD (show-all-methods-sorted :generate-item) ()
  ;1;Modded here by JPR.*
;  (UNLESS (SI:FLAVOR-DEPENDS-ON-ALL data)
;1 *   (SI:COMPOSE-FLAVOR-COMBINATION data))
  1;;Same as (SHOW-ALL-METHODS :generate-item), with output formatted differently.*
  (VALUES
2;;The ASS*oc2 isn't enough--need to check previous mixins too and take into account
;;the method combination type and order (:base-flavor-first/:base-flavor-first).*
    (LET* ((special-comb?)
	   ;1;Modded here by JPR.*
	   (items (LOOP FOR flavor-name IN (get-FLAVOR-DEPENDS-ON-ALL data)
			FOR flavor = (GET flavor-name 'SI:FLAVOR)
			FOR method-table = (SI:FLAVOR-METHOD-TABLE flavor)
			NCONC (MULTIPLE-VALUE-BIND (items comb?)
				  (collect-method-items (m tpl method-table) T NIL T)
				(WHEN comb?
				  (SETQ special-comb? T))
				items))))
      `(,*blank-line-item*
	((:FONT 1 "All Methods of flavor ")
	 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
	 (:FONT 1 ,(IF special-comb? ".  * = special method combination type" ":")))
	,*blank-line-item*
	,*method-display-columns-2*
	1;;This sorts the methods alphabetically by message, then by method type (if any), then by submessage (if any).*
	,@(SORT items
		#'(LAMBDA (x y)
		    (LET* ((x-method-spec (CAR (SEND (THIRD (THIRD x)) :DATA)))
			   (y-method-spec (CAR (SEND (THIRD (THIRD y)) :DATA)))
			   (x-message (si:message x-method-spec))
			   (y-message (si:message y-method-spec)))
		      (IF (EQ x-message y-message)
			  (LET ((x-method-type (si:method-type x-method-spec)))
			    (IF (AND x-method-type (EQ x-method-type (si:method-type y-method-spec)))
				(string-lessp-nil-wins (si:submessage x-method-spec) (si:submessage y-method-spec))
				(string-lessp-nil-wins x-method-type (si:method-type y-method-spec))))
			  (STRING-LESSP x-message y-message)))))))
    `(:FONT fonts:hl12bi :STRING ,(FORMAT NIL "~s's methods (all)" (SI:FLAVOR-NAME data)))))



;1---------------------------------------------------------------------------------------------------------------*

(DEFMETHOD (show-all-handled-messages :generate-item) ()
  (LET ((ops (OR (SI:FLAVOR-WHICH-OPERATIONS data)
		 (LET (messages)
		   (WITH-RECURSION ((comp) (SI:FLAVOR-NAME data))
		     ;1;Modded here by JPR.*
		     (if (GET comp 'SI:FLAVOR)
			 (progn (DOLIST (method-entry (SI:FLAVOR-METHOD-TABLE (GET comp 'SI:FLAVOR)))
				  (PUSHNEW (CAR method-entry) messages))
				(DOLIST (comp (SI:FLAVOR-INCLUDES (GET comp 'SI:FLAVOR)))
				  (RECURSE comp))
				(DOLIST (comp (SI:FLAVOR-DEPENDS-ON (GET comp 'SI:FLAVOR)))
				  (RECURSE comp))
			 )
		         nil))
		   (SORT messages #'STRING-LESSP)))))
    (VALUES
      `(,*blank-line-item*
	((:FONT 1 "Messages handled by flavor ")
	 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
	 (:FONT 1 ":"))
	,*blank-line-item*
	((:FONT 2 "Message                                     Component Flavors Providing Handlers"))
2;;This probably lists too many component flavors--those just having a :COMBINED method.
;;Plus it lists flavor X and Y where flavor X's method shadows flavor Y's method.
;;Once again, need to take method combination into account.*
	,@(LOOP FOR message IN ops
		COLLECT `((:ITEM1 INSTANCE ,(allocate-data 'show-message data message))
			  ,@(LOOP WITH first? = T
				  ;1;Modded here by JPR.*
				  FOR flavor-name IN (get-FLAVOR-DEPENDS-ON-ALL data)
				  FOR flavor = (GET flavor-name 'SI:FLAVOR)
				  WHEN (flavor-message-handler? flavor message)
				  IF first?
				     COLLECT `(:ITEM1 INSTANCE ,(allocate-data 'show-flavor flavor))
				     AND DO (SETQ first? NIL)
				  ELSE
				     COLLECT '(", ")
  				     AND COLLECT `(:ITEM1 INSTANCE ,(allocate-data 'show-flavor flavor))))))
      `(:FONT fonts:hl12bi :STRING ,(FORMAT NIL "~s's messages (all)" (SI:FLAVOR-NAME data))))))


;1---------------------------------------------------------------------------------------------------------------*



;1;;                                     Code for the implementation of the DEBUG command.*


;1---------------------------------------------------------------------------------------------------------------*


(DEFPARAMETER *FLAVOR-OPTIONS-MENU*
   '(("Instance Variables"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-INSTANCE-VARIABLES *FLAVOR-DATA*)
      :DOCUMENTATION "Inspect all instance variables of this flavor")
     ("" :NO-SELECT T)
     ("Local Methods"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-LOCAL-METHODS *FLAVOR-DATA*)
      :DOCUMENTATION "Inspect methods defined locally for this flavor")
     ("All Methods"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-ALL-METHODS *FLAVOR-DATA*)
      :DOCUMENTATION "Inspect methods defined for and inherited by this flavor")
     ("All Methods, Sorted"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-ALL-METHODS-SORTED *FLAVOR-DATA*)
      :DOCUMENTATION "Sorted version of the \" ALL METHODS\" option")
     ("All Handled Messages"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-ALL-HANDLED-MESSAGES *FLAVOR-DATA*)
      :DOCUMENTATION "Inspect all messages handled by this flavor.")
     ("" :NO-SELECT T)
     ("Init Keywords"
      :eval (send ucl:this-application :inspect-thing 'show-init-keywords *flavor-data*)
     :documentation "Show all of the init keywords accepted by this flavor and which flavors defined them.")
     ("Init Keywords, Sorted"
     :eval (send ucl:this-application :inspect-thing 'show-init-keywords-sorted
 *flavor-data*)
     :documentation "Show all of the init keywords accepted by this flavor and which flavors defined them.")
     ("" :NO-SELECT T)
     ;1;This needs better who-line doc.  How it differs from Show Flavor.*
     ("Component Flavors"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-COMPONENT-FLAVORS *FLAVOR-DATA*)
      :DOCUMENTATION "Inspect flavors which make up this flavor (non-heirarchical display).")
     ("Dependent Flavors"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-DEPENDENT-FLAVORS *FLAVOR-DATA*)
      :DOCUMENTATION "Inspect flavors which directly or indirectly depend on this flavor")
     ("" :NO-SELECT T)
     ("Miscellaneous Data"
      :EVAL (SEND UCL::THIS-APPLICATION :INSPECT-THING 'SHOW-MISCELLANEOUS-DATA *FLAVOR-DATA*)
      :DOCUMENTATION "Display miscellaneous data on this flavor")
     ("Edit"
      :EVAL (setq call-edit t)   ;1;;(ed (si:flavor-name *flavor-data*))*
      :DOCUMENTATION "Edit this flavor in a Zmacs buffer.")
     ("" :NO-SELECT T)
     ("Debug"
     :eval (send ucl:this-application :inspect-thing 'debug-flavor *flavor-data*)
     :documentation "Find inconsistencies and dangerous characteristics of this flavor (can be slow)")))


(defun simplify-iv (iv)
"2If passed an instance variable, which is either a symbol or a (symbol default) list.
It returns the symbol either way.*"
  (if (consp iv)
      (first iv)
      iv
  )
)


(defun get-simple-ivs (flavor)
"2If passed a flavor structure returns a list of symbols denoting the names of the
instance variables of that flavor.*"
  (mapcar #'simplify-iv (si:flavor-local-instance-variables flavor))
)


(defun get-ivs (flavor)
"2If passed a flavor structure this function returns values which are:
    The components of the flavor, excluding itself
    A list of all of the instance variables for all of the components.*"
  (let ((components (get-all-components flavor)))
       (values components (mapcar #'get-simple-ivs components))
  )
)


(defun symbols-match (symbol-1 symbol-2 depackage-p)
"2This is a sort of predicate, which tests whether two symbols could be said to
match.  It is passed two symbols and a flag, which determines whether the comparison
is to be made in terms of the symbols themselves (equal test) or of their pnames
 (dep*ackage2-p = t). If the symbols are to be compared in terms of their pnames then
they are said to match if they are no the same symbol and yet their pnames are
string-equal.*"
  (if depackage-p
      (and (not (equal symbol-1 symbol-2)) (string-equal (symbol-name symbol-1) (symbol-name symbol-2)))
      (equal symbol-1 symbol-2)
  )
)


(defun is-in (symbol list test depackage-p)
"2This function is passed a symbol, a list of symbols and a test to determine whether
the symbol can be said to be in the list.  If the symbol is found to be in the list
then the function returns a two-list having as its elements the symbol and the symbol
in the list with which it was matched.  The depackage-p flag is used by the test
function to determine how it is to perform the test.*
"
  (declare (optimize (safety 0)))
  (if (equal nil list)
      nil
      (if (funcall test symbol (first list) depackage-p)
	  (list symbol (first list))
	  (is-in symbol (rest list) test depackage-p)
      )
  )
)


(defun look-for-match (flavor ivs other-flavors other-ivs depackage-p)
"2This function is passed a flavor structure and its local ivs, a list of other
flavors, which are other components like the first flavor and a list of lists of
ivs in the component flavors.  It returns a list of match descriptors, one for
each instance of finding an iv in the current flavor redefined in one of the
other flavors.  Each match descriptor is a three-list.  The first element
is a two-list holding the two instance variable names for the matching names.  The
second and thrid elements are the flavor structures for the multiple defining flavor
and the multiple definee flavor.  The depackage-p argument is non-nil if the type of
multiple definition looked for is to be one of looking for clashes like tv:foo and
zwei:foo.*
"
  (declare (optimize (safety 0)))
  (if (equal other-ivs nil)
      nil
      (if (equal flavor (first other-flavors))
	  (look-for-match flavor ivs (rest other-flavors) (rest other-ivs) depackage-p)
	  (let ((matches (delete nil (map-with-args #'is-in ivs (first other-ivs) #'symbols-match depackage-p))))
	       (if matches
		   (append (map-with-args #'list matches flavor (first other-flavors))
			   (look-for-match flavor ivs (rest other-flavors) (rest other-ivs) depackage-p)
		   )
		   (look-for-match flavor ivs (rest other-flavors) (rest other-ivs) depackage-p)
	       )
	  )
      )
  )
)
	    

(defun find-multiple-definitions-1 (components ivs depackage-p result)
"2This function is passed a list of flavor structures, which are the components of the
structure in question, and a list of the instance variables, for which it is look for
multiple definitions.  It returns a list of match descriptors, one for each multiple
definition that it finds.  Each match descriptor is a three-list.  The first element
is a two-list holding the two instance variable names for the matching names.  The
second and thrid elements are the flavor structures for the multiple defining flavor
and the multiple definee flavor.  The depackage-p argument is non-nil if the type of
multiple definition looked for is to be one of looking for clashes like tv:foo and
zwei:foo.*
"
  (declare (optimize (safety 0)))
  (if (equal nil components)
      result
      (let ((matches (look-for-match (first components) (first ivs) (rest components) (rest ivs) depackage-p)))
	   (find-multiple-definitions-1 (rest components) (rest ivs) depackage-p (append matches result))
      )
  )
)


(defun find-multiple-definitions (components ivs depackage-p)
"2This function is passed a list of flavor structures, which are the components of the
structure in question, and a list of the instance variables, for which it is look for
multiple definitions.  It returns a list of match descriptors, one for each multiple
definition that it finds.  Each match descriptor is a three-list.  The first element
is a two-list holding the two instance variable names for the matching names.  The
second and thrid elements are the flavor structures for the multiple defining flavor
and the multiple definee flavor.  The depackage-p argument is non-nil if the type of
multiple definition looked for is to be one of looking for clashes like tv:foo and
zwei:foo.*
"
  (find-multiple-definitions-1 components ivs depackage-p nil)
)


;1---------------------------------------------------------------------------------------------------------------*


(defun get-primary-methods (methods)
"2Is passed a list of method specs, found in the method table of a flavor structure.
It returns a list of all of the methods, which are primary methods.  These have a
method descriptor length of three (:method foo :bar).*
"
  (remove-if-not
    #'(lambda (element)
	(and (= (length (first (fourth element))) 3)
	     (not (equal :Set (first element)))
	)
      )
      methods
  )
)


(defun recons (a b)
"2Reverse cons.*"
  (cons b a)
)


(defun associate-all-methods (method method-alist)
"2Is passed a method descriptor and a list of method descriptors.  It returns a
list of all of the method descriptors in the alist, which have the same method
name as Method.*
"
  (remove-if-not
    #'(lambda (element) (equal (first method) (first (first element))))
      method-alist
  )
)


(defun check-for-shadowing-methods (flavor)
"2This function is passed a flavor structure.  It returns a two-list.  The
first of the list is a list of all of the primary methods in itself and
its components.  The second is a list of all of the primary methods of its
components, which are shadowed by either itself or any of its components.*
"
  (let ((components (delete flavor (get-all-components flavor)))
	(primary-methods (delete :set (get-primary-methods (si:flavor-method-table flavor))))
       )
       (let ((primary-methods-and-shadows-from-components (mapcar #'check-for-shadowing-methods components)))
	    (let ((primary-methods-from-components
		    (uniqueise (apply #'append (delete nil (mapcar #'first primary-methods-and-shadows-from-components))))
		  )
		  (shadows-from-components
		    (apply #'append (delete nil (mapcar #'second primary-methods-and-shadows-from-components)))
		  )
		  (primary-methods-and-flavor (map-with-args #'list primary-methods flavor))
		 )
		 (let ((new-shadows
			 (apply #'append
				(delete nil (map-with-args #'associate-all-methods primary-methods
							   primary-methods-from-components
					    )
				)
			 )
		       )
		      )     
		      (list (append primary-methods-and-flavor primary-methods-from-components)
			    (uniqueise (append (map-with-args #'recons new-shadows flavor) shadows-from-components))
		      )
		 )
	    )
       )
  )
)


(defun get-method-name-from-shadow (shadow)
"2Given a method structure for a shadowed method, which is a three-list
whose elements are :- the shadowing flavor, the method as it is found
in the flavor's method table and the shadowed flavor, this function
returns the name of the method.*
"
  (first (second shadow))
)


(defun get-shadowed-methods (flavor)
"2Given a flavor structure this function returns a list denoting all of
the cases in which a primary method of a component of this flavor is
shadowed by some higher component.  This list is sorted so that the
method names are in alphabetical order.  Each element in the list is
a three-list with the following structure :-
 (#<shadowing-flavor> (method-data) #<shadowed-flavor>)*
"
  (sort (uniqueise (second (check-for-shadowing-methods flavor)))
	#'string-lessp :Key #'get-method-name-from-shadow
  )
)


;1---------------------------------------------------------------------------------------------------------------*


(defun collect-undefined-components (flavor)
"2Given a flavor structure this function returns a list of all of the
components of that flavor, which have not yet been defined.*"
  (multiple-value-bind (local-components defined-components) (get-local-defined-components flavor)
     (let ((result-from-components (delete nil (mapcar #'collect-undefined-components defined-components))))
	  (uniqueise (append (map-with-args #'list
					    (set-difference local-components (mapcar #'si:flavor-name defined-components))
					    flavor
			     )
			     (mapcar #'append result-from-components)
		     )
	  )
     )
  )
)


;1---------------------------------------------------------------------------------------------------------------*



(defun collect-required-things (flavor selector)
"2This is a general function which is passed a flavor and a keyword which is
the name of the property denoting things that are required by the flavor, e.g.
methods, ivs.  It returns a list of two-lists.  Each two-list contains, first
the name of the required component and second the flavor structure in which
it is required.*"
  (multiple-value-bind (local-components defined-components) (get-local-defined-components flavor)
    (ignore local-components)
    (let ((local-required-things (get (cons nil (si:flavor-plist flavor)) selector))
	  (inherited-required-things (delete nil (map-with-args #'collect-required-things defined-components selector)))
	 )
	 (append (map-with-args #'list local-required-things flavor) (apply #'append inherited-required-things))
    )
  )
)


(defun find-unsatisfied-things (flavor selector test)
"2Given a flavor structure, a keyword which denotes which sort of things
is required, e.g. :Required-Methods, and a function which is able to
determine whether the things that are required are indeed satisfied,
this function returns a list of two-lists describing all of the required
things that have not been defined.*"
  (let ((required-things (collect-required-things flavor selector)))
       (funcall test required-things flavor)
  )
)


;1---------------------------------------------------------------------------------------------------------------*


(defun filter-out-satisfied-flavors (required flavor)
"2Given a list of two-lists which describe required flavors and a
flavor structure, this function returns that subset of the
required components which is not satisfied.*"
  (ignore flavor)
  (remove-if #'(lambda (element) (get (first element) 'si:flavor))
	     required
  )
)


(defun list-of-unsatisfied-required-flavors (flavor)
"2Given a flavor structure this returns a list of two-lists
describing the unsatisfied required flavors for that flavor.*"
  (find-unsatisfied-things flavor :Required-Flavors #'filter-out-satisfied-flavors)
)


;1---------------------------------------------------------------------------------------------------------------*


(defun search-for-method (flavor method)
"2Given a flavor structure and a method name this returns T
if the method is defined by the flavor or one of its components,
otherwise it returns nil.*"
  (if (assoc method (si:flavor-method-table flavor))
      t
        (multiple-value-bind (local-components defined-components) (get-local-defined-components flavor)
	  (ignore local-components)
	  (not (equal nil (delete nil (map-with-args #'search-for-method defined-components method))))
	)
  )
)


(defun filter-out-satisfied-methods (required flavor)
"2Given a list of two-lists which describe required methods and a
flavor structure, this function returns that subset of the
required components which is not satisfied.*"
  (remove-if #'(lambda (element) (search-for-method flavor (first element)))
	     required
  )
)


(defun list-of-unsatisfied-required-Methods (flavor)
"2Given a flavor structure this returns a list of two-lists
describing the unsatisfied required methods for that flavor.*"
  (find-unsatisfied-things flavor :Required-methods #'filter-out-satisfied-methods)
)


;1---------------------------------------------------------------------------------------------------------------*


(defun search-for-iv (flavor iv)
"2Given a flavor structure and an instance variable name this returns T
if the instance variable is defined by the flavor or one of its components,
otherwise it returns nil.*"
  (if (member iv (get-simple-ivs flavor))
      t
        (multiple-value-bind (local-components defined-components) (get-local-defined-components flavor)
	  (ignore local-components)
	  (not (equal nil (delete nil (map-with-args #'search-for-iv defined-components iv))))
	)
  )
)


(defun filter-out-satisfied-ivs (required flavor)
"2Given a list of two-lists which describe required instance
variables and a flavor structure, this function returns
that subset of the required components which is not satisfied.*"
  (remove-if #'(lambda (element) (search-for-iv flavor (first element)))
	     required
  )
)


(defun list-of-unsatisfied-required-IVs (flavor)
"2Given a flavor structure this returns a list of two-lists
describing the unsatisfied required instance variables for
that flavor.*"
  (find-unsatisfied-things flavor :Required-Instance-Variables #'filter-out-satisfied-ivs)
)


;1---------------------------------------------------------------------------------------------------------------*

(defflavor debug-flavor ()
  (flavor-operation-mixin)
  (:Documentation :Special-Purpose
"A flavor used by the flavor inspector in order to represent a request be the user to
execute the debug flavor command for a given flavor.  It is somewhat like Show-Flavor."
  )
)


(DEFMETHOD (debug-flavor :format-concisely) (stream)
"2Prints the instance of Debug-Flavor in a simple manner.  This is used
to display the flavor in the flavor inspector's history window and
such like.*
"
  (FORMAT stream "~s's debug data" (SI:FLAVOR-NAME data))
)


(DEFMETHOD (debug-flavor :generate-item) ()
"2This method generates a window item for displaying in the flavor inspector
which shows debug information associated with the flavor in question.
This window item is made up of a number of window items describing the state
of affairs in detail.*
"
  1;;This makes sure that the flavor's mixin's instance variables have been included into the flavor.*
  (LET* ((flavor-name (SI:FLAVOR-NAME data))
	 (flavor data)
         (clashes (multiple-value-bind (components ivs) (get-ivs flavor)
		    (find-multiple-definitions components ivs nil)))
	 (package-clashes (multiple-value-bind (components ivs) (get-ivs flavor)
			    (find-multiple-definitions components ivs t)))
	 (shadowed-methods (get-shadowed-methods flavor))
	 (undefined-components (collect-undefined-components flavor))
	 (unsatisfied-flavors (list-of-unsatisfied-required-flavors flavor))
	 (unsatisfied-methods (list-of-unsatisfied-required-Methods flavor))
	 (unsatisfied-ivs (list-of-unsatisfied-required-IVs flavor))
	)
    (VALUES
      `(,*blank-line-item*
;1------------------------------*
	,@(if clashes
	      `(((:FONT 1 "Multiple declarations of the same instance variable for flavor ")
		 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
		 (:FONT 1 ":")))
	      nil
	 )
	,@(loop for clash in clashes
	        append `(((:FONT 3 "  ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-instance-variable (first (first clash))))
			  (:FONT 3 " in flavors ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (second clash)))
			  (:FONT 3 " in flavors ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (third clash)))))
	  )
;1------------------------------*
	,@(if package-clashes *blank-line-item*)
	,@(if package-clashes *blank-line-item*)
	,@(if package-clashes
	      `(((:FONT 1 "Declarations of instance variables with the same PName but are in different packages for flavor ")
		 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
		 (:FONT 1 ":")))
	      nil
	 )
	,@(loop for clash in package-clashes
	        append `(((:FONT 3 "  ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-instance-variable (first (first clash))))
			  (:FONT 3 " in flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (second clash)))
			  (:FONT 3 " and ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-instance-variable (second (first clash))))
			  (:FONT 3 " in flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (third clash)))))
	 )
;1------------------------------*
	,@(if undefined-components *blank-line-item*)
	,@(if undefined-components *blank-line-item*)
	,@(if undefined-components
	      `(((:FONT 1 "Undefined components of flavor ")
		 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
		 (:FONT 1 ":")))
	      nil
	 )
	,@(loop for undefined in undefined-components
	        append `(((:ITEM1 INSTANCE ,(allocate-data 'show-undefined-flavor (first undefined)))
			  (:FONT 3 " of flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (second undefined)))))
	 )
;1------------------------------*
	,@(if unsatisfied-flavors *blank-line-item*)
	,@(if unsatisfied-flavors *blank-line-item*)
	,@(if unsatisfied-flavors
	      `(((:FONT 1 "Unsatisfied required flavors of flavor ")
		 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
		 (:FONT 1 ":")))
	      nil
	 )
	,@(loop for unsatisfied in unsatisfied-flavors
	        append `(((:ITEM1 INSTANCE ,(if (get (first unsatisfied) 'si:flavor)
						(allocate-data 'show-flavor (get (first unsatisfied) 'si:flavor))
						(allocate-data 'show-undefined-flavor (first unsatisfied))))
			  (:FONT 3 " for flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (second unsatisfied)))))
	 )
;1------------------------------*
	,@(if unsatisfied-methods *blank-line-item*)
	,@(if unsatisfied-methods *blank-line-item*)
	,@(if unsatisfied-methods
	      `(((:FONT 1 "Unsatisfied required methods of flavor ")
		 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
		 (:FONT 1 ":")))
	      nil
	 )
	,@(loop for unsatisfied in unsatisfied-methods
	        append `(((:FONT 1 ,(format nil "~S" (first unsatisfied)))
			  (:FONT 3 " for flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (second unsatisfied)))))
	 )
;1------------------------------*
	,@(if unsatisfied-ivs *blank-line-item*)
	,@(if unsatisfied-ivs *blank-line-item*)
	,@(if unsatisfied-ivs
	      `(((:FONT 1 "Unsatisfied required instance variables of flavor ")
		 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
		 (:FONT 1 ":")))
	      nil
	 )
	,@(loop for unsatisfied in unsatisfied-ivs
	        append `(((:FONT 1 ,(format nil "~S" (first unsatisfied)))
			  (:FONT 3 " for flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (second unsatisfied)))))
	 )
;1------------------------------*
        ,@(if shadowed-methods *blank-line-item*)
	,@(if shadowed-methods *blank-line-item*)
	,@(if shadowed-methods
	      `(((:FONT 1 "Primary methods of component flavors shadowed by flavor ")
		 (:ITEM1 INSTANCE ,(allocate-data 'show-flavor data))
		 (:FONT 1 ":")))
	      nil
	 )
	,@(loop for shadow in shadowed-methods
	        append `(((:ITEM1 INSTANCE ,(allocate-data 'show-method (fourth (second shadow))))
			  (:FONT 3 " of flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (third shadow)))
			  (:FONT 3 " shadowed by flavor ")
			  (:ITEM1 INSTANCE ,(allocate-data 'show-flavor (first shadow)))))
	 )
       )
      `(:FONT fonts:hl12bi :STRING ,(FORMAT NIL "~S's Debug data" flavor-name)))
  )
)

;-----------------------------------------------------------------------------------------------------------------------------


(DEFCOMMAND (flavor-inspector :debug-flavor) ()
  '(:DESCRIPTION "Displays inconsistencies in a given flavor."
    :DOCUMENTATION "Displays inconsistencies and possibly dangerous aspects of a flavor
whose name is read from the user.  The following lists each condition searched for and
a description of why the condition is considered inconsistent or dangerous.

 1. Flavor has an instance variable supplied by two or more of its flavor components.

    Each component probably makes assumptions about the nature of the instance variable
    and their methods will interfere with each other.


 2. Flavor has two or more instance variables having the same name in different packages;
    for instance, TV:FOO and ZWEI:FOO.

    This is buggy, especially if both are gettable and settable.  The methods :FOO and
    :SET-FOO will only operate on one of these variables, whichever's responsible flavor
    is highest in the heirarchy of component flavors.  Yet methods of the other flavor
    will be addressing the wrong variable in their sending of :FOO and :SET-FOO messages.

 3. Flavor has undefined components.

    This is buggy only if you try to instantiate the flavor.  During development this
    may be reasonable.

 4. Flavor has unsatisfied required component flavors.

    This is buggy only if you try to instantiate the flavor.  During development this
    may be reasonable.

 5. Flavor has unsatisfied required methods.

    This is buggy only if you try to instantiate the flavor.  During development this
    may be reasonable.

 6. Flavor has unsatisfied required instance variables.

    This is buggy only if you try to instantiate the flavor.  During development this
    may be reasonable.

 7. Flavor defines a basic (non-demonic) method which is defined by one of its flavor
    components.

    This is not actually buggy; in fact, many times it is expected that a flavor specialize
    methods defined by their component flavors.  However, it is handy for you to see a
    list of all methods you have shadowed, in case you redefined a method without realizing
    it.  For instance, if your window flavor defines method :SELECT to have the user
    select something from a menu, your window won't be selectable because :SELECT is
    a low-level window method which shouldn't normally be redefined.  Using the Debug-Flavor
    option of the Flavor Inspector, you would quickly see that you have redefined an important
    method.

"
  :KEYS #\SUPER-D)
  (declare (special *flavor-data* frame))
  (SEND UCL:THIS-APPLICATION :inspect-thing 'debug-flavor *flavor-data*)
)

;-------------------------------------------------------------------------------

(defmethod (inspect-history-window :current-object) ()
  (first (first cache)))

(defmethod (inspect-history-window :toggle-lock) ()
  (beep))

;-------------------------------------------------------------------------------

;;; New patches because of bugs found by JPR.  17 Dec 87.

;-------------------------------------------------------------------------------

;;; Didn't dwimify metho names.
(DEFMETHOD (METHOD-SPECS :HANDLE-TYPEIN-P) (EXPRESSION TYPE)
  (if (not (consp expression))
      (if (and (symbolp expression) (boundp expression))
	  (if (consp (symbol-value expression))
	      (setq expression (cdr (symbol-value expression)) type ':implicit-list)
	      (setq expression (symbol-value expression)))
	  (VALUES () (FORMAT NIL "~s is not a defined flavor " EXPRESSION))))
  (WHEN (MEMBER TYPE '(:IMPLICIT-LIST CONS) :TEST #'EQ)
   ;1;Make it (:method <flavor-name> <message-name>) or (:method <flavor-name> <type> <message-name>)*
   ;1;or even (:method <flavor-name> :CASE <message-name> <submessage>)*
    (PUSH :METHOD EXPRESSION)
    (COND
     ;1;Error if no method supplied*
     ((NULL (THIRD EXPRESSION))
      (VALUES () (FORMAT () "No method supplied.  Press META- for correct syntax.")))
     ;1;Error if too many args supplied.*
     ((IF (MEMBER (THIRD EXPRESSION) *METHOD-TYPES* :TEST #'EQ)
	(IF (EQ (THIRD EXPRESSION) :CASE)
	  (SIXTH EXPRESSION)
	  (FIFTH EXPRESSION))
	(FOURTH EXPRESSION))
      (VALUES () (FORMAT () "Extraneous arguments.  Press META- for correct syntax.")))
     ;1;Good method spec => we'll :EXECUTE the expression.*
     ((FDEFINEDP EXPRESSION) (VALUES SELF ()))
     ;1;Bad method name.*
     ((GET (SECOND EXPRESSION) 'SI:FLAVOR)
      ;;; patched here by JPR.
      (let ((method
	      (zwei:symbol-from-string (format nil "~S" expression) nil t)
	    )
	   )
	   (if (consp method)
	       (progn (send self :set-documentation method) (values self nil))
	       (VALUES () (FORMAT () "~{~s ~}is not a defined method."
				  (CDR EXPRESSION)))
	   )
      )
     )
     ;1;Bad flavor name.*
     (T (VALUES () (FORMAT () "~s is not a defined flavor" (SECOND EXPRESSION)))))))

(DEFMETHOD (METHOD-SPECS :EXECUTE) (FLAVOR-INSPECTOR)
  (DECLARE (SPECIAL UCL::INHIBIT-RESULTS-PRINT?))
  (if (and (symbolp -) (boundp -))
      (setq - (cdr (symbol-value -))))
  ;;; Patched here by JPR.
  (if (consp (send self :documentation))
      (setq - (rest (send self :documentation))))
  (LET* ((HISTORY (SEND FLAVOR-INSPECTOR :HISTORY))
	 (FLAVOR (GET (CAR -) 'SI:FLAVOR))
	 (METHOD
	  (INSPECT-REAL-VALUE
	   `(:VALUE
	     ,(ALLOCATE-DATA 'SHOW-METHOD-DETAILS FLAVOR
	       (CAR
		 (CDDDR
		       (ASSOC (OR (THIRD -) (SECOND -)) (SI:FLAVOR-METHOD-TABLE FLAVOR) :TEST
			      #'EQ))))
	     ,HISTORY))))
    (INSPECT-FLUSH-FROM-HISTORY METHOD HISTORY)
    (SEND HISTORY :APPEND-ITEM METHOD)
    (UPDATE-PANES)
    ;1;We don't want our result to be printed.*
    (SETQ UCL::INHIBIT-RESULTS-PRINT? T)))

;-------------------------------------------------------------------------------

;;; Used to loop recursively if Expression was a symbol bound to nil.
(DEFMETHOD (FLAVOR-instance :HANDLE-TYPEIN-P) (EXPRESSION TYPE)
  (if (and (not (consp expression)) expression)
	   (if (not (and (symbolp expression) (boundp expression)))
	       (values () (format nil "~s is not a defined flavor 2" EXPRESSION))
	       (cond ((instancep (eval expression))
		      (IF (AND (MEMBER TYPE '(FIRST-ATOM ATOM SYMBOL) :TEST #'EQ) (SYMBOLP EXPRESSION)
			       (GET (type-of (eval EXPRESSION)) 'SI:FLAVOR))
			  (VALUES self ())
			  (VALUES () (FORMAT NIL "~s is not a defined flavor " EXPRESSION))))
		     ((and (symbolp expression) (boundp expression))
		      (send   self :handle-typein-p (symbol-value expression) (type-of (symbol-value expression))))
		     (t (values () (format NIL "~s is not a defined flavor" EXPRESSION)))))  
	   (values () (format NIL "~s is not a defined flavor" EXPRESSION))))

;;; used to screw up if expression was a symbol bound to nil,
;;; since it would look for a flavor whose name was nil.
(DEFMETHOD (FLAVOR-NAMES :HANDLE-TYPEIN-P) (EXPRESSION TYPE)
  (cond ((AND expression (MEMBER TYPE '(FIRST-ATOM ATOM SYMBOL) :TEST #'EQ) (SYMBOLP EXPRESSION)
       (GET EXPRESSION 'SI:FLAVOR))
	 (VALUES SELF ()))			
	((and (symbolp expression) (boundp expression) expression)
	    (send self :handle-typein-p (symbol-value expression) (type-of (symbol-value expression))))
	(t (VALUES () (FORMAT NIL "~s is not a defined flavor " EXPRESSION)))))

;;; used to screw up if "-" was a symbol bound to nil,
;;; since it would look for a flavor whose name was nil.
tv:
(DEFMETHOD (FLAVOR-NAMES :EXECUTE) (FLAVOR-INSPECTOR)
  (DECLARE (SPECIAL UCL::INHIBIT-RESULTS-PRINT?))
  (if (and (symbolp -) (boundp -) - (symbol-value -))
      (setq - (symbol-value -)))
  (LET* ((HISTORY (SEND FLAVOR-INSPECTOR :HISTORY))
	 (FLAVOR
	  (INSPECT-REAL-VALUE
	   `(:VALUE ,(ALLOCATE-DATA 'SHOW-FLAVOR (GET - 'SI:FLAVOR)) ,HISTORY))))
	 ;;Might not work since not EQ
    (if -
	(progn (INSPECT-FLUSH-FROM-HISTORY FLAVOR HISTORY)
	       (SEND HISTORY :APPEND-ITEM FLAVOR)))
    (UPDATE-PANES)
    ;;We don't want our result to be printed.
    (SETQ UCL::INHIBIT-RESULTS-PRINT? T)))

;;; used to screw up if "-" was a symbol bound to nil,
;;; since it would look for a flavor whose name was nil.
(DEFMETHOD (FLAVOR-instance :EXECUTE) (FLAVOR-INSPECTOR)
  (DECLARE (SPECIAL UCL::INHIBIT-RESULTS-PRINT?))
  (if (and (symbolp -) (boundp -) - (symbol-value -))
      (setq - (symbol-value -)))
  (LET* ((HISTORY (SEND FLAVOR-INSPECTOR :HISTORY)) 
	 (FLAVOR
	  (INSPECT-REAL-VALUE
		`(:VALUE ,(ALLOCATE-DATA 'SHOW-FLAVOR (GET (type-of (eval -)) 'SI:FLAVOR)) ,history))))
	 ;;Might not work since not EQ
    (INSPECT-FLUSH-FROM-HISTORY FLAVOR HISTORY)
    (SEND HISTORY :APPEND-ITEM FLAVOR)
    (UPDATE-PANES)
    ;;We don't want our result to be printed.
    (SETQ UCL::INHIBIT-RESULTS-PRINT? T)))

;;; used to screw up if expression was a symbol bound to nil,
;;; since it would look for a flavor whose name was nil.
(DEFMETHOD (FLAVOR-NAMES :HANDLE-TYPEIN-P) (EXPRESSION TYPE)
  (cond ((AND expression (MEMBER TYPE '(FIRST-ATOM ATOM SYMBOL) :TEST #'EQ) (SYMBOLP EXPRESSION)
       (GET EXPRESSION 'SI:FLAVOR))
	 (VALUES SELF ()))			
	((and (symbolp expression) (boundp expression) expression)
	    (send self :handle-typein-p (symbol-value expression) (type-of (symbol-value expression))))
	(t (VALUES () (FORMAT NIL "~s is not a defined flavor " EXPRESSION)))))

;;; All :Before :Init methods for flavors built on Basic-Inspect-Frame are
;;; supposed to have an "(unless inspectors..." test at the beginning, so that
;;; other flavors can be built on top of them.  This one didn't so things got
;;; mightily screwed up when one built on Flavor-Inspector.

(DEFMETHOD (FLAVOR-INSPECTOR :BEFORE :INIT) (PLIST)
 ;;Specify our panes and constraints.  This differs from (TV:INSPECT-FRAME :BEFORE :INIT) only 
 ;;in some of the who-line messages we provide, which are specific to flavor inspectors.
  (unless inspectors ;;; It didn't use to have this test. JPR
  (LET ((NOI (OR (GET PLIST :NUMBER-OF-INSPECTORS) 3)))
        ;(iobuf (make-default-io-buffer)))
    (SETQ PANES
	  (LIST `(INTERACTOR INSPECTOR-INTERACTION-PANE
                             :LABEL NIL
                             :MORE-P NIL
                             ;:IO-BUFFER ,IOBUFF
                             :FONT-MAP  ,(LIST (first *Inspector-Font-Map*) (second *Inspector-Font-Map*))
                             :WHO-LINE-MESSAGE
                             "To inspect a flavor, type its name.  To inspect a method, type <flavor name> <method name>
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu.")
                `(HISTORY INSPECT-HISTORY-WINDOW  ;-WITH-MARGIN-SCROLLING
;                         :SCROLL-BAR 3
                         :LINE-AREA-MOUSE-DOC (:MOUSE-L-1 "Inspect the indicated data"
                                                          :MOUSE-M-1 "Remove it from the Flavor Inspector")
                         :NORMAL-MOUSE-DOCUMENTATION (:MOUSE-L-1 "Select data to inspect"
                                                      :MOUSE-M-2 "Lock/Unlock inspector pane"
                                                      :MOUSE-R-2 "System Menu"))
                `(MENU INSPECTOR-MENU-PANE)))
    ;;Add an inspector to PANES, taking into account the number of inspector panes requested.  The first
    ;;inspector is given a typeout pane.  Also initialize INSPECTORS.
    (DOTIMES (I NOI)
      (LET ((NAME1 (INTERN (FORMAT () "INSPECTOR-~D" I) "TV")))
	(PUSH `(,NAME1 ,(IF (= I (1- NOI)) 'INSPECT-WINDOW-WITH-TYPEOUT 'INSPECT-WINDOW)
;                :SCROLL-BAR 2
;                :SCROLL-BAR-ALWAYS-DISPLAYED T
                ;;Otherwise we get "More Object Above", etc.
;                :MARGIN-SCROLL-REGIONS ((:TOP "Top" "More Above" FONTS:TR8B) (:BOTTOM "Bottom" "More Below" FONTS:TR8B))
                :CURRENT-OBJECT-WHO-LINE-MESSAGE ,(FUNCTION (LAMBDA (CURRENT-OBJECT)
                                                              (COND
                                                                ((EQUAL CURRENT-OBJECT '(NIL))
                                                                 "Flavor Inspection Pane.  To inspect a flavor, type its name.  To inspect a method, type <flavor name> <method name>
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu.")
                                                                ((TYPEP CURRENT-OBJECT 'FLAVOR-OPERATION-MIXIN)
                                                                 `(:MOUSE-L-1 "Select data to inspect"
                                                                   :MOUSE-M-1 "Help on currently displayed data"
                                                                   :MOUSE-M-2 "Lock/Unlock inspector pane"
                                                                   :MOUSE-R-1 ,(FORMAT () "Menu of operations on flavor ~s"
                                                                                       (SI:FLAVOR-NAME (SEND CURRENT-OBJECT :DATA)))))
                                                                (T '(:MOUSE-L-1 "Choose an item to inspect"))))))
              PANES)
	(PUSH NAME1 INSPECTORS)))
    (SETQ CONSTRAINTS
	  `((:THREE-PANES ,(REVERSE `(INTERACTOR MENU-HISTORY ,@INSPECTORS))
                         ((INTERACTOR 4 :LINES))
                         ((MENU-HISTORY
                            :HORIZONTAL (3 :LINES HISTORY)
                            (MENU HISTORY)
                            ((MENU :ASK :PANE-SIZE))
                            ((HISTORY :EVEN))))
                         ,(MAPCAR #'(LAMBDA (NAME1)
                                      `(,NAME1 :LIMIT (1 36 :LINES)
                                        ,(/ 0.3s0 (1- NOI)) :LINES))
                                  (CDR INSPECTORS))
                         ((,(CAR INSPECTORS) :EVEN)))
	    (:ONE-PANE (,(CAR INSPECTORS) MENU-HISTORY INTERACTOR)
                      ((INTERACTOR 4 :LINES))
                      ((MENU-HISTORY
                         :HORIZONTAL (3 :LINES HISTORY)
                         (MENU HISTORY)
                         ((MENU :ASK :PANE-SIZE))
                         ((HISTORY :EVEN))))
                      ((,(CAR INSPECTORS) :EVEN)))
              (:TWO-HORIZONTAL-PANES ,(REVERSE `(INTERACTOR MENU-HISTORY INSPECTOR-2 INSPECTOR-1))
                    ((INTERACTOR 4 :LINES))
                    ((MENU-HISTORY :HORIZONTAL (3 :LINES HISTORY) (MENU HISTORY)
                                   ((MENU :ASK :PANE-SIZE))
                                   ((HISTORY :EVEN))))
                    ((INSPECTOR-1 0.5))
                    ((INSPECTOR-2 :even)))
              (:TWO-VERTICAL-PANES ,(REVERSE `(INTERACTOR MENU-HISTORY SIDE-BY-SIDE))
                    ((INTERACTOR 4 :LINES))
                    ((MENU-HISTORY :HORIZONTAL (3 :LINES HISTORY) (MENU HISTORY)
                                   ((MENU :ASK :PANE-SIZE))
                                   ((HISTORY :EVEN))))
                    ((SIDE-BY-SIDE :HORIZONTAL (:even)
                                   (INSPECTOR-2 INSPECTOR-1)
                                   ((INSPECTOR-1 0.5))
                                   ((INSPECTOR-2 :even)))))
	    (:debug (,(CAR INSPECTORS) MENU-HISTORY INTERACTOR)
                      ((INTERACTOR 45 :LINES))
                      ((MENU-HISTORY
                         :HORIZONTAL (3 :LINES HISTORY)
                         (MENU HISTORY)
                         ((MENU :ASK :PANE-SIZE))
                         ((HISTORY :EVEN))))
                      ((,(CAR INSPECTORS) :EVEN)))
            )))))

;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;;; Bug in the original Inspector...

;;; Used to use = to compare (nth x ucl::kbd-input) to #\mouse-x-y.  This
;;; would barf if (nth x ucl::kbd-input) was not a number.  This is an
;;; unreasonable assumption
(DEFMETHOD (BASIC-INSPECT-FRAME :AROUND :HANDLE-UNKNOWN-INPUT) (CONT MT IGNORE)
  (LET (inspection-data)
    (COND
      ;;first see if they toggled a pane's locked status
      ((AND (CONSP ucl:kbd-input)
            (EQ (FIRST ucl::kbd-input) :MOUSE-BUTTON)
            (eql (SECOND ucl::kbd-input) #\MOUSE-M-2))
       (SEND (THIRD ucl::kbd-input) :toggle-lock))   
      ((AND (CONSP ucl:kbd-input)
            (eql (fourth ucl::kbd-input) #\MOUSE-M-2))
       (SEND (THIRD ucl::kbd-input) :toggle-lock))   
      ;;If not a blip, let UCL's method handle unknown input
      ((NEQ UCL::INPUT-MECHANISM 'UCL::UNKNOWN)
      ;?((OR (EQ UCL::INPUT-MECHANISM 'UCL::TYPEIN) (not (LISTP THING)))
       (FUNCALL-WITH-MAPPING-TABLE CONT MT :HANDLE-UNKNOWN-INPUT))
      ;;Blip contains an inspection-data instance and we are currently inspecting treating them specially.
      ((AND INSPECTION-DATA-ACTIVE?
            (OR
              ;;Blip in form (INSTANCE (:ITEM1 INSTANCE <inspection-data instance>) <window> <mouse button>).
              ;;These are the standard inspection-data blips from the inspection panes.
              (AND (EQ (FIRST UCL::KBD-INPUT) 'INSTANCE)
                   (EQ (FIRST (SECOND UCL::KBD-INPUT)) :ITEM1)
                   (TYPEP (THIRD (SECOND UCL::KBD-INPUT)) 'INSPECTION-DATA)
                   (SETQ INSPECTION-DATA (THIRD (SECOND UCL::KBD-INPUT))))
              ;;Blip in form (:VALUE <inspection-data instance> <window> <mouse button>).  These blips come from
              ;;the inspection history and always have flavor information in them.
              (AND (EQ (FIRST UCL::KBD-INPUT) :VALUE)
                   (TYPEP (SECOND UCL::KBD-INPUT) 'INSPECTION-DATA)
                   (SETQ INSPECTION-DATA (SECOND UCL::KBD-INPUT)))))
       ;;Have the INSPECTION-DATA handle the mouse blip.  (Each type of info handles the various mouse buttons differently.)
       (SEND INSPECTION-DATA :HANDLE-MOUSE-CLICK UCL::KBD-INPUT SELF))
      ((EQ (FIRST UCL::KBD-INPUT) :LINE-AREA)
       (SELECTOR (FOURTH UCL::KBD-INPUT) eql
         (#\MOUSE-L (SEND SELF :INSPECT-INFO-LEFT-CLICK))
         (#\MOUSE-M
          ;; Delete from line area
          (SEND HISTORY :FLUSH-OBJECT (INSPECT-REAL-VALUE UCL::KBD-INPUT))
          (SEND HISTORY :SET-CACHE NIL)
          ;;make sure the pane is unlocked if they deleted that item
          (LOOP for iw in inspectors
                when (EQ (INSPECT-REAL-VALUE UCL::KBD-INPUT) (SEND iw :current-object))
                do (SEND iw :set-locked-p nil))
          (UPDATE-PANES))
         (T
          (SEND SELF :INSPECT-INFO-RIGHT-CLICK))))
      ;;Middle click on inspected Lisp object--inspect it, leaving source in one of the windows
      ((AND (eql (FOURTH UCL::KBD-INPUT) #\MOUSE-M)
            (MEMBER (THIRD UCL::KBD-INPUT) INSPECTORS :TEST #'EQ))
       (SEND SELF :INSPECT-INFO-MIDDLE-CLICK))
      ;; Right Click on inspected Lisp Object-- inspect its function definition, or itself if no function.
      ((eql (FOURTH UCL::KBD-INPUT) #\MOUSE-R)
       (SEND SELF :INSPECT-INFO-RIGHT-CLICK))
      ((KEY-STATE :HYPER)
       ;; Hyper means modify the slot we are pointing at.
       (IF (OR (NULL (FIRST UCL::KBD-INPUT)) (NULL (GET (FIRST UCL::KBD-INPUT) 'SET-FUNCTION)))
           (FORMAT USER "~&Cannot set this component.")
           (PROGN
             (INSPECT-SET-SLOT UCL::KBD-INPUT USER HISTORY INSPECTORS)
             (UPDATE-PANES)))
       (SEND SELF :HANDLE-PROMPT))
      (T ;; Otherwise inspect UCL:KBD-INPUT.
       (SEND SELF :INSPECT-INFO-LEFT-CLICK)))))

;-------------------------------------------------------------------------------


