;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:USER -*-

;;; File "SAMPLE-EDITOR"
;;; A waveform editor for sampled sound.
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;;  5 Oct 88  Jamie Zawinski    Added ChangeLog.  Converted to use the UCL.
;;;  6 Oct 88  Jamie Zawinski    Got that silly modalism working.  Added online documentation.
;;;  7 Oct 88  Jamie Zawinski    Restructured file, added more keybound commands.  Added typeout-items to doc window.
;;;                              Added system-key binding.
;;; 11 Oct 88  Jamie Zawinski    Added the REVERSE command.
;;;  6 Dec 88  Jamie Zawinski    Removed the *SAMPLE-HASH-TABLE*, added the *SAMPLE-EDITOR-NAMES*, defined fetch-sample.
;;;  7 Dec 88  Jamie Zawinski    Fixed a bug with backwards-play and the current region in PLAY-SAMPLE.
;;; 27 May 89  Jamie Zawinski 	 Modified PLAY-SAMPLE to run WITHOUT-INTERRUPTS instead of WITH-REAL-TIME.  Now you can't
;;;				  slow down the sample-playback just by wagging the mouse.
;;; 23 Jun 89  Jamie Zawinski 	 Made it remember the last seconds-to-record you typed.
;;; 21 May 90  Jamie Zawinski 	 Pounded on the window defs enough to make it work semi-gracefully with the KSL desktop...
;;;


(defparameter *default-sample-editor-magnification* 3
  "The default magnification for newly-created sample editor windows.")


(defvar *sample-editor-names* nil "A list of the names of the samples that the sample editor knows about.")
(defvar *default-sample-file* "samples.xld#>")

(defvar *sample-command-table*)
(defvar *sample-minimalist-command-table*)

(defvar *sample-command-menu*)
(defvar *sample-delete-command-menu*)
(defvar *sample-append-command-menu*)
(defvar *sample-save-command-menu*)
(defvar *sample-add-command-menu*)

(defvar *samples-being-appended* '()
  "When in Append mode, this list is used to build up the append order.
  We can't just use a multiple-choice menu, because the order is important.")

(defvar *sample-name-menu-mode* nil
  "One of NIL, :DELETE, :APPEND, :ADD, or :SAVE, the legal modes for the sample editor to be in.")


(defvar *sample-editor-font*		#+KSL fonts:tr10  #-KSL fonts:tr12)
(defvar *sample-editor-bold-font*	#+KSL fonts:hl10b #-KSL fonts:hl12b)
(defvar *sample-editor-small-font*	#+KSL fonts:tr8   #-KSL fonts:tr10)

(defsubst lookup-sample (name)
  "Returns the sample of the specified name."
  (get name 'TV:SOUND-ARRAY))

(defsubst delete-sample (name)
  "Removes and deallocates the sample called NAME."
  (let* ((array (lookup-sample name)))
    (remprop name 'TV:SOUND-ARRAY)
    (setq *sample-editor-names* (delete name *sample-editor-names*))
    (when array (deallocate-resource 'TV:SOUND-ARRAY array)))
  nil)

(defsubst store-sample (name array)
  "Associates the sample ARRAY with the NAME.  If NAME already defines a sample, it is deallocated first."
  (delete-sample name)
  (setf (get name 'TV:SOUND-ARRAY) array)
  (pushnew name *sample-editor-names*)
  name)


;;; Random array bashing.


(defun sample-array-p (array)
  "T if the array is suitable for passing to TV:PLAY."
  (and (typep array '(vector (integer 0 255)))
       (eql 2 (tv:array-leader-length array))))


(defsubst sound-on-p ()
  "Returns T if sound is currently on, NIL otherwise."
  (not (zerop (tv:sib-sound-bit :query))))


(defmacro safe-with-sound-enabled (&body body)
  "Like TV:WITH-SOUND-ENABLED, except if sound was on before entry, it is not turned off afterwards."
  (let* ((sound-already-on-p (gensym)))
    `(let* ((,sound-already-on-p (sound-on-p)))
       (unwind-protect
	   (progn
	     (unless ,sound-already-on-p (tv:reset-sound t))
	     ,@body)
	 (unless ,sound-already-on-p (tv:reset-sound nil))))))



(defun play-sample (array &optional (start 0) end repeat-p (volume-mod 1) (freq-mod 1) reverse-p)
  "Play an array of sampled sound.  ARRAY may be a sound array or a symbol which has a sound array as its
  TV:SOUND-ARRAY property.  START and END are the segment to play.  If REPEAT-P is T, then it will be played repeatedly.
  Playback stops when there is any keyboard activity.
  VOLUME-MOD and FREQ-MOD are multiplying modifiers for the volume and frequency of the sample.
  FREQ-MOD changes both pitch and duration."
  (when (symbolp array) (setq array (lookup-sample array)))
  (tv:beep-stop-flash)
  (setq end (or end (length array)))
  (tv:with-wired-array array     ; The array MUST be paged in for the duration.
    (tv:without-interrupts	 ; Turn off ALL interrupts, including KBD.  We snarf hardware chars ourself.
      (safe-with-sound-enabled   ; Turn the speaker on, and make sure it goes off.
	(loop
	  (do* ((iterations (1- (floor (- end start) freq-mod)))
		(i (if reverse-p (1- (- end start)) 0)
		   (if reverse-p (1- i) (1+ i)))
		(count 0 (1+ count)))
	       ((= count iterations))
	    (declare (fixnum iterations i count))
	    
	    (let* ((index (the fixnum (+ start (floor (* freq-mod i)))))
		   (sample (aref array (min (the fixnum end) (the fixnum index))))
		   (kbd-char 0))
	      (declare (fixnum index sample kbd-char))
	      
	      ;; If some KBD activity has occurred, bug out.
	      (when (and (tv:kbd-hardware-char-available)
			 (setq kbd-char (tv:kbd-convert-to-software-char (tv:kbd-get-hardware-char))))
		(unless (= kbd-char (char-code #\Space))		; Put it in the input buffer unless it's a space.
		  (tv:io-buffer-put (tv:kbd-get-io-buffer) kbd-char))
		(return-from PLAY-SAMPLE nil))
	      
	      (tv:speech (min 255 (floor (* volume-mod sample))) nil)))  ; Can't guarentee it to be in synch.
	  (unless repeat-p (return-from PLAY-SAMPLE nil)))))))


;;;
;;; We use this instead of TV:RECORD because we were having problems with interference from the function TV:GET-SOUND-ARRAY.
;;; It was second-guessing us, and deallocating things that it shouldn't, so two consecutive records used the same array!
;;;
(defun record-sample (seconds &optional (loud t))
  (tv:with-sound-enabled
    (let* ((n (floor (* seconds 8000)))
	   (vec (allocate-resource 'TV:SOUND-ARRAY n))
	   (bits #xf100))
      (setf (fill-pointer vec) n)
      (if loud (setq bits (+ bits #x200)))
      (si:%nubus-write TV:SIB-SLOT-NUMBER TV:MONITOR-CONTROL bits)
      (tv:with-real-time
	(dotimes (i n) (setf (aref vec i) (tv:voice)))
	vec))))



(defun expand-sample (array ratio)
  "Returns a new sample which is a copy of ARRAY expanded to be RATIO times bigger.  RATIO must be >= 1.
  This changes the pitch and duration of the sample."
  (when (symbolp array) (setq array (lookup-sample array)))
  (let* ((start-size (length array))
	 (end-size (floor (* start-size ratio)))
	 (array2 (allocate-resource 'TV:SOUND-ARRAY end-size)))
    (setf (fill-pointer array2) end-size)
    (dotimes (i end-size)
      (setf (aref array2 i)
	    (aref array (floor i ratio))))
    array2))


(defun make-modified-copy (sample &optional (start 1) end (volume-mod 1) (freq-mod 1) reverse-p)
  "Returns a new array which is the indicated portion of SAMPLE modified by VOL and FREQ."
  (when (symbolp sample) (setq sample (lookup-sample sample)))
  (setq end (or end (length sample)))
  (let* ((length (round (- end start) freq-mod))
	 (new-array (allocate-resource 'TV:SOUND-ARRAY length)))
    (setf (fill-pointer new-array) length)
    (dotimes (i length)
      (let* ((index-in-region (round (* freq-mod i)))
	     (index (if reverse-p
			(- end index-in-region 1)
			(+ start index-in-region)))
	     (byte (aref sample index)))
	(setf (aref new-array i)
	      (min 255 (floor (* volume-mod byte))))))
    new-array))


(defun append-samples (&rest samples)
  "Given any number of sound-arrays, this function returns a new sound array of all of them."
  (let* ((length 0))
    (dolist (sample samples)
      (incf length (length sample)))
    (let* ((new-array (allocate-resource 'TV:SOUND-ARRAY length))
	   (last-x 0))
      (setf (fill-pointer new-array) length)
      (dolist (sample samples)
	(replace new-array sample :start1 last-x)
	(incf last-x (length sample)))
      new-array)))


(defun make-sub-sample (sample start end)
  "Returns a new sample array, which is a subsequence of SAMPLE from START to END."
  (let* ((new-array (allocate-resource 'TV:SOUND-ARRAY (- end start))))
    (setf (fill-pointer new-array) (- end start))
    (sys:copy-array-portion sample start end new-array 0 (- end start))
;    (replace (the array new-array) (the array sample) :start1 0 :start2 start :end2 end)
    new-array))

(defun add-samples (sample-1 sample-2)
  "Returns a new sample, which is SAMPLE-1 and SAMPLE-2 overlayed.  They may be different lengths."
  (let* ((length-1 (length sample-1))
	 (length-2 (length sample-2))
	 (length-3 (max length-1 length-2))
	 (sample-3 (allocate-resource 'TV:SOUND-ARRAY length-3)))
    (setf (fill-pointer sample-3) length-3)
    (dotimes (i length-3)
      (let* ((a (when (< i length-1) (aref sample-1 i)))
	     (b (when (< i length-2) (aref sample-2 i))))
	(setf (aref sample-3 i)
	      (if (and a b) (round (+ a b) 2) (or a b)))))
    sample-3))

(defun add-multiple-samples (&rest samples)
  "Returns a new sample, which is all of the samples overlayed.  They may be different lengths."
  (let* ((max-length 0)
	 (min-length MOST-POSITIVE-FIXNUM))
    (dolist (sample samples)
      (setq max-length (max max-length (length sample))
	    min-length (min min-length (length sample))))
    (let* ((target-sample (allocate-resource 'TV:SOUND-ARRAY max-length)))
      (setf (fill-pointer target-sample) max-length)
      (dotimes (i max-length)
	(let* ((value 0))
	  (dolist (sample samples)
	    (incf value
		  (if (< i (length sample))
		      (aref sample i)
		      0)))
	  (setf (aref target-sample i) (round value (length samples)))))
      target-sample)))


;;; The windows.


(defvar *sample-pane-list* '((menu    sample-menu)
			     (status  sample-status-window)
			     (display sample-display)
			     (scroll-bar sample-scroll-bar)
			     (clip-bar   sample-clip-bar)
			     (samples sample-name-menu)
			     (typein  sample-typein-window)))


(defvar *sample-constraints-list*
	'((default . ((whole)
		      ((whole :horizontal (:even)
			 (left-side right-side)
			 ((right-side :vertical (300)
			    (menu status)
			    ((status #-KSL 230 #+KSL 200))
			    ((menu :even))
			    ))
			  ((left-side :vertical (:even)
			   (samples display scroll-bar clip-bar typein)
			   ((typein 2 :lines))
			   ((display 275))      ; Must be a little bigger than the sample volume resolution of 255.
			   ((scroll-bar 0.05))
			   ((clip-bar 0.05))
			   ((samples :even))
			   ))
			  ))))))


(defflavor sample-frame
	   (main-menu
	    sample-menu
	    display
	    scroll-bar
	    clip-bar
	    status)
	   (ucl:command-loop-mixin
	    ucl:selective-features-mixin
	    tv:inferiors-not-in-select-menu-mixin
	    tv:list-mouse-buttons-mixin
	    tv:bordered-constraint-frame-with-shared-io-buffer
	    w:window)
  (:default-init-plist
    :label nil
    :remove-features '(:LISP-TYPEIN :ALL-UNIVERSAL-COMMANDS)
    :active-command-tables '(*sample-command-table*)
    :all-command-tables    '(*sample-command-table* *sample-minimalist-command-table*)
    
    ;;Specify that TERMINAL-IO be used for command type-in.  The default is a pop-up window.
    ;;See below in :DESIGNATE-IO-STREAMS where TERMINAL-IO is set to the interaction pane.
    :typein-handler nil ; :handle-typein-input
    
    :constraints *sample-constraints-list*
    :panes	 *sample-pane-list*
    :menu-panes '((menu *sample-command-menu*))
    
    :more-p      nil
    :blinker-deselected-visibility nil
    )
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor sample-display
	   ((sample nil)
	    (volume-mod 1)
	    (freq-mod 1)
	    (reverse-p nil)
	    (magnification *default-sample-editor-magnification*)
	    (display-start 0)
	    (start-mark 0)
	    (end-mark 0)
	    )
	   (w:window)
  (:default-init-plist
    :blinker-deselected-visibility nil
    :save-bits t
    :border-margin-width 5
    :label nil)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor sample-scroll-bar
	   ()
	   (w:window)
  (:default-init-plist :blinker-deselected-visibility nil :label nil)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor sample-clip-bar
	   ()
	   (w:window)
  (:default-init-plist :blinker-deselected-visibility nil :label nil)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor sample-menu
	   ((w:item-alignment :center)
	    (w:columns 1))
	   (w:dynamic-item-list-mixin
	    w:command-menu-pane
	    tv:menu)
  (:default-init-plist
    :border-margin-width 2
    :label '(:string "Sample Editor" :font FONTS:METS :centered)
; ## I want this menu to always have a scroll bar and it just won't!!
;    :scroll-bar-on-off :on
;    :scrolling-p t
;    :scroll-bar 10
;    :scroll-bar-always-displayed t
    )
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor sample-name-menu
	   ()
	   (sample-menu)
  (:default-init-plist
    :border-margin-width 2
    :label '(:string "Select a Sample" :font *sample-editor-bold-font*))
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor sample-typein-window
	   ()
	   (ucl:command-and-lisp-typein-window)
  (:default-init-plist :label nil)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor sample-status-window
	   ()
	   (w:basic-mouse-sensitive-items w:window)
  (:default-init-plist
    :label nil
    :font-map '#26(*sample-editor-font* *sample-editor-bold-font* *sample-editor-small-font*)
    :blinker-deselected-visibility nil
    :item-type-alist '((:change-magnification :change-magnification "Change the Magnification.")
		       (:change-pitch :change-pitch "Change the Pitch.")
		       (:change-volume :change-volume "Change the Volume.")
		       (:change-clip-start  :change-clip-start  "Change the starting point of the clip-region.")
		       (:change-clip-end    :change-clip-end    "Change the ending point of the clip-region.")
		       (:change-clip-length :change-clip-length "Change the length point of the clip-region.")
		       (:reverse            :reverse            "Make the sample be played in the opposite direction.")
		       ))
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod (sample-frame	 :screen-manage-deexposed-visibility) () t)
(defmethod (sample-display	 :screen-manage-deexposed-visibility) () t)
(defmethod (sample-scroll-bar	 :screen-manage-deexposed-visibility) () t)
(defmethod (sample-clip-bar	 :screen-manage-deexposed-visibility) () t)
(defmethod (sample-menu		 :screen-manage-deexposed-visibility) () t)
(defmethod (sample-name-menu	 :screen-manage-deexposed-visibility) () t)
(defmethod (sample-typein-window :screen-manage-deexposed-visibility) () t)
(defmethod (sample-status-window :screen-manage-deexposed-visibility) () t)


(defmethod (sample-frame :name-for-selection) ()
  (format nil "Sample Editor editing ~A"
	  (when display
	    (if (send display :sample)
		(format nil "\"~:(~A~)\"" (send display :sample))
		"nothing"))))


(defmethod (sample-display :who-line-documentation-string) ()
  '(:mouse-L-1 "Position the start of the clip region"
    :mouse-R-1 "Position the end of the clip region"
    :mouse-R-2 "System Menu"
    :documentation "This is the window in which the sample waveform is displayed."))

(defmethod (sample-scroll-bar :who-line-documentation-string) ()
  '(:mouse-L-1 "Position the viewed region"
    :mouse-R-2 "System Menu"
    :documentation "This is a scroll bar which shows how much of the whole sample is visible in the Display window."))

(defmethod (sample-clip-bar :who-line-documentation-string) ()
  '(:mouse-L-1 "Position the start of the clip region"
    :mouse-R-1 "Position the end of the clip region"
    :mouse-R-2 "System Menu"
    :documentation "This is a scroll bar which represents the current clip region."))

(defmethod (sample-status-window :who-line-documentation-string) ()
  '(:mouse-L-1 "Select Mouse-Sensitive Item"
    :mouse-R-2 "System Menu"
    :documentation "This window is used for the display of ``useful information.''  Some text is mouse-sensitive."))



(defmethod (sample-display :reset-markers) ()
  (setf start-mark 0
	end-mark (length (lookup-sample sample))
	display-start 0))

(defmethod (sample-frame :designate-io-streams) ()
  (setq *terminal-io* (send self :get-pane 'typein)))



(defmethod (sample-frame :after :init) (&rest ignore)
  ;;
  ;; Initialize the pointers to the subwindows.
  ;;
  (setq main-menu   (send self :get-pane 'menu))
  (setq sample-menu (send self :get-pane 'samples))
  (setq display     (send self :get-pane 'display))
  (setq scroll-bar  (send self :get-pane 'scroll-bar))
  (setq clip-bar    (send self :get-pane 'clip-bar))
  (setq status      (send self :get-pane 'status))
  ;;
  ;; Initialize the command menu.
  ;;
  (setf (send main-menu :item-list) (send main-menu :column-spec-list))
  (send self :grind-sample-menu)
  ;;
  ;; Enable mouse-sensitive-text input.
  ;;
  (push '(:TYPEOUT-EXECUTE :HANDLE-TYPEOUT-EXECUTE)
	ucl:blip-alist)
  )



;;; Displaying the wave.


(defmethod (sample-display :after :refresh) (&optional (type :complete-redisplay))
  (when (and (eq type :complete-redisplay) sample)
    (draw-sample self (lookup-sample sample) display-start magnification volume-mod freq-mod)
    (send (send tv:superior :scroll-bar) :refresh :complete-redisplay)
    (send (send tv:superior :clip-bar) :refresh :complete-redisplay)
    ))

(defmethod (sample-typein-window :after :refresh) (&optional (type :complete-redisplay))
  (when (eq type :complete-redisplay) (send self :home-cursor)))

(defmethod (sample-display :after :set-sample) (new-sample)
  (setq display-start 0)
  (setq start-mark 0)
  (setq reverse-p nil)
  (setq end-mark (length (lookup-sample new-sample)))
;  (send tv:superior :grind-sample-menu)
  (send tv:superior :refresh-status)
  (tv:sheet-force-access (self)
    (send self :refresh :complete-redisplay)
    ))

(defmethod (sample-display :display-end) ()
  (let* ((w (send self :inside-width))
	 (samples-displayable (round w (* freq-mod magnification))))
    (min (length (lookup-sample sample))
	 (+ display-start samples-displayable))))


(defmethod (sample-clip-bar :after :refresh) (&optional (type :complete-redisplay))
  (when (eq type :complete-redisplay)
    (let* ((display (send tv:superior :display))
	   (sample (lookup-sample (send display :sample))))
      (when sample
	(let* ((my-length (send self :inside-width))
	       (my-height (send self :inside-height))
	       (sample-length (length sample)))
	  (when (plusp sample-length)
	    (let* ((start (send display :start-mark))
		   (end (send display :end-mark))
		   (start-ratio (float (/ start sample-length)))
		   (end-ratio (float (/ end sample-length)))
		   (start-pixel (floor (* my-length start-ratio)))
		   (end-pixel (floor (* my-length end-ratio)))
		   (w:*default-texture* W:50%-GRAY))
	      (send self :draw-filled-rectangle start-pixel 0 (- end-pixel start-pixel) my-height))))))))


(defmethod (sample-scroll-bar :after :refresh) (&optional (type :complete-redisplay))
  (when (eq type :complete-redisplay)
    (let* ((display (send tv:superior :display))
	   (sample (lookup-sample (send display :sample))))
      (when sample
	(let* ((my-length (send self :inside-width))
	       (my-height (send self :inside-height))
	       (sample-length (length sample)))
	  (when (plusp sample-length)
	    (let* ((start (send display :display-start))
		   (end (send display :display-end))
		   (start-ratio (float (/ start sample-length)))
		   (end-ratio (float (/ end sample-length)))
		   (start-pixel (floor (* my-length start-ratio)))
		   (end-pixel (floor (* my-length end-ratio)))
		   (w:*default-texture* W:50%-GRAY))
	      (send self :draw-filled-rectangle start-pixel 0 (- end-pixel start-pixel) my-height))))))))


(defsubst end-mark-time (sample-frame)
  (send (send sample-frame :display) :end-mark))

(defsubst start-mark-time (sample-frame)
  (send (send sample-frame :display) :start-mark))


(defun ensure-sample-present ()
  "Barf at the user if there is no sample for the SAMPLE-FRAME in SELF."
  (declare (:self-flavor SAMPLE-FRAME))
  (unless (send display :sample)
    (beep)
    (format t "~&No current sample.~%")
    (signal-condition EH:ABORT-OBJECT)))


(defun draw-sample (window array &optional (start 0) (magnification 1) (volume-mod 1) (freq-mod 1))
  (let* ((w (tv:sheet-inside-width window))
	 (h (tv:sheet-inside-height window))
	 (y-offset (max 0 (round (- h 255) 2)))
	 (last-x 0)
	 (last-y (- h y-offset))
	 (x-mag (* magnification freq-mod)))
    ;;
    ;; Draw the curve
    ;;
    (let* ((active-length (- (length array) start))
	   (max-w (floor (min active-length (/ w x-mag)))))
      (dotimes (x max-w)
	(let* ((real-x (floor (* x x-mag)))
	       (sample (aref array (+ start x)))
	       (mod-sample (min 255 (floor (* volume-mod sample))))
	       (real-y (+ (- h mod-sample) y-offset)))
	(send window :draw-line last-x last-y real-x real-y)
	(setq last-x real-x last-y real-y)))))
  (maybe-draw-markers (tv:sheet-superior window))
  array)


(defun maybe-draw-markers (sample-frame)
  "Draw the markers in the sample window if they are visible."
  (let* ((display (send sample-frame :display))
	 (start (send display :display-start))
	 (magnification (send display :magnification))
	 (freq-mod (send display :freq-mod))
	 (x-mag (float (* magnification freq-mod)))
	 (top-margin 5)
	 (bottom-margin 5)
	 (w (tv:sheet-inside-width display))
	 (h (- (tv:sheet-inside-height display) top-margin bottom-margin))
	 (smt (start-mark-time (send display :superior)))
	 (emt (end-mark-time (send display :superior)))
	 (smtx (round (* (- smt start) x-mag)))
	 (emtx (round (* (- emt start) x-mag)))
	 (thickness (max 1 (round x-mag 2)))
	 (W:NORMAL TV:ALU-XOR))
    (when (< smtx w)
      (send display :draw-line smtx top-margin smtx (+ h top-margin) thickness))
    (when (< emtx w)
      (send display :draw-line emtx top-margin emtx (+ h top-margin) thickness))))


;;; The Commands


(defun prompt-for-keyword (prompt-string &optional default)
  (if default
      (let* ((string (prompt-and-read :STRING-OR-NIL "~&~A (default ~A) " prompt-string default)))
	(if (and string (string/= "" string))
	    (intern (string-upcase string) "KEYWORD")
	    default))
      (let ((string ""))
	(do ()
	    ((string/= "" string))
	  (setq string (prompt-and-read :STRING-TRIM "~&~A" prompt-string)))
	(intern (string-upcase string) "KEYWORD"))))


(defun prompt-for-plusp (prompt-string &optional default)
  (let* ((number 0))
    (do ()
	((plusp number))
      (if default
	  (setq number (or (prompt-and-read '(:NUMBER :OR-NIL T) "~&~A (default ~A) " prompt-string default)
			   default))
	  (setq number (prompt-and-read :NUMBER "~&~A" prompt-string))))
    number))

(defun prompt-for-not-minusp (prompt-string &optional default)
  (let* ((number -1))
    (do ()
	((not (minusp number)))
      (if default
	  (setq number (or (prompt-and-read '(:NUMBER :OR-NIL T) "~&~A (default ~A) " prompt-string default)
			   default))
	  (setq number (prompt-and-read :NUMBER "~&~A" prompt-string))))
    number))


;;; Online Documentation


(defmethod (sample-frame :display-help) ()
  (send status :clear-screen)
  (send status :set-current-font *sample-editor-small-font*)
  (dolist (line '("Welcome to the Sample Editor."
		  "   The top area is a menu of the names of currently"
		  "known samples. Selecting one of these makes it be"
		  "the current sample."
		  "   Below that is where the waveform is displayed."
		  "Below that are two scrollbars.  The first represents"
		  "how much of the sample is visible in the display."
		  "Mouse-Left changes the viewed region."
		  "   The second scroll bar represents the current clip"
		  "region.  This is the part of the sample which is"
		  "used by ``Play Sample,'' among other things."
		  "Mouse-Left moves the start of this region, and"
		  "Mouse-Right moves the end."
;		  "   Control-V and Meta-V page the waveform display."
;		  "Comma, Period, <, and > move the clip-start."
;		  "Minus, Underscore, Plus, and Equal move the end."
;		  "END quits."
		  ))
    (write-string line status) (terpri status))
  )


(defmethod (sample-frame :display-delete-help) ()
  (send status :clear-screen)
  (send status :set-current-font *sample-editor-font*)
  (format status "You are in ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Delete Sample")
  (send status :set-current-font *sample-editor-font*)
  (format status " mode.~%~%")
  (format status "In this mode, the Sample Name Menu is~%")
  (format status "multiple-choice.  Hilight the names of~%")
  (format status "the samples you want to delete, and then~%click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Delete Selected Samples.~%~%")
  (send status :set-current-font *sample-editor-font*)
  (format status "If you don't want to delete anything,~%")
  (format status "then click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Abort Deletion.~%")
  )


(defmethod (sample-frame :display-save-help) ()
  (send status :clear-screen)
  (send status :set-current-font *sample-editor-font*)
  (format status "You are in ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Save Samples")
  (send status :set-current-font *sample-editor-font*)
  (format status " mode.~%~%")
  (format status "In this mode, the Sample Name Menu is~%")
  (format status "multiple-choice.  Hilight the names of~%")
  (format status "the samples you want to save to disk,~%and then click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Save Selected Samples.~%~%")
  (send status :set-current-font *sample-editor-font*)
  (format status "You will then be prompted for the~%")
  (format status "name of the binary file to write.~%~%")
  (format status "If you don't want to delete anything,~%")
  (format status "then click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Abort Save.~%")
  )

(defmethod (sample-frame :display-add-help) ()
  (send status :clear-screen)
  (send status :set-current-font *sample-editor-font*)
  (format status "You are in ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Add Samples")
  (send status :set-current-font *sample-editor-font*)
  (format status " mode.~%~%")
  (format status "In this mode, the Sample Name Menu is~%")
  (format status "multiple-choice.  Hilight the names of~%")
  (format status "the samples you want to add together,~%and then click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Add Selected Samples.~%~%")
  (send status :set-current-font *sample-editor-font*)
  (format status "You will then be prompted for the~%")
  (format status "name of the new sample to create.~%~%")
  (format status "If you don't want to add after all, then~%click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Abort Add.~%")
  )


(defmethod (sample-frame :display-append-help) ()
  (send status :clear-screen)
  (send status :set-current-font *sample-editor-font*)
  (format status "You are in ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Append Samples")
  (send status :set-current-font *sample-editor-font*)
  (format status " mode.~%~%")
  (format status "As you click on the names of samples,~%")
  (format status "a list of your selections is accumulated.~%")
  (format status "When you click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Append Selected~%Samples,")
  (send status :set-current-font *sample-editor-font*)
  (format status " the samples you selected will be~%")
  (format status "appended together in the order you~%")
  (format status "chose them, and will be stored under~%")
  (format status "a prompted-for name.~%~%")
  (format status "To abort, click on ")
  (send status :set-current-font *sample-editor-bold-font*)
  (format status "Abort Append.")
  )


(defmethod (sample-status-window :display-append-list) ()
  (send self :clear-screen)
  (send self :set-current-font *sample-editor-font*)
  (format self "You are in ")
  (send self :set-current-font *sample-editor-bold-font*)
  (format self "Append Samples")
  (send self :set-current-font *sample-editor-font*)
  (format self " mode.~%~%")
  (format self "Selected names:~%~%")
  (send self :set-current-font fonts:hl10b)
  (dolist (name (reverse *samples-being-appended*))
    (format self "~10t~A~%" (string-capitalize (string name))))
  )


(defmethod (sample-frame :refresh-status) ()
  (send status :refresh))



(defmethod (sample-frame :handle-typeout-execute) ()
  (declare (special UCL:KBD-INPUT))
  (let* ((type (second UCL:KBD-INPUT)))
    (ecase type
      (:CHANGE-MAGNIFICATION (send self :change-magnification-command))
      (:CHANGE-VOLUME        (send self :change-volume-command))
      (:CHANGE-PITCH         (send self :change-pitch-command))
      (:CHANGE-CLIP-START    (send self :change-clip-start-command))
      (:CHANGE-CLIP-END      (send self :change-clip-end-command))
      (:CHANGE-CLIP-LENGTH   (send self :change-clip-length-command))
      (:REVERSE              (send self :reverse-command))
      )))


(defmethod (sample-status-window :after :refresh) (&optional (type :complete-redisplay))
  (when (eq type :complete-redisplay)
    (send self :clear-screen)
    (let* ((display (send tv:superior :display))
	   (sample-name (send display :sample))
	   (array (lookup-sample sample-name)))
      (when sample-name
	(send self :set-current-font *sample-editor-font*)
	(princ "Current Sample is " self)
	(send self :set-current-font *sample-editor-bold-font*)
	(princ (string-capitalize (string sample-name)) self) (format self ".~%~%")
	(send self :set-current-font *sample-editor-font*)
	(let* ((r (end-mark-time tv:superior))
	       (l (start-mark-time tv:superior))
	       (mag (rationalize (send display :magnification)))
	       (vol (send display :volume-mod))
	       (freq (send display :freq-mod)))
	(format self "Total Length: ~20t ~,3F second~:P.~%~%~
                      Region start: ~20t ~vM second~:P.~%~
                      Region end:   ~20t ~vM second~:P.~%~
                      Region length:~20t ~vM second~:P.~%~%"
		(float (/ (length array) 8000))
		:change-clip-start  (format nil "~,3F" (float (/ l 8000)))
		:change-clip-end    (format nil "~,3F" (float (/ r 8000)))
		:change-clip-length (format nil "~,3F" (float (/ (- r l) 8000))))
	
	(format self "Current Magnification:     ~35t~vM~%~
                      Current Volume Modifier:   ~35t~vM~%~
                      Current Frequency Modifier:~35t~vM~%~%~
                      This sample is being played ~vM~%"
		:change-magnification (format nil " ~D:~D " (numerator mag) (denominator mag))
		:change-volume        (format nil " ~D " vol)
		:change-pitch         (format nil " ~D " freq)
		:reverse              (if (send display :reverse-p) "backwards." "forwards.")
		)
	)))))


;;;; The Command Definitions.

(defmacro def-sample-command ((method-name string-name description &optional keys) arglist &body body)
  "Shorthand for the calls to DEFCOMMAND that we make for the sample editor."
  `(defcommand (sample-frame ,method-name) ,arglist
     '(:description ,description
       :names ,string-name
       :keys ,keys)
     ,@body))



(def-sample-command (:play-sample-command "Play Sample" "Play the current sample." #\p) (&optional repeat-p)
  (ensure-sample-present)
  (play-sample (lookup-sample (send display :sample))
	       (start-mark-time self) (end-mark-time self) repeat-p
	       (send display :volume-mod) (send display :freq-mod)
	       (send display :reverse-p)))


(def-sample-command (:play-sample-repeating-command "Play Sample Repeating"
		     "Play the current sample repeatedly until a key is pressed." #\r) ()
  (send self :play-sample-command t))


(defvar *sample-default-length* 2)

(def-sample-command (:record-sample-command "Record Sample" "Record a sample under a prompted-for name.") ()
  (let* ((sym (or (send display :sample) :new-sample))
	 (length 1))
    (setq sym (prompt-for-keyword "Name of this sample: " (send display :sample)))
    (setq length (prompt-for-plusp "Seconds to Record: " *sample-default-length*))
    (setq *sample-default-length* length)
    (format t "~&Type any character to begin recording.")
    (read-char)
    (terpri)
    (let* ((array (record-sample length)))
      (store-sample sym array))
    (read-char-no-hang self)
    (send display :set-sample sym)
    (send display :reset-markers)
    (send self :grind-sample-menu)
    (send display :refresh :complete-redisplay)))



(def-sample-command (:change-magnification-command "Change Magnification" "Change the which is used to display the sample."
						   #\m) ()
  (ensure-sample-present)
  (setf (send display :magnification)
	(prompt-for-plusp "Magnification: " (send display :magnification)))
  (send self :refresh-status)
  (send display :refresh :complete-redisplay))


(def-sample-command (:change-pitch-command "Change Pitch"
		     "Change the pitch and duration of the current sample.  This is non-destructive.") ()
  (ensure-sample-present)
  (setf (send display :freq-mod)
	(prompt-for-plusp "New Pitch modification: " 1))
  (send self :refresh-status)
  (send display :refresh :complete-redisplay))


(def-sample-command (:change-volume-command "Change Volume"
		     "Change the volume of the current sample.  This is nondestructive.")
		    ()
  (ensure-sample-present)
  (setf (send display :volume-mod)
	(prompt-for-plusp "New Volume modification: " 1))
  (send self :refresh-status)
  (send display :refresh :complete-redisplay))


(def-sample-command (:reverse-command "Reverse Sample" "Reverse the direction of play of the sample.")
		    ()
  (ensure-sample-present)
  (setf (send display :reverse-p) (not (send display :reverse-p)))
  (send self :refresh-status)
;  (send display :refresh :complete-redisplay)
  )


(defmethod (sample-display :reset-modifications) ()
  (setq volume-mod 1 freq-mod 1 reverse-p nil))

(def-sample-command (:modify-sample-command "Make Changes Permanent"
		     #.(string-append "Destructively modify the current sample so that the current volume, pitch, and "
				      "clipping-region modifications are permanent."))
		    ()
  (ensure-sample-present)
  (let* ((name (send display :sample))
	 (old (lookup-sample name))
	 (new (make-modified-copy old (start-mark-time self) (end-mark-time self)
				  (send display :volume-mod) (send display :freq-mod)
				  (send display :reverse-p))))
    (delete-sample name)
    (store-sample name new)
    (send display :reset-modifications)
    (send display :reset-markers)
    (send self :refresh-status)
    (send display :refresh :complete-redisplay)))



(def-sample-command (:save-sample-region-command "Save Sample Region" "Save the current region under its own name.") ()
  (ensure-sample-present)
  (let* ((sym (prompt-for-keyword "Save region under what name: " :new-sample))
	 (new-sample (make-sub-sample (lookup-sample (send display :sample))
				      (start-mark-time self) (end-mark-time self))))
    (store-sample sym new-sample)
    (when (y-or-n-p "Edit this subsample?")
      (send display :set-sample sym)
      (send display :reset-markers)
      (send display :refresh :complete-redisplay))
    (send self :grind-sample-menu)
    ))


(def-sample-command (:reset-all-command "Reset All Modifications"
		     "Set the clip-region, volume, and pitch to their defaults.")
		    ()
  (send display :reset-markers)
  (setf (send display :volume-mod) 1)
  (setf (send display :freq-mod) 1)
  (setf (send display :reverse-p) nil)
  (send self :refresh-status)
  (send display :refresh)
  )



(def-sample-command (:fetch-sample-command "Fetch Sample"
               "Prompt for a symbol which names a sample (which was previously recorded) and add it to the sample menu.")
		    ()
  (let (new-sample-name sym)
    (unwind-protect
      (do* ()
	   ((and sym (symbolp sym) (get sym 'TV:SOUND-ARRAY))
	    (setq new-sample-name sym))
	(when sym (beep))
	(let* ((*package* (find-package "KEYWORD")))
	  (setq sym (prompt-and-read :READ "~&Type a symbol which names a sample: "))))
      (when new-sample-name (pushnew new-sample-name *sample-editor-names*))
      (send self :grind-sample-menu))))


(def-sample-command (:fetch-all-samples-command "Fetch All Samples"
		     "Search some or all of the packages for symbols which name samples, and add them to the menu.")
		    ()
  (let (package)
    (declare (unspecial package))  ; god this is annoying...
    (do* ((done-once nil t))
	 ((and done-once (or (null package) (find-package package)))
	  (and package (setq package (find-package package))))
      (setq package (prompt-and-read :STRING-OR-NIL
				     "~&Type the name of the package to search, or hit return for \"all\": "))
      (when package (setq package (nstring-upcase package))))
    (format t "~&Searching ")
    (unwind-protect
	(flet ((maybe-push (sym)
		 (when (get sym 'TV:SOUND-ARRAY)
		   (pushnew sym *sample-editor-names*)
		   (princ #\.))))
	  (if package
	      (do-symbols (sym package) (maybe-push sym))
	      (do-all-symbols (sym) (maybe-push sym)))
	  (format t " - done~%"))
      (send self :grind-sample-menu))))



(def-sample-command (:sample-quit-command "Quit" "Exit the sample editor." #\End) ()
  (w:deselect-and-maybe-bury-window self))


(def-sample-command (:redisplay-command "Refresh" "Redraw the sample display." (#\Clear-Screen #\Control-L)) ()
  (send status :refresh)
  (send display :refresh))

(def-sample-command (:redisplay-all-command "Refresh All" "Redraw the entire screen." (#\Control-Clear-Screen #\Meta-L)) ()
  (send self :grind-sample-menu)
  (send self :refresh :complete-redisplay))


(def-sample-command (:help-command "Help!!" "Display some help." #\Help) ()
  (ecase *sample-name-menu-mode*
    (:DELETE (send self :display-delete-help))
    (:APPEND (send self :display-append-help))
    (:SAVE   (send self :display-save-help))
    (:ADD    (send self :display-add-help))
    ((NIL)   (send self :display-help))))


;;; Moving the viewing region.


(def-sample-command (:page-forward "Page Forward" "Show the next window-full of the current sample."
				   #\Control-V)
		    (&optional (ratio 1))
  (ensure-sample-present)
  (let* ((displayed (- (send display :display-end) (send display :display-start))))
    (setf (send display :display-start)
	  (min (send display :display-end)
	       (floor (+ (send display :display-start) (* ratio displayed)))))
    (send scroll-bar :refresh)
    (send display :refresh)))


(def-sample-command (:page-backward "Page Backward" "Show the previous window-full of the current sample."
				    #\Meta-V)
		    (&optional (ratio 1))
  (ensure-sample-present)
  (let* ((displayed (- (send display :display-end) (send display :display-start))))
    (setf (send display :display-start)
	  (max 0 (floor (- (send display :display-start) (* ratio displayed)))))
    (send scroll-bar :refresh)
    (send display :refresh)))


(def-sample-command (:page-forward-short "Page Forward Short" "Like Page Forward, but advances only by 1/4 window-width."
					 #\v) ()
  (send self :page-forward 0.25))

(def-sample-command (:page-backward-short "Page Backward Short"
		     "Like Page Backward, but backs up only by 1/4 window-width." #\V) ()
  (send self :page-backward 0.25))

(def-sample-command (:goto-beginning "Goto Beginning" "Show the beginning of the current sample." #\Meta-<) ()
  (ensure-sample-present)
  (setf (send display :display-start) 0)
  (send display :refresh))

(def-sample-command (:goto-end "Goto End" "Show the end of the current sample." #\Meta->) ()
  (ensure-sample-present)
  (let* ((end (length (lookup-sample (send display :sample))))
	 (width (round (tv:sheet-inside-width display) (* (send display :magnification) (send display :freq-mod)))))
    (setf (send display :display-start) (max 0 (- end width)))
    (send display :refresh)))



;;; Moving the Clip Region.


(def-sample-command (:mark-to-beginning "Mark to Beginning" "Move the start of the clip region to the start of the sample."
		     #\Control-<) ()
  (ensure-sample-present)
  (maybe-draw-markers self)
  (setf (send display :start-mark) 0)
  (maybe-draw-markers self)
  (send clip-bar :refresh))

(def-sample-command (:mark-to-end "Mark to End" "Move the end of the clip region to the end of the sample."
		     #\Control->) ()
  (ensure-sample-present)
  (maybe-draw-markers self)
  (setf (send display :end-mark) (length (lookup-sample (send display :sample))))
  (maybe-draw-markers self)
  (send clip-bar :refresh))


(def-sample-command (:start-clip-forward "Start Clip Forward" "Move the Start Clip Mark forward."
					 #\.)
		    (&optional (increment 10))
  (ensure-sample-present)
  (maybe-draw-markers self)
  (setf (send display :start-mark)
	(min (length (lookup-sample (send display :sample)))
	     (1- (send display :end-mark))
	     (+ (send display :start-mark) increment)))
  (maybe-draw-markers self)
  (send clip-bar :refresh))


(def-sample-command (:start-clip-backward "Start Clip Backward" "Move the Start Clip Mark backward." #\,)
		    (&optional (increment 10))
  (ensure-sample-present)
  (maybe-draw-markers self)
  (setf (send display :start-mark)
	(max 0 (- (send display :start-mark) increment)))
  (maybe-draw-markers self)
  (send clip-bar :refresh))


(def-sample-command (:end-clip-forward "End Clip Forward" "Move the End Clip Mark forward." #\=) (&optional (increment 10))
  (ensure-sample-present)
  (maybe-draw-markers self)
  (setf (send display :end-mark)
	(min (length (lookup-sample (send display :sample)))
	     (+ (send display :end-mark) increment)))
  (maybe-draw-markers self)
  (send clip-bar :refresh))


(def-sample-command (:end-clip-backward "End Clip Backward" "Move the End Clip Mark backward." #\-)
		    (&optional (increment 10))
  (ensure-sample-present)
  (maybe-draw-markers self)
  (setf (send display :end-mark)
	(max 0
	     (1+ (send display :start-mark))
	     (- (send display :end-mark) increment)))
  (maybe-draw-markers self)
  (send clip-bar :refresh))


(def-sample-command (:start-clip-forward-fast "Start Clip Forward Fast" "Move the Start Clip Mark forward fast."
					      #\Control-.) ()
  (send self :start-clip-forward 50))
(def-sample-command (:start-clip-backward-fast "Start Clip Backward Fast" "Move the Start Clip Mark backward fast."
					       #\Control-\,) ()
  (send self :start-clip-backward 50))
(def-sample-command (:end-clip-forward-fast "End Clip Forward Fast" "Move the End Clip Mark forward fast." #\Control-=) ()
  (send self :end-clip-forward 50))
(def-sample-command (:end-clip-backward-fast "End Clip Backward Fast""Move the End Clip Mark backward fast." #\Control--) ()
  (send self :end-clip-backward 50))

(def-sample-command (:start-clip-forward-one "Start Clip Forward One" "Move the Start Clip Mark forward one." #\>) ()
  (send self :start-clip-forward 1))
(def-sample-command (:start-clip-backward-one "Start Clip Backward One" "Move the Start Clip Mark backward one." #\<) ()
  (send self :start-clip-backward 1))
(def-sample-command (:end-clip-forward-one "End Clip Forward One" "Move the End Clip Mark forward one." #\+) ()
  (send self :end-clip-forward 1))
(def-sample-command (:end-clip-backward-one "End Clip Backward One" "Move the End Clip Mark backward one." #\_) ()
  (send self :end-clip-backward 1))


(def-sample-command (:change-clip-start-command "Change Clip start" "Prompt for the position to move the clip-start to.") ()
  (let* ((n -1)
	 (end (float (/ (send display :end-mark) 8000))))
    (do ()
	((and (<= 0 n)
	      (< n end)))
      (setq n (prompt-for-not-minusp (format nil "New clip-start position in seconds (less than ~,3F): " end))))
    (maybe-draw-markers self)
    (setf (send display :start-mark) (floor (* n 8000)))
    (maybe-draw-markers self)
    (send status :refresh)
    (send clip-bar :refresh)))


(def-sample-command (:change-clip-end-command "Change Clip end" "Prompt for the position to move the clip-end to.") ()
  (let* ((n -1)
	 (start (float (/ (send display :start-mark) 8000)))
	 (end (float (/ (length (lookup-sample (send display :sample))) 8000))))
    (do ()
	((< start n end))
      (setq n (prompt-for-plusp (format nil "New clip-end position in seconds (between ~,3F and ~,3F): " start end))))
    (maybe-draw-markers self)
    (setf (send display :end-mark) (ceiling (* n 8000)))
    (maybe-draw-markers self)
    (send status :refresh)
    (send clip-bar :refresh)))

(def-sample-command (:change-clip-length-command "Change Clip length" "Prompt for the length of the clip region.") ()
  (let* ((n -1)
	 (start (float (/ (send display :start-mark) 8000)))
	 (end (float (/ (length (lookup-sample (send display :sample))) 8000)))
	 (len (- end start)))
    (do ()
	((< 0 n len))
      (setq n (prompt-for-plusp (format nil "New clip-region length in seconds (less than ~,3F): " len))))
    (maybe-draw-markers self)
    (setf (send display :end-mark) (ceiling (* (+ start n) 8000)))
    (maybe-draw-markers self)
    (send status :refresh)
    (send clip-bar :refresh)))


;;; Dealing with mouse clicks on various windows.
;;;
;;; This is probably a silly way to handle the mouse, but I couldn't think of a better one.
;;; The documentation on using the mouse with UCL is quite lacking...
;;; The SAMPLE-FRAME has a :MOUSE-CLICK-COMMAND bound to Mouse-L and Mouse-R.
;;; This looks at the blip to see which subwindow it came in under.
;;; If that window handles the :MOUSE-CLICK-COMMAND method, then it is invoked.
;;; Otherwise, we beep.


(def-sample-command (:mouse-click-command "Mouse Click" "All mouse clicks dispatched on here." (#\Mouse-L-1 #\Mouse-R-1)) ()
  (declare (special UCL:KBD-INPUT))
  (ensure-sample-present)
  (let* ((window (third UCL:KBD-INPUT)))
    (if (and (neq window self)
	     (send window :operation-handled-p :mouse-click-command))
	(send window :mouse-click-command)
	(beep))))


(defcommand (sample-scroll-bar :mouse-click-command) ()
  '(:description "Move the current viewing region to be at the place the mouse was clicked."
    :names       "Position Region"
    :keys        (#\Mouse-L-1)
    )
  (declare (special UCL:KBD-INPUT))
  (let* ((blip UCL:KBD-INPUT)
	 (x (fourth blip))
	 (w (tv:sheet-inside-width self))
	 (display (send tv:superior :display))
	 (sample (lookup-sample (send display :sample)))
	 (length (length sample))
	 (ratio (/ length w))
	 (view-width (- (send display :display-start) (send display :display-end)))
	 (new-start (+ (round (* x ratio))
		       (round view-width 2)))
	 )
    (setf (send display :display-start) (max 0 (min length new-start)))
    (send self :refresh)
    (send display :refresh)))


(defcommand (sample-clip-bar :mouse-click-command) ()
  '(:description "Move the start or end of the current clip region to be at the place the mouse was clicked."
    :names       "Position Clip Region"
    :keys        (#\Mouse-L-1 #\Mouse-R-1)
    )
  (declare (special UCL:KBD-INPUT))
  (let* ((blip UCL:KBD-INPUT)
	 (left-p (eql (second blip) #\Mouse-L))
	 (x (fourth blip))
	 (w (tv:sheet-inside-width self))
	 (display (send tv:superior :display))
	 (sample (lookup-sample (send display :sample)))
	 (length (length sample))
	 (ratio (/ length w))
	 (new-start (send display :start-mark))
	 (new-end (send display :end-mark))
	 (clip-width (- new-end new-start))
	 )
    (maybe-draw-markers tv:superior)
    (if left-p
	(setq new-start (round (* x ratio)))
	(setq new-end (round (* x ratio))))
    (when (<= new-end new-start)
      (if left-p
	  (setq new-end (+ new-start clip-width))
	  (setq new-start (- new-end clip-width))))
    (setf (send display :start-mark) (max 0 (min length new-start))
	  (send display :end-mark)   (max 0 (min length new-end)))
    (maybe-draw-markers tv:superior)
    (send self :refresh)))


(defcommand (sample-display :mouse-click-command) ()
  '(:description "Move the start or end of the current clip region to be at the place the mouse was clicked."
    :names       "Position Clip Region"
    :keys        (#\Mouse-L-1 #\Mouse-R-1)
    )
  (declare (special UCL:KBD-INPUT))
  (let* ((blip UCL:KBD-INPUT)
	 (left-p (eql (second blip) #\Mouse-L))
	 (x (fourth blip))
	 (clip-bar (send tv:superior :clip-bar))
	 (length (length (lookup-sample sample)))
	 (click-pos (+ display-start (round x (* magnification freq-mod))))
	 (new-start start-mark)
	 (new-end end-mark)
	 (clip-width (- new-end new-start))
	 )
    (maybe-draw-markers tv:superior)
    (if left-p
	(setq new-start click-pos)
	(setq new-end click-pos))
    (when (<= new-end new-start)
      (if left-p
	  (setq new-end (+ new-start clip-width))
	  (setq new-start (- new-end clip-width))))
    (setf start-mark (max 0 (min length new-start))
	  end-mark   (max 0 (min length new-end)))
    (maybe-draw-markers tv:superior)
    (send clip-bar :refresh)))



(defcommand (sample-status-window :mouse-click-command) ()
  '(:description "Redisplay."
    :names       "Redisplay"
    :keys        (#\Mouse-L-1)
    )
  (send self :refresh))



;;; The Sample Name menu.
;;;
;;; In the top-level state, clicking on this menu makes te selected sample current.
;;; But sometimes executing a command will enter this menu recursively, in which case clicking on it
;;; does a different thing.



(defun set-all-wholine-strings (menu new-string)
  (dolist (list (send menu :item-list))
    (setf (getf (cdr list) :documentation)
	  new-string)))



(defun enter-sample-name-menu-mode (sample-frame mode title-string &optional (wholine-string ""))
  (when (member mode '(:DELETE :SAVE :ADD) :test #'eq)
    (setf (w:sheet-menu-highlighting (send sample-frame :sample-menu)) 1))
  (setf (send (send sample-frame :sample-menu) :label)
	(list :string title-string :font *SAMPLE-EDITOR-BOLD-FONT*))
  (setq *sample-name-menu-mode* mode)
  (set-all-wholine-strings (send sample-frame :sample-menu) wholine-string)
  (setf (send sample-frame :active-command-tables) '(*sample-minimalist-command-table*))
  (setf (send (send sample-frame :main-menu) :item-list)
	(ecase mode
	  (:DELETE *sample-delete-command-menu*)
	  (:APPEND *sample-append-command-menu*)
	  (:SAVE   *sample-save-command-menu*)
	  (:ADD    *sample-add-command-menu*)
	  ))
  )



(defun exit-sample-name-menu-modes (sample-frame)
  (setf (w:sheet-menu-highlighting (send sample-frame :sample-menu)) 0)
  (setf (send (send sample-frame :sample-menu) :highlighted-items) nil)
  (setq *sample-name-menu-mode* nil)
  (setq *samples-being-appended* nil)
  (setf (send (send sample-frame :sample-menu) :label) '(:string "Select a Sample:" :font *sample-editor-bold-font*))
  (setf (send sample-frame :active-command-tables)
	'(*sample-command-table*))
  (setf (send (send sample-frame :main-menu) :item-list)
	*sample-command-menu*)
  (set-all-wholine-strings (send sample-frame :sample-menu) "Click here to make this sample current.")
  (send sample-frame :refresh-status)
  nil)


(defun sample-name-menu-internal (sample-frame sample-name)
  "Called only from the sample-menu."
  (ecase *sample-name-menu-mode*
    ((:DELETE :SAVE :ADD) sample-name)
    (:APPEND
     (push sample-name *samples-being-appended*)
     (send (send sample-frame :status) :display-append-list))
    (NIL
     (let* ((display (send sample-frame :display)))
       (setf (send display :sample) sample-name)
       (send display :reset-markers))))
  sample-name)


(defmethod (sample-frame :grind-sample-menu) ()
  (let* ((old-item-list (send sample-menu :item-list))
	 (old-docstring (if old-item-list
			    (dolist (list old-item-list "")
			      (let* ((string (getf (cdr list) :documentation)))
				(when string (return string))))
			    ""))
	 (result '()))
    (mapcar #'(lambda (name)
		(push `(,(string-capitalize (string name))
			:eval (sample-name-menu-internal ,self ',name)
			:font *sample-editor-font*
			:documentation ,old-docstring)
		      result))
	    *sample-editor-names*)
    (setq result (sort result #'string-lessp :key #'car))
    (setf (send sample-menu :item-list) result)
    nil))


;;; The Mode-commands.

(def-sample-command (:delete-sample-command "Delete Samples" "Remove and deallocate some samples.") ()
  (enter-sample-name-menu-mode self :DELETE "Delete Which Samples: " "Click here to mark this sample for deletion."))

(def-sample-command (:append-samples-command "Append Samples" "Append two or more samples together under a new name.") ()
  (enter-sample-name-menu-mode self :APPEND "Append Which Sample: " "Click here to append this sample."))

(def-sample-command (:save-sample-to-disk-command "Save Samples to Disk" "Save one or more samples to a binary file.") ()
  (enter-sample-name-menu-mode self :SAVE "Save Which Samples: " "Click here to mark this sample for saving."))

(def-sample-command (:add-samples-command "Add Samples" "Create a new sample by adding other samples together.") ()
  (enter-sample-name-menu-mode self :ADD "Select the samples to add together: " "Click here to select this sample."))



(def-sample-command (:abort-mode-command ("ABORT" "Abort Deletion" "Abort Append" "Abort Save" "Abort Add")
		     "Abort whatever sub-mode we are in." #\Abort) ()
  (exit-sample-name-menu-modes self))



(defmacro do-selected-samples ((sample-name) &body body)
  "Execute BODY binding SAMPLE-NAME to each of the samples that are hilighted in the sample menu of SELF."
  `(dolist (,sample-name (mapcar #'(lambda (x) (second (third (third x))))
				 (send sample-menu :highlighted-items)))
     ,@body))



(def-sample-command (:delete-selected-samples-command "Delete Selected Samples"
		     "Delete the samples which are hilighted in the Sample Menu.")
		    ()
  (unwind-protect
      (do-selected-samples (name)
	(delete-one-sample self name))
    (progn
      (exit-sample-name-menu-modes self)
      (send self :grind-sample-menu))))


(def-sample-command (:append-selected-samples-command "Append Selected Samples"
		     "Append the samples which are hilighted in the Sample Menu.") ()
  (unwind-protect
      (when *samples-being-appended*
	(let* ((new-name (prompt-for-keyword "What do you want to call this new sample: "))
	       (array (apply 'append-samples (mapcar #'lookup-sample (nreverse *samples-being-appended*)))))
	  (store-sample new-name array)
	  (send self :grind-sample-menu)))
    (exit-sample-name-menu-modes self)))


(def-sample-command (:append-backspace-command "Backspace" "Undo the previous append selection." #\Rubout) ()
  (setq *samples-being-appended* (cdr *samples-being-appended*))
  (send status :display-append-list)
  )

(def-sample-command (:save-selected-samples-command "Save Selected Samples"
		     "Save the samples which are hilighted in the Sample Menu.") ()
  (unwind-protect
      (let* ((which '()))
	(do-selected-samples (name) (pushnew name which))
	(save-samples-to-disk which))
    (exit-sample-name-menu-modes self)))


(def-sample-command (:add-selected-samples-command "Add Selected Samples"
		     "Add the selected samples together, producing a new sample.") ()
  (unwind-protect
      (let* ((which '()))
	(do-selected-samples (name) (pushnew name which))
	(let* ((new-name (prompt-for-keyword "What do you want to call this new sample: "))
	       (new-sample (apply 'add-multiple-samples
				  (mapcar #'lookup-sample which))))
	  (store-sample new-name new-sample)
	  (when (y-or-n-p "Edit this new sample?")
	    (send display :set-sample new-name)
	    (send display :reset-markers)
	    (send display :refresh :complete-redisplay))
	  (send self :grind-sample-menu)))
    (exit-sample-name-menu-modes self)))


(defun delete-one-sample (sample-frame sample-name)
  (setq *sample-editor-names* (delete sample-name *sample-editor-names*))
  (let* ((array (lookup-sample sample-name)))
    (when array
      (delete-sample sample-name)
      (deallocate-resource 'TV:SOUND-ARRAY array)))
;  (send sample-frame :grind-sample-menu)
  (when (and sample-name (eq sample-name (send (send sample-frame :display) :sample)))
    (setf (send (send sample-frame :display) :sample) nil)
    (send (send sample-frame :display) :refresh)))


(defun save-samples-to-disk (samples)
  (when samples
    (let* ((default (merge-pathnames *default-sample-file*
				     (make-pathname :type (sys:local-binary-file-type) :version :newest
						    :defaults (fs:user-homedir-pathname))))
	   (fs:*name-specified-default-type* (sys:local-binary-file-type))
	   (file (prompt-and-read (list :PATHNAME :DEFAULTS default) "~&Save the samples to what file? (default ~A) "
				  default))
	   (forms ())) ; a list of forms to dump to the binary file.
      (setq *default-sample-file* file)
      
      (dolist (which samples)
	(let* ((array (lookup-sample which))
	       (length (length array)))
	  ;;
	  ;; For each sample, write out a form which will allocate and fill a TV:SOUND-ARRAY, put the array on
	  ;; the name's Plist, and put the name on the list of known sample names.
	  ;;
	  (push `(let* ((array (allocate-resource 'TV:SOUND-ARRAY ,length)))
		   (setf (fill-pointer array) ,length)
		   (replace array ,array)
		   ,(unless (constantp which) `(setq ,which array)) ; If the name is USER::FOO, set USER::FOO to it.
								    ; But not if name is :FOO.
		   (setf (get ',which 'TV:SOUND-ARRAY) array)
		   (pushnew ',which *sample-editor-names*)
		   )
		forms)))
      ;;
      ;; At the front of the file, if *SAMPLE-EDITOR-NAMES* is unbound, proclaim it special and set it to NIL.
      ;;
      (push '(unless (boundp '*sample-editor-names*)
	       (proclaim '(special *sample-editor-names*))
	       (setq *sample-editor-names* nil))
	    forms)
      
      (dump-forms-to-file file forms '(:package "USER"))
      (format t "~&Wrote file ~A.~%" (truename file)))))


;;; The Command Table and Menu definitions.


(build-command-table '*sample-command-table* 'sample-frame
  '(:play-sample-command :play-sample-repeating-command
    :record-sample-command
    :save-sample-region-command
    :delete-sample-command
    :change-magnification-command :change-volume-command :change-pitch-command :reverse-command
    :modify-sample-command
    :reset-all-command
    :save-sample-to-disk-command
    :fetch-sample-command
    :fetch-all-samples-command
    :append-samples-command :add-samples-command
    :sample-quit-command
    :page-forward :page-backward
    :page-forward-short :page-backward-short
    :start-clip-forward :start-clip-backward
    :end-clip-forward :end-clip-backward
    :start-clip-forward-fast :start-clip-backward-fast
    :end-clip-forward-fast :end-clip-backward-fast
    :start-clip-forward-one :start-clip-backward-one
    :end-clip-forward-one :end-clip-backward-one
    :change-clip-start-command :change-clip-end-command :change-clip-length-command
    :goto-beginning :goto-end
    :mark-to-beginning :mark-to-end
    :mouse-click-command
    :redisplay-command :redisplay-all-command :help-command
    ))

(build-command-table '*sample-minimalist-command-table* 'sample-frame
  '(:mouse-click-command
    :redisplay-command :redisplay-all-command
    :abort-mode-command :help-command
    :delete-selected-samples-command
    :append-selected-samples-command
    :save-selected-samples-command
    :add-selected-samples-command
    :append-backspace-command
    ))



(build-menu '*sample-command-menu* 'sample-frame
  :item-list-order
  '(
    :record-sample-command
    :play-sample-command
    :play-sample-repeating-command
    :change-magnification-command
    ""
    :change-volume-command
    :change-pitch-command
    :reverse-command
    :modify-sample-command
    :reset-all-command
    :save-sample-region-command
    ""
    :append-samples-command
    :add-samples-command
    :delete-sample-command
    :save-sample-to-disk-command
    ""
    :fetch-sample-command
    :fetch-all-samples-command
    ""
    :sample-quit-command
    "") :default-item-options '(:font *sample-editor-font*))


(build-menu '*sample-delete-command-menu* 'sample-frame
  :item-list-order
  '("" ""
    :delete-selected-samples-command
    (:abort-mode-command :print-form (:index 1))
    "")
  :default-item-options '(:font *sample-editor-font*))


(build-menu '*sample-append-command-menu* 'sample-frame
  :item-list-order
  '("" ""
    :append-selected-samples-command
    :append-backspace-command
    (:abort-mode-command :print-form (:index 2))
    "")
  :default-item-options '(:font *sample-editor-font*))


(build-menu '*sample-save-command-menu* 'sample-frame
  :item-list-order
  '("" ""
    :save-selected-samples-command
    (:abort-mode-command :print-form (:index 3))
    "")
  :default-item-options '(:font *sample-editor-font*))

(build-menu '*sample-add-command-menu* 'sample-frame
  :item-list-order
  '("" ""
    :add-selected-samples-command
    (:abort-mode-command :print-form (:index 4))
    "")
  :default-item-options '(:font *sample-editor-font*))


;;; Use an icon if running the KSL desktop.

(defvar *sample-editor-icon*
	(make-array '(64 64) :element-type 'bit
	  :displaced-to
	  (make-array 128 :element-type '(unsigned-byte 32)
	    :initial-contents
	    '#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 4294967288 536870911 12 805306368 4 1610612736 4
	       1610612736 4294967044 1627389951 4026516356 1644166599 4026530692 1644150263 679756676 1628509414 679763844
	       1641395575 748962692 1641133126 4278189828 1627389951 4 1610612736 4 1610612736 4 1610612736 4278190084
	       1610613247 260161540 1610843072 528654340 1611023328 1451495428 1611162450 1451634692 1611146066 528748548
	       1611408352 528654340 1610925024 260161540 1610843072 4278190086 1610613247 7 2684354560 5 2684354560 5
	       2684354560 4294966277 2692743167 4194302981 2692742975 5 2684354560 5 2684354560 5 2684354560 4294963205
	       2686451711 4101 2687500288 2147489797 2687500289 3147781 2690652161 3180549 2690783232 117766 1617362944
	       33796 1623326720 1548 1908408320 4294967288 1073741823 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0))))


(defmethod (sample-frame :make-icon) ()
  (make-instance 'w:graphics-icon
		 :window-being-represented self
		 :picture *sample-editor-icon*
		 :reverse-video-p t
		 :borders 1
		 ; :label (list :string w:name :font fonts:hl12b)
		 ))



;;; The top-level function.


(defun sample-editor (&optional edit)
  "Select a new or old Sample Editor window.  An argument sets the current sample."
  (check-type edit (or null symbol))
  (tv:select-or-create-window-of-flavor 'SAMPLE-FRAME)
  (when edit
    (let* ((sample-editor tv:selected-window))
      (assert (get edit 'TV:SOUND-ARRAY) (edit) "~S is not the name of a sampled sound." edit)
      (unless (eq tv:selected-window sample-editor) (send sample-editor :mouse-select))
      (setf (send (send sample-editor :display) :sample) edit)))
  nil)


(tv:remove-system-key #\S)  ;In case one already exists.
(tv:delete-from-system-menu-column :programs "Sample Editor")

(tv:add-system-key #\S 'SAMPLE-FRAME "Edit, play, and modify digitally sampled sounds." t nil "Sample Editor")

(tv:add-to-system-menu-column :PROGRAMS "Sample Editor" '(sample-editor)
 '(:documentation "Select a new or old window to edit, play, and modify digitally sampled sounds.")
 nil)

(setq tv:default-window-types-item-list
      (delete "Sample Editor" tv:default-window-types-item-list :test #'string-equal :key #'car))

(tv:add-window-type "Sample Editor" 'sample-frame "An editor for digitally sampled sounds.")

(compile-flavor-methods SAMPLE-FRAME
			SAMPLE-DISPLAY SAMPLE-SCROLL-BAR SAMPLE-CLIP-BAR
			SAMPLE-MENU SAMPLE-NAME-MENU
			SAMPLE-TYPEIN-WINDOW SAMPLE-STATUS-WINDOW
			)
