;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:USER -*-

;;; Reads X 'bitmap' format files.
;;;
;;; ChangeLog:
;;;
;;; 21 Apr 88  Jamie Zawinski    Created.
;;;  3 Dec 89  Jamie Zawinski    Ported to the Explorer.
;;; 14 Feb 90  Jamie Zawinski 	 Added code to read XBM files.
;;;  9 Mar 90  Jamie Zawinski 	 Optimized SNARF-XBM-DATA for the lispm - instead of reading a character at
;;;				  a time, it operates directly on the stream input buffer.
;;; 27 Apr 90  Jamie Zawinski 	 Fixed BE-STUPID output so that PBM wouldn't complain.
;;;

;;; If Xlib is loaded, defines a function called XLIB-READ-BITMAP-FILE which takes a file name and a display
;;;  as it's arguments.  It constructs a pixmap based on the contents of the file, and returns a new pixmap
;;;  on the DISPLAY.
;;;
;;; If running on a Lisp machine, defines a function called READ-X-BITMAP which produces a 2d bitmap suitable
;;;  for use with BITBLT.  Also defines a function for writing this totally bogus file format.


#+XLIB  (export 'xlib-read-bitmap-file)
#+LISPM (export '(read-x-bitmap write-x-bitmap))

(defun read-xbm-parse-token (stream &optional discard)
  (let* ((string (make-array 20 :element-type 'string-char :fill-pointer 0 :adjustable t)))
    (peek-char t stream) ; discard whitespace.
    (do ()
	((member (peek-char nil stream) '(#\Space #\Tab #\Newline) :test #'char=)
	 (unless discard (copy-seq string)))
      (vector-push-extend (read-char stream) string))))


(defun snarf-#defines (stream)
  "Reads from STREAM until it runs out of C #define statements.
  Returns an association list associng the #defined tokens with their values.
  The keys and vals will be strings."
  (let* ((result nil))
    (do ()
	((char/= (peek-char t stream) #\#))
      ;; Discard the #define
      (read-xbm-parse-token stream t)
      (let ((name (read-xbm-parse-token stream))
	    (val  (read-xbm-parse-token stream)))
	(peek-char #\Newline stream)
	(push (cons name val) result)))
    (nreverse result)))



(defun parse-c-hex (string &optional (start 0) end)
  "Parse from the string a number of the form 0xNN (where N is a hex digit)."
  (declare (string string))
  (unless end (setq end (length string)))
  (let* ((val 0))
    (declare (fixnum val) (optimize speed))
    (assert (char= #\0 (char string start))        () "Not of form 0xNN")
    (assert (char= #\x (char string (incf start))) () "Not of form 0xNN")
    (do* ((i (1+ start) (1+ i)))
	 ((= i end)
	  (values val i))
      (declare (fixnum i))
      (let* ((c (char string i))
	     (n (digit-char-p c 16)))
	(declare (string-char c) (type (or null fixnum) n))
	(if n
	    (setq val (+ (ash val 4) n))
	    (return (values val i)))))))


(defun read-c-hex (stream)
  "Reads a number of the form 0xNN off of STREAM (where N is a hex digit)."
  (let* ((i 0))
    (declare (fixnum i) (optimize speed))
    (assert (char= #\0 (read-char stream)) () "Not of form 0xNN")
    (assert (char= #\x (read-char stream)) () "Not of form 0xNN")
    (loop
      (let* ((c (read-char stream))
	     (n (digit-char-p c 16)))
	(if n
	    (setq i (+ (ash i 4) n))
	    (return (unread-char c stream)))))
    i))


#-LISPM		; a portable version.
(defun snarf-xbm-data (stream into &optional name width pad)
  "Reads the raw data for defining a C bitmap off of STREAM.  INTO should be a 1d array of 8-bit words.  Returns INTO."
  (unless (string= "static" (read-xbm-parse-token stream)) (warn "First word is not 'static'."))
  (unless (string= "char" (read-xbm-parse-token stream)) (warn "Second word is not 'char'."))
  (let* ((read-name (read-xbm-parse-token stream)))
    (when name
      (let* ((should-be (string-append name "_bits[]")))
	(unless (string= read-name should-be)
	  (warn "Third word is not '~A'" should-be)))))
  (peek-char t stream) ; discard whitespace.
  (unless (char= #\= (read-char stream)) (warn "No '=' character!"))
  (peek-char t stream)
  (unless (char= #\{ (read-char stream)) (warn "No '{' character!"))
  (do (#+LISPM (i 0 (1+ i)))
      ((char= #\} (peek-char t stream)))
    #+LISPM (when (and width (= i width))	; This evil hack is so we can keep the thing mod32 wide.
	      (dotimes (i pad) (vector-push 0 into))
	      (setq i 0))
    (vector-push (read-c-hex stream) into)
    (let* ((c (peek-char t stream)))
      (if (char/= #\, c)
	  (unless (char= #\} c) (warn "Missing a ',' character!"))
	  (read-char stream))))
  (read-char stream) ; discard the }
  (unless (char= #\; (read-char stream)) (warn "No terminating ';' character found!"))
  into)


#+LISPM		; an optimized version.
(defun snarf-xbm-data (stream into &optional name width pad)
  "Reads the raw data for defining a C bitmap off of STREAM.  INTO should be a 1d array of 8-bit words.  Returns INTO."
  (unless (string= "static" (read-xbm-parse-token stream)) (warn "First word is not 'static'."))
  (unless (string= "char" (read-xbm-parse-token stream)) (warn "Second word is not 'char'."))
  (let* ((read-name (read-xbm-parse-token stream)))
    (when name
      (let* ((should-be (string-append name "_bits[]")))
	(unless (string= read-name should-be)
	  (warn "Third word is not '~A'" should-be)))))
  (peek-char t stream) ; discard whitespace.
  (unless (char= #\= (read-char stream)) (warn "No '=' character!"))
  (peek-char t stream)
  (unless (char= #\{ (read-char stream)) (warn "No '{' character!"))
  (setf (fill-pointer into) 0)
  (let* ((max-bytes (array-dimension into 0))
	 (deferred 0)
	 (tick -1) 
	 (bufsiz (min (* max-bytes 6) 4096))
	 (line (make-array bufsiz :element-type 'string-char :fill-pointer 0)))
    (declare (optimize speed)
	     (fixnum max-bytes deferred bufsiz tick)
	     (string line))
    (loop
      (setf (fill-pointer line) deferred)
      (unless (zerop deferred)
	(sys:copy-array-portion line (- bufsiz deferred) bufsiz line 0 deferred))
      
      (multiple-value-bind (ignore eof) (send stream :string-in nil line deferred)
	(setf (fill-pointer line) (or (position #\, line :test #'char= :from-end t) bufsiz))
	(setq deferred (- bufsiz (fill-pointer line)))
	
	(let ((i 0) (n 0) (len 0))
	  (declare (fixnum i n len)
		   (inline parse-c-hex))
	  (setq len (fill-pointer line))
	  (unless (zerop len)
	    (loop
	      ;; move i to the start of the next token.
	      (setq i (or (position #\0 line :start i :test #'char=)
			  (return)))
	      (multiple-value-setq (n i) (parse-c-hex line i len))
	      (when width
		(incf tick)
		(when (= tick width)
		  (dotimes (i pad) (vector-push 170 into))
		  (setq tick 0)))
	      (vector-push n into)
	      )))
	(when (or eof (= (fill-pointer into) max-bytes))
	  (return)))))
  into)

#+comment
(zwei:defcom com-snarf "foo" () 
  (let* ((p (getf (zwei:line-plist (zwei:bp-line (zwei:point))) :pathname))
	 w h)
    (multiple-value-setq (b w h) (user:read-x-bitmap p))
    (when (equalp #\Return (view-bitmap b nil nil w h))
      (let* ((n (pathname-name (getf (zwei:line-plist (zwei:bp-line (zwei:point))) :pathname))))
	(setq n (string-trim "." (subseq n 0 (position #\. n))))
	(w:write-bit-array-file (print (string (make-pathname :defaults "jwz:jwz.new;*.bitmap#>" :name n :type "bitmap")))
				b w h))))
  zwei:dis-none)
#+comment
(zwei:set-comtab zwei:*mode-comtab* '(#\~ com-snarf))

(defun read-bitmap-file-1 (stream &optional (pixmap-p t))
  "Reads the data from FILE and returns six values:
     a vector suitable for making a pixmap of;
     the width of this pixmap;
     the height of this pixmap;
     the X of the hotspot of this pixmap (or NIL);
     the Y of the hotspot of this pixmap (or NIL);
     the name of this pixmap (a string).
  If running on a Lispm, the optional argument PIXMAP-P says whether to 
  make a pixmap vector or a normal bitmap."
  #-LISPM (unless pixmap-p (error "Don't know how to read anything but pixmaps in this implementation."))
  (let* ((defines (snarf-#defines stream))
	 (name nil)
	 (width nil) (height nil)
	 (x-hot nil) (y-hot nil))
    ;;
    ;; This way-too-large form is for reading the bitmap's name, size, and hotspot.
    ;;
    (dolist (cons defines)
      (let* ((key (car cons))
	     (val (cdr cons))
	     (last-underscore (position #\_ key :from-end t))
	     (second-to-last-underscore (and last-underscore (position #\_ key :from-end t :end last-underscore)))
	     (possible-name-1 (and last-underscore (subseq key 0 last-underscore)))
	     (possible-name-2 (and last-underscore (subseq key 0 second-to-last-underscore)))
	     (possible-slot-1 (and last-underscore (subseq key (1+ last-underscore))))
	     (possible-slot-2 (and second-to-last-underscore (subseq key (1+ second-to-last-underscore))))
	     new-name slot)
	(cond ((and possible-slot-1
		    (member possible-slot-1 '("width" "height") :test #'string=))
	       (setq slot possible-slot-1
		     new-name possible-name-1))
	      ((and possible-slot-2
		    (member possible-slot-2 '("x_hot" "y_hot") :test #'string=))
	       (setq slot possible-slot-2
		     new-name possible-name-2)))
	(when slot
	  (cond ((and name new-name (string/= name new-name))
		 (error "Multiple pixmaps, ~S and ~S, defined in one file.  I can't cope with that."
			name new-name))
		(t (setq name new-name)))
	  (let* ((num (parse-integer val)))
	    (cond ((string= slot "width")
		   (when width (warn "Width of pixmap ~S being redefined from ~D to ~D." width num))
		   (setq width num))
		  ((string= slot "height")
		   (when height (warn "Height of pixmap ~S being redefined from ~D to ~D." height num))
		   (setq height num))
		  ((string= slot "x_hot")
		   (when x-hot (warn "X hotspot of pixmap ~S being redefined from ~D to ~D." x-hot num))
		   (setq x-hot num))
		  ((string= slot "y_hot")
		   (when y-hot (warn "Y hotspot of pixmap ~S being redefined from ~D to ~D." y-hot num))
		   (setq y-hot num)))))))
    ;;
    ;; This reads the actual bits.
    ;;
    (cond #+LISPM
	  ((not pixmap-p)
	   (let* ((w8  (+  8 (*  8 (floor (1- width)  8))))
		  ;(w16 (+ 16 (* 16 (floor (1- width) 16))))
		  (w32 (+ 32 (* 32 (floor (1- width) 32))))
		  (bitmap (make-array (list height w32) :element-type 'bit))
		  (vector (make-array (ceiling (* (array-dimension bitmap 0) (array-dimension bitmap 1)) 8)
				      :element-type '(unsigned-byte 8) :displaced-to bitmap :fill-pointer 0)))
;	     (print (list (list width height) w8 w16 w32))
;	     (snarf-xbm-data stream vector name (unless (= w16 w32) (/ w16 8)) (/ (- w32 w8) 8))
	     (snarf-xbm-data stream vector name (unless (= w8 w32) (/ w8 8)) (/ (- w32 w8) 8))
	     (values bitmap width height x-hot y-hot name)))
	  (t (let* ((vector (make-array (ceiling (* width height) 8) :element-type '(unsigned-byte 8)
					:fill-pointer 0)))
	       (snarf-xbm-data stream vector name)
	       (values vector width height x-hot y-hot name))))))


#+XLIB
(defun xlib-read-bitmap-file (file display)
  "Returns two values: a pixmap and the name of that pixmap (a string)."
  (multiple-value-bind (vector width height x-hot y-hot name)
      (with-open-file (stream file :direction :input :element-type 'string-char)
	(read-bitmap-file-1 stream))
    (let* ((win (root-window display))
	   (p (create-pixmap :width width :height height :depth 1 :drawable win))
	   (gcontext (create-gcontext :drawable win)))
      (unwind-protect
	  (put-raw-image p gcontext data-vector :x 0 :y 0 :width 32 :height 32 :depth 1 :format :xy-pixmap)
	(free-gcontext gcontext))
      (values p name x-hot y-hot))))

#+LISPM
(defun read-x-bitmap (file)
  "Returns six values: a bitmap (suitable for use with BITBLT), the width and height of the meaningful area of that
  bitmap (since the real width will always be mod32), the name of that bitmap (a string), and an x/y hotspot."
  (multiple-value-bind (array width height x-hot y-hot name)
      (with-open-file (stream file :direction :input :element-type 'string-char)
	(read-bitmap-file-1 stream nil))
    (values array width height name x-hot y-hot)))

#+LISPM
(when (fboundp 'SYS:DECOMPRESSING-OPEN)
  
  (compiler-let ((sys:compile-encapsulations-flag t))
    (sys:advise-within READ-X-BITMAP OPEN :around decompress nil
      ;;
      ;; Instead of the real open, call decompressing-open.
      ;;
      (apply 'sys:decompressing-open sys:arglist)))
  )



;;; Writing

#+LISPM
(defun dump-c-bitmap-1 (bitmap name stream &optional hotspot-x hotspot-y be-stupid)
  (declare (optimize (speed 3) (safety 0)))
  (let* ((w (array-dimension bitmap 1))
	 (h (array-dimension bitmap 0))
	 (i (/ (* w h) 8))
	 (indirect (make-array i :element-type '(unsigned-byte 8) :displaced-to bitmap)))
    (declare (fixnum w h i))
    (format t "~&The bitmap occupies about ~D bytes of memory, so the file will be about ~D bytes long."
	    i (* 6 i))
    (format stream "~&#define ~A_width ~D~%#define ~A_height ~D~%" name w name h)
    (when (and hotspot-x hotspot-y)
      (format stream "#define ~A_x_hot ~D~%#define ~A_y_hot ~D~%" name hotspot-x name hotspot-y))
    (format stream "static char ~A_bits[] = {~% " name)
    (let* ((count 0)
	   (*print-base* 16))
      (dotimes (j (1- i))
	(let* ((n (aref indirect j)))
	  (declare (fixnum n))
	  (write-string "0x" stream)
	  (when (and be-stupid (< n 16))
	    (write-char #\0 stream))
	  (princ n stream)
	  (if be-stupid
	      (write-string ", " stream)
	      (write-string "," stream)))
	(when (> (incf count) 12)
	  (terpri stream)
	  (princ #\Space stream)
	  (setq count 0))))
    (let ((n (aref indirect (1- i))))
      (if (and be-stupid (< n 16))
	  (format stream "0x0~X};~%" n)
	  (format stream "0x~X};~%" n))))
  bitmap)

#+LISPM
(defun write-x-bitmap (bitmap name filename &optional hotspot-x hotspot-y be-stupid)
  "Write the BITMAP to the FILENAME in the format acceptable to 'xsetroot' and 'bitmap' - that is, as C source code.
  NAME can be any string.  It doesn't matter if all you're using this file for is a background pattern.
  If HOTSPOT-X and HOTSPOT-Y are specified, they should be integers less than the width and height of the
  bitmap respectively.  If you plan on using this bitmap for a cursor, you should specify these.
  If BE-STUPID is true, then we output as much superfluous whitespace as the X11 'bitmap' program would."
  (with-open-file (stream filename :direction :output)
    (dump-c-bitmap-1 bitmap name stream hotspot-x hotspot-y be-stupid)
    (truename stream)))


#+LISPM		; I'll leave the non-lispm version of this as an exercise for the reader...
(defun read-xbm-bitmap-1 (stream)
  "Reads a bitmap in Bennet Yee's \"xbm\" format.  STREAM should be 8 bits."
  (let* ((magic-number (char-int #\!)))
    (assert (and (= magic-number (read-byte stream))
		 (= magic-number (read-byte stream)))
	    () "This is not an XBM-format file.")
    (let* ((width (dpb (read-byte stream) (byte 8 8)
		       (read-byte stream)))
	   (height (dpb (read-byte stream) (byte 8 8)
			(read-byte stream)))
	   (bitmap (make-array (list height (+ 32 (* 32 (floor (1- width) 32))))
			       :element-type 'bit))
	   (indirect (make-array (* height (/ width 8)) :element-type '(unsigned-byte 8)
				 :displaced-to bitmap)))
      (dotimes (i (/ (length indirect) 2))
	(let* ((c1 (read-byte stream))		; network byte order - flip the baby.
	       (c2 (read-byte stream)))
	  (setf (aref indirect (ash i 1)) c2
		(aref indirect (1+ (ash i 1))) c1)))
      bitmap)))

#+LISPM
(defun read-xbm-bitmap (pathname)
  "Reads a bitmap in Bennet Yee's \"xbm\" format."
  (with-open-file (stream pathname :direction :input :characters nil :byte-size 8)
    (read-xbm-bitmap-1 stream)))
