;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:TV; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER ADOBE-COURIER14B HL12B CPTFONTBI HL12I) -*-

;1;; File "3READ-SUN-RASTER*".*
;1;; Reads sun-format raster files.  Doesn't work on multi-plane/color files yet, since I don't have any.*
;1;; This code should be portable Common Lisp (modulo machine-dependant byte-order tweaking).*
;1;; There are conditionally-compiled 7way*-cool efficiency hacks for Lispms.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   25 Oct 89*	1Jamie Zawinski*	1Created.*
;1;;*   19 Nov 89*	1Jamie Zawinski*	1Added code for reading byte-encoded rasterfiles, since I finally know the format.*
;1;;*  115 Nov 89*	1Jamie Zawinski *	1Fixed it to work with non-byte-encoded rasterfiles that were not a multiple of 32 wide.*
;1;;*  118 Nov 89*	1Jamie Zawinski *	1Added calls to the 5sys:%buffer-char-map* miscop for faster bit-order reversal.*
;1;;*
;1;; 5File Format:**
;1;;*
;1;;*	1From the Unix man page for 4rasterfile(5)**
;1;;*
;1;;*	1A rasterfile is composed of three* 1parts:* 1first,* 1a* 1header*
;1;;*	1containing* 18* 1integers;* 1second,* 1a (possibly empty) set of*
;1;;*	1colormap values; and third, the pixel image, stored* 1a* 1line* 1at*
;1;;*	1a time, in increasing y order. * 1The image is layed out* 1in*
;1;;*	1the file as in a memory pixrect. * 1Each line of the* 1image* 1is*
;1;;*	1rounded up to the nearest 16 bits.*
;1;;*
;1;;*	1The header is defined by the following structure:*
;1;;*
;1;;*	4struct rasterfile {*
;1;;*		4int ras_magic;*
;1;;*		4int ras_width;*
;1;;*		4int ras_height;*
;1;;*		4int ras_depth;*
;1;;*		4int ras_length;*
;1;;*		4int ras_type;*
;1;;*		4int ras_maptype;*
;1;;*		4int ras_maplength;*
;1;;*		4};*
;1;;*
;1;;*	1The 4ras_magic* field always contains the following constant:*
;1;;*
;1;;*		4#define* 4RAS_MAGIC 0x59a66a95*
;1;;*
;1;;*	1The 4ras_width*, 4ras_height*, and 4ras_depth* fields*
;1;;*	1contain* 1the* 1image's* 1width* 1and* 1height in pixels, and its*
;1;;*	1depth in bits* 1per pixel, respectively. * 1The* 1depth* 1is* 1either* 11*
;1;;*	1or* 18,* 1corresponding* 1to* 1standard* 1frame* 1buffer* 1depths.* 1The*
;1;;*	4ras_length1 field contains the length in bytes of** 1the*
;1;;*	1image* 1data. * 1For an unencoded image, this number is*
;1;;*	1computable from* 1the 4ras_width*, 4ras_height*, and*
;1;;*	4ras_depth1 fields, but for** 1an* 1encoded* 1image* 1it* 1must be*
;1;;*	1explicitly stored in order to be* 1available without decoding*
;1;;*	1the image itself. * 1Note* 1that* 1the* 1length* 1of* 1the* 1header* 1and*
;1;;*	1of the (possibly empty) colormap* 1values are not* 1included*
;1;;*	1in* 1the* 1value* 1of* 1the* 4ras_length* 1field; it is* 1only the*
;1;;*	1image data length. * 1For historical reasons,* 1files of type*
;1;;*	4RT_OLD1 will usually** 1have* 1a* 10* 1in* 1the* 4ras_length* 1field,*
;1;;*	1and software expecting to encounter* 1such* 1files should be*
;1;;*	1prepared to compute the* 1actual* 1image* 1data* 1length* 1if*
;1;;*	1needed. * 1The 4ras_maptype* and* 4ras_maplength1 fields**
;1;;*	1contain the type and length in* 1bytes of the colormap*
;1;;*	1values,* 1respectively. * 1If* 4ras_maptype* 1is* 1not* 4RMT_NONE*
;1;;*	1and* 1the 4ras_maplength* is not 0, then the* 1colormap*
;1;;*	1values* 1are* 1the* 4ras_maplength* 1bytes* 1immediately* 1after*
;1;;*	1the* 1header. * 1These* 1values are either* 1uninterpreted* 1bytes*
;1;;*	1(usually* 1with* 1the* 4ras_maptype* 1set* 1to 4RMT_RAW*) or the*
;1;;*	1equal length red, green* 1and blue vectors, in that order*
;1;;*	1(when* 1the* 4ras_maptype* 1is* 4RMT_EQUAL_RGB1). ** 1In the*
;1;;*	1latter case, the 4ras_maplength* must* 1be three times the*
;1;;*	1size in bytes of any one of the vectors.*
;1;;*
;1;;*	1Each line of the image is rounded out to a multiple of 16*
;1;;*	1bits.*  1This corresponds to the rounding convention used*
;1;;*	1by the memory* 1pixrect* 1package (/usr/include/pixrect/memvar.h)*
;1;;*	1of the SunWindows system.*
;1;;*
;1;;*	1The 4ras_encoding* field (always set to 0 by Sun's supported*
;1;;*	1software)* 1was renamed to 4ras_length* in* 1release 2.0. * 1As a*
;1;;*	1result, rasterfiles* 1of type 0 generated* 1by the old software*
;1;;*	1claim to have 0 length; for* 1compatibility, code reading*
;1;;*	1rasterfiles must be prepared to* 1compute the* 1true length*
;1;;*	1from the width, height, and depth* 1fields.*
;1;;*
;1;;*
;1;;*	1[ 2You may notice that the above information says nothing of the format**
;1;;*	2  of the data in a raster file written with 4ras_encoding* = 4byte_encoded.**
;1;;*	2  Well, neither do 6any* of the Sun manuals.  So I hunted around, and *
;1;;*	2  eventually found the following description in 5Sun-Spots Digest*, *
;1;;*	2  Volume 6, Issue 84: 1]**
;1;;*
;1;;*	4From:    *jpm%green@lanl.gov (Pat McGee)
;1;;*	4Subject: *Re: Format for byte encoded rasterfiles (1)
;1;;*
;1;;*	1The format is composed of many sequences of variable length records.*
;1;;*	1Each record may be 1, 2, or 3 bytes long.*
;1;;*
;1;;*	1 o  If the first byte is not 30x80*, the record is one byte long, and *
;1;;*	1    contains a pixel value.  Output 1 pixel of that value.*
;1;;*	1 o  If the first byte is 30x80* and the second byte is zero, the record*
;1;;*	1     is two bytes long.  Output 1 pixel with value 0x80.*
;1;;*	1 o  If the first byte is 30x80*, and the second byte is not zero, the *
;1;;*	1     record is three bytes long.  The second byte is a count and the *
;1;;*	1     third byte is a value.  Output (2count+1*) pixels of that value.*
;1;;*
;1;;*	1A run is not terminated at the end of a scan line.  So, if there are *
;1;;*	1three lines of red in a picture 100 pixels wide, the first run will *
;1;;*	1be 30x80 0xff 0x2<red>**, and the second will be 30x80 0x2b 0x2<red>**.*
;1;;*
;1;;*		1Pat McGee, jpm@lanl.gov*


(export '(read-sun-raster-file))

(defconstant 4sun-raster-magic-number-lsb* #x59a6 "2The first half of the magic number which opens a Sun Raster File.*")
(defconstant 4sun-raster-magic-number-msb* #x6a95 "2The second half of the magic number which opens a of a Sun Raster File.*")

(defun 4read-sun-raster-file-header *(stream)
  "2Check the magic number, and return the parameters for the image in this file.*"
  (declare (values width height depth length type maptype maplength))
  (labels ((read-short () (dpb (read-byte stream) (byte  8  8) (read-byte stream)))
	   (read-long  () (dpb (read-short)       (byte 16 16) (read-short))))
    (let* ((n1 (read-short))
	   (n2 (read-short)))
      (unless (and (= n1 SUN-RASTER-MAGIC-NUMBER-LSB)
		   (= n2 SUN-RASTER-MAGIC-NUMBER-MSB))
	(cerror "2Try anyway3.**"
		"2Bad magic number: this is not a Sun Raster File.3~%~**
		3Desired magic number:      ~8,'0b ~8,'0b ~8,'0b ~8,'0b ~%~*
		3Encountered magic number:  ~8,'0b ~8,'0b ~8,'0b ~8,'0b ~%*"
		(ldb (byte 8 8) SUN-RASTER-MAGIC-NUMBER-LSB) (ldb (byte 8 0) SUN-RASTER-MAGIC-NUMBER-LSB)
		(ldb (byte 8 8) SUN-RASTER-MAGIC-NUMBER-MSB) (ldb (byte 8 0) SUN-RASTER-MAGIC-NUMBER-MSB)
		(ldb (byte 8 8) n1) (ldb (byte 8 0) n1)
		(ldb (byte 8 8) n2) (ldb (byte 8 0) n2)
		))
      (let* ((width  (read-long))
	     (height (read-long))
	     (depth  (read-long))
	     (length (read-long))
	     (type   (let* ((i (read-long)))
		       (case i
			 (0       :OLD)			;1 Raw pixrect image in 68000 byte order*
			 (1       :STANDARD)		;1 Raw pixrect image in 68000 byte order*
			 (2       :BYTE-ENCODED)	;1 Run-length compression of bytes*
			 (#xFFFF  :EXPERIMENTAL)	;1 Reserved for testing*
			 (t (cerror "2Proceed anyway.*" "2Unknown type code, ~4,'0X*" i) i))))
	     (maptype (let* ((i (read-long)))
			(case i
			  (0    :NONE)		;1 ras_maplength is expected to be 0.*
			  (1    :EQUAL-RGB)	;1 red[ras_maplength/3], green[], blue[]*
			  (2    :RAW)
			  (t (cerror "2Proceed anyway.*" "2Unknown maptype code, ~4,'0X*" i) i))))
	     (maplength (read-long)))
	(when (and (eq maptype :none) (not (eql maplength 0)))
	  (cerror "2Proceed anyway.*" "2MapType was NONE, but MapLength is nonzero (~D).*" maplength))
	(values width height depth length type maptype maplength)))))


(defvar 4*bit-flippage* *(let* ((a (make-array 256 :element-type '(unsigned-byte 8))))
			   (dotimes (i 256)
			     (let* ((flip-byte 0))
			       (dotimes (j 8)
				 (setq flip-byte (dpb (ldb (byte 1 (- 7 j)) i) (byte 1 j) flip-byte)))
			       (setf (aref a i) flip-byte)))
			   a)
  "2A table for quickly reversing the order of bits in a byte.*")

(proclaim '(type (array (unsigned-byte 8) (256)) *bit-flippage*))


(defun 4read-sun-raster-file-bits *(stream pixel-width pixel-height format &optional invert-p into-array)
  "2Read the rest of the file into a bitmap; use INTO-ARRAY if it is exactly the right size, otherwise create a new one.*"
  (declare (optimize (speed 3) (safety 0) (space 0)))
  ;1; If we're running on a Lisp machine, we can indirect an 8-bit array to a 1-bit array, which makes the bit-storing process faster;*
  ;1; we get 8 pixels in one memory-store, instead of having to do one store per pixel.  On a lispm, we can also suck the entire file*
  ;1; into the array in one fell swoop, instead of calling read-byte for each 8 bits in the file.  So we conditionally compile all of this in.*
  ;1;*
  ;1; Unfortunately, on an Explorer, reading a byte-encoded file takes twice as long as reading a non-byte-encoded file.  That's because*
  ;1; I haven't thought of a neat trick that will let me snarf the whole file at once in the byte-encoded case; right now, it still reads from*
  ;1; the file 8 bits at a time.*
  ;1;*
  (flet (#-LISPM
	 (4setmap-byte* (byte bitmap word-x bit-y)
	   "2Bitmap is 2d, 1 bit per pixel; byte is an 8-bit quantity that will be stored into 8 consecutive bits of the bitmap.*
	2    Word-X is expressed in 8-bit bytes, and Word-Y is expressed in pixels.  The order of the bits in byte will be*
	2   reversed before it is stored, to accomidate the Sun's byte ordering.*"
	   (when invert-p (setq byte (lognot byte)))
	   (let* ((bit-x (* word-x 8)))
	     (dotimes (i 8)
	       (setf (aref bitmap bit-x bit-y) (ldb (byte 1 (- 7 i)) byte))
	       (incf bit-x))))
	 #+LISPM
	 (4setmap-byte* (byte bytemap word-x bit-y)
	   "2Bytemap is 2d, 8 bits per pixel.  The order of the bits in byte will be reversed before it is stored, to accomidate the Sun's byte ordering.*"
	   ;1; This function is only called when we are doing a byte-encoded image.  For uncompressed images we can be more efficient.*
	   (declare (type (unsigned-byte 8) byte)
		    (fixnum word-x bit-y)
		    (optimize speed))
	   (setq byte (the (unsigned-byte 8) (aref (the vector *bit-flippage*) byte)))
	   (setf (aref bytemap bit-y word-x) byte))
	 )
    (let* ((byte-width (ceiling pixel-width 8))
	   (scanline-rounded-width (+ 16 (* 16 (floor (1- pixel-width) 16))))
	   (dimensions #+LISPM (list pixel-height (+ 32 (* 32 (floor (1- pixel-width) 32))))
		       #-LISPM (list scanline-rounded-width pixel-height))
	   (bitmap (if (and into-array
			    (or (equal dimensions (array-dimensions into-array))
				(progn (warn "2Supplied array has dimensions ~S but ~S required; creating new array anyway.*"
					     (array-dimensions into-array) dimensions)
				       nil))
			    (or (typep into-array '(ARRAY BIT))
				(progn (warn "2Supplied array has element-type ~A but ~A required; creating new array anyway.*"
					     (array-element-type into-array) 'BIT)
				       nil))
				)
		       into-array
		       (make-array dimensions :element-type 'BIT :initial-element (if invert-p 0 1))))
	   #+LISPM (bytemap (make-array (list pixel-height (round (array-dimension bitmap 1) 8))
					:element-type '(unsigned-byte 8) :displaced-to bitmap))
	   )
      (declare (type (array bit 2) bitmap)
	       #+LISPM (type (array (unsigned-byte 8) 2) bytemap)
	       (fixnum byte-width scanline-rounded-width))
      (ecase format
	((:OLD :STANDARD)
	 #-LISPM (dotimes (row (1- pixel-height))
		   (declare (optimize (speed 3) (safety 0) (space 0)))
		   (dotimes (column byte-width)
		     (setmap-byte (read-byte stream) bitmap column row)))
	 
	 #+LISPM (cond ((zerop (rem scanline-rounded-width 32))
			;1;*
			;1; Oh joy, the image is stored as a multiple of 32 wide, so we can read it in one fell swoop.*
			;1; Make a 1 dimensional 8-bit array displaced to the bitmap.  This means that the two arrays*
			;1; occupy the same memory (this is the Lisp analog of C type-casting).*
			;1; Pass this array to :string-in; this will read one array-full of bytes from the file.*
			;1; Then iterate over the array to reverse the bit-order, since Lispms store their bytes in the*
			;1; opposite order of Suns.*
			;1;*
			(let* ((a (make-array (* pixel-height (round (array-dimension bitmap 1) 8))
					      :element-type '(unsigned-byte 8) :displaced-to bitmap)))
			  (declare (type (array (unsigned-byte 8) 1) a))
			  ;1; This makes the low-level stream handler suck the file directly into the array, bypassing*
			  ;1; an awful lot of method-calling overhead.  After we have the array, then we reverse the bit-order.*
			  (send stream :string-in t a)
			  
                          ;1; *(dotimes (i (length a))
                          ;1;*  (setf (aref a i) (aref (the (array (unsigned-byte 8) (256)) *bit-flippage*)
                          ;1;*                         (aref a i))))
			  ;1;*
			  ;1; Thanks to Paul Fuqua for telling me about this miscop which does the above, but seven times faster...*
			  (sys:%buffer-char-map a 0 (length a) a 0 (length a) *bit-flippage* #xFF 0)
			  ))
		       (t
			;1;*
			;1; The image is stored in the file in scanlines which are a multiple of 16 instead of 32.*
			;1; This means we must read it a scanline at a time.  C'est la vie...*
			;1;*
			;1; Make a 1 dimensional 8-bit array displaced to the bitmap.  This time, the 1d-array is only as long*
			;1; as a scanline in the file.  Call :line-in to snarf one scanline-full from the file, and then 7move* the start-pointer*
			;1; of the displaced array (so that the 0 index of the 1d array is not the same as the 0,0 index of the 2d array).*
			;1; The scanline array will now be length+16 bytes farther along in the 2d array, and we are ready to read the*
			;1; next scanline.  Then iterate over the array to reverse the bit-order, since Lispms store their bytes in the*
			;1; opposite order of Suns.*
			;1;*
			(let* ((bitmap-byte-width (floor (array-dimension bitmap 1) 8))
			       (scanline-byte-width (floor scanline-rounded-width 8))
			       ;1; Note: if you plan to change the displaced-index-offset of an array, you must pass it in as 0 initially.*
			       ;1; Otherwise an array-header word is not allocated for the index-offset, and you can't change it.*
			       (scanline (make-array scanline-byte-width :element-type '(unsigned-byte 8)
						     :displaced-to bitmap :displaced-index-offset 0))
			       (translated-type (array-type scanline))
			       )
			  (declare (type (array (unsigned-byte 8) 1) scanline)
				   (fixnum bitmap-byte-width scanline-byte-width))
			  (dotimes (i pixel-height)
			    (sys:change-indirect-array scanline translated-type scanline-byte-width
						       bitmap (* i bitmap-byte-width))
			    (send stream :string-in t scanline)
			    ;1; Replaced by a later call to 5sys:%buffer-char-map*.*
			    ;1; *(dotimes (j scanline-byte-width)
			    ;1; *  (setf (aref scanline j) (aref (the vector *bit-flippage*) (aref scanline j)))))
			    ;1;*
			    ;1; If we do this here, since we are using a shorter array, we only get a speedup of 1.5 times because of*
			    ;1; the function invocation overhead.  Do it after the whole thing has been read, and get 7x.*
			    ;1;*
			    ;1; *(sys:%buffer-char-map scanline 0 scanline-byte-width scanline 0 scanline-byte-width
                            ;1; *                      *bit-flippage* #xFF 0)
			    )
			  (let* ((a (make-array (* pixel-height (round (array-dimension bitmap 1) 8))
						:element-type '(unsigned-byte 8) :displaced-to bitmap)))
			    (declare (type (array (unsigned-byte 8) 1) a))
			    (sys:%buffer-char-map a 0 (length a) a 0 (length a) *bit-flippage* #xFF 0))
			  )))
	 )
	(:BYTE-ENCODED
	 (let* ((word-x 0)
		(bit-y 0)
		byte)
	   (macrolet ((4push-byte* (byte)
			`(progn
			   (setmap-byte ,byte #-LISPM bitmap #+LISPM bytemap word-x bit-y)
			   (incf word-x)
			   (when (>= word-x byte-width)
			     (setq word-x 0 bit-y (1+ bit-y))))))
	     (do* () ((>= bit-y pixel-height))
	       (unless (setq byte (read-byte stream nil nil))
		 (cerror "2Return the bitmap-so-far.*" "2Premature EOF while reading rasterfile.*")
		 (return))
	       (if (= byte #x80)   ;1 Magic number!*
		   (let* ((next (read-byte stream)))
		     (if (zerop next)						;1 #x80 #x00*	1= exude a literal #x80*
			 (push-byte #x80)
			 (let* ((value (read-byte stream)))			;1 #x80 <x> <y>*	1= exude <y> <x> times.*
			   (dotimes (i (1+ next)) (push-byte value)))))
		   (push-byte byte)))))))					;1 <x>*		1= exude <x>*
      ;1;*
      ;1; This is faster than calling 5lognot* each time we store a byte.*
      #+LISPM (when invert-p (bitblt TV:ALU-SETCA pixel-width pixel-height bitmap 0 0 bitmap 0 0))
      bitmap)))


(defun 4read-sun-raster-file *(pathname &optional invert-p into-array)
  "2Reads a Sun Raster format image from the given file.
  If INVERT-P is true, then the image will be inverted as it is read.
  Returns two values: a bitmap, and its intended width and height in pixels
  (The bitmap's width is always a multiple of 32, so that it can be used with BITBLT.)
  If INTO-ARRAY is an array *of the appropriate size* then we will read the file into that.
  If it is the wrong size, then a warning is printed and a new array is returned anyway.*"
  (declare (values bitmap width height))
  (with-open-file (stream pathname :direction :input :characters nil :byte-size 8)
    (multiple-value-bind (width height depth length type maptype maplength)
			 (read-sun-raster-file-header stream)
      (declare (ignore length maplength))
      (unless (eql 1 depth) (error "2Don't know how to read raster files of more than one bitplane.*"))
      (unless (member type '(:OLD :STANDARD :BYTE-ENCODED) :test #'eq)
	(error "2Don't know how to read ``~:(~A~)'' raster files.*" type))
      (unless (eq maptype :NONE)
	(error "2This raster file has a colormap.  I can't deal.*"))
      (values (read-sun-raster-file-bits stream width height type invert-p into-array)
	      width height))))

#+LISPM
(defun 4read-and-show-sun-raster-file *(pathname &optional (window *terminal-io*) invert-p into-array)
  "2Just like READ-SUN-RASTER-FILE, but draws it on the screen when done.*"
  (multiple-value-bind (bitmap w h) (read-sun-raster-file pathname invert-p into-array)
    (send window :bitblt TV:ALU-SETA w h bitmap 0 0 0 0)
    (values bitmap w h)))
