;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-

; qattraction by John Nguyen  MIT  1988  johnn@hx.lcs.mit.edu

; based on attraction program by John Pezaris at MIT

(defconstant *qattraction-num-frames* 200)

(defvar qorb-list nil)
(defvar qorb-list-length nil)
(defvar queue-x (make-array (list 10 *qattraction-num-frames*) :initial-element 0))
(defvar queue-y (make-array (list 10 *qattraction-num-frames*) :initial-element 0))

(defun qattraction (&optional (stream *terminal-io*) (length 3) (times nil)
		   (max-size 16))

  (setq length (min length 10))

  (macrolet
    ((x-acc (orb) `(first  ,orb))
     (y-acc (orb) `(second ,orb))
     (x-vel (orb) `(third  ,orb))
     (y-vel (orb) `(fourth ,orb))
     (x-pos (orb) `(fifth  ,orb))
     (y-pos (orb) `(sixth  ,orb))
     (mass  (orb) `(seventh ,orb))
     (size  (orb) `(eighth  ,orb))
     (x-pix (orb xlim) `(min ,xlim (max 0 (floor (+ (/ ,xlim 2.) (x-pos ,orb))))))
     (y-pix (orb ylim) `(min ,ylim (max 0 (floor (+ (/ ,ylim 2.) (y-pos ,orb))))))
     (increment (place value)
		"increment a value"
		(declare (arglist place value &key test test-not key))
		(let ((pl (gensym))
		      (val (gensym)))
		  (si:sublis-eval-once `((,val . ,value))
				       (si:sublis-eval-once `((,pl . ,place))
							    `(values (setf ,place (+ ,pl ,val))))))))

  
  (multiple-value-bind (xlim ylim)
      (send stream :inside-size)
    
    (let* ((xlim/2 (/ xlim 2))
	   (ylim/2 (/ ylim 2))
	   (-xlim/2 (* -1.0 xlim/2))
	   (-ylim/2 (* -1.0 ylim/2)))
      
      (let* ((diameter (- (min xlim/2 ylim/2) 50))
	     (head 0)
	     (old-head 1)
	     (list (progn
		     (if (not (and qorb-list (= length qorb-list-length)))
			 (progn
			   (setq qorb-list-length length
				 qorb-list (make-list qorb-list-length))
			   (dotimes (n length)
			     (setf (nth n qorb-list) (list 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0)))))
		     (dotimes (n length)
		       (let ((new-size (min 16 (+ 8 (random (- max-size 9)))))
			     (orb (nth n qorb-list)))
			 (setf (x-acc orb) 0.0)
			 (setf (y-acc orb) 0.0)
			 (setf (x-vel orb) (/ (- 6.0 (random 11)) 8.0))
			 (setf (y-vel orb) (/ (- 6.0 (random 11)) 8.0))
			 (setf (x-pos orb) (* diameter (cos (* n (/ (* 2.0 pi) length)))))
			 (setf (y-pos orb) (* diameter (sin (* n (/ (* 2.0 pi) length)))))
			 (setf (mass orb) (* 7.0 new-size new-size 10.0))
			 (setf (size orb) new-size)
			 ))
		     qorb-list)))
	
	(send stream :clear-screen)
	
	(loop until (if times (= (setf times (1- times)) 0) NIL)
	      do

	      (setq head old-head)
	      (setq old-head (mod (1+ old-head) *qattraction-num-frames*))
	      (dotimes (l length)
		;; calculate attraction of this orb to the other orbs, this will set the x-acc and y-acc.
		(let ((new-x-acc 0.0) (new-y-acc 0.0)
		      (orb (nth l list)))
		  (dotimes (ll length)
		    (let ((other-orb (nth ll list)))
		      (if (not (eq orb other-orb))	; don't do it to yourself !!!
			  (let* ((x-dist (- (x-pos other-orb) (x-pos orb)))
				 (y-dist (- (y-pos other-orb) (y-pos orb)))
				 (dist^2 (+ (* x-dist x-dist)
					    (* y-dist y-dist)))
				 (dist (sqrt dist^2))
				 (new-acc 0.0)
				 (new-acc/dist 0.0))
			    (if (> dist 0.1)
				(progn
				  (setq new-acc (* (/ (mass other-orb) dist^2)
						   (cond ((< dist 100.0) -1.0)
;							 ((< dist 3.0)   -10.0)
;							 ((< dist 1.0)   -100.0)
							 (t 1.0))))
				  (setq new-acc/dist (/ new-acc dist))
				  (increment new-x-acc (* new-acc/dist x-dist))
				  (increment new-y-acc (* new-acc/dist y-dist))
				  )
				(progn
				  (increment new-x-acc (- 5.0 (random 10.0)))
				  (increment new-y-acc (- 5.0 (random 10.0)))
				  )
				)
			    ))
		      ))
		  (setf (x-acc orb) new-x-acc)
		  (setf (y-acc orb) new-y-acc)
		  ) 
		)
	      
	      (dotimes (l length)
		(let* ((orb (nth l list)))
		  
		  ;; set the new velocities
		  (increment (x-vel orb) (x-acc orb))
		  (increment (y-vel orb) (y-acc orb))

		  ;; make sure that if things get too fast, they slow down!
		  (if (< 15.0 (abs (x-vel orb)))
		      (progn
			(setf (x-vel orb) (* (x-vel orb) 0.9))
			(setf (x-acc orb) 0.0)))

		  (if (< 15.0 (abs (y-vel orb)))
		      (progn
			(setf (y-vel orb) (* (y-vel orb) 0.9))
			(setf (y-acc orb) 0.0)))
		  
		  ;; set the new positions
		  (increment (x-pos orb) (x-vel orb))
		  (increment (y-pos orb) (y-vel orb))
		  
		  ;; check for maximum position in x direction
		  (if (<= (- xlim/2 max-size 4) (x-pos orb))
		      (progn
			(setf (x-pos orb) (- xlim/2 max-size 5))
			(setf (x-vel orb) (* -1.0 (max 0.1 (abs (x-vel orb))))))
		      (if (>= (+ -xlim/2 1) (x-pos orb))
			  (progn
			    (setf (x-pos orb) (+ -xlim/2 2))
			    (setf (x-vel orb) (* 1.0 (max 0.1 (abs (x-vel orb))))))))
		  
		  ;; check for maximum position in y direction
		  (if (<= (- ylim/2 max-size 4) (y-pos orb))
		      (progn
			(setf (y-pos orb) (- ylim/2 max-size 5))
			(setf (y-vel orb) (* -1.0 (max 0.1 (abs (y-vel orb))))))
		      (if (>= (+ -ylim/2 1) (y-pos orb))
			  (progn
			    (setf (y-pos orb) (+ -ylim/2 2))
			    (setf (y-vel orb) (* 1.0 (max 0.1 (abs (y-vel orb))))))))
		  
		  (setf (aref queue-x l head) (x-pix orb xlim)
			(aref queue-y l head) (y-pix orb ylim))
		  
		  )
		)
	      (dotimes (l length)
		(let ((l2 (mod (1+ l) length)))
		  (tv:prepare-sheet (stream)
		    (si:%draw-line (aref queue-x l old-head) (aref queue-y l old-head)
				   (aref queue-x l2 old-head) (aref queue-y l2 old-head)
				   w:erase t stream))))
	      (dotimes (l length)
		(let ((l2 (mod (1+ l) length)))
		  (tv:prepare-sheet (stream)
		    (si:%draw-line (aref queue-x l head) (aref queue-y l head)
				   (aref queue-x l2 head) (aref queue-y l2 head)
				   w:normal t stream))))
	      ))
      )
    )
  
  ))

(when (boundp '*screen-saver-hacks-list*)
  (pushnew 'qattraction *screen-saver-hacks-list*))
