;;; -*- Mode:Lisp; Package:USER; Base:10; Syntax:Common-Lisp -*-; 

;;; A simplified version of the UNIX fgrep command. This uses an inefficient algorithm 
;;; but it's network bound anyway so it may not be worth fixing.
;;; JRL, Sept. 1985

(defresource fgrep-buffer (size)
  :constructor (make-array size :element-type 'string-char :fill-pointer 0))


(defmacro expand-wildcards ((bind file-name) &body body)
  `(loop for expand-wildcard-paths in
	       (cdr (fs:directory-list (merge-pathnames ,file-name "x.lisp#>") :fast))
	 for ,bind = (car expand-wildcard-paths)
	 collect
      . ,body))
			  

;;; Figures out what the right stream is then pass on to fgrep1.

(defun fgrep (word &optional (files *standard-input*) &key lisp region)
"The fgrep command does a substring search for the given word in the given file.
 If given the keyword :lisp it will print out the whole lisp expression 
 containing the word rather than just the line on which word appears. If given
 the keyword :region it print out the newline delimited region in which the
 word appears"
  (if (symbolp word) (setq word (symbol-name word)))
  (if (atom files) (setq files (list files)))
  (loop for path in files do
	(if (streamp path)
	    (fgrep1 word path lisp region)
	    (expand-wildcards (file path)
	      (with-open-file (stream file)
		(fgrep1 word stream lisp region))))))


;;; Choose a read function and a print function for the selected mode (line, lisp, or
;;; region) and pass on to fgrep-string.

(defun fgrep1 (word stream lisp-read-mode region-read-mode)
  (let ((buffer (allocate-resource 'fgrep-buffer 500))
	(line-in-p (send stream :operation-handled-p :line-in)))
    (let ((read-function
	    (cond (lisp-read-mode #'(lambda (&optional use-buffer-p)
				       (fgrep-read-lisp-form stream buffer use-buffer-p)))
		  (region-read-mode #'(lambda (&optional use-buffer-p)
					(fgrep-read-region stream buffer use-buffer-p)))
		  (t #'(lambda (&optional use-buffer-p)
			 (fgrep-read-line stream buffer use-buffer-p line-in-p)))))
	  (print-function
	    (cond (lisp-read-mode #'(lambda (buffer) (fgrep-print-no-filename buffer stream)))
		  (region-read-mode #'(lambda (buffer) (fgrep-print-no-filename buffer stream)))
		  (t #'(lambda (buffer) (fgrep-print buffer stream))))))
      (unwind-protect
	  (fgrep-string word read-function print-function)
	(deallocate-resource 'fgrep-buffer buffer)))))

(defvar *null-string* (make-array '(0) :element-type 'string-char :fill-pointer 0)
  "need a null string that has a fill pointer")

;;; Read a buffer of input from the stream, return second value non-nil if reached end of 
;;; file. Reads into the buffer given in stream. If the use-buffer-p argument 
;;; is non-nil then we make sure to use the buffer of the fgrep-stream

(defun fgrep-read-lisp-form (stream buffer use-buffer-p)
  (declare (ignore use-buffer-p))
  (let ((next (read stream nil :fgrep-eof)))
    (if (eq next :fgrep-eof)			;if we hit the end of file
	(values *null-string* :fgrep-eof)	;return null string and indication of eof
	(setf (fill-pointer buffer) 0)		;clear out old data
	;;write into the buffer the expression we just read
	(with-output-to-string (str buffer) (pprint next str))
	(values buffer nil))))			;return new buffer and no end of file found

(defun fgrep-read-line (stream buffer use-buffer-p line-in-p)
  (cond (line-in-p
	 (setf (fill-pointer buffer) 0)
	 ;; just send a :line-in if the stream handles it
	 (multiple-value-bind (line eof) (send stream :line-in)
	   (if use-buffer-p
	       (progn (vector-push-portion-extend line buffer)
		      (values buffer eof))
	       (values line eof))))
	(t (setf (fill-pointer buffer) 0)
	   ;; else we must simulate :line-in using tyi
	   (loop for ch = (read-char stream nil :fgrep-eof)
		 until (or (eq ch :fgrep-eof)
			   (and (char= ch #\newline)
				(vector-push-extend ch buffer)))
		 finally (return (values buffer (eq :fgrep-eof ch)))
		 do
	     (vector-push-extend ch buffer)))))

(defun fgrep-read-region (stream buffer use-buffer-p)
  (declare (ignore use-buffer-p))
  (setf (fill-pointer buffer) 0)
  (loop for ch = (read-char stream nil :fgrep-eof) ;get next character
	for ch2 = nil
	do
    (if (eq ch :fgrep-eof) (return (values buffer t)))
    (when (and (char= ch #\newline)		        ;hit newline
	       (setq ch2 (read-char stream nil nil))
	       (char= ch2 #\newline)		        ;and next char is newline
	       (not (= (fill-pointer buffer) 0)))	;and buffer is not empty
      (vector-push-extend ch buffer)		        ;put in the newline
      (return (values buffer nil)))
    (if ch2 (unread-char ch2 stream))
    (vector-push-extend ch buffer)))


;;; The explorer does not have an vector-push-portion-extend function

(defun vector-push-portion-extend (from-vector to-vector
				  &optional (start 0)
				       (end (if (array-has-fill-pointer-p from-vector)
						(fill-pointer from-vector)
						(array-dimension from-vector 0))))
  (loop for i from start to (1- end) do
    (vector-push-extend (aref from-vector i) to-vector)))


(defun fgrep-string (word read-function print-function)
  (loop with no-more = nil			;iterate over lines in stream
	until no-more
	do
    (multiple-value-bind (buffer end-of-file) (funcall read-function)
      (if end-of-file (setq no-more t))
      (when (search word buffer)
	(funcall print-function buffer)))))


;;; Print out the line and the filename (if possible)

(defun fgrep-print (buffer stream)
  (if (send stream :operation-handled-p :pathname)
      (format *standard-output* "~%~A: " (send (send stream :pathname) :name)))
  (fgrep-print-normal buffer))

(defun fgrep-print-no-filename (buffer stream)
  (declare (ignore stream))
  (format *standard-output* "~%")
  (fgrep-print-normal buffer))

(defun fgrep-print-normal (buffer)
  (format *standard-output* "~A" buffer))
