;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:MAIL; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B) -*-

;1;; Call the function 3BIFF* to put a little icon in your wholine telling you when you have new mail.*
;1;; If 3FONTS:BIFF* is loaded, the icon is a mailbox whose flag goes up when you have mail.  If this font is not loaded, then*
;1;; the icon is blank when you don't have mail, and an asterisk in 3FONTS:CMR18* when you do.*
;1;; The function 3FROM* will print out all lines of your inboxes starting with "From:" so you can decide whether you really want to read it...*
;1;; Terminal-Control-F will do this in a pop-up window.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;    27 Jul 89*	1Jamie Zawinski*	1 Created.*
;1;;    9 Aug 89*	1Jamie Zawinski *	1 Rewrote.*
;1;;   10 Aug 89*	1Jamie Zawinski *	1 Implemented 5from*.  Moved to 5mail:* package.*
;1;;   24 Aug 89*	1Jamie Zawinski *	1 Made 3BIFF-LOOP* never hang in 5mail:probe-inboxes* for more than 5 minutes.*
;1;;*				1 Sometimes the silly TCP code will stay in "TCP Close" or something forever.*
;1;;*   115 Sep 89*	1Jamie Zawinski *	 1Changed it to not put the blinker on the who-line-documentation window, but rather to add a new*
;1;;*				 1 window to the wholine, with the biff-blinker on that.*
;1;;*   118 Sep 89*	1Jamie Zawinski *	 1It was sometimes staying on when the ScreenSaver was on - fixed.*
;1;;*   127 Sep 89*	1Jamie Zawinski *	 1Made biff wakeup as soon as the screensaver toggles state.*
;1;;*   130 Nov 89*	1Jamie Zawinski *	 1Added advice so that Terminal-0-C will complement the biffer as well.*
;1;;*   111 Dec 89*	1Jamie Zawinski *	 1Moved new version of 5mail:probe-inboxes* here from 5mail-modeline.lisp*.*
;1;;*

(export '(biff nobiff from pop-up-from biff-wakeup
	  probe-inboxes ckmail *notify-on-mail-arrival*))

(defvar 4*biff-interval* *30 "2Biff will probe for mail every this-many seconds.*")

(defvar 4*biff-font**	4 *   nil	"2The font for the BIFF icons.*")
(defvar 4*biff-no-mail-char* *nil	"2The character of *BIFF-FONT* to draw when there is no mail.*")
(defvar 4*biff-mail-char* *  4 *nil "2The character of *BIFF-FONT* to draw when there is mail.*")

(unless *biff-font*
  (if (boundp 'fonts:biff)
      (setq *biff-font* 'fonts:biff
	    *biff-no-mail-char* #\A
	    *biff-mail-char* #\B)
      (setq *biff-font* 'fonts:cmr18
	    4*biff-no-mail-char* *#\Space
	    4*biff-mail-char** #\*)))


(defvar 4*biff-wakeup* *nil "2If you set this flag to T, then biff will wake up right away instead of waiting for *biff-interval* to expire.*")

(defun 4biff-wakeup *()
  "2Make biff sit up and take notice.*"
  (setq 4*biff-wakeup** t))

(defflavor 4biff-blinker* () (tv:character-blinker))
(defsubst 4biff-blinker-p *(blinker) (typep blinker 'biff-blinker))

(defflavor 4who-line-biff-window *()
	   (tv:who-line-sheet)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defun 4install-who-line-biff-window *()
  "2Shuffle the who-line windows so that there is room for BIFF over on the right side.*"
  (let* ((max-width (tv:sheet-width tv:who-line-screen))
	 (height (tv:sheet-height tv:who-line-documentation-window))
	 (biff (get tv:who-line-screen 'BIFF-PANE))
	 (biff-width 40))
    (unless biff
      (setq biff (make-instance '4WHO-LINE-BIFF-WINDOW* :superior tv:who-line-screen))
      (setf (get tv:who-line-screen 'BIFF-PANE) biff))
    (send tv:who-line-documentation-window :set-size (- max-width biff-width) height)
    (send biff :set-edges (tv:sheet-width tv:who-line-documentation-window)
			  (tv:sheet-y-offset tv:who-line-documentation-window)
			  (+ (tv:sheet-width tv:who-line-documentation-window) biff-width)
			  (+ (tv:sheet-y-offset tv:who-line-documentation-window) height))
    
    (send biff :set-background-color (send tv:who-line-documentation-window :background-color))
    (send biff :set-char-aluf	     (send tv:who-line-documentation-window :char-aluf))
    (send biff :set-reverse-video-p  (send tv:who-line-documentation-window :reverse-video-p))
    
    (send biff :expose)
    biff))


(defun 4remove-who-line-biff-window *()
  "2If the who-line windows have been shuffled to make room for BIFF, undo that.*"
  (let* ((biff (get tv:who-line-screen 'BIFF-PANE)))
    (when (and biff (send biff :exposed-p))
      (send biff :deactivate)
      (send tv:who-line-documentation-window :set-size
	    (+ (tv:sheet-width tv:who-line-documentation-window) (tv:sheet-width biff))
	    (tv:sheet-height tv:who-line-documentation-window)))
    biff))

(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise TV:KBD-COMPLEMENT :before flip-biff nil
    (when (car sys:arglist)
      (let* ((biffer (find 'who-line-biff-window (tv:sheet-inferiors tv:who-line-screen) :test #'eq :key #'type-of)))
	(when biffer
	  (send biffer :set-reverse-video-p (not (send biffer :reverse-video-p)))))))
  )

(defun 4biff-loop *(icon-when-no-mail)
  "2Main loop of the biff process; does not return.*"
  (let* ((window (install-who-line-biff-window))
	 (blinker (make-instance 'biff-blinker
				 :sheet window :font *biff-font*
				 :character *biff-no-mail-char*
				 :visibility :off))
    	 (list (send window :blinker-list))
	 (new (remove-if #'biff-blinker-p list))
	 (width (tv:sheet-inside-width window))
	 (height (tv:sheet-inside-height window))
	 (offset-from-right height))
    (send blinker :set-cursorpos (- width offset-from-right) 0)
    (push blinker new)
    (sys:set-in-instance window 'TV::BLINKER-LIST new)
    (send window :refresh)
    (loop
      (biff-loop-internal blinker window icon-when-no-mail))))


(defun 4biff-refresh-internal *(blinker window mail-p icon-when-no-mail)
  "2Redraw the biff wholine-sheet.  BLINKER is the character blinker on that window; MAIL-P is whether there is currently unread mail.*"2 *
  (let* ((visibility (if (or icon-when-no-mail mail-p) :on :off)))
    (send blinker :set-character (if mail-p *biff-mail-char* *biff-no-mail-char*))
    (send blinker :set-visibility visibility)
    (cond ((and (boundp 'tv:*the-screen-is-black*) tv:*the-screen-is-black*)
	   ;1;*
	   ;1; If the screensaver is on, then we are tricky - flash it on, then off again, making sure it erases to BLACK.*
	   ;1; But only do this if we have mail.  If we don't have mail, do the erase part anyway, just in case we got refreshed somehow.*
	   ;1;*
	   (when mail-p
	     (send window :refresh)
	     (sleep 1 "3Blink that Icon*"))
	   (send blinker :set-visibility :off)
	   (let* ((color (send window :background-color)))
	     (unwind-protect
		 (progn (send window :set-background-color w:black)
			(tv:prepare-sheet (window)
			  (sys:%draw-rectangle (tv:sheet-width window) (tv:sheet-height window) 0 0 tv:alu-back window)))
	       (send window :set-background-color color))))
	  
	  ;1; If the wholine is disabled, do nothing.*
	  (tv:inhibit-who-line nil)
	  
	  ;1; Otherwise, redraw.*
	  (t (send window :refresh)))))


(defun 4biff-loop-internal *(blinker window icon-when-no-mail)
  "2Probe for mail, then wait.*"
  (catch 'TIMEOUT
    (setq *biff-wakeup* nil)
    (let* ((mail-p (with-timeout ((* 60 60 5) (throw 'TIMEOUT nil))
		     (ckmail))))
      ;1;*
      ;1; Redraw the window, then wait until the interval has expired, or the screensaver has come on or gone off, or someone has set*
      ;1; the biff-wakeup flag.  Then redraw the window again, and return.  We do this because probing for mail takes time, and the *
      ;1; process-wait takes time, but redrawing doesn't.*
      (biff-refresh-internal blinker window mail-p icon-when-no-mail)
      (process-wait "3Biff Sleep*"
		    #'(lambda (start-time interval screen-was-black)
			(or *biff-wakeup*
			    (and (boundp 'tv:*the-screen-is-black*)
				 (not (eq tv:*the-screen-is-black* screen-was-black)))
			    (>= (time-difference (time) start-time)
				interval)))
		    (time)
		    (round (* 60 *biff-interval*))
		    (and (boundp 'tv:*the-screen-is-black*) tv:*the-screen-is-black*))
      (biff-refresh-internal blinker window mail-p icon-when-no-mail))))

;1;;*
;1;; Put advice on 3ZWEI:READ-INBOX* so that when the user reads their mail, the little flag will go down right away.*
;1;;*
(sys:advise zwei:read-inbox :after wake-biff nil
  (biff-wakeup))


(defun 4biff *(&optional (icon-when-no-mail t))
  "2Puts an icon in the wholine window that will tell you whether you have mail.
  If ICON-WHEN-NO-MAIL is T, then the icon is always there, but draws differently when you have mail.
  If it is NIL, then the icon is only visible when there is mail.*"
  (nobiff)
  (process-run-function '(:name "3Biff Background*" :restart-after-boot t :restart-after-reset t)
			#'biff-loop icon-when-no-mail))

(defun 4nobiff *()
  "2Turns off BIFF.*"
  (dolist (proc sys:all-processes)
    (when (string= "3Biff Background*" (send proc :name))
      (send proc :kill)))
  (let* ((window (4remove-who-line-biff-window*)))
    (when window
      (sys:set-in-instance window 'TV::BLINKER-LIST
			   (remove-if #'biff-blinker-p (send window :blinker-list)))))
  nil)


;1;; From and Pop-Up-From.*
;1;;*


(defun 4from *(&rest inboxes)
  "2Say who your mail is from.  INBOXES defaults to all of the inboxes on *MAIL:*INBOX-PROBE-LIST*2.*"
  (unless inboxes (setq inboxes (mapcar #'car mail:*inbox-probe-list*)))
  (let* ((any-mail nil))
    (dolist (file inboxes)
      (block ONE
	(let* ((printed nil))
	  (condition-resume `(error :ABORT-INBOX errorp ("2Give up on listing inbox \"~A\"*" ,file)
				    ,#'(lambda (ignore) (return-from ONE)))
	    (with-open-file (stream file :direction :input :if-does-not-exist nil)
	      (when stream
		(loop
		  (let* ((line (read-line stream nil nil)))
		    (unless line (return-from ONE))
		    (when (string-equal "3From:*" (the simple-string line) :end2 5)
		      (setq any-mail t)
		      (unless printed
			(format t "3~2&In inbox ~A:~%*" (send stream :truename))
			(setq printed t))
		      (princ "3  *")
		      (write-line line))))))))))
    any-mail))


(defun 4pop-up-from *(&optional ignore)
  "2Pop up a window running MAIL:FROM.*"
  (using-resource (window tv:pop-up-finger-window)
    (setf (tv:sheet-truncate-line-out-flag window) 1)
    (send window :set-process current-process)
    (let* ((inboxes (mapcar #'car mail:*inbox-probe-list*)))
      (send window :set-label
	    (cond ((null inboxes) "3No inboxes known.*")
		  ((null (cdr inboxes)) (format nil "3Listing mail in inbox ~A*" (car inboxes)))
		  (t (format nil "3Listing mail in inboxes ~A~{, ~A~}*" (car inboxes) (cdr inboxes)))))
      (tv:window-call (window :deactivate)
	(let* ((*terminal-io* window)
	       (*debug-io* window))
	  (and inboxes (or (from) (format window "3~2&No new mail.*"))))
	(format window "3~2%Any character to flush.*")
	(send window :any-tyi)))))

(tv:add-terminal-key #\Control-F 'pop-up-from "2Pop up a window listing the From: lines of all of the messages in your inboxes.*")


;1;; A better version of MAIL:PROBE-INBOXES.*
;1;; This one will tell you 1: whether you have mail, and 2: whether new mail has just arrived.*

(defvar 4*notify-on-mail-arrival** t "2If T, send a notification when new mail arrives.*")

(defun 4probe-inboxes-avoid-password-query* (host &rest ignore)
  "2Tries to find a password for the given host, and if it can't, throws to *MAIL:PROBE-INBOX-ATTEMPT2.
  This is so we avoid prompting the user for a password down in PROBE-INBOXES.*"
  (let* ((uname (cdr (assoc host fs:user-unames)))
	 (rest (and uname (fs:lookup-password-etc uname host))))
    (cond ((and uname (car rest))
	   (apply #'values uname rest))
	  (t
	   (throw 'MAIL:PROBE-INBOX-ATTEMPT nil)))))


(defun 4probe-inboxes* ()
  "2  Look at all of the known inboxes and see if any of them have mail or have new mail.
  When a file is noticed to have gained mail, and *MAIL:*NOTIFY-ON-MAIL-ARRIVAL* 2is non-NIL, this function
  will send a notification about the arrival.
  Returns two values:
    A list of those inboxes which have changed since the last time this function was called, and
    a list of those inboxes which are not empty.*"
  (declare (values inboxes-with-new-mail inboxes-with-any-mail))
  (let* ((inboxes-with-mail nil)
	 (inboxes-with-new-mail nil))
    (condition-call-if (not *debug-mailer*) (condition)
	(dolist (cons *inbox-probe-list*)
	  (let* ((path (first cons))		;1 The inbox we are checking.*
		 (previous-date (second cons))  ;1 The write-date of this file the last time we looked (or NIL if we've*
		 )				;1  never looked, or it didn't used to exist).*
	    (when path
	      (catch 'MAIL::PROBE-INBOX-ATTEMPT   ;1 If the remote machine wants a password, bug out.*
		(let* ((fs:*generic-login-function* 'probe-inboxes-avoid-password-query)
		       (dirlist (fs:directory-list path :noerror))
		       (file-plist (and (consp dirlist) (cdr (second dirlist)))))
		  
		  (unless (or (errorp dirlist) (null dirlist))  ;1 we couldn't read the file for some reason.  Do nothing.*
		    (let* ((new-date (getf file-plist :creation-date))
			   (new-size (getf file-plist :length-in-bytes))
			   ;1;*
			1    *;1; When the file exists and has something in it, it "has mail".*
			   ;1; When the file has mail, and its date is newer than the last time we looked (or we never looked before, or*
			   ;1; the file didn't exist when we last looked) then it "has new mail".*
			   ;1;*
			   (has-mail (and new-size (plusp new-size)))
			   (has-new-mail (and has-mail (or (null previous-date)
							   (and new-date (> new-date previous-date))))))
		      (and new-date (setf (second cons) new-date))
		      (when has-mail (pushnew path inboxes-with-mail :test #'equalp))
		      (when has-new-mail
			(pushnew path inboxes-with-new-mail :test #'equalp)
			(when *notify-on-mail-arrival* (tv:notify nil "3New mail in ~A*" path)))
		      )))))))
      ((mail::handle-condition-p condition)
       condition))
    
    (values inboxes-with-new-mail inboxes-with-mail)))


(defun 4ckmail* ()
  "2Whether the logged in user has unread mail.  See also *#'MAIL:PROBE-INBOXES2.*"
  (multiple-value-bind (new-mail old-mail) (probe-inboxes)
    (declare (ignore new-mail))
    (not (null old-mail))))


(add-initialization "3Forget Inboxes*" '(setq mail:*inbox-probe-list* nil) :logout)
