;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:ZWEI -*-

;;; File "AUDIO-TAPE-INTERFACE"
;;; Yow, am I hacking ZMACS yet??
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;;  3 Oct 88  Jamie Zawinski    Created.
;;;   7 Nov 88 Jamie Zawinski    WRITE-ONE-ALBUM was doing STRING-TRIM on non-strings and quietly losing on the number 32!!
;;;   7 Feb 89 Jamie Zawinski 	 Added *SIGNATURE-STRING*, made COM-PRINT-ALBUMS take prefix arguments.
;;;	       			 Added band-name and album-name completion in the Album Edit Buffer.
;;;   9 Feb 89 Jamie Zawinski 	 Made the point default to being after the Band Name: field.  Added
;;;				  *TAPES-OF-NEWLY-CREATED-ALBUMS*, and modified Control-U M-X Print Albums to use it.
;;;  14 Feb 89 Jamie Zawinski 	 Made the tape-line parser not discard leading whitespace before the song name.
;;;	       			 Made the Print Albums menu have lots of cross-referencing information in the wholine.
;;;


(use-package "TAPE")


;;;
;;; Grinding the Tape Edit Buffer.
;;;


(defvar *album-edit-buffer* nil)

(defun in-font (string font bp)
  "Inserts the string at the BP in the FONT, and returns a region of the string inserted."
  (let* ((bp-1 (copy-bp bp :normal)))
    (insert-moving bp (in-current-font string font))
    (make-interval bp-1
		   (copy-bp bp :normal))))


(defvar *album-edit-template-1*
	'(("Band Name:" . :nocr)
	  ("Album Name:" . :nocr)
	  ("Year:" . t)
	  ("" . :none)
	  ("Songs:" . nil)
	  ("-------" . :none)
	  ))

(defvar *album-edit-template-2*
	'(("Band Name:" . :nocr)
	  ("Album Name 1:" . :nocr)
	  ("Year:" . t)
	  ("" . :nocr)
	  ("" . :nocr)
	  ("Album Name 2:" . t)
	  ("Songs:" . nil)
	  ("-------" . :none)
	  ))

(defvar *album-edit-template-double*
	'(("Band Name:" . :nocr)
	  ("Album Name:" . :nocr)
	  ("Year:" . t)
	  ("" . :none)
	  ("Songs, Side A:" . nil)
	  ("---------------" . :none)
	  ("" . :none)
	  ("Songs, Side B:" . nil)
	  ("---------------" . :none)
	  ))

(defun select-album-buffer (&optional noclear (n-albums 1) double-p)
  (when (or (null *album-edit-buffer*)
	    (get *album-edit-buffer* :killed))
    (when *album-edit-buffer* (send *album-edit-buffer* :kill)) ; make sure..
    (setq *album-edit-buffer*
	  (make-instance 'zmacs-buffer :name "*Album Edit Buffer*"))
    (set-buffer-fonts *album-edit-buffer* '(fonts:cptfont fonts:tr12b)))
  (make-buffer-not-read-only *album-edit-buffer*)
  (setf (get *album-edit-buffer* 'DOUBLE-ALBUM-P) double-p)
  (setf (get *album-edit-buffer* 'TWO-ALBUMS-P) (if double-p nil (/= n-albums 1)))
  (setf (get *album-edit-buffer* 'RO-REGIONS) '())
  (send *album-edit-buffer* :select)
  (unless noclear
    (zwei:delete-interval *album-edit-buffer*)
    (dolist (cons (if double-p
		      *album-edit-template-double*
		      (if (= n-albums 1)
			  *album-edit-template-1*
			  *album-edit-template-2*)))
      (let* ((nocr-p (eq (cdr cons) :nocr))
	     (none-p (eq (cdr cons) :none))
	     (same-line-p (eq (cdr cons) t))
	     (string (car cons)))
	(push (in-font string 1 (point))
	      (get *album-edit-buffer* 'RO-REGIONS))
	(cond (nocr-p ;(insert-moving (point) #\Space)
		      (insert-moving (point) #\Tab)
		      (insert-moving (point) #\Tab))
	      (none-p (insert-moving (point) #\Return))
	      (same-line-p (insert-moving (point) #\Tab)
			   (insert-moving (point) #\Return))
	      (t (insert-moving (point) #\Return))))))
  nil)


;;; 
;;; Filling and Parsing the Album Edit Buffer.
;;;

(defun grind-album-to-buffer (album region)
  (let* ((artist (album-artist album))
	 (name (album-name album))
	 (year (album-year album))
	 (songs (album-songs album))
	 (other-name (album-other-name album))
	 (double-p (double-album-p album))
	 (other-songs (and double-p (album-songs2 album))))
    
    (select-album-buffer nil (if other-name 2 1) double-p)
    
    (with-bp (mark (interval-first-bp region) :moves)
      
      (flet ((search-and-insert (target string-or-list)
	       "Go to the beginning of the buffer and search for TARGET.  Insert STRING after it."
	       (move-bp mark (zwei:search (interval-first-bp region) target))
	       (cond ((listp string-or-list)
		      (move-bp mark (end-line (forward-line mark 1 t) 0 t))
		      (dolist (cons string-or-list)
			(let* ((name (car cons))
			       (length (cdr cons)))
			  (insert-moving mark #\Newline)
			  (insert-moving mark name)
			  (when length
			    (insert-moving mark #.(make-string 2 :initial-element #\Tab))
			    (insert-moving mark length)))))
		     (t
		      (move-bp mark (forward-char mark 1 t))
		      (insert-moving mark string-or-list)
		      (insert-moving mark #\Space)))))
	
	(search-and-insert "Band Name:" artist)
	(search-and-insert "Year:" year)
	(cond (other-name
	       (search-and-insert "Album Name 1:" name)
	       (search-and-insert "Album Name 2:" other-name))
	      (t
	       (search-and-insert "Album Name:" name)))
	(cond (double-p
	       (search-and-insert "Songs, Side A:" songs)
	       (search-and-insert "Songs, Side B:" other-songs))
	      (t
	       (search-and-insert "Songs:" songs))))))
  nil)


(defun parse-album-from-buffer (interval n-albums double-p)
  (let* (artist name year songs other-name other-songs)
    (with-bp (mark (interval-first-bp interval) :moves)
      
      (flet ((snarf-field (target &optional delimiter-string)
	       "Go to the beginning of the buffer and search for TARGET.  Snarf the string after it."
	       ;;
	       ;; Find the target string and move over any whitespace following it.
	       ;;
	       (move-bp mark (or (zwei:search (interval-first-bp interval) target)
				 (return-from SNARF-FIELD)))
	       (move-bp mark (forward-over '(#\Space #\Tab) mark (interval-last-bp interval)))
	       (with-bp (mark2 mark :normal)   ; MARK2 is moved to the end of the field.
		 (if delimiter-string
		     ;;
		     ;; If there is a delimiter-string, find it or the end of the buffer.
		     ;;
		     (let* ((len (length delimiter-string)))
		       (let* ((new-bp (zwei:search mark2 delimiter-string)))
			 (if new-bp
			     (move-bp mark2 (forward-char new-bp (- len) t)) ; SEARCH leaves us at end.  Goto beginning.
			     (move-bp mark2 (interval-last-bp interval)))))  ; Otherwise, goto end of buffer.
		     ;;
		     ;; Otherwise, move MARK2 to the next tab or newline.
		     ;;
		     (let* ((end-mark (zwei:search-set mark2 '(#\Tab #\Newline))))
		       (if end-mark
			   (move-bp mark2 end-mark)
			   (cerror "" ""))))
		 ;;
		 ;; Move both marks to ditch leading and trailing whitespace.
		 ;;
		 (move-bp mark  (forward-over  '(#\Space #\Tab #\Newline #\-) mark  mark2))
		 (move-bp mark2 (backward-over '(#\Space #\Tab #\Newline #\-) mark2 mark))
		 (string-interval mark mark2))))
	
	(setq artist (snarf-field "Band Name:"))
	(setq year (snarf-field "Year:"))
	(cond ((= n-albums 2)
	       (setq name (snarf-field "Album Name 1:"))
	       (setq other-name (snarf-field "Album Name 2:")))
	      (t
	       (setq name (snarf-field "Album Name:"))))
	(cond (double-p
	       (setq songs       (snarf-field "Songs, Side A:" "Songs, Side B:"))
	       (setq other-songs (snarf-field "Songs, Side B:" "FOOBAR")))
	      (t
	       (setq songs (snarf-field "Songs:" "FOOBAR"))))))

    (setq songs (song-string-to-alist songs))
    (setq other-songs (song-string-to-alist other-songs))
    (if double-p
	(make-double-album :artist artist :name name :year year :songs songs :songs2 other-songs)
	(make-album :artist artist :name name :year year :songs songs :other-name other-name))))


(defun song-string-to-alist (string)
  (let* ((result '()))
    (do* ((start -1 pos)
	  (pos (position #\Newline string :start 0)
	       (and start (position #\Newline string :start (1+ start)))))
	 ((null start))
      (incf start)
      (let* ((substring (subseq string start pos)))
	(setq substring (string-right-trim '(#\Space #\Tab #\Newline) substring))
	(let* ((first-nonblank (position-if-not #'(lambda (x) (member x '(#\Space #\Tab) :test #'char-equal))
						substring))
	       (tab-pos (or (position #\Tab substring :start (or first-nonblank 0))
			    (lisp:search "  " substring :start2 (or first-nonblank 0)))))
	  (if tab-pos
	      (let* ((first (subseq substring 0 tab-pos))
		     (second (string-left-trim '(#\Space #\Tab) (subseq substring tab-pos))))
		(push (cons first second) result))
	      (push (list substring) result)))
	))
    (let* ((first-nonblank (position-if-not #'(lambda (x) (string= (car x) "")) result))
	   (last-nonblank (position-if-not #'(lambda (x) (string= (car x) "")) result :from-end t)))
      (if first-nonblank
	  (setq result (subseq result first-nonblank (if last-nonblank (1+ last-nonblank) nil)))
	  (setq result nil)))
    (nreverse result)))



;;; The Album-Edit Command Hook.
;;; We want field-names to be read-only.
;;;


(defprop album-edit-hook 10 command-hook-priority)

(defun album-edit-hook (char)
  (declare (ignore char))
  (let* ((point (point)))
    (block DONE
      (dolist (region (get *interval* 'RO-REGIONS))
	(let* ((bp-1 (interval-first-bp region))
	       (bp-2 (interval-last-bp region)))
	  (when (and (not (bp-= bp-1 bp-2))
		     (not (bp-< point bp-1))
		     (or (bp-< point bp-2)
			 (bp-= point bp-2)))
	    (make-buffer-read-only *interval*)
	    (return-from DONE))))
      (make-buffer-not-read-only *interval*))))


(defcom com-album-complete "For use in Album Edit Mode - do completion on the current field." ()
  (album-complete-1))

(defcom com-album-display-completions "For use in Album Edit Mode - do completion on the current field." ()
  (album-complete-1 t))


(defun album-complete-1 (&optional display-completions)
  (let* ((this-region nil)
	 (next-region nil)
	 (band-p nil)
	 (band-region nil)
	 (band-next-region nil)
	 )
    (do* ((list (reverse (get *interval* 'RO-REGIONS)) (cdr list)))
	 ((null list))
      (cond ((and (not (bp-< (point) (interval-last-bp (car list))))
		  (or (null (cadr list))
		      (bp-< (point) (interval-first-bp (cadr list))))
		  )
	     (return (setq this-region (car list)
			   next-region (cadr list))))
	    ((string-equal (string-interval (car list)) "Band Name" :end1 9)
	     (setq band-region (car list)
		   band-next-region (cadr list))
	     )))
    (cond ((and this-region
		(or
		  (and (string-equal (string-interval this-region) "Band Name" :end1 9)
		       (setq band-p t))
		  (string-equal (string-interval this-region) "Album Name" :end1 10)
		  ))
	   (let* ((start (forward-over '(#\Space #\Tab) (interval-last-bp this-region)))
		  (end  (backward-over '(#\Space #\Tab) (interval-first-bp next-region)))
		  (interval (if (bp-< end start)
				(make-interval start start)
				(make-interval start end)))
		  (string (string-interval interval))
		  (completion-alist (if band-p
					(build-artist-completion-alist)
					(let* ((bstart (forward-over '(#\Space #\Tab) (interval-last-bp band-region)))
					       (bend (backward-over '(#\Space #\Tab) (interval-first-bp band-next-region)))
					       (band-name (if (bp-< bend bstart)
							      ""
							      (string-interval (make-interval bstart bend)))))
					  (build-album-completing-alist band-name)
					  ))))
	     (cond (display-completions
		    (let* ((*completing-alist* completion-alist)
			   (*completing-delims* nil)
			   (*completing-documenter* nil))
		      (multiple-value-bind (ignore poss) (complete-string string completion-alist nil t)
			(list-completions-internal poss))))
		   
		   (t
		    (multiple-value-bind (completed-string completed-p) (complete-string string completion-alist nil t)
		      (unless (string= completed-string "")
			(delete-interval interval)
			(insert start completed-string)
			(move-bp (point) (forward-char start (length completed-string))))
		      (when (or (not completed-p)
				(string= completed-string "")
				(string= completed-string string))
			(barf))
		      )))))
	  (t (barf "Not in the Band Name or Album Name fields."))))
  DIS-TEXT)



;;;
;;; Prompting for albums in ZMACS.
;;;

(defun build-artist-completion-alist ()
  "Returns an association list suitable for passing to COMPLETING-READ-FROM-MINI-BUFFER of the names of all known bands."
  (mapcar #'(lambda (x) (cons x x))
	  (delete-duplicates (maphash-return #'(lambda (ignore album) (album-artist album))
					     *album-table*)
			     :test #'string-equal)))

(defun build-album-completing-alist (artist)
  "Returns an association list suitable for passing to COMPLETING-READ-FROM-MINI-BUFFER of the names of all of the known
  albums by ARTIST."
  (let* ((album-names '()))
    (maphash #'(lambda (ignore album)
		 (when (string-equal artist (album-artist album))
		   (push (cons (album-name album) album)
			 album-names)
		   (when (album-other-name album)
		     (push (cons (album-other-name album) album)
			   album-names))))
	     *album-table*)
    (nreverse album-names)))



(defun prompt-for-band-name-1 (band-prompt default-band-name must-complete)
  (let* ((band-names (build-artist-completion-alist)))
    (when default-band-name
      (unless (member default-band-name band-names :test #'string-equal :key #'car)
	(setq default-band-name nil)))
    (let* ((name-cons ""))
      (do ((done-once nil t))
	  ((if must-complete
	       (not (stringp name-cons))
	       done-once))
	(when done-once (beep))
	(setq name-cons (completing-read-from-mini-buffer band-prompt band-names nil default-band-name)))
      (if (stringp name-cons) nil (cdr name-cons)))))


(defun prompt-for-album-1 (band-name album-prompt default-album-name must-complete)
  (let* ((album-names (build-album-completing-alist band-name)))
    (when default-album-name
      (unless (member default-album-name album-names :test #'string-equal :key #'car)
	(setq default-album-name nil)))
    (let* ((album-cons ""))
      (do ((done-once nil t))
	  ((if must-complete
	       (not (stringp album-cons))
	       done-once))
	(when done-once (beep))
	(setq album-cons (completing-read-from-mini-buffer album-prompt album-names nil default-album-name)))
      (if (stringp album-cons) nil (cdr album-cons)))))


(defun prompt-for-album (&optional (band-prompt "Band Name:") (album-prompt "Album Name:")
			 default-band-name default-album-name
			 (must-complete t))
  "Prompt the user for the name of a band and an album, and return the ALBUM structure.  If MUST-COMPLETE is NIL, then
   NIL may be returned instead of an album."
  (let* ((name (prompt-for-band-name-1 band-prompt default-band-name must-complete))
	 (album (and name (prompt-for-album-1 name album-prompt default-album-name must-complete))))
    album))


;;;
;;; ZMACS Commands.
;;;

(defcom com-edit-album "Edit an existing Album." ()
  (let* ((album (prompt-for-album)))
    (select-album-buffer t)
    (zwei:delete-interval *album-edit-buffer*)
    (grind-album-to-buffer album *album-edit-buffer*)
    (move-bp (point) (interval-first-bp *album-edit-buffer*))
    (send *album-edit-buffer* :set-major-mode 'album-edit)
    (not-modified *album-edit-buffer*)
    DIS-ALL))


(defcom com-create-album "Create a new album." ()
  (let* ((type (fquery '(:fresh-line t :beep nil
			 :choices (((1 "One Album")    #\1 #\Up-Arrow #\Newline)
				   ((2 "Two Albums")   #\2 #\Down-Arrow)
				   ((0 "Double Album") #\D #\d)))
		       "Is this a normal album, two albums, or a double album? "))
	 (double-p (= type 0)))
    (select-album-buffer nil type double-p)
    (move-bp (point) (forward-char (forward-word (interval-first-bp *album-edit-buffer*) 2 t) 2 t))
    (send *album-edit-buffer* :set-major-mode 'album-edit)
    (not-modified *album-edit-buffer*)
    DIS-ALL))


(defun generate-new-id (album)
  "When ALBUM has a NIL id, assign it one that hasn't been assigned yet."
  (unless (album-id album)
    (let* ((max 0))
      (maphash #'(lambda (n ignore)
		   (when (> n max) (setq max n)))
	       *album-table*)
      (setf (album-id album) (1+ max)))))


(defun delete-album (id)
  "Delete all albums with this ID."
  (remhash id *album-table*))


(defun album-exists (album)
  "If there is an album with the same Artist and Name as ALBUM, return it."
  (maphash #'(lambda (ignore album2)
	       (when (and (string-equal (album-artist album) (album-artist album2))
			  (string-equal (album-name album) (album-name album2)))
		 (return-from ALBUM-EXISTS album2)))
	   *album-table*)
  nil)


(defvar *album-database-modified* nil)
(defvar *tapes-of-newly-created-albums* nil)

(defcom com-exit-album-editor "Exit the Album Editor, saving changes." ()
  (when (buffer-modified-p *album-edit-buffer*)
    (setq *album-database-modified* t)
    (let* ((album (parse-album-from-buffer *album-edit-buffer*
					   (if (get *album-edit-buffer* 'two-albums-p) 2 1)
					   (get *album-edit-buffer* 'DOUBLE-ALBUM-P)))
	   (conflict (album-exists album)))
      (when (or (not conflict)
		(and conflict
		     (or (y-or-n-p "There is already an album ~S by ~S defined.~%Replace it? "
				   (album-name album) (album-artist album))
			 (barf "Not defined."))))
	(when conflict
	  (let* ((id (album-id conflict)))
	    (delete-album id)
	    (setf (album-id album) id)))
	(setf (gethash (or (album-id album) (generate-new-id album)) *album-table*) album))
      (let* ((tape (deal-with-album-tape album)))
	(pushnew tape *tapes-of-newly-created-albums*)
	)))
  (send *window* :exit-special-buffer nil *interval*)
  DIS-BPS)


(defun deal-with-album-tape (album)
  "Make there be a tape for the ALBUM, resolving whatever conflicts might arise.  Returns the tape the album was added to."
  (let* ((old-tape nil)
	 (final-tape nil))
    (dolist (tape *all-tapes*)
      (when (or (eql (album-id album) (tape-side-a tape))
		(eql (album-id album) (tape-side-b tape)))
	(setq final-tape tape)
	(setq old-tape tape)
	(return)))
    (if (double-album-p album)
	(unless old-tape
	  (push (setq final-tape (make-tape :side-a (album-id album)))
		*all-tapes*))
	;;
	;; If the album is already on a tape (which is OLD-TAPE), then the album on the other side is DEFAULT-FLIPSIDE.
	;;
	(let* (default-flipside)
	  (when old-tape
	    (cond ((eql (album-id album) (tape-side-a old-tape)) (setq default-flipside (tape-side-b old-tape)))
		  (t (setq default-flipside (tape-side-a old-tape)))))
	  (when default-flipside (setq default-flipside (lookup-album default-flipside)))
	  
	  (let* ((default-flipband (and default-flipside (album-artist default-flipside)))
		 (default-flipname (and default-flipside (album-name default-flipside)))
		 (flipside (prompt-for-album "Name of band on other side:" "Name of album on other side:"
					     default-flipband default-flipname nil)))
	    (cond (old-tape
		   ;;
		   ;; If the album was already on a tape - 
		   ;;
		   (cond (default-flipside
			  ;;
			  ;; If there was already something on the other side -
			  ;; If what they said was the flipside is different, either create a new tape or modify the old one.
			  ;;
			  (unless (equal flipside default-flipside)
			    (let* ((new-tape-p (fquery '(:choices (((t  "New Tape.") #\N #\)
								   ((nil "Change Old Tape.") #\C #\)))
					    "Is this a copy of the album on a new tape, or shall I change the old tape? "
					    )))
			      (cond (new-tape-p
				     (push (setq final-tape (make-tape :side-a (album-id album)
								       :side-b (album-id flipside)))
					   *all-tapes*))
				    (t
				     (setq final-tape old-tape)
				     (if (eql (tape-side-a old-tape) (album-id album))
					 (setf (tape-side-b old-tape) (album-id flipside))
					 (setf (tape-side-a old-tape) (album-id flipside))))))))
			 (t
			  ;;
			  ;; If the album was already on a tape, but there was nothing on the other side -
			  ;; If they specified a flipside, install it on the existing TAPE object.
			  ;;
			  (when flipside
			    (setq final-tape old-tape)
			    (if (eql (tape-side-a old-tape) (album-id album))
				(setf (tape-side-b old-tape) (album-id flipside))
				(setf (tape-side-a old-tape) (album-id flipside)))))))
		  
		  (t ;;
		     ;; If the album was not on a tape -
		     ;;
		     (let* (old-flipside-tape)
		       (when flipside
			 (dolist (tape *all-tapes*)
			   (when (or (eql (album-id flipside) (tape-side-a tape))
				     (eql (album-id flipside) (tape-side-b tape)))
			     (setq old-flipside-tape tape)
			     (return))))
		       (cond (old-flipside-tape
			      ;;
			      ;; If the album was not on a tape, but its flipside was -
			      ;; Put this album on the existing tape object.
			      ;;
			      (setq final-tape old-flipside-tape)
			      (if (eql (tape-side-a old-flipside-tape) (album-id flipside))
				  (setf (tape-side-b old-flipside-tape) (album-id album))
				  (setf (tape-side-a old-flipside-tape) (album-id album))))
			     (t
			      ;;
			      ;; If the album was not on a tape, and neither was its flipside (or flipside was NIL) -
			      ;; Add a new tape object.
			      ;;
			      (push (setq final-tape (make-tape :side-a (album-id album)
								:side-b (if flipside
									    (album-id flipside)
									    nil)))
				    *all-tapes*)))))
		  ))))
    final-tape))


;;;
;;; Reading and Writing the database files.
;;;


(defvar *default-album-file* nil "The default file from which albums are loaded.")

(defun default-album-file ()
  "The default album database file.  This is derived from *DEFAULT-ALBUM-FILE*."
  (merge-pathnames (or *default-album-file* "")
		   (make-pathname :defaults (fs:user-homedir-pathname)
				  :name "audio-tape"
				  :type "text" :version :newest)))


(defcom com-load-albums "Load the album database file." ()
  (setq *default-album-file* (read-defaulted-pathname "Album Database File:" (default-album-file)))
  (load-albums *default-album-file*)
  DIS-NONE)

(defcom com-save-albums "Save the album database file." ()
  ;;
  ;; This is so this function can be called as a ZMACS command or as a normal function, from a Listener.
  ;; In ZMACS, it uses the minibuffer, otherwise it prompts in the Listener.
  ;;
  (if (boundp '*mini-buffer-window*)
      (setq *default-album-file* (read-defaulted-pathname "Album Database File:" (default-album-file) nil nil :WRITE))
      (setq *default-album-file* (prompt-and-read (list :pathname :defaults (default-album-file))
						  "~&Album Database File (default ~A): " (default-album-file))))
  (save-albums *default-album-file*)
  DIS-NONE)


(defun save-albums (pathname)
  (with-open-file (stream pathname :direction :output)
    (dolist (tape *all-tapes*)
      (let* ((a (tape-side-a tape))
	     (b (tape-side-b tape)))
	(princ a stream)
	(princ "  " stream)
	(when b (princ b stream))
	(terpri stream)))
    (princ "--------------------" stream) (terpri stream)
    (maphash #'(lambda (ignore album)
		 (write-one-album album stream)
		 (princ "--------------------" stream) (terpri stream)
		 )
	     *album-table*))
  (setq *album-database-modified* nil))


(defun write-one-album (album &optional (stream *standard-output*))
  "Write an album in database format."
  (flet ((princ-safe (thing)
	   (cond ((or (null thing)
		      (and (stringp thing)
			   (string= (string-trim '(#\Space #\Tab) thing) "")))
		  (princ "-" stream))
		 (t (princ thing stream)))))
    (princ-safe (album-id album)) (princ #\Tab stream)
    (princ-safe (album-artist album)) (princ #\Tab stream)
    (princ-safe (album-name album)) (princ #\Tab stream)
    (when (and (album-other-name album)
	       (not (string= (string-trim '(#\Space #\Tab) (album-other-name album)) "")))
      (princ (album-other-name album) stream) (princ #\Tab stream))
    (princ-safe (album-year album))
    (when (double-album-p album)
      (princ #\Tab stream) (princ "double" stream))
    (terpri stream) (terpri stream))
  (dolist (cons (album-songs album))
    (princ (car cons) stream)
    (when (cdr cons)
      (princ #\Tab stream) (princ (cdr cons) stream))
    (terpri stream))
  (when (double-album-p album)
    (princ "----" stream) (terpri stream)
    (dolist (cons (album-songs2 album))
      (princ (car cons) stream)
      (when (cdr cons)
	(princ #\Tab stream) (princ (cdr cons) stream))
      (terpri stream)))
  album)


(defun maybe-offer-to-save-albums ()
  (when (and *album-database-modified*
	     (y-or-n-p "There are unsaved changes in the Album Database; save them? "))
    (com-save-albums)))

(sys:add-initialization "Save Album Database" '(maybe-offer-to-save-albums) :logout)
(sys:add-initialization "Clear Tape Signature String" '(setq *signature-string* nil) :logout)


;;;
;;; Dumping the PS files.
;;;

(defcom com-print-albums
	"Produce and maybe print a PostScript file of some albums in the loaded database.
With a numeric argument of Control-U, prompt the user for whether to print the newly created tapes, 
 or all of tapes in the loaded database.
With a numeric argument of a digit, prompt for and print all tapes containing a prompted-for album.
Otherwise, pop up a multiple-choice menu of the albums to print."
	()
  (let* ((chosen '())
	 (new-p nil))
    (cond ((eq *numeric-arg-p* :CONTROL-U)
	   (setq new-p (if *tapes-of-newly-created-albums*
			   (fquery '(:choices (((t  "New Albums.") #\N #\ #\Newline)
					       ((nil "All Albums.") #\A #\)))
				   "Print newly created albums, or print all albums? ")
			   nil))
	   (if new-p
	       (setq chosen *tapes-of-newly-created-albums*)
	       (setq chosen *all-tapes*)))
	  
	  (*numeric-arg-p*
	   (let* ((album (prompt-for-album "Band Name of album to print: " "Album Name of album to print: "))
		  (id (album-id album))
		  (tapes (remove-if-not #'(lambda (tape)
					    (or (eql id (tape-side-a tape))
						(eql id (tape-side-b tape))))
					*all-tapes*)))
	     (setq chosen tapes)))
	  
	  ((not *numeric-arg-p*)
	   (let* ((menu-list '()))
	     (dolist (tape *all-tapes*)
	       (let* ((a (lookup-album (tape-side-a tape)))
		      (b (lookup-album (tape-side-b tape)))
		      (a-artist (and a (album-artist a)))
		      (b-artist (and b (album-artist b)))
		      (a-name (and a (album-name a)))
		      (b-name (and b (album-name b)))
		      (c-name (and a (album-other-name a)))
		      (d-name (and b (album-other-name b)))
		      (double-p (or (and a (double-album-p a))
				    (and b (double-album-p b))))
		      (a-date (and a (album-year a)))
		      (font fonts:hl12b)
		      (ital-font fonts:tr12bi)
		      (date-font fonts:tr12i)
		      )
		 (flet ((insert (artist name this-side flipside other-flipside other-band)
			  (when (and artist name)
			    (push (list (format nil "~A~30t ~A" artist name)
					:VALUE tape
					:DOCUMENTATION
					(append
					  (list :documentation (string-append #\Space artist ": ")
						:font ital-font :documentation name)
					  (if a-date
					      (list :font date-font :documentation (format nil "   ~A~% " a-date))
					      (list :documentation #.(string-append #\Return #\Space)))
					  (list :font font)
					  (cond (double-p '(:documentation "Double Album."))
						((or this-side flipside other-flipside)
						 (append (when this-side
							   (list :documentation "Same side as " :font ital-font
								 :documentation this-side :font font
								 :documentation "; "))
							 (when flipside
							   (append '(:documentation "Backed with ")
							    (unless (equal artist other-band)
							      (list :documentation (format nil "~A: " other-band)))
							    (list :font ital-font :documentation flipside :font font)))
							 (when other-flipside
							   (list :documentation " and " :font ital-font 
								 :documentation other-flipside :font font))
							 '(:documentation ".")))
						(t nil))))
				  menu-list))))
		   (insert a-artist a-name c-name b-name d-name b-artist)
		   (insert b-artist b-name d-name a-name c-name a-artist)
		   (insert a-artist c-name a-name b-name d-name b-artist)
		   (insert b-artist d-name b-name a-name c-name a-artist))))
	     (setq menu-list (sort menu-list #'string-lessp :key #'car))
	     (setq chosen (w:menu-choose menu-list
					 :pop-up t
					 :highlighting t
					 :menu-margin-choices :doit-or-abort
					 )))))
    (when chosen
      (setq chosen (delete-duplicates chosen))
      (unless *signature-string*
	(setq *signature-string* (completing-read-from-mini-buffer "Signature String: " nil t nil
				   "Type the string to print on the back flap of the tape, identifying it as yours.")))
      (let* ((file (read-defaulted-pathname (format nil "PostScript File Name for ~D tape label~:P: "
						    (length chosen))
					    (default-album-file) "ps" nil :write))
	     (print-after (y-or-n-p "Print the file after writing it? ")))
	(dump-tapes-to-file file chosen)
	(when print-after
	  (print-file file)
	  (when new-p (setq *tapes-of-newly-created-albums* nil))))))
  DIS-NONE)


;;;
;;; The ALBUM-EDIT Major Mode.
;;;


(defmajor com-edit-album-1 album-edit "Album-Edit" "" ()
  (set-char-syntax word-alphabetic *mode-word-syntax-table* #\_)
  (set-char-syntax word-alphabetic *mode-word-syntax-table* #\')
  (set-comtab *mode-comtab* '(#\End com-exit-album-editor
			      #\Tab com-insert-tab
			      #\Control-D com-delete-forward  ; no tab-hacking!
			      #\Rubout    com-rubout
			      #\Escape    com-album-complete
			      #\Control-? com-album-display-completions
			      #\Control-/ com-album-display-completions
			      ))
  (command-hook 'album-edit-hook *command-hook*)
  (command-hook 'album-edit-hook *post-command-hook*)
  )


(set-comtab *zmacs-comtab* ()
	    '(("Edit Album"   . com-edit-album)
	      ("Create Album" . com-create-album)
	      ("Load Album Database" . com-load-albums)
	      ("Save Album Database" . com-save-albums)
	      ("Print Albums" . com-print-albums)
	      ))
