/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: common.c,v 1.20 1994/04/27 00:17:56 janssen Exp $
*/

#include <sys/types.h>
#include <unistd.h>		/* for getuid() */
#include <stdlib.h>		/* for getenv() */
#include <pwd.h>		/* for getpwuid() */
#include <sys/stat.h>
#include <time.h>

#include "m3.h"

extern Interface GetInterface(string, string);
static void GenerateIncludesForType (Type t, Context context);

void UnmarkSupertypes (Type t)
{
  Type st;

  if (t == NULL OR type_basic_type(t) != object_Type)
    return;
  for (st = t;  st->supertype != NULL;  st = st->supertype)
    st->marked = FALSE;

  list_enumerate (class_object(st)->superclasses, (EnumProc) UnmarkSupertypes, NULL);
  st->marked = FALSE;
}

Type ultimateType (Type t)
{
  Type st;

  for (st = t;  st != NULL && st->supertype != NULL;  st = st->supertype)
    ;
  return (st);
}

static char *ModTime (char *path)
{
  static char timebuf[30];
  struct stat statbuf;

  stat (path, &statbuf);
  strcpy (timebuf, ctime(&statbuf.st_mtime));
  timebuf[24] = '\0';
  return (timebuf);
}

static char *GoodGetLogin (void)
{
  char *name;
  struct passwd *pw;

  if (((name = cuserid(NULL)) == NULL)
      && (name = (char *) getlogin()) == NULL
      && (name = (char *) (((pw = getpwuid(getuid())) == NULL) ? NULL : pw->pw_name)) == NULL
      && (name = (char *) getenv("USER")) == NULL)
    return "nobody";
  else
    return name;
}

static void PrintImportedFileInfo (Imported s, FILE *file)
{
  Interface i = GetInterface (s->name, s->filename);
  if (i == NULL)
    {
      fprintf (stderr, "Can't find interface \"%s\", supposedly from file \"%s\".\n",
	       s->name, s->filename);
      exit(1);
    }
  else
    fprintf (file, ",\n   and \"%s\" of %s", i->filename, ModTime(i->filename));	   
}

void generate_boilerplate (FILE *file, Interface parse)
{
  time_t clock = time(0);
  char *now;
  extern char ProgramName[];

  now = ilu_strdup(asctime(localtime(&clock)));
  now[strlen(now) - 1] = '\0';
  
  fprintf (file, "(* this file was automatically generated\n   at %s by %s\n   running \"%s\" of %s\n",
	   now, GoodGetLogin(), ProgramName, ModTime(ProgramName));
  fprintf (file, "   on \"%s\" of %s", parse->filename, ModTime(parse->filename));
  if (list_size(parse->imports) > 0)
    list_enumerate (parse->imports, (EnumProc) PrintImportedFileInfo, file);
  fprintf (file, ". *)\n\n");
}

boolean IsBulkyData (Type type)
{
  return (type_basic_type(type) == array_Type OR type_basic_type(type) == record_Type);
}

boolean IsSequenceObject (Type type, Interface i)
{
  return ((i == NULL || type->interface == i)
	  && type->importInterfaceName == NULL
	  && type_basic_type(type) == sequence_Type
	  && (type_basic_type(type_description(type)->structuredDes.sequence.type) != shortcharacter_Type)
	  && (type_basic_type(type_description(type)->structuredDes.sequence.type) != byte_Type));
}

boolean IsPipeObject (Type type, Interface i)
{
  return ((i == NULL || type->interface == i)
	  && type->importInterfaceName == NULL
	  && type_basic_type(type) == pipe_Type);
}

boolean PassedByRef (Type type)
{
  enum PrimitiveTypes t;

  if (type == NULL)
    return (FALSE);

  t = type_basic_type(type);
  return (t == record_Type
	  OR t == union_Type
	  OR t == array_Type
	  OR t == object_Type
	  OR t == pipe_Type
	  OR t == sequence_Type
	  OR t == longcardinal_Type
	  OR t == longinteger_Type);
}

boolean TypeIsNonObjectStruct (Type type)
{
  enum PrimitiveTypes t;

  if (type == NULL)
    return (FALSE);

  t = type_basic_type(type);
  return (t == record_Type OR t == union_Type);
}

boolean TypeIsArray (Type type)
{
  if (type == NULL)
    return (FALSE);

  return (type_basic_type(type) == array_Type);
}

boolean TypeIsString (Type type)
{
  return (type_basic_type(type) == sequence_Type
	  AND (type_basic_type(type_description(type)->structuredDes.sequence.type) == shortcharacter_Type));
}

string iname(string name, Interface interface)
{
  char *interface_name;
  int inamelen;

  if (interface == NULL)
    return (name);

  interface_name = m3_interface_name(interface);
  inamelen = strlen(interface_name);
  if (strncmp (name, interface_name, inamelen) == 0
      AND (name[inamelen] == '.'))
    return (name + inamelen + 1);
  else
    return (name);
}

char * UnionTypeName (Type union_type, Type field_type)
{
  static char buf[1000];
  Interface importInterface = NULL;

  importInterface = (field_type->builtIn OR
		     (ultimateType(union_type)->interface == ultimateType(field_type)->interface)) ? NULL
		       : ultimateType(field_type)->interface;
  sprintf (buf, "%s_%s%s%s",
	   M3_SHORT_TYPE_NAME(union_type),
	   (importInterface != NULL) ? m3_interface_name(importInterface) : "",
	   (importInterface != NULL) ? "_" : "",
	   M3_SHORT_TYPE_NAME(field_type));
  return (buf);
}

void ImportInterface (Imported i, Context context)
{
  char *s = m3_string(i->name);

  if (strcmp(s, "ilu") == 0)
    fprintf (context->file, "IMPORT ilu; <*NOWARN*>\n");
  else
    fprintf (context->file, "IMPORT %s, %s_x; <*NOWARN*>\n", s, s);
  Free(s);
}

char *ModulePrefix(Type type, Context context)
{
  static char buf[1000];
  boolean includeInterface = ((ultimateType(type)->interface != context->interface)
			      OR (context->module == ServerModule));

/*
  fprintf (stderr, "ModulePrefix:  type->interface = %s (0x%x), context->interface = %s (0x%x)\n",
	   interface_name(ultimateType(type)->interface), ultimateType(type)->interface,
	   interface_name(context->interface), context->interface);
  fprintf (stderr, "               context->module == %d, includeInterface = %s\n",
	   context->module, includeInterface ? "TRUE" : "FALSE");
*/

  if (includeInterface)
    sprintf (buf, "%s_x.", m3_interface_name(ultimateType(type)->interface));
  else
    buf[0] = '\0';
  return(buf);
}

void SizeValue (Type type, string name, Context context)
{
  enum PrimitiveTypes t = type_basic_type (type);
  TypeDescription d = type_description(type);

  if (t == integer_Type)
    fprintf (context->file, "IluRuntime.SizeInteger(call_, %s)", name);
  else if (t == enumeration_Type)
    fprintf (context->file, "IluRuntime.SizeEnum(call_, %s%s__EnumValues[ORD(%s)])",
	     ModulePrefix(type, context), M3_SHORT_TYPE_NAME(type), name);
  else if (t == cardinal_Type)
    fprintf (context->file, "IluRuntime.SizeCardinal(call_, %s)", name);
  else if (t == shortinteger_Type)
    fprintf (context->file, "IluRuntime.SizeShortInteger(call_, %s)", name);
  else if (t == shortcardinal_Type)
    fprintf (context->file, "IluRuntime.SizeShortCardinal(call_, %s)", name);
  else if (t == longinteger_Type)
    fprintf (context->file, "IluRuntime.SizeLongInteger(call_, %s)", name);
  else if (t == longcardinal_Type)
    fprintf (context->file, "IluRuntime.SizeLongCardinal(call_, %s)", name);
  else if (t == character_Type)
    fprintf (context->file, "IluRuntime.SizeCharacter(call_, %s)", name);
  else if (t == shortcharacter_Type)
    fprintf (context->file, "IluRuntime.SizeByte(call_, ORD(%s))", name);
  else if (t == real_Type)
    fprintf (context->file, "IluRuntime.SizeReal(call_, %s)", name);
  else if (t == shortreal_Type)
    fprintf (context->file, "IluRuntime.SizeShortReal(call_, %s)", name);
  else if (t == longreal_Type)
    fprintf (context->file, "IluRuntime.SizeLongReal(call_, %s)", name);
  else if (t == boolean_Type)
    fprintf (context->file, "IluRuntime.SizeBoolean(call_, %s)", name);
  else if (t == byte_Type)
    fprintf (context->file, "IluRuntime.SizeByte(call_, %s)", name);
  else if (t == sequence_Type AND type_basic_type (d->structuredDes.sequence.type) == shortcharacter_Type)
    fprintf (context->file, "IluRuntime.SizeString(call_, %s)", name);
  else if (t == sequence_Type AND type_basic_type (d->structuredDes.sequence.type) == byte_Type)
    fprintf (context->file, "IluRuntime.SizeBytes(call_, %s)", name);
  else if (t == array_Type
      AND type_basic_type (d->structuredDes.array.type) == byte_Type
      AND list_size(d->structuredDes.array.dimensions) == 1)
    fprintf (context->file, "IluRuntime.SizeOpaque(call_, %s)", name);
  else if (t == array_Type
      AND type_basic_type (d->structuredDes.array.type) == shortcharacter_Type
      AND list_size(d->structuredDes.array.dimensions) == 1)
    fprintf (context->file, "IluRuntime.SizeStringVec(call_, %s)", name);
  else if (t == record_Type OR t == union_Type OR t == sequence_Type OR t == pipe_Type
	   OR t == array_Type OR t == optional_Type)
    fprintf (context->file, "%sComputeMarshalledSize_%s (call_, %s)",
	     ModulePrefix(type, context), M3_SHORT_TYPE_NAME(type), name);
  else if (t == object_Type)
    fprintf (context->file, "IluRuntime.SizeM3Object(call_, %s, FALSE, %sObjectType_%s)",
	     name, ModulePrefix(type, context), M3_SHORT_TYPE_NAME(type));
  else if (t == void_Type)
    fprintf (context->file, "0");
  else
    {
      fprintf (stderr, "Error:  Can't figure size of argument of type %s (line %d) yet.\n",
	       type_name(type), type->def);
      exit (1);
    }
}

void SizeArgument (Argument arg, Context context)
{
  fprintf (context->file, "\n          + ");
  SizeValue (arg->type, m3_argument_name(arg), context);
}

void MarshallValue (Type type, string name, Context context, int indent)
{
  enum PrimitiveTypes t = type_basic_type(type);
  TypeDescription d = type_description(type);

  if (t == integer_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputInteger (call_, %s);\n", indent, indent, "", name);
  else if (t == enumeration_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputEnum (call_, %s%s__EnumValues[ORD(%s)]);\n", indent, indent, "",
	     ModulePrefix(type, context), M3_SHORT_TYPE_NAME(type), name);
  else if (t == cardinal_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputCardinal (call_, %s);\n", indent, indent, "", name);
  else if (t == shortinteger_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputShortInteger (call_, %s);\n", indent, indent, "", name);
  else if (t == shortcardinal_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputShortCardinal (call_, %s);\n", indent, indent, "", name);
  else if (t == longinteger_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputLongInteger (call_, %s);\n", indent, indent, "", name);
  else if (t == longcardinal_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputLongCardinal (call_, %s);\n", indent, indent, "", name);
  else if (t == character_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputCharacter (call_, %s);\n", indent, indent, "", name);
  else if (t == shortcharacter_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputByte (call_, ORD(%s));\n", indent, indent, "", name);
  else if (t == real_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputReal (call_, %s);\n", indent, indent, "", name);
  else if (t == shortreal_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputShortReal (call_, %s);\n", indent, indent, "", name);
  else if (t == longreal_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputLongReal (call_, %s);\n", indent, indent, "", name);
  else if (t == boolean_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputBoolean (call_, %s);\n", indent, indent, "", name);
  else if (t == byte_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputByte (call_, %s);\n", indent, indent, "", name);
  else if (t == optional_Type)
    {
      char buf[1000];
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.optional);

      fprintf (context->file, "%*.*sIluRuntime.OutputOptional (call_, (%s # NIL));\n", indent, indent, "", name);
      fprintf (context->file, "%*.*sIF %s # NIL THEN\n", indent, indent, "", name);
      sprintf (buf, "%s%s", name, (t2 == object_Type OR t2 == sequence_Type OR t2 == union_Type) ? "" : "^");
      MarshallValue (d->structuredDes.optional, buf, context, indent + 2);
      fprintf (context->file, "%*.*sEND (*if*);\n", indent, indent, "");
    }
  else if (t == sequence_Type AND type_basic_type (d->structuredDes.sequence.type) == shortcharacter_Type)
    {
      fprintf (context->file, "%*.*sIluRuntime.OutputString (call_, %s, %u);\n",
	       indent, indent, "", name, d->structuredDes.sequence.limit);
    }
  else if (t == sequence_Type AND type_basic_type (d->structuredDes.sequence.type) == byte_Type)
    {
      fprintf (context->file, "%*.*sIluRuntime.OutputBytes (call_, %s, %u);\n",
	       indent, indent, "", name, d->structuredDes.sequence.limit);
    }
  else if (t == array_Type
      AND type_basic_type (d->structuredDes.array.type) == byte_Type
      AND list_size(d->structuredDes.array.dimensions) == 1)
    fprintf (context->file, "%*.*sIluRuntime.OutputOpaque(call_, %s, %u);\n",
	     indent, indent, "", name, (unsigned long) list_car(d->structuredDes.array.dimensions));
  else if (t == array_Type
      AND type_basic_type (d->structuredDes.array.type) == shortcharacter_Type
      AND list_size(d->structuredDes.array.dimensions) == 1)
    fprintf (context->file, "%*.*sIluRuntime.OutputStringVec(call_, %s, %u);\n",
	     indent, indent, "", name, (unsigned long) list_car(d->structuredDes.array.dimensions));
  else if (t == record_Type OR t == union_Type OR t == sequence_Type OR t == pipe_Type OR t == array_Type)
    fprintf (context->file, "%*.*s%sMarshall_%s (call_, %s);\n",
	     indent, indent, "", ModulePrefix(type, context), M3_SHORT_TYPE_NAME(type), name);
  else if (t == object_Type)
    fprintf (context->file, "%*.*sIluRuntime.OutputM3Object (call_, %s, FALSE, %sObjectType_%s);\n",
	     indent, indent, "", name, ModulePrefix(type, context), M3_SHORT_TYPE_NAME(type));
  else if (t == void_Type)
    ;
  else
    {
      fprintf (stderr, "Error:  Can't cope with argument of type %s yet.\n", M3_TYPE_NAME(type));
      exit (1);
    }
}

void UnmarshallValue (Context context, Type type, LineNumber line, string name, int indent, boolean terminate)
{
  enum PrimitiveTypes t = type_basic_type(type);
  TypeDescription d = type_description(type);

  if (name != NULL)
    {
      fprintf (context->file, "%*.*s", indent, indent, "");
      if (!IsBulkyData(type))
	fprintf (context->file, "%s := ", name);
    }

  if (t == integer_Type)
    fprintf (context->file, "IluRuntime.InputInteger (call_");
  else if (t == cardinal_Type)
    fprintf (context->file, "IluRuntime.InputCardinal (call_");
  else if (t == shortinteger_Type)
    fprintf (context->file, "IluRuntime.InputShortInteger (call_");
  else if (t == shortcardinal_Type)
    fprintf (context->file, "IluRuntime.InputShortCardinal (call_");
  else if (t == longinteger_Type)
    fprintf (context->file, "IluRuntime.InputLongInteger (call_");
  else if (t == longcardinal_Type)
    fprintf (context->file, "IluRuntime.InputLongCardinal (call_");
  else if (t == character_Type)
    fprintf (context->file, "IluRuntime.InputCharacter (call_");
  else if (t == shortcharacter_Type)
    fprintf (context->file, "VAL(IluRuntime.InputByte(call_), CHAR");
  else if (t == real_Type)
    fprintf (context->file, "IluRuntime.InputReal (call_");
  else if (t == shortreal_Type)
    fprintf (context->file, "IluRuntime.InputShortReal (call_");
  else if (t == boolean_Type)
    fprintf (context->file, "IluRuntime.InputBoolean (call_");
  else if (t == longreal_Type)
    fprintf (context->file, "IluRuntime.InputLongReal (call_");
  else if (t == byte_Type)
    fprintf (context->file, "IluRuntime.InputByte (call_");
  else if (t == sequence_Type AND type_basic_type (d->structuredDes.sequence.type) == shortcharacter_Type)
    {
      fprintf (context->file, "IluRuntime.InputString(call_, %u",
	       d->structuredDes.sequence.limit);
    }
  else if (t == sequence_Type AND type_basic_type (d->structuredDes.sequence.type) == byte_Type)
    {
      fprintf (context->file, "IluRuntime.InputBytes (call_, %u",
	       d->structuredDes.sequence.limit);
    }
  else if (t == array_Type
	   OR t == enumeration_Type
	   OR t == sequence_Type
	   OR t == record_Type
	   OR t == union_Type
	   OR t == optional_Type
	   OR t == pipe_Type
	   OR t == array_Type)
    {
      fprintf (context->file, "%sUnmarshall_%s (call_",
	       ModulePrefix(type, context),
	       M3_SHORT_TYPE_NAME(type));
    }
  else if (t == object_Type)
    fprintf (context->file, "NARROW(IluRuntime.InputM3Object (call_, FALSE, %sObjectType_%s), %s",
	     ModulePrefix(type, context),
	     M3_SHORT_TYPE_NAME(type), M3_TYPE_NAME(type));
  else if (t == void_Type)
    ;
  else
    {
      fprintf (stderr, "Error:  Can't cope with procedure of type %s (line %d) yet.\n",
	       M3_TYPE_NAME(type), line);
      exit (1);
    }
  if (t != void_Type)
    {
      if (name != NULL && IsBulkyData(type))
	fprintf (context->file, ", %s", name);
      fprintf (context->file, ")");
      if (terminate)
	fprintf (context->file, ";\n");
    }
}

boolean BlockingCall (Procedure proc)
{
  return (NOT(proc->asynch)
	  OR type_basic_type(proc->returnType) != void_Type
	  OR list_size(proc->exceptions) > 0);
}

static list InterfacesToInclude = NULL;
static list TypesChecked = NULL;

static boolean matchString (char *s1, char *s2)
{
  return (strcmp(s1,s2) == 0);
}

static boolean matchPointer (char *s1, char *s2)
{
  return (s1 == s2);
}

static void GenerateIncludeForArgument (Argument a, Context context)
{
  GenerateIncludesForType (a->type, context);
}

static void GenerateIncludesForMethods (Procedure m, Context context)
{
  list_enumerate (m->arguments, (EnumProc) GenerateIncludeForArgument, context);
  if (m->returnType != NULL && type_basic_type(m->returnType) != void_Type)
    GenerateIncludesForType (m->returnType, context);
}

extern Class class_object();

static void GenerateIncludesForType (Type t, Context context)
{
  if (t == NULL)
    return;

  if (list_find(TypesChecked, (FindProc) matchPointer, t) != NULL)
    return;
  list_insert(TypesChecked, t);
  if (!t->builtIn && t->interface != NULL && name_base_name(t->interface->name) != NULL && (list_find(InterfacesToInclude, (FindProc) matchString, name_base_name(t->interface->name)) == NULL))
    list_insert (InterfacesToInclude, name_base_name(t->interface->name));

  if (type_basic_type(t) == object_Type)
    {
      list_enumerate (type_description(t)->structuredDes.object->superclasses, (EnumProc) GenerateIncludesForType, context);
      list_enumerate (type_description(t)->structuredDes.object->methods, (EnumProc) GenerateIncludesForMethods, context);
    }
  else if (type_basic_type(t) == record_Type)
    {
      list_enumerate (type_description(t)->structuredDes.record, (EnumProc) GenerateIncludeForArgument, context);
    }
  else if (type_basic_type(t) == union_Type)
    {
      list_enumerate (type_description(t)->structuredDes.uniond.types, (EnumProc) GenerateIncludeForArgument, context);
    }
  else if (type_basic_type(t) == sequence_Type)
    {
      GenerateIncludesForType (type_description(t)->structuredDes.sequence.type, context);
    }
  else if (type_basic_type(t) == array_Type)
    {
      GenerateIncludesForType (type_description(t)->structuredDes.array.type, context);
    }
}

static Interface CurrentInterface = NULL;

static void ListIncludes (Imported i, list l)
{
  Interface imported;

  if ((imported = GetInterface (i->name, i->filename)) != NULL
      && imported != CurrentInterface
      && (strcmp(i->name, "ilu") != 0)
      && (list_find(l, (FindProc) matchString, name_base_name(imported->name)) == NULL))
    list_insert (l, name_base_name(imported->name));    
}

void GenerateNecessaryIncludes (Context context, void (*print_proc) (/* ??? */))
{
  if (InterfacesToInclude == NULL)
    InterfacesToInclude = (list) new_list();

  list_clear(InterfacesToInclude, FALSE);

  CurrentInterface = context->interface;
  list_enumerate (context->interface->imports, (EnumProc) ListIncludes, InterfacesToInclude);

  list_enumerate (InterfacesToInclude, print_proc, context);
}
