(* Ilu.m3 *)
(* Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights
   Reserved.

   Unlimited use, reproduction, and distribution of this software is
   permitted.  Any copy of this software must include both the above
   copyright notice of Xerox Corporation and this paragraph.  Any
   distribution of this software must comply with all applicable United
   States export control laws.  This software is made available AS IS,
   and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND
   NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY LIABILITY
   FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS EXPRESSLY
   DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING NEGLIGENCE)
   OR STRICT LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGES. *)
(* Edited by Mike Spreitzer April 4, 1994 1:24 pm PDT *)

UNSAFE MODULE Ilu EXPORTS Ilu, IluF;
IMPORT Fmt, Ilu, IluBasics, IluF, IluKernel, IntRefTbl, M3asC, M3toC,
       RTScheduler, Stdio, Text, TextToRefanyTable, Thread, Time,
       Uerror, UID, Unix, Utime, Wr;
FROM IluBasics IMPORT Failed, Failure;

<*PRAGMA lL, Ll, Main*>

REVEAL
  Ilu.Root = BRANDED OBJECT END;
  (* This revelation belongs in a module that the application can
     optionally replace.  Is there a way to do this while publicizing
     only one library? *)

EXCEPTION CantHappen;

<*FATAL CantHappen*>

VAR fDump: IluBasics.Failure;

TYPE
  DefaultObjectTable = ObjectTable BRANDED OBJECT
                         n := 1
                       OVERRIDES
                         ObjectToHandle := DefaultObjectToHandle;
                         HandleToObject := DefaultHandleToObject
                       END;

<*Ll, Main unconstrained*>
<*lL >= {the kernel server}*>
<*lL >= {gcmu} if the object is collectible*>

PROCEDURE DefaultObjectToHandle (           dot: DefaultObjectTable;
                                 <*UNUSED*> o  : Object              ):
  ObjectHandle =
  VAR ans := Fmt.Int(dot.n);
  BEGIN
    INC(dot.n);
    RETURN ans;
  END DefaultObjectToHandle;

PROCEDURE DefaultHandleToObject (<*UNUSED*> dot: DefaultObjectTable;
                                 <*UNUSED*> h  : ObjectHandle        ):
  Object =
  BEGIN
    RETURN NIL
  END DefaultHandleToObject;

VAR ssMu: MUTEX;

<*lL >= {ssMu}*>
VAR serverTab: TextToRefanyTable.T;

<*Main Invariant holds; Ll otherwise unconstrained*>
PROCEDURE InitTrueServer (self  : TrueServer;
                          id    : ServerId := NIL;
                          objtab: ObjectTable := NIL ): TrueServer
  RAISES {Failed} =
  VAR uid: UID.T;
  VAR kot: IluKernel.ObjectTable;
  VAR ra: REFANY;
  BEGIN
    IF id = NIL THEN
      UID.Get(uid);
      id :=
        Fmt.FN(
          "%02s%02s%02s%02s%02s%02s%s.%s",
          ARRAY OF
            TEXT{
            Fmt.Int(uid.machine[0], 16), Fmt.Int(uid.machine[1], 16),
            Fmt.Int(uid.machine[2], 16), Fmt.Int(uid.machine[3], 16),
            Fmt.Int(uid.machine[4], 16), Fmt.Int(uid.machine[5], 16),
            Fmt.Unsigned(uid.counter.coarse),
            Fmt.Unsigned(uid.counter.fine)});
    END (*if*);
    LOCK ssMu DO
      IF serverTab.in(id, ra) THEN
        RAISE
          Failed(
            NEW(
              Failure,
              info := "A server with id " & id & " already exists"))
      END (*if*);
      self.id := id;
      self.c_id := M3toC.TtoS(self.id);
      IF objtab = NIL THEN objtab := NEW(DefaultObjectTable) END;
      self.objtab := objtab;
      self.ks := NIL;
      Thread.Acquire(self);
      IF serverTab.put(id, self) THEN
        Thread.Release(self);
        RAISE CantHappen
      END (*if*);
    END (*lock ssMu*);
    TRY
      kot := NEW(IluKernel.ObjectTable);
      kot.object_of_ih := Call_object_of_ih;
      kot.free_self := Call_free;
      kot.rock :=
        LOOPHOLE(objtab, ADDRESS); (* serverTab will hold onto self (and
                                      thus self.objtab) forever *)
      self.ks := IluKernel.ilu_CreateTrueServer(self.c_id, kot);
        (* Technically this is a violation of the locking order
           constraints, which say that an Ilu.Server is held inside a
           kernel server.  No thread should deadlock because no thread
           will hold self.ks 'till after this call on CreateTrueServer
           has acquired all the locks it will need. *)
      IF self.ks = NIL THEN
        RAISE
          Failed(
            NEW(
              Failure,
              info := "ILU kernel failed to create server with id " & id))
      END (*if*);
    FINALLY
      Thread.Release(self);
    END (*try*);
    RETURN self;
  END InitTrueServer;

<*lL.sup < ssMu*>
PROCEDURE SFromKs (ks: IluKernel.Server): Server =
  VAR ra: REFANY;
  VAR ans: SurrogateServer;
  VAR id := M3toC.CopyStoT(IluKernel.ilu_IDOfServer(ks));
  BEGIN
    LOCK ssMu DO
      IF serverTab.in(id, ra) THEN RETURN NARROW(ra, Server) END (*if*);
      IF IluKernel.ilu_TrueServerP(ks) # 0 THEN RAISE CantHappen END;
      ans :=
        NEW(
          SurrogateServer, id := id, c_id := M3toC.CopyTtoS(id),
          ks := ks);
      IF serverTab.put(id, ans) THEN RAISE CantHappen END;
    END (*lock*);
    RETURN ans;
  END SFromKs;

<*lL >= {the server}*>
<*lL >= {gcmu} if result is collectible*>
PROCEDURE Call_object_of_ih (self: IluKernel.ObjectTable;
                             ih  : IluKernel.C_String     ):
  IluKernel.Object =
  VAR objtab := NARROW(LOOPHOLE(self.rock, REFANY), ObjectTable);
  VAR iht := M3toC.StoT(ih);
  VAR o := objtab.HandleToObject(iht);
  BEGIN
    IF o = NIL THEN
      RETURN NIL
    ELSIF o.ilu_is_surrogate THEN
      RAISE CantHappen
    ELSE
      TRY
        RETURN KOfTrueO(o)
      EXCEPT
        Failed (f) => fDump := f; RAISE CantHappen
      END (*try*)
    END (*if*);
  END Call_object_of_ih;

<*lL >= {the server}*>
PROCEDURE Call_free (<*UNUSED*> self: IluKernel.ObjectTable) =
  BEGIN
    RETURN;
  END Call_free;

<*lL, Ll = {}*>

PROCEDURE DefaultHandleListenerFailure (self: TrueServer; f: Failure):
  FailureAction =
  BEGIN
    TRY
      Wr.PutText(
        Stdio.stderr, Fmt.F(
                        "Listener failure on server %s: %s "
                          & "--- quitting listener!\n", self.id,
                        IluBasics.FullInfo(f)));
    EXCEPT
      Wr.Failure, Thread.Alerted =>
    END (*try*);
    RETURN FailureAction.Quit;
  END DefaultHandleListenerFailure;

PROCEDURE DefaultHandleWorkerFailure (self: TrueServer; f: Failure):
  FailureAction =
  BEGIN
    TRY
      Wr.PutText(
        Stdio.stderr,
        Fmt.F(
          "Failure in worker thread for server %s: %s "
            & "--- quitting worker!\n", self.id, IluBasics.FullInfo(f)));
    EXCEPT
      Wr.Failure, Thread.Alerted =>
    END (*try*);
    RETURN FailureAction.Quit;
  END DefaultHandleWorkerFailure;

<*Main Invariant holds; Ll otherwise unconstrained*>

PROCEDURE Export_Server (server: TrueServer;
                         p     : ProtocolInfo;
                         t     : TransportInfo ) RAISES {Failed} =
  VAR pt := EncodeProtocol(p);
  VAR tt := EncodeTransport(t);
  VAR ca := EncodeContact(pt, tt);
  VAR
    new := NEW(
             IluF.Exportation, contactArgs := ca, next := server.exports);
  PROCEDURE WithPS (ps: IluKernel.C_String) =
    PROCEDURE WithTS (ts: IluKernel.C_String) =
      BEGIN
        new.port := IluKernel.ilu_CreatePort(server.ks, ps, ts);
      END WithTS;
    BEGIN
      M3asC.TasS(tt, WithTS);
    END WithPS;
  VAR fd, closed: INTEGER;
  BEGIN
    LOCK server DO
      IF server.ks = NIL THEN
        RAISE
          Failed(
            NEW(
              Failure,
              info := "server " & server.id & " is broken (NIL ks)"))
      END (*if*)
    END (*lock*);
    M3asC.TasS(pt, WithPS);
    IF new.port = NIL THEN
      RAISE
        Failed(
          NEW(
            Failure,
            info := Fmt.F(
                      "couldn't create port from "
                        & "protocolinfo=%s, transportinto=%s", pt, tt)))
    ELSE
      server.exports := new;
    END (*if*);
    fd := IluKernel.ilu_FileDescriptorOfMooringOfPort(new.port, closed);
    IF closed # 0 THEN
      RAISE
        Failed(
          NEW(
            Failure, info := "newly created port (pi=" & pt & ", ti="
                               & tt & ") is closed"))
    END;
    EVAL
      Thread.Fork(
        NEW(ServerListener, ts := server, port := new.port, fd := fd));
  END Export_Server;

TYPE
  ServerListener = Thread.Closure OBJECT
                     <*lL, Ll, Main unconstrained*>

                     ts  : TrueServer;
                     port: IluKernel.Port;
                     fd  : INTEGER
                   OVERRIDES
                     apply := SpawnServers
                   END;

<*lL, Ll = {}*>

PROCEDURE SpawnServers (self: ServerListener): REFANY RAISES {} =
  VAR conn: IluKernel.Connection;
  VAR closed, sure: IluKernel.Bool;
  BEGIN
    LOOP
      TRY
        WaitForInput(self.fd, sure, NIL);
        conn := IluKernel.ilu_HandleNewConnection(self.port, closed);
        IF closed # 0 THEN EXIT END;
        IF conn # NIL THEN
          EVAL
            Thread.Fork(
              NEW(IluF.ConnectionWorker, ts := self.ts, conn := conn));
        ELSIF self.ts.HandleListenerFailure(
                NEW(
                  Failure,
                  info := "unable to create new connection in listener"))
                = FailureAction.Quit THEN
          EXIT;
        END (*if*);
      EXCEPT
      | Failed (f) =>
          IF self.ts.HandleListenerFailure(f) = FailureAction.Quit THEN
            EXIT;
          END (*if*);
      END (*try*);
    END (*loop*);
    IluKernel.ilu_ClosePort(self.port);
    RETURN NIL;
  END SpawnServers;

<*lL, Ll, Main unconstrained*>
VAR nullReads := 0;

PROCEDURE WorkForConnection (self: IluF.ConnectionWorker): REFANY
  RAISES {} =
  VAR stat: IluKernel.ReceiveRequestStatus;
  VAR cd: CallData;
  BEGIN
    WHILE self.work DO
      IF IluKernel.ilu_BlockingWaitForInputOnConnection(self.conn, NIL)
           = 0 THEN
        EXIT
      END;
      stat :=
        IluKernel.ilu_ReceiveRequest(
          self.conn, cd.call, cd.intro_type, cd.method, cd.serialNumber);
      CASE stat OF
      | IluKernel.ReceiveRequestStatus.conn_closed => EXIT;
      | IluKernel.ReceiveRequestStatus.no_request => INC(nullReads);
      | IluKernel.ReceiveRequestStatus.good_request =>
          IF IluKernel.ilu_ThreadPerRequest(self.conn) # 0 THEN
            EVAL Thread.Fork(NEW(CallWorker, cw := self, cd := cd))
          ELSE
            CallWork(self, cd)
          END (*if*);
      | IluKernel.ReceiveRequestStatus.builtin_meth => EVAL 0;
      ELSE
        IF self.ts.HandleWorkerFailure(
             NEW(
               Failure,
               info := "ReceiveRequest failed, stat="
                         & IluKernel.ReceiveRequestStatusName[ORD(stat)]))
             = Ilu.FailureAction.Quit THEN
          EXIT
        END (*handle*);
      END (*case stat*);
    END (*while work do*);
    IluKernel.ilu_CloseConnection(self.conn);
    RETURN NIL;
  END WorkForConnection;

TYPE
  CallData = RECORD
               <*lL, Ll, Main unconstrained*>

               call        : IluKernel.Call;
               intro_type  : IluKernel.ObjectType;
               method      : IluKernel.Method;
               serialNumber: IluKernel.Cardinal;
             END;

TYPE
  CallWorker = Thread.Closure OBJECT
                 <*lL, Ll, Main unconstrained*>

                 cw: IluF.ConnectionWorker;
                 cd: CallData;
               OVERRIDES
                 apply := WorkForCall
               END;

<*Main Invariant holds*>
<*Ll    >=    {conn's callmu, iomu} before,
  Ll disjoint {conn's callmu, iomu} after*>

PROCEDURE WorkForCall (self: CallWorker): REFANY RAISES {} =
  BEGIN
    CallWork(self.cw, self.cd);
    RETURN NIL;
  END WorkForCall;

PROCEDURE CallWork (cw: IluF.ConnectionWorker; cd: CallData) RAISES {} =
  VAR stub: IluKernel.StubProc;
  BEGIN
    stub := LOOPHOLE(cd.method.stubproc, IluKernel.StubProc);
    cd.call.ls_data := LOOPHOLE(cw, ADDRESS);
    stub(cd.call);
  END CallWork;

<*Main Invariant holds; Ll otherwise unconstrained*>

PROCEDURE ReadWait (    fd   : INTEGER;
                    VAR sure : IluKernel.Bool;
                        limit: IluKernel.FineTimePtr ) RAISES {} =
  BEGIN
    TRY
      WaitForInput(fd, sure, limit)
    EXCEPT
      Failed (f) => fDump := f; RAISE CantHappen
    END;
  END ReadWait;

VAR t0 := IluKernel.FineTime{0, 0};

PROCEDURE WaitForInput (    fd   : INTEGER;
                        VAR sure : IluKernel.Bool;
                            limit: IluKernel.FineTimePtr )
  RAISES {Failed} =
  VAR
    status       := -1;
    errno        := Uerror.EINTR;
    rdfds, exfds := Unix.FDSet{fd};
    now, delt    :  IluKernel.FineTime;
    tv           :  Utime.struct_timeval;
    timeout      :  UNTRACED REF Utime.struct_timeval := NIL;
  BEGIN
    WHILE status = -1 AND errno = Uerror.EINTR DO
      IF limit # NIL THEN
        IluKernel.ilu_shim_FineTime_Now(now);
        IluKernel.ilu_shim_FineTime_Sub(limit^, now, delt);
        IF IluKernel.ilu_shim_FineTime_Cmp(delt, t0) < 0 THEN
          sure := 0;
          RETURN
        END;
        tv.tv_sec := delt.s;
        tv.tv_usec :=
          IluKernel.ilu_rescale(
            delt.t, IluKernel.ilu_FineTimeRate, 1000000);
        timeout := ADR(tv);
      ELSE
        timeout := NIL
      END (*if*);
      status :=
        RTScheduler.IOSelect(
          Unix.MAX_FDSET, ADR(rdfds), NIL, ADR(exfds), timeout);
      errno := Uerror.errno;
    END (*while*);
    IF status < 0 THEN
      RAISE
        Failed(
          NEW(
            Failure,
            info := Fmt.F(
                      "select(%s) returned %s, errno %s", Fmt.Int(fd),
                      Fmt.Int(status), Fmt.Int(errno))))
    END (*if*);
    sure := status;
  END WaitForInput;

PROCEDURE WriteWait (    fd   : INTEGER;
                     VAR sure : IluKernel.Bool;
                         limit: IluKernel.FineTimePtr ) RAISES {} =
  VAR
    status       := -1;
    errno        := Uerror.EINTR;
    wrfds, exfds := Unix.FDSet{fd};
    now, delt    :  IluKernel.FineTime;
    tv           :  Utime.struct_timeval;
    timeout      :  UNTRACED REF Utime.struct_timeval := NIL;
  BEGIN
    WHILE status = -1 AND errno = Uerror.EINTR DO
      IF limit # NIL THEN
        IluKernel.ilu_shim_FineTime_Now(now);
        IluKernel.ilu_shim_FineTime_Sub(limit^, now, delt);
        IF IluKernel.ilu_shim_FineTime_Cmp(delt, t0) < 0 THEN
          sure := 0;
          RETURN
        END;
        tv.tv_sec := delt.s;
        tv.tv_usec :=
          IluKernel.ilu_rescale(
            delt.t, IluKernel.ilu_FineTimeRate, 1000000);
        timeout := ADR(tv);
      ELSE
        timeout := NIL
      END (*if*);
      status :=
        RTScheduler.IOSelect(
          Unix.MAX_FDSET, NIL, ADR(wrfds), ADR(exfds), timeout);
      errno := Uerror.errno;
    END (*while*);
    IF status < 0 THEN RAISE CantHappen; END (*if*);
    sure := status;
  END WriteWait;

PROCEDURE Ilu_Close (obj: Ilu.Object) =
  VAR l := obj.ILU_Qua_Type(obj.ILU_Get_Type());
  VAR s := l.ILU_Get_Server();
  VAR ot := l.ILU_Get_Type();
  VAR ko: IluKernel.Object;
  BEGIN
    IF s.ks = NIL THEN
      IF l.ko # NIL THEN RAISE CantHappen END;
      RETURN;
    END (*if*);
    IluKernel.ilu_EnterServer(s.ks, ot);
    IF l.ko # NIL THEN
      ko := l.ko;
      l.ko := NIL;
      KillStrongRef(l.sr);
      l.sr := NIL;
      IluKernel.ilu_RegisterLanguageSpecificObject(ko, NIL);
    END (*if*);
    IluKernel.ilu_ExitServer(s.ks, ot);
    RETURN;
  END Ilu_Close;

PROCEDURE Ilu_Close_Surrogate (obj: Ilu.Object) =
  VAR l := obj.ILU_Qua_Type(obj.ILU_Get_Type());
  BEGIN
    IF (NOT l.ilu_is_surrogate) THEN RAISE CantHappen END;
    Ilu_Close(l);
  END Ilu_Close_Surrogate;

<*lL >= {o's server}*>
<*lL >= {gcmu} if server true and o collectible*>
<*Ll, Main unconstrained*>
PROCEDURE KOfTrueO (o: Object): IluKernel.Object RAISES {Failed} =
  VAR ot := o.ILU_Get_Type();
  VAR l := o.ILU_Qua_Type(ot);
  VAR server: Server;
  VAR ts: TrueServer;
  VAR oh: TEXT := NIL;
  VAR csbh, coh: IluKernel.C_String := NIL;
  BEGIN
    IF l.ko # NIL THEN RETURN l.ko END;
    server := l.ILU_Get_Server();
    IF server # NIL THEN
      TYPECASE server OF
      | TrueServer => ts := server
      ELSE
        RAISE
          Failed(
            NEW(Failure, info := "true object has non-true server"))
      END (*typecase*)
    ELSE
      RAISE Failed(NEW(Failure, info := "true object has NIL server"))
    END (*if*);
    LOCK server DO
      IF server.ks = NIL THEN
        RAISE
          Failed(
            NEW(
              Failure,
              info := "server " & server.id & " is broken (NIL ks)"))
      END
    END;
    oh := ts.objtab.ObjectToHandle(l);
    coh := M3toC.CopyTtoS(oh);
    l.sr := NewStrongRef(l);
    l.ko := IluKernel.ilu_FindOrCreateTrueObject(coh, ts.ks, ot, l.sr);
    csbh := IluKernel.ilu_SBHOfObject(l.ko);
    IF csbh # NIL THEN l.sbh := M3toC.StoT(csbh) END;
    RETURN l.ko;
  END KOfTrueO;

<*lL = {} before;
  after:  Inside(o's server, o's type);
  excpn:  lL = {};
  forall conn: (Ll >= {conn.iomu}) => (Ll >= {conn.callmu});
  Main otherwise unconstrained*>
PROCEDURE KOfO (o: Object): IluKernel.Object RAISES {Failed} =
  VAR ot := o.ILU_Get_Type();
  VAR l := o.ILU_Qua_Type(ot);
  VAR server: Server := l.ILU_Get_Server();
  VAR ko: IluKernel.Object;
  BEGIN
    LOCK server DO
      IF server.ks = NIL THEN
        RAISE
          Failed(
            NEW(
              Failure,
              info := "server " & server.id & " is broken (NIL ks)"))
      END
    END;
    IluKernel.ilu_EnterServer(server.ks, ot);
    IF l.ko # NIL THEN RETURN l.ko END;
    IF NOT l.ilu_is_surrogate THEN
      TRY
        RETURN KOfTrueO(o);
      EXCEPT
        Failed (f) =>
          IluKernel.ilu_ExitServer(server.ks, ot);
          RAISE Failed(f)
      END (*try*);
    END (*if*);
    IluKernel.ilu_ExitServer(server.ks, ot);
    ko := IluKernel.ilu_ObjectOfSBH(l.c_sbh, l.c_mstid, ot);
    IF ko = NIL THEN
      RAISE
        Failed(
          NEW(
            Failure,
            info := "unable to re-create kernel surrogate object"))
    END;
    IF IluKernel.ilu_ClassOfObject(ko) # ot THEN
      (* ObjectOfSBH may not have acquired exactly the locks we must
         have now *)
      RAISE CantHappen
    END;
    IF IluKernel.ilu_GetLanguageSpecificObject(ko) = NIL THEN
      l.ko := ko;
      l.sr := NewStrongRef(l);
      IluKernel.ilu_RegisterLanguageSpecificObject(l.ko, l.sr);
    END (*if*);
    RETURN ko;
  END KOfO;

PROCEDURE SbhFromObject (o: Object): SBH RAISES {Failed} =
  VAR ot := o.ILU_Get_Type();
  VAR l := o.ILU_Qua_Type(ot);
  VAR server: Server := l.ILU_Get_Server();
  VAR ko := KOfO(o);
  VAR kot := IluKernel.ilu_ClassOfObject(ko);
  VAR ans := l.sbh;
  BEGIN
    IluKernel.ilu_ExitServer(server.ks, kot);
    IF ans = NIL THEN
      RAISE Failed(NEW(Failure, info := "Object has NIL SBH"))
    END;
    RETURN ans;
  END SbhFromObject;

<*lL, Ll, Main unconstrained*>

PROCEDURE IdOfObjectType (ot: ObjectType): TEXT =
  BEGIN
    RETURN M3toC.StoT(ot.unique_id);
  END IdOfObjectType;

PROCEDURE EncodeContact (pt, tt: TEXT): TEXT RAISES {Failed} =
  BEGIN
    RETURN pt & "|" & tt;
  END EncodeContact;

PROCEDURE EncodeProtocol (p: ProtocolInfo): TEXT RAISES {Failed} =
  BEGIN
    TYPECASE p OF
    | SunRpc2 (x) =>
        IF x.prognum = 0 THEN
          RETURN "sunrpc_"
        ELSE
          RETURN
            Fmt.F(
              "sunrpc_2_%s_%s", Fmt.Int(x.prognum), Fmt.Int(x.version))
        END;
    | Courier (x) =>
        IF x.prognum = 0 THEN
          RETURN "courier_"
        ELSE
          RETURN
            Fmt.F(
              "courier_%s_%s", Fmt.Int(x.prognum), Fmt.Int(x.version))
        END;
    ELSE
      RAISE
        Failed(
          NEW(Failure, info := "unrecognized kind of ProtocolInfo"))
    END
  END EncodeProtocol;

PROCEDURE EncodeTransport (t: TransportInfo): TEXT RAISES {Failed} =
  VAR host: TEXT;
  BEGIN
    TYPECASE t OF
    | TCP (x) =>
        IF x.host # 0 THEN
          host := FmtIpAddr(x.host)
        ELSE
          host := "localhost"
        END;
        RETURN "tcp_" & host & "_" & Fmt.Int(x.port);
    | UDP (x) =>
        IF x.host # 0 THEN
          host := FmtIpAddr(x.host)
        ELSE
          host := "localhost"
        END;
        RETURN "udp_" & host & "_" & Fmt.Int(x.port);
    | SPP (x) => RETURN "spp_" & FmtXnsAddr(x.addr);
    ELSE
      RAISE
        Failed(
          NEW(Failure, info := "unrecognized kind of TransportInfo"))
    END
  END EncodeTransport;

PROCEDURE FmtIpAddr (host: INTEGER): TEXT =
  VAR
    a1 := (host DIV (256 * 256 * 256)) MOD 256;
    a2 := (host DIV (256 * 256)) MOD 256;
    a3 := (host DIV 256) MOD 256;
    a4 := host MOD 256;
  BEGIN
    RETURN
      Fmt.Int(a1) & "." & Fmt.Int(a2) & "." & Fmt.Int(a3) & "."
        & Fmt.Int(a4);
  END FmtIpAddr;

PROCEDURE FmtXnsAddr (a: XnsAddr): TEXT =
  VAR buf: ARRAY [0 .. 2 * NUMBER(XnsHost) - 1] OF CHAR;
  BEGIN
    FOR i := 0 TO NUMBER(a.host) - 1 DO
      buf[2 * i + 0] := HexChars[a.host[i] DIV 16];
      buf[2 * i + 1] := HexChars[a.host[i] MOD 16];
    END (*for*);
    RETURN
      Fmt.F(
        "%04s%s%02s", Fmt.Unsigned(a.net), Text.FromChars(buf),
        Fmt.Unsigned(a.socket));
  END FmtXnsAddr;

CONST
  HexChars = ARRAY OF
               CHAR{
               '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A',
               'B', 'C', 'D', 'E', 'F'};

VAR srmu: MUTEX;

<*lL >= {srmu}*>

VAR srs: IntRefTbl.T;
VAR srLast: INTEGER := 0;

<*lL.sup < srmu*>

PROCEDURE NewStrongRef (r: REFANY): ADDRESS =
  BEGIN
    IF r = NIL THEN RAISE CantHappen END;
    LOCK srmu DO
      INC(srLast);
      IF srs.put(srLast, r) THEN RAISE CantHappen END;
      RETURN LOOPHOLE(srLast, ADDRESS);
    END (*lock*);
  END NewStrongRef;

PROCEDURE FromStrongRef (sr: ADDRESS): REFANY =
  VAR ans: REFANY;
  VAR key := LOOPHOLE(sr, INTEGER);
  BEGIN
    IF sr = NIL THEN RETURN NIL END;
    IF srs.in(key, ans) THEN RETURN ans END;
    RAISE CantHappen
  END FromStrongRef;

PROCEDURE KillStrongRef (sr: ADDRESS) =
  VAR key := LOOPHOLE(sr, INTEGER);
  VAR val: REFANY;
  BEGIN
    IF sr = NIL THEN RAISE CantHappen END;
    IF NOT srs.delete(key, val) THEN RAISE CantHappen END;
  END KillStrongRef;

(* timu = each Alarm *)

TYPE
  Alarm =
    MUTEX OBJECT
      <*lL >= {self}; Ll, Main unconstrained*>

      set: BOOLEAN := FALSE;
      for: Time.T := Time.Epoch;
      <*Main Invariant holds, Ll otherwise unconstrained, for calling*>
      proc: PROCEDURE (rock: ADDRESS) RAISES {} := NIL;
      rock: ADDRESS := NIL;
      doing := FALSE; (* calling proc(rock), or waiting for for? *)
      change: Thread.Condition;

      <*lL, Ll, Main unconstrained*>
      thread: Thread.T := NIL;
    END;

TYPE
  Alarmer =
    Thread.Closure OBJECT a: Alarm OVERRIDES apply := AnimateAlarm END;

<*lL, Ll = {}*>
PROCEDURE AnimateAlarm (self: Alarmer): REFANY RAISES {} =
  VAR a := self.a;
  VAR now, for: Time.T;
  VAR doit := FALSE;
  VAR proc: PROCEDURE (rock: ADDRESS) RAISES {};
  VAR rock: ADDRESS;
  BEGIN
    LOOP
      LOCK a DO
        EVAL Thread.TestAlert();
        WHILE NOT a.set DO Thread.Wait(a, a.change) END;
        now := Time.Now();
        IF Time.Compare(now, a.for) >= 0 THEN
          a.set := FALSE;
          doit := TRUE;
          proc := a.proc;
          rock := a.rock;
        ELSE
          for := a.for;
          doit := FALSE
        END (*if*);
        a.doing := doit;
      END (*lock a*);
      IF doit THEN
        proc(rock)
      ELSE
        TRY
          Time.AlertPauseUntil(for)
        EXCEPT
          Thread.Alerted => EVAL 0
        END;
      END (*if*);
    END (*loop*);
  END AnimateAlarm;

<*lL.sup < Alarm*>

PROCEDURE CreateAlarm (): ADDRESS =
  VAR a := NEW(Alarm, change := NEW(Thread.Condition));
  BEGIN
    a.thread := Thread.Fork(NEW(Alarmer, a := a));
    RETURN NewStrongRef(a);
  END CreateAlarm;

PROCEDURE SetAlarm (alarm: ADDRESS;
                    t: IluKernel.FineTime;
                    <*Main Invariant holds, Ll otherwise unconstrained*>
                    proc: PROCEDURE (rock: ADDRESS);
                    rock: ADDRESS                    ) =
  VAR a := NARROW(FromStrongRef(alarm), Alarm);
  VAR for := Time.T{t.s, t.t};
  BEGIN
    IF IluKernel.ilu_FineTimeRate # 1000000 THEN
      for.microseconds :=
        IluKernel.ilu_rescale(
          for.microseconds, IluKernel.ilu_FineTimeRate, 1000000)
    END;
    LOCK a DO
      IF NOT a.set THEN
        Thread.Signal(a.change)
      ELSIF (NOT a.doing) AND Time.Compare(for, a.for) < 0 THEN
        Thread.Alert(a.thread)
      END (*if*);
      a.set := TRUE;
      a.for := for;
      a.proc := proc;
      a.rock := rock;
    END (*lock a*);
  END SetAlarm;

PROCEDURE ClearAlarm (alarm: ADDRESS) =
  VAR a := NARROW(FromStrongRef(alarm), Alarm);
  BEGIN
    LOCK a DO
      IF a.set THEN
        a.set := FALSE;
        IF NOT a.doing THEN Thread.Alert(a.thread) END
      END;
    END (*lock a*);
  END ClearAlarm;

<*lL, Ll, Main unconstrained*>

PROCEDURE DumpThreads () =
  VAR m := NEW(MUTEX);
  BEGIN
    Thread.Release(m);
  END DumpThreads;

VAR wt: IluKernel.WaitTech;
VAR ml: IluKernel.MainLoop;

PROCEDURE StartIlu () =
  BEGIN
    ssMu := NEW(MUTEX);
    serverTab := TextToRefanyTable.New();
    srmu := NEW(MUTEX);
    srs := IntRefTbl.New();
    wt := IluKernel.WaitTech{ReadWait, WriteWait};
    ml :=
      IluKernel.MainLoop{
        NIL, NIL, NIL, NIL, NIL, NIL, CreateAlarm, SetAlarm, ClearAlarm};
    IluKernel.ilu_SetWaitTech(ADR(wt));
    IluKernel.ilu_SetMainLoop(ADR(ml));
  END StartIlu;

BEGIN
END Ilu.
