;;; -*- Mode:Lisp; Package: ILU; Syntax:COMMON-LISP; Base:10 -*-
#|
Copyright (c) 1991, 1992, 1993, 1994 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: ilu-franz.lisp,v 1.5 1994/05/10 01:02:31 janssen Exp $
|#

(cl:in-package :ilu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Foreign function utilities:
;;;
;;;  (define-c-function LISP-NAME C-NAME ARGS RETURN-TYPE &key INLINE)
;;;
;;;  (register-lisp-object VAL) => TAG (of type fixnum)
;;;
;;;  (lookup-registered-lisp-object TAG) => VAL
;;;
;;;  (char*-to-string C-POINTER) => STRING
;;;
;;;  (string-to-char* STRING) => C-POINTER
;;;
;;;  (initialize-locking)
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(eval-when (compile eval load)

#+(and excl svr4)
(push :lisp-understands-ansi-c-parameter-passing cl:*features*)

(defvar *debug-franz* nil "when non-NIL, enables debugging messages")

(defun franz-return-type (type)
  (cond
   ((eq type :pointer)
    :integer)
   ((eq type :cardinal)
    :integer)
   ((eq type :short-cardinal)
    :fixnum)
   ((eq type :ilu-call)
    :integer)
   ((eq type :ilu-class)
    :integer)
   ((eq type :ilu-object)
    :integer)
   (t type)))

(defun franz-argument-type (arg)
  (ecase arg
    (:pointer #+allegro-v4.2 'ff:foreign-address
	      #-allegro-v4.2 '(unsigned-byte 32))
    (:cardinal '(unsigned-byte 32))
    (:short-cardinal '(unsigned-byte 16))
    (:long-cardinal '(simple-array (unsigned-byte 32) 2))
    (:integer 'integer)
    (:short-integer '(signed-byte 16))
    (:long-integer '(simple-array (unsigned-byte 32) 2))
    (:string 'simple-string)
    (:wide-string '(simple-array (unsigned-byte 16)))
    (:byte '(unsigned-byte 8))
    (:single-float 'single-float)
    (:double-float 'double-float)
    (:long-float 'double-float)
    (:byte-vector '(simple-array (unsigned-byte 8)))
    (:byte-sequence '(simple-array (unsigned-byte 8)))

    (:ilu-object #+allegro-v4.2 'ff:foreign-address
		 #-allegro-v4.2 '(unsigned-byte 32))
    (:ilu-class #+allegro-v4.2 'ff:foreign-address
		#-allegro-v4.2 '(unsigned-byte 32))
    (:ilu-call #+allegro-v4.2 'ff:foreign-address
	       #-allegro-v4.2 '(unsigned-byte 32))

    (:fixnum 'fixnum)

    (:cardinal-pointer '(simple-array (unsigned-byte 32) 1))
    (:shortcardinal-pointer '(simple-array (unsigned-byte 16) 1))
    (:integer-pointer '(simple-array integer 1))
    (:shortinteger-pointer '(simple-array (signed-byte 16) 1))
    (:byte-pointer '(simple-array (unsigned-byte 8) 1))
    (:real-pointer '(simple-array double-float 1))
    (:shortreal-pointer '(simple-array single-float 1))
    (:string-pointer '(simple-array #+allegro-v4.2 ff:foreign-address #-allegro-v4.2 (unsigned-byte 32) 1))
    (:ilu-object-pointer '(simple-array #+allegro-v4.2 ff:foreign-address #-allegro-v4.2 (unsigned-byte 32) 1))))

)

(defun franz-callable-arg (arg)
  (list (first arg)
	(ecase (second arg)
	  (:fixnum :fixnum)
	  (:pointer :unsigned-long))))

(defmacro define-c-function (lisp-name doc-string c-name args return-type
			     &key inline)
  (declare (ignore doc-string))
  `(ff:defforeign ',lisp-name
		  :arguments ',(mapcar #'franz-argument-type args)
		  :return-type ',(franz-return-type return-type)
		  :entry-point ',(ff:convert-to-lang c-name :language :c)
		  ,@(when (or (find :single-float args) (eq return-type :single-float)) '(:prototype t))
		  ,@(when inline
		      '(:callback nil :arg-checking nil :call-direct t)))
  )

(defmacro register-lisp-object (obj)
  `(ff:register-value ,obj)
  )

(defmacro unregister-lisp-object (index)
  `(setf (ff:lisp-value ,index) nil)
  )

(defmacro lookup-registered-lisp-object (index)
  `(ff:lisp-value ,index)
  )

(defmacro char*-to-string (pointer)
  `(ff:char*-to-string ,pointer))

(defmacro string-to-char* (str)
  `(ff:string-to-char* ,str))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for using UNICODE character strings
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro construct-unicode-character-from-code (code)
  #+excl `(if (and (>= ,code 0) (< ,code 128)) (code-char ,code)
	    (progn
	      (warn "No Lisp character for Unicode value ~a." ,code)
	      (code-char 127)))
  #-excl `(declare (ignore code))
  #-excl `(error "Unable to construct Unicode characters in this lisp"))

(defmacro construct-iso-latin-1-character-from-code (code)
  #+excl `(code-char ,code)
  #-excl `(declare (ignore ,code))
  #-excl `(error "Unable to construct Unicode characters in this lisp"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for coordinating blocking reads & writes
;;;
;;;    These functions have to be callable from C.  Franz provides
;;;    support for registering lisp functions, then calling them
;;;    via an index from C.  There is also Franz-specific code in
;;;    ilu-franz-skin.c to support this.
;;;
;;;  Also defines ``wait-for-input-available'', which is used by ilu-server:
;;;
;;;    (wait-for-input-available UNIX-FILE-DESCRIPTOR)
;;;
;;;  It just blocks until something can be read from UNIX-FILE-DESCRIPTOR.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(ff:defun-c-callable wait-for-input-on-fd ((fd :fixnum) (time :unsigned-long))
  (when *debug-franz*
    (format t "(Franz:  Waiting for input on fd ~d with timeout ~d (milliseconds).)~%" fd time))
  (if (= 0 time)
      (progn (mp:wait-for-input-available fd) 1)
    (if (mp:wait-for-input-available fd :timeout (/ (double-float time) 1000.0))
	1
      0))
  )

;; can't wait for output stream in Allegro

(define-c-function ilufranz_set-wait-tech
    "Register I/O wait functions with kernel"
    "ilufranz_SetWaitTech" (:pointer :pointer) :void)

(defun wait-for-input-available (fd)
  (mp:wait-for-input-available fd))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  An implementation of ILU alarms.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct ilu-alarm
  (cvar)
  (process)
  (time)
  (closure))

#.(defconstant +unix-epoch+ (encode-universal-time 0 0 0 1 1 1970 0))

(defmacro get-unix-time ()
  (- (get-universal-time) +unix-epoch+))

(defvar *alarm-thread-counter* 0 "counter to identify different alarms")

(define-c-function call-alarm-closure "Call the C closure represented by the alarm"
  "ilufranz_CallAlarmClosure" (:pointer :pointer) :void :inline t)

;; each alarm has a process associated with it.  The process runs this loop,
;; forever.  The loop waits for someone to set the alarm (and in the process
;; notify the condition variable), then sleeps until it's time to signal the
;; alarm.  It then checks the alarm time again (someone may have unset the
;; alarm in the meantime), and then calls the indicated C function, indirectly
;; through call-alarm-closure.  It then goes back to wait for someone to set
;; the alarm again.
(defun do-ilu-alarm (alarm-struct)
  (loop
    (catch :unset-alarm
      (unless (ilu-process:without-scheduling (and (ilu-alarm-time alarm-struct) (ilu-alarm-closure alarm-struct)))
	(ilu-process:condition-variable-wait (ilu-alarm-cvar alarm-struct)))
      (when (ilu-alarm-time alarm-struct)
	(when *debug-franz*
	  (format t "(Franz:  ILU alarm ~s:  waiting till ~a)~%" (decode-universal-time (round (+ +unix-epoch+ (ilu-alarm-time alarm-struct))))))
	(mp:process-sleep (- (ilu-alarm-time alarm-struct) (get-unix-time))))
      (when (ilu-process:without-scheduling (and (ilu-alarm-time alarm-struct) (ilu-alarm-closure alarm-struct)))
	(call-alarm-closure (car (ilu-alarm-closure alarm-struct)) (cdr (ilu-alarm-closure alarm-struct)))))))

(ff:defun-c-callable create-alarm ()
  (when *debug-franz*
    (format t "(Franz:  Creating alarm...)~%"))
  (let ((alarm-struct (make-ilu-alarm :cvar (ilu-process:make-condition-variable))))
    (setf (ilu-alarm-process alarm-struct)
      (ilu-process:fork-process
       (format nil "ILU Alarm Thread ~d" (incf *alarm-thread-counter*))
       #'do-ilu-alarm alarm-struct))
    (register-lisp-object alarm-struct)))

(ff:defun-c-callable set-alarm (alarm time p1 p2)
  (let ((alarm-struct (lookup-registered-lisp-object alarm)))
    (ilu-process:without-scheduling
      (setf (ilu-alarm-time alarm-struct) (/ time 1000.0))
      (setf (ilu-alarm-closure alarm-struct) (cons p1 p2)))
    (ilu-process:condition-variable-notify (ilu-alarm-cvar alarm-struct))))

(defun wake-process ()
  (throw :unset-alarm nil))

(ff:defun-c-callable unset-alarm (alarm)
  (let ((alarm-struct (lookup-registered-lisp-object alarm)))
    (ilu-process:without-scheduling
      (setf (ilu-alarm-time alarm-struct) nil)
      (setf (ilu-alarm-closure alarm-struct) nil))
    (ilu-process:process-interrupt (ilu-alarm-process alarm-struct) #'wake-process)))

(define-c-function ilufranz_register-alarms
    "Initialize all the alarm mechanism"
  "ilufranz_SetAlarms" (:pointer :pointer :pointer) :void :inline t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for building and using locks.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; These 9 functions are pretty generic.  They can be used with
;;; any lisp for which the ilu-process package works.

(ff:defun-c-callable create-mutex ((d1 :unsigned-long) (d2 :unsigned-long))
  (let ((s1 (ff:char*-to-string d1))
	(s2 (ff:char*-to-string d2)))
    (when *debug-franz*
      (format t "(Franz:  Creating mutex '~a~a'...)~%" s1 s2))
    (register-lisp-object (ilu-process:make-process-lock :name (format nil "ILU mutex '~a~a'" s1 s2)))))

(ff:defun-c-callable acquire-mutex ((m :fixnum))
  (when *debug-franz*
    (format t "(Franz:  Acquiring mutex ~s)~%" (lookup-registered-lisp-object m)))
  (let ((mutex (lookup-registered-lisp-object m)))
    (unless (eq (ilu-process:current-process) (ilu-process:process-lock-locker mutex))
      (ilu-process:process-lock mutex)))
  (values))

(ff:defun-c-callable hold-mutex ((m :fixnum))
  (when *debug-franz*
    (format t "(Franz:  Hold mutex ~s?)~%" (lookup-registered-lisp-object m)))
  (let ((lock (lookup-registered-lisp-object m)))
    (unless (eq (ilu-process:current-process) (ilu-process:process-lock-locker lock))
      (error "Mutex ~s not held by current process!" lock))))	

(ff:defun-c-callable release-mutex ((m :fixnum))
  (when *debug-franz*
    (format t "(Franz:  Releasing mutex ~s)~%" (lookup-registered-lisp-object m)))
  (ilu-process:process-unlock (lookup-registered-lisp-object m))
  (values))

(ff:defun-c-callable destroy-mutex ((m :fixnum))
  (release-mutex m)
  (unregister-lisp-object m)
  (values))

(ff:defun-c-callable cvar-create ((d1 :unsigned-long)(d2 :unsigned-long))
  (let ((s1 (ff:char*-to-string d1))
	(s2 (ff:char*-to-string d2)))
    (when *debug-franz*
      (format t "(Franz:  Creating cvar '~a~a'...)~%" s1 s2))
    (register-lisp-object
     (ilu-process:make-condition-variable
      :name (format nil "ILU cvar '~a~a'" s1 s2)))))

(ff:defun-c-callable cvar-notify ((v :fixnum))
  (let ((var (lookup-registered-lisp-object v)))
    (if var
	(ilu-process:condition-variable-notify var)))
  (values))

(ff:defun-c-callable cvar-wait ((v :fixnum) (m :fixnum))
  (let ((var (lookup-registered-lisp-object v))
	(mutex (lookup-registered-lisp-object m)))
    (if (and var mutex)
	(ilu-process:without-scheduling
	  (when *debug-franz*
	    (format t "(Franz:  Releasing mutex ~s)~%" mutex))
	  (ilu-process:process-unlock mutex)
	  (ilu-process:condition-variable-wait var))))
  (values))

(ff:defun-c-callable cvar-destroy ((v :fixnum))
  (let ((var (lookup-registered-lisp-object v)))
    (if var
	(ilu-process:condition-variable-notify var)))
  (values))    
  
(define-c-function ilufranz_register-lock-tech
    "Register Franz locking with kernel"
  "ilufranz_SetLockTech"
  (:pointer :pointer :pointer :pointer :pointer  ;; mutex create, acquire, hold, release, destroy
	    :pointer :pointer :pointer :pointer) ;; condition variable create, notify, destroy, wait

  :void)

(defun initialize-locking ()
  (unless mp:*all-processes*
    (mp:start-scheduler))
  (let ((read-index (ff:register-function 'wait-for-input-on-fd)))
    (ilufranz_set-wait-tech read-index 0))
  (ilufranz_register-lock-tech
   (ff:register-function 'create-mutex)
   (ff:register-function 'acquire-mutex)
   (ff:register-function 'hold-mutex)
   (ff:register-function 'release-mutex)
   (ff:register-function 'destroy-mutex)
   (ff:register-function 'cvar-create)
   (ff:register-function 'cvar-notify)
   (ff:register-function 'cvar-destroy)
   (ff:register-function 'cvar-wait))
  (ilufranz_register-alarms
   (ff:register-function 'create-alarm)
   (ff:register-function 'set-alarm)
   (ff:register-function 'unset-alarm)))
