/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: tcp.c,v 1.41 1994/05/12 01:02:22 janssen Exp $ */
/* Last tweaked by Mike Spreitzer April 22, 1994 3:06 pm PDT */

#define _BSD_SOURCE

#include <signal.h>	/* putting this after the #define of BSD loses */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/param.h>		/* for MAXHOSTNAMELEN */
#include <netdb.h>		/* for gethostbyname() */
#include <arpa/inet.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/filio.h>		/* for FIONREAD */
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "ilu.h"
#include "iluntrnl.h"

#include "transprt.h"
#include "mooring.h"

#include "bffrlist.h"

/*L1_sup < trmu; L2, Main unconstrained*/
ilu_TransportClass _ilu_tcp_TransportClass(void);

typedef struct tcpparms {
  /*L1, L2, Main unconstrained*/
  
  ilu_string hostname;
  ilu_cardinal port;
  
  /*L1 >= {connection's iomu}*/
  bufferList buffer;
} *TCPParms;
/* What goes in the data field of a TCP ilu_Transport or ilu_Mooring */

/*L1, L2, Main unconstrained*/

#define TCP_HOSTNAME(a) (((TCPParms)(a))->hostname)
#define TCP_PORT(a) (((TCPParms)(a))->port)

/*L1 >= {connection's iomu}*/
#define TCP_BUFFER(a) (((TCPParms)(a))->buffer)

#define MAXDUMP		10000

/***********************************************************************
************************************************************************
************************************************************************
************************** Some local utilities ************************
************************************************************************
************************************************************************
***********************************************************************/

/*L1 >= {trmu}*/
ilu_string _ilu_tcp_CurrentHostInetName(void)
{
  char name[100];
  static ilu_string inetname = NULL;
  struct hostent *he;
  struct in_addr *hea;

  if (inetname == NULL)
    {
      if (gethostname(name,sizeof(name)) != 0) {
	perror("no hostname for this machine!");
	return (NULL);
      }
  
      he = gethostbyname(name);
      hea = (struct in_addr *) (he->h_addr_list[0]);
      inetname = _ilu_Strdup((ilu_string) inet_ntoa(*hea));
    }
  return (inetname);
}

/**********************************************************************
***********************************************************************
***********************************************************************
***** First, the methods for the TCP Transport ************************
***********************************************************************
***********************************************************************
**********************************************************************/

/*L2, Main unconstrained*/
/*L1 >= {trmu}*/

static ilu_boolean SigPIPEHandler = FALSE;

static void _tcp_HandleSigPIPE (void)
{
  /* Ignore SIGPIPEs, since we can occasionally get them.  There
     should be a way to do this that cooperates with other modules
     that want to handle signals.

     [Thanks to Dave Nichols <nichols@parc.xerox.com> for this code.]  */

typedef void (*sighandler_t)(int);

  sighandler_t old_handler;

  if ((old_handler = signal(SIGPIPE, SIG_IGN)) != SIG_DFL)
    /* Oops!  Someone's using this one */
    signal(SIGPIPE, old_handler);

  SigPIPEHandler = TRUE;
}

/*L1 unconstrained*/

static ilu_private _tcp_InterpretInfo (ilu_string info)
{
  char hostname[1000];
  ilu_cardinal port;
  ilu_boolean buffered = FALSE;

  if (*info == 'b')
    buffered = TRUE;
  if ((sscanf (info + (buffered ? 1 : 0), "tcp_%[^_]_%u",
	       hostname, &port)) == 2)
    {
      TCPParms new = (TCPParms)
			     malloc(sizeof(struct tcpparms));
      new->hostname = _ilu_Strdup(hostname);
      new->port = port;
      if (buffered)
	{
	  new->buffer = (bufferList) malloc(sizeof(struct bufferlist_s));
	  new->buffer->size = 0;
	  new->buffer->head = NULL;
	  new->buffer->tail = NULL;
	}
      else
	new->buffer = NULL;
      return ((ilu_private) new);
    }
  else
    return (NULL);
}

static void DumpPacket (ilu_byte *packet, ilu_shortcardinal length)
{
  ilu_cardinal dumplength;
  int n;
  ilu_byte c;
  register ilu_integer i, j;

  if (length > MAXDUMP)
    {
      fprintf (stderr, "Request to dump packet of %u bytes.  Only %u bytes being dumped.\n",
	       length, MAXDUMP);
      dumplength = MAXDUMP;
    }
  else
    dumplength = length;
  if (packet == NULL)
    {
      fprintf (stderr, "Attempt to dump NULL packet.\n");
      return;
    }
  fprintf (stderr, "DumpPacket of packet %p, length ", (void *) packet);
  fprintf (stderr, "%u bytes, dumping %lu bytes:\n", length, dumplength);
  for (i = 0;  i < dumplength;  i += 16)
    {
      fprintf (stderr, "%6ld:  ", i);
      for (j = 0;  j < 16 AND (i + j) < dumplength;  j += 1)
	fprintf (stderr, "%02x%s ", packet[i + j],
		 ((j % 4) == 3) ? " " : "");
      n = (((16-j)/4)*(13)+((16-j)%4)*3)+1; /* padding before ascii */
      fprintf (stderr, "%*.*s", n, n, "");
      for (j = 0;  j < 16 AND (i + j) < dumplength;  j += 1)
	{
	  c = packet[i + j];
	  fprintf (stderr, "%c", ((c >= ' ') && (c <= '~')) ? (char) c
							    : '.');
	}
      fprintf (stderr, "\n");
    }
}

/*L2 >= {conn's iomu}*/
/*L1, Main unconstrained*/
static ilu_integer _tcp_NDataPresent (ilu_Transport conn)
{
  long count;
  ilu_boolean status = ioctl (conn->tr_fd, FIONREAD, &count);
  return ( (status != 0) ? -1 : (count > 0) ? 1 : 0 );
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_boolean _tcp_WaitForInput (ilu_Transport t, ilu_boolean *sure,
				      ilu_FineTime *limit)
{
  _ilu_WaitForInputOnFD(t->tr_fd, sure, limit);
  return (TRUE);
}

static ilu_boolean _tcp_FlushOutput (ilu_Transport self)
{
  bufferList buf;
  bufferElement new, old;
  ilu_boolean status = FALSE;
  ilu_integer i;

  if (self == NULL)
    return (FALSE);
  
  if ((buf = TCP_BUFFER(transport_data(self))) != NULL && buf->size > 0)
    {
      for (new = buf->head, status = TRUE;
	   status && buf->size > 0 && new != NULL;  buf->size -= 1)
	{
	  ilu_byte lenbuf[4];
	  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		(stderr, "%s %ld bytes from %p to fd %d.\n",
		 "_tcp_FlushOutput:  writing", new->size,
		 (void *) new->packet, self->tr_fd));

	  lenbuf[0] = ((new->size >> 24) & 0xFF) | 0x80;
	    /* what if !new->EOM ? */
	  lenbuf[1] = (new->size >> 16) & 0xFF;
	  lenbuf[2] = (new->size >> 8) & 0xFF;
	  lenbuf[3] = new->size & 0xFF;
	  if ((i = _ilu_Write (self->tr_fd, lenbuf, 4)) < 4)
	    {
	      fprintf (stderr,
		       "%s %u of %d bytes signalled error \"%s\".\n",
		       "_tcp_SendMessage:  output to fd", self->tr_fd, 4,
		       (i < 0) ? ANSI_STRERROR(errno)
			       : ((i == 0) ? "Connection lost"
					   : "not enough bytes written"));
	      status = FALSE;
	    }
	  else if ( (i = _ilu_Write (self->tr_fd, new->packet, new->size))
		    < new->size)
	    {
	      fprintf (stderr,
		       "%s %u of %d bytes signalled error \"%s\".\n",
		       "_tcp_SendMessage:  output to fd", self->tr_fd, 4,
		       (i < 0) ? ANSI_STRERROR(errno)
			       : ((i == 0) ? "Connection lost"
					   : "not enough bytes written"));
	      status = FALSE;
	    }
	  else
	    {
	      old = new;
	      new = new->next;
	      free(old->packet);
	      free(old);
	    }
	}
      if (status)
	{ /* Wrong; what if some worked and some didn't! */
	  buf->size = 0;
	  buf->head = NULL;
	  buf->tail = NULL;
	}
    }

  return (status);
}

/*L1, Main unconstrained*/

static ilu_boolean _tcp_AddPacketToBuffer(ilu_Transport self,
			ilu_byte *packet, ilu_cardinal size,
			ilu_boolean eom)
{
  bufferList buf;
  bufferElement new;

  if (self == NULL)
    return (FALSE);

  buf = TCP_BUFFER(transport_data(self));
  new = (bufferElement) malloc(sizeof(struct bufferlist_element_s));

  new->packet = packet;
  new->size = size;
  new->next = NULL;
  new->EOM = eom;
  if (buf->tail == NULL)
    buf->head = buf->tail = new;
  else
    buf->tail->next = new;
  buf->size += 1;

  return (TRUE);
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_boolean _tcp_SendMessage (ilu_Transport conn, ilu_byte *buffer,
				     ilu_cardinal count)
{
  ilu_boolean status = TRUE;

  if (conn == NULL)
    return (FALSE);

  if (  TCP_BUFFER(transport_data(conn)) == NULL
	|| (!_tcp_AddPacketToBuffer(conn, buffer, count, FALSE)))
    {
      ilu_byte lenbuf[4];
      ilu_boolean status;
      ilu_integer i;

      _tcp_FlushOutput (conn);

      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	    (stderr,
	     "_tcp_SendMessage:  writing %lu bytes from %p to fd %d.\n",
	     count, (void *) buffer, conn->tr_fd));

      if ((_ilu_DebugLevel & PACKET_DEBUG) != 0)
	DumpPacket(buffer, count);

      lenbuf[0] = ((count >> 24) & 0xFF) | 0x80;
      lenbuf[1] = (count >> 16) & 0xFF;
      lenbuf[2] = (count >> 8) & 0xFF;
      lenbuf[3] = count & 0xFF;
      status = TRUE;
      if ((i = _ilu_Write (conn->tr_fd, lenbuf, 4)) < 4)
	{
	  fprintf (stderr, "%s %u of %d bytes signalled error \"%s\".\n",
		   "_tcp_SendMessage:  output to fd", conn->tr_fd, 4,
		   (i < 0) ? ANSI_STRERROR(errno)
			   : ((i == 0) ? "Connection lost"
				       : "not enough bytes written"));
	  status = FALSE;
	}
      else
	{
	  if ((i = _ilu_Write (conn->tr_fd, buffer, count)) < count)
	    {
	      fprintf (stderr,
		       "_tcp_SendMessage:  output to fd %u of %lu bytes"
		       " signalled error \"%s\".\n",
		       conn->tr_fd, count,
		       (i < 0) ? ANSI_STRERROR(errno)
			       : ((i == 0) ? "Connection lost"
					   : "not enough bytes written"));
	      status = FALSE;
	    }
	  else
	    status = TRUE;
	}
      free(buffer);
    };
  return (status);
}

/*L1, L2, Main unconstrained*/

/*Main Invariant holds*/
static ilu_boolean _tcp_Connect (ilu_Transport self)
{
  ilu_integer fd;
  struct sockaddr_in sin;
  struct hostent *hp;
  ilu_boolean status = TRUE;
  int flags;

  _ilu_AutoSetDebugLevel();
  _ilu_AcquireMutex(ilu_trmu);
  if (!SigPIPEHandler)
    _tcp_HandleSigPIPE();
  _ilu_ReleaseMutex(ilu_trmu);

  if (self == NULL)
    return (FALSE);

  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	(stderr, "_tcp_Connect:  connecting to host %s, port %lu...\n",
	 TCP_HOSTNAME(transport_data(self)),
	 TCP_PORT(transport_data(self))));

  memset((ilu_string ) &sin, 0, sizeof(sin));
  if ((fd = OS_SOCKET(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	    (stderr, "_tcp_Connect:  socket call failed:  %s.\n",
	     ANSI_STRERROR(errno)));
      status = FALSE;
    }
  else
    {
      (void) OS_SETSOCKOPT(fd, SOL_SOCKET, SO_REUSEADDR,
			   (ilu_string ) NULL, 0);
      (void) OS_SETSOCKOPT(fd, SOL_SOCKET, SO_USELOOPBACK,
			   (ilu_string ) NULL, 0);

/* The following was suggested by Ken Birman of the Cornell ISIS project.
 * On SunOS it prevents the kernel from delaying this packet in hopes
 * of merging it with a quickly-following one.
 */
#ifdef IPPROTO_TCP
#ifdef TCP_NODELAY
      {
	int one = 1;
	OS_SETSOCKOPT (fd, IPPROTO_TCP, TCP_NODELAY, &one, sizeof(one));
      }
#endif /* TCP_NODELAY */
#endif /* IPPROTO_TCP */     

      sin.sin_family = AF_INET;
      sin.sin_port = htonl(TCP_PORT(transport_data(self)));

      sin.sin_addr.s_addr = OS_INETADDR( TCP_HOSTNAME(
		transport_data( self)));
      if (sin.sin_addr.s_addr == -1 || sin.sin_addr.s_addr == 0)
	if ((hp = gethostbyname (TCP_HOSTNAME(transport_data(self))))
	    != NULL)
	  memcpy ((char *) &sin.sin_addr, hp->h_addr, hp->h_length);

      if (sin.sin_addr.s_addr == -1 || sin.sin_addr.s_addr == 0)
	{
	  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		(stderr, "_tcp_Connect:  Invalid host name (%s)\n",
		 TCP_HOSTNAME(transport_data(self))));
	  status = FALSE;
	}
      else if ((flags = fcntl(fd, F_GETFL, 0)) < 0 OR
	       fcntl(fd, F_SETFL, flags | O_NONBLOCK) < 0)
	{
	  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		(stderr, "_tcp_Connect:  Failed to set socket (descriptor %ld, host %s) non-blocking.  Error \"%s\".\n",
		 fd, TCP_HOSTNAME(transport_data(self)),
		 ANSI_STRERROR(errno)));
	  status = FALSE;
	}
      else
	{
	  if (OS_CONNECT(fd, (struct sockaddr *) &sin, sizeof(sin)) == 0)
	    {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr, "_tcp_Connect:  connected to %s[%lu]\n",
		     inet_ntoa(sin.sin_addr),
		     TCP_PORT(transport_data(self))));
	      self->tr_fd = fd;
	      status = TRUE;
	    }
	  else if (errno == EINPROGRESS) {
	    struct sockaddr peername;
	    int pnlen = sizeof(peername);
	    ilu_boolean sure;
	    _ilu_WaitForOutputOnFD(fd, &sure, NULL);
	    if (getpeername(fd, &peername, &pnlen) == 0) {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr,
		     "_tcp_Connect:  eventually connected to %s[%lu]\n",
		     inet_ntoa(sin.sin_addr),
		     TCP_PORT(transport_data(self))));
	      self->tr_fd = fd;
	      status = TRUE;
	      }
	    else {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr,
		     "_tcp_Connect: connect to %s[%lu] failed "
		     "(no meaningful errno available).\n",
		     OS_INETNTOA(sin.sin_addr),
		     TCP_PORT(transport_data(self)) ));
	      OS_CLOSE(fd);
	      status = FALSE;
	      }
	    }
	  else
	    {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr, "_tcp_Connect: connect to %s[%lu] failed,"
		     " error %d (%s)\n",
		     OS_INETNTOA(sin.sin_addr),
		     TCP_PORT(transport_data(self)), errno,
		     ANSI_STRERROR(errno)));
	      OS_CLOSE(fd);
	      status = FALSE;
	    }
	}
    }
  return (status);
}

/*L2 >= {conn's iomu}*/

static void _tcp_Close (ilu_Transport self)
{
  if (self == NULL)
    return;

  OS_CLOSE (self->tr_fd);
  return;
}

/*Main Invariant holds*/
static ilu_boolean _tcp_ReadMessage (ilu_Transport self,
				     ilu_bytes *buffer,
				     ilu_cardinal *size)
{
  ilu_byte lenbuf[4];
  int i;
  ilu_cardinal bytesToRead, bytesRead, totalSize = 0;
  ilu_bytes packet = NULL;
  ilu_boolean status = FALSE;
  ilu_boolean lastPacket = FALSE;

  if (self == NULL)
    return(FALSE);

  do {
    if ((i = _ilu_Read (self->tr_fd, lenbuf, 4)) < 4)
      {
	DEBUG((INCOMING_DEBUG | TCP_DEBUG),
	      (stderr,
	       "%s %d returns %d (errno is %s) %s.\n",
	       "_tcp_ReadMessage:  read on fd",
	       self->tr_fd, i, ANSI_STRERROR(errno),
	       "while reading packet length"));
	status = FALSE;
      }
    else
      {
	DEBUG((INCOMING_DEBUG | TCP_DEBUG),
	      (stderr, "_tcp_ReadMessage:  lenbuf is %d.%d.%d.%d\n",
	       lenbuf[0], lenbuf[1], lenbuf[2], lenbuf[3]));
	lastPacket = (lenbuf[0] & 0x80) != 0;
	bytesToRead = ((lenbuf[0] & 0x7F) << 24) | (lenbuf[1] << 16)
		      | (lenbuf[2] << 8) | lenbuf[3];
	bytesRead = 0;
	if (packet == NULL)
	  packet = (ilu_bytes) malloc(bytesToRead);
	else
	  packet = (ilu_bytes) realloc (packet, bytesToRead + totalSize);
	status = TRUE;
	i = _ilu_Read (self->tr_fd, packet + totalSize + bytesRead,
		       bytesToRead);
	if ((i < 0) || ((i == 0) && (bytesToRead > 0)))
	  {
	    DEBUG((INCOMING_DEBUG | TCP_DEBUG),
		  (stderr,
		   "_tcp_ReadMessage:  read on fd %d returns %d (%s)"
		   " while reading packet of length %lu,"
		   " having already read %lu bytes.\n",
		   self->tr_fd, i, ANSI_STRERROR(errno),
		   bytesToRead, bytesRead));
	    status = FALSE;
	  }
	else
	  {
	    DEBUG((INCOMING_DEBUG | TCP_DEBUG),
		  (stderr,
		   "_tcp_ReadMessage:  read %d bytes from fd %d"
		   " while asking for %lu bytes.\n",
		   i, self->tr_fd, bytesToRead));
	    bytesRead += i;
	    bytesToRead -= i;
	  }
	totalSize += bytesRead;
      }
  } while (status AND (NOT lastPacket));

  if (status)
    {
      *buffer = packet;
      *size = totalSize;
    }
  else
    {
      if (packet != NULL)
	free(packet);
    }

  if (status AND (_ilu_DebugLevel & PACKET_DEBUG) != 0)
    DumpPacket(*buffer, *size);

  return (status);
}

/**********************************************************************
***********************************************************************
***********************************************************************
**** Now the methods for the TCP Mooring ******************************
***********************************************************************
***********************************************************************
**********************************************************************/

/*L1, L2, Main unconstrained*/

static ilu_string _tcp_FormHandle (ilu_refany parameterblock)
{
  TCPParms parms = (TCPParms) parameterblock;
  char buf[1000];

  if (parms == NULL)
    return (NULL);
  sprintf (buf, "%stcp_%s_%lu", (TCP_BUFFER(parms) == NULL) ? "" : "b",
	   TCP_HOSTNAME(parms), TCP_PORT(parms));
  return (_ilu_Strdup(buf));
}

/*L1_sup < trmu*/
static ilu_Transport _tcp_AcceptClient (ilu_Mooring self)
{
  register ilu_integer ns;
  struct sockaddr_in sin;
  ilu_Transport new = NULL;
  int addrlen;
  int flags;
  char addrDescr[64];

  _ilu_AutoSetDebugLevel();

  if (self == NULL)
    return (NULL);

  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	(stderr, "_tcp_AcceptClient: mooring file descriptor is %d\n",
	 self->mo_fd));

  addrlen = sizeof(sin);

  if ((ns = OS_ACCEPT(self->mo_fd, (struct sockaddr *) &sin, &addrlen)) < 0)
    {
      perror ("_tcp_AcceptClient:  accept");
    }
  else
    {
      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	    (stderr,
	     "_tcp_AcceptClient: new connection, on fd %ld,"
	     " from IP host %s, port %u.\n",
	     ns, inet_ntoa(sin.sin_addr), ntohs(sin.sin_port)));

      new = (ilu_Transport) malloc (sizeof(struct _ilu_Transport_s));
      new->tr_class = _ilu_tcp_TransportClass();
      new->tr_fd = ns;
      new->tr_data = self->mo_data;
      if ((flags = fcntl (ns, F_GETFL, 0)) < 0
	  OR fcntl (ns, F_SETFL, flags | O_NONBLOCK) < 0)
	{
	  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		(stderr,
		 "_tcp_AcceptClient:  Failed to set socket (descriptor"
		 " %ld, from %s) non-blocking.  Error \"%s\".\n",
		 ns, addrDescr, ANSI_STRERROR(errno)));
        }

/* See above comment on similar code below */
#ifdef IPPROTO_TCP
#ifdef TCP_NODELAY
      {
	int one = 1;
	OS_SETSOCKOPT (ns, IPPROTO_TCP, TCP_NODELAY, &one, sizeof(one));
      }
#endif /* TCP_NODELAY */
#endif /* IPPROTO_TCP */     

    }
  return (new);
}

/*L1_sup < trmu*/
static ilu_Mooring _ilu_tcp_CreateMooring (ilu_private mooringInfo)
{
  TCPParms parms = (TCPParms) mooringInfo;
  struct sockaddr_in sin;
  struct linger linger;
  int namelen;
  ilu_integer skt, on = 1;
  ilu_Mooring self;
  
  _ilu_AutoSetDebugLevel();
  _ilu_AcquireMutex(ilu_trmu);
  if (!SigPIPEHandler)
    _tcp_HandleSigPIPE();
  _ilu_ReleaseMutex(ilu_trmu);

  /* setup the new server */
  
  if ((skt = OS_SOCKET(AF_INET, SOCK_STREAM, 0)) < 0) {
    DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	  (stderr, "_tcp_CreateMooring: socket failed:  %s.\n",
	   ANSI_STRERROR(errno)));
    return (NULL);
  }
    
  OS_SETSOCKOPT(skt, SOL_SOCKET, SO_REUSEADDR, (char *) &on, sizeof(on));
  linger.l_onoff = 0;
  OS_SETSOCKOPT(skt, SOL_SOCKET, SO_LINGER,
		(char *) &linger, sizeof(struct linger));

  sin.sin_family = AF_INET;
  sin.sin_port = htonl(TCP_PORT(parms));
  sin.sin_addr.s_addr = INADDR_ANY;	/* me, me! */

  if (OS_BIND(skt, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
    DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	  (stderr, "_tcp_CreateMooring: bind to port %lu failed:  %s.\n",
	   TCP_PORT(parms), ANSI_STRERROR(errno)));
    OS_CLOSE (skt);
    return (NULL);
  }

  /* If servicePort was 0 then discover kernel allocated port */
  
  if (sin.sin_port == 0) {
    namelen = sizeof(sin);
    if (OS_GETSOCKNAME(skt, (struct sockaddr *) &sin, &namelen) < 0)
      {
	DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	      (stderr, "_tcp_CreateMooring: getsockname failed:  %s.\n",
	       ANSI_STRERROR(errno)));
	OS_CLOSE (skt);
	return (NULL);
      }
  }

  if (OS_LISTEN(skt, 4) < 0) {
    DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	  (stderr,
	   "_tcp_CreateMooring:  listen on port %u failed:  %s.\n",
	   ntohs(sin.sin_port), ANSI_STRERROR(errno)));
    OS_CLOSE (skt);
    return (NULL);
  }

  self = (ilu_Mooring) malloc (sizeof(struct _ilu_Mooring_s));

  self->mo_fd = skt;
  self->mo_accept_connection = _tcp_AcceptClient;
  TCP_PORT(parms) = ntohs(sin.sin_port);
  if (TCP_HOSTNAME(parms) != NULL)
    free(TCP_HOSTNAME(parms));
  self->mo_transportClass = _ilu_tcp_TransportClass();
  _ilu_AcquireMutex(ilu_trmu);
  TCP_HOSTNAME(parms) = _ilu_Strdup(_ilu_tcp_CurrentHostInetName());
  _ilu_ReleaseMutex(ilu_trmu);
  self->mo_data = (ilu_private) parms;

  return (self);
}

static void _ilu_tcp_CloseMooring(ilu_Mooring m)
{
  int res;
  res = close(m->mo_fd);
  ASSERT(res==0, buf,
	 (buf, "tcp_CloseMooring: res=%d, errno=%s", res,
	  ANSI_STRERROR(errno)));
}

/*L1_sup < trmu*/
ilu_TransportClass _ilu_tcp_TransportClass (void)
{
  static ilu_TransportClass m = NULL;
  _ilu_AcquireMutex(ilu_trmu);
  if (m == NULL)
    {
      m = (ilu_TransportClass)
	  malloc(sizeof(struct _ilu_TransportClass_s));
      m->tc_type	= ilu_TransportType_TCP;
      m->tc_pFD		= 1;
      m->tc_cFD		= 1;
      m->tc_timesout	= FALSE;
      m->tc_interpret_info	= _tcp_InterpretInfo;
      m->tc_form_info	= _tcp_FormHandle;
      m->tc_connect	= _tcp_Connect;
      m->tc_wait_for_input	= _tcp_WaitForInput;
      m->tc_create_mooring	= _ilu_tcp_CreateMooring;
      m->tc_close_mooring	= _ilu_tcp_CloseMooring;
      m->tc_close		= _tcp_Close;
      m->tc_n_data_present	= _tcp_NDataPresent;
      m->tc_send_message	= _tcp_SendMessage;
      m->tc_read_message	= _tcp_ReadMessage;
      m->tc_flush_output	= _tcp_FlushOutput;
    }
  _ilu_ReleaseMutex(ilu_trmu);
  return (m);
}

