/*
Copyright (c) 1991, 1992, 1993, 1994 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: sunrpc.c,v 1.84 1994/05/11 18:57:04 janssen Exp $ */

#ifdef SUNOS
#define _BSD_SOURCE
#include <sys/types.h>
#include <unistd.h>		/* for gethostname(), getgeid(), etc. */
#else /* not SUNOS */
#define _POSIX_SOURCE
#endif


#ifdef MACOS
#pragma segment ilu
#endif

#include <string.h>
#include "ilu.h"
#include "iluntrnl.h"
#include "packet.h"
#include "sunrpc.h"

#include "call.h"
#include "protocol.h"
#include "connect.h"
#include "pipe.h"
#include "transprt.h"
#include "port.h"
#include "object.h"
#include "type.h"
#include "method.h"
#include "mooring.h"

/*************************************************************/
/*		Implement XDR PACKET			     */
/*************************************************************/

/*L1, Main unconstrained*/
/*L2 >= call's conn's iomu*/

#define ODD(x)		(((x)&0x1)!=0)
#define EVEN(x)		(((x)&0x1)==0)  
#define PADDED_SIZE(x)	((((unsigned) (x))+3) & (~0x3))
  
static ilu_boolean _xdr_put_u_long (PACKET p, ilu_cardinal l)
{
  if (packet_allocation(p) - packet_size(p) >= 4)
    {
      *p->next++ = (l >> 24) & 0xFF;
      *p->next++ = (l >> 16) & 0xFF;
      *p->next++ = (l >> 8) & 0xFF;
      *p->next++ = l & 0xFF;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _xdr_put_u_short (PACKET p,ilu_shortcardinal l)
{
  _ilu_Assert (0, "Attempt to marshall a short value in Sun RPC!");
}

static ilu_boolean _xdr_put_long(PACKET p, long l)
{
  return (_xdr_put_u_long (p, (ilu_cardinal) l));
}

static ilu_boolean _xdr_put_short (PACKET p, short l)
{
  return (_xdr_put_u_short (p, (ilu_shortcardinal) l));
}

static ilu_boolean _xdr_put_byte (PACKET p, char l)
{
  _ilu_Assert (0, "Attempt to marshall 1 byte octet in Sun RPC!");
}

static ilu_boolean _xdr_put_u_byte (PACKET p, ilu_byte l)
{
  _ilu_Assert (0, "Attempt to marshall 1 byte octet in Sun RPC!");
}

static ilu_boolean _xdr_put_float (PACKET p, float l)
{
  float l2 = l;

  return (_xdr_put_u_long (p, ((ilu_cardinal *) &l2)[0]));
}

static ilu_boolean _xdr_put_double (PACKET p, double l)
{
  double l2 = l;

  return (_xdr_put_u_long (p, ((ilu_cardinal *) &l2)[1]) && _xdr_put_u_long (p, ((ilu_cardinal *) &l2)[0]));
}

/*L1, L2, Main unconstrained*/
static ilu_cardinal SunRPCMaxStringSize = 65535;

static ilu_boolean _xdr_put_bytes (PACKET p, ilu_byte *b, ilu_cardinal l)
{
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

/*
  DEBUG(SUNRPC_DEBUG, (stderr, "_xdr_put_bytes:  allocation 0x%x, size 0x%x (diff %d), needed %u\n",
			packet_allocation(p), packet_size(p),
			packet_allocation(p) - packet_size(p),
			(4 + PADDED_SIZE(l))));
*/
			
  if (l > SunRPCMaxStringSize)
    {
      fprintf (stderr,
	       "%s %u, which exceeds SunRPCMaxStringSize value of %u.\n",
	       "Attempt to pass string of length", l,
	       SunRPCMaxStringSize);
      return (FALSE);
    }

  if (packet_allocation(p) - packet_size(p) >= (4 + PADDED_SIZE(l)))
    {
      register ilu_integer i;

      _xdr_put_u_long (p, l);
      for (p1 = p->next, c = l, p2 = b;  c > 0;  c--)
	*p1++ = *p2++;
      for (i = PADDED_SIZE(l) - l;  i > 0;  i--)
	*p1++ = 0;
      p->next = p1;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _xdr_put_opaque (PACKET p, ilu_byte *b, ilu_cardinal l)
{
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

  if (packet_allocation(p) - packet_size(p) >= PADDED_SIZE(l))
    {
      register ilu_integer i;

      for (p1 = p->next, c = l, p2 = b;  c > 0;  c--)
	*p1++ = *p2++;
      for (i = PADDED_SIZE(l) - l;  i > 0;  i--)
	*p1++ = 0;
      p->next = p1;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _xdr_get_u_short (PACKET p, ilu_shortcardinal *l)
{
  _ilu_Assert (0, "Attempt to input 2 byte short in Sun RPC!");
}

static ilu_boolean _xdr_get_u_long (PACKET p, ilu_cardinal *l)
{
  if (packet_allocation(p) - packet_size(p) < 4)
    return(FALSE);
  *l = (p->next[0] << 24) + (p->next[1] << 16) + (p->next[2] << 8) + p->next[3];
  p->next += 4;
  return (TRUE);
}

static ilu_boolean _xdr_get_long (PACKET p, ilu_integer *l)
{
  return (_xdr_get_u_long (p, (ilu_cardinal *) l));
}

static ilu_boolean _xdr_get_short (PACKET p, short *l)
{
  return (_xdr_get_u_short(p, (ilu_shortcardinal *) l));
}

static ilu_boolean _xdr_get_byte (PACKET p, char *l)
{
  _ilu_Assert (0, "Attempt to input 1 byte octet in Sun RPC!");
}

static ilu_boolean _xdr_get_u_byte (PACKET p, ilu_byte *l)
{
  _ilu_Assert (0, "Attempt to input 1 byte octet in Sun RPC!");
}

static ilu_boolean _xdr_get_float(PACKET p, float *l)
{
  return (_xdr_get_u_long (p, (ilu_cardinal *) l));
}

static ilu_boolean _xdr_get_double (PACKET p, double *l)
{
  return (_xdr_get_u_long (p, ((ilu_cardinal *) l) + 1)
	  && _xdr_get_u_long(p, ((ilu_cardinal *) l)));
}

static ilu_boolean _xdr_get_bytes(PACKET p, ilu_byte **b, ilu_cardinal *l, ilu_cardinal limit)
{
  ilu_cardinal l2;
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

  if (packet_allocation(p) - packet_size(p) < 4
      || (!_xdr_get_u_long (p, &l2))
      || packet_allocation(p) - packet_size(p) < PADDED_SIZE(l2))
    return (FALSE);

  *l = 0;
  if (l2 > SunRPCMaxStringSize OR (limit > 0 AND l2 > limit))
    {
      fprintf (stderr,
	       "%s %u, which exceeds SunRPCMaxStringSize value of %u or call limit of %u.\n",
	       "Attempt to read string of length", l2,
	       SunRPCMaxStringSize, limit);
      return (FALSE);
    }

  {
    register ilu_integer i;

    if (*b == NULL)
      *b = (ilu_byte *) malloc(l2 + 1);
    for (p1 = *b, p2 = p->next, c = l2;  c > 0;  c--)
      *p1++ = *p2++;
    for (i = PADDED_SIZE(l2) - l2;  i > 0;  i--)
      p2++;	/* eat padding characters */
    *p1 = '\0';
    p->next = p2;
    *l = l2;
    return (TRUE);
  }
}

static ilu_boolean _xdr_get_opaque (PACKET p, ilu_byte **b, ilu_cardinal len)
{
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;
  register ilu_integer i;

  if (packet_allocation(p) - packet_size(p) < PADDED_SIZE(len))
    return (FALSE);

  if (*b == NULL)
    *b = (ilu_byte *) malloc(len);
  for (p1 = *b, p2 = p->next, c = len;  c > 0;  c--)
    *p1++ = *p2++;
  for (i = PADDED_SIZE(len) - len;  i > 0;  i--)
    p2++;	/* eat padding characters */
  p->next = p2;
    return (TRUE);
}

static void _xdr_destroy (PACKET p)
{
  if (p->buf != NULL && p->count > 0)
    free (p->buf);
  free (p);
}

static PACKET _xdr_CreatePacket (ilu_bytes buf, ilu_cardinal size)
{
  PACKET p = (PACKET) malloc (sizeof(struct ilu_packet_s));
  if (buf != NULL)
    p->buf = buf;
  else if (size > 0)
    p->buf = (ilu_byte *) malloc(size);
  else
    p->buf = NULL;
  p->count = size;
  p->next = p->buf;

  p->put_long = _xdr_put_long;
  p->put_byte = _xdr_put_byte;
  p->put_short = _xdr_put_short;
  p->put_u_long = _xdr_put_u_long;
  p->put_u_byte = _xdr_put_u_byte;
  p->put_u_short = _xdr_put_u_short;
  p->put_float = _xdr_put_float;
  p->put_double = _xdr_put_double;
  p->put_bytes = _xdr_put_bytes;
  p->put_opaque = _xdr_put_opaque;

  p->get_long = _xdr_get_long;
  p->get_byte = _xdr_get_byte;
  p->get_short = _xdr_get_short;
  p->get_u_long = _xdr_get_u_long;
  p->get_u_byte = _xdr_get_u_byte;
  p->get_u_short = _xdr_get_u_short;
  p->get_float = _xdr_get_float;
  p->get_double = _xdr_get_double;
  p->get_bytes = _xdr_get_bytes;
  p->get_opaque = _xdr_get_opaque;

  p->destroy = _xdr_destroy;

  return (p);
}

/**********************************************************************
  End of packet implementation
***********************************************************************/

/*L1, L2, Main unconstrained*/

static ilu_refany _sunrpc_CreateDataBlock (void)
{
  struct SunRPC *new = (struct SunRPC *) malloc(sizeof(struct SunRPC));

  new->sun_in = NULL;
  new->sun_out = NULL;
  return ((ilu_refany) new);
}

static void _sunrpc_FreeDataBlock (struct SunRPC *d)
{
  if (d->sun_in != NULL)
    {
      packet_destroy (d->sun_in);
      d->sun_in = NULL;
    }
  if (d->sun_out != NULL)
    {
      packet_destroy (d->sun_out);
      d->sun_out= NULL;
    }
  free (d);
}

static void formRegistryName (ilu_string buf)
{
  ilu_string registryhome = (ilu_string ) getenv("ILUREGISTRY");

#ifdef MACOS
  if (registryhome != NULL)
    sprintf (buf, "%s:%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    sprintf (buf, "%s:%s", REGISTRY_LAST_RESORT,
	     PROGRAM_NUMBER_REGISTRY_NAME);
#else
  if (registryhome != NULL)
    sprintf (buf, "%s/%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    {
      sprintf (buf, "%s/%s", REGISTRY_LAST_RESORT,
	       PROGRAM_NUMBER_REGISTRY_NAME);
/*      fprintf (stderr, "ILU:  the environment variable ILUREGISTRY is not set, so we're using \"%s\" as the name of the Sun RPC program number registry.\n", buf); */
    }
#endif
}

/*L1 >= {prmu}*/

static HashTable ProgramNumberHashTable = NULL;
/* program number -> sunrpcinfo* */

static HashTable RegistryHashTable = NULL;
/* type unique_id -> sunrpcinfo* */

static void _sunrpc_EnsureRegistries(void)
{
  sunrpcinfo *s = NULL;

  if (RegistryHashTable == NULL)
    {
      char buf[1000], line[1000], name[1000], id[1000];
      FILE *f;
      long lineno;

      RegistryHashTable = _ilu_hash_MakeNewTable (137, NULL, NULL);
      ProgramNumberHashTable = _ilu_hash_MakeNewTable (137,
		_ilu_hash_HashPointer, _ilu_hash_PointerCompare);

      formRegistryName (buf);
      if ((f = fopen(buf, "r")) == NULL)
	{
	  fprintf (stderr, "%s program number registry file %s.\n",
		   "ILU:  _sunrpc_EnsureRegistries:  Can't access", buf);
	  return;
	}
      lineno = 0;
      while (fgets(line, sizeof(line), f) != NULL)
	{
	  lineno++;
     	  s = (sunrpcinfo *) malloc(sizeof(sunrpcinfo));
	  s->sui_class = NULL;
	  if (strncmp(line, ".LastAssignedProgramNumber ", 27) == 0)
	    ; /* skip this record */
	  else {
#ifdef MACOS
            /* AC: all the following code 'fixed' to do a hand-parse due
	       to broken sscanf() */
                /* NB: code now does a return if any line from registry
			is malformed */
		/* NB: We also don't use isspace() to advance over
		 * whitespace, because isspace() is also broken on the
		 * mac...*/
              if ((tmp=strchr(line,':'))==NULL) {
                  fprintf (stderr, "%s %s\"%s\":  %s",
			   "_sunrpc_EnsureRegistries:  Invalid record",
			   "(1) in program number registry file",
			   buf, line);
                  return;
                }
              strncpy(name,line,tmp-line);
              name[tmp-line]='\0';
              tmp++;
              start=tmp;
              ch=*tmp;
              while ((ch!=' ') && (ch!='\0')) {
                  tmp++;
                  ch=*tmp;
                }
              if (ch=='\0') {
                  fprintf (stderr, "%s %s \"%s\":  %s",
			   "_sunrpc_EnsureRegistries:  Invalid record",
			   "(2) in program number registry file",
			   buf, line);
                  return;
                }
              strncpy(id,start,tmp-start);
              id[tmp-start]='\0';
              tmp++;
              /* advance past white-space since strtoul is too stupid
		 to do it for us... */
              while (*tmp==' ')
                        tmp++;
              s->sui_pnumber=_ilu_strtoul(tmp,&tmp,16);
              if (tmp==NULL) {
                  fprintf (stderr, "%s %s \"%s\":  %s",
			   "_sunrpc_EnsureRegistries:  Invalid record",
			   "(3) in program number registry file",
			   buf, line);
                  return;
                }
              while (*tmp==' ')
                  tmp++;
              s->sui_version=_ilu_strtoul(tmp,NULL,10);
#else /* MACOS */
	      if (4 != sscanf (line, "%1000[^:]:%1000s %x %u", name, id,
			       &s->sui_pnumber, &s->sui_version))
                  fprintf (stderr, "%s line %u %s \"%s\":  %s",
			   "_sunrpc_EnsureRegistries:  Invalid record at",
			   lineno, "in program number registry file",
			   buf, line);
	      else
#endif	/* MACOS */
	       {
	         if (_ilu_hash_FindInTable (RegistryHashTable, id)
	   	  OR _ilu_hash_FindInTable (ProgramNumberHashTable,
					    (void *) s->sui_pnumber))
	   	{
		  fprintf (stderr,
			   "%s %s %u %s \"%s\":  <%s> (ignoring)\n",
			   "_sunrpc_EnsureRegistries:  Duplicate type id",
			   "or program number at line", lineno,
			   "in program number registry", buf, line);
		  continue;
		}
	      s->sui_type_id = _ilu_Strdup(id);
	      _ilu_hash_AddToTable (RegistryHashTable, s->sui_type_id, s);
	      _ilu_hash_AddToTable (ProgramNumberHashTable,
				    (void *) s->sui_pnumber, s);
	     }
          }
	}
      fclose (f);
    }
}

static sunrpcinfo *_sunrpc_SunRPCInformationForClass (ilu_Class class)
{
  sunrpcinfo *s = NULL;

  _sunrpc_EnsureRegistries();

  s = (sunrpcinfo *) _ilu_hash_FindInTable (RegistryHashTable,
					    class->cl_unique_id);
  if (s != NULL)
    s->sui_class = class;

  DEBUG(SUNRPC_DEBUG,
	(stderr, "%s \"%s:%s\", pnumber is 0x%x, version is %d.\n",
	 "_sunrpc_SunRPCInformationForClass:  Class", class->cl_name,
	 class->cl_unique_id, s == NULL ? 0 : s->sui_pnumber,
	 s == NULL ? 0 : s->sui_version));

  return (s);
}

static sunrpcinfo *_sunrpc_ClassFromProgramNumber (ilu_cardinal pnumber,
						   ilu_cardinal version)
{
  sunrpcinfo *s = NULL;

  _sunrpc_EnsureRegistries();

  s = (sunrpcinfo *) _ilu_hash_FindInTable (ProgramNumberHashTable,
					    (void *) pnumber);

  DEBUG(SUNRPC_DEBUG,
	(stderr,
	 "%s 0x%x, version is %u(%u), class is \"%s\" (\"%s:%s\").\n",
	 "_sunrpc_ClassFromProgramNumber:  pnumber is", pnumber, version,
	 (s == NULL) ? 0 : s->sui_version,
	 (s == NULL) ? "unknown" : s->sui_type_id,
	 (s == NULL || s->sui_class == NULL) ? "" : s->sui_class->cl_unique_id,
	 (s == NULL || s->sui_class == NULL) ? "" : s->sui_class->cl_name));

  return (s);
}

/*======================================================================*/
/*======================== Basic I/O code ==============================*/
/*======================================================================*/

#define INPUT_ERROR		1
#define OUTPUT_ERROR		2

/*L1, L2, Main unconstrained (this is only for calling from debugger)*/
static ilu_cardinal ilu_sunrpc_SetMaxStringSize (ilu_cardinal size)
{
  ilu_cardinal old_size = SunRPCMaxStringSize;
  if (size > 0)
    SunRPCMaxStringSize = size;
  return (old_size);  
}

/*L2 >= {call's connection's iomu}*/
/*L1, Main unconstrained*/

/* ==================== integer ==================== */

static ilu_boolean _sunrpc_OutputInteger (ilu_Call call, ilu_integer i)
{
  return (packet_put_long (SUNRPC(call_connection(call))->sun_out, i));
}

static ilu_boolean _sunrpc_InputInteger (ilu_Call call, ilu_integer *i)
{
  return (packet_get_long (SUNRPC(call_connection(call))->sun_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfInteger (ilu_Call call, ilu_integer i)
{
  return (4);
}

/* ==================== cardinal ==================== */

static ilu_boolean _sunrpc_OutputCardinal (ilu_Call call, ilu_cardinal i)
{
  return (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, i));
}

static ilu_boolean _sunrpc_InputCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (packet_get_u_long (SUNRPC(call_connection(call))->sun_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfCardinal (ilu_Call call, ilu_cardinal i)
{
  return (4);
}

/* ==================== short integer ==================== */

static ilu_boolean _sunrpc_OutputShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (packet_put_long (SUNRPC(call_connection(call))->sun_out, (ilu_integer) i));
}

static ilu_boolean _sunrpc_InputShortInteger (ilu_Call call, ilu_shortinteger *i)
{
  ilu_integer l;
  ilu_boolean stat;

  stat = packet_get_long (SUNRPC(call_connection(call))->sun_in, &l);
  if (stat)
    *i = l;
  return (stat);
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (4);
}

/* ==================== long integer ==================== */

static ilu_boolean _sunrpc_OutputLongInteger (ilu_Call call, ilu_integer *i)
{
  ilu_boolean stat = ilu_TRUE;
  stat = (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, i[1]));
  return(stat && (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, i[0])));
}

static ilu_boolean _sunrpc_InputLongInteger (ilu_Call call, ilu_longinteger (*i))
{
  ilu_boolean stat = ilu_TRUE;
  stat = (packet_get_long (SUNRPC(call_connection(call))->sun_in, &(*i)[1]));
  return (stat && (packet_get_long (SUNRPC(call_connection(call))->sun_in, &(*i)[0])));
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfLongInteger (ilu_Call call, ilu_integer *i)
{
  return (8);
}

/* ==================== short cardinal ==================== */

static ilu_boolean _sunrpc_OutputShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, (ilu_cardinal) i));
}

static ilu_boolean _sunrpc_InputShortCardinal (ilu_Call call, ilu_shortcardinal *i)
{
  ilu_cardinal l;
  ilu_boolean stat;

  stat = packet_get_u_long (SUNRPC(call_connection(call))->sun_in, &l);
  if (stat)
    *i = l & 0xFFFF;
  return (stat);  
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (4);
}

/* ==================== long cardinal ==================== */

static ilu_boolean _sunrpc_OutputLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  ilu_boolean stat;

  stat = (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, i[1]));
  return (stat && (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, i[0])));
}

static ilu_boolean _sunrpc_InputLongCardinal (ilu_Call call, ilu_longcardinal (*i))
{
  ilu_boolean stat;

  stat = (packet_get_u_long (SUNRPC(call_connection(call))->sun_in, &(*i)[1]));
  return (stat && (packet_get_u_long (SUNRPC(call_connection(call))->sun_in, &(*i)[0])));
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (8);
}

/* ==================== enumeration ==================== */

static ilu_boolean _sunrpc_OutputEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  ilu_cardinal i2 = i;
  return (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, i2));
}

static ilu_boolean _sunrpc_InputEnumeration (ilu_Call call, ilu_shortcardinal *i)
{
  ilu_cardinal i2;
  ilu_cardinal stat;

  stat = packet_get_u_long (SUNRPC(call_connection(call))->sun_in, &i2);
  *i = i2;
  return(stat);
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  return (4);
}

/* ==================== real ==================== */

static ilu_boolean _sunrpc_OutputReal (ilu_Call call, double d)
{
  return (packet_put_double (SUNRPC(call_connection(call))->sun_out, d));
}

static ilu_boolean _sunrpc_InputReal (ilu_Call call, double *d)
{
  return (packet_get_double (SUNRPC(call_connection(call))->sun_in, d));
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfReal (ilu_Call call, double d)
{
  return (8);
}

/* ==================== short real ==================== */

static ilu_boolean _sunrpc_OutputShortReal (ilu_Call call, float f)
{
  float f2;

  f2 = f;
  return (packet_put_float (SUNRPC(call_connection(call))->sun_out, f2));
}

static ilu_boolean _sunrpc_InputShortReal (ilu_Call call, float *f)
{
  return (packet_get_float (SUNRPC(call_connection(call))->sun_in, f));
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfShortReal (ilu_Call call, float d)
{
  return (4);
}

/* ==================== string ==================== */

static ilu_boolean _sunrpc_OutputString (ilu_Call call, ilu_string s, ilu_cardinal len, ilu_cardinal limit)
{
  ilu_cardinal len2 = len;

  return (packet_put_bytes (SUNRPC(call_connection(call))->sun_out, (ilu_bytes) s, len2));
}

static ilu_boolean _sunrpc_InputString (ilu_Call call, ilu_string *s, ilu_cardinal *len, ilu_cardinal limit)
{
  *s = NULL;
  if (!packet_get_bytes (SUNRPC(call_connection(call))->sun_in,
	(ilu_byte **) s, len, limit))
    return (FALSE);
  return (TRUE);
}

  /*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfString (ilu_Call call, ilu_string i, ilu_cardinal l, ilu_cardinal limit)
{
  if (l > SunRPCMaxStringSize)
    {
      fprintf (stderr, "_sunrpc_SizeOfString:  Attempt to pass string of length %u, which exceeds either SunRPCMaxStringSize value of %u, or limit on this sequence type of %u.\n",
	       l, SunRPCMaxStringSize, limit);
      return (0);
    }

  return (4 + PADDED_SIZE(l));
}

/* ==================== bytes ==================== */

static ilu_boolean _sunrpc_OutputBytes (ilu_Call call, ilu_bytes s, ilu_cardinal len, ilu_cardinal limit)
{
  return (packet_put_bytes (SUNRPC(call_connection(call))->sun_out, s, len));
}

static ilu_boolean _sunrpc_InputBytes (ilu_Call call, ilu_bytes *s, ilu_cardinal *len, ilu_cardinal limit)
{
  *s = NULL;
  return (packet_get_bytes (SUNRPC(call_connection(call))->sun_in, s, len, limit));
}

  /*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfBytes (ilu_Call call, ilu_bytes i, ilu_cardinal l, ilu_cardinal limit)
{
  if (((limit > 0) && (l > limit)) || l > SunRPCMaxStringSize)
    {
      fprintf (stderr, "_sunrpc_SizeOfBytes:  Attempt to pass bytestring of length %u, which exceeds either SunRPCMaxStringSize of %u, or limit on this sequence type of %u.\n",
	       l, SunRPCMaxStringSize, limit);
      return (0);
    }
  return (4 + PADDED_SIZE(l));
}

/* ==================== byte ==================== */

static ilu_boolean _sunrpc_OutputByte (ilu_Call call, ilu_byte b)
{
  return (packet_put_u_long (SUNRPC(call_connection(call))->sun_out, (ilu_cardinal) b));
}

static ilu_boolean _sunrpc_InputByte (ilu_Call call, ilu_byte *b)
{
  ilu_cardinal l;
  ilu_boolean stat;

  stat = packet_get_u_long (SUNRPC(call_connection(call))->sun_in, &l);
  if (stat)
    *b = l & 0xFF;
  return (stat);
}

/*ARGSUSED*/
static ilu_cardinal _sunrpc_SizeOfByte (ilu_Call call, ilu_byte i)
{
  return (4);
}

/* ==================== opaque ==================== */

static ilu_boolean _sunrpc_OutputOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (packet_put_opaque (SUNRPC(call_connection(call))->sun_out, o, len));
}

static ilu_boolean _sunrpc_InputOpaque (ilu_Call call, ilu_bytes *o, ilu_cardinal len)
{
  return(packet_get_opaque (SUNRPC(call_connection(call))->sun_in, o, len));
}

static ilu_cardinal _sunrpc_SizeOfOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (PADDED_SIZE(len));
}

static ilu_boolean _sunrpc_OutputSequence (ilu_Call c, ilu_cardinal sequenceLength, ilu_cardinal limit)
{
  return (_sunrpc_OutputCardinal (c, sequenceLength));
}

static ilu_boolean _sunrpc_OutputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  return (TRUE);
}

static ilu_boolean _sunrpc_InputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  return (TRUE);
}

static ilu_boolean _sunrpc_InputSequence (ilu_Call c, ilu_cardinal *sequenceLength, ilu_cardinal limit)
{
  ilu_boolean stat;
  ilu_cardinal len;

  stat = _sunrpc_InputCardinal (c, &len);
  *sequenceLength = len;
  return(stat);
}

static ilu_boolean _sunrpc_EndSequence (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _sunrpc_SizeOfSequence (ilu_Call c, ilu_cardinal length, ilu_cardinal limit)
{
  return (_sunrpc_SizeOfCardinal(c, length));
}

static ilu_boolean _sunrpc_OutputUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_sunrpc_OutputCardinal (c, typeIndex));
}

static ilu_boolean _sunrpc_InputUnion (ilu_Call c, ilu_shortcardinal *typeIndex)
{
  ilu_boolean stat;
  ilu_cardinal i;

  stat = _sunrpc_InputCardinal (c, &i);
  *typeIndex = i;
  return(stat);
}

static ilu_boolean _sunrpc_EndUnion (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _sunrpc_SizeOfUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_sunrpc_SizeOfCardinal(c, typeIndex));
}

static ilu_boolean _sunrpc_OutputArray (ilu_Call c, ilu_cardinal len)
{
  return(TRUE);
}

static ilu_boolean _sunrpc_InputArray (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _sunrpc_EndArray (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _sunrpc_SizeOfArray (ilu_Call c, ilu_cardinal len)
{
  return(0);
}

static ilu_boolean _sunrpc_OutputRecord (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _sunrpc_InputRecord (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _sunrpc_EndRecord (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _sunrpc_SizeOfRecord (ilu_Call c)
{
  return(0);
}

/*======================================================================
**======================================================================
**====================  Non-I/O code ===================================
**======================================================================
**====================================================================*/

/*L2, Main unconstrained*/
/*L1_sup < prmu*/
static ilu_string _sunrpc_FormProtocolHandle (ilu_Object obj)
{
  char buf[100];
  sunrpcinfo *s = NULL;
  ilu_Class class = object_class(obj);

  _ilu_AcquireMutex(ilu_prmu);
  s = _sunrpc_SunRPCInformationForClass (class);
  if (s == NULL)
    {
      _ilu_ReleaseMutex(ilu_prmu);
      fprintf (stderr, "%s program#/version for class %s.\n",
	       "_sunrpc_FormProtocolHandle:  Can't figure", class->cl_name);
      return (NULL);
    }
  else
    {
      sprintf (buf, "sunrpc_2_0x%x_%d", s->sui_pnumber, s->sui_version);
      _ilu_ReleaseMutex(ilu_prmu);
      return (_ilu_Strdup(buf));
    }
}
     
/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_bytes _sunrpc_ReadPacket (ilu_Transport bs,
			      ilu_cardinal *packetLength,
			      ilu_PacketType *type, ilu_cardinal *sn)
{
  ilu_bytes packet;
  ilu_cardinal serialNumber, packetType, size;

  if (!transport_read_message(bs, &packet, &size)) {
    DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	  (stderr, "%s:  error on transport_read_message\n",
	   "_sunrpc_ReadPacket"));
    return (NULL);
  };

  serialNumber =  (packet[0] << 24) | (packet[1] << 16)
		| (packet[2] <<  8) |  packet[3];
  packetType =    (packet[4] << 24) | (packet[5] << 16)
		| (packet[6] <<  8) |  packet[7];

  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr, "%s %d, SN %d, type %d (%s).\n",
	 "_sunrpc_ReadPacket:  reading packet of length", size,
	 serialNumber, packetType,
	 ((packetType == ilu_PacketType_Request) ? "request"
	  : (packetType == ilu_PacketType_Reply) ? "reply"
	  : "unknown type")));

  *packetLength = size;
  *type = packetType;
  *sn = serialNumber;
  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr, "_sunrpc_ReadPacket:  returning packet SN %d\n",
	 serialNumber));
  return (packet);  
}

static ilu_boolean _sunrpc_InterpretPipeRequest (ilu_Call call, ilu_bytes packet,
					  ilu_cardinal len)
{
  struct SunRPC *c = SUNRPC(call_connection(call));
  ilu_cardinal rpcVersion, credSize, verifierSize, programNumber;
  ilu_cardinal programVersion, methodID;
  ilu_cardinal verifierType, credType, serialNumber, packetType;
  ilu_bytes credBuf, verifierBuf;

  c->sun_in = _xdr_CreatePacket (packet, len);

  _sunrpc_InputCardinal (call, &serialNumber);
  _sunrpc_InputCardinal (call, &packetType);
  _sunrpc_InputCardinal (call, &rpcVersion);
  _sunrpc_InputCardinal (call, &programNumber);
  _sunrpc_InputCardinal (call, &programVersion);
  _sunrpc_InputCardinal (call, &methodID);
  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr,
	 "%s:  (call SN %d) prognum 0x%x version %u, method ID = %u\n",
	 "_sunrpc_InterpretPipeRequest", call->ca_SN,
	 programNumber, programVersion, methodID));

  _sunrpc_InputCardinal (call, &credType);
  _sunrpc_InputBytes (call, &credBuf, &credSize, 0);
  _sunrpc_InputCardinal (call, &verifierType);
  _sunrpc_InputBytes (call, &verifierBuf, &verifierSize, 0);
  DEBUG(AUTHENTICATION_DEBUG,
	(stderr, "_sunrpc_InterpretPipeRequest:  (call SN %d) credBuf is %x, credSize is %d, verifierBuf is %x, verifierSize is %d\n",
	 call->ca_SN, (unsigned long) credBuf, credSize,
	 (unsigned long) verifierBuf, verifierSize));
  
  /* not used at the moment */
  FREETOKEN(credBuf);
  FREETOKEN(verifierBuf);
  return (TRUE);
}

static ilu_boolean _sunrpc_InterpretRequest (ilu_Call call, ilu_bytes packet,
				      ilu_cardinal len)
{
  struct SunRPC *c = SUNRPC(call_connection(call));
  sunrpcinfo *s;
  ilu_cardinal rpcVersion, serialNumber, packetType;
  ilu_cardinal credSize, credType, verifierSize, verifierType;
  ilu_cardinal programNumber, programVersion, methodID;
  ilu_bytes credBuf, verifierBuf;

  c->sun_in = _xdr_CreatePacket (packet, len);

  _sunrpc_InputCardinal (call, &serialNumber);
  _sunrpc_InputCardinal (call, &packetType);
  _sunrpc_InputCardinal (call, &rpcVersion);
  _sunrpc_InputCardinal (call, &programNumber);
  _sunrpc_InputCardinal (call, &programVersion);
  _sunrpc_InputCardinal (call, &methodID);
  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr,
	 "%s  (call SN %d) prognum 0x%x (version %u), method ID = %u\n",
	 "_sunrpc_InterpretRequest:", call->ca_SN, programNumber,
	 programVersion, methodID));
  _ilu_Assert(serialNumber == call->ca_SN,
	      "sunrpc_InterpretRequest: serial number snafu");
  _sunrpc_InputCardinal (call, &credType);
  _sunrpc_InputBytes (call, &credBuf, &credSize, 0);
  _sunrpc_InputCardinal (call, &verifierType);
  _sunrpc_InputBytes (call, &verifierBuf, &verifierSize, 0);
  DEBUG(AUTHENTICATION_DEBUG,
	(stderr,
	 "%s %d%s %x, credSize is %d, verfBuf is %x, verfSize is %d\n",
	 "_sunrpc_InterpretRequest:  (call SN", call->ca_SN,
	 ") credBuf is", (unsigned long) credBuf, credSize,
	 (unsigned long) verifierBuf, verifierSize));
  
  /* not used at the moment */
  FREETOKEN(credBuf);
  FREETOKEN(verifierBuf);

  _ilu_AcquireMutex(ilu_prmu);
  s = _sunrpc_ClassFromProgramNumber(programNumber, programVersion);
  call_intro_type(call)
    = (s == NULL) ? NULL
		  : (s->sui_class == NULL) ? ilu_FindClassFromID(s->sui_type_id)
				       : s->sui_class;
  _ilu_ReleaseMutex(ilu_prmu);
  if (s == NULL OR call_intro_type(call) == NULL)
    {
      DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	    (stderr, "%s  (call %d) %s 0x%x, version %u\n",
	     "_sunrpc_InterpretRequest:", call->ca_SN,
	     "Can't find ilu_Class with pn",
	     programNumber, programVersion));
      return (FALSE);
    }

  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr,
	 "_sunrpc_InterpretRequest:  (call SN %d) intro_type is %s:%s\n",
	 call->ca_SN, call->ca_intro_type->cl_name,
	 call->ca_intro_type->cl_unique_id));

  call->ca_method = ilu_FindMethodByID (call_intro_type(call), methodID);
  
  if (call->ca_method == NULL)
    {
      DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	    (stderr,
	     "%s  (call %d) %s \"%s\" (pn %x) with methodID = %u.\n",
	     "_sunrpc_InterpretRequest:", call->ca_SN,
	     "Can't find method on class", call_intro_type(call)->cl_name,
	     programNumber, methodID));
      return (FALSE);
    }
  else
    {
      DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	    (stderr, "%s %d is 0x%x (%s).\n",
	     "_sunrpc_InterpretRequest:  record for method", 
	     methodID, (unsigned long) call->ca_method, call->ca_method->me_name));
    }
  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr, "_sunrpc_InterpretRequest:  returning TRUE\n"));
  return (TRUE);
}

/*L1, Main unconstrained*/
/*L2 >= {conn's iomu}*/

static ilu_ProtocolException _sunrpc_InterpretReply (ilu_Call call,
					      ilu_bytes packet,
					      ilu_cardinal len,
					      ilu_cardinal *estatus)
{
  struct SunRPC *c = SUNRPC(call->ca_connection);
  ilu_cardinal verifierSize, replyStatus, serialNumber, packetType;
  ilu_bytes verifierBuf;
  ilu_cardinal authenticationType;
  static ilu_ProtocolException replyStatusExceptions[] = {
    ilu_ProtocolException_Success,
    ilu_ProtocolException_NoSuchClassAtServer,
    ilu_ProtocolException_ClassVersionMismatch,
    ilu_ProtocolException_NoSuchMethodOnClass,
    ilu_ProtocolException_GarbageArguments
    };
  static ilu_string acceptErrors[] = {
    "Success",
    "program unavailable",
    "program version mismatch",
    "procedure unavailable",
    "garbage arguments",
  };

  c->sun_in = _xdr_CreatePacket (packet, len);

  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr, "_sunrpc_InterpretReply:  SN %d, %d bytes\n",
	 call->ca_SN, len));

  _sunrpc_InputCardinal (call, &serialNumber);
  _sunrpc_InputCardinal (call, &packetType);
  _sunrpc_InputCardinal (call, &replyStatus);

  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	(stderr, "_sunrpc_InterpretReply:  replyStatus is %d\n",
	 replyStatus));

  if (replyStatus == 0)		/* MSG_ACCEPTED */
    {
      _sunrpc_InputCardinal (call, &authenticationType);
      _sunrpc_InputBytes (call, &verifierBuf, &verifierSize, 0);
      FREETOKEN(verifierBuf);
      _sunrpc_InputCardinal (call, &replyStatus);

      DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	    (stderr,
	     "_sunrpc_InterpretReply:  *real* replyStatus is %d\n",
	     replyStatus));

      if (replyStatus > 0)		/* *really* accepted */
	{
	  DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
		(stderr,
		 "Sun RPC call %d signals protocol error %d (%s).\n",
		 call->ca_SN, replyStatus,
		 (replyStatus < 5) ? acceptErrors[replyStatus]
				   : "unknown error"));
	}
      else if (call->ca_method->me_exceptionCount > 0)
	_sunrpc_InputCardinal (call, estatus);
      else
	*estatus = 0;

      if (replyStatus < 5)
	return (replyStatusExceptions[replyStatus]);
      else
	return(ilu_ProtocolException_Unknown);
    }
  else if (replyStatus == 1)	/* MSG_REJECTED */
    {
      DEBUG((INCOMING_DEBUG | SUNRPC_DEBUG),
	    (stderr, "Sun RPC call %d signals \"Message Rejected\".\n",
	     call->ca_SN));
      return(ilu_ProtocolException_RequestRejected);
    }
  return(ilu_ProtocolException_Unknown);
}

/*L1_sup < prmu*/

static ilu_boolean GetSunRPCProgramNumberAndVersion (ilu_Class pclass,
			ilu_cardinal *pnumber, ilu_cardinal *version)
{
  sunrpcinfo *s = NULL;
  _ilu_AcquireMutex(ilu_prmu);
  s = _sunrpc_SunRPCInformationForClass (pclass);
	/* probably consult external DB */
  if (s != NULL) {
      *pnumber = s->sui_pnumber;
      *version = s->sui_version;
    }
  else
    {
      fprintf (stderr, "ILU: (GetSunRPCProgramNumberAndVersion):  Can't get program number information for class \"%s\".\n", pclass->cl_name);
    }
  _ilu_ReleaseMutex(ilu_prmu);
  return (s != NULL);
}

/*L1 >= {prmu}*/
#ifdef _BSD_SOURCE
static int nameset = 0;
static char hostname[255];
static int hostnamelen, hostnamelenr, credlen;
static int gid, uid, ngids;
static gid_t gids[100];
#endif /* _BSD_SOURCE */
static int credlenr;

/*L1_sup < prmu*/

static ilu_boolean _sunrpc_BeginRequest (ilu_Call call, ilu_Class pclass,
					 ilu_Method method,
					 ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  ilu_cardinal pnumber=0, version=0;
  struct SunRPC *c = SUNRPC(call->ca_connection);

  call->ca_method = method;

#if 0
  if (call_object(call)==NULL) { /* no call object, must be pipe req */
#else
  if (0) {
#endif
      DEBUG(CALL_DEBUG,
	    (stderr,
	     "%s 0x%x (sn %d), argSize %d, class %s (%s), meth %s (%d)\n",
	     "_sunrpc_BeginRequest:  pipe call", (unsigned long) call,
	     call_serial_number(call), argSize, class_name(pclass),
	     class_unique_id(pclass), method_name(method),
	     method_id(method)));
    }
  else {
      DEBUG(CALL_DEBUG,
	    (stderr,
	     "%s 0x%x (sn %d), argSize %d, class %s (%s), meth %s (%d)\n",
	     "_sunrpc_BeginRequest:  call", (unsigned long) call,
	     call_serial_number(call), argSize, class_name(pclass),
	     class_unique_id(pclass), method_name(method),
	     method_id(method)));

      if (!GetSunRPCProgramNumberAndVersion (pclass, &pnumber, &version))
        {
	  fprintf (stderr, "%s %s of class \"%s\" on call 0x%x.\n",
		   "_sunrpc_BeginRequest:  Can't determine",
		   "program-number/version",
		   pclass->cl_name,  (unsigned long) call);
          return (FALSE);
        }
    }
  DEBUG(CALL_DEBUG,
	(stderr,
	 "%s 0x%x (sn %d), argSize %d, pnum/vers 0x%x/%d, method id %d\n",
	 "_sunrpc_BeginRequest:  call", (unsigned long) call, call_serial_number(call),
	 argSize, pnumber, version, method_id(method)));

#ifdef _BSD_SOURCE
  if (!nameset) {
      _ilu_AcquireMutex(ilu_prmu);
      if (!nameset) {
          nameset = 1;
          _ilu_Assert(gethostname (hostname, sizeof(hostname)) == 0,
                      "gethostname failed");
          hostnamelen = strlen(hostname);
          hostnamelenr = 4 * ((hostnamelen+3)/4);
          gid = getegid();
          uid = geteuid();
          ngids = getgroups(sizeof(gids)/sizeof(gid), gids);
          _ilu_Assert(ngids >= 0, "getgroups() failed");
          credlen =   4/*stamp*/ + 4+hostnamelenr/*machinename*/
                    + 4/*uid*/ + 4/*gid*/ + 4+4*ngids/*gids*/;
          credlenr = 4 * ((credlen+3)/4);
        }
      _ilu_ReleaseMutex(ilu_prmu);
    }
#else
  credlenr = 4 * 2;
#endif
  
  packetSize = argSize
    + (4*10)		/* for fields of header */
    + credlenr		/* for UNIX credentials */;

  c->sun_out = _xdr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _sunrpc_OutputCardinal (call, call->ca_SN);
  _sunrpc_OutputCardinal (call, 0);	/* message type == CALL */
  _sunrpc_OutputCardinal (call, 2);	/* Sun RPC version 2 */
  _sunrpc_OutputCardinal (call, pnumber);
  _sunrpc_OutputCardinal (call, version);
  _sunrpc_OutputCardinal (call, method->me_id);

#ifdef _BSD_SOURCE
  _sunrpc_OutputCardinal (call, 1);	/* UNIX credentials follow */
  _sunrpc_OutputCardinal (call, credlen); /* credentials length */
  _sunrpc_OutputCardinal (call, 1);	/* stamp */
  _sunrpc_OutputString (call, hostname, hostnamelen, 255);
  _sunrpc_OutputCardinal (call, uid);
  _sunrpc_OutputCardinal (call, gid);
  _sunrpc_OutputCardinal (call, ngids);
  {
    int i;
    for (i=0; i<ngids; i++)
      _sunrpc_OutputCardinal (call, gids[i]);
  }
#else
  _sunrpc_OutputCardinal (call, 0);	/* credentials:  AUTH_NULL */
  _sunrpc_OutputCardinal (call, 0);	/* 		 0 bytes */
#endif

  _sunrpc_OutputCardinal (call, 0);	/* verifier:  AUTH_NULL */
  _sunrpc_OutputCardinal (call, 0);	/*            0 bytes */
  
  DEBUG(CALL_DEBUG,
	(stderr, "_sunrpc_BeginRequest:  request %d begun (size %d).\n",
	 call->ca_SN, packetSize));
  return (TRUE);
}

/*L1, Main unconstrained*/
/*L2 >= {conn's iomu}*/

static void _sunrpc_DropInput (ilu_Call call)
{
  struct SunRPC *s = SUNRPC(call_connection(call));
  if (s->sun_in != NULL)
      packet_destroy(s->sun_in);
  s->sun_in = NULL;
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static void _sunrpc_FlushBuffer (ilu_Call call)
{
  transport_flush_output(call_transport(call));
}

static void _sunrpc_DropRequest (ilu_Call call)
{
  struct SunRPC *s = SUNRPC(call_connection(call));
  if (s->sun_out != NULL)
      packet_destroy(s->sun_out);
  s->sun_out = NULL;
}

static ilu_boolean _sunrpc_SendPacket (ilu_Call call, ilu_boolean isReply)
{
  ilu_Transport bs = call_transport(call);
  struct SunRPC *s = SUNRPC(call_connection(call));
  _ilu_Assert(packet_buffer(s->sun_out) != NULL, "_sunrpc_SendPacket");
  transport_send_message (bs, packet_buffer(s->sun_out), packet_size(s->sun_out));
  if (!bs->tr_class->tc_timesout)
    packet_buffer(s->sun_out) = NULL;
  if (!(method_asynchronous(call_method(call))))
    transport_flush_output(bs);
  if (isReply) {
      if (bs->tr_class->tc_timesout) {
          _ilu_CacheCall(call, packet_buffer(s->sun_out),
                         packet_size(s->sun_out));
          packet_buffer(s->sun_out) = NULL;
        }
      _sunrpc_DropRequest(call);
    }
  return (TRUE);
}

static ilu_boolean _sunrpc_FinishRequest (ilu_Call call)
{
  return (_sunrpc_SendPacket(call, FALSE));
}

static ilu_boolean _sunrpc_FinishReply (ilu_Call call)
{
  return (_sunrpc_SendPacket (call, TRUE));
}

static ilu_boolean _sunrpc_FinishException (ilu_Call call)
{
  return (_sunrpc_SendPacket(call, TRUE));
}

/*L1, Main unconstrained*/
/*L2 >= {call's conn's iomu}*/

static ilu_boolean _sunrpc_BeginReply (ilu_Call call,
				       ilu_boolean exceptions,
				       ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  struct SunRPC *c = SUNRPC(call->ca_connection);

  DEBUG(PACKET_DEBUG,
	(stderr, "%s %d, argSize %d, exceptions %s, fd %d.\n",
	 "_sunrpc_BeginReply:  SN", call->ca_SN, argSize,
	 exceptions ? "TRUE" : "FALSE",
	 transport_file_descriptor(call_transport(call))));

  packetSize = argSize
    + (4*6)			/* for the basic header fields */
    + (exceptions ? 4 : 0);	/* possible extra word for excn code */

  c->sun_out = _xdr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _sunrpc_OutputCardinal (call, call->ca_SN);
  _sunrpc_OutputCardinal (call, 1);		/* message type ==REPLY */
  _sunrpc_OutputCardinal (call, 0);		/* message accepted */
  _sunrpc_OutputCardinal (call, 0);		/* verifier:  AUTH_NULL */
  _sunrpc_OutputCardinal (call, 0);		/*            0 bytes */
  _sunrpc_OutputCardinal (call, 0);		/* successful execution */
  if (exceptions)
    _sunrpc_OutputCardinal (call, 0);		/* ret code is Success */

  DEBUG(PACKET_DEBUG,
	(stderr, "_sunrpc_BeginReply:  started reply %d (size %d).\n",
	 call->ca_SN, packetSize));
  return (TRUE);
}

static ilu_boolean _sunrpc_BeginException (ilu_Call call,
					   ilu_cardinal evalue,
					   ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  struct SunRPC *c = SUNRPC(call->ca_connection);

  /* if "evalue" == 0, then argSize contains a protocol exception
     detail code. */

  if (evalue == 0)	/* signal protocol error */
    packetSize = (4*6);
  else
    packetSize = (4*7) + argSize;

  c->sun_out = _xdr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _sunrpc_OutputCardinal (call, call->ca_SN);
  _sunrpc_OutputCardinal (call, 1);		/* message type ==REPLY */
  _sunrpc_OutputCardinal (call, 0);		/* message accepted */
  _sunrpc_OutputCardinal (call, 0);		/* verifier:  AUTH_NULL */
  _sunrpc_OutputCardinal (call, 0);		/*            0 bytes */
  _sunrpc_OutputCardinal (call, (evalue == 0) ? argSize : 0);
						/* successful execution */
  if (evalue > 0)
    _sunrpc_OutputCardinal (call, evalue);	/* exception value */

  DEBUG(PACKET_DEBUG,
	(stderr, "%s %d:  SN %d, size %d, evalue %d.\n",
	 "_sunrpc_BeginException:  exception started to peer",
	 transport_file_descriptor(call_transport(call)),
	 call_serial_number(call), packetSize, evalue));
  return (TRUE);
}

/*L2, Main unconstrained*/
/*L1 >= {prmu}*/

static ilu_Protocol _sunrpc_NewSunRPC (void)
{
  ilu_Protocol new = (ilu_Protocol)
		     malloc(sizeof(struct _ilu_Protocol_s));

  new->pr_type = ilu_ProtocolType_SunRPC;
  new->pr_concurrent_requests = ilu_FALSE;

  new->pr_begin_request = _sunrpc_BeginRequest;
  new->pr_finish_request = _sunrpc_FinishRequest;
  new->pr_resend_request = _sunrpc_FinishRequest;
  new->pr_drop_request = _sunrpc_DropRequest;
  new->pr_begin_reply = _sunrpc_BeginReply;
  new->pr_finish_reply = _sunrpc_FinishReply;
  new->pr_begin_exception = _sunrpc_BeginException;
  new->pr_finish_exception = _sunrpc_FinishException;

  new->pr_output_integer = _sunrpc_OutputInteger;
  new->pr_input_integer = _sunrpc_InputInteger;
  new->pr_size_of_integer = _sunrpc_SizeOfInteger;

  new->pr_output_shortinteger = _sunrpc_OutputShortInteger;
  new->pr_input_shortinteger = _sunrpc_InputShortInteger;
  new->pr_size_of_shortinteger = _sunrpc_SizeOfShortInteger;

  new->pr_output_longinteger = _sunrpc_OutputLongInteger;
  new->pr_input_longinteger = _sunrpc_InputLongInteger;
  new->pr_size_of_longinteger = _sunrpc_SizeOfLongInteger;

  new->pr_output_cardinal = _sunrpc_OutputCardinal;
  new->pr_input_cardinal = _sunrpc_InputCardinal;
  new->pr_size_of_cardinal = _sunrpc_SizeOfCardinal;

  new->pr_output_shortcardinal = _sunrpc_OutputShortCardinal;
  new->pr_input_shortcardinal = _sunrpc_InputShortCardinal;
  new->pr_size_of_shortcardinal = _sunrpc_SizeOfShortCardinal;

  new->pr_output_longcardinal = _sunrpc_OutputLongCardinal;
  new->pr_input_longcardinal = _sunrpc_InputLongCardinal;
  new->pr_size_of_longcardinal = _sunrpc_SizeOfLongCardinal;

  new->pr_output_real = _sunrpc_OutputReal;
  new->pr_input_real = _sunrpc_InputReal;
  new->pr_size_of_real = _sunrpc_SizeOfReal;

  new->pr_output_shortreal = _sunrpc_OutputShortReal;
  new->pr_input_shortreal = _sunrpc_InputShortReal;
  new->pr_size_of_shortreal = _sunrpc_SizeOfShortReal;

  new->pr_output_longreal = _sunrpc_OutputReal;
  new->pr_input_longreal = _sunrpc_InputReal;
  new->pr_size_of_longreal = _sunrpc_SizeOfReal;

  new->pr_output_optional =
	(ilu_boolean (*)(ilu_Call,ilu_boolean)) _sunrpc_OutputCardinal;
  new->pr_input_optional =
	(ilu_boolean (*)(ilu_Call,ilu_boolean *)) _sunrpc_InputCardinal;
  new->pr_size_of_optional =
	(ilu_cardinal (*)(ilu_Call,ilu_boolean)) _sunrpc_SizeOfCardinal;

  new->pr_output_enum_code = _sunrpc_OutputEnumeration;
  new->pr_input_enum_code = _sunrpc_InputEnumeration;
  new->pr_size_of_enum_code = _sunrpc_SizeOfEnumeration;

  new->pr_output_byte = _sunrpc_OutputByte;
  new->pr_input_byte = _sunrpc_InputByte;
  new->pr_size_of_byte = _sunrpc_SizeOfByte;

  new->pr_output_character = _sunrpc_OutputShortCardinal;
  new->pr_input_character = _sunrpc_InputShortCardinal;
  new->pr_size_of_character = _sunrpc_SizeOfShortCardinal;

  new->pr_output_boolean =
	(ilu_boolean (*)(ilu_Call,ilu_boolean)) _sunrpc_OutputCardinal;
  new->pr_input_boolean =
	(ilu_boolean (*)(ilu_Call,ilu_boolean *)) _sunrpc_InputCardinal;
  new->pr_size_of_boolean =
	(ilu_cardinal (*)(ilu_Call,ilu_boolean)) _sunrpc_SizeOfCardinal;

  new->pr_output_shortchar = _sunrpc_OutputByte;
  new->pr_input_shortchar = _sunrpc_InputByte;
  new->pr_size_of_shortchar = _sunrpc_SizeOfByte;

  new->pr_output_string = _sunrpc_OutputString;
  new->pr_input_string = _sunrpc_InputString;
  new->pr_size_of_string = _sunrpc_SizeOfString;

  new->pr_output_bytes = _sunrpc_OutputBytes;
  new->pr_input_bytes = _sunrpc_InputBytes;
  new->pr_size_of_bytes = _sunrpc_SizeOfBytes;

  new->pr_output_opaque = _sunrpc_OutputOpaque;
  new->pr_input_opaque = _sunrpc_InputOpaque;
  new->pr_size_of_opaque = _sunrpc_SizeOfOpaque;

  new->pr_output_stringvec =
	(ilu_boolean (*)(ilu_Call,ilu_string,ilu_cardinal))
	_sunrpc_OutputOpaque;
  new->pr_input_stringvec =
	(ilu_boolean (*)(ilu_Call,ilu_string *,ilu_cardinal))
	_sunrpc_InputOpaque;
  new->pr_size_of_stringvec =
	(ilu_cardinal (*)(ilu_Call,ilu_string,ilu_cardinal))
	_sunrpc_SizeOfOpaque;

  new->pr_output_pipe = NULL;
  new->pr_input_pipe = NULL;
  new->pr_size_of_pipe = NULL;

  new->pr_output_sequence = _sunrpc_OutputSequence;
  new->pr_output_sequence_mark = _sunrpc_OutputSequenceMark;
  new->pr_input_sequence = _sunrpc_InputSequence;
  new->pr_input_sequence_mark = _sunrpc_InputSequenceMark;
  new->pr_end_sequence = _sunrpc_EndSequence;
  new->pr_size_of_sequence = _sunrpc_SizeOfSequence;

  new->pr_output_record = _sunrpc_OutputRecord;
  new->pr_input_record = _sunrpc_InputRecord;
  new->pr_end_record = _sunrpc_EndRecord;
  new->pr_size_of_record = _sunrpc_SizeOfRecord;

  new->pr_output_array = _sunrpc_OutputArray;
  new->pr_input_array = _sunrpc_InputArray;
  new->pr_end_array = _sunrpc_EndArray;
  new->pr_size_of_array = _sunrpc_SizeOfArray;

  new->pr_output_union = _sunrpc_OutputUnion;
  new->pr_input_union = _sunrpc_InputUnion;
  new->pr_end_union = _sunrpc_EndUnion;
  new->pr_size_of_union = _sunrpc_SizeOfUnion;

  new->pr_form_handle = _sunrpc_FormProtocolHandle;

  new->pr_read_packet = _sunrpc_ReadPacket;
  new->pr_interpret_pipe_request = _sunrpc_InterpretPipeRequest;
  new->pr_interpret_request = _sunrpc_InterpretRequest;
  new->pr_request_read = _sunrpc_DropInput;
  new->pr_interpret_reply = _sunrpc_InterpretReply;
  new->pr_finish_call = _sunrpc_DropInput;

  new->pr_create_data_block = _sunrpc_CreateDataBlock;
  new->pr_free_data_block = (void (*)(void *)) _sunrpc_FreeDataBlock;
  new->pr_flush = _sunrpc_FlushBuffer;

  return (new);
}

/*L1_sup < prmu*/

ilu_Protocol _ilu_sunrpc_Protocol(void)
{
  /*L1 >= {prmu}*/
  static ilu_Protocol StandardSunRPC = NULL;
  _ilu_AcquireMutex(ilu_prmu);
  if (StandardSunRPC == NULL)
    StandardSunRPC = _sunrpc_NewSunRPC();
  _ilu_ReleaseMutex(ilu_prmu);
  return (StandardSunRPC);
}
