/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: gc.c,v 1.30 1994/04/12 02:53:48 janssen Exp $ */
/* Last tweaked by Mike Spreitzer April 4, 1994 3:13 pm PDT */

#define _POSIX_SOURCE

#include <time.h>

#include "ilu.h"

#include "iluntrnl.h"

#include "object.h"
#include "call.h"
#include "server.h"
#include "type.h"
#include "vector.h"

#define max(a,b) (((a)<(b)) ? (a) : (b))


/* ====================== Client & Server Stuff ====================== */

/*for taking address: L1, L2, Main unconstrained*/

static struct _ilu_Method_s GCCallbackMethods[] = {
  /* name, id, cacheable, asynchronous, evector, ecount, stubproc */
  { "Ping", 1, 0, 0,  NULL, 0, NULL }
};

static struct _ilu_Class_s GCCallbackClassRecord = {
  GC_CALLBACK_CLASS_ID,	/* ILU name */
  "",			/* Brand */
  "ilu:OZjT=HxBgeC7hL01Wh2v6zommtc", /* id */
  0,			/* singleton? */
  0,			/* collectible? */
  NULL,			/* authentication */
  GCCallbackMethods,	/* methods table */
  1,			/* method count */
  0,			/* superclass count */
  NULL,			/* superclass ids base ptr */
  NULL,			/* superclass unique_id base ptr */
  FALSE			/* shown */
};

/*L1, L2, Main unconstrained*/

const ilu_Class _ilu_GcCallbackClass = &GCCallbackClassRecord;

/*L1 >= {obj's server};
  obj true && collectible => L1 >= {gcmu}*/
ilu_boolean ilu_VeryInterested(ilu_Object obj)
{
  if (   obj->ob_gclist != NULL
      && _ilu_vector_size(VECTOR(obj->ob_gclist)) != 0)
    return TRUE;
  if (   class_collectible(obj->ob_class)
      && time(NULL) < obj->ob_lastRemote + obj->ob_timeout)
    return TRUE;
  if (obj->ob_notifying && !obj->ob_known)
    return TRUE;
  if (obj->ob_holds != 0)
    return TRUE;
  return FALSE;
}


/* ====================== Server Side Stuff ====================== */

static const ilu_FineTime pingPeriod = {300, 0};

typedef struct {
  /*L1 >= {gcmu}; L2, Main unconstrained*/

  ilu_Alarmette_s gcc;	/* Periodically prompts liveness test of client */
  ilu_Object client;	/* callback interface to client */
  ilu_cardinal nObjs;	/* number of surrogates in client */
  ilu_boolean sweeping;	/* SweepGCCallbacks holds this outside gcmu */
} counted_client;
/* A server program has one of these structs for every client that has some collectible surrogates of this server. */

/*L1 >= {gcmu}; L2, Main unconstrained*/

static ilu_Alarmette_s gcoHead = {&gcoHead, &gcoHead, FALSE, {0,0}};
/* There's an alarm per collectible true object, which goes off timeout after max(lastLocal, lastRemote).  We multiplex these alarms into one alarm, _ilu_gcoAlarm, using mxamu==gcmu. */

static ilu_Alarmette_s gccHead = {&gccHead, &gccHead, FALSE, {0,0}};
/* There's an alarm per interested client, which goes off periodically to prompt liveness testing.  We multiplex these alarms into one alarm, _ilu_gccAlarm, using mxamu==gcmu. */

/*L1_sup = gcmu*/
static void gcoInvoke(ilu_Alarmette a);
/*L1 = {gcmu}*/
static void gccInvoke(ilu_Alarmette a);
static void gcoUrset(ilu_FineTime t);
static void gccUrset(ilu_FineTime t);
static void gcoUrcancel(void);
static void gccUrcancel(void);

static ilu_AlarmRep gcor = {&gcoHead, gcoInvoke, gcoUrset, gcoUrcancel};
static ilu_AlarmRep gccr = {&gccHead, gccInvoke, gccUrset, gccUrcancel};

static HashTable Clients = NULL;
/* Contains a pair <cc->client, cc> for every counted client of this prog.
 */

static ilu_Vector Objects = NULL;
/* The set {obj | obj is a GCed true object} */

/* gcmu invariant:
   cc->nObjs =  |{obj | cc in obj->ob_gclist}|.
   Objects = {obj | obj is true && collectible && not released}.
   Clients = {<cc->client, cc> | cc->nObjs != 0 || cc->sweeping}.
   (cc->nObjs != 0 || cc->sweeping) contributes 1 to cc->client->ob_holds.
   cc->gcc is set iff (cc->nObjs != 0 || cc->sweeping).
   forall collectible true obj:
	obj->ob_gclist is empty => lastRemote is when it became empty;
 */

/*L1 >= {gcmu, cmu, obj's server}*/
void _ilu_TouchedObj(ilu_Object obj)
{
  int should =   (obj->ob_gclist == NULL)
	      || (_ilu_vector_size(VECTOR(obj->ob_gclist)) == 0);
  ASSERT(object_collectible(obj), buf,
	 (buf, "gc.c:TouchedObj: !collectible(%s)", object_oid(obj) ));
  ASSERT(object_is_true(obj), buf,
	 (buf, "gc.c:TouchedObj: surrogate(%s)", object_oid(obj) ));
  if (should) {
       ilu_FineTime t = {0, 0};
       t.ft_s = obj->ob_timeout + obj->ob_lastRemote;
       ilu_MXASet(&gcor, &obj->ob_gco, t);
    }
  else ilu_MXAClear(&gcor, &obj->ob_gco);
  _ilu_VIUpdate(obj);
  return;
}

/*Main Invariant holds*/

static void CallMXO(ilu_private rock)
{
  _ilu_AcquireMutex(ilu_gcmu);
  ilu_MXAProc(ilu_FineTime_Now(), &gcor);
  _ilu_ReleaseMutex(ilu_gcmu);
}

static void CallMXC(ilu_private rock)
{
  _ilu_AcquireMutex(ilu_gcmu);
  ilu_MXAProc(ilu_FineTime_Now(), &gccr);
  _ilu_ReleaseMutex(ilu_gcmu);
}

/*forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu})*/

/*L1 >= {gcmu, cmu}*/
static void DecrementObjCount (counted_client *cc, ilu_boolean fullFlush)
{
  cc->nObjs -= 1;
  if (cc->nObjs == 0 && cc->sweeping == FALSE) {
      _ilu_hash_RemoveFromTable (Clients, cc->client);
      ilu_MXAClear(&gccr, &cc->gcc);
      if (fullFlush == TRUE) {
	  ilu_Server s = object_server(cc->client);
	  _ilu_AcquireMutex(server_lock(s));
	  _ilu_DeltaHolds(cc->client, -1);
	  _ilu_ReleaseMutex(server_lock(s));
	  free(cc);
	}
    }
}

/*L1_sup = gcmu*/
static void gcoInvoke(ilu_Alarmette a)
{
  static struct _ilu_Object_s dumby;
  ilu_Object obj = (ilu_Object) (((char *) a) - (((char *) &dumby.ob_gco) - ((char *) &dumby)));
  ilu_Server s = object_server(obj);
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  ASSERT(object_collectible(obj), buf,
	 (buf, "gc.c:gcoInvoke: !collectible(%s)", object_oid(obj) ));
  ASSERT(object_is_true(obj), buf,
	 (buf, "gc.c:gcoInvoke: surrogate(%s)", object_oid(obj) ));
  _ilu_VIUpdate(obj);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  return;
}

/*L1 >= {gcmu}; L2, Main unconstrained*/

static void gcoUrset(ilu_FineTime t)
{
  ilu_SetAlarm(_ilu_gcoAlarm, t, CallMXO, NULL);
}

static void gccUrset(ilu_FineTime t)
{
  ilu_SetAlarm(_ilu_gccAlarm, t, CallMXC, NULL);
}

static void gcoUrcancel(void)
{
  ilu_UnsetAlarm(_ilu_gcoAlarm);
}

static void gccUrcancel(void)
{
  ilu_UnsetAlarm(_ilu_gccAlarm);
}

void _ilu_StartGCingTrueObj (ilu_Object obj)
{
  _ilu_HoldMutex(ilu_gcmu);
  if (Objects == NULL)
      Objects = _ilu_vector_new(INITIAL_GCDOBJECTSET_SIZE);
  _ilu_vector_add (Objects, obj);
  return;
}

void _ilu_StopGCingTrueObj (ilu_Object obj)
{
  _ilu_HoldMutex(ilu_gcmu);
  if (obj->ob_gclist != NULL)
      _ilu_vector_destroy(VECTOR(obj->ob_gclist),
                          NULL/*vector is empty*/);
  obj->ob_gclist = NULL;
  _ilu_vector_remove (Objects, obj);
  return;
}

/*Main Invariant holds*/
static void AddGCInterest(ilu_Object obj, ilu_Object interest)
{
  counted_client *cc = NULL, **p;
  register ilu_integer i;
  ilu_integer j;
  ilu_Server s, is;

  if (obj == NULL || interest == NULL)
    return;
  s = object_server(obj);
  is = object_server(interest);
  _ilu_AcquireMutex(ilu_gcmu);
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  if (obj->ob_gclist == NULL)
      obj->ob_gclist = (void *) _ilu_vector_new(INITIAL_GCLIST_SIZE);
  if (Clients == NULL)
      Clients = _ilu_hash_MakeNewTable(INITIAL_GCCLIENTS_SIZE,
			_ilu_hash_HashPointer, _ilu_hash_PointerCompare);
  _ilu_Assert(Objects != NULL, "AddGCInterest: Objects == NULL");
    /* Because StartGCingTrueObj must have been called on obj. */
  for ( i = 0, j = _ilu_vector_size(VECTOR(obj->ob_gclist)),
	p = (counted_client **)
	    _ilu_vector_elements(VECTOR(obj->ob_gclist))
       ;  i < j;  i++)
    if (p[i]->client == interest) {
        _ilu_ReleaseMutex(server_lock(s));
        _ilu_ReleaseMutex(ilu_cmu);
        _ilu_ReleaseMutex(ilu_gcmu);
        return;		/* already there */
      }
  cc = (counted_client*) _ilu_hash_FindInTable(Clients, interest);
  if (cc == NULL) {
      ilu_Alarmette_s gcc = {NULL, NULL, FALSE, {0, 0}};
      cc = (counted_client *) malloc(sizeof(counted_client));
      cc->nObjs = 0;
      cc->client = interest;
      cc->sweeping = FALSE;
      cc->gcc = gcc;
      _ilu_hash_AddToTable (Clients, interest, cc);
      ilu_MXASet(&gccr, &cc->gcc,
		 ilu_FineTime_Add(ilu_FineTime_Now(), pingPeriod));
    };
  cc->nObjs += 1;
  if (cc->nObjs == 1)
      _ilu_DeltaHolds(interest, 1);
  _ilu_vector_add (VECTOR(obj->ob_gclist), cc);
  if (_ilu_vector_size(VECTOR(obj->ob_gclist)) == 1) {
      _ilu_TouchedObj(obj);
    }
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  _ilu_ReleaseMutex(ilu_gcmu);
}

/*L1 = {gcmu, cmu, obj's server} before, {gcmu, cmu} after;
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu})*/
static void DropGCInterest (ilu_Object obj, ilu_Object interest,
			    ilu_boolean fullFlush)
{
  register ilu_integer i, j;
  register counted_client **p;

  if (obj == NULL OR obj->ob_gclist == NULL OR interest == NULL)
      return;
  
  for ( i = 0,  j = _ilu_vector_size(VECTOR(obj->ob_gclist)),
	p = (counted_client **)
	      _ilu_vector_elements(VECTOR(obj->ob_gclist))
       ;  i < j;  i++)
    {
      if (p[i]->client == interest) {
	  counted_client *p2 = p[i];
	  _ilu_vector_remove (VECTOR(obj->ob_gclist), p2);
	  if (_ilu_vector_size(VECTOR(obj->ob_gclist)) == 0) {
	      obj->ob_lastRemote = time(NULL);
	      _ilu_TouchedObj(obj);
	    }
	  _ilu_ReleaseMutex(server_lock(object_server(obj)));
	  DecrementObjCount(p2, fullFlush);
	  return;
	}
    }
  _ilu_ReleaseMutex(server_lock(object_server(obj)));
  return;
}

/*Main Invariant holds; L2 otherwise unconstrained*/
  
static void StdFinish(ilu_Object obj)
{
  if (obj != NULL) {
      ilu_Server s = object_server(obj);
      ilu_Class cl = object_class(obj);
      ilu_EnterServer(s, cl);
      _ilu_DeltaHolds(obj, -1);
      ilu_ExitServer(s, cl);
    }
  return;
}

/*L2    >=    {conn's callmu, iomu} before,
  L2 disjoint {conn's callmu, iomu} after*/

void _ilu_HandleGCInterestRegistration(ilu_Call call)
{
  ilu_Object disc, interest;

  ilu_InputObjectID (call, &disc, TRUE, _ilu_rootClass);
  if (disc != NULL) {
      _ilu_DeltaHolds(disc, 1);
      ilu_ExitServer(object_server(disc), _ilu_rootClass);
    }
  ilu_InputObjectID (call, &interest, FALSE, _ilu_GcCallbackClass);
  if (interest != NULL) {
      _ilu_DeltaHolds(interest, 1);
      ilu_ExitServer(object_server(interest), _ilu_GcCallbackClass);
    }
  ilu_RequestRead(call);
  
  if (interest != NULL && object_collectible(interest)) {
      StdFinish(interest);
      interest = NULL;
    }

  if (interest != NULL && disc != NULL) {
      if (object_collectible(disc))
          AddGCInterest(disc, interest);
      if (ilu_BeginReply (call, TRUE, 0))
          ilu_FinishReply (call);
    }
  else {
      if (ilu_BeginException (call, 0,
			  ilu_ProtocolException_GarbageArguments))
          ilu_FinishException (call);
    }
  StdFinish(disc);
  StdFinish(interest);
  return;
}
     
void _ilu_HandleGCInterestDeregistration (ilu_Call call)
{
  ilu_Object disc, interest;
  ilu_Server s;
  
  ilu_InputObjectID (call, &disc, TRUE, _ilu_rootClass);
  if (disc != NULL) {
      _ilu_DeltaHolds(disc, 1);
      ilu_ExitServer(object_server(disc), _ilu_rootClass);
    }
  ilu_InputObjectID (call, &interest, FALSE, _ilu_GcCallbackClass);
  if (interest != NULL) {
      _ilu_DeltaHolds(interest, 1);
      ilu_ExitServer(object_server(interest), _ilu_GcCallbackClass);
    }
  ilu_RequestRead(call);
  
  if (interest != NULL && object_collectible(interest)) {
      StdFinish(interest);
      interest = NULL;
    }

  if (disc != NULL && interest != NULL) {
      s = object_server(disc);
      _ilu_AcquireMutex(ilu_gcmu);
      _ilu_AcquireMutex(ilu_cmu);
      _ilu_AcquireMutex(server_lock(s));
      DropGCInterest(disc, interest, TRUE);
      _ilu_ReleaseMutex(ilu_cmu);
      _ilu_ReleaseMutex(ilu_gcmu);
      if (ilu_BeginReply (call, TRUE, 0))
          ilu_FinishReply (call);
    }
  else {
      if (ilu_BeginException (call, 0,
			      ilu_ProtocolException_GarbageArguments))
          ilu_FinishException (call);
    }
  StdFinish(disc);
  StdFinish(interest);
  return;
}

/*Main Invariant holds; L2 otherwise unconstrained*/

static ilu_boolean TestCallback (counted_client *cc)
{
  ilu_Call call;
  ilu_Object obj;
  ilu_cardinal successCode;
  ilu_ProtocolException pe;
  ilu_boolean status;
  ilu_cardinal reqSize;
  ilu_Server cs;

  if (cc == NULL OR (obj = cc->client) == NULL)
    return (FALSE);
  if ((call = ilu_BeginCall(obj->ob_server)) == NULL)
    return (FALSE);
  cs = object_server(obj);
  _ilu_AcquireMutex(server_lock(cs));
  reqSize = ilu_SizeOfObjectID(call, obj, TRUE, NULL);
  _ilu_ReleaseMutex(server_lock(cs));
  ilu_BeginRequest(call, _ilu_GcCallbackClass, &GCCallbackMethods[0],
		   reqSize);
  ilu_EnterServer(cs, object_class(obj));
  ilu_OutputObjectID (call, obj, TRUE, _ilu_GcCallbackClass);
  ilu_FinishRequest (call);
  
  pe = ilu_GetReply(call, &successCode);
  if (pe==ilu_ProtocolException_Success && successCode==0)
	status = TRUE;
  else	status = FALSE;

  ilu_FinishCall (call);
  return (status);
}

/*L1 = {gcmu};
  for all conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu})*/
static void gccInvoke (ilu_Alarmette a)
{
  counted_client *cc = (counted_client *) a;
  register ilu_cardinal k, m;
  ilu_Object *q;
  ilu_boolean ok;

  _ilu_Assert(cc->nObjs>0 && cc->sweeping==FALSE && cc->client!=NULL,
	      "SweepGCCallbacks: cc malformed");
  cc->sweeping = TRUE;
  _ilu_ReleaseMutex(ilu_gcmu);
  ok = TestCallback(cc);
  _ilu_AcquireMutex(ilu_gcmu);
  cc->sweeping = FALSE;
  if (ok) {
      ilu_MXASet(&gccr, &cc->gcc,
		 ilu_FineTime_Add(ilu_FineTime_Now(), pingPeriod));
    }
  else {
      ilu_Server cs = object_server(cc->client);
      _ilu_AcquireMutex(ilu_cmu);
      q = (ilu_Object *) _ilu_vector_elements(Objects);
      m = _ilu_vector_size(Objects);
      for ( k = 0;  k < m;  k++) {
	  ilu_Object obj = q[k];
	  _ilu_AcquireMutex(server_lock(object_server(obj)));
	  DropGCInterest (obj, cc->client, FALSE);
	}
      ilu_MXAClear(&gccr, &cc->gcc);
      if (cc->nObjs == 0) {
          _ilu_AcquireMutex(server_lock(cs));
          _ilu_DeltaHolds(cc->client, -1);
          _ilu_ReleaseMutex(server_lock(cs));
          free(cc);
        }
      else
        { /* Some gcoInvoke is working on cc */ }
      _ilu_ReleaseMutex(ilu_cmu);
    }
  return;
}


/* ====================== Client Side Stuff ====================== */

/*L1, L2, Main unconstrained*/

static ilu_Object gcInterest = NULL;
/* The object representing this client program. */

void ilu_SetGcClient(ilu_Object i)
{
  _ilu_Assert(gcInterest==NULL, "SetGcClient: already set");
  gcInterest = i;
  return;
}

/*Main Invariant holds; L2 otherwise unconstrained*/

ilu_boolean _ilu_RegisterGCInterest (ilu_Object obj)
{
  ilu_Call call;
  ilu_Object gci = gcInterest;
  ilu_ProtocolException pe;
  ilu_cardinal errorStatus;
  ilu_boolean ans;
  ilu_cardinal reqSize = 0;
  ilu_Server os, is;

  if (obj == NULL OR gci == NULL)
    return (FALSE);
  if ((call = ilu_BeginCall(obj->ob_server)) == NULL)
    return (FALSE);
  os = object_server(obj);
  is = object_server(gci);
  _ilu_AcquireMutex(server_lock(os));
  reqSize += ilu_SizeOfObjectID(call, obj, TRUE, _ilu_rootClass);
  _ilu_ReleaseMutex(server_lock(os));
  _ilu_AcquireMutex(server_lock(is));
  reqSize += ilu_SizeOfObjectID(call, gci, FALSE, _ilu_GcCallbackClass);
  _ilu_ReleaseMutex(server_lock(is));
  ilu_BeginRequest(call, _ilu_rootClass,
		   _ilu_RegisterGCInterestMethod, reqSize);
  ilu_EnterServer(os, object_class(obj));
  ilu_OutputObjectID (call, obj, TRUE, _ilu_rootClass);
  ilu_EnterServer(is, object_class(gci));
  ilu_OutputObjectID (call, gci, FALSE, _ilu_GcCallbackClass);
  ilu_FinishRequest (call);
  pe = ilu_GetReply(call, &errorStatus);
  if (pe == ilu_ProtocolException_Success && errorStatus == 0)
       ans = TRUE;
  else ans = FALSE;
  ilu_FinishCall (call);
  return (ans);
}

ilu_boolean _ilu_UnregisterGCInterest (ilu_Object obj)
{
  ilu_Call call;
  ilu_Object gci = gcInterest;
  ilu_ProtocolException pe;
  ilu_cardinal errorStatus;
  ilu_boolean ans;
  ilu_cardinal reqSize = 0;
  ilu_Server os, is;

  if (obj == NULL OR gci == NULL)
    return (FALSE);
  if ((call = ilu_BeginCall(obj->ob_server)) == NULL)
    return (FALSE);
  os = object_server(obj);
  is = object_server(gci);
  _ilu_AcquireMutex(server_lock(os));
  reqSize += ilu_SizeOfObjectID(call, obj, TRUE, _ilu_rootClass);
  _ilu_ReleaseMutex(server_lock(os));
  _ilu_AcquireMutex(server_lock(is));
  reqSize += ilu_SizeOfObjectID(call, gci, FALSE, _ilu_GcCallbackClass);
  _ilu_ReleaseMutex(server_lock(is));
  ilu_BeginRequest(call, _ilu_rootClass,
		   _ilu_UnregisterGCInterestMethod, reqSize);
  ilu_EnterServer(os, object_class(obj));
  ilu_OutputObjectID (call, obj, TRUE, _ilu_rootClass);
  ilu_EnterServer(is, object_class(gci));
  ilu_OutputObjectID (call, gci, FALSE, _ilu_GcCallbackClass);
  ilu_FinishRequest (call);
  pe = ilu_GetReply(call, &errorStatus);
  if (pe == ilu_ProtocolException_Success && errorStatus == 0)
       ans = TRUE;
  else ans = FALSE;
  ilu_FinishCall (call);
  return (ans);
}


