/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: debug.c,v 1.21 1994/05/10 19:55:17 janssen Exp $ */

#define _POSIX_SOURCE

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "ilu.h"
#include "iluntrnl.h"

/*L1, L2, Main unconstrained*/

unsigned _ilu_DebugLevel = 0;

void ilu_SetDebugLevel (ilu_cardinal level)
{
  if (level != 0 || _ilu_DebugLevel != 0)
    {
      fprintf (stderr, "ILU version %s:\n", ilu_GetILUVersion());
      fprintf (stderr,
	       "ilu_SetDebugLevel:  setting debug mask from 0x%x to 0x%x\n",
	       _ilu_DebugLevel, level);
    }
  _ilu_DebugLevel = level;
}

struct debug_entry {
  ilu_string name;
  ilu_cardinal value;
};

static struct debug_entry debugs[] = {
 { "PACKET", PACKET_DEBUG },
 { "CONNECTION", CONNECTION_DEBUG },
 { "INCOMING", INCOMING_DEBUG },
 { "EXPORT", EXPORT_DEBUG },
 { "AUTHENTICATION", AUTHENTICATION_DEBUG },
 { "OBJECT", OBJECT_DEBUG },
 { "SUNRPC", SUNRPC_DEBUG },
 { "CALL", CALL_DEBUG },
 { "TCP", TCP_DEBUG },
 { "UDP", UDP_DEBUG },
 { "XNSSPP", XNSSPP_DEBUG },
 { "COURIER", COURIER_DEBUG },
 { "GC", GC_DEBUG },
 { "SERVER", SERVER_DEBUG },
 { "LOCK", LOCK_DEBUG },
 { "DCERPC", DCERPC_DEBUG },
 { "ALL", 0xFFFFFFFF }
};

static long _ilu_atoi (ilu_string p, ilu_string *success)
{
 ilu_integer sign = 1;
  ilu_cardinal base = 10;
  ilu_string s = p;
  ilu_string last;

  if (*s == '-')
    {
      s++;
      sign = -1;
    }
  else if (*s == '+')
    {
      s++;
    }

  if (*s == '0')
    {
      switch (*++s)
	{
	case 'b':
	case 'B':
	  ++s;
	  base = 2;
	  break;

	case 'x':
	case 'X':
	  ++s;
	  base = 16;
	  break;

	case 'd':
	case 'D':
	  ++s;
	  base = 10;
	  break;

	case 'o':
	case 'O':
	  ++s;
	  base = 8;
	  break;

	default:
	  --s;
	}
    }

  base = strtol(s, &last, base);
  if (base == 0 && last == s && success != NULL)
    *success = p;
  else if (last > s && success != NULL)
    *success = last;
  return (base * sign);
}  

int _ilu_casefree_cmp (const ilu_string s1, const ilu_string s2)
     /* returns 0 if s1 == s2, -1 if s1 < s2, 1 if s1 > s2 */
{
  register ilu_string p1 = s1;
  register ilu_string p2 = s2;
  register char c1;
  register char c2;

  do
    {
      c1 = tolower(*p1);
      c2 = tolower(*p2);

      if (c1 < c2)
	return (-1);
      else if (c1 > c2)
	return (1);
      else if (*p1 == (char) 0)
	return (0);
      p1++; p2++;
    }
  while (*p1 != (char) 0);
  return ((*p2 == (char) 0) ? 0 : -1);
}

void ilu_SetDebugLevelViaString (ilu_string s)
{
  if (s != NULL)
    {
      char buf[2000];
      ilu_string p = NULL;
      ilu_cardinal debug = 0, i, debugcount;

      if (((debug = _ilu_atoi (s, &p)) == 0 && p == s) || *p != '\0')
	{
	  strcpy (buf, s);
	  for ( debug = 0, p = buf, s = strchr(buf, ':'),
		debugcount = (sizeof(debugs)/sizeof(struct debug_entry));
	       p != NULL;
	       p = s + 1, s = strchr(s + 1, ':'))
	    {
	      if (s != NULL)
		*s = '\0';
	      for (i = 0;  i < debugcount;  i += 1)
		if (_ilu_casefree_cmp(debugs[i].name, p) == 0)
		  {
		    debug |= debugs[i].value;
		    break;
		  }
	      if (i >= debugcount)
		{
		  fprintf (stderr, "_ilu_AutoSetDebugLevel:  Bad debug option \"%s\" specified.  Valid flags are:  ALL", p);
		  for (i = debugcount;  i > 0;  i--)
		    fprintf (stderr, " %s", debugs[i].name);
		  fprintf (stderr, "\n");
		}
	      if (s == NULL)
		break;
	    }
	}
      ilu_SetDebugLevel (debug);
    }
}

void _ilu_AutoSetDebugLevel (void)
{
  static ilu_boolean initialized = FALSE;

  if (!initialized)	/* just do it once */
    {
      ilu_string s = (ilu_string) getenv ("ILU_DEBUG");
      if (s != NULL)
	ilu_SetDebugLevelViaString (s);
      initialized = TRUE;
    }
}

/* added for use in debugging interpreted programs */
static ilu_boolean DumpOnAssertFailure = ilu_TRUE;

void _ilu_SetDumpOnAssertFailure (ilu_boolean v)
{
  DumpOnAssertFailure = v;
}

void _ilu_Assert(int t, ilu_string id)
{
  if (t)
    return;
  fprintf(stderr,
          "\nILU runtime kernel consistency check failed; clue: %s\n",
	  id);
  if (DumpOnAssertFailure)
    *((int *) NULL) = 1;
}
