/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: connect.c,v 1.23 1994/03/19 00:52:51 spreitze Exp $ */
/* Last tweaked by Mike Spreitzer March 17, 1994 11:03 pm PST */

#define _POSIX_SOURCE

#ifdef MACOS
#pragma segment ilu
#endif

#include <string.h>
#include "ilu.h"

#include "iluntrnl.h"

#include "object.h"
#include "server.h"
#include "connect.h"
#include "transprt.h"
#include "protocol.h"

/*L1, L2, Main unconstrained*/

/*L1_sup < prmu, L1_sup < trmu*/
ilu_boolean _ilu_ParseConnectInfo (ilu_string handle,
				   ilu_ProtocolType *protocolType,
				   ilu_string *protocolInfo,
				   ilu_TransportType *transportType,
				   ilu_string *transportInfo)
{
  if (strncmp (handle, "sunrpc_2_", 9) == 0
      && strchr(handle, '|') == NULL)
    {	/* grandfather'ed case */
      ilu_cardinal programNumber, version, port;
      char host[1000], transport[100], buf[1000];

      DEBUG(CONNECTION_DEBUG,
	    (stderr,
	     "_ilu_ParseConnectInfo:  old style Sun RPC handle %s.\n",
	     handle));

      if (protocolType != NULL)
	*protocolType = ilu_ProtocolType_SunRPC;

      if (sscanf (handle, "sunrpc_2_%[^_]_%u_%u_%[^_]_%u", transport,
		  &programNumber, &version, host, &port) < 4)
	return (FALSE);

      if (transportType != NULL)
	{
	  if (strcmp (transport, "tcp") == 0)
	    *transportType = ilu_TransportType_TCP;
	  else if (strcmp (transport, "udp") == 0)
	    *transportType = ilu_TransportType_UDP;
	  else
	    return (FALSE);
	}

      if (protocolInfo != NULL)
	{
	  sprintf (buf, "sunrpc_2_%u_%u", programNumber, version);
	  *protocolInfo = _ilu_Strdup(buf);
	  DEBUG(CONNECTION_DEBUG,
		(stderr, "_ilu_ParseConnectInfo:  protocolInfo is %s.\n",
		 *protocolInfo));
	}
      if (transportInfo != NULL)
	{
	  sprintf (buf, "%s_%s_%u", transport, host, port);
	  *transportInfo = _ilu_Strdup(buf);
	  DEBUG(CONNECTION_DEBUG,
		(stderr, "_ilu_ParseConnectInfo:  transportInfo is %s.\n",
		 *transportInfo));
	}
      return (TRUE);
    }
  else
    {
      char pinfo[1000], tinfo[1000];

#ifndef MACOS
      if (sscanf (handle, "%[^|]|%s", pinfo, tinfo) != 2)
	return (FALSE);
#else
	/* code changed to fix bug in Macintosh sscanf() */
          if ((idx=strchr(handle,'|'))==NULL)
                return (FALSE);

          strncpy(pinfo,handle,idx-handle);
          pinfo[idx-handle]='\0';
          strcpy(tinfo,idx+1);
#endif	/* MACOS */

      if (protocolInfo != NULL)
	{
	  *protocolInfo = _ilu_Strdup(pinfo);
	  DEBUG(CONNECTION_DEBUG,
		(stderr, "_ilu_ParseConnectInfo:  protocolInfo is %s.\n",
		 *protocolInfo));
	}
      if (transportInfo != NULL)
	{
	  *transportInfo = _ilu_Strdup(tinfo);
	  DEBUG(CONNECTION_DEBUG,
		(stderr, "_ilu_ParseConnectInfo:  transportInfo is %s.\n",
		 *transportInfo));
	}

      if (protocolType != NULL
	  && (*protocolType = ilu_FindProtocolTypeFromInfo(pinfo))
	     == ilu_ProtocolType_None)
	return (FALSE);	
      if (transportType != NULL
	  && (*transportType = ilu_FindTransportTypeFromInfo(pinfo))
	     == ilu_TransportType_None)
	return (FALSE);	

      return (TRUE);
    }
}

/*L1_sup < prmu*/
ilu_string _ilu_ContactInfoOfConnection (ilu_Connection conn,
					 ilu_Object obj)
{
  char *ans;
  ilu_string pinfo, tinfo;
  _ilu_AcquireMutex(ilu_prmu);
  pinfo = ((obj == NULL) || (obj->ob_class == NULL))
	  ? _ilu_Strdup("???")
	  : protocol_form_handle(connection_protocol(conn), obj);
  _ilu_ReleaseMutex(ilu_prmu);
  _ilu_AcquireMutex(ilu_trmu);
  tinfo = transport_form_handle(connection_transport(conn));
  _ilu_ReleaseMutex(ilu_trmu);
  ans = _ilu_Strcat3(pinfo, "|", tinfo);
  free(tinfo);
  free(pinfo);
  return (ans);
}

/*L2 >= {conn.iomu}*/
ilu_integer _ilu_NDataPresent (ilu_Connection connection)
{
  return transport_n_data_present(connection_transport(connection));
}

/*Main Invariant holds*/

/*L2 >= {conn's iomu}*/
ilu_bytes _ilu_ReadPacket(ilu_Connection conn, ilu_cardinal *packetLength,
			  ilu_PacketType *packetType,
			  ilu_cardinal *packetSN)
{
  ilu_bytes retval;
  retval = (protocol_read_packet(connection_protocol(conn),
				 connection_transport(conn),
				 packetLength, packetType, packetSN));
  return (retval);
}

ilu_boolean _ilu_BlockingWaitForInputOnConnection(ilu_Connection conn,
						  ilu_boolean *sure,
						  ilu_FineTime *limit)
{
  ilu_Transport x = connection_transport(conn);
  return (transport_wait_for_input(x, sure, limit));
}

ilu_boolean ilu_BlockingWaitForInputOnConnection (ilu_Connection conn,
						  ilu_FineTime *limit)
{
  ilu_boolean sure;
  return (_ilu_BlockingWaitForInputOnConnection(conn, &sure, limit));
}

/*Main unconstrained*/

/*L1 >= {cmu, conn's server}*/
/*L2 >= {conn's iomu}*/
void _ilu_CloseIoingConnection (ilu_Connection conn)
{
  if (!connection_closed(conn)) {
      DEBUG(CONNECTION_DEBUG,
            (stderr, "Closing connection 0x%x via %s|%s FD %d to %s.\n",
             (unsigned long) conn, conn->co_pinfo, conn_tinfo(conn),
	     conn->co_transport->tr_fd, conn->co_server->sr_id));
      if ((conn->co_mucall != NULL) || (conn->co_nOuts > 0))
          ilu_fdsused -= conn->co_transport->tr_class->tc_cFD;
      else {
          ilu_fdsidle -= conn->co_transport->tr_class->tc_cFD;
          _ilu_UnlinkConnection(&ilu_idleConns, conn, ilu_lru);
        }
      _ilu_ClearConnectionFromServer(conn, conn->co_server);
      transport_close(connection_transport(conn));
      protocol_free_data_block( connection_protocol(conn),
				connection_protocol_data(conn));
      while (conn->co_replies != NULL) {
          Reply *r = (Reply *) conn->co_replies;
          Reply *next = r->rp_next;
          free(r->rp_packet);
          free(r);
          conn->co_replies = (ilu_private) next;
        }
      conn->co_closed = TRUE;
      if (_ilu_CanCondition())
          _ilu_NotifyCondition(conn->co_cc);
    }
  return;
}

/*L1_sup = conn's server; L1 >= {cmu}*/
/*L2 not >= {conn's iomu}*/
void _ilu_CloseConnection (ilu_Connection connection)
{
  _ilu_AcquireConnIO(connection);
  _ilu_CloseIoingConnection(connection);
  _ilu_ReleaseConnIO(connection);
  return;
}

/*L2 = {}*/
/*L1_sup < cmu*/
void ilu_CloseConnection (ilu_Connection connection)
{
  ilu_Server s = connection->co_server;
  if (connection_incoming(connection)) {
      _ilu_AcquireMutex(ilu_cmu);
      _ilu_AcquireMutex(server_lock(s));
      _ilu_CloseConnection(connection);
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_cmu);
    }
  else {
      /* Bitch to the caller, when we get our error system. */
      return;
    }
}

/*L2 = {}*/
/*L1_sup < cmu*/
void ilu_DestroyConnection (ilu_Connection connection)
{
  ilu_Server s = connection->co_server;
      DEBUG(CONNECTION_DEBUG,
            (stderr, "Destroying connection 0x%x via %s|%s to %s.\n",
             (unsigned long) connection, connection->co_pinfo, conn_tinfo(connection),
             connection->co_server->sr_id));
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  _ilu_CloseConnection(connection);
  free(conn_tinfo(connection));
  free(connection->co_pinfo);
  free(connection);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  return;
}

/*L1 >= {cmu}*/
/*L2, Main unconstrained*/

ilu_integer ilu_fdbudget = 16;		/* # FDs allowed */
ilu_integer ilu_fdsused = 0;		/* # FDs in use */
ilu_integer ilu_fdsidle = 0;		/* # FDs idle */

ilu_ConnLinks ilu_idleConns = {NULL, NULL};

/*for all conn: (L2 >= {conn's iomu}) => (L2 >= {conn's callmu})*/
/*Main unconstrained*/

/*L1_sup = cmu*/
void _ilu_ReduceFdsTo(ilu_integer goal)
{
  ilu_Connection cur = ilu_idleConns.next, next;
  _ilu_HoldMutex(ilu_cmu);
  while ((cur != NULL) && ((ilu_fdsused + ilu_fdsidle) > goal)) {
      ilu_Server s = cur->co_server;
      next = cur->co_links[ilu_lru].next;
      _ilu_AcquireMutex(server_lock(s));
      if ((cur->co_mucall == NULL) && (cur->co_nOuts == 0)) {
          _ilu_CloseConnection(cur);
        }
      _ilu_ReleaseMutex(server_lock(s));
      cur = next;
    }
  return;
}

/*L1_sup = s; L1 >= {cmu}*/
/*L2, Main unconstrained*/
ilu_Connection _ilu_CreateConnection (ilu_Transport bs, ilu_string tinfo,
				      ilu_Protocol pr, ilu_string pinfo,
				      ilu_Port port, ilu_Server s)
{
  ilu_integer dfd = transport_class(bs)->tc_cFD;
  ilu_Connection new = (ilu_Connection)
		       malloc(sizeof(struct _ilu_Connection_s));

  _ilu_HoldMutex(server_lock(s));
  new->co_mucall = NULL;
  new->co_ioing = FALSE;
  new->co_protocol = pr;
  new->co_protocol_data = protocol_create_data_block(pr);
  new->co_tinfo = _ilu_Strdup(tinfo);
  new->co_pinfo = _ilu_Strdup(pinfo);
  new->co_transport = bs;
  new->co_port = port;
  new->co_auth_info = NULL;
  new->co_server = s;
  new->co_replies = NULL;
  new->co_reader = NULL;
  if (_ilu_CanCondition())
       new->co_cc = _ilu_CreateCondition("a connection of server ", s->sr_id);
  else new->co_cc = NULL;
  new->co_closed = FALSE;
  new->co_next_sn = 1;
  new->co_nOuts = 0;
  if (server_is_true(s)) {
      _ilu_LinkConnection(&port->po_connHead, new, ilu_psl);
    }
  else {
      _ilu_LinkConnection(&s->sr_connHead, new, ilu_psl);
    }
  _ilu_LinkConnection(&ilu_idleConns, new, ilu_lru);
  ilu_fdsidle += dfd;
  DEBUG(CONNECTION_DEBUG,
	(stderr, "%s 0x%x, tinfo %s, fd %d, pinfo %s.\n",
	 "_ilu_CreateConnection:  new connection", 
	 (unsigned long) new, tinfo, transport_file_descriptor(bs), pinfo));

  return (new);
}

/*L1, L2, Main unconstrained*/

ilu_boolean ilu_ThreadPerRequest(ilu_Connection conn)
{
  return (connection_protocol(conn)->pr_concurrent_requests);
}

/*L1_sup < conn's server*/

int ilu_FileDescriptorOfConnection (ilu_Connection conn)
{
 ilu_integer fd;
 fd = (transport_file_descriptor(connection_transport(conn)));
 return (fd);
}

/*L1 >= {conn's server}*/
/*L2 as implied by name*/

void _ilu_AcquireConnIO(ilu_Connection conn)
{
  ilu_Lock sl = server_lock(conn->co_server);
  _ilu_HoldMutex(sl);
  while (conn->co_ioing == TRUE) {
      /* We shouldn't actually get here unless the I/O mutex is held by
         another thread, in which case the runtime should have provided
         a wait-on-condition operation.  We could also get here if the
         caller mistakenly holds the I/O mutex. */
      _ilu_WaitCondition(conn->co_cc, sl);
    }
  conn->co_ioing = TRUE;
  return;
}

/*L1 >= {cmu, conn's server}*/

void _ilu_ReleaseConnIO(ilu_Connection conn)
{
  ilu_Lock sl = server_lock(conn->co_server);
  _ilu_HoldMutex(ilu_cmu);
  _ilu_HoldMutex(sl);
  _ilu_Assert(conn->co_ioing == TRUE, "ReleaseConnIO");
  if (conn->co_server->sr_closing == TRUE)
      _ilu_CloseIoingConnection(conn);
  conn->co_ioing = FALSE;
  if (_ilu_CanCondition())
      _ilu_NotifyCondition(conn->co_cc);
  return;
}

void _ilu_AcquireConnCall(ilu_Connection conn, ilu_Call call)
{
  ilu_Lock sl = server_lock(conn->co_server);
  _ilu_HoldMutex(ilu_cmu);
  _ilu_HoldMutex(sl);
  _ilu_Assert(conn->co_mucall != call, "AcquireConnCall");
  while (conn->co_mucall != NULL) {
      /* We shouldn't actually get here unless the call mutex is held by
         another thread, in which case the runtime should have provided
         a wait-on-condition operation.  We could also get here if the
         caller mistakenly holds the call mutex. */
      _ilu_WaitCondition(conn->co_cc, sl);
    }
  conn->co_mucall = call;
  if (conn->co_nOuts == 0 && !connection_closed(conn)) {
      ilu_fdsidle -= connection_nFD(conn);
      ilu_fdsused += connection_nFD(conn);
      _ilu_UnlinkConnection(&ilu_idleConns, conn, ilu_lru);
    }
  return;
}

void _ilu_ReleaseConnCall(ilu_Connection conn, ilu_Call call)
{
  ilu_Lock sl = server_lock(conn->co_server);
  _ilu_HoldMutex(ilu_cmu);
  _ilu_HoldMutex(sl);
  _ilu_Assert(conn->co_mucall == call, "ReleaseConnCall");
  conn->co_mucall = NULL;
  if (_ilu_CanCondition())
      _ilu_NotifyCondition(conn->co_cc);
  if (conn->co_nOuts == 0 && !connection_closed(conn)) {
      ilu_fdsidle += connection_nFD(conn);
      ilu_fdsused -= connection_nFD(conn);
      _ilu_LinkConnection(&ilu_idleConns, conn, ilu_lru);
    }
  return;
}

/*L2, Main unconstrained*/
/*L1 >= {conn's server if k=psl; cmu if k=lru}*/

void _ilu_LinkConnection(ilu_ConnLinks *head, ilu_Connection conn,
			 ilu_ConnLinkKind k)
{
  _ilu_Assert((head->prev == NULL && head->next == NULL) ||
	      (head->prev != NULL && head->next != NULL &&
	       head->prev->co_links[k].next == NULL &&
	       head->next->co_links[k].prev == NULL),
	      "LinkConnection");
  conn->co_links[k].next = NULL;
  conn->co_links[k].prev = head->prev;
  if ( conn->co_links[k].prev != NULL )
       conn->co_links[k].prev->co_links[k].next = conn;
  else head->next = conn;
  head->prev = conn;
  return;
}

void _ilu_UnlinkConnection(ilu_ConnLinks *head, ilu_Connection conn,
			   ilu_ConnLinkKind k)
{
  _ilu_Assert(  (conn->co_links[k].prev == NULL)
		? head->next == conn
		: conn->co_links[k].prev->co_links[k].next == conn,
	      "UnlinkConnection 1");
  _ilu_Assert(  (conn->co_links[k].next == NULL)
		? head->prev == conn
		: conn->co_links[k].next->co_links[k].prev == conn,
	      "UnlinkConnection 2");
  if ( conn->co_links[k].prev != NULL )
       conn->co_links[k].prev->co_links[k].next = conn->co_links[k].next;
  else head->next			  = conn->co_links[k].next;
  if ( conn->co_links[k].next != NULL )
       conn->co_links[k].next->co_links[k].prev = conn->co_links[k].prev;
  else head->prev			  = conn->co_links[k].prev;
  return;
}
