/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: call.c,v 1.72 1994/05/05 23:26:21 janssen Exp $ */
/* Last tweaked by Mike Spreitzer April 28, 1994 3:10 pm PDT */

#define _POSIX_SOURCE

#include <stddef.h>
#include <time.h>

#include "ilu.h"
#include "iluntrnl.h"
#include "protocol.h"
#include "connect.h"
#include "transprt.h"
#include "port.h"
#include "call.h"
#include "object.h"
#include "method.h"
#include "server.h"
#include "type.h"
#include <string.h>

#define MAX_STRING		65535
#define INPUT_ERROR		1
#define OUTPUT_ERROR		2
#define CONNECTION_ERROR	3
#define SIZEOF_ERROR		4

typedef ilu_shortcardinal ilu_WChar;
typedef struct ilu_WString_s {
  ilu_cardinal len;
  ilu_WChar *chars;
} *ilu_WString;

typedef void (*commErrorHandlerProc)(ilu_Call, ilu_integer err, ilu_cardinal io_dir);
static commErrorHandlerProc commErrorHandler = NULL;

/*L1, L2, Main unconstrained*/

ilu_cardinal _ilu_SafeStrlen(ilu_string s)
{
  if (s == NULL)
    return 0;
  else
    return (strlen(s));
}

void ilu_SetCommunicationsErrorHandler (commErrorHandlerProc handler)
{
  commErrorHandler = handler;
}

static void _ilu_CommunicationsError (ilu_Call call, ilu_integer err,
				      ilu_cardinal io_dir)
{
  ilu_Server s = call_server(call);

  fprintf (stderr,
	   "_ilu_CommunicationsError:  %s (errno %d) on call %d\n",
	   ANSI_STRERROR(err), err,
	   (call == NULL) ? 0 : call->ca_SN);
  fprintf (stderr, "                              %s %s %s (fd %d).\n",
	   ((io_dir == SIZEOF_ERROR)	  ? "sizing of" : 
	    ((io_dir == CONNECTION_ERROR) ? "connection to" :
	    ((io_dir == OUTPUT_ERROR)	  ? "output to" : "input from"))),
	   "server",
	   server_id(s),
	   transport_file_descriptor(
	    connection_transport(call_connection(call))));
  if (commErrorHandler != NULL)
    (*commErrorHandler) (call, err, io_dir);
}

static ilu_Call BuildCall(ilu_Connection connection, ilu_Server server,
			  ilu_cardinal serialNumber)
{
  ilu_Call call = (ilu_Call) malloc(sizeof(struct _ilu_Call_s));
  call->ca_SN = serialNumber;
  call->ca_server = server;
  call->ca_intro_type = NULL;
  call->ca_method = NULL;
  call->ca_connection = connection;
  call->ca_private = NULL;
  return (call);
}

/*Main Invariant holds*/
/*L2 >= {call's conn's callmu, iomu} after*/

ilu_Call ilu_BeginCall (ilu_Server s)
{
  ilu_Connection conn = NULL;
  ilu_boolean conc;
  ilu_Call c;
  ilu_TransportClass tc;
  ilu_integer dfd;
  ilu_Transport t;
  _ilu_Assert(s!=NULL, "BeginCall: s==NULL");
  if (server_is_true(s)) {
      DEBUG(CALL_DEBUG,
	    (stderr,
	     "ilu_BeginCall:  attempted call on true server %s 0x%x.\n",
	     server_id(s), (unsigned long) s));
      return NULL;
    }
  conc = protocol_concurrent(s->sr_protocol);
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  while (1) {
      for (conn = server_connections(s);
	   (conn != NULL) && (conn->co_mucall != NULL);
	   conn = connection_next(conn))
        {}
      if (conn != NULL)
          break;
      tc = _ilu_GetTransportClassByName(s->sr_tinfo);
      _ilu_ReleaseMutex(server_lock(s));
      if (tc == NULL) {
          _ilu_ReleaseMutex(ilu_cmu);
          return (NULL);
        }
      dfd = tc->tc_cFD;
      if (ilu_fdbudget < ilu_fdsused + ilu_fdsidle + dfd) {
          _ilu_ReduceFdsTo(ilu_fdbudget - dfd);
          if ((ilu_fdbudget < ilu_fdsused + ilu_fdsidle + dfd) && (dfd>0))
            {
              DEBUG(CALL_DEBUG,
		    (stderr, "BeginCall: FD budget exhausted.\n"));
              _ilu_ReleaseMutex(ilu_cmu);
              return (NULL);
            }
          _ilu_AcquireMutex(server_lock(s));
        }
      else {
          _ilu_ReleaseMutex(ilu_cmu);
          t = _ilu_GetTransportFromClass(tc, s->sr_tinfo);
          if (t == NULL)
              return (NULL);
          _ilu_AcquireMutex(ilu_cmu);
          _ilu_AcquireMutex(server_lock(s));
          conn = _ilu_CreateConnection (t, s->sr_tinfo, s->sr_protocol,
					s->sr_pinfo, NULL, s);
          if (conn == NULL) {
              _ilu_ReleaseMutex(server_lock(s));
              _ilu_ReleaseMutex(ilu_cmu);
              return (NULL);
            }
        }
    }
  _ilu_Assert(conn->co_mucall == NULL, "BeginCall: conn->co_mucall != NULL");
  _ilu_Assert(!connection_closed(conn), "BeginCall: conn closed");
  if (conn->co_next_sn > 0xFFFFFF)	/* why not 0xFFFF ? */
    conn->co_next_sn = 1;
  c = BuildCall (conn, s, conn->co_next_sn++);
  _ilu_AcquireConnCall(conn, c);
  _ilu_AcquireConnIO(conn);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  DEBUG(CALL_DEBUG,
	(stderr, "%s %d is 0x%x, call->ca_server is %s, %s 0x%x\n",
	 "ilu_BeginCall:  new call SN", call_serial_number(c),
	 (unsigned long) c, server_id(s),
	 "call->ca_connection is", (unsigned long) call_connection(c)));
  return (c);
}

/*L2    >=    {call's conn's callmu, iomu} before,
 *L2 disjoint {call's conn's callmu, iomu} after */
void ilu_FinishCall (ilu_Call call)
{
  ilu_Server s;
  ilu_Connection conn;
  ilu_Protocol proto;
  if (call == NULL)
    return;
  conn = call_connection(call);
  s = connection_server(conn);
  proto = call_proto(call);
  proto->pr_finish_call(call);
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  _ilu_ReleaseConnIO(conn);
  _ilu_ReleaseConnCall(conn, call);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  free(call);
  return;
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_boolean InterpretPipeRequest (ilu_Call call, ilu_bytes packet,
					 ilu_cardinal len)
{
  return (protocol_interpret_pipe_request(call_proto(call), call,
					  packet, len));
}

static enum ilu_InterpretRequestStatus InterpretRequest (ilu_Call call,
	ilu_bytes packet, ilu_cardinal len)
{
  if (protocol_interpret_request(call_proto(call), call, packet, len))
    {
      ilu_Method m = call_method(call);
      ilu_Class intro_type = call_intro_type(call);
      DEBUG(INCOMING_DEBUG,
	    (stderr, "%s:  SN %d, class %s, method %d (%s)\n",
	     "(ILU: InterpretRequest)", call_serial_number(call),
	     class_name(intro_type),
	     (m == NULL) ? -1 : method_id(m),
	     (m == NULL) ? "n/a" : method_name(m) ));
      if (m != NULL && method_id(m) >= 0xFF00 && method_id(m) <  0xFFFF)
	{	/* internal method, handle immediately */
	  DEBUG(INCOMING_DEBUG, (stderr,
		"InterpretRequest:  internal method.\n"));
	  return (ilu_InterpretRequest_Handled);
	}
      else
	return (ilu_InterpretRequest_Succeeded);
    }
  else {
      DEBUG(INCOMING_DEBUG,
	    (stderr, "%s 0x%x, SN %d, server %s.\n",
	     "InterpretRequest:  Couldn't interpret request",
	     (unsigned long) packet, call_serial_number(call), server_id(call->ca_server)));
      return (ilu_InterpretRequest_Failed);
    }
}

/*Main Invariant holds*/

/*before: L2     >=   {conn's callmu, iomu};
  after:  L2 disjoint {conn's callmu, iomu}*/
static void AbortRequest(ilu_Call call, ilu_Connection conn, ilu_Server s)
{
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  _ilu_ReleaseConnIO(conn);
  _ilu_ReleaseConnCall(conn, call);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  free(call);
}

/*before: L2 disjoint {conn's callmu, iomu},
 *after:  L2     >=   {conn's callmu, iomu} if result==ilu_good_request,
 *after:  L2 disjoint {conn's callmu, iomu} if result!=ilu_good_request */

ilu_ReceiveRequestStatus ilu_ReceiveRequest(ilu_Connection conn,
	ilu_Call *pcall,	ilu_Class *pit,
	ilu_Method *meth,	ilu_cardinal *sn)
{
  ilu_Server server = connection_server(conn);
  ilu_integer nd;
  ilu_cardinal pkt_len;
  ilu_PacketType pkt_type;
  ilu_bytes pkt_bytes;
  enum ilu_InterpretRequestStatus req_stat;
  ilu_Call call = BuildCall(conn, server, 0);
  
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(server));
  if (connection_closed(conn)) {
      _ilu_ReleaseMutex(server_lock(server));
      _ilu_ReleaseMutex(ilu_cmu);
      return ilu_conn_closed;
    }
  _ilu_AcquireConnCall(conn, call);
  _ilu_AcquireConnIO(conn);
  _ilu_ReleaseMutex(server_lock(server));
  _ilu_ReleaseMutex(ilu_cmu);
  nd = _ilu_NDataPresent(conn);
  if (nd < 0) {
      AbortRequest(call, conn, server);
      return ilu_conn_closed;
    }
  if (nd == 0) {
      AbortRequest(call, conn, server);
      return ilu_conn_closed;
      /* Not ilu_no_request.  This proc is called directly after
         input is detected, so if 0 bytes are available it means
         our peer has abandoned the connection and ioctl(,FIONREAD,)
         declined to return an error code. */
    }
  pkt_bytes = _ilu_ReadPacket(conn, &pkt_len, &pkt_type, sn);
  if (pkt_bytes == NULL) {
      AbortRequest(call, conn, server);
      return ilu_read_failed;
    }
  if (pkt_type != ilu_PacketType_Request) {
      AbortRequest(call, conn, server);
      return ilu_not_request;
    }
  call_serial_number(call) = *sn;
  req_stat = InterpretRequest(call, pkt_bytes, pkt_len);
  if (req_stat == ilu_InterpretRequest_Failed) {
      AbortRequest(call, conn, server);
      return ilu_interp_failed;
    }
  _ilu_AcquireMutex(server_lock(server));
  if (conn->co_port->po_call_cache != NULL) {
      ilu_CachedCall *cc = conn->co_port->po_call_cache;
      int i;
      for (i=0; i < conn->co_port->po_call_cache_size; i++) {
          if ( (cc[i].cc_sn == *sn) && (cc[i].cc_intro_type == call->ca_intro_type)
              && (cc[i].cc_meth == call_method(call))
              && (cc[i].cc_reply != NULL)
              && (strcmp(cc[i].cc_tinfo, conn_tinfo(conn)) == 0)
             ) {
              _ilu_ReleaseMutex(server_lock(server));
              DEBUG(CALL_DEBUG,
                    (stderr,
                     "ilu_ReceiveRequest: resending cached reply to call 0x%x from %s.\n",
                     cc[i].cc_sn, cc[i].cc_tinfo));
              transport_send_message(conn->co_transport, cc[i].cc_reply,
                                     cc[i].cc_reply_len);
              _ilu_AcquireMutex(ilu_cmu);
              _ilu_AcquireMutex(server_lock(server));
              _ilu_ReleaseConnIO(conn);
              _ilu_ReleaseConnCall(conn, call);
              _ilu_ReleaseMutex(server_lock(server));
              _ilu_ReleaseMutex(ilu_cmu);
              return ilu_builtin_meth;
            }
        }
    }
  _ilu_ReleaseMutex(server_lock(server));
  if (req_stat == ilu_InterpretRequest_Handled) {
      (call_method(call)->me_stubproc)(call);
      return ilu_builtin_meth;
    }
  *pcall = call;
  *pit = call->ca_intro_type;
  *meth = call_method(call);
  return ilu_good_request;
}

ilu_ReceiveRequestStatus ilu_ReceivePipeRequest(ilu_Connection conn,
	ilu_Call *pcall, ilu_Class *pit, ilu_Method *meth,
	ilu_cardinal *sn)
{
  ilu_Server server = connection_server(conn);
  ilu_integer nd;
  ilu_cardinal pkt_len;
  ilu_PacketType pkt_type;
  ilu_bytes pkt_bytes;
  ilu_boolean req_stat;
  ilu_Call call = BuildCall(conn, server, 0);
  
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(server));
  if (connection_closed(conn)) {
      _ilu_ReleaseMutex(server_lock(server));
      _ilu_ReleaseMutex(ilu_cmu);
      return ilu_conn_closed;
    }
  _ilu_AcquireConnCall(conn, call);
  _ilu_AcquireConnIO(conn);
  _ilu_ReleaseMutex(server_lock(server));
  _ilu_ReleaseMutex(ilu_cmu);
  nd = _ilu_NDataPresent(conn);
  if (nd < 0) {
      AbortRequest(call, conn, server);
      return ilu_conn_closed;
    }
  _ilu_ReleaseMutex(ilu_cmu);
  if (nd == 0) {
      AbortRequest(call, conn, server);
      return ilu_no_request;
    }
  pkt_bytes = _ilu_ReadPacket(conn, &pkt_len, &pkt_type, sn);
  if (pkt_bytes == NULL) {
      AbortRequest(call, conn, server);
      return ilu_read_failed;
    }
  if (pkt_type != ilu_PacketType_Request) {
      AbortRequest(call, conn, server);
      return ilu_not_request;
    }
  call_serial_number(call) = *sn;
  req_stat = InterpretPipeRequest(call, pkt_bytes, pkt_len);
  if (req_stat == FALSE) {
      AbortRequest(call, conn, server);
      return ilu_interp_failed;
    }
  *pcall = call;
  *pit = call->ca_intro_type;
  *meth = call_method(call);
  return ilu_good_request;
}

/*L1, L2, Main unconstrained*/
static ilu_FineTime ClipAddTime(ilu_FineTime a, ilu_FineTime b,
				ilu_FineTime l)
{
  ilu_FineTime c = ilu_FineTime_Add(a, b);
  if (ilu_FineTime_Cmp(c, l) > 0)
       return l;
  else return c;
}

/*L1_sup < call's server*/
/*Main unconstrained*/

/* After the client has found a reply packet with the right serial
 * number and type==ilu_PacketType_Reply, he calls this to begin
 * decoding the reply msg.  The result is success or a
 * protocol-level error.  If a protocol error is being reported,
 * the client stub next calls ilu_FinishCall, and then passes the
 * error to client code. This procedure also decodes whether the
 * marshalled results are normal results or an exception parameter;
 * `errorStatus` gets 0 if success is being reported, otherwise the
 * exception index.  The client next calls the appropriate
 * unmarshalling routine, then ilu_FinishCall, and finally returns
 * to the client code. */
/*L2 >= {connection's iomu}*/
/*L1, Main unconstrained*/
static ilu_ProtocolException InterpretReply(ilu_Call call,
		ilu_bytes packet, ilu_cardinal len, ilu_cardinal *estatus)
{
  return(protocol_interpret_reply(call_proto(call), call, packet,
				  len, estatus));
}

/*mxamu = ilu_cmu*/
typedef struct {
  /*L1 >= {ilu_cmu}*/
  
  ilu_Alarmette_s	gra_alarmette;
  ilu_private		gra_cc;
} GetReplyAlarm;

/*L1_sup = ilu_cmu*/

static void GRInvoke(ilu_Alarmette a);
static void GRSet(ilu_FineTime t);
static void GRCancel(void);

static ilu_Alarmette_s grHead = {&grHead, &grHead, FALSE, {0, 0}};
static ilu_AlarmRep grar = {&grHead, GRInvoke, GRSet, GRCancel};

/*Main Invariant holds*/
static void GraInvoke(ilu_private rock)
{
  ilu_FineTime now = ilu_FineTime_Now();
  _ilu_AcquireMutex(ilu_cmu);
  ilu_MXAProc(now, &grar);
  _ilu_ReleaseMutex(ilu_cmu);
  return;
}

static void GRInvoke(ilu_Alarmette a)
{
  GetReplyAlarm *gra = (GetReplyAlarm*) a;
  _ilu_NotifyCondition(gra->gra_cc);
  return;
}

static void GRSet(ilu_FineTime t)
{
  ilu_SetAlarm(_ilu_grAlarm, t, GraInvoke, NULL);
  return;
}

static void GRCancel(void)
{
  ilu_UnsetAlarm(_ilu_grAlarm);
  return;
}

/*Main Invariant holds*/
/*L2 >= {call's conn's callmu, iomu}*/
ilu_ProtocolException ilu_GetReply(ilu_Call call, ilu_cardinal *estatus)
{
  ilu_Connection conn = call_connection(call);
  ilu_Protocol proto = connection_protocol(conn);
  ilu_Server server = connection_server(conn);
  ilu_Transport trans = connection_transport(conn);
  ilu_integer nd;
  ilu_cardinal pktlen, pktsn;
  ilu_PacketType pkttype;
  ilu_bytes pktbytes;
  ilu_ProtocolException ans;
  ilu_boolean input_ok, sure;
  Reply *r, **pr;
  ilu_boolean prco = protocol_concurrent(proto);
  ilu_boolean conc = prco && _ilu_CanCondition();
  ilu_FineTime now;
  ilu_FineTime timeout;
  ilu_FineTime limit, *elimit;
  ilu_FineTime grandLimit;
  GetReplyAlarm gra = {{NULL, NULL, FALSE, {0, 0}}, NULL};

  timeout = trans->tr_to1;
  if (trans->tr_class->tc_timesout) {
      now = ilu_FineTime_Now();
      limit = ilu_FineTime_Add(now, timeout);
      grandLimit = ilu_FineTime_Add(now, trans->tr_tto);
      elimit = &limit;
    }
  else elimit = NULL;
  _ilu_Assert(conn->co_mucall == call, "GetReply: don't hold callmu");
  while  (1) {
    _ilu_AcquireMutex(ilu_cmu);
    _ilu_AcquireMutex(server_lock(server));
    if (conc && connection_reader(conn) == NULL)
      connection_reader(conn) = call;
    if (conc && connection_reader(conn) != call
             && trans->tr_class->tc_timesout) {
        gra.gra_cc = connection_cc(conn);
        ilu_MXASet(&grar, &gra.gra_alarmette, limit);
      }
    _ilu_ReleaseConnIO(conn);
    _ilu_ReleaseMutex(ilu_cmu);
    if (prco) {			/* release the call mutex */
        conn->co_nOuts++;
        conn->co_mucall = NULL;
      }
    
    if (conc && connection_reader(conn) != call) {
        _ilu_WaitCondition(connection_cc(conn), server_lock(server));
        input_ok = TRUE;
        sure = FALSE;
        if (trans->tr_class->tc_timesout) {
            _ilu_AcquireMutex(ilu_cmu);
            ilu_MXAClear(&grar, &gra.gra_alarmette);
            _ilu_ReleaseMutex(ilu_cmu);
          }
      }
    else {
        _ilu_ReleaseMutex(server_lock(server));
        input_ok = _ilu_BlockingWaitForInputOnConnection(conn, &sure,
							 elimit);
        _ilu_AcquireMutex(server_lock(server));
      }
    
    if (conc && connection_reader(conn) == call) {
        connection_reader(conn) = NULL;
        _ilu_NotifyCondition(connection_cc(conn));
      }
    if (connection_closed(conn)) {
        _ilu_ReleaseMutex(server_lock(server));
        if (trans->tr_class->tc_timesout)
            protocol_drop_request(proto, call);
        return ilu_ProtocolException_LostConnection;
      }
    if (prco) {				/* re-acquire the call mutex */
        _ilu_Assert(conn->co_nOuts > 0, "GetReply: conn->co_nOuts==0 a");
        _ilu_AcquireConnCall(conn, call);
        _ilu_Assert(conn->co_nOuts > 0, "GetReply: conn->co_nOuts==0 b");
        conn->co_nOuts--;
      }
    _ilu_AcquireConnIO(conn);		/* re-acquire the I/O mutex */
    _ilu_ReleaseMutex(server_lock(server));
    if (!input_ok) {
        if (trans->tr_class->tc_timesout)
            protocol_drop_request(proto, call);
        return ilu_ProtocolException_Unknown;
      }
    if (prco) {
        for(pr = (Reply **)&conn->co_replies, r = *pr; r!=NULL; pr = &r->rp_next, r = *pr)
          if (r->rp_SN == call_serial_number(call)) {
            if (trans->tr_class->tc_timesout)
                protocol_drop_request(proto, call);
            *pr = r->rp_next;
            ans = InterpretReply(call, r->rp_packet, r->rp_len, estatus);
            free(r);
            return ans;
          }
      }
    while (1) { /* Read all available packets */
      nd = _ilu_NDataPresent(conn);
      if (nd < 0 || (nd == 0 && sure)) {
          if (nd==0) {
              DEBUG(CONNECTION_DEBUG,
                    (stderr, "%s %s because nd == 0 && sure.\n",
                     "Closing a connection to", server->sr_id));
            }
          _ilu_AcquireMutex(ilu_cmu);
          _ilu_AcquireMutex(server_lock(server));
          _ilu_CloseIoingConnection(conn);
          _ilu_ReleaseMutex(server_lock(server));
          _ilu_ReleaseMutex(ilu_cmu);
          if (trans->tr_class->tc_timesout)
              protocol_drop_request(proto, call);
          return ilu_ProtocolException_LostConnection;
        }
      if (nd > 0) {
          if ((pktbytes = _ilu_ReadPacket(conn, &pktlen, &pkttype,
					  &pktsn))
	      == NULL) {
              if (trans->tr_class->tc_timesout)
                  protocol_drop_request(proto, call);
              return (ilu_ProtocolException_Unknown);
            }
          if (pkttype == ilu_PacketType_Reply) {
              if (pktsn == call_serial_number(call)) {
                  if (trans->tr_class->tc_timesout)
                      protocol_drop_request(proto, call);
                  ans = InterpretReply(call, pktbytes, pktlen, estatus);
                  return ans;
                }
              if (prco) {
                  r = (Reply*) malloc(sizeof(Reply));
                  _ilu_Assert(r != NULL, "ilu_GetReply: malloc Reply");
                  r->rp_packet = pktbytes;
                  r->rp_len = pktlen;
                  r->rp_SN = pktsn;
                  r->rp_next = (Reply *) conn->co_replies;
                  conn->co_replies = (ilu_private) r;
                }
              else {
                  /* nowhere to put this reply */
                  free(pktbytes);
                }
            }
          else {
              /* not a reply */
              free(pktbytes);
            }
          sure = FALSE;
        }
      else
      /* nd could be 0 if we let some other thread read the replies. */
      /* nd could be 0 if BlockingWaitForInputOnConnection read some
       packets and needs to make us examine the queued replies. */
          break;
     }
    if (trans->tr_class->tc_timesout) {
        now = ilu_FineTime_Now();
        if (ilu_FineTime_Cmp(now, grandLimit) >= 0)
            break;
        if (ilu_FineTime_Cmp(now, limit) >= 0) {
            DEBUG(CONNECTION_DEBUG,
                  (stderr,
                   "GetReply: retransmitting request on conn 0x%x\n",
                   (unsigned long) conn));
            protocol_resend_request(proto, call);
            timeout = ClipAddTime(timeout, timeout, trans->tr_toN);
            limit = ClipAddTime(now, timeout, grandLimit);
          }
      }
   }
  protocol_drop_request(proto, call);
  return ilu_ProtocolException_RequestTimeout;
}

/*L2 >= {call's conn's callmu, iomu}*/
/*L1, Main unconstrained*/

/*L1_sup < prmu*/
ilu_boolean ilu_BeginRequest (ilu_Call call, ilu_Class intro_type,
			      ilu_Method method, ilu_cardinal argSize)
{
  ilu_Connection conn = call_connection(call);
  if (conn == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return FALSE;
    }
  else if (conn->co_transport->tr_class->tc_timesout
           && method->me_asynchronous)
    return (FALSE);
  else
    return (protocol_begin_request(call_proto(call), call, intro_type,
				   method, argSize));    
}

/*Main Invariant holds*/
ilu_boolean ilu_FinishRequest (ilu_Call call)
{
  if (call_connection(call) == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return FALSE;
    }
  else {
      ilu_boolean ans = protocol_finish_request(call_proto(call), call);
      if (   (!call_transport(call)->tr_class->tc_timesout)
          || method_asynchronous(call_method(call))	/* ACK! */
         )
        protocol_drop_request(call_proto(call), call);
      return (ans);
    }
}

/*L1_sup < cmu*/
/*Main unconstrained*/

/*before: L2     >= {call's conn's callmu, iomu},
 *after:  L2 not >= {call's conn's         iomu},
 *after:  L2     >= {call's conn's callmu} iff protocol not concurrent*/
void ilu_RequestRead(ilu_Call call)
{
  ilu_Connection conn = call_connection(call);
  ilu_Server s = connection_server(conn);
  ilu_Protocol p = call_proto(call);
  p->pr_request_read(call);
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  _ilu_ReleaseConnIO(conn);
  if (connection_concurrent(conn)) {
      conn->co_nOuts++;
      _ilu_ReleaseConnCall(conn, call);
    }
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  return;
}

/*before: L2 not >=   {call's conn's iomu},
	  L2     >=   {call's conn's callmu} iff protocol not concurrent;
   after: L2     >=   {call's conn's callmu, iomu} if result is true,
	  L2 disjoint {call's conn's callmu, iomu} if result is false. */

ilu_boolean ilu_BeginReply (ilu_Call call, ilu_boolean exceptions_p,
			    ilu_cardinal argSize)
{
  ilu_Server s = call_server(call);
  ilu_Connection conn = call_connection(call);
  if (conn == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return FALSE;
    }
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  if (connection_closed(conn)) {
      if (connection_concurrent(conn))
           conn->co_nOuts--;
      else _ilu_ReleaseConnCall(conn, call);
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_cmu);
      return FALSE;    
    }
  if (connection_concurrent(conn)) {
      _ilu_AcquireConnCall(conn, call);
      conn->co_nOuts--;
    }
  _ilu_AcquireConnIO(conn);
  if (protocol_begin_reply(call_proto(call), call, exceptions_p, argSize))
    {
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_cmu);
      return TRUE;
    }
  else {
      _ilu_ReleaseConnIO(conn);
      _ilu_ReleaseConnCall(conn, call);
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_cmu);
      return FALSE;    
    }
}

ilu_boolean ilu_BeginException (ilu_Call call, ilu_cardinal exceptionCode,
				ilu_cardinal argSize)
{
  ilu_Server s = call_server(call);
  ilu_Connection conn = call_connection(call);
  if (conn == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return FALSE;
    }
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  if (connection_closed(conn)) {
      if (connection_concurrent(conn))
           conn->co_nOuts--;
      else _ilu_ReleaseConnCall(conn, call);
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_cmu);
      return FALSE;    
    }
  if (connection_concurrent(conn)) {
      _ilu_AcquireConnCall(conn, call);
      conn->co_nOuts--;
    }
  _ilu_AcquireConnIO(conn);
  if (protocol_begin_exception(call_proto(call), call, exceptionCode,
			       argSize)) {
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_cmu);
      return TRUE;
    }
  else {
      _ilu_ReleaseConnIO(conn);
      _ilu_ReleaseConnCall(conn, call);
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_cmu);
      return FALSE;    
    }
}

/*Main Invariant holds*/
/*L2    >=    {call's conn's callmu, iomu} before,
 *L2 disjoint {call's conn's callmu, iomu} after*/

static void FinishServingRequest(ilu_Call call)
{
  ilu_Server s = call_server(call);
  ilu_Connection conn = call_connection(call);
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  _ilu_ReleaseConnIO(conn);
  _ilu_ReleaseConnCall(conn, call);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  free(call);
}

ilu_boolean ilu_FinishReply (ilu_Call call)
{
  ilu_Connection conn = call_connection(call);
  ilu_boolean ans;
  if (conn == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return FALSE;
    }
  ans = protocol_finish_reply(call_proto(call), call);
  FinishServingRequest(call);
  return ans;
}

ilu_boolean ilu_FinishException (ilu_Call call)
{
  ilu_Connection conn = call_connection(call);
  ilu_boolean ans;
  if (conn == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return FALSE;
    }
  ans = protocol_finish_exception(call_proto(call), call);
  FinishServingRequest(call);
  return ans;
}

/*L1 >= {call's server}; L2, Main unconstrained */
void _ilu_CacheCall(ilu_Call call, ilu_bytes reply, ilu_integer len)
{
  ilu_Connection conn = call_connection(call);
  ilu_Port p = connection_port(conn);
  ilu_string tinfo = conn_tinfo(conn);
  ilu_integer tinfo_len = strlen(tinfo);
  ilu_CachedCall *cc = p->po_call_cache;
  _ilu_Assert(tinfo_len < MAX_CCTINFO_LEN,
              "_ilu_CacheCall: tinfo too long");
  if (cc[p->po_call_cache_finger].cc_reply != NULL)
    free(cc[p->po_call_cache_finger].cc_reply);
  strcpy(cc[p->po_call_cache_finger].cc_tinfo, tinfo);
  cc[p->po_call_cache_finger].cc_sn = call_serial_number(call);
  cc[p->po_call_cache_finger].cc_intro_type = call_intro_type(call);
  cc[p->po_call_cache_finger].cc_meth = call_method(call);
  cc[p->po_call_cache_finger].cc_reply = reply;
  cc[p->po_call_cache_finger].cc_reply_len = len;
  p->po_call_cache_finger++;
  p->po_call_cache_finger %= p->po_call_cache_size;
  return;
}

/*before: L2 not >=   {call's conn's iomu},
	  L2     >=   {call's conn's callmu} iff protocol not concurrent;
  after:  L2 disjoint {call's conn's callmu, iomu}*/
void ilu_NoReply(ilu_Call call)
{
  ilu_Server s = call_server(call);
  ilu_Connection conn = call_connection(call);
  if (conn == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return;
    }
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  if (connection_concurrent(conn)) {
      _ilu_AcquireConnCall(conn, call);
      conn->co_nOuts--;
    }
  _ilu_ReleaseConnCall(conn, call);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
  free(call);
  return;
}

/*L2 >= {call's connection's callmu, iomu} for input & output,
  L2 unconstrained for sizing*/
/*L1, Main unconstrained*/

/* ==================== optional ==================== */

void ilu_OutputOptional (ilu_Call call, ilu_boolean i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_optional(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputOptional (ilu_Call call, ilu_boolean *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_optional(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfOptional (ilu_Call call, ilu_boolean i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_optional(call_proto(call), call, i);
  if (len == 0)
    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
  return (len);
}

/* ==================== integer ==================== */

void ilu_OutputInteger (ilu_Call call, ilu_integer i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_integer(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputInteger (ilu_Call call, ilu_integer *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_integer(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfInteger (ilu_Call call, ilu_integer i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_integer(call_proto(call), call, i);
  if (len == 0)
    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
  return (len);
}

/* ==================== cardinal ==================== */

void ilu_OutputCardinal (ilu_Call call, ilu_cardinal i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_cardinal(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputCardinal (ilu_Call call, ilu_cardinal *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_cardinal(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfCardinal (ilu_Call call, ilu_cardinal i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_cardinal(call_proto(call), call, i);
  if (len == 0)
    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
  else
    return (len);
  return 0;
}

/* ==================== character ==================== */

void ilu_OutputCharacter (ilu_Call call, ilu_character i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_character(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputCharacter (ilu_Call call, ilu_character *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_character(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfCharacter (ilu_Call call, ilu_character i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_character(call_proto(call), call, i);
  if (len == 0)
    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
  else
    return (len);
  return 0;
}

/* ==================== short integer ==================== */

void ilu_OutputShortInteger (ilu_Call call, ilu_shortinteger i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_short_integer(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputShortInteger (ilu_Call call, ilu_shortinteger *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_short_integer(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfShortInteger (ilu_Call call, ilu_shortinteger i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_short_integer(call_proto(call), call, i);
  if (len == 0)
    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
  else
    return (len);
  return 0;
}

/* ==================== long integer ==================== */

void ilu_OutputLongInteger (ilu_Call call, ilu_integer *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_long_integer(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputLongInteger (ilu_Call call, ilu_longinteger (*i))
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_long_integer(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfLongInteger (ilu_Call call, ilu_integer *i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_long_integer(call_proto(call), call, i);
  if (len == 0)
    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
  else
    return (len);
  return 0;
}

/* ==================== short cardinal ==================== */

void ilu_OutputShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_short_cardinal(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputShortCardinal (ilu_Call call, ilu_shortcardinal *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_short_cardinal(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_short_cardinal(call_proto(call), call, i);
  if (len == 0)
    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
  else
    return (len);
  return 0;
}

/* ==================== long cardinal ==================== */

void ilu_OutputLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_long_cardinal(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputLongCardinal (ilu_Call call, ilu_longcardinal (*i))
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_long_cardinal(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_long_cardinal(call_proto(call), call, i);
  if (len == 0)
    {
      _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
      return 0;
    }
  else
    return (len);
}

/* ==================== enumeration ==================== */

void ilu_OutputEnum (ilu_Call call, ilu_shortcardinal i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_enum_code(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputEnum (ilu_Call call, ilu_shortcardinal *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_enum_code(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfEnum (ilu_Call call, ilu_shortcardinal i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_enum_code(call_proto(call), call, i);
  if (len == 0)
    {
      _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
      return(0);
    }
  else
    return (len);
}

/* ==================== real ==================== */

void ilu_OutputReal (ilu_Call call, double i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_real(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputReal (ilu_Call call, double *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_real(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfReal (ilu_Call call, double i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_real(call_proto(call), call, i);
  if (len == 0)
    {
      _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
      return 0;
    }
  else
    return (len);
}

/* ==================== short real ==================== */

void ilu_OutputShortReal (ilu_Call call, float i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_short_real(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputShortReal (ilu_Call call, float *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_short_real(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfShortReal (ilu_Call call, float i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_short_real(call_proto(call), call, i);
  if (len == 0)
    {
      _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
      return(0);
    }
  else
    return (len);
}

/* ==================== long real ==================== */

void ilu_OutputLongReal (ilu_Call call, double i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_long_real(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputLongReal (ilu_Call call, double *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_long_real(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfLongReal (ilu_Call call, double i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_long_real(call_proto(call), call, i);
  if (len == 0)
    {
      _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
      return 0;
    }
  else
    return (len);
}

/* ==================== byte ==================== */

void ilu_OutputByte (ilu_Call call, ilu_byte i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_byte(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputByte (ilu_Call call, ilu_byte *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_byte(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfByte (ilu_Call call, ilu_byte i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_byte(call_proto(call), call, i);
  if (len == 0)
    {
      _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
      return 0;
    }
  else
    return (len);
}

/* ==================== boolean ==================== */

void ilu_OutputBoolean (ilu_Call call, ilu_boolean i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_boolean(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputBoolean (ilu_Call call, ilu_boolean *i)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_boolean(call_proto(call), call, i))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfBoolean (ilu_Call call, ilu_boolean i)
{
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  len = protocol_size_of_boolean(call_proto(call), call, i);
  if (len == 0)
    {
      _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
      return 0;
    }
  else
    return (len);
}

/* ==================== string ==================== */

void ilu_OutputString (ilu_Call call, ilu_string s, ilu_cardinal len,
		       ilu_cardinal limit, ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (limit > 0 && len > limit)
    _ilu_CommunicationsError (call, 0, OUTPUT_ERROR);
  else
    {
      if (opt)
	{
	  if (NOT protocol_output_optional(call_proto(call), call,
					   (s != NULL)))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (s == NULL)
	    return;
	}
      if (NOT protocol_output_string(call_proto(call), call, s,
				     len, limit))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_InputString (ilu_Call call, ilu_string *s, ilu_cardinal *len,
		      ilu_cardinal limit, ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  ilu_boolean test;
	  if (NOT protocol_input_optional(call_proto(call), call, &test))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (!test)
	    {
	      *s = NULL;
	      len = 0;
	      return;
	    }
	}
      if (NOT protocol_input_string(call_proto(call), call, s, len,
				    limit))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfString (ilu_Call call, ilu_string i,
			       ilu_cardinal l, ilu_cardinal limit,
			       ilu_boolean opt)
{
  ilu_cardinal len = 0;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (limit > 0 && l > limit)
    _ilu_CommunicationsError (call, 0, SIZEOF_ERROR);
  else
    {
      if (opt)
	{
	  if (0 == (len = protocol_size_of_optional(call_proto(call),
						    call, (i != NULL))))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (i == NULL)
	    return 0;
	}
      len += protocol_size_of_string(call_proto(call), call, i, l, limit);
      if (len == 0)
	{
	  _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  return 0;
	}
      else
	{
	  return (len);
	}
    }
  return 0;
}

/* ==================== stringvec ==================== */

void ilu_OutputStringVec (ilu_Call call, ilu_string i, ilu_cardinal len,
			  ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  if (NOT protocol_output_optional(call_proto(call), call,
					   (i != NULL)))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (i == NULL)
	    return;
	}
      if (NOT protocol_output_stringvec(call_proto(call), call, i, len))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_InputStringVec (ilu_Call call, ilu_string *i, ilu_cardinal len,
			 ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  ilu_boolean test;
	  if (NOT protocol_input_optional(call_proto(call), call, &test))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (!test)
	    {
	      *i = NULL;
	      len = 0;
	      return;
	    }
	}
      if (NOT protocol_input_stringvec(call_proto(call), call, i, len))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfStringVec (ilu_Call call, ilu_string i,
				  ilu_cardinal len, ilu_boolean opt)
{
  ilu_cardinal len2 = 0;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  if ((len2 = protocol_size_of_optional(call_proto(call), call, (i != NULL))) == 0)
	    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  if (i == NULL)
	    return 0;
	}
      len2 += protocol_size_of_stringvec(call_proto(call), call, i, len);
      if (len2 == 0)
	{
	  _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  return 0;
	}
      else
	return (len2);
    }
  return 0;
}

/* ==================== wstring ==================== */

ilu_cardinal ilu_SizeOfWString (ilu_Call call, ilu_wstring s,
				ilu_cardinal l1	/* size of wstring */,
				ilu_cardinal limit, ilu_string *b,
				ilu_cardinal *l2 /* size of buffer b */,
				ilu_boolean opt)
{
  ilu_string buf;
  register ilu_string p1;
  register ilu_character *p2;
  ilu_cardinal len;
  ilu_cardinal size = 0;

  if (call_connection(call) == NULL)
    {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return 0;
    }
  else if (limit > 0 && l1 > limit)
    {
      _ilu_CommunicationsError (call, 0, SIZEOF_ERROR);
      return 0;
    }
  else
    {
      if (opt)
	{
	  size += protocol_size_of_optional(call_proto(call), call, (s != NULL));
	  if (size == 0)
	    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  if (s == NULL)
	    return(size);
	}
      size += protocol_size_of_cardinal(call_proto(call), call, l1);
      if (b != NULL && *b != NULL && *l2 > 0)
	return (size + protocol_size_of_bytes(call_proto(call), call, (ilu_bytes) *b, *l2, 0));
      else
	{
	  buf = malloc(l1 * 3);
	  if (b != NULL)
	    *b = buf;
	  for (p1 = buf, p2 = s;  *p2 != 0;)
	    {
	      if (*p2 & 0xF800)
		{
		  *p1++ = 0xE0 | (*p2 >> 12);
		  *p1++ = 0x80 | (((ilu_cardinal)(*p2 & 0x0FC0)) >> 6);
		  *p1++ = 0x80 | (*p2++ & 0x3F);
		}
	      else if (*p2 & 0x0780)
		{
		  *p1++ = 0xC0 | ((ilu_cardinal)(*p2 & 0x07C0)) >> 6;
		  *p1++ = 0x80 | (*p2++ & 0x003F);
		}
	      else
		*p1++ = *p2++ & 0x7F;
	    }
	  len = p1 - buf;
	  if (l2 != NULL)
	    *l2 = len;
	  size += protocol_size_of_bytes(call_proto(call), call,
					 (ilu_bytes) buf, len, 0);
	  if (b == NULL || l2 == NULL)
	    free(buf);
	  return(size);
	}
    }
}

void ilu_OutputWString (ilu_Call call, ilu_wstring s, ilu_cardinal l1,
			ilu_cardinal limit, ilu_string *b,
			ilu_cardinal *l2, ilu_boolean opt)
{
  ilu_string buf;
  register ilu_string p1;
  register ilu_character *p2;
  ilu_cardinal len;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (limit > 0 && l1 > limit)
    _ilu_CommunicationsError (call, 0, SIZEOF_ERROR);
  else
    {
      if (opt)
	{
	  if (!protocol_output_optional(call_proto(call), call, (s != NULL)))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (s == NULL)
	    return;
	}
      protocol_output_cardinal(call_proto(call), call, l1);
      if (b != NULL && *b != NULL && *l2 > 0)
	protocol_output_bytes(call_proto(call), call, (ilu_bytes) *b, *l2, 0);
      else
	{
	  buf = (ilu_string) malloc(l1 * 3);
	  if (b != NULL)
	    *b = buf;
	  for (p1 = buf, p2 = s;  *p2 != 0;)
	    {
	      if (*p2 & 0xF800)
		{
		  *p1++ = 0xE0 | (*p2 >> 12);
		  *p1++ = 0x80 | (((ilu_cardinal)(*p2 & 0x0FC0)) >> 6);
		  *p1++ = 0x80 | (*p2++ & 0x3F);
		}
	      else if (*p2 & 0x0780)
		{
		  *p1++ = 0xC0 | (((ilu_cardinal)(*p2 & 0x07C0)) >> 6);
		  *p1++ = 0x80 | (*p2++ & 0x003F);
		}
	      else
		*p1++ = *p2++ & 0x7F;
	    }
	  len = p1 - buf;
	  if (l2 != NULL)
	    *l2 = len;
	  protocol_output_bytes(call_proto(call), call, (ilu_bytes) buf,
				len, 0);
	  if (b == NULL || l2 == NULL)
	    free(buf);
	}
    }
}

void ilu_InputWString (ilu_Call call, ilu_wstring *s, ilu_cardinal *l,
		       ilu_cardinal limit, ilu_boolean opt)
{
  ilu_string buf = NULL;
  ilu_character *ubuf = NULL;
  register ilu_string p2;
  register ilu_character *p1;
  register ilu_cardinal count;
  ilu_cardinal len = 0, len2;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  ilu_boolean test;
	  if (NOT protocol_input_optional(call_proto(call), call, &test))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (NOT test)
	    {
	      *s = NULL;
	      *l = 0;
	      return;
	    }
	}
      if (NOT protocol_input_cardinal(call_proto(call), call, &len))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
      if (NOT protocol_input_string(call_proto(call), call, &buf, &len2, 0))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
      ubuf = (ilu_wstring) malloc(sizeof(ilu_character) * len + 1);
      for (p1 = ubuf, p2 = buf, count = 0;  count < len;  count++)
	{
	  if (*p2 & 0xF0 == 0xE0)
	    *p1++ = ((*p2++ & 0x0F) << 12) & ((*p2++ & 0x3F) << 6) & (*p2++ & 0x3F);
	  else if (*p2 & 0xE0 == 0xC0)
	    *p1++ = ((*p2++ & 0x1F) << 6) & (*p2++ & 0x3F);
	  else
	    *p1++ = *p2++ & 0x7F;
	}
      *s = ubuf;
      *l = len;
    }
}

/* ==================== wstringvec ==================== */

void ilu_OutputWStringVec (ilu_Call call, ilu_wstring i, ilu_cardinal len,
			   ilu_string *buf, ilu_cardinal *blen,
			   ilu_boolean opt)
{
  ilu_OutputWString (call, i, len, len, buf, blen, opt);
}

void ilu_InputWStringVec (ilu_Call call, ilu_wstring *i, ilu_cardinal len,
			  ilu_boolean opt)
{
  ilu_cardinal len2;

  ilu_InputWString (call, i, &len2, len, opt);
  if (len2 != len)
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

ilu_cardinal ilu_SizeOfWStringVec (ilu_Call call, ilu_wstring i,
				   ilu_cardinal len, ilu_string *buf,
				   ilu_cardinal *blen, ilu_boolean opt)
{
  return (ilu_SizeOfWString (call, i, len, len, buf, blen, opt));
}

/* ==================== bytes ==================== */

void ilu_OutputBytes (ilu_Call call, ilu_bytes i, ilu_cardinal len,
		      ilu_cardinal limit, ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (limit > 0 && len > limit)
    _ilu_CommunicationsError (call, 0, OUTPUT_ERROR);
  else
    {
      if (opt)
	{
	  if (NOT protocol_output_optional(call_proto(call), call, (i != NULL)))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (i == NULL)
	    return;
	}
      if (NOT protocol_output_bytes(call_proto(call), call, i, len, limit))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_InputBytes (ilu_Call call, ilu_bytes *i, ilu_cardinal *len,
		     ilu_cardinal limit, ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  ilu_boolean test;
	  if (NOT protocol_input_optional(call_proto(call), call, &test))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (!test)
	    {
	      *i = NULL;
	      len = 0;
	      return;
	    }
	}
      if (NOT protocol_input_bytes(call_proto(call), call, i, len, limit))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfBytes (ilu_Call call, ilu_bytes i,
			      ilu_cardinal len, ilu_cardinal limit,
			      ilu_boolean opt)
{
  ilu_cardinal len2 = 0;

  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (limit > 0 && len > limit)
    _ilu_CommunicationsError (call, 0, SIZEOF_ERROR);
  else
    {
      if (opt)
	{
	  if ((len2 = protocol_size_of_optional(call_proto(call), call, (i != NULL))) == 0)
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (i == NULL)
	    return 0;
	}
      len2 += protocol_size_of_bytes(call_proto(call), call, i, len, limit);
      if (len2 == 0)
	{
	  _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  return 0;
	}
      else
	return (len2);
    }
  return 0;
}

/* ==================== opaque ==================== */

void ilu_OutputOpaque (ilu_Call call, ilu_opaque i, ilu_cardinal len, ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  if (NOT protocol_output_optional(call_proto(call), call, (i != NULL)))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (i == NULL)
	    return;
	}
      if (NOT protocol_output_opaque(call_proto(call), call, i, len))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_InputOpaque (ilu_Call call, ilu_opaque *i, ilu_cardinal len, ilu_boolean opt)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (opt)
	{
	  ilu_boolean test;
	  if (NOT protocol_input_optional(call_proto(call), call, &test))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (!test)
	    {
	      *i = NULL;
	      len = 0;
	      return;
	    }
	}
      if (NOT protocol_input_opaque(call_proto(call), call, i, len))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfOpaque (ilu_Call call, ilu_opaque i,
			       ilu_cardinal len, ilu_boolean opt)
{
  ilu_cardinal len2 = 0;

  if (call_connection(call) == NULL)
    {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return 0;
    }
  else
    {
      if (opt)
	{
	  if ((len2 = protocol_size_of_optional(call_proto(call), call, (i != NULL))) == 0)
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (i == NULL)
	    return 0;
	}
      len2 = protocol_size_of_opaque(call_proto(call), call, i, len);
      if (len2 == 0)
	{
	  _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  return 0;
	}
      else
	return (len2);
    }
}

/* ==================== object ==================== */

/*before: Inside(s, cl);
  after:				 L1 disjoint {cmu, s};
  after: cl collectible		      => L1  not >=  {gcmu};
  after: cl collectible & s surrogate => Main Invariant holds;
  where s = h's server and cl = h's type.
  (We don't really need to hold cmu for surrogate or non-collectible
   objects, but this is convenient because ilu_Enter/ExitServer can
   be used.)*/
void ilu_OutputObjectID (ilu_Call call, ilu_Object h,
			 ilu_boolean discriminator_p,
			 ilu_Class static_type)
{
  ilu_string ostr = NULL;	/* discriminator_p ? OID : SBH */
  ilu_string tstr = NULL;	/* most specific type string */

  if (call_connection(call) == NULL) {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      ilu_ExitServer(object_server(h), object_class(h));
      return;
    }
  if (discriminator_p)
      ostr = object_oid(h);
  else {
      tstr = (object_mstid(h) != NULL) ? object_mstid(h)
				: class_unique_id(object_class(h));
      DEBUG(OBJECT_DEBUG,
	    (stderr, "ilu_OutputObjectID:  m-s type id for %s is <%s>\n",
	     object_oid(h), tstr));
      ostr = ilu_SBHOfObject (h);
    }
  if (object_is_true(h) && object_collectible(h)) {
      h->ob_lastRemote = time(NULL);
      _ilu_TouchedObj(h);
    }
  ilu_ExitServer(object_server(h), object_class(h));

  if (!discriminator_p) {
      if (!protocol_output_string(call_proto(call), call, tstr,
				  strlen(tstr), 0xFFFF)) {
          _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
          return;
        }
    }

  if (!protocol_output_string(call_proto(call), call, ostr,
				  _ilu_SafeStrlen(ostr), 0xFFFF))
      _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

/*before: L1 = {},
  after:  *h!=NULL => Inside(*h's server, static_type);
  after:  *h==NULL => L1 = {};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu});
  L2 >= {call's connection's callmu, iomu};
  Main otherwise unconstrained*/
void ilu_InputObjectID (ilu_Call call, ilu_Object *h,
			ilu_boolean discriminator_p,
			ilu_Class static_type)
{
  ilu_string ostr = NULL;	/* discriminator_p ? OID : SBH */
  ilu_cardinal ostrlen = 0;	/* strlen(ostr) */
  ilu_cardinal namelen = 0;
  ilu_string names = NULL;
  ilu_Server server = connection_server(call_connection(call));

  *h = NULL;
  if (static_type == NULL) {
      DEBUG(INCOMING_DEBUG|OBJECT_DEBUG,
	    (stderr, "ilu_InputObjectID(disc=%d, static_type=NULL)\n",
	     discriminator_p));
      return;
    }
  if (call_connection(call) == NULL)
    {
      _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
      return;
    }
  else
    {
      if (!discriminator_p)
	{
	  if (!protocol_input_string(call_proto(call), call, &names,
				     &namelen, 0xFFFF))
	    {
	      DEBUG(INCOMING_DEBUG|OBJECT_DEBUG,
		    (stderr,
		     "ilu_InputObjectID: Bad read of object type id.\n"));
	      _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	      return;
	    }
	  else if (names != NULL AND namelen > 0)
	    {
	      DEBUG(INCOMING_DEBUG|OBJECT_DEBUG,
		    (stderr, "%s %d, type id is \"%s\"\n",
		     "ilu_InputObjectID:  len of type id is",
		     namelen, names));
	    }
	  else if (namelen == 0)
	    {
	      if (names != NULL)
		free(names);
	      names = NULL;
	    }
	}
      
      if (!protocol_input_string(call_proto(call),
				 call, &ostr, &ostrlen, 0xFFFF))
	{
	  _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  return;
	}
      
      DEBUG(INCOMING_DEBUG,
	    (stderr, "ilu_InputObjectID:  object id/sbh is <%s>\n",
	     ostr));
      if (discriminator_p) {
	  ilu_string ih, sid;
          if (ilu_ParseOID(ostr, &ih, &sid)) {
              ilu_EnterServer(server, static_type);
              if (strcmp(sid, server_id(server)) != 0) {
                  DEBUG(INCOMING_DEBUG,
			(stderr,
			 "%s %s is for wrong server (not %s).\n",
			 "ilu_InputObjectID:  incoming oid", ostr,
			 server_id(server) ));
                  ilu_ExitServer(server, static_type);
                }
              else if (server_objs(server) == NULL) {
                  DEBUG(INCOMING_DEBUG,
			(stderr, "%s %s is in closed server %s.\n",
			 "ilu_InputObjectID:  instance", ih, sid));
                  ilu_ExitServer(server, static_type);
                }
              else if ( (*h = _ilu_FindObjectInServer(ih, server))
			== NULL) {
                  DEBUG(INCOMING_DEBUG,
			(stderr, "%s %s not found in server %s.\n",
			 "ilu_InputObjectID:  instance", ih, sid));
                  ilu_ExitServer(server, static_type);
                }
              free(ih);
              free(sid);
            }
          else {
              DEBUG(INCOMING_DEBUG,
		    (stderr,
		     "ilu_InputObjectID:  parse of oid %s failed.\n",
		     ostr));
            }
        }
      else {
          *h =  ilu_ObjectOfSBH(ostr, (names == NULL)
				      ? class_unique_id(static_type)
				      : names, static_type);
          if (*h == NULL) {
	      DEBUG(INCOMING_DEBUG,
		    (stderr, "%s <%s>.\n",
		     "ilu_InputObjectID:  error:  No object for SBH",
		     ostr));
            }
        }

      FREETOKEN(names);

      if (*h == NULL)
	  _ilu_CommunicationsError (call, errno, INPUT_ERROR);
      free(ostr);
    }
}

/*L1 >= {obj's server}; L2, Main unconstrained*/
ilu_cardinal ilu_SizeOfObjectID(ilu_Call call, ilu_Object h,
				ilu_boolean discriminator_p,
				ilu_Class static_type)
{
  ilu_string ostr;	/* discriminator_p ? OID : SBH */
  ilu_string tstr;	/* empty string if putative class is actual class,
			   most specific type id otherwise */
  ilu_cardinal size = 0;
  ilu_Connection conn = call_connection(call);

  if (conn == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (discriminator_p)
	ostr = object_oid(h);
      else
	{
	  ilu_cardinal size2;

          tstr = (object_mstid(h) != NULL) ? object_mstid(h)
				: class_unique_id(object_class(h));
	  ostr = ilu_SBHOfObject (h);

	  size2 = protocol_size_of_string(connection_protocol(conn), call,
					  tstr, strlen(tstr), 0xFFFF);
	  if (size2 == 0)
	    return 0;
	  size += size2;
	}

      if (ostr == NULL)
	{
	  _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  return 0;
	}
      size += protocol_size_of_string(connection_protocol(conn), call,
				      ostr, _ilu_SafeStrlen(ostr),
				      0xFFFF);
      return (size);
    }
  return 0;
}

/*************************************************************
**************************************************************
**************************************************************

  Code for sequences, records, arrays, and unions

**************************************************************
**************************************************************
*************************************************************/

void ilu_OutputSequence (ilu_Call call, ilu_cardinal i, ilu_cardinal limit, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (limit > 0 && i > limit)
    _ilu_CommunicationsError (call, 0, OUTPUT_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_output_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (NOT provided)
	    return;
	}
      if (NOT protocol_output_sequence(call_proto(call), call, i, limit))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_OutputSequenceMark (ilu_Call call, ilu_cardinal extent)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_output_sequence_mark(call_proto(call), call, extent))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputSequenceMark (ilu_Call call, ilu_cardinal extent)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_input_sequence_mark(call_proto(call), call, extent))
    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
}

void ilu_InputSequence (ilu_Call call, ilu_cardinal *i, ilu_cardinal limit, ilu_boolean optional, ilu_boolean *provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_input_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (NOT *provided)
	    return;
	}
      if (NOT protocol_input_sequence(call_proto(call), call, i, limit))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfSequence (ilu_Call call, ilu_cardinal i, ilu_cardinal limit, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (limit > 0 && i > limit)
    _ilu_CommunicationsError (call, 0, OUTPUT_ERROR);
  else
    {
      ilu_cardinal size = 0;
      if (optional)
	{
	  if ((size = protocol_size_of_optional(call_proto(call), call, provided)) == 0)
	    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  if (NOT provided)
	    return(size);
	}
      return (size + protocol_size_of_sequence(call_proto(call), call, i, limit));
    }
  return 0;
}

void ilu_EndSequence (ilu_Call call)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_end_sequence(call_proto(call), call))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

void ilu_OutputUnion (ilu_Call call, ilu_shortcardinal i, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_output_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (NOT provided)
	    return;
	}
      if (NOT protocol_output_union(call_proto(call), call, i))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_InputUnion (ilu_Call call, ilu_shortcardinal *i, ilu_boolean optional, ilu_boolean *provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_input_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (NOT *provided)
	    return;
	}
      else if (provided != NULL)
	*provided = ilu_TRUE;
      if (NOT protocol_input_union(call_proto(call), call, i))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfUnion (ilu_Call call, ilu_shortcardinal i, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      ilu_cardinal size = 0;
      if (optional)
	{
	  if ((size = protocol_size_of_optional(call_proto(call), call, provided)) == 0)
	    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  if (NOT provided)
	    return(size);
	}
      return (size + protocol_size_of_union(call_proto(call), call, i));
    }
  return 0;
}

void ilu_EndUnion (ilu_Call call)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_end_union(call_proto(call), call))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

void ilu_OutputArray (ilu_Call call, ilu_cardinal length, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_output_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (NOT provided)
	    return;
	}
      if (NOT protocol_output_array(call_proto(call), call,length))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_InputArray (ilu_Call call, ilu_boolean optional, ilu_boolean *provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_input_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (NOT *provided)
	    return;
	}
      if (NOT protocol_input_array(call_proto(call), call))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfArray (ilu_Call call, ilu_cardinal length, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      ilu_cardinal size = 0;
      if (optional)
	{
	  if ((size = protocol_size_of_optional(call_proto(call), call, provided)) == 0)
	    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  if (NOT provided)
	    return(size);
	}
      return (size + protocol_size_of_array(call_proto(call), call, length));
    }
  return 0;
}

void ilu_EndArray (ilu_Call call)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_end_array(call_proto(call), call))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

void ilu_OutputRecord (ilu_Call call, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_output_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
	  if (NOT provided)
	    return;
	}
      if (NOT protocol_output_record(call_proto(call), call))
	_ilu_CommunicationsError (call, errno, OUTPUT_ERROR);
    }
}

void ilu_InputRecord (ilu_Call call, ilu_boolean optional, ilu_boolean *provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      if (optional)
	{
	  if (NOT protocol_input_optional(call_proto(call), call, provided))
	    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
	  if (NOT *provided)
	    return;
	}
      if (NOT protocol_input_record(call_proto(call), call))
	_ilu_CommunicationsError (call, errno, INPUT_ERROR);
    }
}

/*ARGSUSED*/
ilu_cardinal ilu_SizeOfRecord (ilu_Call call, ilu_boolean optional, ilu_boolean provided)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else
    {
      ilu_cardinal size = 0;
      if (optional)
	{
	  if ((size = protocol_size_of_optional(call_proto(call), call, provided)) == 0)
	    _ilu_CommunicationsError (call, errno, SIZEOF_ERROR);
	  if (NOT provided)
	    return(size);
	}
      return (size + protocol_size_of_record(call_proto(call), call));
    }
  return 0;
}

void ilu_EndRecord (ilu_Call call)
{
  if (call_connection(call) == NULL)
    _ilu_CommunicationsError (call, errno, CONNECTION_ERROR);
  else if (NOT protocol_end_record(call_proto(call), call))
    _ilu_CommunicationsError (call, errno, INPUT_ERROR);
}

