/*
** Copyright (c) 1993, 1994 Xerox Corporation.  All Rights Reserved.
**
** Unlimited use, reproduction, and distribution of this software is
** permitted.  Any copy of this software must include both the above
** copyright notice of Xerox Corporation and this paragraph.  Any
** distribution of this software must comply with all applicable United
** States export control laws.  This software is made available AS IS,
** and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
** INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
** AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
** PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
** THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
** CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
** XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: ilu.c,v 1.23 1994/04/14 00:07:33 janssen Exp $ */

#include <malloc.h>
#include <stdio.h>
#include <string.h>

#include "ilu-c.h"
 
#define  TRUE  1
#define  FALSE 0
#define  AND   &&
#define  OR    ||
#define  NOT   !

#define OPTIONAL

#define SEQUENCE_INCREMENT 5

#define MOST_SPECIFIC_ILU_CLASS(cobj)	((cobj)->type)[0]->c
#define ILU_CLASS_OF_C_CLASS(cclass)	(cclass)[0]->c

/* L1, L2, Main unconstrained */
static ilu_boolean ILU_C_Logging = ilu_FALSE;
static ilu_boolean _ILU_C_ServerInitialized = ilu_FALSE;

static void _ILU_C_InitializeDefaultServer(void);

void ilu_CString__Free (ilu_CString s)
{
  if (s != NULL)
    free(s);
}

struct typeRecord_s {
  ilu_Class iluclass;
  ILU_C_Type cclass;
  struct typeRecord_s *next;
};

typedef struct typeRecord_s *typeRecord;

static typeRecord TypeRegistry = NULL;

void _ILU_C_RegisterSurrogateCType (ilu_Class c, ILU_C_Type t)
{
  typeRecord r;

  r = (typeRecord) malloc(sizeof(*r));
  r->iluclass = c;
  r->cclass = t;
  r->next = TypeRegistry;
  TypeRegistry = r;
}

/* Inside(obj->server, obj->class) */
static ILU_C_Object *_ILU_C_CreateSurrogateFromRegistry (ilu_Class c, ilu_Object obj)
{
  typeRecord p;
  ILU_C_Object *lspo = NULL;

  for (p = TypeRegistry;  p != NULL;  p = p->next)
    if (c == p->iluclass)
      break;
  if (p != NULL)
    {
      lspo = ( ILU_C_Object * ) malloc( sizeof( ILU_C_Object ));
      lspo->server = ((ilu_Object) obj)->ob_server;
      lspo->instanceId = ( ilu_Object ) obj;
      lspo->nmsHandle = NULL;
      lspo->data = 0;
      lspo->type = p->cclass;
      ilu_RegisterLanguageSpecificObject(obj, lspo);
    }
  else
    {
      fprintf (stderr, "ILU_C:  attempt to create surrogate instance of class \"%s\", but surrogate code is not loaded.\n", c->cl_name);
    }
  return (lspo);
}
 
ilu_string ILU_C_ClassName (ILU_C_Object *o)
{
  ilu_Class c = ILU_C_ClassRecordOfInstance (o);
  if (o == NULL)
    return (NULL);
  else
    return (c->cl_name);
}

ilu_string ILU_C_ClassID (ILU_C_Object *o)
{
  ilu_Class c = ILU_C_ClassRecordOfInstance (o);
  if (o == NULL)
    return (NULL);
  else
    return (c->cl_unique_id);
}

ilu_Class ILU_C_FindILUClassByTypeName (ilu_string typename)
{
  return (ilu_FindClassFromName(typename));
}

ilu_Class ILU_C_FindILUClassByTypeID (ilu_string typeID)
{
  return (ilu_FindClassFromID(typeID));
}

/* L1, L2, Main unconstrained */
ilu_Class ILU_C_ClassRecordOfInstance(ILU_C_Object *o)
{
  if (o == NULL OR o->type == NULL)
    return (NULL);
  return ((o->type)[0]->c);
}

/* L1, L2, Main unconstrained */
char *_ILU_C_GetNMSHandle(
    ILU_C_Object	*o )
{
  if ( NOT o )
    return( NULL );
  return( o->nmsHandle );
}

/* L1, L2, Main unconstrained */
void _ILU_C_SetNMSHandle(
    ILU_C_Object	*o,
    char	*h )
{
    if ( NOT o )
	return;
    o->nmsHandle = h;
}

/* L1, L2, Main unconstrained */
void *_ILU_C_FindMethod (ILU_C_Object *h, ilu_Class cls, int ndx)
{
  ILU_C_Type c;
  _ILU_C_MethodBlock *d;
 
  c = h->type;
  while ((d = *c++) != NULL) {
    if (d->c == cls)
      {
	return(( void * ) d->methods[ ndx ]);
      }
  }
  return(( void * ) NULL );
}
 
/* Main Invariant holds */
ilu_string ILU_C_SBHOfObject (ILU_C_Object *obj)
{
  ilu_string sbh = NULL;

  ilu_Object kobj = _ILU_C_KernelObjOfObj (obj);
  if (kobj != NULL)
    {
      sbh = ilu_SBHOfObject (kobj);
      ilu_ExitServer (obj->server, MOST_SPECIFIC_ILU_CLASS(obj));
    }
  return (sbh);
}

/*Main Invariant holds*/
ILU_C_Object *ILU_C_SBHToObject (char *sbh, char *ilu_type_id, ilu_Class putative_type)
{
  ILU_C_Object * h = NULL;
  ilu_Object obj;

  obj = ilu_ObjectOfSBH(sbh, ilu_type_id, putative_type);
  if (obj == NULL)
    return(NULL);
  h = ilu_GetLanguageSpecificObject(obj);
  if (h == NULL)
    h = _ILU_C_CreateSurrogateFromRegistry (putative_type, obj);
  ilu_ExitServer (obj->ob_server, obj->ob_class);
  return(h);
}

/* Main Invariant holds */
ilu_string ILU_C_PublishObject (ILU_C_Object *obj)
{
  ilu_string proof = NULL;

  ilu_Object kobj = _ILU_C_KernelObjOfObj (obj);
  if (kobj != NULL)
    return(ilu_PublishObject (kobj));
  else
    return (NULL);
}

/* Main Invariant holds */
ilu_boolean ILU_C_WithdrawObject (ILU_C_Object *obj, char *proof)
{
  ilu_boolean status;

  ilu_Object kobj = _ILU_C_KernelObjOfObj (obj);
  if (kobj != NULL)
    return (ilu_WithdrawObject (kobj, proof));
  else
    return (ilu_FALSE);
}

ILU_C_Object * ILU_C_LookupObject (char *oid, ilu_Class putative_type)
{
  ILU_C_Object * h = NULL;
  ilu_Object obj;

  obj = ilu_LookupObject (oid, putative_type);
  if (obj == NULL)
    return(NULL);
  h = ilu_GetLanguageSpecificObject(obj);
  if (h == NULL)
    h = _ILU_C_CreateSurrogateFromRegistry (putative_type, obj);
  ilu_ExitServer (obj->ob_server, obj->ob_class);
  return(h);
}

/* Main invariant holds */

ILU_C_Object *_ILU_C_GetServerSingleton (ilu_Server s, ilu_Class c)
{
  ILU_C_Object *h;
  ilu_Object kobj;

  if ((kobj = ilu_GetServerSingleton (s, c)) != NULL)
    {
      h = (ILU_C_Object *) ilu_GetLanguageSpecificObject (kobj);
      ilu_ExitServer (s, c);
      return (h);
    }
  else
    return (NULL);
}

/*before: not Inside (cobj->server, cobj->type->c) */
/*after:  return != NULL => Inside(cobj->server, cobj->type->c) */
ilu_Object _ILU_C_KernelObjOfObj (ILU_C_Object *cobj)
{
  ilu_EnterServer (cobj->server, MOST_SPECIFIC_ILU_CLASS(cobj));
  if (cobj->instanceId != NULL)
    return (cobj->instanceId);
  else
    {
      ilu_ExitServer (cobj->server, MOST_SPECIFIC_ILU_CLASS(cobj));
      return (NULL);
    }
}

/* locking unconstrained */
ilu_string _ILU_C_Strdup (ilu_string s)
{
  if (s == NULL)
    return (NULL);
  else
    {
      ilu_string s2 = malloc(strlen(s) + 1);
      strcpy (s2, s);
      return(s2);
    }
}

/* Main invariant holds */
ILU_C_Object *_ILU_C_CreateTrueObject (ILU_C_Type class, OPTIONAL ilu_string instance_handle, OPTIONAL ilu_Server server, void *client_data)
{
    ILU_C_Object	*lspo;
    ilu_Class		iluclass;
    ilu_string		id;
    char		idbuf[10];
    static int		idcounter = 0;

    if (server == NULL AND (NOT _ILU_C_ServerInitialized))
      _ILU_C_InitializeDefaultServer( );

    iluclass = (*class)->c;

    lspo = ( ILU_C_Object * ) malloc( sizeof( ILU_C_Object ));
    lspo->type = class;
    lspo->server = (server == NULL) ? ILU_C_DefaultServer : server;
    lspo->nmsHandle = NULL;
    lspo->data = client_data;

    if (iluclass->cl_singleton)
      id = "0";
    else if (instance_handle != NULL)
      id = _ILU_C_Strdup(instance_handle);
    else
      {
	sprintf (idbuf, "%u", ++idcounter);
	id = _ILU_C_Strdup(idbuf);
      }

    ilu_EnterServer (lspo->server, iluclass);
    lspo->instanceId = ilu_FindOrCreateTrueObject (id, lspo->server, iluclass, (ilu_refany) lspo);
    ilu_ExitServer (lspo->server, iluclass);
    if (lspo->instanceId == NULL)
      {
	free (lspo);
	fprintf (stderr, "ILU_C:  Can't create kernel object for true object of type \"%s\".\n", iluclass->cl_name);
	return (NULL);
      }
    else
      return (lspo);
}

/*before: L1 = {}; L2 >= {call's conn's callmu, iomu}
  after:  same */
ILU_C_Object *_ILU_C_InputObject (ilu_Call call, ilu_Class putative_class, ilu_boolean discriminator)
{
  ilu_Object obj = NULL;
  ILU_C_Object *o;
  ilu_Class c;
 
  ilu_InputObjectID (call, &obj, discriminator, putative_class);
  if (obj == NULL)
    return (NULL);
  /* now Inside(obj->server, obj->class) */
  if ((o = ( ILU_C_Object * ) ilu_GetLanguageSpecificObject( obj )) == NULL)
    {
      if ((c = ilu_ClassOfObject(obj)) != NULL)
	o = _ILU_C_CreateSurrogateFromRegistry (c, obj);
    }
  ilu_ExitServer (obj->ob_server, putative_class);
  return (o);
}
 
/*Main Invariant holds*/
ilu_boolean _ILU_C_OutputObject (ilu_Call	call, 
				ILU_C_Object 	*obj, 
				ilu_Class 	putative_class,
				ilu_boolean	discriminator
				)
{
    if ( obj == NULL )
	return( ilu_FALSE );
    ilu_EnterServer (obj->server, MOST_SPECIFIC_ILU_CLASS(obj));
    ilu_OutputObjectID( call, obj->instanceId, discriminator, putative_class );
    return( ilu_TRUE );
}
 
/* Main invariant holds */
ilu_cardinal _ILU_C_SizeOfObject (ilu_Call	call, 
				 ILU_C_Object 	*obj, 
				 ilu_Class	putative_class,
				 ilu_boolean	discriminator
				 )
{
    ilu_cardinal	sz;

    if ( obj == NULL )
	return( 0 );
    ilu_EnterServer (obj->server, MOST_SPECIFIC_ILU_CLASS(obj));
    sz = ilu_SizeOfObjectID( call, obj->instanceId, discriminator, putative_class );
    ilu_ExitServer (obj->server, MOST_SPECIFIC_ILU_CLASS(obj));
    return( sz );
}
 
/***********************************************************************/
/***********************************************************************/
/***********************************************************************/
/***********************************************************************/

/*		Server code					       */

/***********************************************************************/
/***********************************************************************/
/***********************************************************************/
/***********************************************************************/

#include <stdio.h>      /* I/O defs (including popen and pclose) */
#include <string.h>     /* strdup() */
#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>
#include <sys/errno.h>

static ilu_boolean 	_ILU_C_ReadConnectionRequest( int, ilu_private );
static ilu_boolean 	_ILU_C_ReadServiceRequest( int, ilu_private );

extern ilu_Exception	ex_ilu_ProtocolError;

ilu_Port 	ILU_C_DefaultPort = NULL;
ilu_Server 	ILU_C_DefaultServer = NULL;
 
/* Main invariant holds */
ilu_Server ILU_C_InitializeServer(char * serverID, ilu_ObjectTable obj_tab, char * protocol, char * transport, ilu_boolean setdefaultport)
{
  ilu_Server server;
  ilu_Port port;
  ilu_boolean closed = 0;
  char *DefaultTransport = "tcp_localhost_0";
  char *DefaultProtocol = "sunrpc_";
  int fd;
  char *sid = (serverID == NULL) ? ilu_InventID() : serverID;

  /* 
   ** create a true server then open an ILU port 
   ** so we've got something to export objects on 
   ** this function should be called only once
   */

  server = ilu_CreateTrueServer( sid, obj_tab );
  if ( server == NULL )
    {
      fprintf(stderr, "ILU_C_InitializeServer:  Error:  Couldn't create true server from (\"%s\", \"%s\", \"%s\", 0x%lx).\n",
	      sid, (protocol == NULL) ? DefaultProtocol : protocol, (transport == NULL) ? DefaultTransport : transport, (unsigned long) obj_tab);
      exit( 1 );
    }
  if (protocol != NULL AND transport != NULL)
    {
      port = ilu_CreatePort (server, (protocol == NULL) ? DefaultProtocol : protocol,
			     (transport == NULL) ? DefaultTransport : transport);
      if ( port == NULL )
	{
	  fprintf (stderr, "_ILU_C_InitializeServer:  Error:  Couldn't open Port to export object with protocol = \"%s\" and transport = \"%s\".\n", (protocol == NULL) ? DefaultProtocol : protocol, (transport == NULL) ? DefaultTransport : transport );
	  ilu_BankServer (server);
	  return (NULL);
	}
      fd = ilu_FileDescriptorOfMooringOfPort( port, &closed );
      if (setdefaultport)
	ilu_SetServerDefaultPort (server, port);
      if ( NOT ilu_RegisterInputSource (fd, ( void (*)(int, void *)) _ILU_C_ReadConnectionRequest,
					port ))
	{
	  fprintf( stderr, "_ILU_C_InitializeServer:  Error:  Can't register input source.\n" );
	  ilu_BankServer (server);
	  return (NULL);
	}
    }
  return (server);
}

static void _ILU_C_InitializeDefaultServer()
{
    char 		*DefaultTransport = "tcp_localhost_0";
    char 		*DefaultProtocol = "sunrpc_";

    /* 
    ** create a true server then open an ILU port 
    ** so we've got something to export objects on 
    ** this function should be called only once
    */

    if (_ILU_C_ServerInitialized)
      return;
    _ILU_C_ServerInitialized = TRUE;

    if ( ILU_C_DefaultPort != NULL OR ILU_C_DefaultServer != NULL )
	return;

    if ((ILU_C_DefaultServer = ILU_C_InitializeServer (NULL, NULL, DefaultProtocol, DefaultTransport, ilu_TRUE)) == NULL)
      {
        fprintf(stderr, "_ILU_C_InitializeDefaultServer:  Error:  Couldn't create true server.\n" );
        exit( 1 );
      }
    if ((ILU_C_DefaultPort = ILU_C_DefaultServer->sr_default_port) == NULL)
      {
        fprintf (stderr, "_ILU_C_InitializeDefaultServer:  Error:  Couldn't open Port to export object.\n" );
        exit( 1 );
      }
}

/* L1, L2, Main unconstrained */
ilu_Method _ILU_C_MethodOfCall( 
    ilu_Call	call )
{
    return( call->ca_method );
}

/* Main invariant holds */
static ilu_boolean _ILU_C_ReadConnectionRequest( 
    int 	fd, 
    ilu_private	arg )
{
    ilu_Port 		p;
    int 		fd2;
    ilu_boolean		closed = 0;
    ilu_Connection 	conn;
 
    p = ( ilu_Port ) arg;
    if (ILU_C_Logging)
      printf ("Mooring 0x%lx (fd %d):  ", (unsigned long) p, fd);
    if (( conn = ilu_HandleNewConnection( p, &closed )) == NULL ) {
      if (ILU_C_Logging)
	printf ("Unable to accept connection.\n");
      else
	fprintf(stderr, "_ILU_C_ReadConnectionRequest was unable to accept connection.\n");
      return( ilu_FALSE );
    }
    else if (( fd2 = ilu_FileDescriptorOfConnection( conn )) < 0 ) {
      if (ILU_C_Logging)
	printf ("Unable to get file descriptor of new conn 0x%lx!\n", (unsigned long) conn);
      else
        fprintf(stderr, "_ILU_C_ReadConnectionRequest:  Unable to get file desc of connection 0x%lx.\n", (unsigned long) conn);
      return( ilu_FALSE );
    }
    else
      {
        if ( NOT ilu_RegisterInputSource(fd2, (void (*)(int, void *))_ILU_C_ReadServiceRequest,
					 ( ilu_private ) conn ))
	  {
	    if (ILU_C_Logging)
	      printf ("Unable to register input source fd %d of conn 0x%lx!\n", fd2, (unsigned long) conn);
	    else
	      fprintf( stderr, "_ILU_C_ReadConnectionRequest:  Cannot register input source fd %d\n", fd2 );
	    return( ilu_FALSE );
	  }
	if (ILU_C_Logging)
	  printf ("New connection 0x%lx on fd %d\n", (unsigned long) conn, fd2);
        return( ilu_TRUE );
      }
}
 
/*Main Invariant holds
  before: L2 disjoint {arg's callmu, iomu}
  after:  L2     >=   {conn's callmu, iomu} if result==ilu_good_request,
  after:  L2 disjoint {conn's callmu, iomu} if result!=ilu_good_request */
static ilu_boolean _ILU_C_ReadServiceRequest(
    int 		fd, 
    ilu_private		arg )
{
    ilu_Call 			call;
    ilu_Class			class;
    ilu_Connection 		conn;
    ilu_Method			method;
    ilu_cardinal 		SN;
    ilu_ReceiveRequestStatus	stat;
 
    conn = ( ilu_Connection ) arg;
    if (ILU_C_Logging)
      printf ("Activity on connection 0x%lx, fd %d:  ", (unsigned long) conn, fd);
    ilu_UnregisterInputSource (fd);
    stat = ilu_ReceiveRequest( conn, &call, &class, &method, &SN );
    ilu_RegisterInputSource (fd, (void (*)(int, void*)) _ILU_C_ReadServiceRequest, arg);
    if (ILU_C_Logging)
      {
	int e;

	if (stat == ilu_read_failed)
	  e = errno;
	printf ("%s",
		(stat == ilu_good_request) ? "request"
		: ((stat == ilu_no_request) ? "ghost input -- OK"
		   : ((stat == ilu_builtin_meth) ? "builtin method, already handled -- OK"
		      : ((stat == ilu_conn_closed) ? "connection closed -- OK"
			 : ((stat == ilu_read_failed) ? "read failed -- Error"
			    : ((stat == ilu_not_request) ? "received message not a request -- Error"
			       : ((stat == ilu_interp_failed) ? "kernel request interpreter failed -- Error"
				  : "unknown ReceiveRequest error -- Bad Error")))))));
	if (stat == ilu_good_request)
	  printf (", SN %lu, method \"%s\" of class \"%s\"",
		  SN, method->me_name, class->cl_name);
	else if (stat == ilu_read_failed)
	  printf (", errno is %d", e);

	printf (".\n");
      }
    if ( stat == ilu_good_request )
      {
        ( *method->me_stubproc )( call );
        return( ilu_TRUE );
      }
    else if (stat == ilu_no_request OR stat == ilu_builtin_meth)
      {
	return (ilu_TRUE);
      }
    else if (stat == ilu_conn_closed)
      {
	ilu_UnregisterInputSource (fd);
	ilu_CloseConnection (conn);
	return (ilu_TRUE);
      }
    else
      return (ilu_FALSE);
}

/*Main invariant holds
 *before:  L2     >=   {call's conn's callmu, iomu}
 *after:  L2 not >= {call's conn's         iomu},
 *after:  L2     >= {call's conn's callmu} iff protocol not concurrent*/
ilu_boolean _ILU_C_FinishParameters (ilu_Call call, ILU_C_Object *obj)
{
  ilu_RequestRead (call);
  if (obj == NULL)
    {
      ilu_BeginException (call, 0, (ilu_cardinal) ilu_ProtocolException_GarbageArguments);
      ilu_FinishException (call);
      return (ilu_FALSE);
    }
  else
    return (ilu_TRUE);
}

/* Main Invariant holds */
void ILU_C_Run( )
{
    int		stop = 0;

    ilu_RunMainLoop( &stop );
}

/* unrestricted */
unsigned int _ILU_C_SafeStrlen (char *str)
{
  if (str == NULL)
    return 0;
  else
    return (strlen(str));
}

/*============================================================*/
/*============================================================*/
/*                 Sequence operations                        */
/*============================================================*/
/*============================================================*/

#define Alloc( n, t )   ( t * ) _xallocate( n * sizeof( t ))
#define Realloc( p, n, t ) ( t * ) _xreallocate( p, n * sizeof( t ))

static char *_xallocate(unsigned int);
static char *_xreallocate(char *, unsigned int);

void _ILU_C_AppendGeneric (ILU_C_Sequence h, char *p, int sz)
{
  char *ptr;

  /*
   ** place the item pointed to by p
   ** at the end of the sequence
   */

  if (h->_length >= h->_maximum)
    {
      h->_maximum = (h->_maximum + SEQUENCE_INCREMENT);
      if (h->_buffer != NULL)
	h->_buffer = realloc(h->_buffer, h->_maximum * sz);
      else
	h->_buffer = malloc(h->_maximum*sz);
    }
  ptr = h->_buffer + (h->_length * sz);
  memcpy (ptr, p, sz);
  h->_length += 1;
}

void _ILU_C_EveryElement(ILU_C_Sequence h, void (*proc)(void *,void *), int sz, void *data)
{
  int i;
  char *p;

  if (!h || h->_length <= 0)
    return;
  for(p = h->_buffer, i = 0; i < h->_length; i++, p += sz)
    (*proc)((void *) p, data);
}

void _ILU_C_PopGeneric (ILU_C_Sequence h, char *p, int sz)
{
    char	*ptr;

    /*
    ** return the top element
    ** in the sequence in p then
    ** remove it from the list.
    */

    if ( !h || h->_length <= 0 )
	return;
    memcpy( p, h->_buffer, sz );
    h->_length--;
    ptr = h->_buffer + sz;
    memmove (h->_buffer, ptr, h->_length * sz);
}

void _ILU_C_PushGeneric (ILU_C_Sequence h, char *p, int sz)
{
    int		l = h->_length;
    int		n;
    char	*ptr;

    /*
    ** place the item pointed to by p
    ** at the beginning of the sequence
    */

    h->_length++;
    n = h->_length * sz;
    if ( h->_length > h->_maximum ) {
        if ( h->_buffer )
    	    h->_buffer = Realloc( h->_buffer, n, char );
        else
    	    h->_buffer = Alloc( n, char );
	h->_maximum = h->_length;
    }
    ptr = h->_buffer + sz;

    memmove (ptr, h->_buffer, l * sz);
    memcpy (h->_buffer, p, sz);
}


static char *_xallocate (unsigned int n)
{
    char        *p = ( char * ) malloc( n );
 
    if ( !p )
        fprintf( stderr, "ILU_C:  out of memory, could not allocate %d bytes\n", n );
    return( p );
}
 
static char *_xreallocate (char * q, unsigned int n)
{
    char        *p = ( char * ) realloc( q, n );
 
    if ( !p )
        fprintf( stderr, "ILU_C:  out of memory, could not reallocate %d bytes\n", n );
    return( p );
}
 

