function ypilabel(h,label)
%YPILABEL Label y-axis in units of PI.
%	YPILABEL changes the Y-axis tick marks locations
%	to multiples or fractions of PI and labels tick
%	marks accordingly in the form "3pi" or "3pi/2" or
%	something similar only "pi" is displayed in the 
%	Greek alphabet.
%
%	YPILABEL(H) applies the changes to the axes pointed
%	to by the axes handle H.
%
%	YPILABEL('LABEL') and YPILABEL(H,'LABEL') also label
%	the y-axis in place of the ylabel command.
%
%	Caution: Because the axis labels and title take
%	the font used set in the axes object, using the 
%	YLABEL, XLABEL, or TITLE commands must be executed 
%	before XPILABEL and YPILABEL to avoid the labels
%	using the Symbol font.  The axes font is change to
%	the sympol font to display the Greek PI character.
%
%	See also XPILABEL

%       Dennis W. Brown 5-24-94, DWB 6-5-94
%       Copyright (c) 1994 by Dennis W. Brown
%       May be freely distributed.
%       Not for use in commercial products.

% Fractions are not shown is lowest terms. This could be modified
%   if you have a fraction reduction function.

if nargin < 1,
	h = gca;
	label = [];
elseif nargin == 1,
	if isstr(h),
		label = h;
		h = gca;
	else,
		label = [];
	end;
elseif nargin > 2,
	error('ypilabel: Invalid number of input arguments.');
end;


% get current ticks and limits
cticks = get(h,'YTick');
cylim = get(h,'YLim');

% number of labels already there
cnbr = length(cticks);

% interval between their ticks
cint = abs(cticks(2) - cticks(1));

% decide lower and upper tick
lower = pi*(floor(cylim(1)/pi));
upper = pi*(ceil(cylim(2)/pi));

% pick interval on pi close to current
int = fix(cint/pi)*pi;
k = 1;


% if less than pi, make smaller
if int < pi,
	k = 2^ceil(log2(pi/cint));
	int = pi/k;
	lower = pi/k*(floor(k*cylim(1)/pi));
	upper = pi/k*(ceil(k*cylim(2)/pi));
end;

% decide how many to put in their place
nbr = fix(abs((upper-lower))/int);


% generate tick marks
ticks = lower:int:upper;

% generate labels
bigstr = [];
labels = [];
all = [];
for i = 1:length(ticks);
	if ~isempty(bigstr), bigstr = [bigstr ',']; end;
	temp = round(ticks(i)/int);
	if k == 1,
		if temp == 0,
			str = '0';
		elseif temp == 1,
			str = 'p';
		elseif temp == -1,
			str = '-p';
		else
			str = [int2str(temp) 'p'];
		end;
	else,
		if temp == 0,
			str = '0';
		else,
			temp2 = rem(temp,k);
			if temp2,
				g = gcd(temp,k);
				kstr = int2str(fix(k/g));
				temp = fix(temp/g);
				if abs(temp) ~= 1,
					str = [int2str(temp) 'p/' kstr];
				elseif temp == 1,
					str = ['p/' kstr];
				else,
					str = ['-p/' kstr];
				end;
			else,
				g = gcd(temp,k);
				kstr = int2str(fix(k/g));
				temp = fix(temp/g);
				if abs(temp) ~= 1,
					str = [int2str(temp) 'p'];
				elseif temp == 1,
					str = 'p';
				else,
					str = '-p';
				end;
			end;
		end;
	end;
	bigstr = [bigstr ' ''' str ''' '];

	if ~rem(i,9),
		eval(['tstr = str2mat(' bigstr ');']);
		if ~isempty(labels),
			labels = str2mat(labels,tstr);
		else,
			labels = tstr;
		end;
		bigstr = [];
	end;
end;
if ~isempty(bigstr),
	eval(['tstr = str2mat(' bigstr ');']);
	if ~isempty(labels),
		labels = str2mat(labels,tstr);
	else,
		labels = tstr;
	end;
end;

% set in axis
set(h,'YLim',[lower upper]);

set(h,'YTick',ticks','YTickLabels',labels);

% set fonts
old = get(get(h,'ylabel'),'FontName');
set(h,'Fontname','symbol');
if ~isempty(label),
	set(get(h,'ylabel'),'String',label,'FontName',old,'Visible','on');
end;

