;;;-*-Mode:LISP; Package:(PCL LISP 1000); Base:10; Syntax:Common-lisp -*-
;;;
;;;
;;; Defsystem stuff, ripped off from March 17th PCL, and only very
;;; slightly altered to allow for loading source.
;;;
;;;
;;; *************************************************************************
;;; Copyright (c) 1985, 1986, 1987, 1988 Xerox Corporation.
;;; All rights reserved.
;;;
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted.  Any distribution of this
;;; software or derivative works must comply with all applicable United
;;; States export control laws.
;;; 
;;; This software is made available AS IS, and Xerox Corporation makes no
;;; warranty about the software, its performance or its conformity to any
;;; specification.
;;; 
;;; Any person obtaining a copy of this software is requested to send their
;;; name and post office or electronic mail address to:
;;;   CommonLoops Coordinator
;;;   Xerox Artifical Intelligence Systems
;;;   2400 Hanover St.
;;;   Palo Alto, CA 94303
;;; (or send Arpanet mail to CommonLoops-Coordinator.pa@Xerox.arpa)
;;;
;;; Suggestions, comments and requests for improvements are also welcome.
;;; *************************************************************************



(in-package 'pcl)


(eval-when (compile load eval)

(defun get-system (name)
  (get name 'system-definition))

(defun set-system (name new-value)
  (setf (get name 'system-definition) new-value))

(defmacro defsystem (name directory files)
  `(set-system ',name (list #'(lambda () ,directory)
			    (make-modules ',files)
			    ',(mapcar #'car files))))
)


;;;
;;; The internal datastructure used when operating on a system.
;;; 
(defstruct (module (:constructor make-module (name))
                   (:print-function
                     (lambda (m s d)
                       (declare (ignore d))
                       (format s "#<Module ~A>" (module-name m)))))
  name
  load-env
  comp-env
  recomp-reasons)

(defun make-modules (system-description)
  (let ((modules ()))
    (labels ((get-module (name)
               (or (find name modules :key #'module-name)
                   (progn (setq modules (cons (make-module name) modules))
                          (car modules))))
             (parse-spec (spec)
               (if (eq spec 't)
                   (reverse (cdr modules))                   
                   (mapcar #'get-module spec))))
      (dolist (file system-description)
        (let* ((name (car file))
               (port (car (cddddr file)))
               (module nil))
          (when (or (null port)
                    (member port *port*))
            (setq module (get-module name))
            (setf (module-load-env module) (parse-spec (cadr file))
                  (module-comp-env module) (parse-spec (caddr file))
                  (module-recomp-reasons module) (parse-spec
                                                   (cadddr file))))))
      (let ((filenames (mapcar #'car system-description)))
	(sort modules #'(lambda (name1 name2)
			  (member name2 (member name1 filenames)))
	      :key #'module-name)))))


(defun make-transformations (modules filter make-transform)
  (let ((transforms (list nil)))
    (dolist (m modules)
      (when (funcall filter m transforms)
        (funcall make-transform m transforms)))
    (reverse (cdr transforms))))

(defun make-compile-transformation (module transforms)
  (unless (dolist (trans transforms)
            (and (eq (car trans) ':compile)
                 (eq (cadr trans) module)
                 (return trans)))    
    (dolist (c (module-comp-env module))
      (make-load-transformation c transforms))
    #+symbolics-release-6 (make-load-transformation module transforms)
    (push `(:compile ,module) (cdr transforms)))
  transforms)

(defun make-load-transformation (module transforms)
  (unless (dolist (trans transforms)
            (when (eq (cadr trans) module)
              (cond ((eq (car trans) ':compile) (return nil))
                    ((eq (car trans) ':load)   (return trans)))))
    (dolist (l (module-load-env module))
      (make-load-transformation l transforms))
    (push `(:load ,module) (cdr transforms)))
  transforms)

(defun make-load-source-transformation (module transforms)
  (unless (dolist (trans transforms)
            (when (eq (cadr trans) module)
              (cond ((eq (car trans) ':compile) (return nil))
                    ((eq (car trans) ':load)   (return trans)))))
    (dolist (l (module-load-env module))
      (make-load-transformation l transforms))
    (push `(:load-source ,module) (cdr transforms)))
  transforms)

(defun make-load-without-dependencies-transformation (module transforms)
  (unless (dolist (trans transforms)
            (and (eq (car trans) ':load)
                 (eq (cadr trans) module)
                 (return trans)))
    (push `(:load ,module) (cdr transforms)))
  transforms)

(defun make-load-source-without-dependencies-transformation
  (module transforms)
  (unless (dolist (trans transforms)
            (and (eq (car trans) ':load)
                 (eq (cadr trans) module)
                 (return trans)))
    (push `(:load-source ,module) (cdr transforms)))
  transforms)

(defun compile-filter (module transforms)
  (or (dolist (r (module-recomp-reasons module))
        (when (dolist (transform transforms)
                (when (and (eq (car transform) ':compile)
                           (eq (cadr transform) r))
                  (return t)))
          (return t)))
      (null (probe-file (make-binary-pathname (module-name module))))
      (> (file-write-date (make-source-pathname (module-name module)))
         (file-write-date (make-binary-pathname (module-name module))))))

(defun operate-on-system (name mode &optional arg print-only)
  (let ((system (get-system name)))
    (unless system (error "Can't find system with name ~S." name))
    (let ((*system-directory* (funcall (car system)))
	  (modules (cadr system))
	  (transformations ()))
      (labels ((load-source (name pathname)
		 (format t "~&Loading source of ~A..." name)
		 (or print-only (load pathname)))
	       (load-binary (name pathname)
		 (format t "~&Loading binary of ~A..." name)
		 (or print-only (load pathname)))
	       
	       (load-module (m)
		 (let* ((name (module-name m))
			(*load-verbose* nil)
			(source (make-source-pathname name))
			(binary (make-binary-pathname name)))
		   (if (dolist (trans transformations)
			 (and (eq (car trans) :compile)
			      (eq (cadr trans) m)
			      (return trans)))
		       (cond ((null (probe-file binary))
			      (load-source name source))
			     ((null (probe-file source))
			      (load-binary name binary))
			     ((not
				(yes-or-no-p
				  "The definition of this system requires ~
                                   that the module ~A be loaded now even~%~
                                   though it will later be compiled.  Should ~
                                   the existing binary version be loaded ~%~
                                   instead of the source?"
				  name))
			      (load-source name source))
			     (t
			      (load-binary name binary)))
		       (if (null (probe-file binary))
			   (load-source name source)
			   (load-binary name binary)))))
	       
	       (compile-module (m)
		 (format t "~&Compiling ~A..." (module-name m))
		 (unless print-only
		   (let  ((name (module-name m)))
		     (compile-file (make-source-pathname name)
				   :output-file
				   (make-binary-pathname name)
				   :load t))))
	       
	       (true (&rest ignore) (declare (ignore ignore)) 't))
	
	(setq transformations
	      (ecase mode
		(:compile
		  ;; Compile any files that have changed and any other files
		  ;; that require recompilation when another file has been
		  ;; recompiled.
		  (make-transformations
		    modules
		    #'compile-filter
		    #'make-compile-transformation))
		(:recompile
		  ;; Force recompilation of all files.
		  (make-transformations
		    modules
		    #'true
		    #'make-compile-transformation))
		(:recompile-some
		  ;; Force recompilation of some files.  Also compile the
		  ;; files that require recompilation when another file has
		  ;; been recompiled.
		  (make-transformations
		    modules
		    #'(lambda (m transforms)
			(or (member (module-name m) arg)
			    (compile-filter m transforms)))
		    #'make-compile-transformation))
		(:query-compile
		  ;; Ask the user which files to compile.  Compile those
		  ;; and any other files which must be recompiled when
		  ;; another file has been recompiled.
		  (make-transformations
		    modules
		    #'(lambda (m transforms)
			(or (compile-filter m transforms)
			    (y-or-n-p "Compile ~A?"
				      (module-name m))))
		    #'make-compile-transformation))
		(:confirm-compile
		  ;; Offer the user a chance to prevent a file from being
		  ;; recompiled.
		  (make-transformations
		    modules
		    #'(lambda (m transforms)
			(and (compile-filter m transforms)
			     (y-or-n-p "Go ahead and compile ~A?"
				       (module-name m))))
		    #'make-compile-transformation))
		(:load
		  ;; Load the whole system.
		  (make-transformations
		    modules
		    #'true
		    #'make-load-transformation))
		(:load-source
		  ;; Load just the source for the whole system,
		  ;; regardless of whether the binaries exist.
		  (make-transformations
		    modules
		    #'true
		    #'make-load-source-transformation))
		(:query-load
		  ;; Load only those files the user says to load.
		  (make-transformations
		    modules
		    #'(lambda (m transforms)
			(declare (ignore transforms))
			(y-or-n-p "Load ~A?" (module-name m)))
		    #'make-load-without-dependencies-transformation))
		(:query-load-source
		  ;; Load only those source files the user says to load.
		  (make-transformations
		    modules
		    #'(lambda (m transforms)
			(declare (ignore transforms))
			(y-or-n-p "Load source of ~A?" (module-name m)))
		    #'make-load-source-without-dependencies-transformation))))
	
	(#+Symbolics
	 compiler:compiler-warnings-context-bind
	 #+TI
	 COMPILER:COMPILER-WARNINGS-CONTEXT-BIND
	 #+:LCL3.0
	 lucid-common-lisp:with-deferred-warnings
	 #-(or Symbolics TI :LCL3.0)
	 progn
	 (loop (when (null transformations) (return t))
	       (let ((transform (pop transformations)))
		 (ecase (car transform)
		   (:compile (compile-module (cadr transform)))
		   (:load (load-module (cadr transform)))
		   (:load-source (load-source
				  (module-name (cadr transform))
				  (make-source-pathname name)))))))))))


(defun make-source-pathname (name) (make-pathname-internal name :source))
(defun make-binary-pathname (name) (make-pathname-internal name :binary))

(defun make-pathname-internal (name type)
  (let* ((extension (ecase type
                      (:source (car *pathname-extensions*))
                      (:binary (cdr *pathname-extensions*))))
         (directory (pathname
		      (etypecase *system-directory*
			(string *system-directory*)
			(pathname *system-directory*)
			(cons (ecase type
				(:source (car *system-directory*))
				(:binary (cdr *system-directory*)))))))
         (pathname
           (make-pathname
             :name #-VMS (string-downcase (string name))
                   #+VMS (string-downcase (substitute #\_ #\- (string name)))
             :type extension
             :defaults directory)))

    #+Symbolics
    (setq pathname (zl:send pathname :new-raw-name (pathname-name pathname))
          pathname (zl:send pathname :new-raw-type (pathname-type pathname)))

    pathname))

