;;; -*- Mode:Common-Lisp; Package:USER; Base:10 -*-

;;;; Copyright (c) 1992 by Hwee Tou Ng. This program may be freely
;;;; copied, used, or modified provided that this copyright notice 
;;;; is included in each copy of this code and parts thereof.

(in-package :user)

(proclaim '(optimize (speed 3) (compilation-speed 0)))

(export '(free-assumption? free-predicate?
	  fault-mode-assumption? fault-mode-predicate?
	  behavior-mode-assumption? behavior-mode-predicate? 
	  simplicity diag-simplicity coherence
	  simplicity-only coherence-only
	  coherence-then-simplicity
	  pr-compute-estimate pr-combine-estimates
	  diag-compute-estimate diag-combine-estimates
	  check-answer))

(defvar *components*)

;------------------------------------------
;     Evaluation metrics for explanations
;------------------------------------------

(defvar *fault-mode-predicates*)

(defun free-assumption? (a)
  (free-predicate? (predicate a)))

(defun free-predicate? (p)
  (member p *free-assumption-predicates*))

(defun fault-mode-assumption? (a)
  (fault-mode-predicate? (predicate a)))

(defun fault-mode-predicate? (p)
  (member p *fault-mode-predicates*))

(defun behavior-mode-assumption? (a)
  (behavior-mode-predicate? (predicate a)))

(defun behavior-mode-predicate? (p)
  (member p *behavior-mode-predicates*))

; Simplicity metric = 1/A.

(defun simplicity (as rules final-goal-datum?
		   &aux num-non-free-assumptions)
  (declare (ignore rules final-goal-datum?))
  (setf num-non-free-assumptions (count-if-not #'free-assumption? as))
  (if (zerop num-non-free-assumptions)
      most-positive-fixnum
      (/ 1 num-non-free-assumptions)))

(defun diag-simplicity (as rules final-goal-datum?
			&aux num-bad-comps num-fault-mode-comps)
  (declare (ignore rules final-goal-datum?))
  (setf num-bad-comps (count-if-not #'free-assumption? as))
  (setf num-fault-mode-comps (count-if #'fault-mode-assumption? as))
  (list (if (zerop num-bad-comps)
	    most-positive-fixnum (/ 1 num-bad-comps))
	num-fault-mode-comps))

(defun pr-compute-estimate-coherence (e &aux plans)
  (dolist (a (aaa::environment-assumptions e))
    (when (and (inst-atom? a)
	       (get (third a) 'plan))
      (push (third a) plans)))
  (setf (aaa::environment-estimate e)
	(cons plans (length (aaa::environment-assumptions e)))))

(defun pr-combine-estimates-coherence (e1 e2 &aux (count 0) num-as)
  (dolist (p2 (car (aaa::environment-estimate e2)))
    (dolist (p1 (car (aaa::environment-estimate e1)))
      (when (compatible-sort-pair p1 p2)
	(incf count)
	(return))))
  (setf num-as
	(+ (cdr (aaa::environment-estimate e1))
	   (cdr (aaa::environment-estimate e2))))
  (list count
	(if (zerop num-as) most-positive-fixnum (/ 1 num-as))))

(defun pr-compute-estimate-simplicity (e)
  (setf (aaa::environment-estimate e)
	(length (aaa::environment-assumptions e))))

(defun pr-combine-estimates-simplicity (e1 e2 &aux num-as)
  (setf num-as
	(+ (aaa::environment-estimate e1)
	   (aaa::environment-estimate e2)))
  (list (if (zerop num-as)
	    most-positive-fixnum
	    (/ 1 num-as))))

(defun diag-compute-estimate (e)
  (setf (aaa::environment-estimate e) nil)
  (dolist (a (aaa::environment-assumptions e))
    (when (behavior-mode-assumption? a)
      (push a (aaa::environment-estimate e)))))

; Check for behavior mode consistency.  If inconsistent, return nil.
; Else return diag-simplicity metric.

(defun diag-combine-estimates (e1 e2 &aux mode (num-bad-comps 0)
			       (num-fault-mode-comps 0))
  (dolist (c *components*)
    (setf mode nil)
    (dolist (a1 (aaa::environment-estimate e1))
      (when (eq (second a1) c)
	(if (null mode)
	    (setf mode (predicate a1))
	    (when (not (eq (predicate a1) mode))
	      (return-from diag-combine-estimates nil)))))
    (dolist (a2 (aaa::environment-estimate e2))
      (when (eq (second a2) c)
	(if (null mode)
	    (setf mode (predicate a2))
	    (when (not (eq (predicate a2) mode))
	      (return-from diag-combine-estimates nil)))))
    (when (and mode
	       (not (free-predicate? mode)))
      (incf num-bad-comps)
      (when (fault-mode-predicate? mode)
	(incf num-fault-mode-comps))))
  (list (if (zerop num-bad-comps)
	    most-positive-fixnum (/ 1 num-bad-comps))
	num-fault-mode-comps))

(defstruct entry
  datum
  (children nil)
  (visited? nil)
  (descendants nil))

; Return the coherence metric value of an environment.
; Note: if final-goal-datum? = nil, return 0.

(defun coherence (as rules final-goal-datum? &aux input-atoms data)
  (declare (ignore as))
  (if (not final-goal-datum?) (return-from coherence 0))
  (multiple-value-setq (input-atoms data)
    (set-up-coherence-hash-table rules))
  (compute-coherence input-atoms data))

; Compute input-atoms and data. In addition,
; store all (distinct) data in *coherence-hash-table*. We also setup
; the entry-children field of each hash-table entry.

(defun set-up-coherence-hash-table (rules &aux brule b input-atom input-atoms
				    data conseq antes)
  (clrhash *coherence-hash-table*)
  (dolist (r rules)
    (setf brule (get (predicate r) 'brule))
    (setf b (unify (brule-id brule) r))
    (cond ((final-goal-datum? r)
	   (setf input-atom
		 (substitute-bindings (first (last (brule-antes brule))) b))
	   (when (null (gethash input-atom *coherence-hash-table*))
	     (setf (gethash input-atom *coherence-hash-table*)
		   (make-entry :datum input-atom))
	     (push input-atom data))
	   (pushnew input-atom input-atoms :test #'equal))
	  (t
	   (setf conseq (substitute-bindings (brule-conseq brule) b))
	   (setf antes (substitute-bindings (brule-antes brule) b))
	   (when (null (gethash conseq *coherence-hash-table*))
	     (setf (gethash conseq *coherence-hash-table*)
		   (make-entry :datum conseq))
	     (push conseq data))
	   (dolist (ante antes)
	     (when (null (gethash ante *coherence-hash-table*))
	       (setf (gethash ante *coherence-hash-table*)
		     (make-entry :datum ante))
	       (push ante data))
	     (pushnew conseq
		      (entry-children (gethash ante *coherence-hash-table*))
		      :test #'equal)))))
  (values input-atoms data))

; input-atoms is a list of (distinct) input atoms.
; data is a list of (distinct) data/nodes in the proof graph.
; This function assumes that all (distinct) data have been stored 
; in *coherence-hash-table*, and that the entry-children field of each 
; hash-table entry has been setup appropriately.

(defun compute-coherence (input-atoms data
			  &aux (num-input-atoms 0) (pos -1) (sum 0))
  (declare (fixnum num-input-atoms pos sum))
  (setf num-input-atoms (length input-atoms))
  (if (= num-input-atoms 1) (return-from compute-coherence 0))
  (dolist (d data)
    (setf (entry-descendants (gethash d *coherence-hash-table*))
	  (make-array num-input-atoms
		      :element-type 'bit :initial-element 0)))
  (dolist (i input-atoms)
    (setf (sbit (entry-descendants (gethash i *coherence-hash-table*))
		(incf pos))
	  1))
  (dolist (d data) (dfs d))
  (dotimes (i num-input-atoms)
    (declare (fixnum i))
    (do ((j (1+ i) (1+ j))
	 descendants)
	((= j num-input-atoms))
      (declare (fixnum j))
      (dolist (d data)
	(setf descendants (entry-descendants
			    (gethash d *coherence-hash-table*)))
	(when (and (= (sbit descendants i) 1)
		   (= (sbit descendants j) 1))
	  (incf sum)
	  (return)))))
  (/ sum
     (the fixnum (n-choose-2 num-input-atoms))))

; Depth first search

(defun dfs (data &aux entry)
  (setf entry (gethash data *coherence-hash-table*))
  (unless (entry-visited? entry)
    (setf (entry-visited? entry) t)
    (dolist (child (entry-children entry))
      (dfs child)
      (bit-ior (entry-descendants entry)
	       (entry-descendants
		 (gethash child *coherence-hash-table*)) t))))


(defun simplicity-only (as rules final-goal-datum?)
  (list (simplicity as rules final-goal-datum?)))

(defun coherence-only (as rules final-goal-datum?)
  (list (coherence as rules final-goal-datum?)))

(defun coherence-then-simplicity (as rules final-goal-datum?)
  (list (coherence as rules final-goal-datum?)
	(simplicity as rules final-goal-datum?)))


; Utility functions to check that the answers computed are correct.
; Return a list (M/T E/A), where
; M = missing assumptions,
; T = target (correct) assumptions,
; E = extra assumptions,
; A = answer (computed) assumptions.

(defun check-answer (target-assumptions answer-assumptions
		     &aux extra missing)
  (multiple-value-setq (extra missing)
    (match-assumptions target-assumptions answer-assumptions))
  (cond ((and (null extra) (null missing))
	 (format t "~%Perfect match~%"))
	(t (format t "~%Extra:")
	   (row-print-list extra nil nil nil t 3)
	   (format t "Missing:")
	   (row-print-list missing nil nil nil t 3)))
  (setf missing (remove-if #'(lambda (m) (eq (first m) 'user::precede))
			   missing))
  (list (/ (length missing) (length target-assumptions))
	(/ (length extra) (length answer-assumptions))))

(defun match-assumptions (target-assumptions answer-assumptions
			  &aux bindings b2 match extra missing)
  (setf bindings *empty-bindings*)
  (setf extra answer-assumptions)
  (dolist (tg target-assumptions)
    (do ((e extra (cdr e)))
	((null e) (push tg missing))
      (setf b2 (unify (first e) tg bindings))
      (when (and b2
		 (renaming-subst? b2 match))
	(setf bindings b2)
	(push (first e) match)
	(setf extra
	      (remove (first e) extra :test #'equal))
	(return))))
  (values (substitute-bindings extra bindings)
	  (reverse missing)))
