;;; -*- Mode:Common-Lisp; Package:AAA; Base:10 -*-

;;;; Copyright (c) 1992 by Hwee Tou Ng. This program may be freely
;;;; copied, used, or modified provided that this copyright notice 
;;;; is included in each copy of this code and parts thereof.

(in-package :aaa)

(proclaim '(optimize (speed 3) (compilation-speed 0)))

(use-package 'user)

(defvar user::*inter-batch-beam-width*)
(defvar user::*intra-batch-beam-width*)
(defvar user::*bchain-depth*)
(defvar user::*caching*)
(defvar user::*factoring*)
(defvar user::*remove-superset?*)
(defvar user::*remove-superset-fn*)
(defvar user::*explanation-eval-metric*)
(defvar user::*compute-estimate-fn*)
(defvar user::*combine-estimates-fn*)

(defvar user::*nogoods*)
(defvar user::*assumption-nogoods*)
(defvar *input-atoms*)
(defvar *start-time*)
(defvar *trace* nil)
(defvar *trace-progress* nil)
(defvar *trace-select* nil)

(defun print-envs (envs)
  (dolist (e envs)
    (print-env e)
    (format t "M = ~a~%" (environment-eval-metric e))))

(defun eq-env (e1 e2)
  (equal (environment-assumptions e1)
	 (environment-assumptions e2)))

; Add environments envs as the last environments in
; the label of node.

(defun add-environments (node envs)
  (if (null (node-label node))
      (setf (node-label node) envs)
      (nconc (node-label node) envs))
  (incf (node-label-size node) (length envs)))

(defun change-environments (node envs)
  (setf (node-label node) envs)
  (setf (node-label-size node) (length envs)))

(defun enough-environments (node)
  (>= (node-label-size node) user::*intra-batch-beam-width*))

(defun uniquify-datum+env (datum env &aux d-a-r)
  (setf d-a-r
	(uniquify (list datum
			(environment-assumptions env)
			(environment-rules env))))
  (values (first d-a-r) (second d-a-r) (third d-a-r)))


(defun init-abduce ()
  (format t "~%Running AAA...~%")
  (format t "~%Inter-observation beam width = ~d" user::*inter-batch-beam-width*)
  (format t "~%Intra-observation beam width = ~d" user::*intra-batch-beam-width*)
  (format t "~%Backward-chain depth bound = ~d" user::*bchain-depth*)
  (format t "~%Caching = ~a" user::*caching*)
  (format t "~%Factoring = ~a" user::*factoring*)
  (format t "~%Remove superset = ~a~%" user::*remove-superset?*)
  (setf *start-time* (get-internal-run-time))
  (gensym 0)
  (init-var-count)
  (init-const-count)
  (clear-cache)
  (setf *input-atoms* nil))


(defun abduce (input-atoms correct-ans &aux vars (count 0) final-goal
	       run-time best-envs recall+precision-errors)
  (init-abduce)
  (setf vars (collect-vars input-atoms))
  (dolist (atom input-atoms)
    (setf final-goal
	  (inc-abduce atom (incf count) vars)))
  (setf run-time (compute-run-time *start-time*))
  (remove-current-indices)
  (format t "~%Run Time = ~,2F min~%" run-time)
  (format t "~%Total number of explanations = ~d~%"
	  (length (node-label final-goal)))
  ;(print-envs (node-label final-goal))
  (format t "~%Best explanations:~%")
  (setf best-envs (select-best-environments (node-label final-goal)))
  (print-envs best-envs)
  (dolist (best-env best-envs)
    (push (check-answer correct-ans (environment-assumptions best-env))
	  recall+precision-errors))
  (cons run-time
	(if (null recall+precision-errors) ; no answers
	    (list 1 1)
	    (list (average (mapcar #'first recall+precision-errors))
		  (average (mapcar #'second recall+precision-errors)))))
  )

; e.g. goal-name-count = g.5
; e.g. goal-datum = (g.5 ?x ?y)
; Return the final goal node.

(defun inc-abduce (input-atom count vars
		   &aux goal-name-count goal-datum brule goal)
  (format t "~%Adding observation #~D: ~a~%" count input-atom)
  (push input-atom *input-atoms*)
  (setf goal-name-count (final-goal-name-count count))
  (setf goal-datum (final-goal-datum count vars))
  (setf brule
	(make-brule :id goal-datum
		    :conseq goal-datum
		    :antes (if (= count 1)
			       (list input-atom)
			       (list (final-goal-datum (1- count) vars) input-atom))))
  (setf (get goal-name-count 'user::brule) brule)
  (index-brule brule)
  (setf goal (compute-label goal-datum user::*bchain-depth*))
  (change-environments
    goal (truncate-envs (node-label goal)
			user::*inter-batch-beam-width* t))
  (print-run-time *start-time*)
  (when *trace-progress*
    (print-envs (node-label goal)))
  ;(print-envs (node-label goal))
  ;(print-envs (select-best-environments (node-label goal)))
  goal)

; Retrun the node correspoding to goal-datum after computing its label.
; Note: After calling the function "inconsistent?" to check the
; consistency of goal-datum, a more instantiated goal-datum may be
; returned.

(defun compute-label (goal-datum depth &optional (consistency-check t)
		      &aux goal env)
  (trace-print *trace* "~%Start computing the label of ~s~%" goal-datum)
  (when (and (or user::*caching* (final-goal-datum? goal-datum))
	     (setf goal (goal-cached goal-datum depth)))
    (return-from compute-label goal))
  (setf goal (make-node :datum goal-datum))
  (setf env (make-environment :assumptions (list goal-datum)))
  (cond
    ((and consistency-check
	  (inconsistent? env))
     (change-environments goal nil))
    ((equal (environment-subst env) *empty-bindings*)
     (resolve-with-facts goal (assume goal))
     (unless (enough-environments goal)
       (backward-chain goal depth))
     (when (and user::*remove-superset?*
		(enough-environments goal))
       (rename-environments (node-label goal))
       (change-environments
	 goal (minimal-set (node-label goal)
			   :keyfn #'environment-renamed-assumptions
			   :fn-<= user::*remove-superset-fn*))))
    (t
     (change-environments
       goal (mapcar #'(lambda (e)
			(make-environment
			  :assumptions (environment-assumptions e)
			  :rules (environment-rules e)
			  :subst (join-bindings (environment-subst env)
						(environment-subst e))))
		    (node-label
		      (compute-label
			(substitute-bindings goal-datum (environment-subst env))
			depth nil))))))
  (when (or user::*caching* (final-goal-datum? goal-datum))
    (insert-cache goal depth))
  (trace-print *trace* "~%Label of ~s computed =~%~s~%"
	       goal-datum (node-label goal))
  goal)

; If goal-datum is cached, then return a node whose label is:
; (i) the same (shared) node label if an exact match;
; (ii) an appropriately instantiated copy if goal-datum is an instance.
; If not cached, return nil.

(defun goal-cached (goal-datum depth &aux node exact-match? goal
		    subst-node-datum assumptions rules b)
  (multiple-value-setq (node exact-match?)
    (query-cache goal-datum depth))
  (when node
    (cond
      (exact-match?
       (setf goal node))
      (t
       (setf goal (make-node :datum goal-datum))
       (dolist (env (node-label node))
	 (setf subst-node-datum
	       (substitute-bindings (node-datum node) (environment-subst env)))
	 (multiple-value-setq (subst-node-datum assumptions rules)
	   (uniquify-datum+env subst-node-datum env))
	 (when (setf b (unify subst-node-datum goal-datum))
	   (add-environments
	     goal (list (make-environment
			  :assumptions (substitute-bindings assumptions b)
			  :rules (substitute-bindings rules b)
			  :subst b)))))))
    (trace-print *trace* "~%Label of ~s already cached =~%~s~%"
		 goal-datum (node-label goal))
    (return-from goal-cached goal)))

; Note: Return t if goal is assumed; otherwise return nil.

(defun assume (goal &aux datum)
  (setf datum (node-datum goal))
  (when (assumable? datum)
    (add-environments
      goal (list (make-environment :assumptions (list datum))))
    t))

; If goal-datum is an instance of a fact (i.e., goal-datum itself 
; is also a fact), then discard the abductive self-assumption, if any.
; Return goal-assumed?, which indicates whether the goal is still assumed.

(defun resolve-with-facts (goal goal-assumed? &aux goal-datum
			   f n b e)
  (setf goal-datum (node-datum goal))
  (dolist (facts (get-facts goal-datum))
    (dolist (fact facts)
      (multiple-value-setq (n f)
	(uniquify-fact fact))
      (when (setf b (unify f goal-datum))
	(when (and goal-assumed?
		   (instance-of? goal-datum f))
	  (change-environments goal (rest (node-label goal)))
	  (setf goal-assumed? nil))
	(setf e (make-environment :subst b))
	(add-environments goal (list e))
	(when (enough-environments goal)	
	  (return-from resolve-with-facts goal-assumed?)))))
  goal-assumed?)

(defun backward-chain (goal depth &aux goal-datum final-goal-datum? id conseq
		       antes b ante ante-node partial-envs)
  (if (zerop depth)
      (return-from backward-chain))
  (setf goal-datum (node-datum goal))
  (setf final-goal-datum? (final-goal-datum? goal-datum))
  (dolist (brules (get-brules goal-datum))
    (dolist (brule brules)
      (multiple-value-setq (id conseq antes) (uniquify-brule brule))
      (when (setf b (unify conseq goal-datum))
	(setf antes (substitute-bindings antes b))
	(trace-print *trace* "~%Start: ~s <- ~s~%"
		     (substitute-bindings conseq b) antes)
	; process first antecedent
	(setf ante (pop antes))
	(setf ante-node (compute-label ante (1- depth)))
	(cond
	  ((null (node-label ante-node))
	   (trace-print *trace* "~%Abort this rule~%"))
	  (t
	   (setf partial-envs
		 (mapcar #'(lambda (env)
			     (make-environment
			       :assumptions (environment-assumptions env)
			       :rules (environment-rules env)
			       :subst (join-bindings b (environment-subst env))))
			 (node-label ante-node)))
	   ; iterate over the remaining antecedents
	   (loop
	     (when (null antes)
	       ; finished processing all antecedents of this brule
	       (trace-print *trace* "~%Done.~%")
	       (dolist (e partial-envs)
		 (push (substitute-bindings id (environment-subst e))
		       (environment-rules e)))
	       (add-environments goal partial-envs)
	       (return))  ; carry on processing the next brule
	     
	     ; compute cross product of the label of the antecedents
	     ; so far and the next antecedent
	     (setf partial-envs (cross-product partial-envs (pop antes) depth))
	     (when (null partial-envs)
	       ; go on to the next brule
	       (trace-print *trace* "~%Abort this rule~%")
	       (return))
	     (when user::*remove-superset?*
	       (rename-environments partial-envs)
	       (setf partial-envs
		     (minimal-set partial-envs
				  :keyfn #'environment-renamed-assumptions
				  :fn-<= user::*remove-superset-fn*)))
	     (when (>= (length partial-envs) user::*intra-batch-beam-width*)
	       (setf partial-envs
		     (truncate-envs partial-envs user::*intra-batch-beam-width*
				    nil)))))))))
  (when (enough-environments goal)
    (change-environments
      goal (truncate-envs (node-label goal) user::*intra-batch-beam-width*
			  final-goal-datum?))))

; Return the new environments which are obtained by taking the
; cross product of partial-envs (representing the label of the
; antecedents so far) and the label of ante (the next antecedent).
; Note: partial-envs is non-nil when this function is called.

(defun cross-product (partial-envs ante depth
		      &aux old-partial-envs subst-ante subst-ante-node
		      same-instantiation new-envs)
  (setf old-partial-envs partial-envs)
  (setf partial-envs nil)
  ; check if the substituted antecedents are the same
  (setf subst-ante
	(substitute-bindings ante (environment-subst (first old-partial-envs))))
  (when (and user::*caching*
             (every #'(lambda (env)
			(equal subst-ante
			       (substitute-bindings ante (environment-subst env))))
		    (rest old-partial-envs)))
    (setf same-instantiation t)
    (setf subst-ante-node (compute-label subst-ante (1- depth)))
    (when (> (* (length old-partial-envs) (node-label-size subst-ante-node))
	     user::*intra-batch-beam-width*)
      (return-from cross-product
	(selective-cross-product old-partial-envs (node-label subst-ante-node)))))
  (dolist (env1 old-partial-envs)
    (unless same-instantiation
      (setf subst-ante (substitute-bindings ante (environment-subst env1)))
      (setf subst-ante-node (compute-label subst-ante (1- depth))))
    (dolist (env2 (node-label subst-ante-node))
      (when (setf new-envs (cprod env1 env2))
	(if (null partial-envs)
	    (setf partial-envs new-envs)
	    (nconc partial-envs new-envs)))))
  partial-envs)

; Note: est-cps is a list of elements of the form:
; (combined-estimate . (env1-pos . env2-pos))

(defun selective-cross-product (envs1 envs2
				&aux estimate est-cps new-envs
				(num-all-new-envs 0) all-new-envs)
  (mapc user::*compute-estimate-fn* envs1)
  (mapc user::*compute-estimate-fn* envs2)

  (when *trace-select*
    (format t "~%envs1:~%")
    (dolist (e envs1)
      (print-env e)
      (format t "est = ~a~%" (environment-estimate e)))
    (format t "~%envs2:~%")
    (dolist (e envs2)
      (print-env e)
      (format t "est = ~a~%" (environment-estimate e))))

  (do ((rem-envs1 envs1 (rest rem-envs1))
       (index1 0 (1+ index1)))
      ((null rem-envs1))
    (do ((rem-envs2 envs2 (rest rem-envs2))
	 (index2 0 (1+ index2)))
	((null rem-envs2))
      (setf estimate
	    (funcall user::*combine-estimates-fn*
		     (first rem-envs1) (first rem-envs2)))
      (when estimate
	(push (cons estimate (cons index1 index2))
	      est-cps))))
  (setf est-cps (sort est-cps #'lex> :key #'car))

  (when *trace-select*
    (format t "~%")
    (dolist (est-cp est-cps)
      (format t "est = ~a  elt = ~a~%" (car est-cp) (cdr est-cp))))

  (dolist (est-cp est-cps)
    (when *trace-select*
      (format t "~%Processing elt = ~a~%" (cdr est-cp)))
    (setf new-envs (cprod (elt envs1 (cadr est-cp))
			  (elt envs2 (cddr est-cp))))
    (when *trace-select*
      (format t "~%New envs = ~a~%" new-envs))
    (when new-envs
      (if (null all-new-envs)
	  (setf all-new-envs new-envs)
	  (nconc all-new-envs new-envs))
      (incf num-all-new-envs (length new-envs))
      (when (>= num-all-new-envs user::*intra-batch-beam-width*)
	(return))))
  all-new-envs)

; Take the cross product of env1 and env2.

(defun cprod (env1 env2 &aux env1-assumptions new-assumptions
	      new-rules new-subst new-env)
  (setf env1-assumptions
	(delete-duplicates
	  (substitute-bindings (environment-assumptions env1)
			       (environment-subst env2))
	  :test #'equal :from-end t))
  (setf new-assumptions
	(remove-duplicates
	  (append env1-assumptions (environment-assumptions env2))
	  :test #'equal :from-end t))
  (setf new-rules
	(nconc (substitute-bindings (environment-rules env1)
				    (environment-subst env2))
	       (environment-rules env2)))
  (setf new-subst (join-bindings (environment-subst env1)
				 (environment-subst env2)))
  (setf new-env (make-environment :assumptions new-assumptions
				  :rules new-rules
				  :subst new-subst))
  (unless (inconsistent? new-env)
    (if user::*factoring*
	(delete-duplicates
	  (factoring env1-assumptions (environment-assumptions env2) new-env)
	  :test #'eq-env)
	(list new-env))))

; Return a list of all possible factors of env (= a1s + a2s),
; by factoring an assumption of a1s with an assumption of a2s.

(defun factoring (a1s a2s env &aux factors b
		  subst-a1s subst-a2s subst-as subst-env)
  (setf factors (list env))
  (dolist (a1 a1s factors)
    (dolist (a2 a2s)
      (when (and (eq (predicate a1) (predicate a2))
		 (setf b (unify a1 a2))
		 (not (equal b *empty-bindings*)))
	(setf subst-a1s
	      (delete-duplicates (substitute-bindings a1s b)
				 :test #'equal :from-end t))
	(setf subst-a2s
	      (delete-duplicates (substitute-bindings a2s b)
				 :test #'equal :from-end t))
	(setf subst-as
	      (remove-duplicates (append subst-a1s subst-a2s)
				 :test #'equal :from-end t))
	(setf subst-env
	      (make-environment
		:assumptions subst-as
		:rules (substitute-bindings (environment-rules env) b)
		:subst (join-bindings (environment-subst env) b)))
	(unless (inconsistent? subst-env)
	  (nconc factors (factoring subst-a1s subst-a2s subst-env)))))))

(defun truncate-envs (envs bw final-goal-datum?)
  (compute-eval-metric envs final-goal-datum?)
  (setf envs (sort-environments envs))
  (first-n-destr envs bw))

; Compute the evaluation metric for each environment.

(defun compute-eval-metric (envs final-goal-datum?)
  (dolist (env envs)
    (setf (environment-eval-metric env)
	  (funcall user::*explanation-eval-metric*
		   (environment-assumptions env)
		   (environment-rules env)
		   final-goal-datum?))))

; Sort the environments according to the evaluation metric.

(defun sort-environments (envs)
  (sort envs #'lex> :key #'(lambda (e) (environment-eval-metric e))))

; Assume envs is sorted according to the evaluation metric.

(defun select-best-environments (envs &aux best-metric)
  (when envs
    (setf best-metric (environment-eval-metric (first envs)))
    (do ((rem-envs (rest envs))
	 (best-envs (list (first envs))))
	((or (null rem-envs)
	     (not (equal-eval-metric 
		    (environment-eval-metric (first rem-envs))
		    best-metric)))
	 (nreverse best-envs))
      (push (pop rem-envs) best-envs))))

; Return t iff m1 and m2 are the same evaluation metric.

(defun equal-eval-metric (m1 m2)
  (do ((rem-m1 m1 (rest rem-m1))
       (rem-m2 m2 (rest rem-m2)))
      ((null rem-m1) t)
    (if (/= (first rem-m1) (first rem-m2))
	(return nil))))

; Uniquify the variables in environments.

(defun rename-environments (envs)
  (dolist (env envs)
    (when (eq (environment-renamed-assumptions env) :not-renamed)
      (setf (environment-renamed-assumptions env)
	    (uniquify (environment-assumptions env))))))
