%{
/*	This file is part of the software similarity tester SIM.
	Written by Dick Grune, Vrije Universiteit, Amsterdam.
	$Header: clang.l,v 2.2 91/09/17 21:32:38 dick Exp $
*/

/*
	C language front end for the similarity tester.
	Author:	Dick Grune <dick@cs.vu.nl>
*/

#include	"cport.h"
#include	"private.h"
#include	"token.h"
#include	"idf.h"
#include	"stream.h"
#include	"lex.h"

/* Language-dependent Code */

extern char options[];

/* Data for module idf */

CONST PRIVATE struct idf ppcmd[] = {
	"define",	META('d'),
	"else",		META('e'),
	"endif",	META('E'),
	"if",		META('i'),
	"ifdef",	META('I'),
	"ifndef",	META('x'),
	"include",	MTCT('I'),
	"line",		META('l'),
	"undef",	META('u')
};

CONST PRIVATE struct idf reserved[] = {
	"auto",		NORM('a'),
	"break",	NORM('b'),
	"case",		NORM('c'),
	"char",		NORM('C'),
	"continue",	CTRL('C'),
	"default",	NORM('d'),
	"do",		NORM('D'),
	"double",	CTRL('D'),
	"else",		NORM('e'),
	"enum",		NORM('E'),
	"extern",	CTRL('E'),
	"float",	NORM('f'),
	"for",		NORM('F'),
	"goto",		NORM('g'),
	"if",		NORM('i'),
	"int",		NORM('I'),
	"long",		NORM('l'),
	"register",	SKIP,
	"return",	NORM('r'),
	"short",	NORM('s'),
	"sizeof",	NORM('S'),
	"static",	CTRL('S'),
	"struct",	META('s'),
	"switch",	META('S'),
	"typedef",	NORM('t'),
	"union",	NORM('u'),
	"unsigned",	NORM('U'),
	"void",		SKIP,
	"while",	NORM('w')
};

/* Special treatment of identifiers */

PRIVATE TOKEN
idf2token(hashing)
	int hashing;
{
	register TOKEN tk;
	
	tk = idf_in_list(yytext, reserved, IDF);	
	if (TOKEN_EQ(tk, IDF) && hashing) {
		/* return a one-token hash code */
		tk = idf_hashed(yytext);
	}
	return tk;
}

/* Token sets for module algollike */
CONST TOKEN Headers[] = {
	IDF,		/* identifier */
	NORM('{'),
	NORM('('),
	NORM('a'),	/* auto */
	NORM('b'),	/* break */
	NORM('c'),	/* case */
	NORM('C'),	/* char */
	CTRL('C'),	/* continue */
	NORM('d'),	/* default */
	NORM('D'),	/* do */
	CTRL('D'),	/* double */
	NORM('E'),	/* enum */
	CTRL('E'),	/* extern */
	NORM('f'),	/* float */
	NORM('F'),	/* for */
	NORM('g'),	/* goto */
	NORM('i'),	/* if */
	NORM('I'),	/* int */
	NORM('l'),	/* long */
	NORM('r'),	/* return */
	NORM('s'),	/* short */
	CTRL('S'),	/* static */
	META('s'),	/* struct */
	META('S'),	/* switch */
	NORM('t'),	/* typedef */
	NORM('u'),	/* union */
	NORM('U'),	/* unsigned */
	NORM('w'),	/* while */
	NOTOKEN
};
CONST TOKEN Trailers[] = {
	NORM(')'),
	NORM('}'),
	NORM(';'),
	NOTOKEN
};
CONST TOKEN Openers[] = {
	NORM('{'),
	NORM('('),
	NORM('['),
	NOTOKEN
};
CONST TOKEN Closers[] = {
	NORM('}'),
	NORM(')'),
	NORM(']'),
	NOTOKEN
};

/* Language-INdependent Code */

yystart() {
	BEGIN INITIAL;
}

PRIVATE int
yywrap() {
	return 1;
}

%}

%Start	Comment

Layout		([ \t])

AnyQuoted	(\\.)
StrChar		([^"\n\\]|{AnyQuoted})
ChrChar		([^'\n\\]|{AnyQuoted})

StartComment	("/*")
EndComment	("*/")
SafeComChar	([^*\n])
UnsafeComChar	("*")

Digit		([0-9a-fA-F])
Idf		([A-Za-z][A-Za-z0-9_]*)

%%

{StartComment}	{
		/*	We do not have one single pattern to match a comment
			(although one can be written), for two reasons.
			The matched string might overflow lex-internal buffers
			like yysbuf and yytext; and the pattern would be very
			complicated and overtax lex.
			So we break up the string into safe chunks and keep
			track of where we are in a start condition <Comment>.
		*/
		BEGIN Comment;
	}

<Comment>{SafeComChar}+	{		/* safe comment chunk */
	}

<Comment>{UnsafeComChar}	{	/* unsafe char, read one by one */
	}

<Comment>"\n"		{		/* to break up long comments */
		return_eol();
	}

<Comment>{EndComment}	{		/* end-of-comment */
		BEGIN INITIAL;
	}

\"{StrChar}*\"	{			/* strings */
		return_ch('"');
	}

\'{ChrChar}+\'	{			/* characters */
		return_ch('\'');
	}

#{Layout}*include.*	{		/* skip #include line */
	}

#{Layout}*{Idf}	{			/* a preprocessor line */
		register char *n = yytext+1;

		/* skip layout in front of preprocessor identifier */
		while (*n == ' ' || *n == '\t') {
			n++;
		}
		return_tk(idf_in_list(n, ppcmd, NORM('#')));
	}

(0x)?{Digit}+("l"|"L")?	{		/* numeral, passed as an identifier */
		return_tk(IDF);
	}

{Idf}/"("	{			/* identifier in front of ( */
		register TOKEN tk;

		tk = idf2token(options['F']);
		if (!TOKEN_EQ(tk, SKIP)) return_tk(tk);
	}

{Idf}	{				/* identifier */
		register TOKEN tk;

		tk = idf2token(0 /* no hashing */);
		if (!TOKEN_EQ(tk, SKIP)) return_tk(tk);
	}

\;	{				/* semicolon, conditionally ignored */
		if (options['f']) return_ch(yytext[0]);
	}

{Layout}	{			/* ignore layout */
	}

\n	{				/* count newlines */
		return_eol();
	}

.	{				/* copy other text */
		if (!(yytext[0]&0200)) return_ch(yytext[0]);
	}

%%
