; dataFlow.scm; simple data driven data flow executive.  

;================================================================
;| Brian Beckman                  | brian@topaz.jpl.nasa.gov    |
;| Computer Graphics Laboratory   | (818) 397-9207              |
;| Mail Stop 510-202              | (818) 397-9344              |
;| Jet Propulsion Laboratory      |                             |
;| Pasadena, CA 91109             | 6 July 1989                 |
;================================================================

(load "methods.scm")
(load "queues.scm")

; This file contains a simple data-driven data flow executive, some 
; basic primitive data flow node classes, and an extended example.
; The example is a data flow program for a quadratic equation solver
; for quadratic equations with real roots.
;
; The procedures in this file make up a small data flow language. This
; language has the following structure.  A data flow ``graph'' is a
; list of ``nodes''. A ``node'' is either a ``primitive node'' or a
; graph.  Thus, a graph may contain embedded or nested sub-graphs.
; Every node has zero or more ``input arcs'' and zero or more ``output
; arcs''.  A node operates by consuming tokens from its input arcs,
; performing an operation on the tokens, and appending tokens to its
; output arcs.  Primitive arcs differ from graphs in how the operation
; is implemented.  The operation of a primitive node is represented by
; a procedure that takes K arguments and produces a LIST of L output
; values, where K is the number of input arcs and L is the number of
; output arcs of the node.  The operation of a graph is represented by
; a number of internal nodes and arcs that together perform a
; composite action.  The input arcs and output arcs of a graph are
; those with only one end connected to a node belonging to the graph.
; 
; Arcs are FIFO queues.  They may be represented graphically as arrows.
; The node at the tail of an arc appends tokens to the tail of the 
; queue, and the node at the head of an arc consumes tokens from the
; head of the queue.  A node is ``ready to fire'' when all of its
; input arcs are non-empty.  When a node fires, exactly one token is
; taken from each input arc and exactly one token is pushed into each
; output arc.
; 
; Some nodes types have no input arcs.  They are ``data sources'', and
; they are always ready to fire, by convention.  Other node types have
; no output arcs.  They are called ``data sinks''. 
; 
; Every node has methods for the following messages.
; 'ready?  -- return #t if the node is ready to fire.
; 'fire!   -- fire the node, modifying the input and output arcs
; 'print   -- print a list of the front tokens in each input arc
; 'clear!  -- clear all input and output arcs.
;
; This file contains the following procedures, most of which 
; represent node classes:
;   new-pnode . . . . . . . create a primitive node
;   new-dfgraph . . . . . . create a data flow graph
;   execute-data-flow . . . execute a data flow graph until done
;   new-add-node. . . . . . create an adder
;   new-mul-node. . . . . . create a multiplier
;   new-div-node. . . . . . create a divider
;   new-negate-node . . . . create a unary minus operator
;   new-sub-node. . . . . . create a subtracter
;   new-dup-node. . . . . . create a duplicator
;   new-sqr-node. . . . . . create a squaring node
;   new-const-node. . . . . create a constant producer
;   new-sqrt-node . . . . . create a square root-taking node
;   new-disc-node . . . . . create a discriminant calculator
;   new-dbl-node. . . . . . create a doubler
;   new-quadratic-solver. . create a real root finder for quadratics
;
; 
; Create a primitive node.
; 
;;;------------------------------'-----------------------------;;;
;;;                      N E W - P N O D E                     ;;;
;;;------------------------------.-----------------------------;;;

(define (new-pnode proc input-queues output-queues)
  
  (let* (  (supers ())  ) ;;; a root class
    
    (define (all-qs-non-empty? qlist)
      (cond
       (  (null? qlist)  #t  ) 
       ;;; if there are no queues, then all queues are non-empty, 
       ;;; by convention
       (  else  (and (not (send (car qlist) 'empty?))
                     (all-qs-non-empty? (cdr qlist)))  )))
    
    (define (ready-to-fire?)
      (all-qs-non-empty? input-queues))
    
    (define (fire-node!)
      (let* (  (input-list 
                (map (lambda (q) (send q 'front!)) input-queues))
               (output-list (apply proc input-list))  )
        
        (define (iter output-items output-qs)
          (cond
           (  (null? output-qs)     self  )  ;;; return self for method
           (  (null? output-items)  self  )  ;;; composition purposes
           (  else  (let (  (o (car output-items))
                            (q (car output-qs))  )
                      (send q 'append! o)
                      (iter (cdr output-items) (cdr output-qs)))  )))
        
        (iter output-list output-queues)))

    (define (print)
      (define (front q)
        (if (send q 'empty?) "Empty" (send q 'front)))
      (display
       (map front input-queues))
      (newline)
      self)
    
    (define (clear!)
      (map (lambda (q) (send q 'clear!)) input-queues)
      (map (lambda (q) (send q 'clear!)) output-queues)
      self)
    
    (define (self msg)
      (cond
       (  (eq? msg 'ready?)  ready-to-fire?  )
       (  (eq? msg 'fire!)   fire-node!  )
       (  (eq? msg 'print)   print  )
       (  (eq? msg 'clear!)  clear!  )
       (  (search-supertypes supers msg)  )
       (  else  (make-error-method "Data-flow-node" msg)  )))
    
    self))

; Create a data flow graph given a list of nodes, a list of 
; input arcs, and a list of output arcs.  These arcs are the
; ones connected at exactly one end to nodes in the graph.
; 
;;;------------------------------'-----------------------------;;;
;;;                    N E W - D F G R A P H                   ;;;
;;;------------------------------.-----------------------------;;;

(define (new-dfgraph graph input-queues output-queues)
  
  (let* (  (supers ())  )
    
    (define (all-qs-non-empty? qlist)
      (cond
       (  (null? qlist)  #t  )
       (  else  (and (not (send (car qlist) 'empty?))
                     (all-qs-non-empty? (cdr qlist)))  )))
    
    (define (ready-to-fire?)
      (all-qs-non-empty? input-queues))
    
    (define (print graph)
      (cond
       (  (null? graph)  self  )
       (  else  (send (car graph) 'print)
                (print (cdr graph)))))
    
    (define (clear! graph)
      (cond
       (  (null? graph)  self  )
       (  else  (send (car graph) 'clear!)
                (clear! (cdr graph)))))
    
    (define (self msg)
      (cond
       (  (eq? msg 'ready?)  ready-to-fire?  )
       (  (eq? msg 'fire!)   (lambda () (execute-data-flow graph))  )
       (  (eq? msg 'print)   (lambda () (print graph))  )
       (  (eq? msg 'clear!)  (lambda () (clear! graph)) )
       (  (search-supertypes supers msg)  )
       (  else  (make-error-method "DFGraph" msg)  )))
    
    self))

; This is the data flow executve.  It takes a
; graph as input, which is a list of nodes.  It loops over the 
; list until nothing more is ready.  The nodes, which are
; connected by shared arcs, must form a directed acyclic graph.
; 
;;;------------------------------'-----------------------------;;;
;;;              E X E C U T E - D A T A - F L O W             ;;;
;;;------------------------------.-----------------------------;;;

(define (execute-data-flow graph)
  
  (define (one-pass graph)
    (cond
     (  (null? graph)  #f  ) ;;; so far, nothing is ready.
     
     (  (send (car graph) 'ready?)  (send (car graph) 'fire!)
                                    (or #t (one-pass (cdr graph)))  )
     
        ;;; at least one node was ready, so we OR in #t.
     
     (  else (one-pass (cdr graph))  )))
  
  (define (iter) ;;; loop until nothing is ready.
    (cond
     (  (not (one-pass graph))  'done  )
     (  else  (iter)  )))
  
  (iter))

; The first primitive node is the ``add-node''.  An add-node
; adds the tokens in its input arcs.  It has two input arcs and one
; output arc.  Since the add operation is inherently commutative,
; the behavior of an add-node is insensitive to the order of its
; input arcs.  That is, data coming down the left-input is 
; treated the same as data coming down the right-input by this
; type of node.
; 
;;;------------------------------'-----------------------------;;;
;;;                   N E W - A D D - N O D E                  ;;;
;;;------------------------------.-----------------------------;;;

(define (new-add-node left-input right-input output)
  
    (let* ((supers (list
                      (new-pnode
                       (lambda (a b) (list (+ a b)))
                       (list left-input right-input)
                       (list output)))))

    (define (self msg)
      (cond
       (  (search-supertypes supers msg)  )
       (  else  (make-error-method "Add-node" msg)  )))
    
    self))

;;; Test suite for add-nodes and execute-data-flow 

(define arc1 (new-queue '(10 20 30 40)))

(define arc2 (new-queue '(5 5 5)))

(define o-arc (new-queue))

(define an-add-node (new-add-node arc1 arc2 o-arc))

(define (print-all)
  (send arc1 'print)
  (send arc2 'print)
  (send o-arc 'print))

(print-all)
;(10 20 30 40)
;(5 5 5)
;()

(execute-data-flow (list an-add-node))
;'done

(print-all)
;(40)
;()
;(15 25 35)
; 
; Data has flowed out of the input queues and into the 
; output queue.  One input token is left behind because it had no
; positionally corresponding token in the other input queue.

;;; End of test suite

; The next pnode is a multiplier, another commutative operator.
; 
;;;------------------------------'-----------------------------;;;
;;;                    N E W - M U L - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-mul-node left-input right-input output)
  
  (let* ((supers (list
                  (new-pnode
                   (lambda (a b) (list (* a b)))
                   (list left-input right-input)
                   (list output)))))
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Mul-node" msg)  )))
    
    self))

; The next pnode is a divider, where the dividend comes in the left
; arc and the divisor comes in the right arc.  
; 
;;;------------------------------'-----------------------------;;;
;;;                    N E W - D I V - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-div-node left-input right-input output)
  
  (let* ((supers (list
                  (new-pnode
                   (lambda (a b) (list (/ a b)))
                   (list left-input right-input)
                   (list output)))))
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Div-node" msg)  )))
    
    self))

; The next pnode is unary minus operator.
; 
;;;------------------------------'-----------------------------;;;
;;;                N E W - N E G A T E - N O D E               ;;;
;;;------------------------------.-----------------------------;;;

(define (new-negate-node input output)
  
  (let* ((negate-proc (lambda (in) (list (- in))))
         (supers (list
                  (new-pnode
                   negate-proc
                   (list input)
                   (list output)))))
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Negate-node" msg)  )))
    
    self))

; The subtraction node class is defined as a dfgraph
; composing a negation and an addition.
; 
;;;------------------------------'-----------------------------;;;
;;;                    N E W - S U B - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-sub-node left-input right-input output)   ;;; subtraction
  
  (let* (  (AQ (new-queue))
           (N  (new-negate-node right-input AQ))
           (P  (new-add-node left-input AQ output))
           (supers (list (new-dfgraph
                          (list n p)
                          (list left-input right-input)
                          (list output))))  )
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Sub-node" msg)  )))
    
    self))

; We need a duplicator node class, for replicating flows.
; 
;;;------------------------------'-----------------------------;;;
;;;                    N E W - D U P - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-dup-node input left-output right-output)

  (let* ((supers (list
                  (new-pnode
                   (lambda (in) (list in in))
                   (list input)
                   (list left-output right-output)))))
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Dup-node" msg)  )))
    
    self))

; The square "node" is a dfgraph comprising a dup and multiply.
; 
;;;------------------------------'-----------------------------;;;
;;;                    N E W - S Q R - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-sqr-node in out)
  
  (let* (  (AQ (new-queue))
           (BQ (new-queue))
           (D  (new-dup-node in AQ BQ))
           (M  (new-mul-node AQ BQ out))
           (supers (list (new-dfgraph
                          (list D M)
                          (list in)
                          (list out))))  )
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Sqr-node" msg)  )))
    
    self))

; We need a node to produce a constant value when it has anything 
; whatever in its input queue.  
; 
;;;------------------------------'-----------------------------;;;
;;;                N E W - C O N S T - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-const-node constant inq outq)
  
  (let* (  (supers (list
                    (new-pnode
                     (lambda (x) (list constant))
                     (list inq)
                     (list outq))))  )
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Const-node" msg)  )))
    
    self))

; A sqrt node is needed that returns both branches of the square root
; function (real values only).  This node will blow up on input of
; a negative number.  
; 
;;;------------------------------'-----------------------------;;;
;;;                  N E W - S Q R T - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-sqrt-node input out1 out2)
  
  (let* (  (supers (list
                    (new-pnode
                     (lambda (x) (list (sqrt x) (- (sqrt x))))
                     (list input)
                     (list out1 out2))))  )
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Sqrt-node" msg)  )))
    
    self))

; Here's the graph for the quadratic equation solution
;
;              ----------
;        +    /  2
;    -b  -  \/  b  - 4ac
;  --------------------------
;           2 a
;
; 
; The graph is somewhat complex, and it is really quite
; impossible to write this code without drawing the graph first.  
; It is also impossible to paste a picture of the graph in
; these comments, so the reader will have to reconstruct the graph
; from the written description.  It is a little tedious, though
; not difficult to do so.
; 
; First, a pair of helper node classes, ``disc'' to evaluate
; discriminants, and ``dbl'' to double its input.  These classes
; will encapsulate the constants 4 and 2.
; 
;;;------------------------------'-----------------------------;;;
;;;                  N E W - D I S C - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-disc-node A B C D)
  
  (let* ((Z (new-queue))
         (Y (new-queue))
         (D1 (new-dup-node A Z Y))
         (X (new-queue))
         (M1 (new-mul-node Y C X))
         (W (new-queue))
         (C1 (new-const-node 4 Z W))
         (V (new-queue))
         (S1 (new-sqr-node B V))
         (U (new-queue))
         (M2 (new-mul-node W X U))
         (SUB1 (new-sub-node V U D))
         (supers (list
                    (new-dfgraph
                     (list S1  D1  M1  C1  M2  SUB1)
                     (list A   B   C)
                     (list D)))))
    
    (define (self msg)
      (cond
       (  (search-supertypes supers msg)  )
       (  else  (make-error-method "Disc-node" msg)  )))
    
    self))

;;;------------------------------'-----------------------------;;;
;;;                    N E W - D B L - N O D E                 ;;;
;;;------------------------------.-----------------------------;;;

(define (new-dbl-node A D)
  
  (let* ((Z (new-queue))
         (Y (new-queue))
         (D1 (new-dup-node A Y Z))
         (X (new-queue))
         (C1 (new-const-node 2 Z X))
         (M1 (new-mul-node Y X D))
         (supers (list
                  (new-dfgraph
                   (list  D1  C1  M1)
                   (list  A)
                   (list  D)))))
    
    (define (self msg)
      (cond
       (  (search-supertypes supers msg)  )
       (  else  (make-error-method "Dbl-node" msg)  )))
    
    self))

;;;------------------------------'-----------------------------;;;
;;;          N E W - Q U A D R A T I C - S O L V E R           ;;;
;;;------------------------------.-----------------------------;;;

(define (new-quadratic-solver A B C rroot1 rroot2)
  
  (let* ((Z (new-queue))
         (Y (new-queue))
         (D1 (new-dup-node A Z Y))
         (X (new-queue))
         (W (new-queue))
         (D2 (new-dup-node B X W))
         (V (new-queue))
         (DBL (new-dbl-node Z V))
         (U (new-queue))
         (DISC (new-disc-node Y X C U))
         (T (new-queue))
         (N1 (new-negate-node W T))
         (S (new-queue))
         (R (new-queue))
         (D4 (new-dup-node V S R))
         (Q (new-queue))
         (P (new-queue))
         (S1 (new-sqrt-node U Q P))
         (O (new-queue))
         (N (new-queue))
         (D3 (new-dup-node T O N))
         (M (new-queue))
         (P1 (new-add-node Q O M))
         (L (new-queue))
         (P2 (new-add-node P N L))
         (/1 (new-div-node M S RROOT1))
         (/2 (new-div-node L R RROOT2))
         (supers
          (list
           (new-dfgraph
            (list  D1   D2   N1   DISC
                   DBL  D3   S1   D4 
                   P1   P2   /1   /2)
            (list  A    B    C)
            (list  RROOT1    RROOT2)))))
    
    (define (self msg)
      (cond
        (  (search-supertypes supers msg)  )
        (  else  (make-error-method "Quadratic-solver" msg)  )))
    
    self))

; Whew!  Now, lets try it.  

(define a (new-queue '( 8  2  3)))
(define b (new-queue '(10 -7 -9)))
(define c (new-queue '( 2  5  4)))
(define r1 (new-queue))
(define r2 (new-queue))
(define (print)
  (send a 'print)
  (send b 'print)
  (send c 'print)
  (send r1 'print)
  (send r2 'print))

(define qs (new-quadratic-solver a b c r1 r2))

(print)
(execute-data-flow (list qs))
(print)
