(herald spis)

(define *offset-from-template* 10)

(define sparc/bcc
  (object (lambda (bv i cc disp)
	    (let ((displ (branch-target-offset i disp)))
	      (if (neq? cc jump-op/jl)
		  (branch-type bv i (cc->code cc) #b010 displ)
		  (call-type bv i displ))))
    ((instruction-as-string self i cc disp)
     (if (neq? cc jump-op/jl)
	 (format nil "b~a ~a" (cc->string cc)
		 (fx+ i (fixnum-ashl (branch-target-offset i disp) 2)))
	 (format nil "call ~a"
		 (fx+ i (fixnum-ashl (branch-target-offset i disp) 2)))))
    ((identification self) "bcc")))

(define jbr-a-inst
  (object (lambda (bv i cc disp)
	    (let ((displ (branch-target-offset i disp)))
	      (branch-a-type bv i (cc->code cc) #b010 displ)))
    ((instruction-as-string self i cc disp)
     (format nil "b~a,a ~a" (cc->string cc)
	     (fx+ i (fixnum-ashl (branch-target-offset i disp) 2))))
    ((identification self) "bcc")))


(define (cc->string jump-op)
  (cond ((fx>= jump-op 0)
	 (vref '#("a" "ne" "gtz" "gez" "gu" "cc" "pos" "vc") jump-op))
	(else
	 (vref '#("a" "e" "lez" "ltz" "leu" "cs" "neg" "vs") (fx- 0 jump-op)))))

(define (cc->code jump-op)
  (cond ((fx>= jump-op 0)
	 (vref '#(8 9 10 11 12 13 14 15) jump-op))
	(else
	 (vref '#(8 1 2 3 4 5 6 7) (fx- 0 jump-op)))))

(define (branch-target-offset pc thing)
  (cond ((fixnum? thing) (fixnum-ashr thing  2))
	(else
	 (let ((addr (address-of (cdr thing))))
	   (fixnum-ashr (fx- (xcase (car thing)
				    ((label) addr)
				    ((template) (fx+ addr 12))
				    ((label+1) (fx+ addr 4)))
				  pc) 2)))))
	
	
(define (normal-3op name op3)
  (object (lambda (bv i s1 s2 d)
	    (cond ((atom? s1)
		   (3-reg-format bv i op3 (rnum s2) (rnum s1) (rnum d)))
		  (else
		   (imm-format bv i op3 (rnum s2) (rnum d) (get-literal i s1)))))
	  ((instruction-as-string self i s1 s2 d)
	    (cond ((atom? s1)
		   (format nil "~a ~a,~a,~a" name (rname s1) (rname s2)
			   (rname d)))
		  (else
		   (format nil "~a $~d,~a,~a" name (get-literal i s1)
			   (rname s2) (rname d)))))
	  ((identification self) name)))

(define risc/add (normal-3op "add" #b010000))
(define risc/sub (normal-3op "sub" #b010100))
(define risc/or (normal-3op "or"   #b010010))
(define risc/and (normal-3op "and" #b010001))
(define risc/xor (normal-3op "xor" #b010011))
(define risc/sra (normal-3op "sra" #b100111))
(define risc/srl (normal-3op "srl" #b100110))
(define risc/sll (normal-3op "sll" #b100101))
(define sparc/save (normal-3op "save" #b111100))
(define sparc/restore (normal-3op "restore" #b111101))
(define sparc/iflush (normal-3op "iflush" #b111011))

(define sparc/jmpl
  (object (lambda (bv i ro d)
	    (cond ((eq? (car ro) 'reg-reg)
		   (3-reg-format bv i #b111000 (rnum (cadr ro))
				 (rnum (caddr ro)) (rnum d)))
		  (else
		   (receive (base offset) (get-reg-and-offset ro)
			    (imm-format bv i #b111000 (rnum base) (rnum d) offset)))))
	  ((instruction-as-string self i ro d)
	   (cond ((eq? (car ro) 'reg-reg)
		  (format nil "jmpl (~a:~a),~a"
			  (rname (cadr ro))
			  (rname (caddr ro)) (rname d)))
		 (else
		  (receive (base offset) (get-reg-and-offset ro)	   
			   (format nil "jmpl ~d(~a),~a"  offset
				   (rname base) (rname d))))))
	  ((identification self) "jmpl")))

(define risc/load
  (object (lambda (bv i size ro d)
	    (receive (base offset) (get-reg-and-offset ro)
	      (load-store-type bv i (load-op size) (rnum base) (rnum d) offset)))
	  ((instruction-as-string self i size ro d)
	   (receive (base offset) (get-reg-and-offset ro)	   
             (format nil "~a ~d(~a),~a" (load-op-name size) offset
		     (rname base) (rname d))))
	  ((identification self) "load")))	  

(define (load-op size)
  (xcase size
    ((l) #b000000)
    ((uw) #b000010)
    ((sw) #b001010)
    ((ub) #b000001)
    ((sb) #b001001)
    ((d) #b000011)))
(define (load-op-name size)
  (xcase size
    ((l) "ld")
    ((uw) "lduh")
    ((sw) "ldsh")
    ((ub) "ldub")
    ((sb) "ldsb")
    ((d) "ldd")))

(define risc/store
  (object (lambda (bv i size d ro)
	    (receive (base offset) (get-reg-and-offset ro)
	      (load-store-type bv i (store-op size) (rnum base) (rnum d) offset)))
	  ((instruction-as-string self i size d ro)
	   (receive (base offset) (get-reg-and-offset ro)	   
             (format nil "~a ~a,~d(~a)" (store-op-name size) (rname d)
		     offset (rname base))))
	  ((identification self) "store")))

(define (store-op size)
  (xcase size
    ((l) #b000100)
    ((w) #b000110)
    ((b) #b000101)
    ((d) #b000111)))
(define (store-op-name size)
  (xcase size
    ((L) "st")
    ((w) "sth")
    ((b) "stb")
    ((d) "std")))


(define sparc/fload
  (object (lambda (bv i ro d)
	    (receive (base offset) (get-reg-and-offset ro)
	      (load-store-type bv i #b100000 (rnum base) d offset)))
	  ((instruction-as-string self i ro d)
	   (receive (base offset) (get-reg-and-offset ro)	   
             (format nil "ldf ~d(~a),$f~a"  offset
		     (rname base) d)))
	  ((identification self) "fload")))

(define sparc/fstore
  (object (lambda (bv i d ro)
	    (receive (base offset) (get-reg-and-offset ro)
	      (load-store-type bv i #b100100 (rnum base) d offset)))
	  ((instruction-as-string self i d ro)
	   (receive (base offset) (get-reg-and-offset ro)	   
             (format nil "stf $f~a,~d(~a)" d
		     offset (rname base))))
	  ((identification self) "fstore")))

(define-constant (13bit? x)
  (and (fx<= #x-1000 x) (fx< x #x1000)))

(define-constant (u13bit? x)
  (and (fx>= x 0) (fx<= x #x1fff)))

(define (get-reg-and-offset ro)
  (xcase (car ro)
    ((reg-offset) (return (cadr ro) (enforce 13bit? (caddr ro))))))

(define (get-literal i lit)
  (xcase (car lit)
    ((unsigned)
     (enforce u13bit? (cdr lit)))
    ((tp-offset)
     (fixnum-logand #x3ff		;low 10 bits!
      (fx- (fx+ (ib-address (cdr lit)) 10) (fx- i 8)))) ;second instruction
    ((handler-diff)
     (fixnum-logand #x3ff (fx- (fx+ (ib-address (cadr lit)) 12)
			       (ib-address (cddr lit)))))
    ((lit) (enforce 13bit? (cdr lit)))
    ((label-offset)
     (enforce 13bit? (fx- (ib-address (cdr lit)) (fx- i 4))))))

(define (get-high i lit)
  (xcase (car lit)
    ((unsigned) (cdr lit))
    ((tp-offset)
     (fixnum-ashr (fx- (fx+ (ib-address (cdr lit)) 10) ;high 22 bits!
		       (fx- i 4)) 10)) ;first instruction
    ((handler-diff)
     (fixnum-ashr (fx- (fx+ (ib-address (cadr lit)) 12)
		       (ib-address (cddr lit))) 10))))



(define sparc/sethi
  (object (lambda (bv i lit reg)
	    (branch-type bv i  (rnum reg) #b100 (get-high i lit)))
    ((instruction-as-string self i lit reg)
     (format nil "sethi $~x,~a" (get-high i lit) (rname reg)))
    ((identification self) "sethi")))

(define sparc/noop
  (object (lambda (bv i)
	    (branch-type bv i 0 #b100 0))
    ((instruction-as-string self i)
     "noop")))

#|  
(define (rnum r)
  (cond ((not (fixnum? r))
	 (cond ((assq r native-registers) => cdr)
	       (else (bug "bad native register ~s" r))))
	((fx< r 0)
	 (vref '#(nil 0 10 11 12 7 24 13 14 15) (fx- 0 r)))
	((fx< r *real-registers*)
	 (vref '#(1 2 3 4 5 6 8 9 16 17 18 19 20 21 22 23 25 26 27 28 29) r))))
;;;		 g1g2g3g4g5g6o0o1 l0 l1 l2 l3 l4 l5 l6 l7 i1 i2 i3 i4 i5 

(define *reg-names* (make-vector *real-registers*))
(set (vref *reg-names* 0) "p")
(do ((i 1 (fx+ i 1)))
    ((fx= i AN)
     (set (vref *reg-names* AN) "an")
     (set (vref *reg-names* AN+1) "an+1")
     (do ((i 0 (fx+ i 1)))
	 ((fx= i *stack-registers*))
       (set (vref *reg-names* (fx+ i S0)) (format nil "s~d" i))))
    (set (vref *reg-names* i)
       (format nil "a~d" i)))

(define (rname r)
  (cond ((not (fixnum? r))
	 (cond ((assq r native-registers) => car)
	       (else (bug "bad native register ~s" r))))
	((fx>= r 0)
	 (vref *reg-names* r))
	(else
	 (vref '#("nil" "zero" "eargs" "extra" "scratch" "nil" "pex" "vector"
			"sp" "link")
	       (fx- 0 r)))))
|#

(define (rnum r)
  (cond ((fx>= r 0)
	 (if (fx= r an+1) 3 (fx+ r 16)))
	(else
	 (vref '#(nil 0 9  10 13 29 11 12  2  1 15  8 31 14) (fx- 0 r)))))
;;;                  g0 o1 o2 o5 i5 o3 o4 g2 g1 o7 o0 i7 o6

(define *reg-names* (make-vector *real-registers*))
(set (vref *reg-names* 0) "p")
(do ((i 1 (fx+ i 1)))
    ((fx= i AN)
     (set (vref *reg-names* AN) "an")
     (set (vref *reg-names* AN+1) "an+1"))
  (set (vref *reg-names* i)
       (format nil "a~d" i)))

(define (rname r)
  (cond ((fx>= r 0)
	 (vref *reg-names* r))
	(else
	 (vref '#("nil" "zero" "eargs" "extra" "scratch" "nil" "pex" "vector"
			"t" "sp" "link" "ass" "crit" "ssp" "%fp" "%o0")
	       (fx- 0 r)))))

(define lap-table (make-table 'lap-table))
(define (define-lap x y)
  (set (table-entry lap-table x) y))


	 
(define jbr-inst sparc/bcc)
(define noop-inst `(,sparc/noop))


(define (set-16 bv i val)
  (set (bref bv (fx+ i 1)) (fixnum-logand #xff val))
  (set (bref bv i) (fixnum-logand #xff (fixnum-ashr val 8))))

(define (set-24 bv i val)
  (set (bref bv (fx+ i 2)) (fixnum-logand #xff val))
  (set (bref bv (fx+ i 1)) (fixnum-logand #xff (fixnum-ashr val 8)))
  (set (bref bv i) (fixnum-logand #xff (fixnum-ashr val 16))))



;| annotation offsetSHI | handler offset |
;|           code vector offset          |    
;|      pointer       | nargs |?template | 

(define (template1 bv i l h)
  (set-16 bv (fx+ i 2)
       (if h
	   (fx- (address-of h) (fx+ i 10)) ;this template is at i+10
	   0))
  (set-16 bv i
       (get-template-annotation l)))

(define (template2 bv i l)
  (cond ((table-entry *template-descriptors* l)
	 => (lambda (pair) 
	      (set (car pair) (fixnum-ashr (fx+ i 8) 2))))) ;longwords
  (set (bref bv i) 0)
  (set-24 bv (fx+ i 1) (fx+ i 8)))

(define (template3 bv i l)
  (set-16 bv i (get-template-cells l))
  (set (bref-8-u bv (fx+ i 2)) (get-template-nargs l))
  (set (bref-8-u bv (fx+ i 3))
       (if (template-nary l) (fx+ header/template 128) header/template)))
	   
(define (stemplate1 bv i l)
  (set-16 bv (fx+ i 2) 0)	;handler offset
  (set-16 bv i
       (if (not l) 0 (get-template-annotation l))))

(define (stemplate3 bv i l encloser)
  (set-16 bv i
       (let ((n (lambda-max-temps encloser)))
	 (if (fx< n *real-registers*)
	     0
	     (fx+ (fx- n *real-registers*) 1))))
  (set (bref-8-u bv (fx+ i 2)) (if (not l) -2 (get-template-nargs l)))
  (set (bref-8-u bv (fx+ i 3))
       (if (and l (template-nary l))
	   (fx+ header/template 128) header/template)))
	   
(define (laptemplate3 bv i pointer nargs nary?)
  (set-16 bv i pointer)
  (set (bref-8-u bv (fx+ i 2)) nargs)
  (set (bref-8-u bv (fx+ i 3))
       (if nary? (fx+ header/template 128) header/template)))

(define (branch-type bv i cc op2 displ)
  (set-16 bv i
       (fx-ior (fixnum-ashl cc 9)
	       (fx-ior (fixnum-ashl op2 6)
		       (fixnum-logand #x3f
				      (fixnum-ashr displ 16))))) ;high 6 of displ
  (set-16 bv (fx+ i 2) displ))

(define (branch-a-type bv i cc op2 displ)
  (set-16 bv i
       (fx-ior (fx-ior (fixnum-ashl cc 9) (fixnum-ashl 1 13)) ;annul bit
	       (fx-ior (fixnum-ashl op2 6)
		       (fixnum-logand #x3f
				      (fixnum-ashr displ 16))))) ;high 6 of displ
  (set-16 bv (fx+ i 2) displ))

(define (call-type bv i displ)
  (set-16 bv i
       (fx-ior (fixnum-ashl 1 14)
	       (fixnum-logand #x3fff (fixnum-ashr displ 16)))) ;high 14 of displ
  (set-16 bv (fx+ i 2) displ))
  
(define (imm-format bv i op3 rs1 rd displ)
  (set-16 bv i
       (fx-ior (fixnum-ashl 2 14)
	       (fx-ior (fixnum-ashl rd 9)
		       (fx-ior (fixnum-ashl op3 3)
			       (fixnum-ashr rs1 2))))) ;high 3 of rs1
  (set-16 bv (fx+ i 2)
	  (fx-ior (fixnum-ashl (fixnum-logand 3 rs1) 14) ;low 2 of sr1
		  (fx-ior (fixnum-ashl 1 13) ;i bit on
			  (fixnum-logand #x1fff displ)))))

(define (3-reg-format bv i op3 rs1 rs2 rd)
  (set-16 bv i
       (fx-ior (fixnum-ashl 2 14)
	       (fx-ior (fixnum-ashl rd 9)
		       (fx-ior (fixnum-ashl op3 3)
			       (fixnum-ashr rs1 2))))) ;high 3 of rs1
  (set-16 bv (fx+ i 2)
	  (fx-ior (fixnum-ashl (fixnum-logand 3 rs1) 14) ;low 2 of sr1
		  rs2)))

(define (load-store-type bv i op3 rs1 rd displ)
  (set-16 bv i
       (fx-ior (fixnum-ashl 3 14)
	       (fx-ior (fixnum-ashl rd 9)
		       (fx-ior (fixnum-ashl op3 3)
			       (fixnum-ashr rs1 2))))) ;high 3 of rs1
  (set-16 bv (fx+ i 2)
	  (fx-ior (fixnum-ashl (fixnum-logand 3 rs1) 14) ;low 2 of sr1
		  (fx-ior (fixnum-ashl 1 13) ;i bit on
			  (fixnum-logand #x1fff displ)))))


(define (write-i-bytes bv i)
  (let ((write-byte
	 (lambda (byte)
	   (writec (terminal-output) (digit->char (fx-ashr byte 4) 16))  
	   (writec (terminal-output) (digit->char (fx-and byte 15) 16)))))
    (write-byte (bref-8-u bv (fx+ i 0)))
    (write-byte (bref-8-u bv (fx+ i 1)))
    (write-byte (bref-8-u bv (fx+ i 2)))
    (write-byte (bref-8-u bv (fx+ i 3)))))



